/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.packages;

import java.util.*;
import org.freebuilder.system.classes.*;
import org.freebuilder.system.ideengine.Globals;
import org.freebuilder.system.debug.Debug;

/** <code>PacketItemI</code> is for organizing hirerachically packets and projects
  * every element in this hierarchy (item) have to implement this interface
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 12/12/97
*/

abstract public class PacketsContainer extends StringList implements PacketItemI {
  private   PacketItemI ParentItem;
  private   PacketItemI RootItem;
  protected int         ItemKind = PacketItemI.cpikUnknown;

  protected String      FileName = "";    // just file name, example : 'classes.java'
  protected String      FilePath = "";
  protected String      RootFullPath = ""; // Root full dos path. example : c:\jdk1.1.5\lib

  /** AFileName have to be only the the file name, without path to it */
  public PacketsContainer(PacketItemI AParentItem, PacketItemI ARootItem, String AFileName) {
    RootItem   = ARootItem;
    ParentItem = AParentItem;
    FileName   = AFileName;
    if (ParentItem == null) // This is a Root item
      csetFileName(AFileName);
  }

  protected void csetFileName(String AFileName) {
    if (ParentItem != null) return;
    RootItem     = null;
    ParentItem   = null;
    RootFullPath = AFileName;
    FilePath     = Globals.getPath(FileName);
    FileName     = Globals.ExtractFileName(FileName);
  }

  // Begin of the methods from RootPathI interface ---------------------------------------
  // Always returns the full path to the Root Element.
  public String getRootFullPath() {
    PacketItemI RootItem = getRootItem();
    if (RootItem != null) return RootItem.getRootFullPath();
    else                  return RootFullPath;
  }

  public String getRootRelativePath() { return getRootFullPath(); }
  // End   of the methods from RootPathI interface ---------------------------------------

  // Begin of the methods from PacketItemI interface ---------------------------------------
  /** Returns full dos path for that Item. If it is in a .Zip or .Jar file ,
    * returns Path to the Archive + path in the archive. */
  public  String getFullDosPath() {     // Ok. Example : Returns 'c:\FBProject\FBuilder\fb\system\classes\Strings.java'
    String Result  = getRootFullPath();
    String RelPath = getRelativeDosPath();

    Result = Globals.AddFName2Path(Result,RelPath);
    return Globals.validateFileName(Result);
  }

  /** Returns the relative dos path corresponding to the Root Item */
  public  String getRelativeDosPath() { // Ok. Example : Returns 'fb\system\classes\Strings.java'
    String CrntPath;

    if (ParentItem == null) return ""; // This is the Root Item, so relative path is empty.
    CrntPath = ParentItem.getRelativeDosPath();
    CrntPath = Globals.AddFName2Path(CrntPath,getFileName());
    return CrntPath;
  }

  public  String getPath() {            // Ok. Example : Returns 'fb\system\classes\Strings.java'
    if (ParentItem == null) return ""; // This is the Root Item, so relative path is empty.
    String CrntPath = ParentItem.getFullDosPath();
    return CrntPath;
  }

  public  String getDotName() {         // Example : Returns 'org.freebuilder.system.classes.Strings.java'
    String DosPath = getRelativeDosPath();
    if (DosPath.length() == 0) return ""; // Dot Name for Root Item = empty string.
    String Result  = DosPath.replace(Globals.FileSeparatorChar,'.');
    return Result;
  }

  public  String getFileName() { // Ok. Example : Returns 'Strings.java'
    return FileName;
  }

  /** Compares ADotName with the string returned by getDotName() */
  public  boolean MatchDotName(String ADotName) { // Ok.
    String ThisDotName = getDotName();
    if (ADotName == null) return false;
    if ((ADotName.length() == 0) || (ThisDotName.length() == 0)) return false;
    return ThisDotName.compareTo(ADotName) == 0;
  }

  /** Compares with FullDosPath */
  public  boolean MatchPath (String APath) { // Ok.
    String FullDosPath = getFullDosPath();
    if (APath == null) return false;
    if ((APath.length() == 0) || (FullDosPath.length() == 0)) return false;
    return FullDosPath.equalsIgnoreCase(APath);
  }

  /** Compares with FullDosPath */
  public  boolean MatchFileName(String AFileName) {
    String LFileName = getFileName();
    if (AFileName == null) return false;
    if ((AFileName.length() == 0) || (LFileName.length() == 0)) return false;
    return LFileName.equalsIgnoreCase(AFileName);
  }

  /** get Packet Item by It's index in Parent's item */
  public  PacketItemI getPacketItemByFileName(String AFileName) {
    int Cntr, To;
    PacketItemI Result = null;

    if (MatchFileName(AFileName)) return this;

    To = getPacketItemsCount();
    for (Cntr = 0; Cntr < To; Cntr++) {
      Result = getPacketItemAt(Cntr); if (Result == null) continue;
      if (Result.MatchFileName(AFileName))
        return Result;
    }
    return null;
  }

  /** get Packet Item by It's dot name */
  public  PacketItemI getPacketItemByDotName(String ADotName) {
    int Cntr, To;
    PacketItemI Result = null;

    if (MatchDotName(ADotName)) return this;
    if (!isInThisContainer(ADotName, getDotName()))
      return null;

    To = getPacketItemsCount();
    for (Cntr = 0; Cntr < To; Cntr++) {
      Result = getPacketItemAt(Cntr); if (Result == null) continue;
      Result = Result.getPacketItemByDotName(ADotName);
      if (Result != null)
        return Result;
    }
    return null;
  }

  public boolean isInThisContainer (String Search, String This) {
    if (Search == null)
      return false;

    String ThisName = This.toLowerCase();
    String SearchedItemName = Search.toLowerCase();
    if (SearchedItemName.startsWith (ThisName))
      return true;
    return false;
  }

  /** looks for this packet only in current (this) PacketItemI. (Not recursivelly) */
  public PacketItemI findPacketItem(String AFullDosName) {
    int Cntr, To;
    PacketItemI Result = null;

    if (MatchPath(AFullDosName)) return this;

    if (!isInThisContainer(AFullDosName, getFullDosPath()))
      return null;

    To = getPacketItemsCount();
    for (Cntr = 0; Cntr < To; Cntr++) {
      Result = getPacketItemAt(Cntr); if (Result == null) continue;
      if (Result.MatchPath(AFullDosName))
        return Result;
    }
    return null;
  }

  /** looks for this packet recursivelly */
  public PacketItemI findPacketItemR(String AFullDosName) {
    int Cntr, To;
    PacketItemI Result = null;

    if (MatchPath(AFullDosName)) return this;

    if (!isInThisContainer(AFullDosName, getFullDosPath()))
      return null;

    To = getPacketItemsCount();
    for (Cntr = 0; Cntr < To; Cntr++) {
      Result = getPacketItemAt(Cntr); if (Result == null) continue;
      Result = Result.findPacketItemR(AFullDosName);
      if (Result != null)
        return Result;
    }
    return Result;
  }

  /** retunrs IndexOf(this) in parent item */
  public  int getIndex() { // Ok.
    int Result = -1;
    if (ParentItem != null) Result = ParentItem.IndexOfPacketItem(this);
    return Result;
  }

  protected int setPacketItemKind() {
    PacketItemI PItem = null;
    int Cntr = 0, To = getPacketItemsCount();
    int CrntItemKind = PacketItemI.cpikUnknown;

    ItemKind = PacketItemI.cpikDirectory;

    for (Cntr = 0; Cntr < To; Cntr++) {
      PItem = getPacketItemAt(Cntr);
      if (PItem == null) continue;
      CrntItemKind = PItem.getPacketItemKind();
      if ((CrntItemKind == PacketItemI.cpikPacket) || (CrntItemKind == PacketItemI.cpikSource) ||
          (CrntItemKind == PacketItemI.cpikClass)) {
        ItemKind = PacketItemI.cpikPacket;
        return ItemKind;
      }
    }
    return ItemKind;
  }

  public int     getPacketItemKind() {
    if ((ItemKind == PacketItemI.cpikUnknown) || (isChanged())) {
      setPacketItemKind();
    }
    return ItemKind;
  }

  public void    Refresh() {
    setPacketItemKind();
  }

  /** returns true if it is in a .jar or .zip file */
  abstract public boolean isStable();                 // true - if Is In Library
  abstract public boolean isChanged();


  public  PacketItemI getParentItem() {
    return ParentItem;
  }

  public PacketItemI getRootItem() {
    return RootItem;
  }

  public void setParentItem(PacketItemI AParentItem) {
    ParentItem = AParentItem;
  }

  public void setRootItem  (PacketItemI ARootItem) {
    RootItem = ARootItem;
  }

  public int IndexOfPacketItem(PacketItemI APacketItem) {
    return IndexOf(APacketItem);
  }

  public int getPacketItemsCount() {
    return getCount();
  }

  public PacketItemI getPacketItemAt(int AIndex) {
    try {
      return (PacketItemI)getObject(AIndex);
    }
    catch(Exception e) {
      return null;
    }
  }

  public void setPacketItemAt(int AIndex, PacketItemI APacketItem) {
    setObject(AIndex, APacketItem);
  }

  public int addPacketItem(PacketItemI APacketItem) {
    if (APacketItem == null) return -1;
    String AFileName = APacketItem.getFileName();
    return Add(AFileName, APacketItem);
  }
  // End   of the methods from PacketItemI interface ---------------------------------------

  public String getKindAsString() {
    int iKind = getPacketItemKind();
    switch (iKind) {
      case PacketItemI.cpikDirectory        : return "Directory";
      case PacketItemI.cpikPacket           : return "Packet";
      case PacketItemI.cpikSource           : return "Source";
      case PacketItemI.cpikClass            : return "Class";
      case PacketItemI.cpikHtml             : return "Html";
      case PacketItemI.cpikAny              : return "Any";
    }
    return "Unknown";
  }

  public void TestPrint() {
    int i;
    PacketItemI PItem = null;

    System.out.println ("Contenets of " + getFullDosPath());
    System.out.println("------------------------------------------------- ");
    System.out.println("getFullDosPath()      = " + getFullDosPath());
    System.out.println("getRelativeDosPath()  = " + getRelativeDosPath());
    System.out.println("getPath()             = " + getPath());
    System.out.println("getDotName()          = " + getDotName());
    System.out.println("getFileName()         = " + getFileName());
    System.out.println("getIndex()            = " + getIndex());
    System.out.println("isStable()            = " + isStable());

    System.out.println("getPacketItemKind()   = " + getKindAsString());
    System.out.println("isChanged()           = " + isChanged());
    System.out.println("getParentItem()       = " + getParentItem());
    System.out.println("getRootItem()         = " + getRootItem());
    System.out.println("getPacketItemsCount() = " + getPacketItemsCount());
    System.out.println("------------------------------------------------- ");

    for (i = 0; i < getCount(); i ++) {
      PItem = getPacketItemAt(i);
      if (PItem != null) PItem.TestPrint();
    }
    System.out.println ("End of " + getFullDosPath());
  }

  public void ShortTestPrint() {
    int i;
    PacketItemI PItem = null;
    System.out.println(getRelativeDosPath() + "     " + getKindAsString());

    for (i = 0; i < getCount(); i ++) {
      PItem = getPacketItemAt(i);
      if (PItem != null) PItem.ShortTestPrint();
    }
    System.out.println ("");
  }

}
