/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.dpaths;

import java.io.*;
import java.util.*;
import org.freebuilder.system.debug.*;
import org.freebuilder.system.classes.*;
import org.freebuilder.system.packages.*;
import org.freebuilder.system.projectman.*;
import org.freebuilder.system.ideengine.*;
import org.freebuilder.system.classes.events.*;
import org.freebuilder.system.classes.options.*;
import org.freebuilder.gui.ideoptions.*;
import org.freebuilder.system.classes.events.engine.*;

/** <code>RootPathsColl</code> is a collection of all the unique paths defined in
  * SourcePath, ClassPath, OutPath and Project Path.
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 12/12/97
*/

public class RootPathsColl extends StringList {
  private StringList AllItems = null; // This is used only OnRefresh, to discard not used Items.

  public RootPathsColl() {
    OptionsEventAdapter OptionsRAdapter = null;
    OptionsEventAdapter OptionsCAdapter = null;

    AllItems = new StringList();

    try {
      OptionsRAdapter = new OptionsEventAdapter(PathsOptions.PathsOptionsName, OptionsEvent.msgOptionsRefresh,
                                             IdeEventListenerI.CALL_MASK, this, "RefreshPathsByEvent");
      OptionsCAdapter = new OptionsEventAdapter(PathsOptions.PathsOptionsName, OptionsEvent.msgOptionsChanged,
                                             IdeEventListenerI.CALL_MASK, this, "RefreshPathsByEvent");
    } catch (NoSuchMethodException e) {
      Debug.Assert (e, this, "RootPathsColl : No such Method.");
    }
    Globals.EventsEngine.AddEventListener (OptionsRAdapter);
    Globals.EventsEngine.AddEventListener (OptionsCAdapter);
  }

  /** Creates Collection of distinct root paths */
  private void RebuildPaths() {
    Clear();
    System.gc();
    RefreshPaths();
  }

  private void RefreshPaths() {
    PathsOptions POptions = (PathsOptions)(Globals.Options.getOptions(PathsOptions.PathsOptionsName));
    Globals.EventsEngine.SendEvent (new SystemEvent (SystemEvent.msgRootPathsBeginUpdate, this, ""));
    if (POptions == null) return;
    try {
      AllItems.Clear();
      RefreshPath (Globals.getProjectPath());
      RefreshPaths(POptions.getSourcePath());
      RefreshPaths(POptions.getClassPath());
      RefreshPath (POptions.getClassOutPath());
    }
    finally {
      AssignNewItems();
      System.gc();
    }
    Globals.EventsEngine.PostEvent (new SystemEvent (SystemEvent.msgRootPathsChanged, this, ""));
  }

  /** Refreshs only the new paths */
  public void RefreshPathsByEvent (IdeSuperEvent e) {
    RefreshPaths();
  }

  public void RefreshPaths(String[] FullDosPaths) {
    if (FullDosPaths == null) return;
    for (int Cntr = 0; Cntr < FullDosPaths.length; Cntr++)
       RefreshPath(FullDosPaths[Cntr]);
  }

  public int  Add (final String S, Object AObject) { // Override
    int Index = IndexOf(S);
    if (Index < 0)
      Index = super.Add(S, AObject);
    return Index;
  }

  void RefreshPath(String FullDosPath) {
    PacketItemI NewEntry = getRootPath(FullDosPath);

    if (NewEntry != null) { // If this root item already exist, just will copy it's reference
      AllItems.Add(FullDosPath, NewEntry);
      return;
    }
    AddNewPath2AllItems(FullDosPath); // Add this new item to AllItems.
  }

  void AddNewPath2AllItems(String FullDosPath) {
    PacketItemI NewEntry = null;
    if (FullDosPath == null) return;
    if (FullDosPath.length() == 0) return;

    NewEntry = PCreateEntry.CreateEntry(FullDosPath);
    AllItems.Add(FullDosPath, NewEntry);
  }

  void AssignNewItems() {
    BeginUpdate();
    try {
      Clear();
      Add(AllItems);
    }
    finally {
      EndUpdate();
    }
  }

  void RebuildPath(String FullDosPath) {
    PacketItemI NewEntry = null;
    if (FullDosPath == null) return;
    if (FullDosPath.length() == 0) return;

    NewEntry = PCreateEntry.CreateEntry(FullDosPath);
    Add(FullDosPath, NewEntry);
  }

  public PacketItemI getRootPath(String FullDosPath) {
    if (FullDosPath == null) return null;
    PacketItemI Result = (PacketItemI)getObject(FullDosPath);
    return Result;
  }

  public PacketItemI findPacketItemI (String FullDosPath) {
    int Count = getCount();
    PacketItemI RetPacketItemI = null;
    int i;
    for (i = 0; i < Count; i++) {
      RetPacketItemI = (PacketItemI) getObject(i);
      RetPacketItemI = RetPacketItemI.findPacketItemR(FullDosPath);
      if (RetPacketItemI != null)
         break;
     }
    if (i == Count)
      return null;
    return RetPacketItemI;
  }

  /** get Packet Item by It's dot name */
  public  PacketItemI getPacketItemByDotName(String ADotName) {
    int Count = getCount();
    PacketItemI RetPacketItemI = null;
    if (ADotName == null) {
      Debug.Assert(this, "ADotName == null");
      return null;
    }
    int i;
    for (i = 0; i < Count; i++) {
      RetPacketItemI = (PacketItemI) getObject(i);
      if (RetPacketItemI != null)
        RetPacketItemI = RetPacketItemI.getPacketItemByDotName(ADotName);
      if (RetPacketItemI != null)
        break;
    }
    if (i == Count)
      return null;
    return RetPacketItemI;
  }

  public void TestPrint() {
     PacketItemI CrntItem = null;
     String      ItemName = "";

     for (int Cntr = 0; Cntr < getCount(); Cntr++) {
       ItemName = getString(Cntr);
       CrntItem = getRootPath(ItemName);
       if (CrntItem != null)
         CrntItem.TestPrint();
     }
  }

  public void ShortTestPrint() {
     PacketItemI CrntItem = null;
     String      ItemName = "";

     for (int Cntr = 0; Cntr < getCount(); Cntr++) {
       ItemName = getString(Cntr);
       CrntItem = getRootPath(ItemName);

       System.out.println("RootPath : " + ItemName);
       if (CrntItem != null)
         CrntItem.ShortTestPrint();
       else
         System.out.println ("CrntItem == null   !!!  ---- " + ItemName);
     }
  }


}
