/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.gui.ideoptions.abstr;
import java.io.*;
import java.awt.*;
import java.awt.Image;
import org.freebuilder.gui.beans.*;
import org.freebuilder.gui.ide.*;
import org.freebuilder.system.classes.*;
import org.freebuilder.system.ideengine.Globals;

/** This class represents a bean in a palette page.
  * @author	<a href=mailto:wreck@nat.bg> Peter Radkov </a>
    @version 0.7, 07-01-98
*/
public class ABean implements SimpleItem, NamedIcon {

  /** The page in which this Bean resides. */
  private PalettePage Owner;

  /** The Class object, representing this Bean. */
  private transient Class BeanClass;

  /** The icon of this this Bean. */
  private transient Image BeanIcon;

  /** The Bean icon file name. */
  private String BeanIconFile;

  /** The Bean class file name. */
  private String BeanClassFile;

  /** The user defined neame if this Bean. */
  private String Name = null;

  /**
    Constructs the Bean with no icon.
    @exception ClassNotFoundException If the class could
    not be found.
  */
  public ABean (PalettePage Owner, String BeanClassFile) throws ClassNotFoundException {
    this(Owner, BeanClassFile, null);
  }

  /**
    Constructs the Bean.
    @exception ClassNotFoundException If the class could not be found.
  */
  public ABean (PalettePage Owner,
                String BeanClassFile,
                String BeanIconFile,
                boolean foo) throws ClassNotFoundException {
    this (Owner, BeanClassFile, null);
    this.BeanIconFile = BeanIconFile;
    BeanIcon = Globals.ResourceMngr.loadImage (BeanIconFile);
  }

  /**
    Constructs the Bean.
    @exception ClassNotFoundException If the class could not be found.
  */
  public ABean (PalettePage Owner,
                String BeanClassFile,
                String BeanIconFile) throws ClassNotFoundException {

    this.Owner         = Owner;
    this.BeanClassFile = BeanClassFile;
    this.BeanIconFile  = BeanIconFile;
    init();
  }

  /** Loads the image from the specified file */
  private void loadIcon () {
    if (BeanIconFile != null)
      BeanIcon = Globals.ResourceMngr.loadImage (BeanIconFile);
    else
      BeanIcon = null;
  }

  /**
    Initializes the Bean. Constructs the Class object. Loads The Icon,
    if present. Sets the default name to be the name of the class, if
    no user defined name is given.
    @exception ClassNotFoundException If the class could not be found.
  */
  private void init () throws ClassNotFoundException {
    String ClassFile = BeanClassFile;
    int i;

    while (true)
     try {
      BeanClass = Class.forName (ClassFile);
      break;
     } catch (ClassNotFoundException e) {
       i = ClassFile.indexOf('.');
       if (i == -1)
         throw e;
       ClassFile = ClassFile.substring (i+1,ClassFile.length());
     }

    if (getName() == null)
      setName (BeanClass.getName());
    loadIcon();
  }

  /**
    Deserializes the Bean.
    @exception ClassNotFoundException If the class could not be found.
    @exception IOException If a read error occurs.
  */
  private void readObject(java.io.ObjectInputStream in)
     throws IOException, ClassNotFoundException {

    in.defaultReadObject ();
    init();
  }

  /** Returns the icon of the Bean. */
  public Image getIcon () {
    return BeanIcon;
  }

  /**  Loads a new icon for the Bean. */
  public void setIcon (String BeanIconFile) {
    this.BeanIconFile = BeanIconFile;
    loadIcon ();
  }

  /** Returns the name of the icon file */
  public String getIconFile () {
    return BeanIconFile;
  }

  /** Returns the name of the Bean. */
  public String getName () {
    return Name;
  }

  /** Returns the name of the Bean. */
  public String toString () {
    return Name;
  }

  /** Sets the name of the Bean. */
  public void setName (String NewName) {
    Name = NewName;
  }

  /**
    Removes this Bean from its Owner.
    Returns this Bean, after it is being removed.
  */
  public ABean remove () {
    Owner.removeBean (this);
    return this;
  }

  public ABean clone (PalettePage NewOwner) {
    try {
      return new ABean (NewOwner, BeanClassFile, BeanIconFile, true);
    } catch (ClassNotFoundException e) {return null;}
  }

  public Class getBeanClass () {
    return BeanClass;
  }

  public boolean equals (ABean AnotherBean) {
    if ((getBeanClass().equals(AnotherBean.getBeanClass())) &&
        (getIconFile ().equals(AnotherBean.getIconFile ())))
      return true;
    return false;
  }
}
