UNIT BrdHndlr;
  {Board handling routines for Fumiko}


INTERFACE


  USES
    CRT, GloblCTV, Arcana;


  FUNCTION Legal_Move (var x, y: byte; player: color_type; var board: board_type): boolean;
    {Returns true if the move in question is legal}


  FUNCTION Suicidal (var x, y: byte; var player: color_type; var board: board_type): boolean;
    {Returns true if the move in question would result in its own capture}


  PROCEDURE Draw_Board (var board: board_type);
    {Draws the board on the screen}


  PROCEDURE Get_User_Move (player: color_type; var board: board_type);
    {Gets a move from the user and updates the board}


  PROCEDURE Pass (var board: board_type);
    {Increments the pass and turn counters and clears the ko variable}


  PROCEDURE Play_At (var x, y: byte; var player: color_type; var board: board_type; depth: byte);
    {Plays at the indicated place and updates the groups, passes,}
    {turn, and ko variables}


  PROCEDURE Estimate_Score (var board: board_type; var b, w: integer; var points: integer_board);
    {Calculates influence for each point and adds up the score}


  PROCEDURE Show_Score (var board: board_type);
    {Estimates the score and shows the influence around the board}


  PROCEDURE Save_Board (var board: board_type; var filename: string);
    {Saves the board to a file and increments board_number}


  PROCEDURE Load_Board (var board: board_type; var filename: string);
    {Loads board from a file}


  PROCEDURE Delete_Board (var filename: string);
    {Deletes a saved board}


  PROCEDURE Show_Final_Score (var board: board_type);
    {Removes groups of aliveness 1-2, counts the score, and displays it}


IMPLEMENTATION


  USES
    GroupMgr, Sequence;


  FUNCTION A_Neighboring_Point_Is_Unoccupied (var x, y: byte; var board: board_type): boolean;
    {Returns true if any adjacent point is empty.  LOCAL}
    Var
      direction: direction_type;
      neighbor: location_type;
      found_one: boolean;
    Begin  {A Neighboring Point Is Unoccupied}
      found_one := false;
      For direction := north to west do
        Begin
          neighbor := neighbors [x, y, direction];
          If on_board (neighbor.x, neighbor.y) and (board.data [neighbor.x, neighbor.y].color = empty)
            Then found_one := true
        End;
      a_neighboring_point_is_unoccupied := found_one
    End;  {A Neighboring Point Is Unoccupied}


  FUNCTION Legal_Move (var x, y: byte; player: color_type; var board: board_type): boolean;
    {Returns true if the move in question is legal}
    Begin  {Legal Move}
      If board.data [x, y].color <> empty
        Then legal_move := false
        Else If a_neighboring_point_is_unoccupied (x, y, board)
               Then legal_move := true
               Else legal_move := not ((board.ko.x = x) and (board.ko.y = y))
    End;  {Legal Move}


  FUNCTION Suicidal (var x, y: byte; var player: color_type; var board: board_type): boolean;
    {Returns true if the move in question would result in its own capture}
    Var
      direction: direction_type;
      safe: boolean;
      adjacent_point: location_type;
    Begin  {Suicidal}
      If a_neighboring_point_is_unoccupied (x, y, board)
        Then safe := true
        Else Begin
               safe := false;
               For direction := north to west do
                 Begin
                   adjacent_point := neighbors [x, y, direction];
                   If on_board (adjacent_point.x, adjacent_point.y)
                     Then If board.data [adjacent_point.x, adjacent_point.y].color = player
                            Then Begin
                                   If board.groups [board.data [adjacent_point.x, adjacent_point.y].group]^.liberties > 1
                                     Then safe := true
                                 End
                            Else If (board.data [adjacent_point.x, adjacent_point.y].color in [black_stone, white_stone])
                                      and (board.groups [board.data [adjacent_point.x, adjacent_point.y].group]^.liberties = 1)
                                   Then safe := true
                 End
             End;
      suicidal := not safe
    End;  {Suicidal}


  PROCEDURE Draw_Board (var board: board_type);
    {Draws the board on the screen}
    Var
      x, y: byte;
      c: char;
    Begin  {Draw Board}
      ClrScr;
      WriteLn (board_index_line);
      For y := board_size downto 1 do
        Begin
          Write (y:2, ' ');
          For x := 1 to board_size do
            Begin
              c := screen_char (board.data [x, y].color);
              If (c = #250) and (x in [4, 10, 16]) and (y in [4, 10, 16])
                Then c := '+';
              Write (c, ' ')
            End;
          WriteLn (y)
        End;
      WriteLn (board_index_line);
      WriteLn
    End;  {Draw Board}


  PROCEDURE Show_Influence (var group: group_type);
    {Displays the status of a group.  LOCAL}
    Var
      x, y: byte;
    Begin  {Show Influence}
      ClrScr;
      WriteLn (board_index_line);
      For y := board_size downto 1 do
        Begin
          Write (y:2, ' ');
          For x := 1 to board_size do
            If group.interpretation [x, y].too_distant
              Then Write ('. ')
              Else If group.interpretation [x, y].connected
                     Then Write ('@ ')
                     Else Write ('+ ');
          WriteLn (y)
        End;
      WriteLn (board_index_line);
      WriteLn;
      WriteLn ('Size: ', group.size, '   Liberties: ', group.liberties);
      WriteLn ('Aliveness: ', group.aliveness, '   Room: ', group.room);
      WriteLn ('To kill: ', group.to_kill.x, ',', group.to_kill.y, '   To save: ', group.to_save.x, ',', group.to_save.y);
      WriteLn ('(Return to continue)');
      ReadLn
    End;  {Show Influence}


  PROCEDURE Show_Points_To_Attack (var group: group_type; var board: board_type);
    {Performs and displays results of find_points_to_attack.  LOCAL}
    Var
      result: boolean_board;
      x, y: byte;
      points, here: location_list;
    Begin  {Show Points To Attack}
      points := critical_points_for (group, board);
      ClrScr;
      WriteLn ('Points to attack that group: ');
      WriteLn;
      Draw_Board (board);
      here := points;
      While here <> nil do
        Begin
          GotoXY (2 + (2 * here^.location.x), 2 + board_size - here^.location.y);
          Write ('x');
          here := here^.next
        End;
      GotoXY (1, 4 + board_size);
      WriteLn ('(Return to continue)');
      Clean_Out_Location_List (points);
      ReadLn
    End;  {Show Points To Attack}


  PROCEDURE Get_User_Move (player: color_type; var board: board_type);
    {Gets a move from the user and updates the board}
    Var
      move: string;
      x, y: byte;
      error_code: integer;
      legal: boolean;
      command: char;
    Begin  {Get User Move}
      Repeat
        Draw_Board (board);
        Write ('Turn #', board.turn, '   ');
        If player = black_stone
          Then Write ('Black (*) to move: ')
          Else Write ('White (O) to move: ');
        ReadLn (move);
        legal := true;
        command := ' ';
        If move <> 'pass'
          Then Begin
                 If move [1] in ['@', '!', '?', '<']
                   Then Begin
                          legal := false;
                          command := move [1];
                          move := copy (move, 2, 3)
                        End;
                 Case upcase (move [1]) of
                   'A'..'H': x := Ord (upcase (move [1])) - Ord ('A') + 1;
                   'J'..'T': x := Ord (upcase (move [1])) - Ord ('A')
                   Else legal := false
                 End;
                 Val (copy (move, 2, 2), y, error_code);
                 If legal
                   Then legal := (error_code = 0) and (on_board (x, y));
                 If legal
                   Then legal := legal_move (x, y, player, board)
               End;
        If not legal
          Then If command <> ' '
                 Then Begin
                        If (on_board (x, y) and (board.data [x, y].group <> 0)) or (command in ['?', '<'])
                          Then Case command of
                                 '@': Show_Influence (board.groups [board.data [x, y].group]^);
                                 '!': Show_Points_To_Attack (board.groups [board.data [x, y].group]^, board);
                                 '?': Begin
                                        WriteLn (strategy (board, player));
                                        WriteLn ('(Return to continue)');
                                        ReadLn
                                      End;
                                 '<': Begin
                                        Load_Board (board, last_board_filename);
                                        Draw_Board (board)
                                      End
                               End
                      End
                 Else Begin
                        Write ('That is not a legal move -- return to continue');
                        ReadLn
                      End
      Until legal;
      Save_Board (physical_board, last_board_filename);
      If move = 'pass'
        Then Pass (board)
        Else Play_At (x, y, player, board, 0)
    End;  {Get User Move}


  PROCEDURE Pass (var board: board_type);
    {Increments the pass and turn counters and clears the ko variable}
    Begin  {Pass}
      Inc (board.passes);
      Inc (board.turn);
      board.ko.x := 0;
      board.ko.y := 0
    End;  {Pass}


  PROCEDURE Update_Relevant_Groups (var board: board_type; var changed: boolean_board; var depth: byte);
    {An ugly part of play_at.  Updates all groups who do not consider all of}
    {the affected points too_distant.  LOCAL}
    Var
      color: color_type;
      group, x, y: byte;
    Begin  {Update Relevant Groups}
      group := 1;
      While board.groups [group]^.size <> 0 do
        Begin
          For x := 1 to board_size do
            For y := 1 to board_size do
              If changed [x, y] and (not board.groups [group]^.interpretation [x, y].too_distant)
                   and (board.groups [group]^.last_update < board.turn)
            Then Update_Interpretation (group, board, depth);
          Inc (group)
        End
    End;  {Update Relevant Groups}


  PROCEDURE Play_At (var x, y: byte; var player: color_type; var board: board_type; depth: byte);
    {Plays at the indicated place and updates the groups, passes,}
    {turns, and ko variables}
    Var
      direction: direction_type;
      neighbor: location_type;
      stones_captured: byte;
      changes: boolean_board;
      trickle: text;
    Begin  {Play At}
      board.data [x, y].color := player;
      stones_captured := 0;
      Make_New_Group (x, y, board);
      changes := false_board;
      changes [x, y] := true;
      For direction := north to west do
        Begin
          neighbor := neighbors [x, y, direction];
          If on_board (neighbor.x, neighbor.y)
            Then Begin
                   If board.data [neighbor.x, neighbor.y].color = player
                     Then Begin
                            If board.data [x, y].group <> board.data [neighbor.x, neighbor.y].group
                              Then Merge_Groups (board.data [x, y].group, board.data [neighbor.x, neighbor.y].group, board)
                          End
                     Else If board.data [neighbor.x, neighbor.y].color in [black_stone, white_stone]
                            Then Begin  {It's occupied by the other player}
                                   Update_Interpretation (board.data [neighbor.x, neighbor.y].group, board, depth);
                                   If board.groups [board.data [neighbor.x, neighbor.y].group]^.liberties = 0
                                     Then Begin
                                            board.ko := neighbor;
                                            Inc (stones_captured,
                                                  board.groups [board.data [neighbor.x, neighbor.y].group]^.size);
                                            Capture (board.data [neighbor.x, neighbor.y].group, board, changes, depth)
                                          End
                                 End
                 End
        End;
      Update_Interpretation (board.data [x, y].group, board, depth);
      If (stones_captured = 0) and (board.groups [board.data [x, y].group]^.liberties = 0)
        Then Capture (board.data [x, y].group, board, changes, depth)
        Else If (stones_captured <> 1) or (board.groups [board.data [x, y].group]^.size <> 1)
                  or (board.groups [board.data [x, y].group]^.liberties <> 1)
               Then Begin
                      board.ko.x := 0;
                      board.ko.y := 0
                    End;
      Update_Relevant_Groups (board, changes, depth);
      board.passes := 0;
      If depth = 0
        Then Begin
               Assign (trickle, trickle_filename);
               Append (trickle);
               WriteLn (trickle, '+', board.turn);
               WriteLn (trickle, x);
               WriteLn (trickle, y);
               Close (trickle)
             End;
      Inc (board.turn)
    End;  {Play At}


  PROCEDURE Estimate_Score (var board: board_type; var b, w: integer; var points: integer_board);
    {Calculates influence for each point and adds up the score}
    Var
      x, y, group: byte;
      influence: integer;
      color: color_type;
    Begin  {Estimate Score}
      b := 0;
      w := 0;
      For x := 1 to board_size do
        For y := 1 to board_size do
          Begin
            points [x, y] := 0;
            group := 1;
            While board.groups [group]^.size <> 0 do
              Begin
                If board.groups [group]^.interpretation [x, y].connected
                  Then Begin
                         If board.groups [group]^.interpretation [x, y].member
                           Then influence := board.groups [group]^.aliveness * 200
                           Else influence := (board.groups [group]^.aliveness * 100)
                                               div board.groups [group]^.interpretation [x, y].distance;
                         If board.groups [group]^.owner = black_stone
                           Then points [x, y] := points [x, y] + influence
                           Else points [x, y] := points [x, y] - influence
                       End;
                Inc (group)
              End;
            If points [x, y] > 0
              Then b := b + 1
              Else If points [x, y] < 0
                     Then w := w + 1
          End
    End;  {Estimate Score}


  PROCEDURE Show_Score (var board: board_type);
    {Estimates the score and shows the influence around the board}
    Var
      x, y: byte;
      b, w: integer;
      points: integer_board;
    Begin  {Show Score}
      Estimate_Score (board, b, w, points);
      ClrScr;
      WriteLn (board_index_line);
      For y := board_size downto 1 do
        Begin
          Write (y:2, ' ');
          For x := 1 to board_size do
            Case points [x, y] of
              -maxint..-1000: Write ('W ');
              -999..-100: Write ('w ');
              -99..-1: Write ('- ');
              0: Write ('. ');
              1..99: Write ('| ');
              100..999: Write ('b ');
              1000..maxint: Write ('B ')
            End;
          WriteLn (y)
        End;
      WriteLn (board_index_line);
      WriteLn;
      WriteLn ('Black: ', b, '  White: ', w);
      ReadLn
    End;  {Show Score}


  PROCEDURE Save_Board (var board: board_type; var filename: string);
    {Saves the board to a file and increments board_number}
    Var
      board_file: board_file_type;
      group_file: group_file_type;
      count: byte;
    Begin  {Save Board}
      If Copy (filename, 3, 8) <> 'PHYSICAL'
        Then Begin
               Str (board_number, filename);
               filename := drive_name + 'F' + filename;
               board_number := (board_number + 1) mod maxint
             End;
      Assign (board_file, filename + '.BRD');
      Rewrite (board_file);
      Write (board_file, board);
      Close (board_file);
      Assign (group_file, filename + '.GRP');
      Rewrite (group_file);
      count := 1;
      While (count <= max_groups) and (board.groups [count]^.size <> 0) do
        Begin
          Write (group_file, board.groups [count]^);
          Inc (count)
        End;
      Close (group_file)
    End;  {Save Board}


  PROCEDURE Load_Board (var board: board_type; var filename: string);
    {Loads board from a file}
    Var
      board_file: board_file_type;
      group_file: group_file_type;
      count: byte;
    Begin  {Load Board}
      Assign (board_file, filename + '.BRD');
      {$I-}
      Reset (board_file);
      {$I+}
      If ioresult = 0
        Then Begin
               Read (board_file, board);
               Close (board_file);
               Assign (group_file, filename + '.GRP');
               Reset (group_file);
               count := 1;
               While not eof (group_file) do
                 Begin
                   Read (group_file, board.groups [count]^);
                   Inc (count)
                 End;
               Close (group_file);
               If (count <= max_groups)
                 Then board.groups [count]^.size := 0
             End
    End;  {Load Board}


  PROCEDURE Delete_Board (var filename: string);
    {Deletes a saved board}
    Var
      deadfile: file;
    Begin  {Delete Board}
      Assign (deadfile, filename + '.BRD');
      Erase (deadfile);
      Assign (deadfile, filename + '.GRP');
      Erase (deadfile)
    End;  {Delete Board}


  PROCEDURE Show_Final_Score (var board: board_type);
    {Removes groups of aliveness 1-2, counts the score, and displays it}
    Var
      count, changes, x, y, black_neighbors, white_neighbors: byte;
      counted: boolean_board;
      black_score, white_score: real;
      score_array: integer_board;
      done: boolean;
      direction: direction_type;
      neighbor: location_type;
    Begin  {Show Final Score}
      count := 1;
      time_left := maxint;
      WriteLn ('Counting score...');
      Repeat
        changes := 0;
        While board.groups [count]^.size <> 0 do
          Begin
            Update_Interpretation (count, board, 0);
            If board.groups [count]^.aliveness < 3
              Then Begin
                     Capture (count, board, counted, 0);
                     Inc (changes)
                   End
              Else Inc (count);
            WriteLn ('Next: size ', board.groups [count]^.size)
          End
      Until changes = 0;
      Draw_Board (board);
      black_score := 0;
      white_score := 0;
      score_array := zero_board;
      counted := false_board;
      Repeat
        For x := 1 to board_size do
          For y := 1 to board_size do
            If not counted [x, y]
              Then Begin
                     count := 0;
                     If board.data [x, y].color in [black_stone, white_stone]
                       Then If board.data [x, y].color = black_stone
                              Then score_array [x, y] := 12
                              Else score_array [x, y] := -12
                       Else Begin
                              white_neighbors := 0;
                              black_neighbors := 0;
                              For direction := north to west do
                                Begin
                                  neighbor := neighbors [x, y, direction];
                                  If on_board (neighbor.x, neighbor.y)
                                    Then If board.data [neighbor.x, neighbor.y].color = black_stone
                                           Then Inc (black_neighbors)
                                           Else If board.data [neighbor.x, neighbor.y].color = white_stone
                                                  Then Inc (white_neighbors)
                                End;
                              Case (black_neighbors + white_neighbors) of
                                1: score_array [x, y] := (12 * black_neighbors) - (12 * white_neighbors);
                                2: score_array [x, y] := (6 * black_neighbors) - (6 * white_neighbors);
                                3: score_array [x, y] := (4 * black_neighbors) - (4 * white_neighbors);
                                4: score_array [x, y] := (3 * black_neighbors) - (3 * white_neighbors)
                                Else Begin
                                       For direction := north to west do
                                         Begin
                                           neighbor := neighbors [x, y, direction];
                                           If on_board (neighbor.x, neighbor.y)
                                             Then Begin
                                                    If counted [neighbor.x, neighbor.y]
                                                      Then Inc (count);
                                                    If score_array [neighbor.x, neighbor.y] > 0
                                                      Then Inc (black_neighbors)
                                                      Else If score_array [neighbor.x, neighbor.y] < 0
                                                           Then Inc (white_neighbors)
                                                  End
                                         End;
                                       If (black_neighbors > 0) and (white_neighbors > 0)
                                         Then score_array [x, y] := 0
                                         Else Case (black_neighbors + white_neighbors) of
                                                1: score_array [x, y] := (12 * black_neighbors) - (12 * white_neighbors);
                                                2: score_array [x, y] := (6 * black_neighbors) - (6 * white_neighbors);
                                                3: score_array [x, y] := (4 * black_neighbors) - (4 * white_neighbors);
                                                4: score_array [x, y] := (3 * black_neighbors) - (3 * white_neighbors)
                                              End
                                     End
                              End
                            End;
                     counted [x, y] := (black_neighbors > 0) or (white_neighbors > 0);
                     If counted [x, y]
                       Then If score_array [x, y] > 0
                              Then black_score := black_score + (score_array [x, y] / 12)
                              Else white_score := white_score - (score_array [x, y] / 12)
                       Else score_array [x, y] := 0
                   End;
        done := true;
        For x := 1 to board_size do
          For y := 1 to board_size do
            If not counted [x, y]
              Then done := false
      Until done;
      WriteLn;
      WriteLn ('Black:  ', black_score:6:2, '  White: ', white_score:6:2)
    End;  {Show Final Score}


END.  {BrdHndlr}

