/******************************************************************************
 *                                                                            *
 *      Header          :       ProTracker Music Interface Kit                *
 *      Author          :       Marq/Fit                                      *
 *      Version         :       0.5                                           *
 *      Description     :       Functions for 4-channel PT/NST/ST module      *
 *                              playing with DAC, SB & GUS                    *
 *      Implemented cmds:       0  - Arpeggio                                 *
 *                              1  - Porta Up                                 *
 *                              2  - Porta Dn                                 *
 *                              3  - Tone porta                               *
 *                              4  - Vibrato                                  *
 *                              5  - ToneP + VolSlide                         *
 *                              6  - Vibrato + VolSlide                       *
 *                              7  - Tremolo                                  *
 *                              9  - Sample offset                            *
 *                              A  - Volume slide                             *
 *                              B  - Jump                                     *
 *                              C  - Volume set                               *
 *                              D  - Pattern break                            *
 *                              E1 - FineSlideUp                              *
 *                              E2 - FineSlideDn                              *
 *                              E6 - Jumploop                                 *
 *                              E9 - Retrig note                              *
 *                              EA - FineVolUp                                *
 *                              EB - FineVolDn                                *
 *                              EC - Cut note                                 *
 *                              ED - Note delay                               *
 *                              EE - Pattern delay                            *
 *                              F  - Speed/Tempo set                          *
 *      Comments        :       Thanks to Yzi & Pete for their support and    *
 *                              -testing!                                    *
 *                                                                            *
 ******************************************************************************/

#pragma inline

#include <alloc.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <mem.h>
#include <stdio.h>
#include <stdlib.h>

#define pt_MASTER_CLK 3546836l
#define pt_ERROR 0
#define pt_OK 1
#define pt_DAC 0
#define pt_SB 1
#define pt_GUS 2
#define pt_LPT1 peek(0x40,0x6+2)
#define pt_LPT2 peek(0x40,0x6+4)
#define pt_LPT3 peek(0x40,0x6+6)
#define pt_LPT4 peek(0x40,0x6+8)

#define GUS_START 2
#define GUS_END 4
#define GUS_CURRENT 0xa
#define GUS_GF1_RESET 0x4c     
#define GUS_VOICE_CONTROL 0
#define GUS_ACTIVE_VOICES 0xe

int     pt_curpos,
        pt_index,
        pt_inst,
        pt_itemp,
        pt_mainvol=64,
        pt_pattdelay=0,
        pt_curper[4],
        pt_vol[4],
        pt_cvol[4],
        pt_tonedest[4],
        pt_tonespd[4],
        pt_jumpdest[4],
        pt_jumpcnt[4];

unsigned        pt_freq,
                pt_device,
                pt_port,
                pt_SB_port,
                pt_counter,
                pt_VBmax,
                pt_VBcnt,
                pt_oldmax,
                pt_speed,
                pt_save_speed,
                pt_row,
                pt_off,
                pt_temp,
                pt_slen[31],
                pt_roff[31],
                pt_rlen[31],
                pt_sseg[31],
                pt_clen[4],
                pt_sdec[4],
                pt_snum[4],
                pt_next[4],
                pt_add_off[4],
                pt_add_dec[4],
                pt_vib_off[4],
                pt_vib_size[4],
                pt_vib_spd[4],
                pt_tre_off[4],
                pt_tre_size[4],
                pt_tre_spd[4],
                pt_note[4],
                pt_per_off[1024],
                pt_per_dec[1024],
                gus_baseaddr,
                gus_maxfrequency;

unsigned char   pt_modlen,
                pt_cmd,
                pt_prm,
                pt_ptns,
                pt_on[4],
                pt_pos[128],
                pt_loop[31],
                pt_svol[31],
                pt_finetune[31],
                pt_songname[21],
                *pt_patt[100],
                *pt_smp[31],
                gus_pan[4]={4,12,12,4};

unsigned long   pt_address[4],
                gus_start[31],
                gus_end[31],
                gus_repeat[31],
                gus_meminstalled,
                gus_availstart;

unsigned pt_period[]={
            856,808,762,720,678,640,604,570,538,508,480,453,    /*  0   */
            428,404,381,360,339,320,302,285,269,254,240,226,
            214,202,190,180,170,160,151,143,135,127,120,113,

            850,802,757,715,674,637,601,567,535,505,477,450,    /*  1   */
            425,401,379,357,337,318,300,284,268,253,239,225,
            213,201,189,179,169,159,150,142,134,126,119,113,

            844,796,752,709,670,632,597,563,532,502,474,447,    /*  2   */
            422,398,376,355,335,316,298,282,266,251,237,224,
            211,199,188,177,167,158,149,141,133,125,118,112,

            838,791,746,704,665,628,592,559,528,498,470,444,    /*  3   */
            419,395,373,352,332,314,296,280,264,249,235,222,
            209,198,187,176,166,157,148,140,132,125,118,111,

            832,785,741,699,660,623,588,555,524,495,467,441,    /*  4   */
            416,392,370,350,330,312,294,278,262,247,233,220,
            208,196,185,175,165,156,147,139,131,124,117,110,

            826,779,736,694,655,619,584,551,520,491,463,437,    /*  5   */
            413,390,368,347,328,309,292,276,260,245,232,219,
            206,195,184,174,164,155,146,138,130,123,116,109,

            820,774,730,689,651,614,580,547,516,487,460,434,    /*  6   */
            410,387,365,345,325,307,290,274,258,244,230,217,
            205,193,183,172,163,154,145,137,129,122,115,109,

            814,768,725,684,646,610,575,543,513,484,457,431,    /*  7   */
            407,384,363,342,323,305,288,272,256,242,228,216,
            204,192,181,171,161,152,144,136,128,121,114,108,

            907,856,808,762,720,678,640,604,570,538,508,480,    /*  -8  */
            453,428,404,381,360,339,320,302,285,269,254,240,
            226,214,202,190,180,170,160,151,143,135,127,120,

            900,850,802,757,715,675,636,601,567,535,505,477,    /*  -7  */
            450,425,401,379,357,337,318,300,284,268,253,238,
            225,212,200,189,179,169,159,150,142,134,126,119,

            894,844,796,752,709,670,632,597,563,532,502,474,    /*  -6  */
            447,422,398,376,355,335,316,298,282,266,251,237,
            223,211,199,188,177,167,158,149,141,133,125,118,

            887,838,791,746,704,665,628,592,559,528,498,470,    /*  -5  */
            444,419,395,373,352,332,314,296,280,264,249,235,
            222,209,198,187,176,166,157,148,140,132,125,118,

            881,832,785,741,699,660,623,588,555,524,494,467,    /*  -4  */
            441,416,392,370,350,330,312,294,278,262,247,233,
            220,208,196,185,175,165,156,147,139,131,123,117,

            875,826,779,736,694,655,619,584,551,520,491,463,    /*  -3  */
            437,413,390,368,347,328,309,292,276,260,245,232,
            219,206,195,184,174,164,155,146,138,130,123,116,

            868,820,774,730,689,651,614,580,547,516,487,460,    /*  -2  */
            434,410,387,365,345,325,307,290,274,258,244,230,
            217,205,193,183,172,163,154,145,137,129,122,115,

            862,814,768,725,684,646,610,575,543,513,484,457,    /*  -1  */
            431,407,384,363,342,323,305,288,272,256,242,228,
            216,203,192,181,171,161,152,144,136,128,121,114};

int     pt_vib_table[64]={
            0,24,49,74,97,120,141,161,180,197,212,224,235,244,250,253,
            255,253,250,244,235,224,212,197,180,161,141,120,97,74,49,24,
            0,-24,-49,-74,-97,-120,-141,-161,-180,-197,-212,-224,-235,-244,
            -250,-253,-255,-253,-250,-244,-235,-224,-212,-197,-180,-161,
            -141,-120,-97,-74,-49,-24};

int pt_init(int device,unsigned port);
int pt_load(char *path);
int pt_play(unsigned freq);
void pt_stop(void);
void pt_pause(void);
void pt_continue(void);
void pt_SB_handler(void);
void pt_DAC_handler(void);
void interrupt pt_GUS_handler(void);
void pt_commands(void);
void pt_retrig(int n);
void pt_checkloop(void);
void interrupt (*pt_oldfunc)();
void pt_calc_inc(void);
void pt_settimer(unsigned freq);
void pt_resettimer(void);
void pt_restoreclock(void);
unsigned pt_bcd2d(unsigned bcd);
int pt_resetSB(int ba);
void gus_pokeb(unsigned addr_hi,unsigned addr_lo,unsigned char data);
unsigned char gus_peekb(unsigned addr_hi,unsigned addr_lo);
void gus_outb(int reg,unsigned char data);
void gus_outw(int reg,unsigned data);
void gus_setperiod(unsigned per);
void gus_resetGF1(void);
void gus_delay(void);
int gus_reset(int portti);
void gus_vol(unsigned vol);
void gus_setlocation(unsigned char l_type,long loca);
void gus_selectvoice(unsigned chan);
void gus_transferblock(unsigned seg0,unsigned ofs0,unsigned adhi,unsigned adlo,unsigned tsize);
void gus_handler(void);
void gus_retrig(int n);

int pt_init(int device,unsigned port)
{
        if(device==pt_SB)
        {
                pt_device=pt_SB;
                if(pt_resetSB(port))
                {
                        pt_port=port;
                        pt_SB_port=port+0xc;
                        return(pt_OK);
                }
                else
                        return(pt_ERROR);
        }

        if(device==pt_GUS)
        {
                pt_device=pt_GUS;
                if(gus_reset(port)==pt_ERROR)
                        return(pt_ERROR);

                return(pt_OK);
        }

        if(device==pt_DAC)
        {
                pt_device=pt_DAC;
                pt_port=port;
                return(pt_OK);
        }

        return(pt_ERROR);
}

int pt_load(char *path)
{
        int     n,
                hand;

        unsigned        l,
                        po,
                        cnt,
                        help,
                        helpseg,
                        helpoff;

        unsigned char   temp[4],
                        calc[4],
                        mk[5],
                        *modhead,
                        *gusblok;

        hand=open(path,O_BINARY|O_RDONLY);
	if(hand==-1)
                return(pt_ERROR);

        lseek(hand,1080,SEEK_SET);
        read(hand,mk,4);
        mk[4]=0;
        if(!stricmp(mk,"M.K.") || !stricmp(mk,"FLT4"))
                pt_inst=31;
        else
        {
                if(!stricmp(mk,"6CHN") || !stricmp(mk,"8CHN") || !stricmp(mk,"FLT8"))
                        return(pt_ERROR);
                else
                        pt_inst=15;
        }

        lseek(hand,0,SEEK_SET);
        read(hand,pt_songname,20);
        modhead=malloc(pt_inst*30);
        read(hand,modhead,pt_inst*30);

        for(n=0,help=0;n<pt_inst;n++,help+=30)
        {
                pt_slen[n]=modhead[help+22];
                pt_slen[n]<<=8;
                pt_slen[n]+=modhead[help+23];
                pt_slen[n]*=2;

                pt_finetune[n]=modhead[help+24];

                pt_svol[n]=modhead[help+25];
                if(pt_svol[n]>63)
                        pt_svol[n]=64;

                pt_roff[n]=modhead[help+26];
                pt_roff[n]<<=8;
                pt_roff[n]+=modhead[help+27];

                pt_rlen[n]=modhead[help+28];
                pt_rlen[n]<<=8;
                pt_rlen[n]+=modhead[help+29];

                if(pt_inst==31)
                {
                        pt_roff[n]*=2;
                        pt_rlen[n]*=2;
                }
        }

        free(modhead);

        read(hand,&pt_modlen,1);
        if(!pt_modlen)
                return(pt_ERROR);
        read(hand,NULL,1); 
        read(hand,pt_pos,128);

        if(pt_inst==31)
                read(hand,NULL,4);

        pt_ptns=pt_pos[0];
        for(n=1;n<128;n++)
                if(pt_pos[n]>pt_ptns)
                        pt_ptns=pt_pos[n];
        pt_ptns++;

        for(n=0;n<pt_ptns;n++)
        {
                pt_patt[n]=malloc(1024);
                if(pt_patt[n]==NULL)
                        return(pt_ERROR);
                read(hand,pt_patt[n],1024);
        }

        if(pt_device==pt_GUS)
        {
                for(n=0;n<16;n++)           /*  This might not be necessary  */
                        gus_pokeb(0,n,0);

                gus_availstart=16;
                gusblok=farmalloc(65534l+16l);
                if(gusblok==NULL)
                        return(pt_ERROR);
                help=FP_SEG(gusblok)+1;

                for(n=0;n<pt_inst;n++)
                {
                        if(pt_slen[n])
                        {
                                read(hand,MK_FP(help,0),pt_slen[n]);

                                gus_transferblock(help,0,gus_availstart>>16u,gus_availstart&0xffff,pt_slen[n]);
                                gus_start[n]=gus_availstart;
                                gus_availstart+=pt_slen[n];
                        }
                        else
                                gus_start[n]=0;

                        if(pt_rlen[n]<3)
                        {
                                pt_loop[n]=0;
                                gus_repeat[n]=gus_start[n];
                                gus_end[n]=gus_start[n]+(unsigned long)pt_slen[n]-1;
                        }
                        else
                        {
                                pt_loop[n]=1;
                                gus_repeat[n]=gus_start[n]+(unsigned long)pt_roff[n]-1;
                                gus_end[n]=gus_repeat[n]+(unsigned long)pt_rlen[n];
                        }
                }

                farfree(gusblok);
        }
        else
        {
                for(n=0;n<pt_inst;n++)
                {
                        pt_smp[n]=farmalloc((long)pt_slen[n]+16l);
                        if(pt_smp[n]==NULL)
                                return(pt_ERROR);

                        if(pt_slen[n])
                        {
                                pt_smp[n]=MK_FP(FP_SEG(pt_smp[n])+1,0);
                                read(hand,pt_smp[n],pt_slen[n]);
                                pt_sseg[n]=FP_SEG(pt_smp[n]);
                        }
                        else
                                pt_slen[n]=1;   /*      It has to be something.. */

                        if(pt_rlen[n]<3)
                                pt_loop[n]=0;
                        else
                        {
                                pt_loop[n]=1;
                                pt_slen[n]=pt_roff[n]+pt_rlen[n];
                        }
                }
        }

        close(hand);

/*      This is the old slow converter (perhaps easier to understand):

        for(n=0;n<pt_ptns;n++)
        {
		for(l=0;l<1024;l+=4)
		{
                        memmove(temp,&pt_patt[n][l],4);

                        calc[3]=temp[3];
                        calc[2]=temp[2] & 0xf;
			calc[0]=(temp[0] & 0xf0) + (temp[2]>>4);

			po=(temp[0] & 0xf);
			po<<=8;
                        po+=temp[1];

                        calc[1]=0;
                        if(po)
                        {
                                for(cnt=0;cnt<36;cnt++)
                                        if(po>=pt_period[cnt])
                                        {
                                                calc[1]=cnt+1;
                                                break;
                                        }
                        }

                        memmove(&pt_patt[n][l],calc,4);
		}
        }       */

        for(n=0;n<pt_ptns;n++)
        {
                helpseg=FP_SEG(pt_patt[n]);
                helpoff=FP_OFF(pt_patt[n]);

                asm     push    es
                asm     push    di
                asm     push    dx

                asm     mov     ax,helpseg
                asm     mov     es,ax
                asm     mov     di,helpoff
                asm     mov     cx,1024/4

                smultron:
                asm     mov     bh,es:[di]
                asm     mov     bl,es:[di+1]
                asm     mov     dh,es:[di+2]

                asm     mov     al,dh
                asm     and     al,0fh
                asm     mov     es:[di+2],al

                asm     mov     ah,dh
                asm     shr     ah,1
                asm     shr     ah,1
                asm     shr     ah,1
                asm     shr     ah,1
                asm     mov     al,bh
                asm     and     al,10h
                asm     add     al,ah
                asm     mov     es:[di],al

                asm     mov     ah,bh
                asm     and     ah,0fh
                asm     mov     al,bl
                asm     test    ax,0ffffh
                asm     jz      movo

                asm     xor     bx,bx

                gertrud:
                asm     cmp     ax,ds:pt_period[bx]
                asm     jae     entti

                asm     add     bx,2
                asm     cmp     bx,36*2
                asm     jne     gertrud

                movo:
                asm     xor     bx,bx
                asm     mov     es:[di+1],bl
                asm     jmp     anataas

                entti:
                asm     shr     bx,1
                asm     inc     bx
                asm     mov     es:[di+1],bl

                anataas:
                asm     add     di,4
                asm     loop    smultron

                asm     pop     dx
                asm     pop     di
                asm     pop     es
        }
        /*      Bytes after the (fast!) conversion:
                1 - Sample number
                2 - Note number
                3 - Special command
                4 - Command parameter   */

        return(pt_OK);
}

int pt_play(unsigned freq)
{
        int     n;

        pt_freq=freq;

        if(pt_device==pt_SB && freq>44000u)
                return(pt_ERROR);

        if(pt_device!=pt_GUS)
        {
                pt_VBmax=pt_oldmax=freq/50u;
                pt_calc_inc();
        }

        if(pt_device==pt_SB)
        {
                while(inp(pt_SB_port) & 0x80)
                        ;
                outp(pt_SB_port,0xd1);
        }

        pt_speed=6;
        pt_counter=5;
        pt_row=63;
        pt_curpos=0;
        pt_index=-1;
        pt_VBcnt=1;
        for(n=0;n<4;n++)
        {
                pt_add_dec[n]=pt_add_off[n]=0;
                pt_on[n]=pt_vol[n]=pt_cvol[n]=0;
                pt_address[n]=0;
                pt_clen[n]=65535u;
                pt_jumpcnt[n]=-1;
                pt_snum[n]=0;
        }

        pt_oldfunc=getvect(0x8);

        disable();

        switch(pt_device)
        {
                case pt_SB  : pt_settimer(pt_freq);
                              pt_SB_handler(); break;
                case pt_DAC : pt_settimer(pt_freq);
                              pt_DAC_handler(); break;
                case pt_GUS : gus_outb(GUS_ACTIVE_VOICES,13);
                              gus_maxfrequency=44100u;
                              gus_outb(GUS_GF1_RESET, 1+2);
                              pt_settimer(50);
                              setvect(0x8,pt_GUS_handler); break;
                default : enable(); return(pt_ERROR);
        }

        enable();
        return(pt_OK);
}

void pt_stop(void)
{
        pt_resettimer();
        setvect(0x8,pt_oldfunc);

        if(pt_device==pt_SB)
                pt_resetSB(pt_port);

        if(pt_device==pt_GUS)
        {
                gus_resetGF1();
                pt_resettimer();
        }

        pt_restoreclock();
}

void pt_pause(void)
{
        char    value;

        value=inp(0x21);
        outp(0x21,value|1);

        if(pt_device==pt_GUS)
                for(value=0;value<4;value++)
                {
                        gus_selectvoice(value);
                        gus_vol(0);
                }
}

void pt_continue(void)
{
        char    value;

        value=inp(0x21);
        outp(0x21,value &0xfe);

        if(pt_device==pt_GUS)
                for(value=0;value<4;value++)
                {
                        gus_selectvoice(value);
                        gus_vol(pt_cvol[value]);
                }
}

void pt_DAC_handler(void)
{        
        asm     push    ds
        asm     push    dx
        asm     push    ax

        asm     mov     dx,seg pt_handst
        asm     mov     ds,dx
        asm     mov     dx,offset pt_handst
        asm     mov     al,8
        asm     mov     ah,025h
        asm     int     021h

        asm     pop     ax
        asm     pop     dx
        asm     pop     ds

        if(pt_curpos!=666)
                return;

check1: asm     mov     ax,0
        pt_checkloop();
        asm     jmp     chan2

check2: asm     mov     ax,1
        pt_checkloop();
        asm     jmp     chan3

check3: asm     mov     ax,2
        pt_checkloop();
        asm     jmp     chan4

check4: asm     mov     ax,3
        pt_checkloop();
        asm     jmp     output_byte

misc:   asm     jmp     newamigablank

temp1:  asm     jmp     check1
temp2:  asm     jmp     check2

asm     pt_handst  proc    far

        asm     push    ax
        asm     push    bx
        asm     push    dx
        asm     push    es
        asm     push    ds

        asm     mov     dx,DGROUP
        asm     mov     ds,dx

        asm     dec     pt_VBcnt
        asm     jz      misc

        startmixing:
        asm     les     bx,dword ptr pt_address
        asm     mov     al,byte ptr es:[bx]

        asm     imul    byte ptr pt_cvol
        asm     mov     dx,ax

        asm     mov     ax,word ptr pt_add_dec
        asm     add     word ptr pt_sdec,ax
        asm     adc     bx,word ptr pt_add_off
        asm     mov     word ptr pt_address,bx

        asm     cmp     bx,pt_clen
        asm     jae     temp1

        chan2:
        asm     les     bx,dword ptr pt_address+4
        asm     mov     al,byte ptr es:[bx]

        asm     imul    byte ptr pt_cvol+2
        asm     add     dx,ax

        asm     mov     ax,word ptr pt_add_dec+2
        asm     add     word ptr pt_sdec+2,ax
        asm     adc     bx,word ptr pt_add_off+2
        asm     mov     word ptr pt_address+4,bx

        asm     cmp     bx,pt_clen+2
        asm     jae     temp2

        chan3:
        asm     les     bx,dword ptr pt_address+8
        asm     mov     al,byte ptr es:[bx]

        asm     imul    byte ptr pt_cvol+4
        asm     add     dx,ax

        asm     mov     ax,word ptr pt_add_dec+4
        asm     add     word ptr pt_sdec+4,ax
        asm     adc     bx,word ptr pt_add_off+4
        asm     mov     word ptr pt_address+8,bx

        asm     cmp     bx,pt_clen+4
        asm     jae     temp3

        chan4:
        asm     les     bx,dword ptr pt_address+12
        asm     mov     al,byte ptr es:[bx]

        asm     imul    byte ptr pt_cvol+6
        asm     add     dx,ax

        asm     mov     ax,word ptr pt_add_dec+6
        asm     add     word ptr pt_sdec+6,ax
        asm     adc     bx,word ptr pt_add_off+6
        asm     mov     word ptr pt_address+12,bx

        asm     cmp     bx,pt_clen+6
        asm     jae     temp4

        output_byte:
        asm     xor     dh,128
        asm     mov     al,dh
        asm     mov     dx,pt_port
        asm     out     dx,al

        asm     mov     al,020h
        asm     out     020h,al

        asm     pop     ds
        asm     pop     es
        asm     pop     dx
        asm     pop     bx
        asm     pop     ax
        asm     iret

temp3:  asm     jmp     check3
temp4:  asm     jmp     check4

        newamigablank:
        asm     push    cx
        asm     push    si
        asm     push    di
        asm     push    bp
	asm     mov     bp,sp

        pt_commands();

	asm     mov     sp,bp
	asm     pop     bp
	asm     pop     di
	asm     pop     si
	asm     pop     cx

        asm     jmp     startmixing

asm     pt_handst  endp
}

void pt_SB_handler(void)
{        
        asm     push    ds
        asm     push    dx
        asm     push    ax

        asm     mov     dx,seg pt_SB_handst
        asm     mov     ds,dx
        asm     mov     dx,offset pt_SB_handst
        asm     mov     al,8
        asm     mov     ah,025h
        asm     int     021h

        asm     pop     ax
        asm     pop     dx
        asm     pop     ds

        if(pt_curpos!=666)
                return;

check1: asm     mov     ax,0
        pt_checkloop();
        asm     jmp     chan2

check2: asm     mov     ax,1
        pt_checkloop();
        asm     jmp     chan3

check3: asm     mov     ax,2
        pt_checkloop();
        asm     jmp     chan4

check4: asm     mov     ax,3
        pt_checkloop();
        asm     jmp     output_byte

misc:   asm     jmp     newamigablank

temp1:  asm     jmp     check1
temp2:  asm     jmp     check2

asm     pt_SB_handst  proc    far

        asm     push    ax
        asm     push    bx
        asm     push    dx
        asm     push    es
        asm     push    ds

        asm     mov     dx,DGROUP
        asm     mov     ds,dx

        asm     mov     dx,pt_SB_port
wait1:  asm     in      al,dx
        asm     rcl     al,1
        asm     jc      wait1
        asm     mov     al,010h
        asm     out     dx,al

        asm     dec     pt_VBcnt
        asm     jz      misc

        startmixing:
        asm     les     bx,dword ptr pt_address
        asm     mov     al,byte ptr es:[bx]

        asm     imul    byte ptr pt_cvol
        asm     mov     dx,ax

        asm     mov     ax,word ptr pt_add_dec
        asm     add     word ptr pt_sdec,ax
        asm     adc     bx,word ptr pt_add_off
        asm     mov     word ptr pt_address,bx

        asm     cmp     bx,pt_clen
        asm     jae     temp1

        chan2:
        asm     les     bx,dword ptr pt_address+4
        asm     mov     al,byte ptr es:[bx]

        asm     imul    byte ptr pt_cvol+2
        asm     add     dx,ax

        asm     mov     ax,word ptr pt_add_dec+2
        asm     add     word ptr pt_sdec+2,ax
        asm     adc     bx,word ptr pt_add_off+2
        asm     mov     word ptr pt_address+4,bx

        asm     cmp     bx,pt_clen+2
        asm     jae     temp2

        chan3:
        asm     les     bx,dword ptr pt_address+8
        asm     mov     al,byte ptr es:[bx]

        asm     imul    byte ptr pt_cvol+4
        asm     add     dx,ax

        asm     mov     ax,word ptr pt_add_dec+4
        asm     add     word ptr pt_sdec+4,ax
        asm     adc     bx,word ptr pt_add_off+4
        asm     mov     word ptr pt_address+8,bx

        asm     cmp     bx,pt_clen+4
        asm     jae     temp3

        chan4:
        asm     les     bx,dword ptr pt_address+12
        asm     mov     al,byte ptr es:[bx]

        asm     imul    byte ptr pt_cvol+6
        asm     add     dx,ax

        asm     mov     ax,word ptr pt_add_dec+6
        asm     add     word ptr pt_sdec+6,ax
        asm     adc     bx,word ptr pt_add_off+6
        asm     mov     word ptr pt_address+12,bx

        asm     cmp     bx,pt_clen+6
        asm     jae     temp4

        output_byte:
        asm     xor     dh,128
        asm     mov     ah,dh
        asm     mov     dx,pt_SB_port

wait2:  asm     in      al,dx
        asm     rcl     al,1
        asm     jc      wait2
        asm     mov     al,ah
        asm     out     dx,al

        asm     mov     al,020h
        asm     out     020h,al

        asm     pop     ds
        asm     pop     es
        asm     pop     dx
        asm     pop     bx
        asm     pop     ax
        asm     iret

temp3:  asm     jmp     check3
temp4:  asm     jmp     check4

        newamigablank:
        asm     push    cx
        asm     push    si
        asm     push    di
        asm     push    bp
	asm     mov     bp,sp

        pt_commands();

	asm     mov     sp,bp
	asm     pop     bp
	asm     pop     di
	asm     pop     si
	asm     pop     cx

        asm     jmp     startmixing

asm     pt_SB_handst  endp
}

void pt_commands(void)
{
        int     n;

        pt_VBcnt=pt_VBmax;

        pt_counter++;
        if(pt_counter>=pt_speed)
        {
                pt_counter=0;

                pt_row++;
                if(pt_row>63)
                {
                        pt_row-=64;
                        pt_index++;
                        if(pt_index==pt_modlen)
                                pt_index=0;

                        pt_curpos=pt_pos[pt_index];
                }
                pt_off=pt_row<<4;
        }

        if(pt_pattdelay && !pt_counter)
        {
                pt_speed=pt_save_speed;
                pt_pattdelay=0;
        }

        for(n=0;n<4;n++,pt_off+=4)
        {
                pt_cmd=pt_patt[pt_curpos][pt_off+2];
                pt_prm=pt_patt[pt_curpos][pt_off+3];

                if(!pt_counter)
                {
                        if(pt_cmd==0xe && (pt_prm>>4)==0xd)    /*  Notedelay */
                        {
                                if(pt_patt[pt_curpos][pt_off])
                                        pt_vol[n]=pt_svol[pt_patt[pt_curpos][pt_off]-1];
                        }
                        else
                                pt_retrig(n);

                        if(pt_cmd|pt_prm)
                        {
                                switch(pt_cmd)
                                {
                                        case 3  : if(pt_prm)
                                                        pt_tonespd[n]=pt_prm;
                                                  break;

                                        case 4  : if(pt_prm>>4)
                                                        pt_vib_spd[n]=pt_prm>>4;
                                                  if(pt_prm&0xf)
                                                        pt_vib_size[n]=pt_prm&0xf;
                                                  break;

                                        case 7  : if(pt_prm>>4)
                                                        pt_tre_spd[n]=pt_prm>>4;
                                                  if(pt_prm&0xf)
                                                        pt_tre_size[n]=pt_prm&0xf;
                                                  break;

                                        case 9  : pt_address[n]&=0xffff0000;
                                                  pt_address[n]+=((unsigned)pt_prm)<<8u;
                                                  break;

                                        case 0xb: pt_row=63;
                                                  pt_index=pt_prm-1;
                                                  break;

                                        case 0xc: pt_vol[n]=pt_prm;
                                                  break;

                                        case 0xd: pt_row=63+(pt_prm&0xf)+(pt_prm>>4)*10;
                                                  break;

                                        case 0xe: switch(pt_prm>>4)
                                                  {
                                                        case 1: pt_curper[n]-=pt_prm&0xf;
                                                                pt_add_off[n]=pt_per_off[pt_curper[n]];
                                                                pt_add_dec[n]=pt_per_dec[pt_curper[n]];
                                                                break;
                                                        case 2: pt_curper[n]+=pt_prm&0xf;
                                                                pt_add_off[n]=pt_per_off[pt_curper[n]];
                                                                pt_add_dec[n]=pt_per_dec[pt_curper[n]];
                                                                break;
                                                        case 6: if(pt_prm&0xf)
                                                                {
                                                                    if(pt_jumpcnt[n]==-1)
                                                                    {
                                                                        pt_jumpcnt[n]=(pt_prm&0xf)-1;
                                                                        pt_row=pt_jumpdest[n]-1;
                                                                    }
                                                                    else
                                                                        if(pt_jumpcnt[n])
                                                                        {
                                                                            pt_jumpcnt[n]--;
                                                                            pt_row=pt_jumpdest[n]-1;
                                                                        }
                                                                        else
                                                                            pt_jumpcnt[n]=-1;
                                                                }
                                                                else
                                                                        pt_jumpdest[n]=pt_row;
                                                                break;
                                                        case 0xb: pt_vol[n]-=pt_prm&0xf;
                                                                  if(pt_vol[n]<0)
                                                                        pt_vol[n]=0;
                                                                  break;
                                                        case 0xa: pt_vol[n]+=pt_prm&0xf;
                                                                  if(pt_vol[n]>64)
                                                                        pt_vol[n]=64;
                                                                  break;
                                                        case 0xe: pt_save_speed=pt_speed;
                                                                  pt_speed=pt_speed*((int)(pt_prm&0xf)+1);
                                                                  pt_pattdelay=1;
                                                                  break;
                                                  }
                                                  break;

                                        case 0xf: if(pt_prm<0x20)
                                                        pt_speed=pt_prm;
                                                  else
                                                  {
                                                        pt_VBmax=((long)pt_oldmax)*125l/(long)pt_prm;
                                                        pt_VBcnt=pt_VBmax;
                                                  }
                                                  break;
                                }
                        }
                }

                if((pt_cmd|pt_prm) && pt_counter)
                {
                        recurse:
                        switch(pt_cmd)
                        {
                                case 0  : switch(pt_counter%3)
                                          {
                                                case 0: pt_temp=pt_curper[n];
                                                        break;
                                                case 1: pt_temp=pt_note[n]+36*pt_finetune[pt_snum[n]]+(pt_prm>>4);
                                                        pt_temp=pt_period[pt_temp];
                                                        break;
                                                case 2: pt_temp=pt_note[n]+36*pt_finetune[pt_snum[n]]+(pt_prm&0xf);
                                                        pt_temp=pt_period[pt_temp];
                                                        break;
                                          }
                                          pt_add_off[n]=pt_per_off[pt_temp];
                                          pt_add_dec[n]=pt_per_dec[pt_temp];
                                          break;

                                case 1  : pt_curper[n]-=pt_prm;
                                          if(pt_curper[n]<113)
                                                pt_curper[n]=113;
                                          pt_add_off[n]=pt_per_off[pt_curper[n]];
                                          pt_add_dec[n]=pt_per_dec[pt_curper[n]];
                                          break;

                                case 2  : pt_curper[n]+=pt_prm;
                                          if(pt_curper[n]>856)
                                                pt_curper[n]=856;
                                          pt_add_off[n]=pt_per_off[pt_curper[n]];
                                          pt_add_dec[n]=pt_per_dec[pt_curper[n]];
                                          break;

                                case 3  : if(pt_curper[n]<pt_tonedest[n])
                                          {
                                                pt_curper[n]+=pt_tonespd[n];
                                                if(pt_curper[n]>=pt_tonedest[n])
                                                        pt_curper[n]=pt_tonedest[n];
                                          }
                                          else
                                          {
                                                pt_curper[n]-=pt_tonespd[n];
                                                if(pt_curper[n]<=pt_tonedest[n])
                                                        pt_curper[n]=pt_tonedest[n];
                                          }
                                          pt_add_off[n]=pt_per_off[pt_curper[n]];
                                          pt_add_dec[n]=pt_per_dec[pt_curper[n]];
                                          break;

                                case 4  : pt_itemp=pt_vib_table[pt_vib_off[n]]*(int)(pt_vib_size[n]);
                                          pt_itemp/=128;

                                          pt_temp=pt_curper[n]+pt_itemp;
                                          pt_add_off[n]=pt_per_off[pt_temp];
                                          pt_add_dec[n]=pt_per_dec[pt_temp];

                                          pt_vib_off[n]+=pt_vib_spd[n];
                                          pt_vib_off[n]&=63;
                                          break;

                                case 5  : if(pt_curper[n]<pt_tonedest[n])
                                          {
                                                pt_curper[n]+=pt_tonespd[n];
                                                if(pt_curper[n]>pt_tonedest[n])
                                                        pt_curper[n]=pt_tonedest[n];
                                          }
                                          else
                                          {
                                                pt_curper[n]-=pt_tonespd[n];
                                                if(pt_curper[n]<pt_tonedest[n])
                                                        pt_curper[n]=pt_tonedest[n];
                                          }
                                          pt_add_off[n]=pt_per_off[pt_curper[n]];
                                          pt_add_dec[n]=pt_per_dec[pt_curper[n]];
                                          pt_cmd=0xa;
                                          goto recurse;

                                case 6  : pt_vib_off[n]+=pt_vib_spd[n];
                                          pt_vib_off[n]&=63;
                                          pt_itemp=pt_vib_table[pt_vib_off[n]]*(int)(pt_vib_size[n]);
                                          pt_itemp/=128;

                                          pt_temp=pt_curper[n]+pt_itemp;
                                          pt_add_off[n]=pt_per_off[pt_temp];
                                          pt_add_dec[n]=pt_per_dec[pt_temp];
                                          pt_cmd=0xa;
                                          goto recurse;

                                case 7  : pt_tre_off[n]+=pt_tre_spd[n];
                                          pt_tre_off[n]&=63;
                                          pt_itemp=pt_vib_table[pt_tre_off[n]]*(int)(pt_tre_size[n]);
                                          pt_itemp/=64;

                                          pt_vol[n]=pt_svol[pt_snum[n]]+pt_itemp;
                                          if(pt_vol[n]>64)
                                                pt_vol[n]=64;
                                          if(pt_vol[n]<0)
                                                pt_vol[n]=0;
                                          break;

                                case 0xa: if(pt_prm&0xf && !(pt_prm>>4))
                                          {
                                                pt_vol[n]-=pt_prm&0xf;
                                                if(pt_vol[n]<0)
                                                        pt_vol[n]=0;
                                          }
                                          if(pt_prm>>4)
                                          {
                                                pt_vol[n]+=pt_prm>>4;
                                                if(pt_vol[n]>64)
                                                        pt_vol[n]=64;
                                          }
                                          break;
                                case 0xe: switch(pt_prm>>4)
                                          {
                                                case 9  : if(!(pt_counter%(pt_prm&0xf)))
                                                                pt_retrig(n);
                                                          break;
                                                case 0xc: if(pt_counter>=(pt_prm&0xf))
                                                                pt_vol[n]=0;
                                                          break;
                                                case 0xd: if(pt_counter==(pt_prm&0xf))
                                                                pt_retrig(n);
                                                          break;
                                          }
                                          break;
                        }
                }
        }

        for(n=0;n<4;n++)
        {
                if(pt_on[n])
                        pt_cvol[n]=(pt_mainvol*pt_vol[n])/64;
                else
                        pt_cvol[n]=0;
        }
        pt_off-=16;
}

void pt_retrig(int n)
{
        if(pt_patt[pt_curpos][pt_off])
        {
                pt_next[n]=pt_patt[pt_curpos][pt_off]-1;
                pt_vol[n]=pt_svol[pt_next[n]];
        }

        if(pt_patt[pt_curpos][pt_off+1])        /*      New note        */
        {
                pt_on[n]=1;

                if(pt_cmd!=0x3 && pt_cmd!=5)    /*      No ToneP        */
                {
                        pt_tre_off[n]=pt_vib_off[n]=0;

                        pt_note[n]=pt_patt[pt_curpos][pt_off+1]-1;
                        pt_sdec[n]=0;

                        asm     mov     bx,n
                        asm     shl     bx,1
                        asm     shl     bx,1
                        asm     mov word ptr pt_address[bx],0

                        pt_snum[n]=pt_next[n];
                        pt_temp=pt_note[n]+36*pt_finetune[pt_snum[n]];
                        pt_curper[n]=pt_period[pt_temp];
                        pt_clen[n]=pt_slen[pt_snum[n]];

                        asm     mov     bx,n
                        asm     shl     bx,1
                        asm     mov     ax,pt_snum[bx]
                        asm     mov     bx,ax
                        asm     shl     bx,1
                        asm     mov     ax,pt_sseg[bx]
                        asm     mov     bx,n
                        asm     shl     bx,1
                        asm     shl     bx,1
                        asm     mov word ptr pt_address[bx+2],ax
                        
                        pt_add_off[n]=pt_per_off[pt_curper[n]];
                        pt_add_dec[n]=pt_per_dec[pt_curper[n]];
                }
                else
                {
                        if(pt_patt[pt_curpos][pt_off])
                                pt_snum[n]=pt_next[n];
                }

                pt_temp=pt_patt[pt_curpos][pt_off+1]-1; /*      New Dest     */
                pt_tonedest[n]=pt_period[pt_temp];
        }
}

void pt_checkloop(void)
{
        asm     push    ax
        asm     mov     bx,ax
        asm     shl     bx,1
        asm     mov     ax,pt_snum[bx]
        asm     mov     bx,ax
        asm     cmp byte ptr pt_loop[bx],0
        asm     jne     yes_loop

        asm     pop     bx
        asm     mov byte ptr pt_on[bx],0
        asm     shl     bx,1
        asm     mov word ptr pt_cvol[bx],0
        asm     mov word ptr pt_add_dec[bx],0
        asm     mov word ptr pt_add_off[bx],0
        asm     shl     bx,1
        asm     mov word ptr pt_address[bx],0
        asm     jmp     get_out
        
        yes_loop:
        asm     shl     bx,1
        asm     mov     ax,pt_roff[bx]
        asm     pop     bx
        asm     shl     bx,1
        asm     shl     bx,1
        asm     mov word ptr pt_address[bx],ax

        get_out:
                ;

/*      if(!pt_loop[pt_snum[num]])
        {
                pt_address[num]&=0xffff0000;
                pt_cvol[num]=0;
                pt_on[num]=0;
                pt_add_dec[num]=0;
                pt_add_off[num]=0;
        }
        else
        {
                pt_address[num]&=0xffff0000;
                pt_address[num]|=pt_roff[pt_snum[num]];
        }       */
}

void interrupt pt_GUS_handler(void)
{
        int     n;

        pt_counter++;
        if(pt_counter>=pt_speed)
        {
                pt_counter=0;

                pt_row++;
                if(pt_row>63)
                {
                        pt_row-=64;
                        pt_index++;
                        if(pt_index==pt_modlen)
                                pt_index=0;

                        pt_curpos=pt_pos[pt_index];
                }
                pt_off=pt_row<<4;
        }

        if(pt_pattdelay && !pt_counter)
        {
                pt_speed=pt_save_speed;
                pt_pattdelay=0;
        }

        for(n=0;n<4;n++,pt_off+=4)
        {
                gus_selectvoice(n);

                pt_cmd=pt_patt[pt_curpos][pt_off+2];
                pt_prm=pt_patt[pt_curpos][pt_off+3];

                if(!pt_counter)
                {
                        if(pt_cmd==0xe && (pt_prm>>4)==0xd)    /*  Notedelay */
                        {
                                if(pt_patt[pt_curpos][pt_off])
                                        pt_vol[n]=pt_svol[pt_patt[pt_curpos][pt_off]-1];
                        }
                        else
                                gus_retrig(n);

                        if(pt_cmd|pt_prm)
                        {
                                switch(pt_cmd)
                                {
                                        case 3  : if(pt_prm)
                                                        pt_tonespd[n]=pt_prm;
                                                  break;

                                        case 4  : if(pt_prm>>4)
                                                        pt_vib_spd[n]=pt_prm>>4;
                                                  if(pt_prm&0xf)
                                                        pt_vib_size[n]=pt_prm&0xf;
                                                  break;

                                        case 7  : if(pt_prm>>4)
                                                        pt_tre_spd[n]=pt_prm>>4;
                                                  if(pt_prm&0xf)
                                                        pt_tre_size[n]=pt_prm&0xf;
                                                  break;

                                        case 9  : if(pt_loop[pt_snum[n]])
                                                  {
                                                        gus_setlocation(GUS_CURRENT,gus_start[pt_snum[n]]+((unsigned)pt_prm<<8u));
                                                        gus_outb(0,8);
                                                  }
                                                  else
                                                  {
                                                        gus_setlocation(GUS_CURRENT,gus_start[pt_snum[n]]+((unsigned)pt_prm<<8u));
                                                        gus_outb(0,0);
                                                  }
                                                  break;

                                        case 0xb: pt_row=63;
                                                  pt_index=pt_prm-1;
                                                  break;

                                        case 0xc: pt_vol[n]=pt_prm;
                                                  break;

                                        case 0xd: pt_row=63+(pt_prm&0xf)+(pt_prm>>4)*10;
                                                  break;

                                        case 0xe: switch(pt_prm>>4)
                                                  {
                                                        case 1: pt_curper[n]-=pt_prm&0xf;
                                                                gus_setperiod(pt_curper[n]);
                                                                break;
                                                        case 2: pt_curper[n]+=pt_prm&0xf;
                                                                gus_setperiod(pt_curper[n]);
                                                                break;
                                                        case 6: if(pt_prm&0xf)
                                                                {
                                                                    if(pt_jumpcnt[n]==-1)
                                                                    {
                                                                        pt_jumpcnt[n]=(pt_prm&0xf)-1;
                                                                        pt_row=pt_jumpdest[n]-1;
                                                                    }
                                                                    else
                                                                        if(pt_jumpcnt[n])
                                                                        {
                                                                            pt_jumpcnt[n]--;
                                                                            pt_row=pt_jumpdest[n]-1;
                                                                        }
                                                                        else
                                                                            pt_jumpcnt[n]=-1;
                                                                }
                                                                else
                                                                        pt_jumpdest[n]=pt_row;
                                                                break;
                                                        case 0xb: pt_vol[n]-=pt_prm&0xf;
                                                                  if(pt_vol[n]<0)
                                                                        pt_vol[n]=0;
                                                                  break;
                                                        case 0xa: pt_vol[n]+=pt_prm&0xf;
                                                                  if(pt_vol[n]>64)
                                                                        pt_vol[n]=64;
                                                                  break;
                                                        case 0xe: pt_save_speed=pt_speed;
                                                                  pt_speed=pt_speed*((int)(pt_prm&0xf)+1);
                                                                  pt_pattdelay=1;
                                                                  break;
                                                  }
                                                  break;

                                        case 0xf: if(pt_prm<0x20)
                                                        pt_speed=pt_prm;
                                                  else
                                                  {
                                                        pt_settimer((50*(int)pt_prm)/125);
                                                  }
                                                  break;
                                }
                        }
                }

                if((pt_cmd|pt_prm) && pt_counter)
                {
                        recurse:
                        switch(pt_cmd)
                        {
                                case 0  : switch(pt_counter%3)
                                          {
                                                case 0: pt_temp=pt_curper[n];
                                                        break;
                                                case 1: pt_temp=pt_note[n]+36*pt_finetune[pt_snum[n]]+(pt_prm>>4);
                                                        pt_temp=pt_period[pt_temp];
                                                        break;
                                                case 2: pt_temp=pt_note[n]+36*pt_finetune[pt_snum[n]]+(pt_prm&0xf);
                                                        pt_temp=pt_period[pt_temp];
                                                        break;
                                          }
                                          gus_setperiod(pt_temp);
                                          break;

                                case 1  : pt_curper[n]-=pt_prm;
                                          if(pt_curper[n]<113)
                                                pt_curper[n]=113;
                                          gus_setperiod(pt_curper[n]);
                                          break;

                                case 2  : pt_curper[n]+=pt_prm;
                                          if(pt_curper[n]>856)
                                                pt_curper[n]=856;
                                          gus_setperiod(pt_curper[n]);
                                          break;

                                case 3  : if(pt_curper[n]<pt_tonedest[n])
                                          {
                                                pt_curper[n]+=pt_tonespd[n];
                                                if(pt_curper[n]>=pt_tonedest[n])
                                                        pt_curper[n]=pt_tonedest[n];
                                          }
                                          else
                                          {
                                                pt_curper[n]-=pt_tonespd[n];
                                                if(pt_curper[n]<=pt_tonedest[n])
                                                        pt_curper[n]=pt_tonedest[n];
                                          }
                                          gus_setperiod(pt_curper[n]);
                                          break;

                                case 4  : pt_itemp=pt_vib_table[pt_vib_off[n]]*(int)(pt_vib_size[n]);
                                          pt_itemp/=128;

                                          pt_temp=pt_curper[n]+pt_itemp;
                                          gus_setperiod(pt_temp);

                                          pt_vib_off[n]+=pt_vib_spd[n];
                                          pt_vib_off[n]&=63;
                                          break;

                                case 5  : if(pt_curper[n]<pt_tonedest[n])
                                          {
                                                pt_curper[n]+=pt_tonespd[n];
                                                if(pt_curper[n]>pt_tonedest[n])
                                                        pt_curper[n]=pt_tonedest[n];
                                          }
                                          else
                                          {
                                                pt_curper[n]-=pt_tonespd[n];
                                                if(pt_curper[n]<pt_tonedest[n])
                                                        pt_curper[n]=pt_tonedest[n];
                                          }
                                          gus_setperiod(pt_curper[n]);
                                          pt_cmd=0xa;
                                          goto recurse;

                                case 6  : pt_vib_off[n]+=pt_vib_spd[n];
                                          pt_vib_off[n]&=63;
                                          pt_itemp=pt_vib_table[pt_vib_off[n]]*(int)(pt_vib_size[n]);
                                          pt_itemp/=128;

                                          pt_temp=pt_curper[n]+pt_itemp;
                                          gus_setperiod(pt_temp);
                                          pt_cmd=0xa;
                                          goto recurse;

                                case 7  : pt_tre_off[n]+=pt_tre_spd[n];
                                          pt_tre_off[n]&=63;
                                          pt_itemp=pt_vib_table[pt_tre_off[n]]*(int)(pt_tre_size[n]);
                                          pt_itemp/=64;

                                          pt_vol[n]=pt_svol[pt_snum[n]]+pt_itemp;
                                          if(pt_vol[n]>64)
                                                pt_vol[n]=64;
                                          if(pt_vol[n]<0)
                                                pt_vol[n]=0;
                                          break;

                                case 0xa: if(pt_prm&0xf && !(pt_prm>>4))
                                          {
                                                pt_vol[n]-=pt_prm&0xf;
                                                if(pt_vol[n]<0)
                                                        pt_vol[n]=0;
                                          }
                                          if(pt_prm>>4)
                                          {
                                                pt_vol[n]+=pt_prm>>4;
                                                if(pt_vol[n]>64)
                                                        pt_vol[n]=64;
                                          }
                                          break;
                                case 0xe: switch(pt_prm>>4)
                                          {
                                                case 9  : if(!(pt_counter%(pt_prm&0xf)))
                                                                gus_retrig(n);
                                                          break;
                                                case 0xc: if(pt_counter>=(pt_prm&0xf))
                                                                pt_vol[n]=0;
                                                          break;
                                                case 0xd: if(pt_counter==(pt_prm&0xf))
                                                                gus_retrig(n);
                                                          break;
                                          }
                                          break;
                        }
                }
        }

        for(n=0;n<4;n++)
        {
                gus_selectvoice(n);
                if(pt_on[n])
                        pt_cvol[n]=(pt_mainvol*pt_vol[n])/64;
                else
                        pt_cvol[n]=0;
                gus_vol(pt_cvol[n]);
        }

        pt_off-=16;
        outp(0x20,0x20);
}

void gus_retrig(int n)
{
        if(pt_patt[pt_curpos][pt_off])
        {
                pt_next[n]=pt_patt[pt_curpos][pt_off]-1;
                pt_vol[n]=pt_svol[pt_next[n]];
        }

        if(pt_patt[pt_curpos][pt_off+1])        /*      New note        */
        {
                pt_on[n]=1;

                if(pt_cmd!=0x3 && pt_cmd!=5)    /*      No ToneP        */
                {
                        pt_tre_off[n]=pt_vib_off[n]=0;

                        pt_note[n]=pt_patt[pt_curpos][pt_off+1]-1;
                        pt_sdec[n]=0;

                        pt_snum[n]=pt_next[n];
                        pt_temp=pt_note[n]+36*pt_finetune[pt_snum[n]];
                        pt_curper[n]=pt_period[pt_temp];

                        gus_outb(0,2);
                        gus_setperiod(pt_curper[n]);
                        gus_outb(0xc,gus_pan[n]);

                        if(pt_loop[pt_snum[n]])
                        {
                                gus_setlocation(GUS_START,gus_repeat[pt_snum[n]]);
                                gus_setlocation(GUS_END,gus_end[pt_snum[n]]);
                                gus_setlocation(GUS_CURRENT,gus_start[pt_snum[n]]);
                                gus_outb(0,8);
                        }
                        else
                        {
                                gus_setlocation(GUS_START,gus_start[pt_snum[n]]);
                                gus_setlocation(GUS_END,gus_end[pt_snum[n]]);
                                gus_setlocation(GUS_CURRENT,gus_start[pt_snum[n]]);
                                gus_outb(0,0);
                        }
                }
                else
                {
                        if(pt_patt[pt_curpos][pt_off])
                                pt_snum[n]=pt_next[n];
                }

                pt_temp=pt_patt[pt_curpos][pt_off+1]-1; /*      New Dest     */
                pt_tonedest[n]=pt_period[pt_temp];
        }
}

void pt_calc_inc(void)
{
        unsigned        prd;

        unsigned long   trap;

        for(prd=1;prd<1024;prd++)       /*      Precalculate values     */
        {                               /*      for each period         */
                trap=pt_MASTER_CLK/prd;
                trap<<=16;
                trap/=pt_freq;
                
                pt_per_off[prd]=trap>>16;
                pt_per_dec[prd]=trap;
        }
}

void pt_settimer(unsigned freq)
{
	unsigned	help;

        help=(unsigned)(1193182l/(long)freq);

        outp(0x40,help & 0xff);
        outp(0x40,help >> 8);
        outp(0x43,0);
}

void pt_resettimer(void)
{
        outp(0x40,0);
        outp(0x40,0);
        outp(0x43,0);
}

void pt_restoreclock(void)
{
	struct REGPACK	reg1,reg2;

	reg1.r_ax=0x0200;
	intr(0x1a,&reg1);

        reg1.r_cx=(pt_bcd2d(reg1.r_cx>>8)<<8)+pt_bcd2d(reg1.r_cx&0xff);
        reg1.r_dx=pt_bcd2d(reg1.r_dx>>8)<<8;
	reg2.r_ax=0x0400;
	intr(0x1a,&reg2);

        reg2.r_cx=pt_bcd2d(reg2.r_cx>>8)*100+pt_bcd2d(reg2.r_cx&0xff);
        reg2.r_dx=(pt_bcd2d(reg2.r_dx>>8)<<8)+pt_bcd2d(reg2.r_dx&0xff);
	reg2.r_ax=0x2b00;
	intr(0x21,&reg2);

	reg1.r_ax=0x2d00;
	intr(0x21,&reg1);
}

unsigned pt_bcd2d(unsigned bcd)
{
        unsigned help;

        help=bcd/16;
        help=bcd-16*help+help*10;

        return(help);
}

int pt_resetSB(int base)
{
        unsigned        tries=100;      /*      Try 100 times before exit */

        outp(base+6,1);
        delay(1);
        outp(base+6,0);
        delay(1);

        /*      SB will return AAh if it is reset                       */
        while(inp(base+0xa)!=0xaa && tries)
	{
                outp(base+6,1);
                delay(1);
                outp(base+6,0);
                delay(1);
		tries--;
	}

        if(!tries)
                return(0);              /*      SB Not found            */
        else
                return(1);              /*      SB found                */
}

int gus_reset(int portti)
{
        int     n;

        gus_baseaddr=portti;
	gus_resetGF1();
	gus_pokeb(0,0,66);
	if(gus_peekb(0,0)!=66)
                return(pt_ERROR);

        gus_meminstalled=65536l*4;
	gus_pokeb(4,0,0xaa);
        if(gus_peekb(4,0)==0xaa)
                gus_meminstalled=65536l*8;
	gus_pokeb(8,0,0xaa);
        if(gus_peekb(8,0)==0xaa)
                gus_meminstalled=65536l*16;

        for(n=0;n<32;n++)
        {
                gus_selectvoice(n);
                gus_outb(0,3);
        }

        return(pt_OK);
}

void gus_delay(void)
{
        asm     mov     al, 0x34
        asm     out     0x43, al
        asm     sub     al, 128
        asm     out     0x40, al
        asm     out     0x40, al

        asm     sub     al, al
        asm     out     0x43, al
        asm     in      al, 0x40
        asm     mov     bl, al
        asm     in      al, 0x40
        asm     mov     bh, al
        asm     sub     bx, 6
again:  asm     sub     al, al
        asm     out     0x43, al
        asm     in      al, 0x40
        asm     mov     cl, al
        asm     in      al, 0x40
        asm     mov     ch, al
        asm     cmp     cx, bx
        asm     ja      again
        ;
}

void gus_resetGF1(void)
{
        int     n;

        gus_outb(GUS_GF1_RESET,0);
        outp(gus_baseaddr,9);

        for(n=0;n<3000;n++);
                gus_delay();

        gus_outb(GUS_GF1_RESET,7);

        for(n=0;n<32;n++)
        {
                gus_selectvoice(n);
                gus_outb(0,3);
                gus_outb(0xd,3);
                gus_delay();
                gus_outb(0,3);
                gus_outb(0xd,3);
        }
}

void gus_pokeb(unsigned addr_hi,unsigned addr_lo,unsigned char data)
{
        outp(gus_baseaddr+0x103,0x43);
        outport(gus_baseaddr+0x104,addr_lo);
        outp(gus_baseaddr+0x103,0x44);
        outp(gus_baseaddr+0x105,addr_hi);
	outp(gus_baseaddr+0x107,data);
}

unsigned char gus_peekb(unsigned addr_hi,unsigned addr_lo)
{
        outp(gus_baseaddr+0x103,0x43);
        outport(gus_baseaddr+0x104,addr_lo);
        outp(gus_baseaddr+0x103,0x44);
        outp(gus_baseaddr+0x105,addr_hi);
        return(inp(gus_baseaddr+0x107));
}

void gus_outb(int reg,unsigned char data)
{
        outp(gus_baseaddr+0x103,reg);
        outp(gus_baseaddr+0x105,data);
}

void gus_outw(int reg,unsigned data)
{
        outp(gus_baseaddr+0x103,reg);
        outport(gus_baseaddr+0x104,data);
}

void gus_transferblock(unsigned seg0,unsigned ofs0,unsigned adhi,unsigned adlo,unsigned tsize)
{
        asm     push    ds
        asm     pushf
        asm     cli
        asm     mov     dx, gus_baseaddr
        asm     mov     ds, seg0
        asm     mov     si, ofs0
        asm     mov     bx, adhi
        asm     mov     di, adlo
        asm     cld
        asm     mov     cx, tsize
        asm     add     dx, 0x103
tdlp:   asm     mov     al, 0x43
        asm     out     dx, al
        asm     inc     dx
        asm     mov     ax, di
        asm     out     dx, ax
        asm     dec     dx
        asm     mov     al, 0x44
        asm     out     dx, al
        asm     add     dx, 2
        asm     mov     al, bl
        asm     out     dx, al
        asm     add     dx, 2
        asm     lodsb
/*        asm     add     al, 128       */
        asm     out     dx, al
        asm     add     di, 1
        asm     adc     bl, 0
        asm     sub     dx, 4
        asm     loop    tdlp
        asm     popf
        asm     pop     ds

        ;
}

void gus_selectvoice(unsigned chan)
{
        outp(gus_baseaddr+0x102,chan);
}

void gus_setlocation(unsigned char l_type,long loca)
{
        gus_outw (l_type,(unsigned)(loca>>7));
        gus_outw (l_type+1,(unsigned)(loca<<9));
}

void gus_setperiod(unsigned per)
{
        if(!per)
                return;
        gus_outw(1,((long)(pt_MASTER_CLK/gus_maxfrequency)<<9l)/per<<1);
}

void gus_vol(unsigned vol)
{
	unsigned        erk=0xaa00;

        if(vol<=15)
                erk+=vol*0x230;
        else
                if(vol<=31)
                        erk+=0x2300+(vol-0x10)*0x110;
                else
                        if(vol<=47)
                                erk+=0x3400+(vol-0x20)*0x80;
                        else
                                if(vol<=58)
                                        erk+=0x3c00+(vol-0x30)*0x60;
                                else
                                        erk+=0x3fc0+(vol-58)*0x48;

        if(vol)
                gus_outw(9,erk);
        else
                gus_outw(9,0);
}

/*** End of File ***/
