/*
 * xpal.c - higher level palette manipulation routines
 *
 * DESCRIPTION
 * Includes routines to black-out, fade in and out, and rotate the palette.
 * The actual palette setting routines x_set_palette and x_set_part_palette are
 * in xpalette.s
 *
 * USAGE
 *
 * NOTES
 * Based on code from VGL 2.0 by Mark Morley.
 * To use with VR timing module, modify palette setting routines to not
 * wait for the VR before writing.
 *
 * REVISION HISTORY
 * Date         Reason
 * 27 Jun 95    Initial Release
 *
 */

#include "defines.h"
#include "xlib.h"

static BYTE Pal[768];
int XPaletteLevel = 32;    /* range = 0-32, set by fading routines */
int XPaletteStep = 1;      /* number of levels to change with each step */

/***************************************************************************
 Sets the active palette to all black.  Useful for drawing a screen and then
 fading in later.
***************************************************************************/
void x_black_palette()
{
   int i;
   for (i=0;i<768;i++)
     Pal[i]=0;
   x_set_palette( Pal );
   XPaletteLevel = 0;
}

/***************************************************************************
 Fades in from black to the specified palette.
***************************************************************************/
void x_fade_in( BYTE * palette )
{
   int i, j;

   for( i = 0; i < 32; i+= XPaletteStep )
   {
      for( j = 0; j < 768; j++ )
	 Pal[j] =  (i * (unsigned)palette[j]) >> 5;
      x_set_palette( Pal );
   }
   XPaletteLevel = 32;
}

/***************************************************************************
 Fades to black from a specified palette.
***************************************************************************/
void x_fade_out( BYTE * palette )
{
   int i, j;

   for( i = 31; i >= 0; i -= XPaletteStep)
   {
      for( j = 0; j < 768; j++ )
	 Pal[j] =  (i * (unsigned)palette[j]) >> 5;
      x_set_palette( Pal );
   }
   XPaletteLevel = 0;
}

/***************************************************************************
 Fades in only a part of the palette.
***************************************************************************/
void x_part_fade_in( int first, int num, BYTE * palette )
{
   int i, j, v;
   
   v = first * 3;
   for( i = 0; i < 32; i += XPaletteStep )
   {
      for( j = 0; j < (num * 3); j++ )
	 Pal[j] = (i * (int)palette[j+v]) >> 5;
      x_set_part_palette( first, num, Pal );
   }
   XPaletteLevel = 32;
}

/***************************************************************************
 Fades out only a part of the palette.
***************************************************************************/
void x_part_fade_out( int first, int num, BYTE * palette )
{
   int i, j, v;
   
   v = first * 3;
   for( i = 31; i >= 0; i -= XPaletteStep )
   {
      for( j = 0; j < (num * 3); j++ )
	 Pal[j] = (i * (int)palette[j+v]) >> 5;
      x_set_part_palette( first, num, Pal );
   }
   XPaletteLevel = 0;
}

/***************************************************************************
 Cycles 'num' palette entries, starting at the 'first' entry.  Entries are
 cycled to the "left".
***************************************************************************/
void x_part_cycleL( int first, int num, BYTE * palette )
{
   int i, v;
   BYTE a, b, c;

   v = first * 3;
   a = palette[0+v];
   b = palette[1+v];
   c = palette[2+v];
   for( i = 0; i < (num-1) * 3; i++ )
      palette[i+v] = palette[i + 3+v];
   palette[i+v] = a;
   palette[i + 1+v] = b;
   palette[i + 2+v] = c;
   x_set_part_palette( first, num, &palette[v] );
}

/***************************************************************************
 Cycles 'num' palette entries, starting at the 'first' entry.  Entries are
 cycled to the "right".
***************************************************************************/
void x_part_cycleR( int first, int num, BYTE * palette )
{
   int i, v;
   BYTE a, b, c;

   i = (num * 3) - 1;
   v = first * 3;
   a = palette[i - 2+v];
   b = palette[i - 1+v];
   c = palette[i+v];
   for( ; i >= 3; i-- )
      palette[i+v] = palette[i - 3+v];
   palette[0+v] = a;
   palette[1+v] = b;
   palette[2+v] = c;
   /*x_set_palette( palette );*/
   x_set_part_palette( first, num, &palette[v] );
}


/* NoVR functions - the following all perform the same functions
** except that none wait for vertical retrace.  For use with VR timing
** module, will produce bad results if used alone.
*/

/***************************************************************************
 Fades in from black to the specified palette.
***************************************************************************/
void x_Fade_in_noVR( BYTE * palette )
{
  int i, j;

  for( i = 0; i < 32; i++ )
  {
     for( j = 0; j < 768; j += XPaletteStep )
       Pal[j] =  (i * (unsigned)palette[j]) >> 5;
     x_set_palette_noVR( Pal );
  }
  XPaletteLevel = 32;
}

/***************************************************************************
 Fades to black from a specified palette.
***************************************************************************/
void x_fade_out_noVR( BYTE * palette )
{
   int i, j;

   for( i = 31; i >= 0; i -= XPaletteStep)
   {
      for( j = 0; j < 768; j++ )
	 Pal[j] =  (i * (unsigned)palette[j]) >> 5;
      x_set_palette_noVR( Pal );
   }
   XPaletteLevel = 0;
}

/***************************************************************************
 Fades in only a part of the palette.
***************************************************************************/
void x_part_fade_in_noVR( int first, int num, BYTE * palette )
{
   int i, j, v;

   v = first * 3;
   for( i = 0; i < 32; i += XPaletteStep )
   {
      for( j = 0; j < num * 3; j++ )
	 Pal[j] =  (i * (unsigned)palette[j+v]) >> 5;
      x_set_part_palette_noVR( first, num, Pal );
   }
   XPaletteLevel = 32;
}

/***************************************************************************
 Fades out only a part of the palette.
***************************************************************************/
void x_part_fade_out_noVR( int first, int num, BYTE * palette )
{
   int i, j, v;

   v = first * 3;
   for( i = 31; i >= 0; i -= XPaletteStep)
   {
      for( j = 0; j < num * 3; j++ )
	 Pal[j] =  (i * (unsigned)palette[j+v]) >> 5;
      x_set_part_palette_noVR( first, num, Pal );
   }
   XPaletteLevel = 0;
}

/***************************************************************************
 Cycles 'num' palette entries, starting at the 'first' entry.  Entries are
 cycled to the "left".
***************************************************************************/
void x_part_cycleL_noVR( int first, int num, BYTE * palette )
{
   int i, v;
   BYTE a, b, c;

   v = first * 3;
   a = palette[0+v];
   b = palette[1+v];
   c = palette[2+v];
   for( i = 0; i < (num-1) * 3; i += XPaletteStep )
      palette[i+v] = palette[i + 3+v];
   palette[i+v] = a;
   palette[i + 1+v] = b;
   palette[i + 2+v] = c;
   x_set_part_palette_noVR( first, num, &palette[v] );
}

/***************************************************************************
 Cycles 'num' palette entries, starting at the 'first' entry.  Entries are
 cycled to the "right".
***************************************************************************/
void x_part_cycleR_noVR( int first, int num, BYTE * palette )
{
   int i, v;
   BYTE a, b, c;

   i = (num * 3) - 1;
   v = first * 3;
   a = palette[i - 2+v];
   b = palette[i - 1+v];
   c = palette[i+v];
   for( ; i >= 3; i-- )
      palette[i+v] = palette[i - 3+v];
   palette[0+v] = a;
   palette[1+v] = b;
   palette[2+v] = c;
   x_set_part_palette_noVR( first, num, &palette[v] );
}

/* Stepped palette fading routines - do not wait for retrace */

/***************************************************************************
 Increases intensity by one step
***************************************************************************/
void x_fade_in_step( BYTE * palette )
{
  int j;

  XPaletteLevel += XPaletteStep;
  if (XPaletteLevel > 32)
    XPaletteLevel = 32;
  
  for( j = 0; j < 768; j++ )
    Pal[j] =  (XPaletteLevel * (unsigned)palette[j]) >> 5;
  x_set_palette_noVR( Pal );
}

/***************************************************************************
 Fades out one step
***************************************************************************/
void x_fade_out_step( BYTE * palette )
{
   int j;
   
   XPaletteLevel -= XPaletteStep;
   if (XPaletteLevel < 0)
     XPaletteLevel = 0;

   for( j = 0; j < 768; j++ )
     Pal[j] =  (XPaletteLevel * (unsigned)palette[j]) >> 5;
   x_set_palette_noVR( Pal );
}

/***************************************************************************
 Fades partial palette in one step
***************************************************************************/
void x_part_fade_in_step( int first, int num, BYTE * palette )
{
   int j, v;

  XPaletteLevel += XPaletteStep;
  if (XPaletteLevel > 32)
    XPaletteLevel = 32;
   
   v = first * 3;
      for( j = 0; j < num * 3; j++ )
	 Pal[j] =  (XPaletteLevel * (unsigned)palette[j+v]) >> 5;
      x_set_part_palette_noVR( first, num, Pal );
}

/***************************************************************************
 Fades partial palette out one step
***************************************************************************/
void x_part_fade_out_step( int first, int num, BYTE * palette )
{
   int j, v;
   
   XPaletteLevel -= XPaletteStep;
   if (XPaletteLevel < 0)
     XPaletteLevel = 0;

   v = first * 3;
   for( j = 0; j < num * 3; j++ )
     Pal[j] =  (XPaletteLevel * (unsigned)palette[j+v]) >> 5;
   x_set_part_palette_noVR( first, num, Pal );
}
