/*
 * xmouse.c - a simple polled mouse driver for mode-x resolutions
 *
 * DESCRIPTION
 * Functions to initialize, show and hide mouse.  Cursor can be replaced
 * with any valid planar bitmap using XSetMouseCursor().  XMouseGetEvent()
 * polls the mouse, and updates the display if it has moved.  The background
 * is saved near the end of video memory, as defined by MOUSE_BG_OFFSET.
 * Should handle mouse cursors up to about 80 x 64.
 * Cursor movement is limited to the screen coordinates - cursor size to
 * prevent clobbering the undisplayed page.  To access the very edges of
 * the page, make each page a little bigger (ie add a non-visible offset
 * between the pages) where the cursor can safely go.
 * NEW - mouse callback driver added.  Use #define switch in this file
 * to select when compiling.  Be sure to call x_hide_mouse() when drawing.
 * Must also call x_remove_mouse() on exit when using callback.
 *
 * USAGE
 *
 * NOTES
 * copy to background uses 80 pixel width starting at MOUSE_BG_OFFSET as 
 * the basis for the background page.  This limits cursor size, but you 
 * can squeeze in more lines.  Be careful not to clobber this with
 * multiple/background pages!
 *
 * REVISION HISTORY
 * Date         Reason
 * 27 Jun 95    Initial Release
 *  3 Aug 95    Changed bitmaps to word w and h
 *
 */

#include <go32.h>
#include <dpmi.h>
#include "defines.h"
#include "xlib.h"

#define MOUSE_BG_OFFSET 60000

static BYTE * pMouseCursor;

/* default 12x16 cursor (planar format) */
static BYTE default_mouse_cursor[] = { 3, 0, 16, 0,
     15,  0,  0, 15,  0,  0, 15,  0,  0, 15,  0,  0,
     15, 15,  0, 15, 15,  0, 15, 15,  0, 15,  0,  0,
     15,  0,  0,  0, 15,  0,  0, 15,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0, 15,  0,  0, 15,  0,  0, 15,  0,  0,
     15,  0,  0, 15, 15,  0, 15, 15,  0, 15,  0,  0,
      0,  0,  0,  0, 15,  0,  0, 15,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0, 15,  0,  0, 15,  0,  0,
     15,  0,  0, 15,  0,  0, 15, 15,  0, 15,  0,  0,
     15,  0,  0, 15,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0, 15,  0,  0,
     15,  0,  0, 15,  0,  0, 15,  0,  0, 15,  0,  0,
     15,  0,  0, 15,  0,  0, 15,  0,  0,  0,  0,  0,
      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
};

int MouseX;
int MouseY;
int MouseButtonStatus;
extern BOOLEAN MouseDriverActive;
BOOLEAN MouseHidden;
extern int ScrnPhysicalPixelWidth;
extern int ScrnPhysicalHeight;
extern int VisiblePageOffs; /* cursor will always be drawn here */

int XMouseClipMin;
int XMouseClipMax;
int YMouseClipMin;
int YMouseClipMax;

static int OldMouseX;
static int OldMouseY;
static int OldMousePageBase;


/***********************************************
** Initializes mouse to display size
***********************************************/
int x_init_mouse()
{
  __dpmi_regs r;
  
  /* Initialize mouse */
  r.x.ax = 0;
  __dpmi_int(0x33, &r);
  /* Hide mouse cursor */
  r.x.ax = 2;
  __dpmi_int(0x33, &r);
  /* Set min/max horizontal position */
  r.x.ax = 7;
  r.x.cx = 0;  /* min */
  r.x.dx = (ScrnPhysicalPixelWidth - 12) * 2; /* don't let cursor wrap */
  __dpmi_int(0x33, &r);
  /* Set min/max vertical position */
  r.x.ax = 8;
  r.x.cx = 0;
  r.x.dx = ScrnPhysicalHeight - 16;
  __dpmi_int(0x33, &r);
  
  pMouseCursor = &default_mouse_cursor[0];
  
  XMouseClipMin = 0;
  YMouseClipMin = 0;
  XMouseClipMax = ScrnPhysicalPixelWidth - ((int)pMouseCursor[0] * 4);
  YMouseClipMax = ScrnPhysicalHeight - (int)pMouseCursor[2];
  
  /* Set Horiz/Vert mouse resolution */
  r.x.ax = 0x0f;
  r.x.cx = 4;
  r.x.dx = 8;
  __dpmi_int(0x33, &r);
  /* Get mouse position and button status */
  r.x.ax = 3;
  __dpmi_int(0x33, &r);
  MouseX = (int)(r.x.cx >> 1);
  MouseY = (int)r.x.dx;
  MouseButtonStatus = (int)r.x.bx;

  MouseHidden = TRUE;
  
  MouseDriverActive = TRUE;
  return(0);
}

/* Remove mouse driver - call only if using callback */
void x_remove_mouse()
{
  __dpmi_regs r;

  x_hide_mouse();
  r.x.ax = 12;
  r.x.cx = 0;
  MouseDriverActive = FALSE;
}

void x_hide_mouse()
{
  if (!MouseHidden && MouseDriverActive)
  {
    MouseHidden = TRUE;
    x_cp_vid_rect((MouseX & 3), 0, (MouseX & 3)+(pMouseCursor[0] << 2), 
                   pMouseCursor[2], MouseX, MouseY, MOUSE_BG_OFFSET, 
                   VisiblePageOffs, 80, ScrnPhysicalPixelWidth);
  }
}

void x_show_mouse()
{
  if (MouseHidden && MouseDriverActive)
  {
    
    __dpmi_regs r;
    /* This will always be updated by callback routine */
    r.x.ax = 3;
    __dpmi_int(0x33, &r);
    MouseButtonStatus = r.x.bx;
    MouseX = (r.x.cx >> 1);
    MouseY = r.x.dx;

    OldMouseX = MouseX;
    OldMouseY = MouseY;
    OldMousePageBase = VisiblePageOffs;
    x_cp_vid_rect(MouseX, MouseY, MouseX+(pMouseCursor[0] << 2), 
                  MouseY+pMouseCursor[2], (MouseX & 3), 0, 
                  VisiblePageOffs, MOUSE_BG_OFFSET, ScrnPhysicalPixelWidth, 80);
    x_put_PBM_masked(MouseX, MouseY, VisiblePageOffs, pMouseCursor);
    MouseHidden = FALSE;
  }
}

/**********************************************************
** sets a new mouse cursor - max size ~= 80 x 64
**********************************************************/
void x_set_mouse_cursor(BYTE * bitmap)
{
  __dpmi_regs r;

  if (!MouseDriverActive)
    return;

  pMouseCursor = &bitmap[0];
  
  /* Adjust limits for new cursor size */
  XMouseClipMin = 0;
  YMouseClipMin = 0;
  XMouseClipMax = ScrnPhysicalPixelWidth - (pMouseCursor[0] << 2);
  YMouseClipMax = ScrnPhysicalHeight - pMouseCursor[2];
  /* Set min/max horizontal position */
  r.x.ax = 7;
  r.x.cx = 0;  /* min */
  r.x.dx = ScrnPhysicalPixelWidth - (pMouseCursor[0] << 2);
  r.x.dx = r.x.dx << 1; /* cursor steps by 2 pixels */
  __dpmi_int(0x33, &r);
  /* Set min/max vertical position */
  r.x.ax = 8;
  r.x.cx = 0;
  r.x.dx = ScrnPhysicalHeight - pMouseCursor[2];
  __dpmi_int(0x33, &r);
  
  if (!MouseHidden)
  {
    x_cp_vid_rect((MouseX & 3), 0, (MouseX & 3)+(pMouseCursor[0] << 2), 
                  pMouseCursor[2], MouseX, MouseY, MOUSE_BG_OFFSET, 
                  VisiblePageOffs, 80, ScrnPhysicalPixelWidth);
    x_put_PBM_masked(MouseX, MouseY, VisiblePageOffs, pMouseCursor);
  }
}

/****************************************************************
** restricts mouse movement to the current mouse clip region
*****************************************************************/
void x_mouse_window()
{
  __dpmi_regs r;
  r.x.ax = 7;
  r.x.cx = XMouseClipMin << 1;
  r.x.dx = XMouseClipMax << 1;
  __dpmi_int(0x33, &r);
  
  r.x.ax = 8;
  r.x.cx = YMouseClipMin;
  r.x.dx = YMouseClipMax;
  __dpmi_int(0x33, &r);
}

/********************************************************
** Polls current mouse position, updates coordinate and
** button status, redraws mouse cursor if it has moved
********************************************************/
void x_mouse_get_event()
{
  __dpmi_regs r;

  if (!MouseDriverActive)
    return;
  r.x.ax = 3;
  __dpmi_int(0x33, &r);
  MouseButtonStatus = (r.x.bx & 7);
  /* see if mouse has moved */
  if ((MouseX != (r.x.cx >> 1)) || (MouseY != r.x.dx))
  {
    MouseX = (r.x.cx >> 1);
    MouseY = r.x.dx;
    if (!MouseHidden)
    {
      /* restore bg at old location */
      /* NOTE -  only uses low byte of bitmap since cursor must be <= 64x64) */
      x_cp_vid_rect((OldMouseX & 3), 0, (OldMouseX & 3)+(pMouseCursor[0] << 2), 
                    pMouseCursor[2], OldMouseX, OldMouseY, MOUSE_BG_OFFSET, 
                    OldMousePageBase, 80, ScrnPhysicalPixelWidth);
      /* save bg at new location */
      OldMouseX = MouseX;
      OldMouseY = MouseY;
      OldMousePageBase = VisiblePageOffs;
      x_cp_vid_rect(MouseX, MouseY, MouseX+(pMouseCursor[0] << 2), 
                    MouseY+pMouseCursor[2], (MouseX & 3), 0, VisiblePageOffs, 
                    MOUSE_BG_OFFSET, ScrnPhysicalPixelWidth, 80);
      x_put_PBM_masked(MouseX, MouseY, VisiblePageOffs, pMouseCursor);
    }
  }
}

