/*
 * archive.c - routines to read and write multiple file data archives
 *
 * DESCRIPTION
 * archive stores multiple data files end-to-end after a header which
 * contains the filenames and offsets (from top of file) of the files
 * stored.
 *
 * USAGE
 *
 * NOTES
 *
 * REVISION HISTORY
 * Date         Reason
 * 14 Mar 95    Initial Coding
 * 3  May 95    Changed to new layout style
 * 8  Jun 95    Fixed 12 byte filename bug, documented
 * 29 Jun 95    Added to xlib package
 *
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <dos.h>
#include <sys/stat.h>

#include "defines.h"
#include "archive.h"

const char arc_info[] = "RES";
BYTE arc_maj_version = 1;
BYTE arc_min_version = 1;
int ArcNumFiles;

/***********************************************************
   ARC FILE FORMAT:

   3 bytes:               "RES"
   1 byte:                Major version # (currently 1)
   1 byte:                Minor version # (currently 1)
   1 long int (4 bytes):  Number of files in archive
   ARC_MAX_FILES * 20:    file info block
   ??? bytes              raw binary data from files

   Each file info block is:
   12 bytes:              filename
   1 long int:            file offset in bytes (from beginning of archive)
   1 long int:            file size in bytes

   ARC_MAX_FILES is defined in archive.h, currently set to 200,
   so an empty archive will be 4009 bytes.

***********************************************************/

/******************************************** 
** creates the named archive
** Returns: 0 success
**         -1 couldn't create archive file
**         -4 archive already exists
********************************************/
int arc_create(char * arcname)
{
  FILE * arcfile, * arcfile1;
  int i;
  char nothing[] = "\0\0\0\0\0\0\0\0\0\0\0\0";

  arcfile1 = fopen(arcname, "rb");
  if (arcfile1 != NULL)
  {
    /* file already exists */
    fclose(arcfile1);
    return(ARC_EXISTS);
  }
  arcfile = fopen(arcname, "wb");
  if (arcfile == NULL)
  {
    /* Couldn't create arc file to write */
    return(ARC_NOT_FOUND);
  }

  /* write archive header block with no data */
  ArcNumFiles = 0;

  fwrite(arc_info, 1, 3, arcfile);
  fwrite(&arc_maj_version, 1, 1, arcfile);
  fwrite(&arc_min_version, 1, 1, arcfile);
  fwrite(&ArcNumFiles, sizeof(int), 1, arcfile);
  for (i=0; i<ARC_MAX_FILES; i++)
  {
    fwrite(nothing, 1, 12, arcfile);
    fwrite(&ArcNumFiles, sizeof(int), 1, arcfile);  /* file offset block */
    fwrite(&ArcNumFiles, sizeof(int), 1, arcfile);  /* file size block */
  }
  fclose(arcfile);
  return(ARC_SUCCESS);
}

/********************************************************
** searches for a filename in the named archive
** Returns:  0 file not found
**          -1 archive doesn't exist / not readable
**           # length of file in bytes
********************************************************/
int arc_query(char * arcname, char * filename)
{
  int i, offset, length;
  FILE * arcfile;
  char buffer[16];
  
  arcfile = fopen(arcname, "rb");
  if (arcfile == NULL)
  {
    /* archive file not found */
    return(ARC_NOT_FOUND);
  }
  fread(buffer, 1, 5, arcfile);
  fread(&ArcNumFiles, sizeof(int), 1, arcfile);
  for (i=0; i<ArcNumFiles; i++)
  {
    fread(buffer, 1, 12, arcfile);
    fread(&offset, sizeof(int), 1, arcfile);
    fread(&length, sizeof(int), 1, arcfile);
    if ((strncmp(buffer, filename, 12) == 0))
    {
      /* success - filenames match */
      fclose(arcfile);
      return(length);
    }
  }
  /* filename not found in arcfile */
  fclose(arcfile);
  return(0);
}

/********************************************************
** adds the file filename to the first open slot
** in the archive file
** Returns:  0 success
**          -1 archive doesn't exist / not writeable
**          -3 filename already exists in archive
********************************************************/
int arc_add(char * arcname, char * filename, char * data, int length)
{
  FILE * arcfile;
  int i, offset;
  char buffer[24];
  struct stat statbuf;

  i = arc_query(arcname, filename);
  if (i > 0)
  {
    /* This filename already exists in archive */
    return(FILE_EXISTS);
  }
  if (i == ARC_NOT_FOUND)
  {
    /* Archive file not present / readable */
    return(ARC_NOT_FOUND);
  }
  
  stat(arcname, &statbuf);
  offset = statbuf.st_size;

  arcfile = fopen(arcname, "rb+");
  if (arcfile == NULL)
  {
    /* archive file not found */
    return(ARC_NOT_FOUND);
  }
  fread(buffer, 1, 5, arcfile);
  fread(&ArcNumFiles, sizeof(int), 1, arcfile);
  if (ArcNumFiles >= ARC_MAX_FILES)
  {
    /* archive is full - can't add more */
    fclose(arcfile);
    return(ARC_FULL);
  }
  fseek(arcfile, 20 * ArcNumFiles, 1);
  if (strlen(filename) > 12)
    fwrite(filename, 1, 12, arcfile);
  else
    fwrite(filename, 1, strlen(filename), arcfile);
  fseek(arcfile, 20 * ArcNumFiles+12+ARC_HEADER_SIZE, 0);
  fwrite(&offset, sizeof(int), 1, arcfile);
  fwrite(&length, sizeof(int), 1, arcfile);
  fseek(arcfile, 0, 2); /* EOF */  
  fwrite(data, 1, length, arcfile);   /* append data to eof */
  ArcNumFiles ++;
  fseek(arcfile, 5, 0);
  fwrite(&ArcNumFiles, sizeof(int), 1, arcfile);
  fclose(arcfile);
  return(ARC_SUCCESS);
}

/************************************************************
** Deletes the named file from archive
** Returns:  0 success 
**          -1 archive doesn't exist / not writeable
**          -2 file not found in archive
************************************************************/
int arc_remove(char * arcname, char * filename)
{
  FILE * arcfile, * tempfile;
  int i, j, count = 0, offset, length, file_found = 0;
  int toffset, tlength;  /* temporary for updating shifted offsets */
  char buffer[48];
  struct stat statbuf;
  
  i = arc_query(arcname, filename);
  if (i == 0)
  {
    /* This filename does not exist in archive */
    return(FILE_NOT_FOUND);
  }
  if (i == ARC_NOT_FOUND)
  {
    /* Archive file not present / readable */
    return(ARC_NOT_FOUND);
  }

  stat(arcname, &statbuf);

  arcfile = fopen(arcname, "rb+");
  if (arcfile == NULL)
  {
    /* archive file not found */
    return(ARC_NOT_FOUND);
  }
  tempfile = fopen("~~temp~~.001", "wb");
  fread(buffer, 1, 5, arcfile);
  fread(&ArcNumFiles, sizeof(int), 1, arcfile);
  while (!file_found)
  {
    fread(buffer, 1, 12, arcfile);
    fread(&offset, sizeof(int), 1, arcfile);
    fread(&length, sizeof(int), 1, arcfile);
    count ++;
    if (strncmp(buffer, filename, 12) == 0)
    {
      /* success - cut out file info */
      file_found = TRUE;
      fseek(arcfile, -20, 1);
      for (i = 0; i < (ArcNumFiles - count); i++)
      {
        fseek(arcfile, 20, 1);
        fread(buffer, 1, 12, arcfile);
        fread(&toffset, sizeof(int), 1, arcfile);
        fread(&tlength, sizeof(int), 1, arcfile);
        toffset -= length;
        fseek(arcfile, -40, 1);
        fwrite(buffer, 1, 12, arcfile);
        fwrite(&toffset, sizeof(int), 1, arcfile);
        fwrite(&tlength, sizeof(int), 1, arcfile);
      }
      j = 0;
      for (i=0; i<5; i++)
        fwrite(&j, sizeof(int), 1, arcfile);   /* blank out last entry */
      
      /* cut out file data */
      ArcNumFiles --;
      rewind(arcfile);
      for (i=0; i<offset; i++)
      {
        fread(buffer, 1, 1, arcfile);
        fwrite(buffer, 1, 1, tempfile);
      }
      fseek(arcfile, (offset+length), 0);
      for (i=0; i<(statbuf.st_size - offset - length); i++)
      {
        fread(buffer, 1, 1, arcfile);
        fwrite(buffer, 1, 1, tempfile);
      }
      fseek(tempfile, 5, 0);
      fwrite(&ArcNumFiles, sizeof(int), 1, tempfile);
      
      fclose(tempfile);
      fclose(arcfile);

      sprintf(buffer, "del %s", arcname);
      system(buffer);
      sprintf(buffer, "ren ~~temp~~.001 %s", arcname);
      system(buffer);
    }
  }
  return(ARC_SUCCESS);
}

/***************************************************************
** extracts the named file from the archive and saves
** it in the current directory as filename.
** DOES NOT DELETE FILE FROM ARCHIVE!
**
** NOTE:  Will overwrite existing files without backing up!
**        Check for existance of file first.
** Returns:  0 success
**          -1 archive doesn't exist / not writeable
**          -2 file not found in archive
**          -5 couldn't write file in directory
***************************************************************/
int arc_extract(char * arcname, char * filename)
{
  /* do:  arc_read, save file to disk, arc_remove */
  int i, length;
  FILE * datafile;
  char * buffer;

  length = arc_query(arcname, filename);
  if (length == 0)
  {
    /* This filename does not exist in archive */
    return(FILE_NOT_FOUND);
  }
  if (length == ARC_NOT_FOUND)
  {
    /* Archive file not present / readable */
    return(ARC_NOT_FOUND);
  }

  datafile = fopen(filename, "wb");
  if (datafile == NULL)
  {
    /* couldn't open file to be extracted */
    return(FILE_CANT_WRITE);
  }
  
  buffer = (char *)malloc(length);
  if (buffer == NULL)
  {
    /* can't allocate memory for buffer */
    fclose(datafile);
    return(ARC_MALLOC_ERROR);
  }
  
  i = arc_read(arcname, filename, buffer);
  if (i != ARC_SUCCESS)
  {
    /* error reading file from archive */
    fclose(datafile);
    free(buffer);
    return(ARC_NOT_FOUND);
  }
  fwrite(buffer, 1, length, datafile);
  fclose(datafile);
  free(buffer);

  return(ARC_SUCCESS);
}

/**************************************************************
** reads the contents of the specified file from archive
** into the array data (must be pre-malloc'd - use 
** arc_query to get size of file)
** Returns:  0 success
**          -1 archive doesn't exist / not readable
**          -2 file not found in archive
*************************************************************/
int arc_read(char * arcname, char * filename, char * data)
{
  FILE * arcfile;
  int i, offset, length;
  char buffer[16];

  arcfile = fopen(arcname, "rb");
  if (arcfile == NULL)
  {
    /* archive file not found */
    return(ARC_NOT_FOUND);
  }
  fread(buffer, 1, 5, arcfile);
  fread(&ArcNumFiles, sizeof(int), 1, arcfile);
  for (i=0; i<ArcNumFiles; i++)
  {
    fread(buffer, 1, 12, arcfile);
    fread(&offset, sizeof(int), 1, arcfile);
    fread(&length, sizeof(int), 1, arcfile);
    if (strncmp(buffer, filename, 12) == 0)
    {
      /* success - read file data */
      fseek(arcfile, offset, 0);
      fread(data, 1, length, arcfile);
      
      fclose(arcfile);
      return(ARC_SUCCESS);
    }
  }
  /* filename not found in archive */
  fclose(arcfile);
  return(FILE_NOT_FOUND);
}

