/*****************************************************************************
 *
 * xpbmclip.s - Clipped Planar Bitmap Functions - System RAM <-> Video RAM
 *
 * DESCRIPTION
 * Routines to draw planar and masked-planar bitmaps to video memory with
 * clipping.  All routines return either success (0) or failure (1) in AX.
 *
 * NOTES
 * Works with djgpp v. 1.12 or later.
 * Parts based on XLIB60 by Themie Goumas.
 *
 * ASSUMPTIONS
 * 1.  The bitmap is not larger than the clipping area, i.e., it can be clipped
 *     at most against one X border and one Y border.
 * 2.  The clipping borders are greater than or equal to zero.
 * 3.  The bitmap dimentions are greater than zero.
 * 4.  The bitmap address is not zero.
 *
 * REVISION HISTORY
 * Date       Who               Reason
 * 30-AUG-95  Ed. Mueller       Initial Release
 * 31 Aug 95  Paul Fenwick      Minor revision and added unmasked PBMs
 *
 *****************************************************************************/

        .align  4
        .data        
BMHeight:
        .long   0
BMWidth:
        .long   0
CType:
        .long   0
DataInc:
        .long   0
LeftSkip:
        .long   0
LineInc:
        .long   0
PlaneInc:
        .long   0
TopRow:
        .long   0
Plane:
        .byte   0
        
        .text

        .globl  _x_put_PBM_masked_clipx
        .globl  _x_put_PBM_masked_clipxy
        .globl  _x_put_PBM_masked_clipy
        .globl  _x_set_cliprect
        .globl  _x_put_PBM_clipx
        .globl  _x_put_PBM_clipxy
        .globl  _x_put_PBM_clipy
        .globl  _ScrnLogicalByteWidth
        .globl  _XClipMin
        .globl  _XClipMax
        .globl  _YClipMin
        .globl  _YClipMax
        .extern _core_select
        
/*****************************************************************************
 *
 * x_put_PBM_masked_clipx(int x, int y, int PageBase, BYTE * bitmap);
 * Stack offset:             +8    +12      +16           +20
 *
 * Clipping region variables:  XClipMin, XClipMax
 *
 * X and Y values may be negative.
 *  
 *****************************************************************************/
        .align  4
_x_put_PBM_masked_clipx:
        pushl   %ebp
        movl    %esp, %ebp

        # Save all registers.
        pushl   %esi
        pushl   %edi
        pushl   %eax
        pushl   %ebx
        pushl   %ecx
        pushl   %edx
        pushw   %es

        xorl    %eax, %eax
        movl    %eax, CType             # Clear Clip type descision var
        movl    20(%ebp), %esi          # ESI - Bitmap pointer
        movzwl  2(%esi), %ebx           # EBX - Height
        movzwl  (%esi), %eax            # EAX - Width
        movl    8(%ebp), %edi           # EDI - X (pixel)
        movl    %edi, %ecx              # Save in ECX
        sarl    $2, %edi                # EDI - X (planar)

        # Clip against the left border.
        movl    _XClipMin, %edx
        subl    %edi, %edx
        jle     XNotLeftClip
        cmpl    %eax, %edx
        jnl     XNotVisible
        addl    %edx, %edi
        movl    %edx, LeftSkip
        movl    %edx, DataInc
        subl    %edx, %eax
        movl    $1, CType
        jmp     XHorizClipDone

XNotVisible:
        movl    $1, %eax                # Set return code
        jmp     XDone

XNotLeftClip:
        # Clip against the right border.
        movl    _XClipMax, %edx
        subl    %edi, %edx
        js      XNotVisible
        movl    $0, LeftSkip
        movl    $0, DataInc
        cmpl    %eax, %edx
        jge     XHorizClipDone
        incl    %edx
        subl    %edx, %eax
        movl    %eax, DataInc
        movl    %edx, %eax
        movl    $-1, CType

XHorizClipDone:
        movl    %eax, BMWidth           # Save width and height of clipped image
        movl    %ebx, BMHeight

        movl    $4, %eax                # Skip dimension bytes in source
        addl    LeftSkip, %eax          # Skip clipped pixels in front of row
        addl    %eax, %esi

        # Compute starting screen addr.
        movl    _ScrnLogicalByteWidth, %ebx
        movl    12(%ebp), %eax
        imull   %ebx, %eax
        addl    16(%ebp), %eax
        addl    $0xa0000, %eax
        addl    %eax, %edi              # EDI - starting screen address
        subl    BMWidth, %ebx           # Calc. diff. from end of bitmap in
        movl    %ebx, LineInc           #  curr. line to bitmap in next line

        movw    _core_select, %ax       # We're using post-DPMI extender
        movw    %ax, %es                # Use real area selector

        # Prepare VGA for CPU to video writes.
        movb    $0x11, %ah              # Map mask - loaded to rotate
        andl    $0x03, %ecx
        shlb    %cl, %ah
        movw    $0x3c4, %dx
        movb    $0x02, %al
        outb    %al, %dx
        incw    %dx
        movb    $4, Plane

XPlaneLoop:
        pushl   %edi                    # Save starting screen address
        movl    BMHeight, %ebx
        movb    %ah, %al
        outb    %al, %dx

        movl    BMWidth, %ebp
XRowLoop:
        movl    %ebp, %ecx           # Reset column counter
        jecxz   XNoWidth

XColLoop:        
        movb    (%esi), %al             # Get next source bitmap byte
        incl    %esi
        orb     %al, %al                # If not zero, write to dest.
        jz      XNoPixel
        movb    %al, %es:(%edi)

XNoPixel:
        incl    %edi
        decl    %ecx
        jnz     XColLoop                # Loop if more columns left

XNoWidth:
        addl    DataInc, %esi           # Move to next source row
        addl    LineInc, %edi           # Move to next row on screen
        decl    %ebx                    # Decrement row counter
        jnz     XRowLoop                # Jump if more rows left
        popl    %edi                    # Restore starting screen address
        rolb    $1, %ah                 # Shift mask for next plane
        jnb     XNoCarry                # Test for a plane transition

        # A transition has occurred from plane 3 to plane 0.  At this point,
        # if we started in plane n, we may need to go back and draw pixels in 
        # planes 0..n-1.  If we've clipped pixels to the left, increment the
        # working width and start putting pixels one byte earlier in the source.
        movl    CType, %eax
        addl    %eax, BMWidth           # Adjust bitmap width
        subl    %eax, DataInc           # Update vars. to account for new width
        subl    %eax, LineInc
        cmpl    $0, %eax                # Left or right clip?
        movb    $0x11, %ah              # Restore plane mask
        jg      XRightAdvance           # Jump if clipping against the left edge
        incl    %edi
        jmp     XNoCarry

XRightAdvance:
        decl    %esi                    # Start image processing one byte earlier

XNoCarry:
        decb    Plane                   # Decrement plane counter
        jnz     XPlaneLoop              # Jump if more planes left
        xorl    %eax, %eax              # Set return code

XDone:
        popw    %es                     # Restore registers.
        popl    %edx
        popl    %ecx
        popl    %ebx
        popl    %eax
        popl    %edi
        popl    %esi
        popl    %ebp
        ret


/*****************************************************************************
 *
 * x_put_PBM_masked_clipxy(int x, int y, int PageBase, BYTE * bitmap);
 * Stack offset:              +8    +12      +16           +20
 *
 * Clipping region variables:  XClipMin, XClipMax, YClipMin, YClipMax
 *
 * X and Y values may be negative.
 *  
 *****************************************************************************/
        .align  4
_x_put_PBM_masked_clipxy:
        pushl   %ebp
        movl    %esp, %ebp

        # Save all registers.
        pushl   %esi
        pushl   %edi
        pushl   %eax
        pushl   %ebx
        pushl   %ecx
        pushl   %edx
        pushw   %es

        xorl    %eax, %eax
        movl    %eax, CType             # Clear Clip type descision var
        movl    20(%ebp), %esi          # ESI - Bitmap pointer
        movzwl  2(%esi), %ebx           # EBX - Height
        movzwl  (%esi), %eax            # EAX - Width
        movl    %eax, BMWidth           # Save width
        imull   %ebx, %eax
        movl    %eax, PlaneInc          # Num. of bytes/plane
        movl    BMWidth, %eax           # Restore width
        movl    8(%ebp), %edi           # EDI - X (pixel)
        movl    %edi, %ecx              # Save in ECX
        sarl    $2, %edi                # EDI - X (planar)

        # Clip against the top border.
        movl    _YClipMin, %edx
        subl    12(%ebp), %edx
        jle     XYNotTopClip
        cmpl    %ebx, %edx
        jnl     XYNotVisible
        movl    %edx, TopRow
        subl    %edx, %ebx
        addl    %edx, 12(%ebp)
        jmp     XYVertClipDone

XYNotVisible:
        movl    $1, %eax                # Set return code
        jmp     XYDone

        # Clip against the bottom border.
XYNotTopClip:
        movl    _YClipMax, %edx
        subl    12(%ebp), %edx
        js      XYNotVisible
        movl    $0, TopRow
        cmpl    %ebx, %edx
        jge     XYVertClipDone
        incl    %edx
        movl    %edx, %ebx

XYVertClipDone:
        # Clip against the left border.
        movl    _XClipMin, %edx
        subl    %edi, %edx
        jle     XYNotLeftClip
        cmpl    %eax, %edx
        jnl     XYNotVisible
        addl    %edx, %edi
        movl    %edx, LeftSkip
        movl    %edx, DataInc
        subl    %edx, %eax
        movl    $1, CType
        jmp     XYHorizClipDone

XYNotLeftClip:
        # Clip against the right border.
        movl    _XClipMax, %edx
        subl    %edi, %edx
        js      XYNotVisible
        movl    $0, LeftSkip
        movl    $0, DataInc
        cmpl    %eax, %edx
        jge     XYHorizClipDone
        incl    %edx
        subl    %edx, %eax
        movl    %eax, DataInc
        movl    %edx, %eax
        movl    $-1, CType

XYHorizClipDone:
        movl    %eax, BMWidth           # Save width and height of clipped image
        movl    %ebx, BMHeight

        addl    DataInc, %eax           # EAX = Original width of image
        imull   TopRow, %eax            # Skip clipped top rows
        addl    $4, %eax                # Skip dimension bytes in source
        addl    LeftSkip, %eax          # Skip clipped pixels in front of row
        addl    %eax, %esi

        # Compute starting screen addr.
        movl    _ScrnLogicalByteWidth, %ebx
        movl    12(%ebp), %eax
        imull   %ebx, %eax
        addl    16(%ebp), %eax
        addl    $0xa0000, %eax
        addl    %eax, %edi              # EDI - starting screen address
        subl    BMWidth, %ebx           # Calc. diff. from end of bitmap in
        movl    %ebx, LineInc           #  curr. line to bitmap in next line

        movw    _core_select, %ax       # We're using post-DPMI extender
        movw    %ax, %es                # Use real area selector

        # Prepare VGA for CPU to video writes.
        movb    $0x11, %ah              # Map mask - loaded to rotate
        andl    $0x03, %ecx
        shlb    %cl, %ah
        movw    $0x3c4, %dx
        movb    $0x02, %al
        outb    %al, %dx
        incw    %dx
        movb    $4, Plane

XYPlaneLoop:
        pushl   %edi                    # Save starting screen address
        pushl   %esi                    # Save bitmap address
        movl    BMHeight, %ebx
        movb    %ah, %al
        outb    %al, %dx

        movl    BMWidth, %ebp
XYRowLoop:
        movl    %ebp, %ecx           # Reset column counter
        jecxz   XYNoWidth

XYColLoop:        
        movb    (%esi), %al             # Get next source bitmap byte
        incl    %esi
        orb     %al, %al                # If not zero, write to dest.
        jz      XYNoPixel
        movb    %al, %es:(%edi)

XYNoPixel:
        incl    %edi
        decl    %ecx
        jnz     XYColLoop               # Loop if more columns left

XYNoWidth:
        addl    DataInc, %esi           # Move to next source row
        addl    LineInc, %edi           # Move to next row on screen
        decl    %ebx                    # Decrement row counter
        jnz     XYRowLoop               # Jump if more rows left
        popl    %esi                    # Restore bitmap address and set to
        addl    PlaneInc, %esi          #  addr. of 1st. pixel in next plane
        popl    %edi                    # Restore starting screen address
        rolb    $1, %ah                 # Shift mask for next plane
        jnb     XYNoCarry               # Test for a plane transition

        # A transition has occurred from plane 3 to plane 0.  At this point,
        # if we started in plane n, we may need to go back and draw pixels in 
        # planes 0..n-1.  If we've clipped pixels to the left, increment the
        # working width and start putting pixels one byte earlier in the source.
        movl    CType, %eax
        addl    %eax, BMWidth           # Adjust bitmap width
        subl    %eax, DataInc           # Update vars. to account for new width
        subl    %eax, LineInc
        cmpl    $0, %eax                # Left or right clip?
        movb    $0x11, %ah              # Restore plane mask
        jg      XYRightAdvance          # Jump if clipping against the left edge
        incl    %edi
        jmp     XYNoCarry

XYRightAdvance:
        decl    %esi                    # Start image processing one byte earlier

XYNoCarry:
        decb    Plane                   # Decrement plane counter
        jnz     XYPlaneLoop             # Jump if more planes left
        xor     %eax, %eax              # Set return code

XYDone:
        popw    %es                     # Restore registers.
        popl    %edx
        popl    %ecx
        popl    %ebx
        popl    %eax
        popl    %edi
        popl    %esi
        popl    %ebp
        ret


/*****************************************************************************
 *
 * x_put_PBM_masked_clipy(int x, int y, int PageBase, BYTE * bitmap);
 * Stack offset:             +8    +12      +16           +20
 *
 * Clipping region variables:  YClipMin, YClipMax
 *
 * X and Y values may be negative.
 *  
 *****************************************************************************/
        .align  4
_x_put_PBM_masked_clipy:
        pushl   %ebp
        movl    %esp, %ebp

        # Save all registers.
        pushl   %esi
        pushl   %edi
        pushl   %eax
        pushl   %ebx
        pushl   %ecx
        pushl   %edx
        pushw   %es

        movl    20(%ebp), %esi          # ESI - Bitmap pointer
        movzwl  2(%esi), %ebx           # EBX - Height
        movzwl  (%esi), %eax            # EAX - Width
        movl    %eax, BMWidth
        imull   %ebx, %eax
        movl    %eax, PlaneInc          # Num. of bytes/plane

        # Clip against the top border.
        movl    _YClipMin, %edx
        subl    12(%ebp), %edx
        jle     YNotTopClip
        cmpl    %ebx, %edx
        jnl     YNotVisible
        movl    %edx, TopRow
        subl    %edx, %ebx
        addl    %edx, 12(%ebp)
        jmp     YVertClipDone

YNotVisible:
        movl    $1, %eax                # Set return code
        jmp     YDone

        # Clip against the bottom border.
YNotTopClip:
        movl    _YClipMax, %edx
        subl    12(%ebp), %edx
        js      YNotVisible
        movl    $0, TopRow
        cmpl    %ebx, %edx
        jge     YVertClipDone
        incl    %edx
        movl    %edx, %ebx

YVertClipDone:
        movl    %ebx, BMHeight
        movl    BMWidth, %eax
        imull   TopRow, %eax
        addl    $4, %eax                # Skip dimension bytes in source
        addl    %eax, %esi              # Skip top rows that aren't visible

        # Compute starting screen addr.
        movl    _ScrnLogicalByteWidth, %ebx
        movl    12(%ebp), %eax
        imull   %ebx, %eax
        addl    16(%ebp), %eax
        addl    $0xa0000, %eax
        movl    8(%ebp), %ecx           # ECX - X
        movl    %ecx, %edx
        shrl    $2, %edx
        addl    %edx, %eax
        movl    %eax, %edi              # EDI - starting screen address
        subl    BMWidth, %ebx           # Calc. diff. from end of bitmap in
        movl    %ebx, LineInc           #  curr. line to bitmap in next line

        movw    _core_select, %ax       # We're using post-DPMI extender
        movw    %ax, %es                # Use real area selector

        # Prepare VGA for CPU to video writes.
        movb    $0x11, %ah              # Map mask - loaded to rotate
        andl    $0x03, %ecx
        shlb    %cl, %ah
        movw    $0x3c4, %dx
        movb    $0x02, %al
        outb    %al, %dx
        incw    %dx
        movb    $4, Plane

YPlaneLoop:
        pushl   %edi                    # Save starting screen address
        pushl   %esi                    # Save bitmap address
        movl    BMHeight, %ebx
        movb    %ah, %al
        outb    %al, %dx

        movl    BMWidth, %ebp
YRowLoop:
        movl    %ebp, %ecx           # Reset column counter

YColLoop:        
        movb    (%esi), %al             # Get next source bitmap byte - Use lodsb
        incl    %esi
        orb     %al, %al                # If not zero, write to dest.
        jz      YNoPixel
        movb    %al, %es:(%edi)

YNoPixel:
        incl    %edi
        decl    %ecx
        jnz     YColLoop                # Loop if more columns left
        addl    LineInc, %edi           # Move to next row on screen
        decl    %ebx                    # Decrement row counter
        jnz     YRowLoop                # Jump if more rows left
        popl    %esi                    # Restore bitmap address and set to
        addl    PlaneInc, %esi          #  addr. of 1st. pixel in next plane
        popl    %edi                    # Restore starting screen address
        rolb    $1, %ah                 # Shift mask for next plane
        adcl    $0, %edi                # If carry, increment screen addr.
        decb    Plane                   # Decrement plane counter
        jnz     YPlaneLoop              # Jump if more planes left
        xorl    %eax, %eax              # Set return code

YDone:
        popw    %es                     # Restore registers.
        popl    %edx
        popl    %ecx
        popl    %ebx
        popl    %eax
        popl    %edi
        popl    %esi
        popl    %ebp
        ret


/*****************************************************************************
 *
 * x_set_cliprect(int left, int top, int right, int bottom);
 * Stack offset:     +8       +12      +16         +20
 *
 * Clipping region variables:  XClipMin, XClipMax, YClipMin, YClipMax
 *
 * NOTE: Clipping is byte oriented. "left" and "right" are in bytes not pixels.
 *       Only selected functions perform any clipping at all.
 *  
 *****************************************************************************/
        .align  4
_x_set_cliprect:
        pushl   %ebp
        movl    %esp, %ebp

        # Save all registers.
        pushl   %eax
        pushl   %ebx

        movl    8(%ebp), %eax
        movl    16(%ebp), %ebx
        cmpl    %eax, %ebx
        jns     CorrectXOrder
        xchgl   %eax, %ebx

CorrectXOrder:
        movl    %eax, _XClipMin
        movl    %ebx, _XClipMax
        movl    12(%ebp), %eax
        movl    20(%ebp), %ebx
        cmpl    %eax, %ebx
        jns     CorrectYOrder
        xchgl   %eax, %ebx

CorrectYOrder:
        movl    %eax, _YClipMin
        movl    %ebx, _YClipMax

        popl    %ebx                    # Restore registers.
        popl    %eax
        popl    %ebp
        ret



/*****************************************************************************
 *
 * x_put_PBM_clipx(int x, int y, int PageBase, BYTE * bitmap);
 * Stack offset:      +8    +12      +16           +20
 *
 * Clipping region variables:  XClipMin, XClipMax
 *
 * X and Y values may be negative.
 *  
 *****************************************************************************/
        .align  4
_x_put_PBM_clipx:
        pushl   %ebp
        movl    %esp, %ebp

        # Save all registers.
        pushl   %esi
        pushl   %edi
        pushl   %eax
        pushl   %ebx
        pushl   %ecx
        pushl   %edx
        pushw   %es

        xorl    %eax, %eax
        movl    %eax, CType             # Clear Clip type descision var
        movl    20(%ebp), %esi          # ESI - Bitmap pointer
        movzwl  2(%esi), %ebx           # EBX - Height
        movzwl  (%esi), %eax            # EAX - Width
        movl    8(%ebp), %edi           # EDI - X (pixel)
        movl    %edi, %ecx              # Save in ECX
        sarl    $2, %edi                # EDI - X (planar)

        # Clip against the left border.
        movl    _XClipMin, %edx
        subl    %edi, %edx
        jle     XNotLeftClipNM
        cmpl    %eax, %edx
        jnl     XNotVisibleNM
        addl    %edx, %edi
        movl    %edx, LeftSkip
        movl    %edx, DataInc
        subl    %edx, %eax
        movl    $1, CType
        jmp     XHorizClipDoneNM

XNotVisibleNM:
        movl    $1, %eax                # Set return code
        jmp     XDoneNM

XNotLeftClipNM:
        # Clip against the right border.
        movl    _XClipMax, %edx
        subl    %edi, %edx
        js      XNotVisibleNM
        movl    $0, LeftSkip
        movl    $0, DataInc
        cmpl    %eax, %edx
        jge     XHorizClipDoneNM
        incl    %edx
        subl    %edx, %eax
        movl    %eax, DataInc
        movl    %edx, %eax
        movl    $-1, CType

XHorizClipDoneNM:
        movl    %eax, BMWidth           # Save width and height of clipped image
        movl    %ebx, BMHeight

        movl    $4, %eax                # Skip dimension bytes in source
        addl    LeftSkip, %eax          # Skip clipped pixels in front of row
        addl    %eax, %esi

        # Compute starting screen addr.
        movl    _ScrnLogicalByteWidth, %ebx
        movl    12(%ebp), %eax
        imull   %ebx, %eax
        addl    16(%ebp), %eax
        addl    $0xa0000, %eax
        addl    %eax, %edi              # EDI - starting screen address
        subl    BMWidth, %ebx           # Calc. diff. from end of bitmap in
        movl    %ebx, LineInc           #  curr. line to bitmap in next line

        movw    _core_select, %ax       # We're using post-DPMI extender
        movw    %ax, %es                # Use real area selector

        # Prepare VGA for CPU to video writes.
        movb    $0x11, %ah              # Map mask - loaded to rotate
        andl    $0x03, %ecx
        shlb    %cl, %ah
        movw    $0x3c4, %dx
        movb    $0x02, %al
        outb    %al, %dx
        incw    %dx
        movb    $4, Plane

XPlaneLoopNM:
        pushl   %edi                    # Save starting screen address
        movl    BMHeight, %ebx
        movb    %ah, %al
        outb    %al, %dx

        movl    BMWidth, %ebp
XRowLoopNM:
        movl    %ebp, %ecx           # Reset column counter
        jecxz   XNoWidthNM

        shrl    $1, %ecx
        rep
        movsw
        adcl    $0, %ecx
        rep
        movsb

XNoWidthNM:
        addl    DataInc, %esi           # Move to next source row
        addl    LineInc, %edi           # Move to next row on screen
        decl    %ebx                    # Decrement row counter
        jnz     XRowLoopNM              # Jump if more rows left
        popl    %edi                    # Restore starting screen address
        rolb    $1, %ah                 # Shift mask for next plane
        jnb     XNoCarryNM              # Test for a plane transition

        # A transition has occurred from plane 3 to plane 0.  At this point,
        # if we started in plane n, we may need to go back and draw pixels in 
        # planes 0..n-1.  If we've clipped pixels to the left, increment the
        # working width and start putting pixels one byte earlier in the source.
        movl    CType, %eax
        addl    %eax, BMWidth           # Adjust bitmap width
        subl    %eax, DataInc           # Update vars. to account for new width
        subl    %eax, LineInc
        cmpl    $0, %eax                # Left or right clip?
        movb    $0x11, %ah              # Restore plane mask
        jg      XRightAdvanceNM         # Jump if clipping against the left edge
        incl    %edi
        jmp     XNoCarryNM

XRightAdvanceNM:
        decl    %esi                    # Start image processing one byte earlier

XNoCarryNM:
        decb    Plane                   # Decrement plane counter
        jnz     XPlaneLoopNM            # Jump if more planes left
        xor     %eax, %eax              # Set return code

XDoneNM:
        popw    %es                     # Restore registers.
        popl    %edx
        popl    %ecx
        popl    %ebx
        popl    %eax
        popl    %edi
        popl    %esi
        popl    %ebp
        ret


/*****************************************************************************
 *
 * x_put_PBM_clipxy(int x, int y, int PageBase, BYTE * bitmap);
 * Stack offset:       +8    +12      +16           +20
 *
 * Clipping region variables:  XClipMin, XClipMax, YClipMin, YClipMax
 *
 * X and Y values may be negative.
 *  
 *****************************************************************************/
        .align  4
_x_put_PBM_clipxy:
        pushl   %ebp
        movl    %esp, %ebp

        # Save all registers.
        pushl   %esi
        pushl   %edi
        pushl   %eax
        pushl   %ebx
        pushl   %ecx
        pushl   %edx
        pushw   %es

        xorl    %eax, %eax
        movl    %eax, CType             # Clear Clip type descision var
        movl    20(%ebp), %esi          # ESI - Bitmap pointer
        movzwl  2(%esi), %ebx           # EBX - Height
        movzwl  (%esi), %eax            # EAX - Width
        movl    %eax, BMWidth           # Save width
        imull   %ebx, %eax
        movl    %eax, PlaneInc          # Num. of bytes/plane
        movl    BMWidth, %eax           # Restore width
        movl    8(%ebp), %edi           # EDI - X (pixel)
        movl    %edi, %ecx              # Save in ECX
        sarl    $2, %edi                # EDI - X (planar)

        # Clip against the top border.
        movl    _YClipMin, %edx
        subl    12(%ebp), %edx
        jle     XYNotTopClipNM
        cmpl    %ebx, %edx
        jnl     XYNotVisibleNM
        movl    %edx, TopRow
        subl    %edx, %ebx
        addl    %edx, 12(%ebp)
        jmp     XYVertClipDoneNM

XYNotVisibleNM:
        movl    $1, %eax                # Set return code
        jmp     XYDoneNM

        # Clip against the bottom border.
XYNotTopClipNM:
        movl    _YClipMax, %edx
        subl    12(%ebp), %edx
        js      XYNotVisibleNM
        movl    $0, TopRow
        cmpl    %ebx, %edx
        jge     XYVertClipDoneNM
        incl    %edx
        movl    %edx, %ebx

XYVertClipDoneNM:
        # Clip against the left border.
        movl    _XClipMin, %edx
        subl    %edi, %edx
        jle     XYNotLeftClipNM
        cmpl    %eax, %edx
        jnl     XYNotVisibleNM
        addl    %edx, %edi
        movl    %edx, LeftSkip
        movl    %edx, DataInc
        subl    %edx, %eax
        movl    $1, CType
        jmp     XYHorizClipDoneNM

XYNotLeftClipNM:
        # Clip against the right border.
        movl    _XClipMax, %edx
        subl    %edi, %edx
        js      XYNotVisibleNM
        movl    $0, LeftSkip
        movl    $0, DataInc
        cmpl    %eax, %edx
        jge     XYHorizClipDoneNM
        incl    %edx
        subl    %edx, %eax
        movl    %eax, DataInc
        movl    %edx, %eax
        movl    $-1, CType

XYHorizClipDoneNM:
        movl    %eax, BMWidth           # Save width and height of clipped image
        movl    %ebx, BMHeight

        addl    DataInc, %eax           # EAX = Original width of image
        imull   TopRow, %eax            # Skip clipped top rows
        addl    $4, %eax                # Skip dimension bytes in source
        addl    LeftSkip, %eax          # Skip clipped pixels in front of row
        addl    %eax, %esi

        # Compute starting screen addr.
        movl    _ScrnLogicalByteWidth, %ebx
        movl    12(%ebp), %eax
        imull   %ebx, %eax
        addl    16(%ebp), %eax
        addl    $0xa0000, %eax
        addl    %eax, %edi              # EDI - starting screen address
        subl    BMWidth, %ebx           # Calc. diff. from end of bitmap in
        movl    %ebx, LineInc           #  curr. line to bitmap in next line

        movw    _core_select, %ax       # We're using post-DPMI extender
        movw    %ax, %es                # Use real area selector

        # Prepare VGA for CPU to video writes.
        movb    $0x11, %ah              # Map mask - loaded to rotate
        andl    $0x03, %ecx
        shlb    %cl, %ah
        movw    $0x3c4, %dx
        movb    $0x02, %al
        outb    %al, %dx
        incw    %dx
        movb    $4, Plane

XYPlaneLoopNM:
        pushl   %edi                    # Save starting screen address
        pushl   %esi                    # Save bitmap address
        movl    BMHeight, %ebx
        movb    %ah, %al
        outb    %al, %dx

        movl    BMWidth, %ebp
XYRowLoopNM:
        movl    %ebp, %ecx           # Reset column counter
        jecxz   XYNoWidthNM

        shrl    $1, %ecx
        rep
        movsw
        adcl    $0, %ecx
        rep
        movsb

XYNoWidthNM:
        addl    DataInc, %esi           # Move to next source row
        addl    LineInc, %edi           # Move to next row on screen
        decl    %ebx                    # Decrement row counter
        jnz     XYRowLoopNM             # Jump if more rows left
        popl    %esi                    # Restore bitmap address and set to
        addl    PlaneInc, %esi          #  addr. of 1st. pixel in next plane
        popl    %edi                    # Restore starting screen address
        rolb    $1, %ah                 # Shift mask for next plane
        jnb     XYNoCarryNM             # Test for a plane transition

        # A transition has occurred from plane 3 to plane 0.  At this point,
        # if we started in plane n, we may need to go back and draw pixels in 
        # planes 0..n-1.  If we've clipped pixels to the left, increment the
        # working width and start putting pixels one byte earlier in the source.
        movl    CType, %eax
        addl    %eax, BMWidth           # Adjust bitmap width
        subl    %eax, DataInc           # Update vars. to account for new width
        subl    %eax, LineInc
        cmpl    $0, %eax                # Left or right clip?
        movb    $0x11, %ah              # Restore plane mask
        jg      XYRightAdvanceNM        # Jump if clipping against the left edge
        incl    %edi
        jmp     XYNoCarryNM

XYRightAdvanceNM:
        decl    %esi                    # Start image processing one byte earlier

XYNoCarryNM:
        decb    Plane                   # Decrement plane counter
        jnz     XYPlaneLoopNM           # Jump if more planes left
        xor     %eax, %eax              # Set return code

XYDoneNM:
        popw    %es                     # Restore registers.
        popl    %edx
        popl    %ecx
        popl    %ebx
        popl    %eax
        popl    %edi
        popl    %esi
        popl    %ebp
        ret


/*****************************************************************************
 *
 * x_put_PBM_clipy(int x, int y, int PageBase, BYTE * bitmap);
 * Stack offset:      +8    +12      +16           +20
 *
 * Clipping region variables:  YClipMin, YClipMax
 *
 * X and Y values may be negative.
 *  
 *****************************************************************************/
        .align  4
_x_put_PBM_clipy:
        pushl   %ebp
        movl    %esp, %ebp

        # Save all registers.
        pushl   %esi
        pushl   %edi
        pushl   %eax
        pushl   %ebx
        pushl   %ecx
        pushl   %edx
        pushw   %es

        movl    20(%ebp), %esi          # ESI - Bitmap pointer
        movzwl  2(%esi), %ebx           # EBX - Height
        movzwl  (%esi), %eax            # EAX - Width
        movl    %eax, BMWidth
        imull   %ebx, %eax
        movl    %eax, PlaneInc          # Num. of bytes/plane

        # Clip against the top border.
        movl    _YClipMin, %edx
        subl    12(%ebp), %edx
        jle     YNotTopClipNM
        cmpl    %ebx, %edx
        jnl     YNotVisibleNM
        movl    %edx, TopRow
        subl    %edx, %ebx
        addl    %edx, 12(%ebp)
        jmp     YVertClipDoneNM

YNotVisibleNM:
        movl    $1, %eax                # Set return code
        jmp     YDoneNM

        # Clip against the bottom border.
YNotTopClipNM:
        movl    _YClipMax, %edx
        subl    12(%ebp), %edx
        js      YNotVisibleNM
        movl    $0, TopRow
        cmpl    %ebx, %edx
        jge     YVertClipDoneNM
        incl    %edx
        movl    %edx, %ebx

YVertClipDoneNM:
        movl    %ebx, BMHeight
        movl    BMWidth, %eax
        imull   TopRow, %eax
        addl    $4, %eax                # Skip dimension bytes in source
        addl    %eax, %esi              # Skip top rows that aren't visible

        # Compute starting screen addr.
        movl    _ScrnLogicalByteWidth, %ebx
        movl    12(%ebp), %eax
        imull   %ebx, %eax
        addl    16(%ebp), %eax
        addl    $0xa0000, %eax
        movl    8(%ebp), %ecx           # ECX - X
        movl    %ecx, %edx
        shrl    $2, %edx
        addl    %edx, %eax
        movl    %eax, %edi              # EDI - starting screen address
        subl    BMWidth, %ebx           # Calc. diff. from end of bitmap in
        movl    %ebx, LineInc           #  curr. line to bitmap in next line

        movw    _core_select, %ax       # We're using post-DPMI extender
        movw    %ax, %es                # Use real area selector

        # Prepare VGA for CPU to video writes.
        movb    $0x11, %ah              # Map mask - loaded to rotate
        andl    $0x03, %ecx
        shlb    %cl, %ah
        movw    $0x3c4, %dx
        movb    $0x02, %al
        outb    %al, %dx
        incw    %dx
        movb    $4, Plane

YPlaneLoopNM:
        pushl   %edi                    # Save starting screen address
        pushl   %esi                    # Save bitmap address
        movl    BMHeight, %ebx
        movb    %ah, %al
        outb    %al, %dx

        movl    BMWidth, %ebp
YRowLoopNM:
        movl    %ebp, %ecx           # Reset column counter

        shrl    $1, %ecx
        rep
        movsw
        adcl    $0, %ecx
        rep
        movsb
        
        addl    LineInc, %edi           # Move to next row on screen
        decl    %ebx                    # Decrement row counter
        jnz     YRowLoopNM              # Jump if more rows left
        popl    %esi                    # Restore bitmap address and set to
        addl    PlaneInc, %esi          #  addr. of 1st. pixel in next plane
        popl    %edi                    # Restore starting screen address
        rolb    $1, %ah                 # Shift mask for next plane
        adcl    $0, %edi                # If carry, increment screen addr.
        decb    Plane                   # Decrement plane counter
        jnz     YPlaneLoopNM            # Jump if more planes left
        xorl    %eax, %eax              # Set return code

YDoneNM:
        popw    %es                     # Restore registers.
        popl    %edx
        popl    %ecx
        popl    %ebx
        popl    %eax
        popl    %edi
        popl    %esi
        popl    %ebp
        ret

