/*
 * xcbitmap.c - linear bitmap compiler and blitter
 *
 * DESCRIPTION
 * Compiles linear bitmaps.  Memory must be pre-allocated.  Use
 * x_sizeof_cbitmap to determine how much memory you will need.
 * The reference in xlib60 and personal experience suggest that
 * a typical compiled bitmap requires about 4x the memory of it's linear
 * version.
 *
 * USAGE
 *
 * NOTES
 * Based on code from xlib60 for BC by Themie Goumas.
 *
 * REVISION HISTORY
 * Date         Reason
 * 27 Jun 95    Initial Release
 *  3 Aug 95    Changed bitmaps to have word w and h
 * 31 Aug 95    Fix to get rid of spurious warning message.
 *
 */

#include "defines.h"
#include "xlib.h"

/* Macros */
#define LBMHeight(lbitmap) (lbitmap[2] + (lbitmap[3] << 8))
#define LBMWidth(lbitmap) (lbitmap[0] + (lbitmap[1] << 8))
#define PBMHeight(pbitmap) (pbitmap[2] + (pbitmap[3] << 8))
#define PBMWidth(pbitmap) ((pbitmap[0] << 2) + (pbitmap[1] << 10))
#define PBMByteWidth(pbitmap) (pbitmap[0] + (pbitmap[1] << 8))

#define LBMPutPix(x,y,lbitmap,color)  \
	lbitmap[4 + (x) + (y) * LBMWidth(lbitmap)] = color

#define LBMGetPix(x,y,lbitmap)  \
	(lbitmap[4 + (x) + (y) * LBMWidth(lbitmap)])

/* 386 Opcodes */
#define ROL_AL          0xc0d0  /* rol al,1 */
#define SHORT_STORE_8   0x46c6  /* mov [esi + disp8], imm8 */
#define STORE_8         0x86c6  /* mov [esi + disp32], imm8 */
#define SHORT_STORE_32  0x46c7  /* mov [esi + disp8], imm16 */
#define STORE_32        0x86c7  /* mov [esi + disp32], imm16 */
#define ADC_SI_IMMED    0xd683  /* adc esi, imm8 */
#define OUT_AL          0xee    /* out dx, al */
#define RETURN          0xc3    /* ret */

#define WORD_PREFIX    0x66     /* 16-bit prefix */

int output_used = 0;

/* static variables to hold asm parameters */
static int x_pos, y_pos, page_offset;
static BYTE * spr_ptr;

/***********************************************************************
* void x_put_cbitmap(int XPos, int YPos,
*                    unsigned int PageOffset, BYTE * Sprite);
*
* Displays a compiled bitmap generated by x_compile_bitmap at given
* coordinates, on a given screen page.
*********************************************************************/
void x_put_cbitmap(int XPos, int YPos, int PageOffset, BYTE * Sprite)
{
  x_pos = XPos;
  y_pos = YPos;
  page_offset = PageOffset;
  spr_ptr = Sprite;

  asm ("
    pushl       %eax
    pushl       %ebx
    pushl       %ecx
    pushl       %edx
    pushl       %edi
    pushl       %esi
    pushw       %es

    movl        _ScrnLogicalByteWidth, %edx
    movl        _y_pos, %eax
    movl        _x_pos, %ecx
    imul        %edx
    movl        %ecx, %esi
    sarl        $2, %esi
    addl        %eax, %esi
    addl        _page_offset, %esi
    addl        $0xa0000, %esi
    addl        $128, %esi

    movb        $0x11, %ah
    andl        $0x03, %ecx
    movw        $0x3c4, %dx
    movb        $0x02, %al
    outb        %al, %dx
    incw        %dx
    movb        %ah, %al
    rolb        %cl, %al
    movl        _spr_ptr, %edi

    movw        %ds, %cx
    movw        %cx, %es
    movw        _core_select, %ebx
    movw        %bx, %ds

    call        %edi
    
    movw        %cx, %ds        /* restore old ds */

    popw        %es
    popl        %esi
    popl        %edi
    popl        %edx
    popl        %ecx
    popl        %ebx
    popl        %eax
  ");
}

void Emitb(char * output, BYTE x)
{
  *(output + output_used) = x;
  output_used++;
}

void Emitw(char * output, WORD x)
{
  *(output + output_used) = x & 255;
  *(output + output_used+1) = x >> 8;
  output_used += 2;
}

void Emitd(char * output, LONG x)
{
  *(output + output_used) = x & 255;
  *(output + output_used+1) = x >> 8;  
  *(output + output_used+2) = x >> 16; 
  *(output + output_used+3) = x >> 24;
  output_used += 4;
}

int x_compile_bitmap(int logical_screen_width, BYTE * bitmap, BYTE * output)
{
  long pos;
  int pix0 = 0;
  int pix1 = 0; 
  int pix2 = 0; 
  int pix3 = 0; 
  int numpix;
  int column = 0, set_column = 0;
  int scanx = 0, scany = 0;

  int height = LBMHeight(bitmap);
  int width = LBMWidth(bitmap);
  int margin = width - 1;
  int margin2 = margin - 4;
  int margin4 = margin - 12;

  output_used = 0;

  Emitb (output, OUT_AL);
  while (column < 4) 
  {
    numpix = 1;
    pix0 = LBMGetPix(scanx, scany, bitmap);
    if (pix0 != 0) 
    {
      if (set_column != column) 
      {
	Emitw (output, ROL_AL);
	Emitw (output, ADC_SI_IMMED);
	Emitb (output, 0);
	set_column++;
	Emitb (output, OUT_AL);
      }
      if (scanx <= margin2) 
      {
	pix1 = LBMGetPix(scanx + 4, scany, bitmap);
	if ((pix1 != 0) && (scanx <= margin4)) 
	{
	  numpix = 2;
	  pix2 = LBMGetPix(scanx + 8, scany, bitmap);
	  pix3 = LBMGetPix(scanx + 12, scany, bitmap);
	  if ((pix2 != 0) && (pix3 != 0)) 
	  {
	    numpix = 4;
	  }
	}
      }
      pos = (scany * logical_screen_width) + (scanx >> 2) - 128;
      if ((pos >= -128) && (pos <= 127)) 
      {
	if (numpix == 1) 
	{
	  Emitw (output, SHORT_STORE_8);
	  Emitb (output, pos);
	  Emitb (output, pix0);
	} 
	else 
	{
	  if( numpix != 4 )
	    Emitb (output, WORD_PREFIX);
	  Emitw (output, SHORT_STORE_32);
	  Emitb (output, pos);
	  Emitb (output, pix0);
	  Emitb (output, pix1);
	  if (numpix == 4) 
	  {
	    Emitb (output, pix2);
	    Emitb (output, pix3);
	  }
	}
      } 
      else 
      {
	if (numpix == 1) 
	{
	  Emitw (output, STORE_8);
	  Emitd (output, pos);
	  Emitb (output, pix0);
	} 
	else 
	{
	  if( numpix != 4 )
	    Emitb (output, WORD_PREFIX);
	  Emitw (output, STORE_32);
	  Emitd (output, pos);
	  Emitb (output, pix0);
	  Emitb (output, pix1);
	  if (numpix == 4) 
	  {
	    Emitb (output, pix2);
	    Emitb (output, pix3);
	  }
	}
      }
    }
    scanx += (numpix << 2);
    if (scanx > margin) 
    {
      scanx = column;
      scany++;
      if (scany == height) 
      {
	scany = 0;
	column++;
	scanx=column;
      }
    }
  }
  Emitb(output, RETURN);
  return(output_used);
}

int x_sizeof_cbitmap(int logical_screen_width, BYTE * bitmap)
{
  int pix0, pix1, pix2, pix3, numpix, pos;
  int column = 0, set_column = 0;
  int scanx = 0, scany = 0;
  int output_used = 1;    /* for RET */

  int height = LBMHeight(bitmap);
  int width = LBMWidth(bitmap);
  int margin = width - 1;
  int margin2 = margin - 4;
  int margin4 = margin - 12;

  output_used ++;      /* first out %al, %dx */
  while (column < 4) 
  {
    numpix = 1;
    pix0 = LBMGetPix(scanx, scany, bitmap);
    if (pix0 != 0) 
    {
      if (set_column != column) 
      {
	output_used += 5;
	set_column++;
	output_used++;
      }
      if (scanx <= margin2) 
      {
	pix1 = LBMGetPix(scanx + 4, scany, bitmap);
	if ((pix1 != 0) && (scanx <= margin4)) 
	{
	  numpix = 2;
	  pix2 = LBMGetPix(scanx + 8, scany, bitmap);
	  pix3 = LBMGetPix(scanx + 12, scany, bitmap);
	  if ((pix2 != 0) && (pix3 != 0)) 
	  {
	    numpix = 4;
	  }
	}
      }
      pos = (scany * logical_screen_width) + (scanx >> 2) - 128;
      if ((pos >= -128) && (pos <= 127)) 
      {
	if (numpix == 1) 
	{
	  output_used += 4;
	} 
	else 
	{
	  if( numpix != 4 )
	    output_used++;
	  output_used += 5;
	  if (numpix == 4)
	    output_used += 2;
	}
      } 
      else 
      {
	if (numpix == 1) 
	{
	  output_used += 7;
	} 
	else 
	{
	  if( numpix !=4 )
	    output_used++;
	  output_used += 8;
	  if (numpix == 4)
	    output_used += 2;
	}
      }
    }
    scanx += (numpix << 2);
    if (scanx > margin) 
    {
      scanx = column;
      scany++;
      if (scany == height) 
      {
	scany = 0;
	column++;
	scanx=column;   
      }
    }
  }
  return(output_used);
}
