/*
   This package contains functions to implement a set of drawing functions
   (BitBlt) to allow block copying to and from video memory.  The video modes
   currently supported are as follows:

    16 color       screen resolution
    --------       -----------------
    EGA/VGA        640x350/480
    SVGA           800x600, 1024x768, 1280x1024  (via VESA)

    256 color
    ---------
    VGA            320x200, 320x240 if supported
    SVGA           640x400/480, 800x600, 1024x768, 1280x1024  (via VESA)

    True color (via VESA)
    ----------
    SVGA           320x200, 640x480

    VGA paged mode
    --------------
    VGA-X          256x256, 320x200 ,320x240, 360x270, 400x300, 400x600

   See your favorite VESA paper for an explanation of the VESA specification
   for PC video programming.  I found most of the
   info used here on the net in a file called vesasp12.txt.
   Also used is the "VESA BIOS EXTENSION (VBE) Core Functions" document,
   version 2.0, revision 1.1,  available from the 
   Video Electronics Standards Association.

   Written by Arve Holmbo.
   1.0 Released to public domain March 1996.
   2.0 Update release Sep 1996
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <graph.h>
#include <i86.h>
#include <dos.h>

#include "vesa.h"
#include "vga16.h"
#include "vga256.h"
#include "vgapage.h"
#include "svga.h"
#include "wvga.h"
#include "pal.h"
#include "mem.h"
#include "pragmas.h"


typedef int   (*GetSetFunc)( uchar *, int, int, uchar *, int );
typedef void *(*MemCpyFunc)( void  *, const void *, size_t );
typedef int   (*drawFn)( int, int, int, int, uchar *);

static drawFn  drawFunc = NULL;
static drawFn  getFunc  = NULL;
static drawFn  drawTranspFunc = NULL;
static MemCpyFunc MemCopyFunction;
static SVGAdata   SvgaData;   /* SVGA info for current mode */

static int SVGA256drawBlock( int x, int y, int xlen, int ylen, uchar *src );
static int SVGA256getBlock(  int x, int y, int xlen, int ylen, uchar *dest );

static int SVGA16drawBlock( int x, int y, int xlen, int ylen, uchar *src );
static int SVGA16drawTranspBlock( int x, int y, int xlen, int ylen, uchar *src);
static int SVGA16getBlock( int x, int y, int xlen, int ylen, uchar *dest );
static int SVGA16drawGetBlock( int x, int y, int xlen, int ylen,
                               uchar *src, GetSetFunc func );
static int SVGA16getLine(uchar *adr, int x, int xlen, uchar *dest, int page);
static int SVGA16setLine(uchar *adr, int x, int xlen, uchar *src,  int page);


typedef struct _internal_state
{
  short isTrueColorMode;
  short isExtendedTrueC;
  short isHicolMode;
  short initOk;
} SVGAintState;

static SVGAintState SvgaState = { 0, 0, 0, 0 };



/* Convenient short forms create slightly more readable code.
*/
#define  BYPP  SvgaData.BytesPerPixel /* Pixel size in # of bytes */
#define  BYPL  SvgaData.BytesPerLine  /* Bytes per scanline */
#define  XRES  SvgaData.XPixels       /* Max pixels in X */
#define  YRES  SvgaData.YPixels       /* Max pixels in Y */




/* Does all initialization and returns EXIT_SUCCESS if init went OK,
   else EXIT_FAILURE, along with reason ( *SvgaStatus )
   */
int  SVGAinit( short mode,          /* Desired video mode */
               int   iUseWatcomLib, /* To use Watcom functions or not */
               int   iUseLinearBuffer, /* To use a linear frame buffer */
               int   iDoSetMode,    /* Clear if mode not to be set here */
               int  *SvgaStatus     /* O: Status */)
{
  int        iStat = EXIT_SUCCESS, status;
  VESAdata  *VesaData;    /* In case video card supports VESA */
  uchar     *tmp;

  drawFunc = getFunc = drawTranspFunc = NULL;
  *SvgaStatus = SvgaModeValid;

  /* Set up some default values. */
  SvgaData.Granularity = 4;  /* this was found to be same in all CL modes */
  SvgaData.Mode   = mode;    /* Remember mode */
  SvgaData.WinSize= 64;      /* Window size */
  SvgaData.VidAdd = 0xA0000; /* Works in most VGA modes, not CGA, Hercules */
  SvgaData.BytesPerPixel = 1;  /* Default that works the best */

  /* Initialise internal SVGA state */
  SvgaState.isTrueColorMode = 0;
  SvgaState.isHicolMode = 0;
  SvgaState.isExtendedTrueC = 0;
  SvgaState.initOk = 0;

  SVGAgetNonSVGAresolution( mode, &SvgaData.XPixels, &SvgaData.YPixels );

  /* If the user has specified to use the Watcom graphics library's
     primitives or given mode is not supported, then set up pointers
     to those functions and return.
     */
  if ( iUseWatcomLib || mode < 16 )
  {
    drawFunc = WVGAdrawBlock;
    getFunc  = WVGAgetBlock;
    drawTranspFunc = WVGAdrawTranspBlock;
    WVGAsetResolution( SvgaData.XPixels, SvgaData.YPixels );
    *SvgaStatus    = SvgaVesaModeValid;

    if ( !_setvideomode( mode ))
      return EXIT_FAILURE;
    SvgaState.initOk = 1;
    return   EXIT_SUCCESS;
  }

  /* Process regular EGA/VGA modes and VGA mode X first
   */
  if ( mode >= VGAPAGEmode256x256 && mode <= VGAPAGEmode400x600 )
  {
    /* VGA mode X/paged 256 color lowres modes */
    VGAPAGEinit( mode );

    drawFunc       = VGAPAGEdrawBlock;
    getFunc        = VGAPAGEgetBlock;
    drawTranspFunc = VGAPAGEdrawTranspBlock;
  }
  else if ( mode == _ERESCOLOR || mode == _VRES16COLOR) /* 640x350/480 E/VGA */
  {
    SvgaData.BytesPerLine = 80;

    drawFunc       = VGA16drawBlock;
    getFunc        = VGA16getBlock;
    drawTranspFunc = VGA16drawTranspBlock;
  }
  else if ( mode == _MRES256COLOR )
  {
    SvgaData.BytesPerLine = 320;       /* 320x200 256 color lo-res VGA */

    drawFunc       = VGA256drawBlock;
    getFunc        = VGA256getBlock;
    drawTranspFunc = VGA256drawTranspBlock;
  }

  else if ( mode < _URES256COLOR )
  {
    /* Modes below 256-color lowres SVGA not currently supported
     */
    *SvgaStatus = SvgaModeNotSupported;
    iStat = EXIT_FAILURE;
  }

  if ( mode >= _URES256COLOR )
  {
    /* Attempt to use VESA to set SVGA video modes of 640x480x256 or higher.
    */
    if ( iDoSetMode )
      iStat = VESAsetMode( mode, iUseLinearBuffer, &status );
    else
      iStat = VESAqueryMode( mode, iUseLinearBuffer, &status );
    if ( iStat == EXIT_SUCCESS )
    {
      *SvgaStatus = SvgaVesaModeValid;

      VesaData = VESAgetData();

      SvgaData.Granularity  = VesaData->Granularity;
      SvgaData.BytesPerLine = VesaData->BytesPerLine;
      SvgaData.WinSize      = (int )VesaData->WinSize;
      SvgaData.XPixels      = VesaData->XPixels;
      SvgaData.YPixels      = VesaData->YPixels;
      SvgaData.VidAdd       = VesaData->Asegment * 16;
      SvgaData.BitsPerPixel = VesaData->BitsPerPixel;
      SvgaData.Banks        = VesaData->Banks;
      SvgaData.BankSize     = VesaData->BankSize;
      SvgaData.RedMaskSiz   = VesaData->RedMaskSiz;   /* Used for 15/16 bit */
      SvgaData.GreenMaskSiz = VesaData->GreenMaskSiz;
      SvgaData.BlueMaskSiz  = VesaData->BlueMaskSiz;
      SvgaData.RedFieldPos  = VesaData->RedFieldPos;   /* Used for 15/16 bit */
      SvgaData.GreenFieldPos= VesaData->GreenFieldPos;
      SvgaData.BlueFieldPos = VesaData->BlueFieldPos;

      /* SVGA modes set from here
       */
      if ( VesaData->MemoryModel == 3 )   /* 4-planar modes */
      {
        drawFunc       = SVGA16drawBlock;    /* 16-color modes */
        getFunc        = SVGA16getBlock;
        drawTranspFunc = SVGA16drawTranspBlock;
      }
      else if ( VesaData->MemoryModel == 4 )  /* 256 color modes */
      {
        drawFunc       = SVGA256drawBlock;   /* 256-color 640x400/480, 800x600,*/
        getFunc        = SVGA256getBlock;    /* 1024x768 and 1280x1024 modes */
        drawTranspFunc = SVGA256drawBlock;
      }

      /* Bytes per pixel only apply to 256 and higher color modes
         Trick: Add 1 to get correct # of bytes for 15-bit modes.
      */
      SvgaData.BytesPerPixel = max( 1, (SvgaData.BitsPerPixel+1) / 8 );

      if ( mode == _SVGAmode320true ||   /* true color modes */
           mode == _SVGAmode640true ||
           mode == _SVGAmode800true )
      {
        drawFunc       = SVGA256drawBlock;
        getFunc        = SVGA256getBlock;
        drawTranspFunc = SVGA256drawBlock;

        SvgaState.isTrueColorMode = 1;

        /* This is intended used to enable properly copying of
           24-bit color information into a 32-bit pixel location.
           Some future release.  See the use of memcpy functions
           in the drawing functions below.
           */
        if ( SvgaData.BitsPerPixel > 24 )
          SvgaState.isExtendedTrueC = 1;   /* Assume 32-bit++  pixel */
        else
          SvgaState.isExtendedTrueC = 0;
      }
      else if ( mode == _SVGAmode640hi15 ||  /* 15/16-bit hi-color modes */
                mode == _SVGAmode640hi16 ||
                mode == _SVGAmode800hi15 ||
                mode == _SVGAmode800hi16 ||
                mode == _SVGAmode1024hi15 ||
                mode == _SVGAmode1024hi16 )
      {
        if ( SvgaData.RedMaskSiz   == 5 && SvgaData.BlueMaskSiz == 5 &&
           ( SvgaData.GreenMaskSiz == 5 || SvgaData.GreenMaskSiz == 6 ))
        {
          /* Support basic 15- and 16-bit color modes */

          SvgaState.isHicolMode = 1;

          drawFunc       = SVGA256drawBlock;
          getFunc        = SVGA256getBlock;
          drawTranspFunc = SVGA256drawBlock;
        }
        else
        {
          *SvgaStatus = SvgaModeNotSupported;
          iStat = EXIT_FAILURE;
        }        
      }
    }
    else
    {
      /* VESA set mode failed
      */
      if ( status == VesaInitFail )
        *SvgaStatus = SvgaInitFail;
      else if ( status == VesaModeInValid )
        *SvgaStatus = SvgaModeInValid;
      else if ( status == VesaInitFailLFB )
        *SvgaStatus = SvgaLFBnotAvail;
      else
        *SvgaStatus = SvgaModeNotSupported;
      iStat = EXIT_FAILURE;
    }
  }
  else if ( mode <= _MRES256COLOR )
  {
    if ( iDoSetMode )
      if ( !_setvideomode( mode ))
        iStat = EXIT_FAILURE;
  }

  /* Calculate start address of next page in video memory.
     Also, check if we want a linear frame buffer and if one exists
  */
  if ( iStat == EXIT_SUCCESS && iUseLinearBuffer && mode >= _URES256COLOR )
  {
    if ( VesaData->PhysBasePtr != 0 &&
       ( VesaData->ModeAttr & 0x80 ))
    {
      /* Point video start address at start of linear frame buffer and
         make the window size = total amount of video mem. to avoid
         the bank switching
         */
      SvgaData.WinSize = VesaData->TotalMemory;  /*Assume all video memory OK*/

      /* Get start address to linear frame buffer */
      if (( tmp = VESAgetLinearAddr()) == NULL )
      {
        *SvgaStatus = SvgaLinAdrNotAvail;
        iStat = EXIT_FAILURE;
      }
      else
        SvgaData.VidAdd = (uint )tmp;
    }
    else
    {
      /* Linear frame buffer not available. Return error
      */
      *SvgaStatus = SvgaLFBnotAvail;
      iStat = EXIT_FAILURE;
    }    
  }

  SvgaData.WinSize *= 1024;  /* Convert to number of bytes */
  SvgaData.VidOffs = SvgaData.VidAdd + SvgaData.WinSize;


  SvgaState.initOk = ( iStat == EXIT_SUCCESS );

  /* Clear display memory */
  if ( iDoSetMode && iStat == EXIT_SUCCESS )
    SVGAclearScreen();

  return ( iStat );
}




/* Cleanup SVGA initial state
*/
void  SVGAend( void )
{
  if ( SvgaState.initOk )
  {
    SvgaState.initOk = 0;
    VESAcleanup();
  }
}




/* Return pointer to SVGA data structure.
   The 2 info functions above should have been called a priori.
*/
SVGAdata  *SVGAgetData( void )
{
  return  &SvgaData;
}



/* Clear display
*/
int  SVGAclearScreen( void )
{
  int    i, iStat = EXIT_SUCCESS;
  int    dx, dy;
  uchar *p, *cp;
  if ( SvgaState.initOk )
  {
    dx = SvgaData.XPixels;
    dy = SvgaData.YPixels;
    if (( p = calloc( dx, dy/2 )) != NULL )
    {
      if (( cp = SVGAbufferConvert( p, dx, dy/2 )) != NULL )
      {
        /* Clear the 2 halves */
        SVGAdrawBlock( 0, 0,    dx, dy/2, cp );
        SVGAdrawBlock( 0, dy/2, dx, dy/2, cp );
        free( cp );
      }
      free( p );
    }
  }
  else iStat = EXIT_FAILURE;
  return iStat;
}




/*******************************************************/
/* Drawing routines start here */





/* Draw to display the block, starting at x, y.
   Length of each line in the block is xlen.
 */
int  SVGAdrawBlock( int x, int y, int xlen, int ylen, uchar *src )
{
  MemCopyFunction = memcpy;

  if ( ylen > 25 )        // larger cursors cause flicker, but the ..
    VGAwaitVertRetrace(); // wait for retrace don't work on all SVGAs..

  if ( drawFunc && SvgaState.initOk )
    return (*drawFunc)( x, y, xlen, ylen, src );

  return ( EXIT_FAILURE );
}




/* Draw pixels from the src that are not NULL.  This differs from
   SVGAdrawBlock only in the way pixels are drawn.  The effect is that
   when a rectangle with zero pixels in it is drawn the background
   pixels are not overwritten by a null-pixel in the source.
   This can be used to draw any sprite shape on the screen.
   This function uses the MEMtcpy routine to copy transparently.
*/
int  SVGAdrawTranspBlock( int x, int y, int xlen, int ylen, uchar *src )
{
  MemCopyFunction = MEMtcpy;

  if ( ylen > 25 )
    VGAwaitVertRetrace();

  if ( drawTranspFunc && SvgaState.initOk )
    return (*drawTranspFunc)( x, y, xlen, ylen, src );

  return EXIT_FAILURE;
}





/* Get from display the block, starting at x, y.
   Length of each line in the block is xlen.
   Opposite function of SVGAdrawBlock
 */
int  SVGAgetBlock( int x, int y, int xlen, int ylen, uchar *dest )
{
  if ( getFunc && SvgaState.initOk )
    return (*getFunc)( x, y, xlen, ylen, dest );
  return ( EXIT_FAILURE );
}





/* 256 color generic get block function
   XRES is video resolution in X for the current video mode.
   YRES is that in the Y direction.
   BYPP is number of bytes per pixel in the current mode.
   BYPL is the number of bytes per scanline. Note that BYPL cannot be
           derived from the XRES and BYPP together, but is mode dependent.
*/
static int SVGA256getBlock( int x, int y, int xlen, int ylen, uchar *dest )
{
  int  page,    len, clippedLen;
  int  offset,  left_of_page;
  uint currPage;

  // Perform clipping
  clippedLen = min( xlen, XRES - x );   /* do clipping */
  clippedLen *= BYPP;           /* Convert from pixels to # of bytes */

  ylen       = min( ylen, YRES - y );
  if ( clippedLen <= 0 || ylen <= 0 )
    return -1;

  offset = y * BYPL + x * BYPP;  /* # of bytes off video start adrs */
  page   = offset / SvgaData.WinSize;
  left_of_page = (page+1)*SvgaData.WinSize - offset;

  currPage = VESAgetPage();
  if ( page != currPage )
    VESAsetPage( page );

  while ( ylen-- )
  {
    len = ( left_of_page >= clippedLen ? clippedLen : left_of_page );
    memcpy( dest, (char *)(SvgaData.VidOffs-left_of_page), len);
    if ( len != clippedLen )
    {
      VESAsetPage( ++page );
      left_of_page = SvgaData.WinSize; /* start next page and complete line */
      memcpy( dest+len, (char *)(SvgaData.VidOffs-left_of_page),
              clippedLen-len );
      left_of_page -= ( BYPL - len );
    }
    else
    {
      left_of_page -= BYPL;
      if ( left_of_page <= 0 )
      {
        VESAsetPage( ++page );   /* Start next page */
        left_of_page += SvgaData.WinSize;
      }
    }

    dest += xlen * BYPP;   /* Each pixel in source assumped BYPP bytes */
  }

  /* Must make sure that the current video page is the same as when
     this routine was entered because of a 'bug' in the Watcom
     graph. libraries that assume a certain video page
     */
  if ( page != currPage )
    VESAsetPage( currPage );   /* Restore current video page */

  return ( EXIT_SUCCESS );
}






/* Draw block in SVGA 256 color modes, transparent or straight..
 * That depends on what MemCopyFunction is set to.
 */
static int SVGA256drawBlock( int x, int y, int xlen, int ylen, uchar *src )
{
  int  page,    len, clippedLen;
  int  offset,  left_of_page;
  uint currPage;

  // Perform clipping
  clippedLen = min( xlen, XRES - x );   /* do clipping */
  clippedLen *= BYPP;           /* Convert from pixels to # of bytes */

  ylen       = min( ylen, YRES - y );
  if ( clippedLen <= 0 || ylen <= 0 )
    return -1;

  offset = y * BYPL + x * BYPP;
  page   = offset / SvgaData.WinSize;
  left_of_page = (page+1)*SvgaData.WinSize - offset;

  currPage = VESAgetPage();
  if ( page != currPage )
    VESAsetPage( page );

  while ( ylen-- )
  {
    len = ( left_of_page >= clippedLen ? clippedLen : left_of_page );
    (*MemCopyFunction)((char *)(SvgaData.VidOffs-left_of_page), src, len);
    if ( len != clippedLen )
    {
      VESAsetPage( ++page );
      left_of_page = SvgaData.WinSize; /* start next page and complete line */
      (*MemCopyFunction)((char *)(SvgaData.VidOffs-left_of_page), src+len,
                         clippedLen-len );
      left_of_page -= ( BYPL - len );
    }
    else
    {
      left_of_page -= BYPL;
      if ( left_of_page <= 0 )
      {
        VESAsetPage( ++page );   /* Start next page */
        left_of_page += SvgaData.WinSize;
      }
    }
    src += xlen * BYPP;  /* Each pixel in source assumed BYPP bytes */
  }

  if ( page != currPage )
    VESAsetPage( currPage );   /* Restore current video page */

  return ( EXIT_SUCCESS );
}









/*
  SVGA drawing routines for 16 color mode.  Set pixel values in SVGA
  16-color graphics mode with values in *buf.  The pixel line set is
  horizontal starting at (x,y) and line is xlen pixels long.
  Each pixel value is assumed to be one byte of data.
  Taken from book by Wilton p.142 and p.96 and reworked for efficiency.
*/
static int Transparent;
static int SVGA16drawBlock( int x, int y, int xlen, int ylen, uchar *src )
{
  int iStat;
  Transparent = 0;
  VGA16setupRegs();
  iStat = SVGA16drawGetBlock( x, y, xlen, ylen, src, SVGA16setLine );
  VGA16resetRegs();
  return iStat;
}

static int SVGA16drawTranspBlock( int x, int y, int xlen, int ylen, uchar *src)
{
  int iStat;
  Transparent = 1;
  VGA16setupRegs();
  iStat = SVGA16drawGetBlock( x, y, xlen, ylen, src, SVGA16setLine );
  VGA16resetRegs();
  return iStat;
}

static int SVGA16getBlock( int x, int y, int xlen, int ylen, uchar *dest )
{
  Transparent = 0;
  return SVGA16drawGetBlock( x, y, xlen, ylen, dest, SVGA16getLine );
}

static int SVGA16drawGetBlock( int x, int y, int xlen, int ylen,
                               uchar *src, GetSetFunc getset_func )
{
  int  page, offset;
  char *adr;
  int  clippedLen;
  uint currPage;

  // Perform clipping
  clippedLen = min( xlen, SvgaData.XPixels - x );   /* do clipping */
  ylen       = min( ylen, SvgaData.YPixels - y );
  if ( clippedLen <= 0 || ylen <= 0 )   return EXIT_FAILURE;

  offset = y * BYPL + x / 8;   /* 8 pixels/byte in video mem. */

  page   = offset / SvgaData.WinSize;
  adr    = (char *)( SvgaData.VidAdd + offset - page*SvgaData.WinSize );

  currPage = VESAgetPage();
  if ( page != currPage )
    VESAsetPage( page );

  while ( ylen-- )
  {
    page = (*getset_func)( adr, x, clippedLen, src, page );

    adr += BYPL;
    if ( adr >= SvgaData.VidOffs )
    {
      adr -= SvgaData.WinSize;
      VESAsetPage( ++page );
    }
    src += xlen;
  }

  if ( page != currPage )
    VESAsetPage( currPage );   /* Restore current video page */

  return EXIT_SUCCESS;
}







/* Draw line in 16-color SVGA modes
*/
static int  SVGA16setLine(uchar *adr,
                          int    x,
                          int    xlen,
                          uchar *BufP,
                          int    page )
{
  int pixels_left, len;
  int  PIXPBYT = 8;  /* 8 pixels / byte in video mem. (4 bits/pix)/4 planes */

  /* Figure out number of pixels left in this video page
  */
  pixels_left = (SvgaData.VidOffs - (int )adr) * PIXPBYT - (x & 7);

  VGA16setTransparent( Transparent );
  
  len = min( xlen, pixels_left );

  VGA16drawLine( x, adr, len, BufP );
  if ( len < xlen )
  {
    VESAsetPage( ++page );
    adr = (uchar *)SvgaData.VidAdd;  /* Reset to next page in video memory */
    VGA16drawLine( x+len, adr, xlen-len, BufP+len );
  }

  return  page;
}
        




/* Get line in 16-color SVGA modes
*/
static int SVGA16getLine(uchar *adr,
                         int    x,
                         int    xlen,
                         uchar *dest,
                         int    page )
{
  uchar bits, tmp;
  int   i;
  int   PIXPBYT = 8;

  int pixels_left, len;

  /* Figure out number of pixels left in this video page
  */
  pixels_left = (SvgaData.VidOffs - (int )adr) * PIXPBYT - (x & 7);
  
  len = min( xlen, pixels_left );

  VGA16getLine( x, adr, len, dest );
  if ( len < xlen )
  {
    VESAsetPage( ++page );
    adr = (uchar *)SvgaData.VidAdd;  /* Reset to next page in video memory */
    VGA16getLine( x+len, adr, xlen-len, dest+len );
  }
  return  page;
}





/******************************************************************

   The following pixel manipulation routines have some
   limitations
   */


/* Put pixel at (x, y).
*/
void SVGAputPixel( int x, int y, uchar color )
{
  if ( SvgaData.Mode >= _SVGAXMODE256x256 &&
       SvgaData.Mode <= _SVGAXMODE400x600 )
    VGAPAGEputPixel( x, y, color );
  else if ( SvgaData.Mode <= _XRES256COLOR )
  {
    _setcolor( color );
    _setpixel( x, y );
  }
}

/* Get pixel at (x, y).
*/
int  SVGAgetPixel( int x, int y )
{
  if ( SvgaData.Mode >= _SVGAXMODE256x256 &&
       SvgaData.Mode <= _SVGAXMODE400x600 )
    return VGAPAGEgetPixel( x, y );
  else if ( SvgaData.Mode <= _XRES256COLOR )
  {
    return _getpixel( x, y );
  }
  else
    return 0;
}

/* Moveto/lineto routines to emulate Microsoft/Watcom functionality
*/
void  SVGAmoveto( int x, int y )
{
  if ( SvgaData.Mode >= _SVGAXMODE256x256 &&
       SvgaData.Mode <= _SVGAXMODE400x600 )
    VGAPAGEmoveto( x, y );
  else if ( SvgaData.Mode <= _XRES256COLOR )
    _moveto( x, y );
}

void SVGAlineto( int x, int y, uchar Color )
{
  if ( SvgaData.Mode >= _SVGAXMODE256x256 &&
       SvgaData.Mode <= _SVGAXMODE400x600 )
  {
    VGAPAGEsetColor( Color );
    VGAPAGElineto( x, y );   /* Works only for hor/ver lines */
  }
  else if ( SvgaData.Mode <= _XRES256COLOR )
  {
    _setcolor( Color );   /* Works only in Watcom modes */
    _lineto( x, y );
  }
}



/* Print failure string according to reason at init time
*/
int  SVGAfailPrint( int reason )
{
  switch ( reason )
  {
    case  SvgaModeValid:
      printf( "SVGA mode is valid\n" );
      break;
    case  SvgaModeInValid:
      printf( "Mode is invalid\n" );
      break;
    case  SvgaInitFail:
      printf( "SVGA init failed\n" );
      break;
    case  SvgaModeNotSupported:
      printf( "Mode is not supported\n" );
      break;
    case  SvgaLFBnotAvail:
      printf( "SVGA linear frame buffer not available\n" );
      break;
    case  SvgaLinAdrNotAvail:
      printf( "SVGA linear frame buffer addressing error\n" );
      break;
    default:
      printf( "SVGA init error.  Reason (%d) not known\n", reason );
  }
  return reason;
}





/* Return TRUE if a HIGH- or TRUE color mode in use
*/
int  SVGAisTrueColorMode( void )
{
  int iStat = (int )( SvgaState.isHicolMode | SvgaState.isTrueColorMode );
  return iStat;
}

int  SVGAis16colorMode( void )
{
  if ( SvgaState.initOk )
  {
    return ( SVGA16colorMode( SvgaData.Mode ));
  }
  return FALSE;
}




/* Get the video resolution for the current video mode.
   SVGAinit must have been successfully called prior to
   calling this
   */
int SVGAgetVideoRes( int *Xpix, int *Ypix )
{
  if ( SvgaState.initOk )
  {
    *Xpix = SvgaData.XPixels;
    *Ypix = SvgaData.YPixels;
  }
  return (int )SvgaState.initOk;
}




/* Allocate a buffer large enough to hold Ypix * Xpix pixels
   Minimum size for a pixel unit is 1 byte.
   On successful return the buffer will be zeroed.
*/
uchar *SVGAbufferAlloc( uint Xpix, uint Ypix )
{
  uchar *p = NULL;

  if ( SvgaState.initOk )
    p = (uchar *)calloc( BYPP, Xpix * Ypix );

  return p;
}





/* Routine to convert a standard 256 color VGA buffer of pixel
   values to a buffer of high - or true color pixel value correspondents.
   A buffer to hold the hi/true color pixel values is created and returned.
   When the application is done using the buffer it should be freed.
   */
static uchar rgb[ 3*256 ];
uchar *SVGAbufferConvert( uchar *buf, uint Xpix, uint Ypix )
{
  int i, j, index, shift_mask, shift_mask2;
  uchar *cbuf = NULL;
  int  redindex, greenindex, blueindex;

  if ( !SvgaState.initOk )
    return cbuf;

  if (( cbuf = (uchar *)SVGAbufferAlloc( Xpix, Ypix )) != NULL )
  {
    /* Use the current palette values to create the proper values to
       store into the true color buffer.
    */
    VESAgetPal( 0, 256, rgb );
    if ( SvgaState.isTrueColorMode )
    {
      /* Convert R G B bytes' position in video memory to index into
         buffer memory
         */
      redindex   = SvgaData.RedFieldPos / 8;
      greenindex = SvgaData.GreenFieldPos / 8;
      blueindex  = SvgaData.BlueFieldPos / 8;
      for ( i=j=0; i < Xpix*Ypix; i++, j += BYPP )
      {
        index = buf[ i ] * 3;   /* Copy from palette table */
        cbuf[ j+redindex  ] = rgb[ index ];
        cbuf[ j+greenindex] = rgb[ index+1 ];
        cbuf[ j+blueindex ] = rgb[ index+2 ];
      }
    }
    else if ( SvgaState.isHicolMode )
    {
      if ( SvgaData.GreenMaskSiz == 6 )
        shift_mask = 2, shift_mask2 = 3;    /* 16-bit color */
      else
        shift_mask = 3, shift_mask2 = 2;    /* 15-bit */
      for ( i=j=0; i < Xpix*Ypix; i++, j += BYPP )
      {
        uchar tmp;
        uchar r, g, b;

        index = buf[ i ] * 3;

        /* This is a kludge that demands a cleanup someday..
        */
        if ( SvgaData.RedFieldPos > SvgaData.BlueFieldPos )
        {
          r = rgb[ index ] >> 3;
          g = rgb[ index+1 ] >> shift_mask;   /* 5,6,5- or 5,5,5 bit */
          b = rgb[ index+2 ] >> 3;
        }
        else
        {
          b = rgb[ index ] >> 3;
          g = rgb[ index+1 ] >> shift_mask;   /* 5,6,5- or 5,5,5 bit */
          r = rgb[ index+2 ] >> 3;
        }

        cbuf[ j ] = b & 0x1F;
        tmp = g << 5;
        cbuf[ j ] |= tmp;
  
        cbuf[ j+1 ] = g >> 3;
        tmp = r << shift_mask2;
        cbuf[ j+1 ] |= tmp;
      }
    }
    else if ( SVGA16colorMode( SvgaData.Mode ))
    {
      /* Reduce number of colors from 256 to 16
      */
      j = Xpix*Ypix;
      for ( i=0; i < j; i++ )   cbuf[i] = PALget16colorIndex( buf[ i ] );
    }
  }
  return  cbuf;
}




/* Set new palette.  This is used prior to drawing graphics in
   hi-color or true color modes only.
*/
int  SVGAsetPal( uchar *palValues )
{
  PALsetCustomTable( palValues );
  return ( EXIT_SUCCESS );
}


/* Set display logical scan length in pixels.  The new scan line
   length is returned on success, zero else.  This applies to VESA
   modes only.
*/
int  SVGAsetLogicalScanLen( int iLen )
{
  return VESAsetLogicalScanLen( iLen );
}


/* Set display start address at (x,y).  This applies only to VESA modes.
   Could also do this in mode-X.
   Returns EXIT_SUCCESS on success.
*/
int  SVGAsetDisplayStart( int x, int y )
{
  return VESAsetDisplayStart( x, y );
}



typedef struct
{
  short xres, yres;
} SVGAmodeResMap;

static SVGAmodeResMap ResolTab[] =
{
  /* Resolutions for non-SVGA- and X-modes listed in this table,
     basically to avoid a lot of 'if-then-elses' in the init code.
   */
  40, 25,
  40, 25,
  80, 25,
  80, 25,
  320, 200,
  320, 200,
  640, 200,
  80, 25,
0,0,
0,0,
0,0,
  720, 350,
0,0,
  320, 200,
  640, 200,
  640, 350,
  640, 350,
  640, 480,
  640, 480,
  320, 200,

  256, 256, /* X-modes */
  320, 200,
  320, 240,
  360, 270,
  400, 300,
  400, 600
};

#define MaxNonSVGAmode  (sizeof(ResolTab) / sizeof( SVGAmodeResMap ))

/* If TRUE, then valid values for xresolution and yresolution are returned.
   */
int  SVGAgetNonSVGAresolution( short mode, ushort *XPixels, ushort *YPixels )
{
  if ( mode >= 0 && mode < MaxNonSVGAmode )
  {
    *XPixels = ResolTab[mode].xres;
    *YPixels = ResolTab[mode].yres;
    return 1;
  }
  else
    return 0;
}
