/* This package contains functions to implement a partial buffer
   refresh and retrieval.
   Written by Arve Holmbo.
   Released to public domain March 1997.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "svga.h"
#include "bitb.h"
#include "pragmas.h"

static uchar *AllocBuffer( uchar *bufPtr, int xsiz, int ysiz, int *siz );
static uchar *screen = NULL;   // 'Previous' screen cache buffer
static uchar *screen2= NULL;   // Current screen cache buffer
static int    siz, siz2, lx, ly, ux, uy;
static int    pixelsiz=1;      // Color depth in number of pixels

/* Update the background as required.  If the old and new rectangles
   don't overlap, then refresh display memory, else store background
   in screen cache memory for later refresh.
*/
void  BITBdrawBlock( int old_x, int old_y, int old_xsiz, int old_ysiz,
                     int     x, int y,     int xsiz,     int ysiz,
                     uchar  *backgnd )
{
  int  old_endx, old_endy, endx, endy;

  old_endx = old_x + old_xsiz;
  old_endy = old_y + old_ysiz;
  endx     = x + xsiz;
  endy     = y + ysiz;

  if (  old_endx < x || endx < old_x || old_endy < y || endy < old_y )
    // Simply refresh background and quit.
    SVGAdrawBlock( old_x, old_y, old_xsiz, old_ysiz, backgnd );
  else
  {
    /* Use screen buffer as cache between system memory & display mem.
    */
    if (( screen = AllocBuffer( screen, old_xsiz, old_ysiz, &siz )) != NULL )
      memcpy( screen, backgnd, siz*pixelsiz );
  }
}





/* Get new background into the background buffer.  Use partially
   screen cache buffer, partially display memory.
*/
void  BITBgetBlock( int old_x, int old_y, int old_xsiz, int old_ysiz,
                    int     x, int y,     int xsiz,     int ysiz,
                    uchar  *backgnd )
{
  int    big_xsiz, big_ysiz, dx, dx2, dy;
  uchar *s, *d;
  int    old_endx, old_endy, endx, endy, xoffs, xoffs2;
  int    rectUnion, i;

  old_endx = old_x + old_xsiz;
  old_endy = old_y + old_ysiz;
  endx     = x + xsiz;
  endy     = y + ysiz;

  if ( old_endx < x || endx < old_x ||
       old_endy < y || endy < old_y )
  {
    rectUnion = 0;
    lx = x, ly = y;
    ux = endx, uy = endy;
    big_xsiz = xsiz;
    big_ysiz = ysiz;
  }
  else
  {
    rectUnion = 1;
    lx = min( old_x, x );
    ly = min( old_y, y );
    ux = max( endx, old_endx );
    uy = max( endy, old_endy );
    big_xsiz = ux - lx;
    big_ysiz = uy - ly;
  }

  if (( screen2 = AllocBuffer( screen2, big_xsiz, big_ysiz, &siz2 )) == NULL )
    return;

  /* Get whole area from display memory in to screen cache # 2
  */
  if ( SVGAgetBlock( lx, ly, big_xsiz, big_ysiz, screen2 ) !=
       EXIT_SUCCESS )
    return;
  if ( rectUnion )
  {
    if ( screen != NULL )
    {
      /* Update screen cache # 2 with sub area in screen cache # 1
      */
      dx = old_x - lx;
      dy = old_y - ly;
      xoffs  = 0;
      xoffs2 = (dx + dy*big_xsiz) * pixelsiz;
      dx = old_xsiz * pixelsiz;
      dx2= big_xsiz * pixelsiz;
      for ( i=0; i < old_ysiz; i++, xoffs += dx, xoffs2 += dx2 )
        memcpy( &screen2[ xoffs2 ], &screen[ xoffs ], dx );
    }
  }

  /* Get new background */
  dx = x - lx;
  dy = y - ly;
  xoffs = 0;
  xoffs2= (dx + dy*big_xsiz) * pixelsiz;
  dx = pixelsiz * xsiz;
  dx2= pixelsiz * big_xsiz;
  for ( i=0; i < ysiz; i++, xoffs += dx, xoffs2 += dx2 )
    memcpy( &backgnd[ xoffs ], &screen2[ xoffs2 ], dx );
}



/* Draw the buffer to display memory in transparent mode, i.e. don't draw
   pixels with color index 0.
*/
void  BITBdrawTranspBlock( int x, int y, int xsiz, int ysiz,  char *buf )
{
  int    big_xsiz, big_ysiz, dx, dy, i, j, soffs, doffs;
  uchar *s, *d;

  if ( screen2 == NULL )  { putch(7); return; }

  big_xsiz = ux - lx;
  big_ysiz = uy - ly;

  soffs = pixelsiz * xsiz;
  doffs = pixelsiz * big_xsiz;
  dx = x - lx;
  dy = y - ly;
  s  = buf;
  d  = &screen2[ (dx + dy * big_xsiz)*pixelsiz ];
  for ( i=0; i < ysiz; i++ )
  {
    for ( j=0; j < soffs; j++ )    if ( s[j] )  d[j] = s[j];
    s += soffs;
    d += doffs;
  }  
  SVGAdrawBlock( lx, ly, big_xsiz, big_ysiz, screen2 );
}



/* Erase the background buffer.  This is used as indication that the
   content of screen buffer is out of date.
 */
void  BITBdropBackground( void )
{
  if ( screen ) free( screen );
  screen = NULL;
}






/* Where the new background is different from the old background and
   the cursor has no pixel set, update the old with the value of the
   pixel of the new background
   */
int  BITBupdateBackground(uchar          *old_backgnd, /* IO */
                          uchar          *new_backgnd, /* I */
                          uchar          *sprite,      /* I */
                          int             sprtSiz      /* I */)
//                          VMOUSEcurShape *cursor       /* I */)
{
  int  i, siz = sprtSiz;

#ifdef DONT_WORK
  SVGAtrueT *buf, *old, *new;

  if ( !SVGAisTrueColorMode())
#endif

  {
    for ( i=0; i < siz; i++ )
    {
      if ( !sprite[i] )
      {
        /* Non-sprite pixels in sprite rectangle
         */
        old_backgnd[i] = new_backgnd[i];     
      }
      else
      {
        /* This is the tricky part.  If the sprite and the new background
           had the same color at a paricular pixel, how would we know if
           it was a background pixel or a sprite pixel?
           One solution is to make sure the sprite uses unique colors.
           */
        if ( sprite[i] != new_backgnd[i] )   /* New background may have */
          old_backgnd[i] = new_backgnd[i];   /* overwritten sprite pixels */
      }
    }
  }


#ifdef DONT_WORK
  else
  {
    /* Should really make this function's code more elegant.
       Case for C++ ?
       */
    buf = (SVGAtrueT *)&sprite[0];    /* 3 bytes per pixel here */
    old = (SVGAtrueT *)&old_backgnd[0];
    new = (SVGAtrueT *)&new_backgnd[0];

    for ( i=0; i < siz; i++, buf++, old++, new++ )
    {
      if ( buf->r == 0 && buf->g == 0 && buf->b == 0 )
      {
        /* Non-sprite pixels in sprite rectangle
         */
        *old = *new;
      }
      else
      {
        /* This is the tricky part.  If the sprite and the new background
           had the same color at a paricular pixel, how would we know if
           it was a background pixel or a sprite pixel?
           One solution is to make sure the sprite uses unique colors.
           */
        if ( buf->r != new->r ||
             buf->g != new->g ||
             buf->b != new->b )    /* New background may have */
          *old = *new;             /* overwritten sprite pixels */
      }
    }
  }
#endif

  return 0;
}








/* Allocate buffer large enough to hold a rectangle of size xsiz, ysiz
*/
static uchar *AllocBuffer( uchar *bufPtr, int xsiz, int ysiz, int *siz )
{
  SVGAdata *sd;

  if ( xsiz == 0 || ysiz == 0 )
  {
    *siz = 0;
    if ( bufPtr ) free( bufPtr );
    bufPtr = NULL;
  }
  else
  if ( bufPtr == NULL || *siz < xsiz * ysiz )
  {
    sd = SVGAgetData();
    pixelsiz = sd->BytesPerPixel;  /* Colordepth = bytes per pixel */

    if ( bufPtr ) free( bufPtr );
    *siz = xsiz * ysiz;
    bufPtr = SVGAbufferAlloc( xsiz, ysiz );
  }

  return  bufPtr;
}
