/*
   This package contains the functions to implement a mouse cursor that
   works in some of the most 'popular' PC video modes.  If you want a
   text cursor, then this is not for you.
   Part of the stuff in here is based on material found in the Watcom manual,
   other parts are based on things people have been posting on the net.
   An example of the latter is the CopyDs2Es() function which ensure that
   ES=DS in the mouse ISR below.

   Written by Arve Holmbo.
   Released to public domain March 1996.
*/
static char  VMOUSEstring[] = "VMOUSE2.2 by Arve Holmbo";

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <conio.h>
#include <i86.h>
#include <graph.h>

#define   VMOUSE_PACKAGE
#include  "vmouse.h"
#include  "svga.h"
#include  "wvga.h"
#include  "vga.h"
#include  "lock.h"
#include  "scan.h"
#include  "bitb.h"
#include  "int.h"
#include  "pragmas.h"


/* It was found that the mouse driver returns every 8th coordinate count.
   Therefore, the resolution must be increased by dividing the returned
   coordinate values to give a resolution of 1 pixel.  This value is
   RESOLUTION
   */
#define  RESOLUTION   8


/* Mask to detect a mouse button event
*/
#define VMOUSEclickEvent   0x7E


/* Buffer to store mouse button events.  EVENTBUFSIZ must be
             n
   a number 2
*/
#define   EVENTBUFSIZ  1023

static volatile uchar eventBuf[ EVENTBUFSIZ+1 ];
static volatile uint  inPtr = 0, outPtr = 0, outPtr2 = 0;
static volatile int   vminx, vminy, vmaxx, vmaxy;
static volatile int   isDrawing = 0;

static int   DrawCursor( int DrawFlag );
static void  Move16to256Index( void );
static int   InstallKbd( void );
static void  KeyBend( void );
static void  AddEvent( uchar Event );

#pragma inline  AddEvent;


/*
 * Mouse control to handle mouse events.  Supports all VGA and some SVGA modes
 */
static int       MousePresent = 0;     /* assume no mouse as default */
static int       MouseHidden  = 1;     /* Initially, mouse is hidden */

/* Current cursor shape data */
static VMOUSEcurShape  cursor = { 0,0,0,0,NULL };
static short     videoMode;     /* Current video mode */


static void   (__interrupt __far *OrigKbdHandler)();


/* Data touched at mouse callback time -- they are in a structure to
  simplify calculating the size of the region to lock.
*/
struct callback_data
{
  ushort mouse_code;
  ushort mouse_bx;
  ushort mouse_cx;
  ushort mouse_dx;
  short  mouse_si;
  short  mouse_di;
} vcbd = { 0 };

int     VMOUSEx, VMOUSEy;  /* where the mouse currently is */
int     VMOUSEdummy;       /* to delineate data segment to lock. */

int     VMOUSEbufferedX, VMOUSEbufferedY;


extern void    CopyDs2Es( void );
#pragma inline CopyDs2Es;
#pragma aux    CopyDs2Es = "push ds" "pop  es";


/* Mouse callback routine
*/
void _loadds far vclick_handler (int m_ax, int m_cx, int m_dx)
{
#pragma aux vclick_handler parm [EAX][ECX][EDX]

// m_ax;   mouse events
// m_bx;   button state
// m_cx;   x-coordinate
// m_dx;   y-coordinate
// m_si;   last raw mickey-y cnt
// m_di;   last raw mickey-x cnt

  /* Set ES = DS as BIOS code as well as compiler generated code accesses
     data via both the ES and DS registers.  This should always be 1st
     code line
   */
  CopyDs2Es();
  AddEvent((uchar )m_ax );
  if ( m_ax & VMOUSEclickEvent )
  {
    /* Generate an artificial mouse event to reset mouse cursor when a
       button event has been detected
       */
    AddEvent( VMOUSEclickEvent );
  }

  VMOUSEx = m_cx/RESOLUTION;   /* Get current mouse coordinates */
  VMOUSEy = m_dx/RESOLUTION;
}




/* Keyboard handler to generate simulated mouse events.
   This handler snoops the keys pressed and gets the generated
   scan codes which are then converted to mouse events.
*/
static volatile int leftd=0, rightd=0;
static void interrupt  VmouseKbdHandler( void )
{
  uint   cc;
  uchar  _event;
  extern uchar getkey( void );     /* Get keyboard scan code */
#pragma aux getkey = "in  al, 60h" value [al];
#define MOVE  8

  _event = VMOUSEnonEvent;
  cc     = getkey();

  if ( cc < 0xBC )   // happens to be greatest code
  {
    if ( cc == SCANctrlDown || cc == SCANenterDown)
    {
      if ( !leftd ) { leftd = 1; _event = VMOUSEleftPres; }
    }
    else if ( cc == SCANctrlUp || cc == SCANenterUp)
    {
      leftd = 0; _event = VMOUSEleftRele;
    }
    else if ( cc == SCANaltDown )
    {
      if ( !rightd ) { rightd = 1; _event = VMOUSErightPres; }
    }
    else if ( cc == SCANaltUp )
    {
      rightd = 0;  _event = VMOUSErightRele;
    }
    else if ( cc == SCANup || cc == SCANdown || cc == SCANleft || cc == SCANright )
    {
      _event = VMOUSEmoved;
      if      ( cc == SCANup    && VMOUSEy > (vminy+MOVE)) VMOUSEy -= MOVE;
      else if ( cc == SCANdown  && VMOUSEy < (vmaxy-MOVE)) VMOUSEy += MOVE;
      else if ( cc == SCANright && VMOUSEx < (vmaxx-MOVE)) VMOUSEx += MOVE;
      else if ( cc == SCANleft  && VMOUSEx > (vminx+MOVE)) VMOUSEx -= MOVE;
    }
  }

  if ( _event != VMOUSEnonEvent )
  {
    AddEvent( _event );
    if ( _event & VMOUSEclickEvent )
    {
      /* Generate an artificial mouse event to reset mouse cursor when a
         button event has been detected
         */
      AddEvent( VMOUSEclickEvent );
    }
  }

  _chain_intr( OrigKbdHandler );
}


/*
  Dummy function so we can calculate size of
  code to lock (vcbc_end - vclick_handler)
*/
void vcbc_end (void) {}



/*
* VMOUSEinit: Initialize the mouse driver if exist and return EXIT_SUCCESS,
* otherwise, return EXIT_FAILURE.
* If maxX and maxY are zero on input, then these values will be derived
* automatically from the current video mode.
*/
int  VMOUSEinit( short VGAmode,         /* Video mode */
                 VMOUSEinitType *init,  /* Various input parameters */
                 int minX, int maxX, int minY, int maxY /* mouse region */)
{
  union  REGS  inregs, outregs;
  struct SREGS sregs;
  int          reason;
  void (far *function_ptr)();
  char        *p;
  int          xr, yr;

  MouseHidden = 1;    /* set global flag that mouse cursor is hidden */
  videoMode   = VGAmode;        /* Remember video mode */
  if ( SVGAinit( VGAmode,
                 (int )init->UseWatcomLib,
                 (int )init->UseLFB,
                 (int )init->SetVideoMode,
                 &reason ) != EXIT_SUCCESS )
  {
    _setvideomode( 3 );
    SVGAfailPrint( reason );
    (void )getch();
    return EXIT_FAILURE;
  }

  if ( maxX == 0 && maxY == 0 )
  {
    /* If undefined then set max x and y screen resolution parameters
    */
    if ( SVGAgetVideoRes( &xr, &yr ))
    {
      maxX = xr-1; maxY = yr-1;
    }
    else
      maxX = maxY = 200;  /* Some useful default if fatal error */
  }

  inregs.w.ax = 0;
  int386( VMOUSEint, &inregs, &outregs );  /* Initialize mouse */
  if ( !outregs.w.ax )
  {
//    _setvideomode( 3 );
//    printf("Mouse initialization failed\n" );
//    return  EXIT_FAILURE; /* if no mouse */
    MousePresent = 1;    /* Allow keyboard use instead */
    if ( InstallKbd() != EXIT_SUCCESS )
      return  EXIT_FAILURE;
  }
  else
    MousePresent = 1;       /* Remember that we have a mouse */


  /* lock data and code segments that we don't want swapped out in
     an VMM environment.  Note the 4 ints for VMOUSExxx data
     */
  if ((! LOCKregion( &vcbd, sizeof(vcbd)+4*sizeof(int))) ||
      (! LOCKregion ((void near *) vclick_handler,
         (char *)vcbc_end - (char near *)vclick_handler)))
  {
//    printf( "locks failed\n" );
    return  EXIT_FAILURE;
  }

  /* install click handler */

  segread( &sregs );
  inregs.w.ax = 0xC;
  inregs.w.cx = 0x007F;   /* Generate all buttons press and release events*/
  function_ptr = (void (*)(void ))vclick_handler;
  inregs.x.edx = FP_OFF (function_ptr);
  sregs.es     = FP_SEG (function_ptr);
  int386x( VMOUSEint, &inregs, &outregs, &sregs);


  /* Set mouse active range rectangle
  */
  VMOUSEsetVerticalRange(   minY, maxY );
  VMOUSEsetHorisontalRange( minX, maxX );


  inregs.w.ax = 0xF; /* Set mickey to pixel ratio to 1 for good sensitivity */
  inregs.w.cx = 1;
  inregs.w.dx = 1;
  int386( VMOUSEint, &inregs, &inregs );


  /* Set up color indexing if 256 color mode
  */
  Move16to256Index();


  /* Set default cursor shape */
  return  VMOUSEsetCursor( VMOUSEcurPointer );
}



/* Functions to set horisontal and vertical mouse pointer ranges
*/
void VMOUSEsetHorisontalRange( int minX, int maxX )
{
  union REGS  regs;  
  regs.x.eax = 7;       /* Horizontal mouse region */
  regs.x.ecx = minX * RESOLUTION;
  regs.x.edx = maxX * RESOLUTION;
  int386( VMOUSEint, &regs, &regs );

  vminx = minX, vmaxx = maxX;
}

void VMOUSEsetVerticalRange( int minY, int maxY )
{
  union REGS  regs;
  regs.x.eax = 8;       /* Vertical mouse region */
  regs.x.ecx = minY * RESOLUTION;
  regs.x.edx = maxY * RESOLUTION;
  int386( VMOUSEint, &regs, &regs );

  vminy = minY, vmaxy = maxY;
}


/* VMOUSEsetPos:  Set mouse cursor position
*/
void    VMOUSEsetPos( int x, int y )
{
  union REGS regs;

  /* Offset for hotspot */
  x -= cursor.xhot;
  y -= cursor.yhot;

  /* kick start mouse coordinates */
  VMOUSEx = VMOUSEbufferedX = x;
  VMOUSEy = VMOUSEbufferedY = y;

  x *= RESOLUTION;
  y *= RESOLUTION;

  regs.x.eax = 4;       /* Position mouse cursor */
  regs.x.ecx = x;
  regs.x.edx = y;
  int386( VMOUSEint, &regs, &regs );
}




/* VMOUSEshow: Show the mouse cursor
*/
void  VMOUSEshow()
{
  if ( !MouseHidden ) return;
  MouseHidden = 0;
  DrawCursor( TRUE );
}



/* VMOUSEhide: Hide the mouse cursor
*/
void  VMOUSEhide()
{
  if ( MouseHidden ) return;
  MouseHidden = 1;       /* turn on flag to indicate that mouse is off */
  DrawCursor( FALSE );   /* Hide cursor */
}



/* Get the application's coordinates of the mouse pointer.
   These are the real mouse coordinates offset for hotspot.
   This function also updates the buffered mouse coordinates
   variables VMOUSEbufferedX and VMOUSEbufferedY.
*/
void VMOUSEgetCoord( int *x, int *y )
{
  /* Get mouse ISR's current coordinates and buffer them until
     next time this function is called.
   */
  VMOUSEbufferedX = VMOUSEx;
  VMOUSEbufferedY = VMOUSEy;

  *x = VMOUSEbufferedX + cursor.xhot;  /* Correct for hot spot offsets */
  *y = VMOUSEbufferedY + cursor.yhot;
}



/* Reset mouse 
*/
void  VMOUSErestore( void )
{
  union REGS inregs, outregs;

  if ( MousePresent )
  {
    MousePresent = 0;   /* turn it off for now */

    inregs.w.ax = 0;
    int386( VMOUSEint, &inregs, &outregs );  /* ReInitialize mouse */

    SVGAend();
  }
}



/* Return mouse hide status
*/
int  VMOUSEhidden( void )  { return MouseHidden; }




/* Routine to return the next buffered mouse event.
   This buffer is simply a ring buffer, where events
   are copied to the buffer by the mouse ISR, and read
   by this routine.  Also, this routine draws the mouse
   cursor when the static MouseHidden flag is FALSE.
   
   Special processing is performed when the read event
   = VMOUSEclickEvent.  This means that the cursor is to
   be redrawn after a real mouse click event has been processed.
   The VMOUSEclickEvent is an artificial event generated by the
   mouse ISR to allow us to redraw the mouse cursor.
*/
uint VMOUSEgetEvent( int *x, int *y )
{
  uint       Event;
  
  VMOUSEgetCoord( x, y );   /* Update application mouse coordinates. */

  if ( inPtr == outPtr )  return 0;   /* no event */

  Event  = eventBuf[ outPtr++ ];     /* Get next event */
  outPtr &= EVENTBUFSIZ;            /* Advance ring buffer pointer */


  if ( Event == VMOUSEclickEvent )
  {
    /* If DrawCursor has been called after the mouse cursor was hidden
       when a button was pressed then don't get the new background as there
       may well be a new cursor there now.
       */
    if ( !MouseHidden)
      DrawCursor( TRUE );   /* Redisplay cursor */

    return 0;      /* Don't return this event */
  }

  else if ( Event & VMOUSEclickEvent )
    DrawCursor( FALSE ); /* Hide mouse cursor if button has been pressed*/

  else if ( !MouseHidden )
  {
    /* Draw the mouse at any new position
     */
    DrawCursor( TRUE );
  }

  return ( Event );
}






/*********************************************************************/
/* Color defines for cursor shape */
/* Regular VGA colors are used for 16 color modes */

/* This code is a bit iffy in that it 'allocates' two indices in the
   256-color palette table.  If you don't like this way of doing
   cursor color inversion, then use your own custom cursors.
 */

#define VMOUSEc16to256     240  /* increment to get from 16 to 256 color
                                   indexing */

#define VMOUSEdefColor       8  /* Cursor shape definition */
#define VMOUSEfillColor     15  /* Cursor shape internal color */
#define VMOUSEhDefColor   (VMOUSEdefColor +VMOUSEc16to256) /*Inverse fill.*/
#define VMOUSEhFillColor  (VMOUSEfillColor+VMOUSEc16to256) /*Inverse shape def*/

#define I   VMOUSEdefColor
#define X   VMOUSEfillColor


/* Palette values for black and white colors
*/
#define P_BLACK  0
#define P_WHITE  0x3F


/* Data for the clock cursor
*/
static uchar clock[] = 
{
   0,0,0,0,0,0,I,I,I,I,0,0,0,0,0,0,
   0,0,0,0,0,I,I,I,I,I,I,0,0,0,0,0,
   0,0,0,I,I,X,X,I,I,X,X,I,I,0,0,0,
   0,0,I,X,X,X,X,I,I,X,X,X,X,I,0,0,
   0,I,X,X,X,X,X,X,X,X,X,X,X,X,I,0,
   0,I,X,X,X,X,X,X,X,X,X,X,I,X,I,0,
   I,X,X,X,X,X,X,X,X,X,X,I,X,X,X,I,
   I,X,X,X,X,X,X,X,X,X,I,X,X,X,X,I,
   I,I,X,X,X,X,X,X,X,I,X,X,X,X,I,I,
   I,I,X,X,X,X,X,X,I,X,X,X,X,X,I,I,
   I,X,X,X,X,X,X,I,I,X,X,X,X,X,X,I,
   I,X,X,X,X,X,X,I,I,X,X,X,X,X,X,I,
   0,I,X,X,X,X,X,I,I,X,X,X,X,X,I,0,
   0,I,X,X,X,X,X,X,X,X,X,X,X,X,I,0,
   0,0,I,X,X,X,X,X,X,X,X,X,X,I,0,0,
   0,0,0,I,I,X,X,I,I,X,X,I,I,0,0,0,
   0,0,0,0,0,I,X,I,I,X,I,0,0,0,0,0,
   0,0,0,0,0,I,I,I,I,I,I,0,0,0,0,0,
   0,0,0,0,0,0,I,I,I,I,0,0,0,0,0,0
};



/* Data for the standard pointer cursor
*/
static uchar pointer[16*24] =
{
  I,I,I,0,0,0,0,0,0,0,0,0,0,0,0,0,
  I,I,I,I,0,0,0,0,0,0,0,0,0,0,0,0,
  I,I,X,X,I,0,0,0,0,0,0,0,0,0,0,0,
  I,I,X,X,X,I,0,0,0,0,0,0,0,0,0,0,
  I,I,X,X,X,X,I,0,0,0,0,0,0,0,0,0,
  I,I,X,X,X,X,X,I,0,0,0,0,0,0,0,0,
  I,I,X,X,X,X,X,X,I,0,0,0,0,0,0,0,
  I,I,X,X,X,X,X,X,X,I,0,0,0,0,0,0,
  I,I,X,X,X,X,X,X,X,X,I,0,0,0,0,0,
  I,I,X,X,X,X,X,X,X,X,X,I,0,0,0,0,
  I,I,X,X,X,X,X,X,X,X,X,X,I,0,0,0,
  I,I,X,X,X,X,X,X,X,X,X,X,X,I,0,0,
  I,I,X,X,X,X,X,X,X,X,X,X,X,X,I,0,
  I,I,X,X,X,X,X,X,X,X,X,X,X,X,X,I,
  I,I,X,X,X,X,X,X,X,X,X,X,X,X,X,I,
  I,I,X,X,X,X,X,X,X,I,I,I,I,I,I,I,
  I,I,X,X,X,X,X,X,X,I,I,I,I,I,I,0,
  I,I,X,X,X,I,I,I,X,X,X,I,0,0,0,0,
  I,I,X,I,I,0,I,I,X,X,X,I,0,0,0,0,
  I,I,X,I,I,0,0,I,X,X,X,I,0,0,0,0,
  I,I,I,0,0,0,0,0,I,X,X,X,I,0,0,0,
  0,0,0,0,0,0,0,0,I,X,X,X,I,0,0,0,
  0,0,0,0,0,0,0,0,I,X,X,X,I,0,0,0,
  0,0,0,0,0,0,0,0,0,I,I,I,I,0,0,0
};




/* Size of question mark cursor is 15 x 20
*/
static uchar QuestionMark[] =
{
  0,0,0,X,X,X,X,X,X,X,X,X,X,0,0,
  0,0,0,X,X,X,X,X,X,X,X,X,X,0,0,
  0,0,0,X,X,X,X,X,X,X,X,X,X,0,0,
  X,X,X,X,X,0,0,0,0,0,X,X,X,X,X,
  X,X,X,X,X,0,0,0,0,0,X,X,X,X,X,
  0,0,0,0,0,0,0,0,0,0,X,X,X,X,X,
  0,0,0,0,0,0,0,0,0,0,X,X,X,X,X,
  0,0,0,0,0,0,0,0,0,0,X,X,X,X,X,
  0,0,0,0,0,0,0,0,X,X,X,X,X,0,0,
  0,0,0,0,0,0,0,0,X,X,X,X,X,0,0,
  0,0,0,0,0,X,X,X,X,X,0,0,0,0,0,
  0,0,0,0,0,X,X,X,X,X,0,0,0,0,0,
  0,0,0,0,0,X,X,X,X,X,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,X,X,X,X,X,0,0,0,0,0,
  0,0,0,0,0,X,X,X,X,X,0,0,0,0,0,
  0,0,0,0,0,X,X,X,X,X,0,0,0,0,0,
  0,0,0,0,0,X,X,X,X,X,0,0,0,0,0,
  0,0,0,0,0,X,X,X,X,X,0,0,0,0,0
};



/*
 * Data for the crosshair cursor. 441 bytes
 */
uchar crossPicture[] =
{
//   21, 21,   /* xlen and ylen of button */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  X,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,X,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,X,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,X,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,X,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  X,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,X,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,X,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  X,X,X,X,X,X,X,X,0,0,0,X,X,X,X,
  X,X,X,X,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,X,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,X,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,X,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,X,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  X,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,X,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,X,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,X,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0
};



uchar handPicture[720] =
{
  0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,0,0,0,0
  ,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,15,15
  ,15,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0
  ,0,4,4,4,15,15,15,4,15,15,4,0,0,0,0,0,0,0
  ,0,0,0,0,0,0,4,15,15,4,15,15,15,4,15,15,4,4
  ,4,4,0,0,0,0,0,0,0,0,0,0,4,15,15,4,15,15
  ,15,4,15,15,15,4,15,15,4,0,0,0,0,0,0,0,0,0
  ,4,15,15,4,15,15,15,4,15,15,15,4,15,15,4,7,0,0
  ,0,0,0,0,0,0,4,15,15,4,15,15,15,4,15,15,15,4
  ,15,15,4,7,0,0,0,0,0,0,0,0,4,15,15,4,15,15
  ,15,4,15,15,15,4,15,15,4,7,0,0,0,0,0,0,0,0
  ,4,15,15,4,15,15,15,4,15,15,15,4,15,15,4,7,0,0
  ,0,4,4,7,0,0,4,15,15,4,15,15,15,4,15,15,15,4
  ,15,15,4,7,0,0,4,15,15,4,7,0,4,15,15,15,15,15
  ,15,15,15,15,15,15,15,15,4,7,0,0,4,15,15,4,7,0
  ,4,15,15,15,15,15,15,15,15,15,15,15,15,15,4,7,0,0
  ,4,15,15,15,4,4,4,15,15,15,15,15,15,15,15,15,15,15
  ,15,15,4,7,0,0,0,4,15,15,15,4,4,15,15,15,15,15
  ,15,15,15,15,15,15,15,15,4,7,0,0,0,4,15,15,15,15
  ,15,15,15,15,15,15,15,15,15,15,15,15,15,15,4,7,0,0
  ,0,0,4,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15
  ,15,15,4,7,0,0,0,0,4,15,15,15,15,15,15,15,15,15
  ,15,15,15,15,15,15,15,15,4,7,0,0,0,0,0,4,15,15
  ,15,15,15,15,15,15,15,15,15,15,15,15,15,15,4,7,0,0
  ,0,0,0,4,15,15,15,15,15,15,15,15,15,15,15,15,15,15
  ,15,4,7,7,0,0,0,0,0,0,4,15,15,15,15,15,15,15
  ,15,15,15,15,15,15,15,4,7,0,0,0,0,0,0,0,0,4
  ,4,15,15,15,15,15,15,15,15,15,15,15,15,4,7,0,0,0
  ,0,0,0,0,0,0,0,4,15,15,15,15,15,15,15,15,15,15
  ,4,7,0,0,0,0,0,0,0,0,0,0,0,4,15,15,15,15
  ,15,15,15,15,15,15,4,7,0,0,0,0,0,0,0,0,0,0
  ,0,0,4,15,15,15,15,15,15,15,15,15,4,7,0,0,0,0
  ,0,0,0,0,0,0,0,0,4,15,15,15,15,15,15,15,15,15
  ,4,7,0,0,0,0,0,0,0,0,0,0,0,0,4,15,15,15
  ,15,15,15,15,15,4,4,7,0,0,0,0,0,0,0,0,0,0
  ,0,0,0,4,4,4,4,4,4,4,4,7,7,0,0,0,0,0
  ,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,0
  ,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,7
  ,7,7,7,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0
  ,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};


/* List of availiable cursors
*/
static  VMOUSEcurShape  builtInCursor[ VMOUSEcursors ] =
{
  16, 24, 0, 0, pointer,
  16, 19, 7, 9, clock,
  15, 20, 7, 9, QuestionMark,
  21, 21,10,10, crossPicture,
  24, 30,10,10, handPicture
};


static int  BuiltInPointer( uchar *ptr )
{
  int i;
  for ( i=0; i < VMOUSEcursors; i++ )
    if ( ptr == builtInCursor[i].buf == ptr )
      return 1;
  return 0;
}





/* Convert the specified mouse object from 256 color standard VGA to
   TRUE color mode or 16-color mode.  Status is returned at *piStat.
*/
static  VMOUSEcurShape ConvertColor( VMOUSEcurShape *c, int *piStat )
{
  static uchar *buf = NULL;
  static VMOUSEcurShape shape;
  uchar *tmp;
  int    i;

  /* Be sure not to allocate the buffer at the exact same spot or else
     Draw will fail.
  */
  tmp = buf;
  if (( buf = (uchar *)SVGAbufferConvert( c->buf, c->xsiz, c->ysiz )) != NULL )
  {
    if ( !BuiltInPointer( c->buf ))   free( c->buf );

    shape = *c;  /* Copy mouse object */
    shape.buf = buf;
    *piStat = EXIT_SUCCESS;
  }
  else
    *piStat = EXIT_FAILURE;

  if ( tmp ) free( tmp );   /* Delete previous buffer */
  return shape;
}




/* Set cursor shape.  EXIT_SUCCESS is returned if the cursor shape
   is set successfully
*/
int VMOUSEsetCursor( int CurNum )
{
  int iStat = EXIT_SUCCESS;

  if ( CurNum >= VMOUSEcursors || CurNum < -1 )
    iStat = EXIT_FAILURE;
  else
  {
    if ( SVGAisTrueColorMode())
      cursor = ConvertColor( &builtInCursor[ CurNum ], &iStat );
    else
      cursor = builtInCursor[ CurNum ];  // Work for 16- and 256-color modes
  }
  return iStat;
}






/* Set custom cursor shape.  For hi- or true color modes or 16-color
   modes a byte-per pixel conversion is performed automatically.
*/
int  VMOUSEsetCustomCursor( VMOUSEcurShape *shape )
{
  int iStat = EXIT_SUCCESS;

  if ( SVGAisTrueColorMode() || SVGA16colorMode( videoMode ))
    cursor = ConvertColor( shape, &iStat );
  else
    cursor = *shape;

  return iStat;
}



/* Set custom cursor shape for hi- or true color mode
   It is up to the caller to know the number of bytes per
   pixel that is required.  This lets the application have
   full control of the color mix to use.
   This requires hi- or true-color mode to work.
*/
int  VMOUSEsetCustomCurFullColor( VMOUSEcurShape *shape )
{
  int iStat = EXIT_SUCCESS;

  if ( SVGAisTrueColorMode())
    cursor = *shape;
  else
    iStat = EXIT_FAILURE;

  return iStat;
}







/* Convert all built- in cursor's colors from 16-color to 256-color
   in 256 color modes
*/
static void  Move16to256Index( void )
{
  int   i, j;
  uchar pals[3];
  VMOUSEcurShape *cur;

  if ( SVGA256colorMode( videoMode ))
  {
    for ( i=0; i < VMOUSEcursors; i++ )
    {
      cur = &builtInCursor[i];
      for ( j=0; j < cur->xsiz*cur->ysiz; j++ )
      {
        if ( cur->buf[j] == VMOUSEdefColor ||
             cur->buf[j] == VMOUSEfillColor )
          cur->buf[j] += VMOUSEc16to256;     /*Move to 256-color indexing*/
      }
    }

    /* Set up palette values for the hi-color indexes
    */
    pals[0] = pals[1] = pals[2] = P_WHITE;   /* White */
    VGAsetPal( VMOUSEhFillColor, 1, pals, 0 );
    pals[0] = pals[1] = pals[2] = P_BLACK;   /* Black */
    VGAsetPal( VMOUSEhDefColor, 1, pals, 0 );
  }
}




/* Interrupt service handler for cursor animation.
*/
static volatile VMOUSEcurShape  *CurAnimArray;
static volatile int              CurAnimCnt;
static volatile int              counter=0;
static volatile int              curCnt=0;
static volatile int              counterMax=0;
static volatile int              iAnimRefresh=FALSE;
static volatile void  animationHandler( void )
{
  int x, y;

  if ( isDrawing )  return;       // Can't do it now
  if ( ++counter == counterMax )  // time to change cursor shape
  {
    counter = 0;
    VMOUSEsetCustomCursor( &CurAnimArray[ curCnt++ ] );
    if ( curCnt == CurAnimCnt )   curCnt = 0;

    AddEvent( VMOUSEmoved );      // fake mouse event to force redraw
  }

  if ( iAnimRefresh && !MouseHidden )
  {
    /* Refresh the mouse cursor */
    VMOUSEgetCoord( &x, &y );  // Get new mouse coordinates
    DrawCursor(TRUE);         //  Redraw the cursor in the ISR
  }
}



/* Start cursor animation with the given array of cursor images.
   iCnt is the number of images in the array.
   iFreq is the frequency of change of cursor.  Max = 18 Hertz
   assuming that the ISR 8 is called the standard 18.2 times/second.
   iRefresh is used to enable cursor refresh during the ISR to
   enable immediate display updates.

   If iRefresh is FALSE, then the cursor will be refreshed via the
   queue (VMOUSEgetEvent).  Set iRefresh to TRUE when you know
   that the image on the display is frozen.  If something else
   on the display is being repainted when the ISR is called we get
   garbled display, so iRefresh=TRUE will only work when the image
   on the display is frozen.
*/
int  VMOUSEanimateCursor( VMOUSEcurShape *shapes,
                          int iCnt,
                          int iFreq,
                          int iRefresh )
{
  int  count;

  if ( iFreq < 0 )       iFreq = 1;
  else if ( iFreq > 18 ) iFreq = 18;
  count = (int )(18.2 / (float )iFreq );

  CurAnimArray = (VMOUSEcurShape *)malloc( iCnt * sizeof(VMOUSEcurShape));
  if ( CurAnimArray == NULL )  return  EXIT_FAILURE;
  memcpy( CurAnimArray, shapes, iCnt * sizeof(VMOUSEcurShape));
  CurAnimCnt   = iCnt;
  curCnt  = 0;         /* Current cursor index */
  counter = 0;         /* inverse frequency counter */
  counterMax = count;  /* inverse frequency max count */
  iAnimRefresh = iRefresh;
  INTinit( animationHandler );   /* Chain to int 8 handler */
  return  EXIT_SUCCESS;
}


/* Stop cursor animation
 */
void  VMOUSEanimateCurStop( void )
{
  INTend();
  if ( CurAnimArray ) free( CurAnimArray );
  CurAnimArray = NULL;
}







/* Draw mouse cursor at the current position.
*/
static int  DrawCursor( int Draw )
{
  static int    first = 1, old_x, old_y;
  static int    DrewAtLastCall;
  static uchar *backgnd;
  static uchar *newest_backgnd;
  static VMOUSEcurShape old_cursor;

  int x = VMOUSEbufferedX,y = VMOUSEbufferedY;   /* Get mouse coordinates */

  if (x == old_x && y == old_y && !first && Draw && DrewAtLastCall && old_cursor.buf == cursor.buf)
    return 0;

  isDrawing=1;  /* Semaphore to keep ISRs away */
  if ( first )
  {
    /* Allocate space to store two sets of background
     */
    newest_backgnd = SVGAbufferAlloc( cursor.xsiz, cursor.ysiz );
    backgnd        = SVGAbufferAlloc( cursor.xsiz, cursor.ysiz );
    if ( backgnd == NULL || newest_backgnd == NULL )
    { isDrawing=0;  return -1; }

    first = 0;
    old_x = x, old_y = y;
    old_cursor = cursor;  /* Save in case the cursor shape changes */
  }
  else
  {
    /* If the background has changed since this background was stored, then
       to avoid that the display gets distorted, take a new sample of the
       display background to determine what areas of the background to update.
       Get the latest background to see if some of the background pixels
       have changed.
       */
    if ( DrewAtLastCall )
    {
      SVGAgetBlock( old_x, old_y, old_cursor.xsiz, old_cursor.ysiz, newest_backgnd );
      BITBupdateBackground( backgnd, newest_backgnd, old_cursor.buf,
                            old_cursor.xsiz * old_cursor.ysiz );

      /* Draw background to display */
      if ( !Draw )
        SVGAdrawBlock( old_x, old_y, old_cursor.xsiz, old_cursor.ysiz, backgnd);
      else
        BITBdrawBlock( old_x, old_y, old_cursor.xsiz, old_cursor.ysiz,
                       x, y, cursor.xsiz, cursor.ysiz, backgnd);
    }
  }

  if ( old_cursor.buf != cursor.buf )   /* Cursor changed shape ? */
  {
    if ( old_cursor.xsiz*old_cursor.ysiz != cursor.xsiz*cursor.ysiz )
    {
      /* Cursor shape has changed.  Resize the background buffers
       */
      free( backgnd );
      free( newest_backgnd );

      newest_backgnd = SVGAbufferAlloc( cursor.xsiz, cursor.ysiz );
      backgnd        = SVGAbufferAlloc( cursor.xsiz, cursor.ysiz );
      if ( backgnd == NULL || newest_backgnd == NULL )
      { isDrawing=0;   return -1; }
    }
  }

  /* Get background before plotting the cursor
   */
//  SVGAgetBlock( x, y, cursor.xsiz, cursor.ysiz, backgnd );
  BITBgetBlock( old_x, old_y, old_cursor.xsiz, old_cursor.ysiz,
                x, y, cursor.xsiz, cursor.ysiz,    backgnd );

  if ( Draw )   /* Draw cursor shape */
  {
//    SVGAdrawTranspBlock( x, y, cursor.xsiz, cursor.ysiz, cursor.buf );
    BITBdrawTranspBlock( x, y, cursor.xsiz, cursor.ysiz, cursor.buf );

    DrewAtLastCall = TRUE;   /* Indicate a Draw operation for next call */
  }
  else
  {
    BITBdropBackground();
    DrewAtLastCall = FALSE;
  }
  old_x = x, old_y = y;  /* Save current cursor pos. as old */
  old_cursor = cursor;   /* Make copy of current mouse cursor */

  isDrawing=0;  /* Reset semaphore */
  return 0;
}




/* If the current video mode is one of the SVGA 256 color modes
   then redefine the colors of the mouse.
   Use white for the definition and black for the interior.
   Use 2 of the uppermost color indexes in the 256 color palette.
   Return EXIT_SUCCESS on success, EXIT_FAILURE otherwise.
   */
int  VMOUSEinvertCursor( void )
{
  uchar pals[3];

  if ( !SVGAisTrueColorMode())   /* Not supported in TRUE Color mode yet */
  {
    if ( SVGA256colorMode( videoMode ))
    {
      /* Just swap the 2 colors used.
      */
      VGAgetPal( VMOUSEhFillColor, 1, pals );
/*      VESAgetPal( VMOUSEhFillColor, 1, pals );*/
      if ( pals[0] == P_WHITE )
      {
        pals[0] = pals[1] = pals[2] = P_BLACK;
        VGAsetPal( VMOUSEhFillColor, 1, pals, 0 );
        pals[0] = pals[1] = pals[2] = P_WHITE;
        VGAsetPal( VMOUSEhDefColor, 1, pals, 0 );    
      }
      else
      {
        pals[0] = pals[1] = pals[2] = P_WHITE;
        VGAsetPal( VMOUSEhFillColor, 1, pals, 0 );
        pals[0] = pals[1] = pals[2] = P_BLACK;
        VGAsetPal( VMOUSEhDefColor, 1, pals, 0 );
      }
    }
  }
  return EXIT_SUCCESS;
}



/* Install keyboard handler that will produce mouse simulated
   event codes for certain keys
   */
static int set=0;
static int InstallKbd( void )
{
  if ( !set )
  {
    if ( atexit( KeyBend ) != 0 )  return  EXIT_FAILURE;

    OrigKbdHandler = _dos_getvect( VMOUSEkeyInt );
    _dos_setvect( VMOUSEkeyInt, VmouseKbdHandler );

    set = 1;
  }
  return  EXIT_SUCCESS;
}



/* Terminate keyboard driver
*/
static void  KeyBend( void )
{
  if ( set )
  {
    set = 0;
    _dos_setvect( VMOUSEkeyInt, OrigKbdHandler );
  }
}



/* Add event to event buffer
*/
static void AddEvent( uchar Event )
{
  eventBuf[ inPtr++ ] = Event;   /* Store event in event buffer */
  inPtr &= EVENTBUFSIZ;          /* Update ring buffer pointer */
}


/* EOF */
