/*
   This package contains functions to implement a VESA 1.2 video interface.
   Only a subset of functions are provided in this package:
     00   VESA init
     01   Get VESA mode info
     02   Set VESA video mode
     05   Get/set page
   See your favorite VESA paper for an explanation of these functions and
   the VESA specification for PC video programming.  I found most of the
   info used here on the net in a file called vesasp12.txt.

   Written by Arve Holmbo.
   Released to public domain March 1996.
   Update release with some VBE 2.0 support (linear frame buffer) Sep 1996.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <graph.h>
#include <i86.h>
#include <dos.h>

#include "vesa.h"
#include "pal.h"
#include "pm.h"


typedef struct
{
  char    Signature[4];   /* should say "VESA" on return */
  char    Version[2];
  ushort  OEMinfoOffset;  /* offset:segment pointer to OEM string */
  ushort  OEMinfoSegment;
  char    Capabilities[4];
  ushort  VideoModesOffset;
  ushort  VideoModesSegment;
  ushort  TotalMem;
  char    Reserved[236];
  char    VBE2OEMinfo[ 256]; /* Extra OEM info if VBE2.0 */
} VESAinfo;

typedef struct
{
  ushort ModeAttr;
  char   WindowAattr;
  char   WindowBattr;
  ushort Granularity;
  ushort WinSize;
  short  Asegment;
  short  Bsegment;
  void   (*WindowPosFunc)( long );
  ushort BytesPerLine;

  /* optional information (when bit 1 of modeattr is set)  */

  ushort  XPixels;
  ushort  YPixels;
  char   CharWidth;
  char   CharHeight;
  char   Planes;
  char   BitsPerPixel;
  char   Banks;
  char   MemoryModel;
  char   BankSize;               /* in k bytes */
  char   NumOfImPages;
  char   Reserved;

  uchar  RedMaskSiz;
  uchar  RedFieldPos;

  uchar  GreenMaskSiz;
  uchar  GreenFieldPos;

  uchar  BlueMaskSiz;
  uchar  BlueFieldPos;

  uchar  RsvdMaskSiz;
  uchar  RsvdFieldPos;
  uchar  DirColModeInfo;

  uint   PhysBasePtr;       /* VBE 2.0: Linear frame buffer start address */
  uint   OffScreenMemOffs;  /* Pointer to start of off-screen memory */
  short  OffScreenMemSiz;   /* # of 1K blocks of off-screen memory */

  char   Reserved2[206];
} VESAmodeInfo;



/* Real mode call structure used by the DPMI interface
*/
static  PM_RMI  rmi;


static int       SetMode( short mode );
static VESAdata  VesaData;   /* VESA info in PM memory */
static int       blockSiz;   /* Number of blocks in a VESA window */
static short     linMode = 0; /* VBE2.0: linear frame buffer */


#define DPMI_INTR  0x31


/* Keep pointers to 2 real mode memory buffers
*/
static PM_REALPTR  v_info, v_mode_info;




/* Sets a desired VESA video mode if provided by the hardware.
   If linearMode is TRUE, then a linear frame buffer mode will
   be chosen if available.
   On success, EXIT_SUCCESS is returned, else EXIT_FAILURE, along
   with VesaStatus reason.  This function calls VESAinit.
   */
int  VESAsetMode( short mode, int linearMode, int *VesaStatus )
{
  if ( VESAinit( mode, VesaStatus ) != EXIT_SUCCESS )
  {
    if ( linearMode )
      *VesaStatus = VesaInitFailLFB;
    return  EXIT_FAILURE;
  }

  linMode = (linearMode != 0 ? 0x4000 : 0 );
  return SetMode( mode | linMode );
}



/* Gets the parameters for a desired VESA video mode if provided by
   the hardware.
   If linearMode is TRUE, then a linear frame buffer mode will
   be chosen if available.
   On success, EXIT_SUCCESS is returned, else EXIT_FAILURE, along
   with VesaStatus reason.  This function calls VESAinit.
   This function is similar to VESAsetMode() except that the
   mode is not set.  It is then the responsiblity of the application
   to ensure that the mode is available and valid.
   */
int  VESAqueryMode( short mode, int linearMode, int *VesaStatus )
{
  if ( VESAinit( mode, VesaStatus ) != EXIT_SUCCESS )
  {
    if ( linearMode )
      *VesaStatus = VesaInitFailLFB;
    return  EXIT_FAILURE;
  }

  linMode = (linearMode != 0 ? 0x4000 : 0 );
  return EXIT_SUCCESS;
}



/* Does all initialization and returns EXIT_SUCCESS if init went OK,
   else EXIT_FAILURE, along with reason ( *VesaStatus )
   */
int  VESAinit( short mode, int *VesaStatus )
{
  VesaData.Granularity = 4;  /* This was equal in all CL-5429 modes */
  VesaData.WinSize     = 64; /* Acceptable default */

  *VesaStatus = VesaModeValid;

  if ( VESAgetInfo() == EXIT_SUCCESS )
  {
    if ( VESAgetModeInfo( mode ) == EXIT_SUCCESS )
      return ( EXIT_SUCCESS );
    *VesaStatus = VesaModeInValid;
  }
  else
    *VesaStatus = VesaInitFail;
  return ( EXIT_FAILURE );
}





/* Get VESA info.  Return EXIT_SUCCESS on success, EXIT_FAILURE otherwise
*/
int   VESAgetInfo( void )
{
  union  REGS   regs;
  struct SREGS  sregs;
  VESAinfo far *vesa_info;
  char     far *p;
  int    i, iStat = EXIT_SUCCESS;

  memset( &VesaData, 0, sizeof( VESAdata ));

  if (( vesa_info = (VESAinfo far *)PMallocDosMem( sizeof( VESAinfo ),
                                                  &v_info )) == NULL )
  {
    printf("Failed to alloc DOS memory. Press a key\n");
    (void )getch();
    return EXIT_FAILURE;   /* Issue error message here */
  }

  vesa_info->Signature[0] = 'V';
  vesa_info->Signature[1] = 'B';
  vesa_info->Signature[2] = 'E';
  vesa_info->Signature[3] = '2';  /* To get any extended VBE 2.0 info */

  memset( &rmi, 0, sizeof( PM_RMI ));

  rmi.EAX = 0x4F00;       /* Call service 4F, subfunction 0 */
  rmi.ES  = v_info.segment; /* Put DOS seg:offs into ES:DI */
  rmi.DS  = v_info.segment;
  rmi.EDI = 0;

  /* Use DPMI call 300h to issue BIOS interrupt */
  memset( &regs,  0, sizeof( regs ));
  memset( &sregs, 0, sizeof( sregs ));
  segread( &sregs  );

  regs.w.ax = 0x300;
  regs.w.bx = 0x0010;   /* Video services */
  regs.w.cx = 0;        /* No PM stack to real mode stack usage */
  sregs.es  = FP_SEG( &rmi );
  regs.x.edi= FP_OFF( &rmi );
  int386x( DPMI_INTR, &regs, &regs, &sregs );
  if ( rmi.EAX == 0x4F )
  {
    VesaData.TotalMemory = vesa_info->TotalMem;
    VesaData.TotalMemory *= 64;      /* Get mem in kBytes */
    VesaData.OEMinfoOffset = vesa_info->OEMinfoOffset;
    VesaData.OEMinfoSegment= vesa_info->OEMinfoSegment;
    VesaData.Version[0]    = vesa_info->Version[0];
    VesaData.Version[1]    = vesa_info->Version[1];
    VesaData.VidModesOffs  = vesa_info->VideoModesOffset;
    VesaData.VidModesSeg   = vesa_info->VideoModesSegment;

    VesaData.OEMinfo[0] = 0;
    if ( VesaData.Version[1] >= 2 )
    {
      /* Get VESA OEM info when VBE 2.0 or higher */
      p = (char far *)&vesa_info->VBE2OEMinfo[0];
      for ( i=0; i < 256; i++, p++ )
      {
        if ( *p == 0 )  VesaData.OEMinfo[i] = ' ';
        else            VesaData.OEMinfo[i] = *p;
      }
    }
  }
  else
    iStat = EXIT_FAILURE;

  PMfreeDosMem( &v_info );
  return iStat;
}




/* Get VESA video mode information.
   SVGAmode example is _SVRES256COLOR
   Return EXIT_SUCCESS on success, EXIT_FAILURE otherwise
*/
static int  VESAgetModeInfo( short SVGAmode )
{
  union  REGS  regs;
  struct SREGS sregs;
  VESAmodeInfo far *vesa_mode_info;
  int    iStat = EXIT_SUCCESS;

  if ((vesa_mode_info = (VESAmodeInfo far *)PMallocDosMem( sizeof(VESAmodeInfo),
     &v_mode_info )) == NULL)
  {
    printf("Failed to allocate DOS memory.  Press a key\n" );
    (void )getch();
    return EXIT_FAILURE;   /* Issue error message here */
  }

  memset( &rmi, 0, sizeof( PM_RMI ));

  rmi.EAX = 0x4F01;    /* Call service 4F, subfunction 1 */
  rmi.ES  = v_mode_info.segment;   /* Put DOS seg:offs into ES:DI */
  rmi.DS  = v_mode_info.segment;
  rmi.EDI = 0;
  rmi.ECX = (long )SVGAmode;      /* SVGA mode */

  /* Use DPMI call 300h to issue BIOS interrupt */
  memset( &sregs, 0, sizeof( sregs ));
  segread( &sregs  );

  regs.w.ax = 0x300;
  regs.w.bx = 0x0010;   /* Video services */
  regs.w.cx = 0;        /* No PM stack to real mode stack usage */
  sregs.es  = FP_SEG( &rmi );
  regs.x.edi= FP_OFF( &rmi );
  int386x( DPMI_INTR, &regs, &regs, &sregs );
  if ( rmi.EAX == 0x4F )
  {
    /* Copy key VESA information to PM data store.
       */
    VesaData.ModeAttr    = vesa_mode_info->ModeAttr;
    VesaData.WindowAattr = vesa_mode_info->WindowAattr;
    VesaData.WindowBattr = vesa_mode_info->WindowBattr;
    VesaData.Granularity = vesa_mode_info->Granularity;
    VesaData.WinSize     = vesa_mode_info->WinSize;
    VesaData.BytesPerLine= vesa_mode_info->BytesPerLine;
    VesaData.Asegment    = vesa_mode_info->Asegment;
    VesaData.Bsegment    = vesa_mode_info->Bsegment;
    VesaData.Planes      = vesa_mode_info->Planes;
    VesaData.BitsPerPixel= vesa_mode_info->BitsPerPixel;
    VesaData.Banks       = vesa_mode_info->Banks;
    VesaData.BankSize    = vesa_mode_info->BankSize;
    VesaData.NumOfImPages= vesa_mode_info->NumOfImPages;
    VesaData.XPixels     = vesa_mode_info->XPixels;
    VesaData.YPixels     = vesa_mode_info->YPixels;
    VesaData.MemoryModel = vesa_mode_info->MemoryModel;

    VesaData.RedMaskSiz   = vesa_mode_info->RedMaskSiz;
    VesaData.GreenMaskSiz = vesa_mode_info->GreenMaskSiz;
    VesaData.BlueMaskSiz  = vesa_mode_info->BlueMaskSiz;

    VesaData.RedFieldPos  = vesa_mode_info->RedFieldPos; 
    VesaData.GreenFieldPos= vesa_mode_info->GreenFieldPos;
    VesaData.BlueFieldPos = vesa_mode_info->BlueFieldPos; 

    VesaData.DirColModeInfo = vesa_mode_info->DirColModeInfo;
    VesaData.PhysBasePtr  = vesa_mode_info->PhysBasePtr;

    if ( VesaData.Granularity != 0 )
      blockSiz = VesaData.WinSize / VesaData.Granularity;
    else
      iStat = EXIT_FAILURE;
  }
  else
    iStat = EXIT_FAILURE;

  PMfreeDosMem( &v_mode_info );
  return iStat;
}



/* Set the given VESA mode.  No check to see if valid mode.
   If Ok, EXIT_SUCCESS is returned, else EXIT_FAILURE
 */
#define  VESA_DONT_CLEAR   0x8000

static int SetMode( short mode )
{
  union  REGS  regs;

  regs.w.ax = 0x4f02;         /* Set video mode */
  regs.w.bx = (mode & 0x7FFF ) | VESA_DONT_CLEAR;  /* Dont clear screen */
  int386(0x10,&regs,&regs);

  if ( regs.w.ax == 0x4F )
    return EXIT_SUCCESS;
  return EXIT_FAILURE;
}



/* If linear frame buffer mode has been successfully set
   up, return ist video start address.
   */
uchar *VESAgetLinearAddr( void )
{
  uchar *tmp = NULL;

  if ( VesaData.PhysBasePtr != 0 && VesaData.ModeAttr & 0x80 )
    tmp = (uchar *)PMgetLinearAddr( VesaData.PhysBasePtr,
                                    VesaData.TotalMemory*1024 );
  return tmp;
}




/* Cleanup internals 
*/
void VESAcleanup( void )
{
  PMcleanup();
}





/* Return pointer to VESA data structure.
   The 2 info functions above should have been called a priori.
*/
VESAdata  *VESAgetData( void )
{
  return  &VesaData;
}





/********************************************
 Set the given vesa screen page.

 Assumes the WinSize and Granularity parameters
 have been setup with correct values.

*********************************************/

void  VESAsetPage( uint vpage )
{
  struct SREGS sregs;
  union  REGS  regs;
  ushort winPos;

  if ( !linMode )
  {
    winPos = vpage * blockSiz;

    regs.w.ax = 0x4f05;
    regs.w.bx = 0;   /* Window A */
    regs.w.cx = 0;
    regs.w.dx = winPos;

    segread( &sregs  );
    int386x( 0x10, &regs, &regs, &sregs );

    regs.w.ax = 0x4f05;
    regs.w.bx = 1;   /* Window B */
    regs.w.cx = 0;
    regs.w.dx = winPos;

    segread( &sregs  );
    int386x( 0x10, &regs, &regs, &sregs );
  }
}



/********************************************
 Get the current vesa screen page.

 Assumes the WinSize and Granularity parameters
 have been setup with correct values.
*/

uint VESAgetPage( void )
{
  union  REGS  regs;
  uint   vpage;
  ushort winPos;
  static int  first=1;

  if ( linMode )
    return 0;  /* Default to zero in linear frame buffer mode */

  if ( first )
  {
    first = 0;     /*Initialize to the first page */
    VESAsetPage(0);
  }
  regs.w.ax = 0x4f05;
  regs.w.bx = 0x100;   /* Window A */
  regs.w.cx = 0;
  regs.w.dx = 0;
  int386(0x10,&regs,&regs);
  if ( regs.w.ax == 0x4F )
  {
    winPos = regs.w.dx;
    vpage = winPos * VesaData.Granularity / VesaData.WinSize;
    return vpage;
  }
  return 0xFFFFFFFF;   /* return 32-bit -1 */
}




/* Get palette values
*/
int  VESAgetPal( uint index, uint howmany, uchar *pals )
{
  uchar *p;
  int    iStat = EXIT_SUCCESS;

  p = PALgetTable();
  p += ( index * 3 );
  memcpy( pals, p, howmany * 3 );

  return iStat;
}



/* Set display scan line length in pixels.  If success, then the new
   length is returned, else zero
*/
int  VESAsetLogicalScanLen( int iLen )
{
  union  REGS  regs;
  int    iReturnLen = 0;

  regs.w.ax = 0x4f06;
  regs.w.bx = 0;    /* Set scan-line length in pixels. */
  regs.x.ecx = iLen;
  int386(0x10,&regs,&regs);
  if ( regs.w.ax == 0x4F )
    iReturnLen = (int )regs.w.cx;

  return  iReturnLen;
}


/* Set display start address in terms of x- and y-location.
   Returns EXIT_SUCCESS on success, EXIT_FAILURE else.
*/
int  VESAsetDisplayStart( int x, int y )
{
  union  REGS  regs;
  int    iStat = EXIT_FAILURE;

  regs.w.ax = 0x4f07;
  regs.w.bx = 0x80;    /* Set start during vertical retrace */
  regs.x.ecx = x;
  regs.x.edx = y;
  int386(0x10,&regs,&regs);
  if ( regs.w.ax == 0x4F )
    iStat = EXIT_SUCCESS;
  return  iStat;
}



/* Return list of available VESA graphics modes.  A call must have
   been made to VESAinit().  The list is terminated with -1.
*/
#define MAX_MODES  2048

short  *VESAgetModeList( void )
{
  uint   addr;      /* Real mode address */
  short *modeAddr;  /* Protected mode address equivalent */
  short *modeList;
  short *d, *s;
  int    i, imax;

  /* Create a linear address assuming a list no longer than
     1k modes
     */
  addr  = (uint )( VesaData.VidModesSeg ) << 4;
  addr += (uint )VesaData.VidModesOffs;
  modeAddr = (short *)PMgetLinearAddr( addr, MAX_MODES * sizeof( short ));
  modeList = (short *)malloc( MAX_MODES * sizeof( short ));
  if ( modeList == NULL )
    return  NULL;
  imax = MAX_MODES;  i=0;
  for ( d=modeList, s=modeAddr; *s != -1; s++, d++ )   *d = *s;
  *d = -1;
  return  modeList;
}




/* Print failure string according to reason
*/
int  VESAfailPrint( int reason )
{
  switch ( reason )
  {
    case  0:
      printf( "VESA mode is valid\n" );
      break;
    case -1:
      printf( "Mode is invalid\n" );
      break;
    case -2:
      printf( "VESA init failed\n" );
      break;
    case -3:
      printf( "Mode is not supported\n" );
      break;
    case -4:
      printf( "VESA init internal failure\n" );
      break;
    default:
      printf( "VESA init error.  Reason (%d) not known\n", reason );
  }
  return reason;
}

/* EOF */
