/* wav2raw.c */

#define BUFFER_SIZE 8192

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define TRUE  1
#define FALSE 0

#define MAX(a, b) (((a) > (b)) ? (a) : (b))
#define MIN(a, b) (((a) < (b)) ? (a) : (b))

typedef struct
  {
   /* RIFF header */
    char  rID[4];          // "RIFF"
    long  rLen;

   /* WAVE form definition */
    char  wID[4];          // "WAVE"

   /* Format chunk */
    char  fID[4];          // "fmt\0"
    long  fLen;
    short wFormatTag;
    short nChannels;       // 1 = mono
    long  nSamplesPerSec;  // sampling rate
    long  nAvgBytesPerSec; // nChannels * nSamplesPerSec * (nBitsPerSample/8)
    short nBlockAlign;     // nChannels * (nBitsPerSample / 8)
    short nBitsPerSample;

   /* Data chunk */
    long  dID;             // "data"
    long  dLen;            // data size;
  } WAVE_HEADER;

void main(int argc, char *argv[])
  {
    char  in[64];
    char  out[64];
    int   i;

    FILE *infile;
    FILE *outfile;

    WAVE_HEADER wave_header;

    char buffer[BUFFER_SIZE];

    long remaining;
    long count;

    switch(argc)
      {
        case 2:
          strcpy(in, argv[1]);  strcpy(out, argv[1]);
          strcat(in, ".WAV");   strcat(out, ".RAW");
          break;

        case 3:
          strcpy(in, argv[1]);
          strcpy(out, argv[2]);
          break;

        default:
          printf("Syntax:  WAV2RAW <input.wav> <output.raw>    (input.wav    -> output.raw)   \n");
          printf("Syntax:  WAV2RAW <filename>                  (filename.wav -> filename.raw) \n");
          exit(EXIT_FAILURE);
          break;
      }

   printf("Converting %s to %s \n", in, out);

   if ((infile = fopen(in, "rb")) == NULL)
     {
       printf("ERROR:  Can't open input file `%s'! \n", infile);
       exit(EXIT_FAILURE);
     }

   if ((outfile = fopen(out, "wb")) == NULL)
     {
       printf("ERROR:  Can't open output file `%s'! \n", outfile);
     }

   fread(&wave_header, sizeof(wave_header), 1, infile);

   if (wave_header.wFormatTag != 1)
     {
       printf("ERROR:  WAV file must contain PCM data \n");
       exit(EXIT_FAILURE);
     }

   if (wave_header.nChannels != 1)
     {
       printf("ERROR:  WAV file must contain mono sound \n");
       exit(EXIT_FAILURE);
     }

   if (wave_header.nBitsPerSample != 8)
     {
       printf("ERROR:  WAV file must contain 8-bit samples \n");
       exit(EXIT_FAILURE);
     }

   if ((wave_header.nSamplesPerSec < 22000) || (wave_header.nSamplesPerSec > 23000))
     {
       printf("WARNING:  Data not sampled at 22khz \n");
       printf(" Data is actually sampled at %u HZ \n", wave_header.nSamplesPerSec);
       printf(" Data will be converted, but may play incorrectly \n");
     }

   remaining = wave_header.dLen;

   while (remaining > 0)
     {
       count = MIN(remaining, BUFFER_SIZE);

       fread(buffer, 1, count, infile);

       for (i = 0; i < count; i++)
         buffer[i] -= 128;

       fwrite(buffer, 1, count, outfile);

       remaining -= count;
     }


   free(buffer);

   fclose(infile);
   fclose(outfile);
  }