/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */

#define KEY_BUFFER_SIZE 256
#define KEY_LSHIFT      42
#define KEY_RSHIFT      54
#define KEY_CAPSLOCK    58

#if defined (FOR_WATCOM)
#define inportb  inp
#define outportb outp
#define enable   _enable
#define disable  _disable
#define KB_UPDATE
#elif defined (FOR_LINUX)
#define disable()
#define enable()
#define __interrupt
#define __far
#define KB_UPDATE keyboard_update()
#elif defined (FOR_X11)
#define KB_UPDATE keyboard_update()
#define disable()
#define enable()
#else
#define KB_UPDATE
#define __interrupt
#define __far
#endif

/* array of scan codes -> ascii values */
static UBYTE __jlib_ascii_tab[128] =
{
	0, 27, '1', '2', '3', '4', '5', '6', '7', '8', '9', '0',
	'-', '=', 8, 9, 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p',
	'[', ']', 13, 0, 'a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l',
	';', '\'', '`', 0, 0, 'z', 'x', 'c', 'v', 'b', 'n', 'm',
	',', '.', '/', 0, '*', 0, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, '-', 0, '5', 0, '+', 0, 0, 0, 0, 16, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

static UBYTE __jlib_shift_tab[128] =
{
	0, 27, '!', '@', '#', '$', '%', '^', '&', '*', '(', ')',
	'_', '+', 8, 9, 'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P',
	'{', '}', 13, 0, 'A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L',
	':', '|', '~', 0, 0, 'Z', 'X', 'C', 'V', 'V', 'B', 'M',
	'<', '>', '?', 0, '*', 0, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, '-', 0, '5', 0, '+', 0, 0, 0, 0, 16, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

static UBYTE __jlib_caps_tab[128] =
{
 0, 27, '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=', 8, 9,
	'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', '[', ']', 13, 0, 'A', 'S',
	'D', 'F', 'G', 'H', 'J', 'K', 'L', ';', 39, '`', 0, 92, 'Z', 'X', 'C', 'V',
	'B', 'N', 'M', ',', '.', '/', 0, '*', 0, ' ', 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, '-', 0, 0, 0, '+', 0,
	0, 0, 0, 16, 0, 0, 92, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	13, 0, '/', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 127,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

int __jlib_kb_initted = 0;
unsigned __jlib_caps_state;
int __jlib_last_ext = 0;	/* last scancode was extended key */
volatile char __jlib_kb_state[128];	/* key pressed flags */
int __jlib_ctrl_press, __jlib_alt_press, __jlib_lshift_press, __jlib_rshift_press;

volatile USHORT __jlib_kb_buff[KEY_BUFFER_SIZE];
static volatile UBYTE __jlib_kb_start = 0;
static volatile UBYTE __jlib_kb_end = 0;


#define buff_add(val) \
{ \
 __jlib_kb_buff[__jlib_kb_end] = (USHORT)(val); \
 __jlib_kb_end++; /* autowraps */ \
 if (__jlib_kb_end == __jlib_kb_start) {    /* buffer full */ \
     __jlib_kb_start++; \
 } \
}


#if !defined (FOR_LINUX) && !defined (FOR_X11)
static void __interrupt __far __jlib_keyint(void);
static void __interrupt __far __jlib_keyint(void)
{
	int temp = inportb(0x60);	/* read keyboard byte */
	int temp2;

	disable();

	temp2 = temp & 0x7f;

	if ((__jlib_last_ext != 0) && ((temp2 == 42) || (temp2 == 55) || (temp2 == 111))) {
		__jlib_last_ext = 0;	/* skip these ext keys */
	} else {
		if (temp == 224) {
			__jlib_last_ext = 1;	/* extended key coming next time */
			enable();
			outportb(0x20, 0x20);	/* ack. the interrupt */
			return;
		} else {
			__jlib_last_ext = 0;
		}

		if (temp & 0x80) {	/* key was released */
			temp &= 0x7f;
			__jlib_kb_state[temp] = 0;

			switch (temp) {
			case KEY_LSHIFT:
				buff_add(KB_LSHIFT_UP);
				break;
			case KEY_RSHIFT:
				buff_add(KB_RSHIFT_UP);
				break;
			case KEY_CONTROL:
				buff_add(KB_CTRL_UP);
				break;
			case KEY_ALT:
				buff_add(KB_ALT_UP);
				break;
			case KEY_UP:
				buff_add(KB_UP_UP);
				break;
			case KEY_DOWN:
				buff_add(KB_DOWN_UP);
				break;
			case KEY_LEFT:
				buff_add(KB_LEFT_UP);
				break;
			case KEY_RIGHT:
				buff_add(KB_RIGHT_UP);
				break;
			case KEY_F1:
				buff_add(KB_F1_UP);
				break;
			case KEY_F2:
				buff_add(KB_F2_UP);
				break;
			case KEY_F3:
				buff_add(KB_F3_UP);
				break;
			case KEY_F4:
				buff_add(KB_F4_UP);
				break;
			case KEY_F5:
				buff_add(KB_F5_UP);
				break;
			case KEY_F6:
				buff_add(KB_F6_UP);
				break;
			case KEY_F7:
				buff_add(KB_F7_UP);
				break;
			case KEY_F8:
				buff_add(KB_F8_UP);
				break;
			case KEY_F9:
				buff_add(KB_F9_UP);
				break;
			case KEY_F10:
				buff_add(KB_F10_UP);
				break;
			default:
				if (__jlib_ascii_tab[temp]) {
					if (__jlib_caps_state) {
						buff_add(__jlib_caps_tab[temp] | (1 << 15));
					} else {
						if ((__jlib_kb_state[KEY_LSHIFT]) || (__jlib_kb_state[KEY_RSHIFT])) {
							buff_add(__jlib_shift_tab[temp] | (1 << 15));
						} else {
							buff_add(__jlib_ascii_tab[temp] | (1 << 15));
						}
					}
				}
				break;
			}
		} else {	/* key was pressed */
			temp &= 0x7f;
			__jlib_kb_state[temp] = 1;

			if (__jlib_ascii_tab[temp]) {
				if (__jlib_caps_state) {
					buff_add(__jlib_caps_tab[temp]);
				} else {
					if ((__jlib_kb_state[KEY_LSHIFT]) || (__jlib_kb_state[KEY_RSHIFT])) {
						buff_add(__jlib_shift_tab[temp]);
					} else {
						buff_add(__jlib_ascii_tab[temp]);
					}
				}
			} else {
				switch (temp) {
				case KEY_CAPSLOCK:
					__jlib_caps_state = (!__jlib_caps_state);	/* flip shift state */
					break;
				case KEY_LSHIFT:
					buff_add(KB_LSHIFT);
					break;
				case KEY_RSHIFT:
					buff_add(KB_RSHIFT);
					break;
				case KEY_CONTROL:
					buff_add(KB_CTRL);
					break;
				case KEY_ALT:
					buff_add(KB_ALT);
					break;
				case KEY_UP:
					buff_add(KB_UP);
					break;
				case KEY_DOWN:
					buff_add(KB_DOWN);
					break;
				case KEY_LEFT:
					buff_add(KB_LEFT);
					break;
				case KEY_RIGHT:
					buff_add(KB_RIGHT);
					break;
				case KEY_F1:
					buff_add(KB_F1);
					break;
				case KEY_F2:
					buff_add(KB_F2);
					break;
				case KEY_F3:
					buff_add(KB_F3);
					break;
				case KEY_F4:
					buff_add(KB_F4);
					break;
				case KEY_F5:
					buff_add(KB_F5);
					break;
				case KEY_F6:
					buff_add(KB_F6);
					break;
				case KEY_F7:
					buff_add(KB_F7);
					break;
				case KEY_F8:
					buff_add(KB_F8);
					break;
				case KEY_F9:
					buff_add(KB_F9);
					break;
				case KEY_F10:
					buff_add(KB_F10);
					break;
				default:
					break;
				}
			}
		}
	}
	enable();
	outportb(0x20, 0x20);	/* ack. the interrupt */
}
#ifndef FOR_WATCOM
static int __jlib_kb_int_end;
#endif
#endif


/*+------------------------------------------------------------------------+ */
/*| Clear the key buffer.                                                  | */
/*+------------------------------------------------------------------------+ */
void kb_clear(void)
{
	if (!__jlib_kb_initted) {
		return;
	}
	KB_UPDATE;

#ifdef FOR_LINUX
	keyboard_clearstate();
#endif

	disable();
	__jlib_kb_start = __jlib_kb_end = 0;
	enable();

	__jlib_alt_press = kb_keydown(KEY_ALT);
	__jlib_ctrl_press = kb_keydown(KEY_CONTROL);
	__jlib_lshift_press = kb_keydown(KEY_LSHIFT);
	__jlib_rshift_press = kb_keydown(KEY_RSHIFT);
}


/*+------------------------------------------------------------------------+ */
/*| Indicate if a given key is currently being pressed.                    | */
/*+------------------------------------------------------------------------+ */
int kb_keydown(int keynum)
{
	if ((!__jlib_kb_initted) || ((keynum < 0) || (keynum >= 128))) {
		return 0;
	}
	KB_UPDATE;

	return __jlib_kb_state[keynum];
}


/*+------------------------------------------------------------------------+ */
/*| Indicate if there are any keys waiting in the key buffer.              | */
/*+------------------------------------------------------------------------+ */
int kb_key_hit(void)
{
	UBYTE i;

	if (!__jlib_kb_initted) {
		return 0;
	}
	KB_UPDATE;

	/* only return regular characters */
	i = __jlib_kb_start;

	while (i != __jlib_kb_end) {
		if (KB_IS_SPECIAL_CHAR(__jlib_kb_buff[i]) || KB_WAS_RELEASED(__jlib_kb_buff[i])) {
			i++;
		} else {
			return 1;
		}
	}
	return 0;
}


/*+------------------------------------------------------------------------+ */
/*| Indicate if there are any extended keys waiting in the key buffer.     | */
/*+------------------------------------------------------------------------+ */
int kb_ext_key_hit(void)
{
	if (!__jlib_kb_initted) {
		return 0;
	}
	KB_UPDATE;

	return (__jlib_kb_start != __jlib_kb_end);
}


/*+------------------------------------------------------------------------+ */
/*| Get the next key from the key buffer.                                  | */
/*+------------------------------------------------------------------------+ */
char kb_get_next_key(void)
{
	USHORT val;

	if (!__jlib_kb_initted) {
		return 0;
	}
	/* loop through key buffer, skipping special keys and key-up events */
	do {
		do {
			KB_UPDATE;
		} while (__jlib_kb_start == __jlib_kb_end);	/* wait for a press */
		disable();
		val = __jlib_kb_buff[__jlib_kb_start];
		__jlib_kb_start++;	/* autowrap the unsigned char */
		enable();
		/* printf("read val %d\n",val);
		   printf("val special: %d\n",KB_IS_SPECIAL_CHAR(val));
		   printf("val release: %d\n",KB_WAS_RELEASED(val));
		 */
	} while ((KB_IS_SPECIAL_CHAR(val)) || (KB_WAS_RELEASED(val)));

	return (char) (val & 0xff);
}


/*+------------------------------------------------------------------------+ */
/*| Get the next key from the key buffer including special chars.          | */
/*+------------------------------------------------------------------------+ */
USHORT kb_get_next_ext_key(void)
{
	UBYTE val;
	USHORT ret = 0;

	if (!__jlib_kb_initted) {
		return 0;
	}
	/* loop through key buffer, processing special keys */
	do {
		KB_UPDATE;
		do {
		} while (__jlib_kb_start == __jlib_kb_end);	/* wait for a press */
		disable();
		val = __jlib_kb_buff[__jlib_kb_start];
		__jlib_kb_start++;	/* autowrap the unsigned char */
		enable();

		if (KB_IS_SPECIAL_CHAR(val)) {
			switch (val) {
			case KB_LSHIFT_UP:
				__jlib_lshift_press = 0;
				break;
			case KB_RSHIFT_UP:
				__jlib_rshift_press = 0;
				break;
			case KB_CTRL_UP:
				__jlib_ctrl_press = 0;
				break;
			case KB_ALT_UP:
				__jlib_alt_press = 0;
				break;

			case KB_LSHIFT:
				__jlib_lshift_press = 1;
				break;
			case KB_RSHIFT:
				__jlib_rshift_press = 1;
				break;
			case KB_CTRL:
				__jlib_ctrl_press = 1;
				break;
			case KB_ALT:
				__jlib_alt_press = 1;
				break;

			case KB_UP:
				ret |= EXT_UP;
				val = 0;
				break;
			case KB_DOWN:
				ret |= EXT_DOWN;
				val = 0;
				break;
			case KB_LEFT:
				ret |= EXT_LEFT;
				val = 0;
				break;
			case KB_RIGHT:
				ret |= EXT_RIGHT;
				val = 0;
				break;
			}
		}
		if (!KB_IS_SPECIAL_CHAR(val) && !KB_WAS_RELEASED(val)) {
			break;
		}
	} while (1);

	if (__jlib_alt_press) {
		ret = ret | EXT_ALT;
	}
	if ((__jlib_lshift_press) || (__jlib_rshift_press)) {
		ret = ret | EXT_SHIFT;
	}
	if (__jlib_ctrl_press) {
		ret = ret | EXT_CTRL;
	}
	ret |= val;

	if (__jlib_kb_start == __jlib_kb_end) {
		__jlib_alt_press = kb_keydown(KEY_ALT);
		__jlib_ctrl_press = kb_keydown(KEY_CONTROL);
		__jlib_lshift_press = kb_keydown(KEY_LSHIFT);
		__jlib_rshift_press = kb_keydown(KEY_RSHIFT);
	}
	return ret;
}


/*+------------------------------------------------------------------------+ */
/*| Get the next code from the key buffer.                                 | */
/*+------------------------------------------------------------------------+ */
USHORT kb_get_next_code(void)
{
	USHORT val;

	if (!__jlib_kb_initted) {
		return 0;
	}
	KB_UPDATE;

	if (__jlib_kb_start == __jlib_kb_end) {
		return KB_NO_CODE;
	}
	disable();
	val = __jlib_kb_buff[__jlib_kb_start];
	__jlib_kb_start++;
	enable();

	return val;
}
