/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <stdlib.h>
#include <stdio.h>
#include <jlib.h>

JINLINE int __jlib_intersect(int x1, int y1, UBYTE * r1, int x2, int y2, UBYTE * r2);

#define SDR	sprite_data_rec
#define SS	sprite_system
#define SR	sprite_record


/*+----------------------------------------------------------------------+ */
/*|sprite_init - intitialises the sprite system.                         | */
/*+----------------------------------------------------------------------+ */
SS *sprite_init(int max_sprites, int max_frames)
{
	SS *s;
	int counter;

	JLIB_ENTER("sprite_init");

#ifndef JLIB_PRODUCTION
	if (!max_sprites && !max_frames)
		jlib_exit(jlib_msg(JLIB_ENULL));

	if ((max_sprites < 0) || (max_frames < 0))
		jlib_exit(jlib_msg(JLIB_ENULL));
#endif

	/* create space for sprite system record */
	if ((s = (SS *) malloc(sizeof(SS))) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* fill in max sprite info */
	SPR_MAX_SPRITES(s) = max_sprites;

	JLIB_SPRINTF("%d sprites supported.", max_sprites);

	/* create 'is active' array */
	if (SPR_MAX_SPRITES(s) == 0)
		s->active_sprites = NULL;
	else {
		int ns = SPR_MAX_SPRITES(s);

		if ((s->active_sprites = (UBYTE *) malloc(ns * sizeof(UBYTE))) == NULL)
			jlib_exit(jlib_msg(JLIB_EMALLOC));

		/* turn 'em all off */
		MEM_STORE_LONG(s->active_sprites, 0, ns);
	}

	SPR_NUM_ACTIVE(s) = 0;

	/* create 'sprite' array */
	if (SPR_MAX_SPRITES(s) == 0)
		s->sprites = NULL;
	else
		if ((s->sprites = (SR **) malloc(SPR_MAX_SPRITES(s) * sizeof(SR *))) == NULL)
			jlib_exit(jlib_msg(JLIB_EMALLOC));

	SPR_MAX_FRAMES(s) = max_frames;
	SPR_NUM_LOADED(s) = 0;

	JLIB_SPRINTF("%d frames supported.", max_frames);

	/* create 'sprite data' array */
	if ((s->sprite_data = (SDR **) malloc(SPR_MAX_FRAMES(s) * sizeof(SDR *))) == NULL) {
		jlib_exit(jlib_msg(JLIB_EMALLOC));
	}
	/* NULL them all */
	for (counter = 0; counter < SPR_MAX_FRAMES(s); counter++)
		s->sprite_data[counter] = NULL;

	/* initialise the sprites */
	for (counter = 0; counter < SPR_MAX_SPRITES(s); counter++) {
		if ((s->sprites[counter] = (SR *) calloc(sizeof(SR), 1)) == NULL)
			jlib_exit(jlib_msg(JLIB_EMALLOC));

		s->sprites[counter]->mode = SPR_MODE_WHOLE;
		s->sprites[counter]->animframes = NULL;

		if ((s->sprites[counter]->buffer = (UBYTE *) malloc(SPR_MAX_X * SPR_MAX_Y * sizeof(UBYTE))) == NULL)
			jlib_exit(jlib_msg(JLIB_EMALLOC));
	}

	s->biggest_rle = 0;	/* no loaded sprites yet */

	JLIB_LEAVE;
	return s;
}


/*+----------------------------------------------------------------------+ */
/*|Decrease the memory requirements of a sprite system.                  | */
/*+----------------------------------------------------------------------+ */
void sprite_optimise_mem(SS * s)
{
	int i;

	JLIB_ENTER("sprite_optimise_mem");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
#endif

	JLIB_SPRINTF("Biggest rle = %d",s->biggest_rle);
	JLIB_SPRINTF("Optimise memory saving = %d",((SPR_MAX_X * SPR_MAX_Y) - s->biggest_rle) * SPR_MAX_SPRITES(s));
	
	for (i = 0; i < SPR_MAX_SPRITES(s); i++)
		if (s->sprites[i]->buffer != NULL) {
			if (s->biggest_rle)
				s->sprites[i]->buffer = realloc(s->sprites[i]->buffer,s->biggest_rle);
			else
				s->sprites[i]->buffer = realloc(s->sprites[i]->buffer,(SPR_MAX_X * SPR_MAX_Y));
			if (s->sprites[i]->buffer == NULL)
				jlib_exit(jlib_msg(JLIB_EMALLOC));
		}

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|sprite_turn_on - turn on a sprite                                     | */
/*+----------------------------------------------------------------------+ */
JINLINE void sprite_turn_on(SS * s, int snum)
{
	JLIB_ENTER("sprite_turn_on");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_sprite(s,snum);
#endif

	/* set the sprite as active if it isn't allready */
	if (s->active_sprites[snum] == 0) {
		s->active_sprites[snum] = 1;
		s->no_active++;
	}
	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|sprite_set_an_frame - set a sprites animation frame                   | */
/*+----------------------------------------------------------------------+ */
JINLINE void sprite_set_an_frame(SS * s, int snum, int frame)
{
	JLIB_ENTER("sprite_set_an_frame");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_sprite(s,snum);
	jlib_check_frame(s,frame);
#endif

	s->sprites[snum]->frame = frame;

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|sprite_set_xy - change a sprites position                             | */
/*+----------------------------------------------------------------------+ */
JINLINE void sprite_set_xy(SS * s, int snum, int x, int y)
{
	JLIB_ENTER("sprite_set_xy");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_sprite(s,snum);
#endif

	s->sprites[snum]->y = y;
	s->sprites[snum]->x = x;

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|find_first_free - find the number of the first free sprite.           | */
/*+----------------------------------------------------------------------+ */
JINLINE int sprite_find_first_free(SS * s)
{
	int count;

	JLIB_ENTER("sprite_find_first_free");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
#endif

	/* NOTE : this routine exits if there are no free sprites left. */

	if (SPR_NUM_ACTIVE(s) == (SPR_MAX_SPRITES(s) - 1))
		jlib_exit("No Free Sprites.");

	for (count = 0; s->active_sprites[count] != 0; count++);

	JLIB_LEAVE;
	return count;
}


/*+----------------------------------------------------------------------+ */
/*|find_first_frame - find the number of the first free sprite frame.    | */
/*+----------------------------------------------------------------------+ */
JINLINE int sprite_find_first_frame(SS * s)
{
	JLIB_ENTER("sprite_find_first_frame");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
#endif

	if (SPR_NUM_LOADED(s) >= SPR_MAX_FRAMES(s))
		jlib_exit("No Free Sprite Frames.");

	JLIB_LEAVE;

	return SPR_NUM_LOADED(s);
}


/*+----------------------------------------------------------------------+ */
/*|set_mode - Set the mode of a sprite (fixed point or whole).           | */
/*+----------------------------------------------------------------------+ */
JINLINE void sprite_set_mode(SS * s, int snum, int mode)
{
	JLIB_ENTER("sprite_set_mode");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_sprite(s,snum);

	if ((mode != SPR_MODE_WHOLE) & (mode != SPR_MODE_FIXED))
		jlib_exit("Invalid sprite mode");
#endif

	s->sprites[snum]->mode = mode;

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|set_move_info - Set up the movement info for a sprite.                | */
/*+----------------------------------------------------------------------+ */
JINLINE void sprite_set_move_info(SS * s, int snum,
				  UBYTE speed, BYTE xinc, BYTE yinc)
{
	JLIB_ENTER("sprite_set_move_info");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_sprite(s,snum);
#endif

	s->sprites[snum]->speed = speed;

	if (s->sprites[snum]->mode == SPR_MODE_WHOLE)
		s->sprites[snum]->speed_counter = speed;
	else
		s->sprites[snum]->speed_counter = 0;

	s->sprites[snum]->xinc = xinc;
	s->sprites[snum]->yinc = yinc;

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|set_anim_info - Set up the animation info for a sprite.               | */
/*+----------------------------------------------------------------------+ */
JINLINE void sprite_set_anim_info(SS * s, int snum,
			     UBYTE anspeed, UBYTE noframes, int *animpat)
{
	JLIB_ENTER("sprite_set_anim_info");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite(s,snum);

	if(anspeed){
		if (noframes == 0)
			jlib_exit("Zero animation frames given");
#ifdef JLIB_PARANOID
		else {
			int test;
			JLIB_ENTER("Checking animation frame array");
			test = animpat[0];
			test = animpat[noframes-1];
			JLIB_LEAVE;
		}
#endif
	}
#endif

	s->sprites[snum]->animspeed = anspeed;

	if (s->sprites[snum]->mode == SPR_MODE_WHOLE)
		s->sprites[snum]->animspeed_counter = anspeed;
	else
		s->sprites[snum]->animspeed_counter = 0;

	s->sprites[snum]->noframes = noframes;
	if (noframes)
		s->sprites[snum]->frame = animpat[0];
	s->sprites[snum]->frame_count = 0;
	s->sprites[snum]->animframes = animpat;

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|sprite_turn_off - turn off a sprite                                   | */
/*+----------------------------------------------------------------------+ */
JINLINE void sprite_turn_off(sprite_system * s, int snum)
{
	JLIB_ENTER("sprite_turn_off");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_sprite(s,snum);
#endif

	/* turn off if it was on */
	if (s->active_sprites[snum] == 1) {
		/* restore default values to sprite */
		sprite_set_mode(s, snum, SPR_MODE_WHOLE);
		sprite_set_move_info(s, snum, 0, 0, 0);
		sprite_set_anim_info(s, snum, 0, 0, NULL);

		s->active_sprites[snum] = 0;
		s->no_active--;
	};

	JLIB_LEAVE;
}


/*+---------------------------------------------------------------------+ */
/*|update_anim_and_move - update a sprites posistion and animframe.     | */
/*+---------------------------------------------------------------------+ */
JINLINE void sprite_update_anim_and_move(SS * s, int snum)
{
	SR *current;

	JLIB_ENTER("sprite_update_anim_and_move");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_sprite(s,snum);
#endif

	current = s->sprites[snum];

	if (SPR_IS_ON(s, snum)) {
		if (current->mode == SPR_MODE_WHOLE) {
			/* check movement */
			if (current->speed) {
				if ((--current->speed_counter) == 0) {
					current->speed_counter = current->speed;
					current->x += current->xinc;
					current->y += current->yinc;
				}
			}
			/* check animation */
			if (current->animspeed) {
				if ((--current->animspeed_counter) == 0) {
					current->animspeed_counter = current->animspeed;

					if ((++current->frame_count) == current->noframes)
						current->frame_count = 0;
					current->frame = current->animframes[current->frame_count];
				}
			}
		}
		else {
			/* check fixed point movement */
			if (current->speed) {
				current->speed_counter += current->speed;
				current->x += ((current->speed_counter >> 4) * current->xinc);
				current->y += ((current->speed_counter >> 4) * current->yinc);
				current->speed_counter = (current->speed_counter & 0xf);

			}
			/* check fixed point animation */
			if (current->animspeed) {
				current->animspeed_counter += current->animspeed;
				current->frame_count += (current->animspeed_counter >> 4);

				while (current->frame_count >= current->noframes)
					current->frame_count -= current->noframes;
				current->frame = current->animframes[current->frame_count];
			}
			current->animspeed_counter = (current->animspeed_counter & 0xf);
		}
	}
	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|sprite_update_all_anim_and_move                                       | */
/*+----------------------------------------------------------------------+ */
void sprite_update_all_anim_and_move(SS * s)
{
	int count;

	JLIB_ENTER("sprite_update_all_anim_and_move");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
#endif

	for (count = 0; count < s->no_sprites; count++)
		if (SPR_IS_ON(s, count))
			sprite_update_anim_and_move(s, count);

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|sprite_do_all_anim_and_move_n_times                                   | */
/*+----------------------------------------------------------------------+ */
void sprite_do_all_anim_and_move_n_times(SS * s, int n)
{
	int count, i;

	JLIB_ENTER("sprite_do_all_anim_and_move_n_times");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
#endif

	for (count = 0; count < s->no_sprites; count++)
		if (SPR_IS_ON(s, count))
			for (i = n; i != 0; i--)
				sprite_update_anim_and_move(s, count);

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|__jlib_intersect - this routine is private!                           | */
/*+----------------------------------------------------------------------+ */
JINLINE int __jlib_intersect(int x1, int y1, UBYTE * r1, int x2, int y2, UBYTE * r2)
{
	if (((x1 + r1[0]) < (x2 + r2[2])) &&
	    ((x1 + r1[2]) > (x2 + r2[0])) &&
	    ((y1 + r1[1]) < (y2 + r2[3])) &&
	    ((y1 + r1[3]) > (y2 + r2[1]))) {
		return 1;
	}
	return 0;
}


/*+----------------------------------------------------------------------+ */
/*|sprite_do_intersect                                                   | */
/*+----------------------------------------------------------------------+ */
int sprite_do_intersect(SS * s, int s1, int s2)
{
	int s1x, s1y, s2x, s2y;
	UBYTE *s1rect, *s2rect;
	int s1count, s2count;

	JLIB_ENTER("sprite_do_intersect");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_sprite(s,s1);
	jlib_check_sprite(s,s2);
#endif

	/* if either sprite is off,  return no */
	if ((!SPR_IS_ON(s, s1)) || (!SPR_IS_ON(s, s2))) {
		JLIB_LEAVE;
		return 0;
	}

	/* set up preliminary information */
	s1x = s->sprites[s1]->x;
	s1y = s->sprites[s1]->y;
	s2x = s->sprites[s2]->x;
	s2y = s->sprites[s2]->y;

	s1rect = s->sprite_data[s->sprites[s1]->frame]->rect_coords;
	s2rect = s->sprite_data[s->sprites[s2]->frame]->rect_coords;

	/* for each bounding rectangle of sprite 1,  check each rect of sprite 2 */
	for (s1count = s->sprite_data[s->sprites[s1]->frame]->no_rects; s1count != 0; s1count--) {
		for (s2count = s->sprite_data[s->sprites[s2]->frame]->no_rects; s2count != 0; s2count--) {
			if (__jlib_intersect(s1x, s1y, s1rect, s2x, s2y, s2rect)) {
				JLIB_LEAVE;
				return 1;
			}
			 else {
				/* move to the 2nd sprites next rectangle coordinates */
				s2rect += 4;
			}
		}
		s2rect -= (s->sprite_data[s->sprites[s2]->frame]->no_rects * 4);

		s1rect += 4;	/* move to the 1st sprites next rect coords */
	}

	JLIB_LEAVE;
	return 0;
}


/*+----------------------------------------------------------------------+ */
/*|sprites_do_intersect                                                  | */
/*+----------------------------------------------------------------------+ */
int sprites_do_intersect(SS * sys1, int s1, SS * sys2, int s2)
{
	int s1x, s1y, s2x, s2y;
	UBYTE *s1rect, *s2rect;
	int s1count, s2count;

	JLIB_ENTER("sprites_do_intersect");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(sys1);
	jlib_check_sprite_system(sys2);
	jlib_check_sprite(sys1,s1);
	jlib_check_sprite(sys2,s2);
#endif

	/* if either sprite is off,  return no */
	if ((!SPR_IS_ON(sys1, s1)) || (!SPR_IS_ON(sys2, s2))) {
		JLIB_LEAVE;
		return 0;
	}

	/* set up preliminary information */
	s1x = sys1->sprites[s1]->x;
	s1y = sys1->sprites[s1]->y;
	s2x = sys2->sprites[s2]->x;
	s2y = sys2->sprites[s2]->y;

	s1rect = sys1->sprite_data[sys1->sprites[s1]->frame]->rect_coords;
	s2rect = sys2->sprite_data[sys2->sprites[s2]->frame]->rect_coords;

	/* for each bounding rectangle of sprite 1,  check each rect of sprite 2 */
	for (s1count = sys1->sprite_data[sys1->sprites[s1]->frame]->no_rects; s1count != 0; s1count--) {
		for (s2count = sys2->sprite_data[sys2->sprites[s2]->frame]->no_rects; s2count != 0; s2count--) {
			if (__jlib_intersect(s1x, s1y, s1rect, s2x, s2y, s2rect)) {
				JLIB_LEAVE;
				return 1;
			} else {
				/* move to the 2nd sprites next rectangle coordinates */
				s2rect += 4;
			}
		}
		s2rect -= (sys2->sprite_data[sys2->sprites[s2]->frame]->no_rects * 4);
		s1rect += 4;	/* move to the 1st sprites next rect coords */
	}

	JLIB_LEAVE;
	return 0;
}



/*+-----------------------------------------------------------------------+ */
/*|sprite_free - free the memory used by a sprite system.                 | */
/*+-----------------------------------------------------------------------+ */
SS *sprite_free(SS * s)
{
	int counter;

	JLIB_ENTER("sprite_free");

	if (s == NULL) {
		JLIB_LEAVE;
		return NULL;
	}
	if (s->sprites != NULL)
		for (counter = 0; counter < SPR_MAX_SPRITES(s); counter++) {
			if (s->sprites[counter]->buffer != NULL)
				free(s->sprites[counter]->buffer);
			free(s->sprites[counter]);
		}

	free(s->sprites);

	if (s->sprite_data != NULL) {
		for (counter = 0; counter < SPR_MAX_FRAMES(s); counter++) {
			if (s->sprite_data[counter] != NULL) {
				if (s->sprite_data[counter]->data != NULL)
					free(s->sprite_data[counter]->data);
				if (s->sprite_data[counter] != NULL)
					free(s->sprite_data[counter]->pattern);
				if (s->sprite_data[counter] != NULL)
					free(s->sprite_data[counter]->rect_coords);
				if (s->sprite_data[counter] != NULL)
					free(s->sprite_data[counter]);
			}
		}
	}
	free(s->sprite_data);

	free(s->active_sprites);

	free(s);

	JLIB_LEAVE;
	return NULL;
}


#define _MY_HEIGHT  (s->sprite_data[frame]->height)
#define _MY_WIDTH   (s->sprite_data[frame]->width)

/*+-----------------------------------------------------------------------+ */
/*|sprite_build_from_buff - build a sprite data frame from a buffer.      | */
/*+-----------------------------------------------------------------------+ */
void sprite_build_from_buff(SS * s, int frame, buffer_rec * bf, int x1, int y1, int x2, int y2)
{
	int width;
	UBYTE *store;

	JLIB_ENTER("sprite_build_from_buff");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_buffer(bf);
#endif

	width = B_X_SIZE(bf);

	if (x1 > x2)
		JLIB_SWAP(x2, x1);
	if (y1 > y2)
		JLIB_SWAP(y2, y1);

	/* check width & height,  truncate if too big */
	if ((x2 - x1) > SPR_MAX_X)
		x2 = x1 + SPR_MAX_X;

	if ((y2 - y1) > SPR_MAX_Y)
		y2 = y1 + SPR_MAX_Y;

	/* check to see if the sprite frame already exists */
	if (s->sprite_data[frame] != NULL) {
		JLIB_SPRINTF("Overwriting Frame %d.", frame);

		/* kill existing sprite frame */
		free(s->sprite_data[frame]->data);
		free(s->sprite_data[frame]->pattern);

		if (s->sprite_data[frame]->rect_coords != NULL)
			free(s->sprite_data[frame]->rect_coords);
		free(s->sprite_data[frame]);
	}
	else
		s->number_loaded++;

	/* make a sprite_data_rec for the sprite */
	if ((s->sprite_data[frame] = (SDR *) malloc(sizeof(SDR))) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* set width and height */
	_MY_HEIGHT = y2 - y1 + 1;
	_MY_WIDTH = x2 - x1 + 1;

	/* create with no bounding rectangles */
	s->sprite_data[frame]->no_rects = 0;
	s->sprite_data[frame]->rect_coords = NULL;

	if ((s->sprite_data[frame]->data = (UBYTE *) malloc((_MY_HEIGHT * _MY_WIDTH) * sizeof(UBYTE))) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	store = s->sprite_data[frame]->data;

	/* copy in the data */
	for (; y1 <= y2; y1++) {
		int j;
		UBYTE *start = B_OFFSET(bf, y1);
		for (j = x1; j <= x2; j++) {
			*store = start[j];
			store++;
		}
	}

	/* generate the RLE pattern */
	generate_rle(s,s->sprite_data[frame]);
	sprite_optimise_mem(s);

#ifndef JLIB_PRODUCTION
	jlib_check_frame(s,frame);
#endif
	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Add a bounding rectangle to a sprite frame.                            | */
/*+-----------------------------------------------------------------------+ */
void sprite_add_rect(SS * s, int frame, UBYTE x1, UBYTE y1, UBYTE x2, UBYTE y2)
{
	UBYTE *rc;

	JLIB_ENTER("sprite_add_rect");

#ifndef JLIB_PRODUCTION
	jlib_check_sprite_system(s);
	jlib_check_frame(s,frame);
#endif

	/* add the rectangle to the sprite */
	s->sprite_data[frame]->no_rects++;	/* increment count */

	/* increase memory to hold new coords */
	s->sprite_data[frame]->rect_coords = 
		(UBYTE *) realloc(s->sprite_data[frame]->rect_coords, (s->sprite_data[frame]->no_rects * 4));

	if (s->sprite_data[frame] == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* write out the rectangles */
	rc = s->sprite_data[frame]->rect_coords + ((s->sprite_data[frame]->no_rects - 1) * 4);
	rc[0] = x1;
	rc[1] = y1;
	rc[2] = x2;
	rc[3] = y2;

	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Remove the background saving buffers from a sprite system.             | */
/*+-----------------------------------------------------------------------+ */
void sprite_kill_buffers(SS * s)
{
	int i;

	JLIB_ENTER("sprite_kill_buffers");

#ifndef JLIB_PRODUCTION
	if (s == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
	jlib_check_sprite_system(s);
#endif

	for (i = 0; i < SPR_MAX_SPRITES(s); i++) {
		if (s->sprites[i]->buffer != NULL) {
			free(s->sprites[i]->buffer);
			s->sprites[i]->buffer = NULL;
		}
	}

	JLIB_LEAVE;
}