/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details.  */
#include <jlib.h>


/*+----------------------------------------------------------------------+ */
/*|stamp a sprite frame shadow on buff without clipping                  | */
/*+----------------------------------------------------------------------+ */
void buff_stamp_sprite_colorNC(sprite_system * ssys, int frame, buffer_rec * buff, int x, int y, UBYTE col)
{
	UBYTE *pattern, *out;
	int bwidth, height;

	JLIB_ENTER("buff_stamp_sprite_colorNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(ssys);
	jlib_check_frame(ssys,frame);
#endif

	pattern = ssys->sprite_data[frame]->pattern;
	height = ssys->sprite_data[frame]->height;
	bwidth = B_X_SIZE(buff) - ssys->sprite_data[frame]->width;
	out = B_OFFSET(buff, y) + x;

	while (height--) {
		int width = *pattern++;
		while (width--) {
			UBYTE datum = *pattern++;
			SPR_STAMP_COLOR(out, col, datum);
			out += (datum & 15);
		}
		out += bwidth;
	}

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Stamp a sprite frame shadow on buff with clipping.                    | */
/*+----------------------------------------------------------------------+ */
void buff_stamp_sprite_color(sprite_system * ssys, int frame, buffer_rec * buff, int x, int y, UBYTE col)
{
	int width, height, clip = 0;
	UBYTE *data = (UBYTE *) 0;

	JLIB_ENTER("buff_stamp_sprite_color");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(ssys);
	jlib_check_frame(ssys,frame);
#endif

	width = ssys->sprite_data[frame]->width;
	height = ssys->sprite_data[frame]->height;

	/* x clipping */
	if (x + width > B_CX2(buff)) {
		width = B_CX2(buff) - x + 1;
		clip |= C_RIGHT;
	}
	else
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (x < B_CX1(buff)) {
		x -= B_CX1(buff);
		width += x;
		data -= x;
		x = B_CX1(buff);
		clip |= C_LEFT;
	}
	else
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}

	if (clip) {
		/* worst case: x is clipped */
		int sprwidth = ssys->sprite_data[frame]->width;
		UBYTE *out;
		data += (unsigned int) ssys->sprite_data[frame]->data;

		/* y clipping */
		if (y + height > B_CY2(buff)) {
			height = B_CY2(buff) - y + 1;
			clip |= C_DOWN;
		}
		else
			if (y + height < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y < B_CY1(buff)) {
			y -= B_CY1(buff);
			height += y;
			do {
				data += sprwidth;
			} while (++y != 0);
			y = B_CY1(buff);
			clip |= C_UP;
		}
		else
			if (y > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		out = B_OFFSET(buff, y) + x;

		while (height--) {
			int w = width;
			while (w--)
				if (data[w])
					out[w] = col;
				else
					out[w] = 0;
			out += B_X_SIZE(buff);
			data += sprwidth;
		}
		JLIB_LEAVE;
		return;
	}
	else {
		UBYTE *pattern = ssys->sprite_data[frame]->pattern;

		/* y clipping */
		if (y + height > B_CY2(buff)) {
			height = B_CY2(buff) - y + 1;
			clip |= C_DOWN;
		}
		else
			if (y + height < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y < B_CY1(buff)) {
			y -= B_CY1(buff);
			height += y;
			do {
				pattern += *pattern + 1;	/* skip through RLE pattern */
			} while (++y != 0);
			y = B_CY1(buff);
			clip |= C_UP;
		}
		else
			if (y > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (clip) {
			/* average case: y is clipped */
			UBYTE *out = B_OFFSET(buff, y) + x;
			int bwidth = B_X_SIZE(buff) - width;

			while (height--) {
				int w = *pattern++;
				while (w--) {
					UBYTE datum = *pattern++;
					SPR_STAMP_COLOR(out, col, datum);
					out += (datum & 15);
				}
				out += bwidth;
			}
		}
		else
			buff_stamp_sprite_colorNC(ssys, frame, buff, x, y, col);
	}
	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Stencil a sprite frame shadow on buff without clipping                | */
/*+----------------------------------------------------------------------+ */
void buff_stencil_sprite_colorNC(sprite_system * ssys, int frame, buffer_rec * buff, int x, int y, UBYTE col)
{
	UBYTE *pattern, *out;
	int bwidth, height;

	JLIB_ENTER("buff_stencil_sprite_colorNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(ssys);
	jlib_check_frame(ssys,frame);
#endif

	pattern = ssys->sprite_data[frame]->pattern;
	height = ssys->sprite_data[frame]->height;
	bwidth = B_X_SIZE(buff) - ssys->sprite_data[frame]->width;
	out = B_OFFSET(buff, y) + x;

	while (height--) {
		int width = *pattern++;
		while (width--) {
			UBYTE datum = *pattern++;
			SPR_STENCIL_COLOR(out, col, datum);
			out += (datum & 15);
		}
		out += bwidth;
	}

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Stencil a sprite frame shadow on buff without clipping.               | */
/*+----------------------------------------------------------------------+ */
void buff_stencil_sprite_color(sprite_system * ssys, int frame, buffer_rec * buff, int x, int y, UBYTE col)
{
	int width, height, clip = 0;
	UBYTE *data = (UBYTE *) 0;

	JLIB_ENTER("buff_stencil_sprite_color");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(ssys);
	jlib_check_frame(ssys,frame);
#endif

	width = ssys->sprite_data[frame]->width;
	height = ssys->sprite_data[frame]->height;

	/* x clipping */
	if (x + width > B_CX2(buff)) {
		width = B_CX2(buff) - x + 1;
		clip |= C_RIGHT;
	}
	else
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (x < B_CX1(buff)) {
		x -= B_CX1(buff);
		width += x;
		data -= x;
		x = B_CX1(buff);
		clip |= C_LEFT;
	}
	else
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}

	if (clip) {
		/* worst case: x is clipped */
		int sprwidth = ssys->sprite_data[frame]->width;
		UBYTE *out;
		data += (unsigned int) ssys->sprite_data[frame]->data;

		/* y clipping */
		if (y + height > B_CY2(buff)) {
			height = B_CY2(buff) - y + 1;
			clip |= C_DOWN;
		}
		else
			if (y + height < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y < B_CY1(buff)) {
			y -= B_CY1(buff);
			height += y;
			do {
				data += sprwidth;
			} while (++y != 0);
			y = B_CY1(buff);
			clip |= C_UP;
		}
		else
			if (y > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		out = B_OFFSET(buff, y) + x;

		while (height--) {
			int w = width;
			while (w--)
				if (data[w])
					out[w] = col;
			out += B_X_SIZE(buff);
			data += sprwidth;
		}
		JLIB_LEAVE;
		return;
	}
	else {
		UBYTE *pattern = ssys->sprite_data[frame]->pattern;

		/* y clipping */
		if (y + height > B_CY2(buff)) {
			height = B_CY2(buff) - y + 1;
			clip |= C_DOWN;
		}
		else
			if (y + height < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y < B_CY1(buff)) {
			y -= B_CY1(buff);
			height += y;
			do {
				pattern += *pattern + 1;	/* skip through RLE pattern */
			} while (++y != 0);
			y = B_CY1(buff);
			clip |= C_UP;
		}
		else
			if (y > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (clip) {
			/* average case: y is clipped */
			UBYTE *out = B_OFFSET(buff, y) + x;
			int bwidth = B_X_SIZE(buff) - width;

			while (height--) {
				int w = *pattern++;
				while (w--) {
					UBYTE datum = *pattern++;
					SPR_STENCIL_COLOR(out, col, datum);
					out += (datum & 15);
				}
				out += bwidth;
			}
		}
		else
			buff_stencil_sprite_colorNC(ssys, frame, buff, x, y, col);
	}
	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Stamp a textured sprite frame out of sbuff to buff without clipping  | */
/*+----------------------------------------------------------------------+ */
void buff_stamp_sprite_buffNC(sprite_system * ssys, int frame, buffer_rec * buff, int x, int y, buffer_rec * sbuff)
{
	UBYTE *pattern, *out, *texture;
	int bwidth, twidth, height;

	JLIB_ENTER("buff_stamp_sprite_buffNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_buffer(sbuff);
	jlib_check_sprite_system(ssys);
	jlib_check_frame(ssys,frame);
#endif

	pattern = ssys->sprite_data[frame]->pattern;
	height = ssys->sprite_data[frame]->height;
	bwidth = B_X_SIZE(buff) - ssys->sprite_data[frame]->width;
	out = B_OFFSET(buff, y) + x;
	twidth = B_X_SIZE(sbuff) - ssys->sprite_data[frame]->width;
	texture = B_BUFF_PTR(sbuff);

	while (height--) {
		int width = *pattern++;
		while (width--) {
			UBYTE datum = *pattern++;
			SPR_STAMP_COPY(out, texture, datum);
			out += (datum & 15);
			texture += (datum & 15);
		}
		out += bwidth;
		texture += twidth;
	}

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Stamp a textured sprite frame out of sbuff to buff with clipping     | */
/*+----------------------------------------------------------------------+ */
void buff_stamp_sprite_buff(sprite_system * ssys, int frame, buffer_rec * buff, int x, int y, buffer_rec * sbuff)
{
	int twidth, width, height, clip = 0;
	UBYTE *data = (UBYTE *) 0, *texture;

	JLIB_ENTER("buff_stamp_sprite_buff");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_buffer(sbuff);
	jlib_check_sprite_system(ssys);
	jlib_check_frame(ssys,frame);
#endif

	width = ssys->sprite_data[frame]->width;
	height = ssys->sprite_data[frame]->height;
	twidth = B_X_SIZE(sbuff);
	texture = B_BUFF_PTR(sbuff);

#ifndef JLIB_PRODUCTION
	/* Check texture size */
	if ((B_MAX_X(sbuff) < width) || (B_MAX_Y(sbuff) < height))
		jlib_exit("Texture too small");
#endif

	/* x clipping */
	if (x + width > B_CX2(buff)) {
		width = B_CX2(buff) - x + 1;
		clip |= C_RIGHT;
	}
	else
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (x < B_CX1(buff)) {
		x -= B_CX1(buff);
		width += x;
		data -= x;
		texture -= x;
		x = B_CX1(buff);
		clip |= C_LEFT;
	}
	else
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}

	if (clip) {
		/* worst case: x is clipped */
		int sprwidth = ssys->sprite_data[frame]->width;
		UBYTE *out;
		data += (unsigned int) ssys->sprite_data[frame]->data;

		/* y clipping */
		if (y + height > B_CY2(buff)) {
			height = B_CY2(buff) - y + 1;
			clip |= C_DOWN;
		}
		else
			if (y + height < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y < B_CY1(buff)) {
			y -= B_CY1(buff);
			height += y;
			do {
				data += sprwidth;
				texture += twidth;
			} while (++y != 0);
			y = B_CY1(buff);
			clip |= C_UP;
		}
		else
			if (y > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		out = B_OFFSET(buff, y) + x;

		while (height--) {
			int w = width;
			while (w--)
				if (data[w])
					out[w] = texture[w];
				else
					out[w] = 0;

			out += B_X_SIZE(buff);
			data += sprwidth;
			texture += twidth;
		}
		JLIB_LEAVE;
		return;
	} else {
		UBYTE *pattern = ssys->sprite_data[frame]->pattern;

		/* y clipping */
		if (y + height > B_CY2(buff)) {
			height = B_CY2(buff) - y + 1;
			clip |= C_DOWN;
		}
		else
			if (y + height < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y < B_CY1(buff)) {
			y -= B_CY1(buff);
			height += y;
			do {
				pattern += *pattern + 1;	/* skip through RLE pattern */
				texture += twidth;
			} while (++y != 0);
			y = B_CY1(buff);
			clip |= C_UP;
		}
		else
			if (y > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (clip) {
			/* average case: y is clipped */
			UBYTE *out = B_OFFSET(buff, y) + x;
			int bwidth = B_X_SIZE(buff) - width;
			twidth -= ssys->sprite_data[frame]->width;

			while (height--) {
				int w = *pattern++;
				while (w--) {
					UBYTE datum = *pattern++;
					SPR_STAMP_COPY(out, texture, datum);
					out += (datum & 15);
					texture += (datum & 15);
				}
				out += bwidth;
				texture += twidth;
			}
		}
		else
			/* best case: no clipping */
			buff_stamp_sprite_buffNC(ssys, frame, buff, x, y, sbuff);
	}
	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Stencil a textured sprite frame out of sbuff to buff without clipping | */
/*+----------------------------------------------------------------------+ */
void buff_stencil_sprite_buffNC(sprite_system * ssys, int frame, buffer_rec * buff, int x, int y, buffer_rec * sbuff)
{
	UBYTE *pattern, *out, *texture;
	int bwidth, twidth, height;

	JLIB_ENTER("buff_stencil_sprite_buffNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_buffer(sbuff);
	jlib_check_sprite_system(ssys);
	jlib_check_frame(ssys,frame);
#endif

	pattern = ssys->sprite_data[frame]->pattern;
	height = ssys->sprite_data[frame]->height;
	bwidth = B_X_SIZE(buff) - ssys->sprite_data[frame]->width;
	out = B_OFFSET(buff, y) + x;
	twidth = B_X_SIZE(sbuff) - ssys->sprite_data[frame]->width;
	texture = B_BUFF_PTR(sbuff);

	while (height--) {
		int width = *pattern++;
		while (width--) {
			UBYTE datum = *pattern++;
			SPR_STENCIL_COPY(out, texture, datum);
			out += (datum & 15);
			texture += (datum & 15);
		}
		out += bwidth;
		texture += twidth;
	}

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Stencil a textured sprite frame out of sbuff to buff with clipping    | */
/*+----------------------------------------------------------------------+ */
void buff_stencil_sprite_buff(sprite_system * ssys, int frame, buffer_rec * buff, int x, int y, buffer_rec * sbuff)
{
	int twidth, width, height, clip = 0;
	UBYTE *data = (UBYTE *) 0, *texture;

	JLIB_ENTER("buff_stencil_sprite_buff");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_buffer(sbuff);
	jlib_check_sprite_system(ssys);
	jlib_check_frame(ssys,frame);
#endif

	width = ssys->sprite_data[frame]->width;
	height = ssys->sprite_data[frame]->height;
	twidth = B_X_SIZE(sbuff);
	texture = B_BUFF_PTR(sbuff);

#ifndef JLIB_PRODUCTION
	/* Check texture size */
	if ((B_MAX_X(sbuff) < width) || (B_MAX_Y(sbuff) < height))
		jlib_exit("Texture too small");
#endif

	/* x clipping */
	if (x + width > B_CX2(buff)) {
		width = B_CX2(buff) - x + 1;
		clip |= C_RIGHT;
	}
	else
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (x < B_CX1(buff)) {
		x -= B_CX1(buff);
		width += x;
		data -= x;
		texture -= x;
		x = B_CX1(buff);
		clip |= C_LEFT;
	}
	else
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}

	if (clip) {
		/* worst case: x is clipped */
		int sprwidth = ssys->sprite_data[frame]->width;
		UBYTE *out;
		data += (unsigned int) ssys->sprite_data[frame]->data;

		/* y clipping */
		if (y + height > B_CY2(buff)) {
			height = B_CY2(buff) - y + 1;
			clip |= C_DOWN;
		}
		else
			if (y + height < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y < B_CY1(buff)) {
			y -= B_CY1(buff);
			height += y;
			do {
				data += sprwidth;
				texture += twidth;
			} while (++y != 0);
			y = B_CY1(buff);
			clip |= C_UP;
		}
		else
			if (y > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		out = B_OFFSET(buff, y) + x;

		while (height--) {
			int w = width;
			while (w--)
				if (data[w])
					out[w] = texture[w];

			out += B_X_SIZE(buff);
			data += sprwidth;
			texture += twidth;
		}
		JLIB_LEAVE;
		return;
	} else {
		UBYTE *pattern = ssys->sprite_data[frame]->pattern;

		/* y clipping */
		if (y + height > B_CY2(buff)) {
			height = B_CY2(buff) - y + 1;
			clip |= C_DOWN;
		}
		else
			if (y + height < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y < B_CY1(buff)) {
			y -= B_CY1(buff);
			height += y;
			do {
				pattern += *pattern + 1;	/* skip through RLE pattern */
				texture += twidth;
			} while (++y != 0);
			y = B_CY1(buff);
			clip |= C_UP;
		}
		else
			if (y > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (clip) {
			/* average case: y is clipped */
			UBYTE *out = B_OFFSET(buff, y) + x;
			int bwidth = B_X_SIZE(buff) - width;
			twidth -= ssys->sprite_data[frame]->width;

			while (height--) {
				int w = *pattern++;
				while (w--) {
					UBYTE datum = *pattern++;
					SPR_STENCIL_COPY(out, texture, datum);
					out += (datum & 15);
					texture += (datum & 15);
				}
				out += bwidth;
				texture += twidth;
			}
		}
		else
			/* best case: no clipping */
			buff_stencil_sprite_buffNC(ssys, frame, buff, x, y, sbuff);
	}
	JLIB_LEAVE;
}