/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <jlib.h>

#define	BR buffer_rec


/*+------------------------------------------------------------------------+ */
/*|Draw part of src scaled into dest at a given pos without clipping.      | */
/*+------------------------------------------------------------------------+ */
void buff_scale_buff_toNC(BR *d, int dx1, int dy1, int dx2, int dy2, BR *s, int x1, int y1, int x2, int y2)
{
	int rxs, xx, yy, y_step, width, height;
	int swidth, sheight, dwidth, sx;
	UBYTE *dptr, *sptr, *prev_sptr = NULL;

	JLIB_ENTER("buff_scale_buff_toNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(s);
	jlib_check_buffer(d);
#endif

	/* sort coordinates if needed */
	if (x2 < x1) JLIB_SWAP(x1, x2);
	if (y2 < y1) JLIB_SWAP(y1, y2);
	if (dx2 < dx1) JLIB_SWAP(dx1, dx2);
	if (dy2 < dy1) JLIB_SWAP(dy1, dy2);

	swidth = x2 - x1 + 1;
	sheight = y2 - y1 + 1;

	dptr = B_OFFSET(d, dy1) + dx1;
	sptr = B_OFFSET(s, y1) + x1;

	width = dx2 - dx1 + 1;
	height = dy2 - dy1 + 1;
	dwidth = B_X_SIZE(d);
	sx = B_X_SIZE(s);
	rxs = (width + swidth - 1);

	if (sheight < height)
		y_step = sheight;
	else
		y_step = height - 1;

	for (yy = height; yy > 0; yy--) {
		UBYTE *dsav = dptr;

		y_step -= sheight;

		if (y_step > 0)
			MEM_COPY_ALIGN_1(prev_sptr, dptr, width);
		else {
			int x_step = rxs;
			UBYTE *ssav = sptr;

			prev_sptr = dptr;

			for (xx = width; xx > 0; xx--) {
				x_step -= swidth;
				if (x_step < 0)
					do {
						sptr++;
						x_step += width;
					} while (x_step < 0);
				*dptr = *sptr;
				dptr++;
			}
			sptr = ssav;
			do {
				sptr += sx;
				y_step += height;
			} while (y_step < 0);
		}
		dptr = (dsav + dwidth);
	}

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|Draw part of src scaled into dest at a given pos with clipping.         | */
/*+------------------------------------------------------------------------+ */
void buff_scale_buff_to(BR * d, int dx1, int dy1, int dx2, int dy2, BR * s, int x1, int y1, int x2, int y2)
{
	int swidth, sheight, dwidth, dheight, diff;
	int smin_x, smin_y, smax_x, smax_y;
	int dmin_x, dmin_y, dmax_x, dmax_y;

	JLIB_ENTER("buff_scale_buff_to");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(s);
	jlib_check_buffer(d);
#endif

	/* sort coordinates if needed */
	if (x2 < x1) JLIB_SWAP(x1, x2);
	if (y2 < y1) JLIB_SWAP(y1, y2);
	if (dx2 < dx1) JLIB_SWAP(dx1, dx2);
	if (dy2 < dy1) JLIB_SWAP(dy1, dy2);

	dmin_x = B_CX1(d);
	dmax_x = B_CX2(d);
	dmin_y = B_CY1(d);
	dmax_y = B_CY2(d);

	smin_x = B_CX1(s);
	smax_x = B_CX2(s);
	smin_y = B_CY1(s);
	smax_y = B_CY2(s);

	swidth = x2 - x1 + 1;
	sheight = y2 - y1 + 1;
	dwidth = dx2 - dx1 + 1;
	dheight = dy2 - dy1 + 1;

	/* handle trivial rejection cases */
	if ((dx1 > dmax_x) || (dy1 > dmax_y) || (dx2 < dmin_x) || (dy2 < dmin_y)) {
		JLIB_LEAVE;
		return;
	}
	if ((x1 > smax_x) || (y1 > smax_y) || (x2 < smin_x) || (y2 < smin_y)) {
		JLIB_LEAVE;
		return;
	}

	/* if src area is clipped, shrink dest area by the appropriate amount */
	if (x1 < smin_x) {
		diff = smin_x - x1;
		dx1 += ((diff * dwidth) / swidth);
		x1 = smin_x;
	}
	if (x2 > smax_x) {
		diff = x2 - smax_x;
		dx2 -= ((diff * dwidth) / swidth);
		x2 = smax_x;
	}
	if (y1 < smin_y) {
		diff = smin_y - y1;
		dy1 += ((diff * dwidth) / swidth);
		y1 = smin_y;
	}
	if (y2 > smax_y) {
		diff = y2 - smax_y;
		dy2 -= ((diff * dwidth) / swidth);
		y2 = smax_y;
	}

	/* if dest area is clipped, shrink src area by the appropriate amount */
	if (dx1 < dmin_x) {
		diff = dmin_x - dx1;
		x1 += ((diff * swidth) / dwidth);
		dx1 = dmin_x;
	}
	if (dx2 > dmax_x) {
		diff = dx2 - dmax_x;
		x2 -= ((diff * swidth) / dwidth);
		dx2 = dmax_x;
	}
	if (dy1 < dmin_y) {
		diff = dmin_y - dy1;
		y1 += ((diff * swidth) / dwidth);
		dy1 = dmin_y;
	}
	if (dy2 > dmax_y) {
		diff = dy2 - dmax_y;
		y2 -= ((diff * swidth) / dwidth);
		dy2 = dmax_y;
	}

	/* re-check rejection cases */
	if ((dx1 > dmax_x) || (dy1 > dmax_y) || (dx2 < dmin_x) || (dy2 < dmin_y)) {
		JLIB_LEAVE;
		return;
	}
	if ((x1 > smax_x) || (y1 > smax_y) || (x2 < smin_x) || (y2 < smin_y)) {
		JLIB_LEAVE;
		return;
	}

	/* do scale with clipped parameters */
	buff_scale_buff_toNC(d, dx1, dy1, dx2, dy2, s, x1, y1, x2, y2);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*| Draw src scaled into dest in the given coordinates without clipping.   | */
/*+------------------------------------------------------------------------+ */
void buff_scale_full_buff_toNC(BR *d, int x1, int y1, int x2, int y2, BR *s)
{
	JLIB_ENTER("buff_scale_full_buff_toNC");

	buff_scale_buff_toNC(d, x1, y1, x2, y2, s, 0, 0, B_MAX_X(s), B_MAX_Y(s));

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*| Draw src scaled into dest in the given coordinates with clipping.      | */
/*+------------------------------------------------------------------------+ */
void buff_scale_full_buff_to(BR *d, int x1, int y1, int x2, int y2, BR* s)
{
	JLIB_ENTER("buff_scale_full_buff_to");

	buff_scale_buff_to(d, x1, y1, x2, y2, s, 0, 0, B_MAX_X(s), B_MAX_Y(s));

	JLIB_LEAVE;
}