/************************************************************************/
/* playing.c, the tracking code.                                        */
/************************************************************************/
#include "playing.h"

extern unsigned char error[]; // error string
extern samrec samples[];
extern channelinforec channel[];
extern void mastervolume(signed int vol);
extern output_type _output;   // output device
instrument_type *instruments;

void hoida_effyt(int rc, row_type row);
void hoida_xm_volume_effy(int rc, row_type row);
void do_row(int rc, row_type row);

int playing_master;

void playing_mastervolume(signed int vol)
{
 if (vol > 64) vol = 64; else if (vol < 0) vol = 0;
 playing_master = vol;
}

unsigned int finetunes[16]=
{
 8363,8413,8463,8529,8581,8651,8723,8757,
 7895,7941,7985,8046,8107,8169,8232,8280
};

unsigned int periods[132]= // for mod and s3m
{
27392,25856,24384,23040,21696,20480,19328,18240,17216,16256,15360,14496,
13696,12928,12192,11520,10848,10240, 9664, 9120, 8608, 8128, 7680, 7248,
 6848, 6464, 6096, 5760, 5424, 5120, 4832, 4560, 4304, 4064, 3840, 3624,
 3424, 3232, 3048, 2880, 2712, 2560, 2416, 2280, 2152, 2032, 1920, 1812,
 1712, 1616, 1524, 1440, 1356, 1280, 1208, 1140, 1076, 1016,  960,  906,
  856,  808,  762,  720,  678,  640,  604,  570,  538,  508,  480,  453,
  428,  404,  381,  360,  339,  320,  302,  285,  269,  254,  240,  226,
  214,  202,  190,  180,  170,  160,  151,  143,  135,  127,  120,  113,
  107,  101,   95,   90,   85,   80,   75,   71,   67,   63,   60,   56,
   53,   50,   47,   45,   42,   40,   37,   35,   33,   31,   30,   28,
   26,   25,   23,   22,   21,   20,   18,   17,   16,   15,   15,   14
}; // from fmoddoc2


ulong xm_PeriodTab[12 * 8 + 8] = // for xm
{
      907,900,894,887,881,875,868,862,856,850,844,838,832,826,820,814,
      808,802,796,791,785,779,774,768,762,757,752,746,741,736,730,725,
      720,715,709,704,699,694,689,684,678,675,670,665,660,655,651,646,
      640,636,632,628,623,619,614,610,604,601,597,592,588,584,580,575,
      570,567,563,559,555,551,547,543,538,535,532,528,524,520,516,513,
      508,505,502,498,494,491,487,484,480,477,474,470,467,463,460,457,
      453,450,447,443,440,437,434,431
}; // from xm.txt. the last 8 are from midas.

static volatile void begin_of_playing(){}

inline ulong xm_lin_period(slong Note, slong FineTune)
{
  return (10*12*16*4 - Note*16*4 - FineTune/2);
}
/*
inline ulong xm_amiga_period(slong Note, slong FineTune)
{
 slong nuotti   = (Note % 12);
 slong oktaavi  = (Note / 12);
 slong kfine    = (FineTune / 16);
 slong period1  = xm_PeriodTab[nuotti*8 + kfine + 8];
 slong period2;

 if (FineTune < 0)
 {
   kfine--;
   FineTune = -FineTune;
 } else kfine++;

 period2  = xm_PeriodTab[nuotti*8 + kfine + 8];
 kfine    = FineTune & 0x0F;
 period1 *= (16 - kfine);
 period2 *= kfine;

 return ((period1 + period2) * 2) >> oktaavi;
}
*/

inline int xmpt_clip(int x)
{
 if (x < 0) return 0;
 if (x >= 12 * 8 + 8) return (12 * 8 + 8 - 1);
 return x;
}

inline ulong xm_amiga_period(slong Note, slong FineTune)
{
 slong nuotti   = (Note % 12);       // note
 slong oktaavi  = (Note / 12);       // octave
 slong ifine    = (FineTune / 16);   // finetune's integer part
 slong ffine    = (FineTune & 0x0F); // finetune's fractional part
 slong period1  = xm_PeriodTab[xmpt_clip(nuotti*8 + ifine + 8)] * (16 - ffine);
 slong period2  = xm_PeriodTab[xmpt_clip(nuotti*8 + ifine + 8 + 1)] * ffine;

 period1 = ((period1 + period2) * 2) >> oktaavi;

 return period1;
}


inline ulong xm_period(slong Note, ulong FineTune)
{
 if (Note < 0)
   Note = 0; else
 if (Note > 118)
   Note = 118;

 switch (module.linear_freq)
 {
   case 0:
     return xm_amiga_period(Note, FineTune);
     break;
   case 1:
     return xm_lin_period(Note, FineTune);
     break;
   default:
     break;
 }
 return 5000;
}

ulong get_period(slong note, int sample)
{
  switch (module.type)
  {
    case XM:
      if (sample != _nothing)
        return xm_period((slong)note +
                         (slong)samples[sample].relativenote,
                         (slong)samples[sample].finetune);
      break;
    default:
    case S3M:
    case MOD:
      if (note < 0)
        note = 0; else
      if (note > 131)
        note = 131;
      return periods[note];
      break;
  }
  return 69;
}

inline void set_xm_freq(int Achanna, unsigned int Aperiod)
{
  unsigned long long sett;

  switch (module.linear_freq)
  {
    case 0:
      if (Aperiod == 0) return;
      sett = (module.basefreq * 1712L) / Aperiod;
      break;
    case 1:
      sett = ((float)module.basefreq) * pow(2.0f, (6*12*16*4.0f - (float)Aperiod) / (float)(12*16*4.0f));
      break;
    default:
      sett = module.basefreq;
      break;
  }

  _output.set_freq(Achanna, sett);
}

inline int get_pattern_nro(void)
{
  unsigned int pat = module.order[play.order];
  play.physpatta = pat;
  if (pat > module.patterns) return -1;
  return pat;
}

void set_bpm(int bpm)
{
  if (bpm < 1) bpm = 1;
  if (bpm > 256) bpm = 256;
  play.bpm = bpm;
  kello.play_timspeed = (1193181 * 5) / (bpm * 2);
  kello.play_counter %= kello.play_timspeed;
}

inline void actual_setamigafreq(unsigned int Achanna, unsigned int Aperiod, unsigned int c2spd)
{
  unsigned long long sett;

  if (Aperiod == 0) return;

  sett  = 14317056L * (unsigned long long)c2spd;
  sett /= 8363L * (unsigned long long)Aperiod;

  _output.set_freq(Achanna,sett);
}

void setamigafreq(unsigned int Achanna, unsigned int Aperiod, unsigned int c2spd)
{
  switch (module.type)
  {
    default:
    case MOD:
    case S3M:
      actual_setamigafreq(Achanna, Aperiod, c2spd);
      break;
    case XM:
      set_xm_freq(Achanna, Aperiod);
      break;
  }
}

inline void playing_set_volume(int kanava, int volume)
{
  volume = (volume * 64) / module.max_volume;
  volume = (volume * playing_master) / 64;
  if (module.type == XM) volume = (volume * channa[kanava].xm_vol_fade) / 65536;
  if (volume < 0) volume = 0; else
  if (volume > 64) volume = 64;
  _output.set_volume(kanava, volume);
}

void do_portatonote(int c)
{
  channa[c].effect.portanote_period = channa[c].period;

  if (channa[c].effect.portanote_notetoportato > channa[c].effect.portanote_period)
  {
    channa[c].effect.portanote_period +=
      channa[c].effect.portanote_portaspeed;

    if (channa[c].effect.portanote_period >
      channa[c].effect.portanote_notetoportato)
    channa[c].effect.portanote_period =
      channa[c].effect.portanote_notetoportato;

    channa[c].period = channa[c].effect.portanote_period;
  }
  else
  if (channa[c].effect.portanote_notetoportato < channa[c].effect.portanote_period)
  {
    channa[c].effect.portanote_period -=
      channa[c].effect.portanote_portaspeed;

    if (channa[c].effect.portanote_period <
      channa[c].effect.portanote_notetoportato)
    channa[c].effect.portanote_period =
      channa[c].effect.portanote_notetoportato;

    channa[c].period = channa[c].effect.portanote_period;
  }
  else channa[c].whicheffect = _nothing;

  if (channa[c].lastinstru < module.maxsamples)
    setamigafreq(c, channa[c].period, samples[channa[c].lastinstru].c2spd);
}

void do_vslide(int c)
{
  int i;

  if (channa[c].vslide == 1)
  {
    i = (channa[c].volume+channa[c].effect.volumeslide_add);

    if (i < 0) i = 0; else if (i > module.max_volume) i = module.max_volume;

    channa[c].volume = i;
    playing_set_volume(c, i);
  }
}


void do_porta(int c, signed int amount)
{
 channa[c].period += amount;

 if (channa[c].period < module.minporta) channa[c].period = module.minporta; else
 if (channa[c].period > module.maxporta) channa[c].period = module.maxporta;

 if ((channa[c].period == module.minporta) || (channa[c].period == module.maxporta))
   channa[c].whicheffect = _nothing;

 if (channa[c].lastinstru < module.maxsamples)
   setamigafreq(c, channa[c].period, samples[channa[c].lastinstru].c2spd);
}


void do_tremor(int c)
{
 int x = 1 + ((channa[c].effect.tremor_para>>4)&0x0F);
 int y = 1 + (channa[c].effect.tremor_para&0x0F);

 channa[c].effect.tremor_count %= (x + y);

 if (channa[c].effect.tremor_count < x)
   playing_set_volume(c,channa[c].volume);
 else playing_set_volume(c, 0);

 channa[c].effect.tremor_count++;
}

void do_volume_envelopes() // xm
{
 int rc, volume, xd, yd, x1, y1, x2, y2;
 if (module.type != XM) return;

 for (rc = 1; rc <= module.channels; rc++)
 {
   if (channa[rc].keyon == 0)
   {
     channa[rc].vol_env_sustain = 0;
     channa[rc].xm_vol_fade -=
       instruments[channa[rc].lastXMinstrument].fadeout;
     if (channa[rc].xm_vol_fade < 0)
       channa[rc].xm_vol_fade = 0;
   }

   if ((channa[rc].vol_env_on) && (channa[rc].vol_env_sustain == 0))
   {
     // get the current x-point :
     x1 = instruments[channa[rc].lastXMinstrument].vol_env[
            (channa[rc].vol_env_pos)*2];

     // get the target (next) x-point :
     x2 = instruments[channa[rc].lastXMinstrument].vol_env[
            (channa[rc].vol_env_pos+1)*2];

     // calculate x delta :
     xd = x2 - x1;
     if (xd <= 0) xd = 1;

     // get the current y-point :
     y1 = instruments[channa[rc].lastXMinstrument].vol_env[
            (channa[rc].vol_env_pos)*2 + 1];

     // get the target (next) y-point :
     y2 = instruments[channa[rc].lastXMinstrument].vol_env[
            (channa[rc].vol_env_pos+1)*2 + 1];

     // y delta :
     yd = y2 - y1;

     // volume is current y-point plus (vol_env_x / xd * 100) percent
     // of yd
     volume = y1 + (((channa[rc].vol_env_x-channa[rc].vol_env_xs) * yd) / xd);

     volume *= channa[rc].volume;
     volume /= 64;

     playing_set_volume(rc, volume);

     // increase x-position :
     channa[rc].vol_env_x ++;

     // if x-position has reached the target (next) x-point :
     if (channa[rc].vol_env_x >= x2)
     {
       // then make the target point current :
       channa[rc].vol_env_pos++;
       channa[rc].vol_env_xs = channa[rc].vol_env_x; // xstart = x

       // if envelope loop is on and we have reached the loop end
       // then goto loop start :
       if ( (instruments[channa[rc].lastXMinstrument].vol_type & 2) &&
            (channa[rc].vol_env_pos >= instruments[channa[rc].lastXMinstrument].vol_loop_end)
          )
         channa[rc].vol_env_pos =
           instruments[channa[rc].lastXMinstrument].vol_loop_start;

       // if volume sustain is on and we are at the sustain point
       if ( (instruments[channa[rc].lastXMinstrument].vol_type & 4) &&
            (channa[rc].vol_env_pos == instruments[channa[rc].lastXMinstrument].vol_sustain)
          )
         channa[rc].vol_env_sustain = 1;

       // if we are at the end of the envelope then stop processing it :
       if (channa[rc].vol_env_pos >= instruments[channa[rc].lastXMinstrument].vol_pts)
         channa[rc].vol_env_on = 0;
     }
   }
 }
}

void trigger_note(ubyte note, ubyte effe, int rc, int sample) // note, effect, channel, sample
{
  if ( (note != _nothing) &&
       (note != _keyoff) &&
       (sample < module.maxsamples) &&
       (effe != 0x03) &&
       (effe != 0x05)
     )
  {
    _output.playsample(rc, sample, -2, -2, -2, channa[rc].offset);
    channa[rc].keyon = 1;

    if (module.type != XM) return;

    channa[rc].xm_vol_fade = 65536;

    if (instruments[channa[rc].lastXMinstrument].vol_type & 1)
    {
      channa[rc].vol_env_pos     = 0;
      channa[rc].vol_env_x       = 0;
      channa[rc].vol_env_on      = 1;
      channa[rc].vol_env_xs      = 0;
      channa[rc].vol_env_sustain = 0;
    } else channa[rc].vol_env_on = 0;
  }
}

void update_effect()
{
 signed int vibe[32]= // pt vibe
 {
   0,24,49,74,97,120,141,161,180,197,212,224,235,244,250,253,
   255,253,250,244,235,224,212,197,180,161,141,120,97,74,49,24
 };

 int c, temp = 0, d = 0, w = 0;

 for (c = 1; c <= module.channels; c++)
 {
  do_vslide(c); // slide the volume

  switch (channa[c].whicheffect)
  {
    case _nothing:
      break;

    case _globalvslide:
      if (channa[c].effect.global_vslide_para & 0xF0)
        channa[c].effect.global_vslide_para &= 0xF0;

      channa[c].volume += (channa[c].effect.global_vslide_para >> 4);
      channa[c].volume -= (channa[c].effect.global_vslide_para & 0x0F);

      if (channa[c].volume > module.max_volume)
        channa[c].volume = module.max_volume;
      else if (channa[c].volume < 0) channa[c].volume = 0;

      playing_set_volume(c, channa[c].volume);
      break;

    case _panningslide:
      if (channa[c].effect.panning_slide_para & 0xF0)
        channa[c].effect.panning_slide_para &= 0xF0;

      channa[c].pan += (channa[c].effect.panning_slide_para >> 4);
      channa[c].pan -= (channa[c].effect.panning_slide_para & 0x0F);
      if (channa[c].pan > 255) channa[c].pan = 255;
      else if (channa[c].pan < 0) channa[c].pan = 0;

      _output.set_pan(c,channa[c].pan);
      break;

    case _tremor:
      do_tremor(c);
      break;

    case _notecut:
      if (channa[c].effect.notecut_ontick == play.tick)
      {
        playing_set_volume(c, 0);
        channa[c].volume      = 0;
        channa[c].whicheffect = _nothing;
      }
      break;

    case _notedelay:
      if (play.tick == channa[c].effect.newrow.epar)
      {
        channa[c].effect.newrow.effe = 0;
        channa[c].effect.newrow.epar = 0;
        do_row(c, channa[c].effect.newrow);
      }
      break;

    case _retrignote:
      if (channa[c].effect.retrignote_ontick) // 0 -> crash boom bang.
      if ((play.tick % channa[c].effect.retrignote_ontick) == 0)
      {
        switch (channa[c].effect.retrig_para & 0xF0)
        { // pretty.
          case 0x00:break;
          case 0x10:channa[c].volume--;break;
          case 0x20:channa[c].volume-=2;break;
          case 0x30:channa[c].volume-=4;break;
          case 0x40:channa[c].volume-=8;break;
          case 0x50:channa[c].volume-=16;break;
          case 0x60:channa[c].volume=(channa[c].volume*2)/3; break;
          case 0x70:channa[c].volume/=2;break;
          case 0x80:break;
          case 0x90:channa[c].volume++;break;
          case 0xA0:channa[c].volume+=2;break;
          case 0xB0:channa[c].volume+=4;break;
          case 0xC0:channa[c].volume+=8;break;
          case 0xD0:channa[c].volume+=16;break;
          case 0xE0:channa[c].volume=(channa[c].volume*3)/2; break;
          case 0xF0:channa[c].volume*=2;break;
        }

        if (channa[c].volume > module.max_volume)
          channa[c].volume = module.max_volume;
        else if (channa[c].volume < 0) channa[c].volume = 0;

        playing_set_volume(c, channa[c].volume);
        // note, effect, channel, sample
        trigger_note(0, 0, c, channa[c].effect.retrignote_sample);
      }
      break;

    case _portatonote:
      do_portatonote(c);
      break;

    case _arpeggio:
     switch (play.tick%3)
     {
      case 0:
        w = get_period(channa[c].effect.arpeggio_note, channa[c].lastinstru);
        break;
      case 1:
        w = get_period(channa[c].effect.arpeggio_note +
              channa[c].effect.arpeggio_xfine, channa[c].lastinstru);
        break;
      case 2:
        w = get_period(channa[c].effect.arpeggio_note +
              channa[c].effect.arpeggio_yfine, channa[c].lastinstru);
        break;
     }

     if (channa[c].lastinstru < module.maxsamples)
       setamigafreq(c, w, samples[channa[c].lastinstru].c2spd);
     break;

    case _porta:
     do_porta(c,channa[c].effect.porta_add);
     break;

    case _vibrato:
     temp = (channa[c].effect.vibrato_vibepos & 31);
     switch (waveform.vibrato&3) {
      case 0: // sine
        d = vibe[temp];
        break;
      case 1: // ramp down
        temp *= 8;
        if (channa[c].effect.vibrato_vibepos < 0) temp = (255 - temp);
        d = temp;
        break;
      case 2: // square ramp
        d=255;
        break;
      case 3: // "random"
        d = vibe[temp];
        break;
     }

     d = (d * channa[c].effect.vibrato_depth) / 128;
     if (channa[c].lastinstru < module.maxsamples)
      {
       if (channa[c].effect.vibrato_vibepos < 0)
         setamigafreq(c, channa[c].period - d,
           samples[channa[c].lastinstru].c2spd);
       else
         setamigafreq(c, channa[c].period + d,
           samples[channa[c].lastinstru].c2spd);
      }

     channa[c].effect.vibrato_vibepos += channa[c].effect.vibrato_speed;
     if (channa[c].effect.vibrato_vibepos > 31)
       channa[c].effect.vibrato_vibepos -= 64;
     break;

    case _tremolo:
     temp = (channa[c].effect.tremolo_vibepos & 31);
     switch (waveform.tremolo)
     {
      case 0:
        d = vibe[temp];
        break;
      case 1:
        temp *= 8;
        if (channa[c].effect.tremolo_vibepos < 0) temp = (255 - temp);
        d = temp;
        break;
      case 2:
        d = 255;
        break;
      case 3:
        d = vibe[temp];
        break;
     }
     d = (d * channa[c].effect.tremolo_depth) / 64;

     if (channa[c].effect.tremolo_vibepos < 0) d = (channa[c].volume - d);
       else d = channa[c].volume + d;

     playing_set_volume(c, d);

     channa[c].effect.tremolo_vibepos += channa[c].effect.tremolo_speed;
     if (channa[c].effect.tremolo_vibepos > 31)
       channa[c].effect.tremolo_vibepos -= 64;
     break;
   }
 }
 return;
}


void setupvslide(int cha, unsigned char efpara)
{
// make sure efpara is not 0, but that the other of the n(y/i)bbles is.
 if ( (((efpara & 0xF0) == 0) || ((efpara & 0x0F) == 0)) && (efpara != 0) )
  {
   if ((efpara & 0xF0) > 0) channa[cha].effect.volumeslide_add = (efpara >> 4);
    else
   if ((efpara & 0x0F) > 0) channa[cha].effect.volumeslide_add = -(int)(efpara&0x0F);

   channa[cha].vslide = 1; // true
  }
}

void hoida_effyt(int rc, row_type row) /* handle the effects */
{
  int j;
  channa[rc].effy=row.effe;

  switch (row.effe)
  {
   case 0x19: // xm global volume slide
     if (row.epar) channa[rc].effect.global_vslide_para = row.epar;
     channa[rc].whicheffect = _globalvslide;
     break;

   case 0x20: // xm panning slide
     if (row.epar) channa[rc].effect.panning_slide_para = row.epar;
     channa[rc].whicheffect = _panningslide;
     break;

   case 0x11: /*s3m "D", volumeslide.*/
    if (row.epar!=0) channa[rc].effect.s3m_vslide_para=row.epar;
     else row.epar=channa[rc].effect.s3m_vslide_para;

    if ((row.epar&0x0F)==0x0F) channa[rc].volume+=((row.epar>>4)&0xF);
     else
    if ((row.epar&0xF0)==0xF0) channa[rc].volume-=(row.epar&0x0F);
     else
      {
       setupvslide(rc,row.epar);
       if (module.fast_vslides==1) do_vslide(rc);
      }
    break;

   case 0x12: /*s3m "E", portamento down*/
    if (row.epar!=0) channa[rc].effect.s3m_porta_down=row.epar;

    switch (channa[rc].effect.s3m_porta_down&0xF0)
    {
     case 0xF0: /*fine slide*/
      do_porta(rc,4*((int)channa[rc].effect.s3m_porta_down&0x0F));
      break;
     case 0xE0: /*extra fine slide*/
      do_porta(rc,((int)channa[rc].effect.s3m_porta_down&0x0F));
      break;
     case 0x00:
      break;
     default: /* normal slide */
      channa[rc].whicheffect=_porta;
      channa[rc].effect.porta_add=((signed int)row.epar)*4;
      break;
    }
    break;

   case 0x13: /*s3m "F", portamento up*/
    if (row.epar!=0) channa[rc].effect.s3m_porta_down=row.epar;

    switch (channa[rc].effect.s3m_porta_down&0xF0)
    {
     case 0xF0: /* fine slide */
      do_porta(rc,(-4)*((int)channa[rc].effect.s3m_porta_down&0x0F));
      break;
     case 0xE0: /* extra fine slide */
      do_porta(rc,-((int)channa[rc].effect.s3m_porta_down&0x0F));
      break;
     case 0x00:
      break;
     default:/* normal slide */
      channa[rc].whicheffect=_porta;
      channa[rc].effect.porta_add=(-4)*((signed int)row.epar);
      break;
    }
    break;

   case 0x14: /*s3m "I", tremor*/
    if (row.epar!=0) channa[rc].effect.tremor_para=row.epar;
    channa[rc].whicheffect=_tremor;
    do_tremor(rc);
    break;

   case 0x15: /*retrig with volume slide support*/
    if ((row.note!=_nothing) && (row.samp<module.maxsamples))
     {
      if (row.epar>0) channa[rc].effect.retrig_para=row.epar;
       else row.epar=channa[rc].effect.retrig_para;
      channa[rc].whicheffect=_retrignote;
      channa[rc].effect.retrignote_sample=row.samp;
      channa[rc].effect.retrignote_ontick=(row.epar&0x0F);
     }
    break;

   case 0x16: /*fine vibrato*/
    channa[rc].whicheffect=_vibrato;

    if ((row.epar&0xF0)>0)
     channa[rc].effect.vibrato_speed=(row.epar>>4);

    if ((row.epar&0x0F)>0)
     channa[rc].effect.vibrato_depth=(int)(row.epar&0x0F);
    break;

   case 0x17: /*set global volume*/
    playing_mastervolume(row.epar);
    break;

   case 0x18: // delay note [not dealt here]
     break;

   case 0x08: /*set pan*/
    channa[rc].pan=row.epar;
    _output.set_pan(rc,channa[rc].pan);
    break;

   case 0x0C: /*set volume*/
    if (row.epar>module.max_volume) row.epar=module.max_volume;
    channa[rc].volume=row.epar;
    break;

   case 0x0F: /*set speed*/
    play.speed=row.epar;
    break;

   case 0x10: /*set bpm*/
    set_bpm(row.epar);
    break;

   case 0x09: /*set sample offset*/
    if (channa[rc].lastinstru<module.maxsamples)
     {
      if (row.epar>0) channa[rc].lastsampleoffset=row.epar;

      j=((unsigned int)channa[rc].lastsampleoffset)<<8;

      if (j>=samples[channa[rc].lastinstru].length)
       j=samples[channa[rc].lastinstru].length-1;

      channa[rc].offset=j;
     }
    break;

   case 0x0B: /*jump to pattern*/
    play.patternjump=1;
    row.epar++;
    if (row.epar>module.length) row.epar=1;
    play.order=row.epar;
    play.row=0;
    break;

   case 0x0D: /*pattern break*/
    row.epar=(row.epar>>4)*10+(row.epar&0x0F)+1;
    j = get_pattern_nro();
    if (j == -1)
    {
      play.playing = 0;
      break;
    }
    if (row.epar > module.pattern[j].rows) row.epar=1;
    play.row=row.epar-1;
    if ((play.patternjump==0)&&(play.patternbreak==0)) play.order++;
    if (play.order>module.length) play.order=1;
    play.patternbreak=1;
    break;

   case 0x0E: /*.mod extended commands.*/

    switch ((unsigned int)row.epar>>4) {

     case 0x0C: /* note cut / cut note */
      channa[rc].effect.notecut_ontick=row.epar&0x0F;
      if ((row.epar&0x0F)>0) channa[rc].whicheffect=_notecut;
      break;

     case 0x04: /*set vibrato waveform*/
      waveform.vibrato=(row.epar&0x0F)&3;
      if ((row.epar&0x0F)>=4) waveform.retrigvib=0;
       else waveform.retrigvib=1;
      break;

     case 0x07: /*set tremolo waveform*/
      waveform.tremolo=(row.epar&0x0F)&3;
      if ((row.epar&0x0F)>=4) waveform.retrigtre=0;
       else waveform.retrigtre=1;
      break;

     case 0x05: /*set finetune*/
      samples[channa[rc].lastinstru].c2spd=finetunes[row.epar&0x0F];
      break;

     case 0x06: /*pattern loop*/
      if ((row.epar&0x0F)==0) channa[rc].pattern_loop_row = play.row;
      else
      {
        if (channa[rc].pattern_loop_loop == 0)
          channa[rc].pattern_loop_loop = (row.epar&0x0F);
        else channa[rc].pattern_loop_loop--;

        if (channa[rc].pattern_loop_loop > 0)
          play.row = channa[rc].pattern_loop_row-1;
      }
      break;

     case 0x0A: /*fine volume slide up*/
      channa[rc].volume+=(row.epar&0x0F);
      break;

     case 0x0B: /*fine volume slide down*/
      channa[rc].volume-=(row.epar&0x0F);
      break;

     case 0x0E: /*pattern delay*/
      play.patterndelay=row.epar&0x0F;
      break;

     case 0x08: /*16 position pan*/
      channa[rc].pan=(row.epar&0x0F)*16+7;
      _output.set_pan(rc,channa[rc].pan);
      break;

     default:
      break;
    }
    break;


   case 0x03: // porta to note / tone portamento
     channa[rc].whicheffect             = _portatonote;
     channa[rc].effect.portanote_period = channa[rc].period;
     switch (module.type)
     {
       case MOD:
         if (row.epar>0)
           channa[rc].effect.portanote_portaspeed=4*((row.epar>>4)*10+(row.epar&0x0F));
         channa[rc].effect.portanote_notetoportato=get_period(channa[rc].lastnote, 0);
         break;
       case XM:
         if (channa[rc].lastinstru >= module.maxsamples)
         {
           channa[rc].whicheffect = _nothing;
           break;
         }
         if (row.epar)
           channa[rc].effect.portanote_portaspeed=4*((slong)row.epar);

         channa[rc].effect.portanote_notetoportato=
           get_period(channa[rc].lastnote, channa[rc].lastinstru);
         break;
       case S3M:
         channa[rc].effect.portanote_notetoportato=get_period(channa[rc].lastnote, 0);
         if (row.epar>0)
           channa[rc].effect.portanote_portaspeed=4*((slong)row.epar);
         break;
     }
     break;

    /* volume slide,porta+volume slide,vibrato+volume slide*/
   case 0x05:
     setupvslide(rc, row.epar);
     break;
   case 0x06:
     setupvslide(rc, row.epar);
     break; /*doesn't affect sample's triggering*/

   case 0x00: // arpeggio / appregio
    if (channa[rc].lastnote!=_nothing)
    {
      channa[rc].whicheffect=_arpeggio;
      channa[rc].effect.arpeggio_note=channa[rc].lastnote;
      channa[rc].effect.arpeggio_xfine=(row.epar>>4);
      channa[rc].effect.arpeggio_yfine=(row.epar&0x0F);
    }
    break;

   case 0x01: /*porta up*/
    channa[rc].whicheffect=_porta;
    channa[rc].effect.porta_add=-((signed int)row.epar)*4;
    if (module.type == S3M) do_porta(rc,channa[rc].effect.porta_add);
    break;

   case 0x02: /*porta down*/
    channa[rc].whicheffect=_porta;
    channa[rc].effect.porta_add=((signed int)row.epar)*4;
    if (module.type == S3M) do_porta(rc,channa[rc].effect.porta_add);
    break;

   case 0x04: /*vibrato*/
    channa[rc].whicheffect=_vibrato;
    if ((row.epar&0xF0)>0) channa[rc].effect.vibrato_speed=(row.epar>>4);
    if ((row.epar&0x0F)>0) channa[rc].effect.vibrato_depth=(int)(row.epar&0x0F)*4;
    break;

   case 0x07: /*tremolo*/
    channa[rc].whicheffect=_tremolo;
    if ((row.epar&0xF0)>0) channa[rc].effect.tremolo_speed=(row.epar>>4);
    if ((row.epar&0x0F)>0) channa[rc].effect.tremolo_depth=(row.epar&0x0F);
    break;

   default:
    break;
  }
  return;
} /*end of hoida_effyt*/

void hoida_xm_volume_effy(int rc, row_type row)
{
    ubyte volume_column_effect = (ubyte)row.volu >> 4;
    sbyte efparam = row.volu & 0x0F;

    switch (volume_column_effect)
    {
      default:
      case 0x0: // do nothing
        break;
      case 0x1: // set volume
      case 0x2:
      case 0x3:
      case 0x4:
      case 0x5:
        channa[rc].volume = row.volu - 0x10;
        break;
      case 0x6: // volume slide down
        channa[rc].effect.volumeslide_add = -(int)(efparam);
        channa[rc].vslide = 1; // true
        break;
      case 0x7: // volume slide up
        channa[rc].effect.volumeslide_add = (int)(efparam);
        channa[rc].vslide = 1; // true
        break;
      case 0x8: // fine volume slide down
        channa[rc].volume-=efparam;
        break;
      case 0x9: // fine volume slide up
        channa[rc].volume+=efparam;
        break;
      case 0xA: // set vibrato speed
        channa[rc].effect.vibrato_speed = efparam;
        break;
      case 0xB: // vibrato
        channa[rc].whicheffect=_vibrato;
        if (efparam) channa[rc].effect.vibrato_depth=(int)(efparam)*4;
        break;
      case 0xC: // set panning
        channa[rc].pan = 7 + efparam * 16;
        _output.set_pan(rc,channa[rc].pan);
        break;
      case 0xD: // panning slide left
        break;
      case 0xE: // panning slide right
        break;
      case 0xF: // tone porta
        if (channa[rc].lastinstru >= module.maxsamples) break;
        channa[rc].whicheffect=_portatonote;
        if (efparam>0) channa[rc].effect.portanote_portaspeed=4*16*(int)efparam;
        channa[rc].effect.portanote_period=channa[rc].period;
        channa[rc].effect.portanote_notetoportato=
          get_period(channa[rc].lastnote, channa[rc].lastinstru);
        row.note = _nothing; // don't trigger
        break;
    } // end of switch
    return;
} /*end of hoida_xm_volume_effy*/

void do_row(int rc, row_type row)
{
  // set the volume even if there is no sample or note
  // xm volume column is dealt in hoida_xm_volume_effyt()
  if ((module.type!=XM)&&(row.volu!=_nothing)) channa[rc].volume=row.volu;

  // if there is a note and it isn't keyoff
  if ((row.note!=_nothing)&&(row.note!=_keyoff))
  {
    if (waveform.retrigvib==1) channa[rc].effect.vibrato_vibepos=0;
    if (waveform.retrigtre==1) channa[rc].effect.tremolo_vibepos=0;

    channa[rc].lastnote=row.note;
  }
  else if (row.note==_keyoff)
  {
    channa[rc].keyon = 0;
    if (module.type != XM) _output.stopchannel(rc);
  }

  // if there is an instrument number
  if (row.samp < module.maxsamples)
  {
    if ( (module.type == XM) && (row.samp < module.maxinstrus) )
    {
      if ( channa[rc].lastnote < 96 )
        channa[rc].lastinstru     = instruments[row.samp].basesample +
                                    instruments[row.samp].samplenro[
                                    channa[rc].lastnote];
      channa[rc].lastXMinstrument = row.samp;
      channa[rc].pan              = samples[channa[rc].lastinstru].pan;
      if (row.volu == 0)
        channa[rc].volume         = samples[channa[rc].lastinstru].volume;
    }
    else // MOD, S3M
    {
      channa[rc].lastinstru = row.samp;
      if (row.volu == _nothing) channa[rc].volume = samples[channa[rc].lastinstru].volume;
      else channa[rc].volume = row.volu;
    }
  }

  // if there is a note and it isn't keyoff and the effect is not a porta
  // then set period
  if ((row.note!=_nothing)&&(row.note!=_keyoff)&&(row.effe!=0x03)&&(row.effe!=0x05))
    channa[rc].period = get_period(row.note, channa[rc].lastinstru);

  // set note, effy and instru for user to check
  channa[rc].note = row.note;
  channa[rc].effy = _nothing;
  if (row.samp < module.maxsamples) channa[rc].instru = row.samp;

  channa[rc].vslide = 0; // volume slide off

  // if there is a combined effect (vib+vslid,port+vslid,etc)
  // then don't stop the first effect (vib,port)
  if ((row.effe!=0x05)&&(row.effe!=0x06))
    channa[rc].whicheffect=_nothing;

  // if we have a valid sample, then reset its starting offset :
  if (channa[rc].lastinstru < module.maxsamples)
    channa[rc].offset=0;

  // handle the XM volume column effects :
  if ((module.type == XM) && (row.volu != 0)) hoida_xm_volume_effy(rc, row);

  // handle the standard effects :
  if ((row.effe != 0) || (row.epar != 0)) hoida_effyt(rc, row);

  // check volume against limits :
  if (channa[rc].volume < 0)
    channa[rc].volume = 0;
  else if (channa[rc].volume > module.max_volume)
    channa[rc].volume = module.max_volume;

  // suggest the settings calculated to the sound device :
  playing_set_volume(rc, channa[rc].volume);
  _output.set_pan(rc, channa[rc].pan);

  if (channa[rc].lastinstru < module.maxsamples)
    setamigafreq(rc, channa[rc].period, samples[channa[rc].lastinstru].c2spd);

  // if the sample should be triggered, play note :
  trigger_note(row.note, row.effe, rc, channa[rc].lastinstru);
}

void update_row()
{
  int pat,roffs, rc;
  row_type row = {0,0,0,0,0};

  pat = get_pattern_nro();
  if (pat == -1)
  {
    play.playing = 0;
    return;
  }

  roffs = (play.row - 1) * module.channels * 5;

  play.patternjump  = 0;
  play.patternbreak = 0;

  for (rc = 1; rc <= module.channels; rc++)
  {
    roffs += 5;

    row.samp = (ulong)module.pattern[pat].data[roffs+0]; // get sample
    row.note = (ulong)module.pattern[pat].data[roffs+1]; // get note
    row.effe = (ulong)module.pattern[pat].data[roffs+2]; // get effect
    row.epar = (ulong)module.pattern[pat].data[roffs+3]; // get effect parameter
    row.volu = (ulong)module.pattern[pat].data[roffs+4]; // get volume

    if ((row.effe == 0x18) && (row.epar) && (row.epar < play.speed)) // delay note
    {
      channa[rc].whicheffect = _notedelay;
      memcpy(&channa[rc].effect.newrow, &row, sizeof(row_type));
    } else do_row(rc, row); // process row

  }
}

void do_play()
{
 int i;

 if (play.playing == 0) return; // are we still playing?

 play.tick++;
 if (play.tick >= play.speed)
 {
   play.tick=0;
   if (play.patterndelay==0)
   {
     update_row();
     play.row++;
     if (get_pattern_nro() == -1) return;
     if (play.row > module.pattern[get_pattern_nro()].rows)
      {
       play.row=1;
       play.order++;

       if (play.order>module.length) // are we at the end of the module?
       {
         play.playing = 0;           // yes sire, we are.
         play.row     = 0;
         play.order   = 0;
         for (i = 1; i <= module.channels; i++)
         {
           playing_set_volume(i, 0); // mute all channels
           _output.stopchannel(i);
         }
       }

      }
   } else play.patterndelay--;
 } else update_effect();

 if (module.type == XM) do_volume_envelopes();
}

static void modplay()
{
 kello.kelloja++;

 kello.play_counter+=kello.timspeed;
 if (kello.play_counter>kello.play_timspeed)
 {
   kello.play_counter-=kello.play_timspeed;
   do_play(); /* update module */
 }

 kello.counter+=kello.timspeed;
 if (kello.counter>65535) // call the old interrupt 8 handler at ~18.2 Hz
 {
   kello.counter-=65536;
   // push flags on stack because the old interrupt handler will
   // pop 'em. then we'll do a far call (long call) to the old handler
   asm volatile (
   "pushfl
    lcall %0
   "
   :
   :"g" (kello.call_me));
 } else outportb(0x20,0x20);

 asm volatile ("sti");
}

static volatile void end_of_playing(){}

void rewind_playing()
{
 disable();
 do_play();
 enable();
}

void deinit_playing()
{
 disable();
 play.playing = 0;

 // reprogram the pit to its normal frequency :
 outportb(0x43, 0x36);
 outportb(0x40, 0);
 outportb(0x40, 0);

 // restore the old interrupt vector :
 _go32_dpmi_set_protected_mode_interrupt_vector(8, &kello.vanha);
 enable();
}

void init_playing()
{
 int i;
 printf("\rplaying v1.05, MOD/S3M/XM \n");

 // lock some regions used by the tracking code :
 _go32_dpmi_lock_code(begin_of_playing,((char *)end_of_playing-(char *)begin_of_playing));
 _go32_dpmi_lock_data(&kello, sizeof(kello_type));
 _go32_dpmi_lock_data(&play, sizeof(play_type));
 _go32_dpmi_lock_data(&module, sizeof(module_type));

 // initialize play variables :
 play.speed        = module.speed;
 play.bpm          = module.bpm;
 play.tick         = play.speed;
 play.order        = 1;
 play.row          = 1;
 play.patterndelay = 0;
 play.patternbreak = 0;
 play.patternjump  = 0;
 play.playing      = 1;
 mastervolume(64);
 playing_mastervolume(module.init_master);

 // initialize channel structures :
 for (i = 1; i < 33; i++)
 {
   memset(&channa[i].effect, 0, sizeof(modplay_effectrec));
   channa[i].xm_vol_fade       = 0;
   channa[i].keyon             = 0;
   channa[i].vol_env_on        = 0;
   channa[i].vol_env_pos       = 0;
   channa[i].vol_env_x         = 0;
   channa[i].lastXMinstrument  = _nothing;
   channa[i].lastinstru        = _nothing;
   channa[i].lastnote          = _nothing;
   channa[i].whicheffect       = _nothing;
   channa[i].period            = 1712;
   channa[i].vslide            = 0;
   channa[i].pattern_loop_row  = 0;
   channa[i].lastsampleoffset  = 0;
   channa[i].offset            = 0;
   channa[i].pan               = module.initial_pan[i];
   channel[i].freq             = 8363;
 }

 if (module.max_volume == 0) module.max_volume = 64;

 // initialize vibrato and tremolo :
 waveform.retrigvib = 1;
 waveform.retrigtre = 1;
 waveform.vibrato   = 0;
 waveform.tremolo   = 0;

 for (i = 0; i < module.maxsamples; i++)
   samples[i].offset = 0;

 // initialize clock variables :
 kello.counter  = 0;
 kello.timspeed = (1193181L / 200L);

 // error variable
 i=0;

 // save the address of the old interrupt handler :
 i+=_go32_dpmi_get_protected_mode_interrupt_vector(8,&kello.vanha);

 // build a far pointer pointing at the old int 8 handler :
 kello.call_me = (uhuge)kello.vanha.pm_offset+ // first 32 bits are the offset
   ( ((uhuge)kello.vanha.pm_selector) << 32 ); // and the last 16 selector

 // install new int 8 handler :
 kello._info.pm_offset=(unsigned long int)modplay;
 kello._info.pm_selector=_my_cs();

 disable();
 i+=_go32_dpmi_allocate_iret_wrapper(&kello._info);
 i+=_go32_dpmi_set_protected_mode_interrupt_vector(8,&kello._info);
 // i strongly recommend that your extender does not remap ints

 // ok, now the new handler is up.. lets reprogram the PIT so
 // that int 8 shall be called 200 times per second :
 disable();
 outportb(0x43, 0x36);
 outportb(0x40, lo(kello.timspeed));
 outportb(0x40, hi(kello.timspeed));
 set_bpm(play.bpm);

 // and finally enable interrupts thus allowing our int 8 handle to
 // start playing some music.
 enable();
 asm volatile ("sti");

 if (atexit(deinit_playing) != 0)
   strcat(error,"init_playing: auto clean-up not inited.\n");

 if (i != 0)
   strcat(error,"init_playing: error initializing playing.\n");
 fflush(stdout);
}

// end of playing.h

