/**
 ** SVGA24.C ---- the 16M color Super VGA frame driver
 **
 ** Copyright (c) 1995 Csaba Biegl, 820 Stirrup Dr, Nashville, TN 37221
 ** [e-mail: csaba@vuse.vanderbilt.edu] See "doc/copying.cb" for details.
 **/

#include "grdriver.h"
#include "libgrx.h"
#include "arith.h"
#include "mempeek.h"

#define C24BYTE(color,idx)	(((uchar *)(&color))[idx])

static long readpixel(GrFrame *c,int x,int y)
{
	long cval   = 0L;
	long offs   = umul32(y,SCRN->gc_lineoffset) + (x + x + x);
	int  bank   = BANKNUM(offs);
	char far *p = &SCRN->gc_baseaddr[0][BANKPOS(offs)];
	setup_far_selector(SCRN->gc_selector);
	CHKBANK(bank);
	C24BYTE(cval,0) = peek_b_f(p);
	if((ushort)(int)(++p) == 0) { SETBANK(++bank); p = SCRN->gc_baseaddr[0]; }
	C24BYTE(cval,1) = peek_b_f(p);
	if((ushort)(int)(++p) == 0) { SETBANK(++bank); p = SCRN->gc_baseaddr[0]; }
	C24BYTE(cval,2) = peek_b_f(p);
	return(cval);
}

static void drawpixel(int x,int y,long color)
{
	long cmask;
	long offs   = umul32(y,CURC->gc_lineoffset) + (x + x + x);
	int  bank   = BANKNUM(offs);
	char far *p = &CURC->gc_baseaddr[0][BANKPOS(offs)];
	switch(C_OPER(color)) {
	    case C_XOR: cmask = -1L;		    break;
	    case C_OR:  cmask = ~color;		    break;
	    case C_AND: cmask =  color; color = 0L; break;
	    default:	cmask =  0L;		    break;
	}
	setup_far_selector(CURC->gc_selector);
	CHKBANK(bank);
	poke_b_f(p,((peek_b_f(p) & C24BYTE(cmask,0)) ^ C24BYTE(color,0)));
	if((ushort)(int)(++p) == 0) { SETBANK(++bank); p = CURC->gc_baseaddr[0]; }
	poke_b_f(p,((peek_b_f(p) & C24BYTE(cmask,1)) ^ C24BYTE(color,1)));
	if((ushort)(int)(++p) == 0) { SETBANK(++bank); p = CURC->gc_baseaddr[0]; }
	poke_b_f(p,((peek_b_f(p) & C24BYTE(cmask,2)) ^ C24BYTE(color,2)));
}

static void drawhline(int x,int y,int w,long color)
{
	long offs = umul32(y,CURC->gc_lineoffset) + (x + x + x);
	uint c1 = (uint)(((ushort *)(&color))[0]);
	uint c2 = (uint)(((ushort *)(&color))[1]);
	int  op = C_OPER(color);
	setup_far_selector(CURC->gc_selector);
	do {
	    uint      fillc = BANKLFT(offs) / 3;
	    char far *fillp;
	    if(fillc == 0) {
		/* let 'drawpixel' handle split pixels! */
		drawpixel(x,y,color);
		x++; w--;
		continue;
	    }
	    fillp = &CURC->gc_baseaddr[0][BANKPOS(offs)];
	    CHKBANK(BANKNUM(offs));
	    fillc = umin(w,fillc);
	    offs += (fillc + fillc + fillc);
	    x	 += fillc;
	    w	 -= fillc;
	    switch(op) {
	      case C_XOR:
		do {
		    poke_w_f_xor(fillp,c1); fillp += 2;
		    poke_b_f_xor(fillp,c2); fillp += 1;
		} while(--fillc != 0);
		break;
	      case C_OR:
		do {
		    poke_w_f_or(fillp,c1); fillp += 2;
		    poke_b_f_or(fillp,c2); fillp += 1;
		} while(--fillc != 0);
		break;
	      case C_AND:
		do {
		    poke_w_f_and(fillp,c1); fillp += 2;
		    poke_b_f_and(fillp,c2); fillp += 1;
		} while(--fillc != 0);
		break;
	      default:
		do {
		    poke_w_f(fillp,c1); fillp += 2;
		    poke_b_f(fillp,c2); fillp += 1;
		} while(--fillc != 0);
		break;
	    }
	} while(w != 0);
}

static
#include "fdrivers/generic/vline.c"

static
#include "fdrivers/generic/block.c"

static
#include "fdrivers/generic/line.c"

static
#include "fdrivers/generic/bitmap.c"

static
#include "fdrivers/generic/pattern.c"

static void bitblt(GrFrame *dst,int dx,int dy,GrFrame *src,int sx,int sy,int w,int h,long op)
{
	if(GrColorMode(op) == GrIMAGE) _GrFrDrvGenericBitBlt(
	    dst,dx,dy,
	    src,sx,sy,
	    w,h,
	    op
	);
	else _GrFrDrvPackedBitBltV2V(
	    dst,(dx * 3),dy,
	    src,(sx * 3),sy,
	    (w * 3),h,
	    op
	);
}

static void bltv2r(GrFrame *dst,int dx,int dy,GrFrame *src,int sx,int sy,int w,int h,long op)
{
	if(GrColorMode(op) == GrIMAGE) _GrFrDrvGenericBitBlt(
	    dst,dx,dy,
	    src,sx,sy,
	    w,h,
	    op
	);
	else _GrFrDrvPackedBitBltV2R(
	    dst,(dx * 3),dy,
	    src,(sx * 3),sy,
	    (w * 3),h,
	    op
	);
}

static void bltr2v(GrFrame *dst,int dx,int dy,GrFrame *src,int sx,int sy,int w,int h,long op)
{
	if(GrColorMode(op) == GrIMAGE) _GrFrDrvGenericBitBlt(
	    dst,dx,dy,
	    src,sx,sy,
	    w,h,
	    op
	);
	else _GrFrDrvPackedBitBltR2V(
	    dst,(dx * 3),dy,
	    src,(sx * 3),sy,
	    (w * 3),h,
	    op
	);
}

GrFrameDriver _GrFrameDriverSVGA24 = {
    GR_frameSVGA24,		/* frame mode */
    GR_frameRAM24,		/* compatible RAM frame mode */
    TRUE,			/* onscreen */
    4,				/* line width alignment */
    1,				/* number of planes */
    24,				/* bits per pixel */
    24*16*1024L*1024L,		/* max plane size the code can handle */
    NULL,
    readpixel,
    drawpixel,
    drawline,
    drawhline,
    drawvline,
    drawblock,
    drawbitmap,
    drawpattern,
    bitblt,
    bltv2r,
    bltr2v
};

