// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

// Merge to rectangular parts of a bitmap into a third.

#include "misc/template.h"
#include "gfx/bitmap.h"
#include "gfx/merge.h"

static int FactorTable[18] = { 0,1,2,3,4,5,6,7,8,16,15,14,13,12,11,10,9,8 };

// Precompute values for accelerated PackedPixel8 merging.
void MergeTablePP8::Construct(Palette &Pal)
  {
  #ifdef DEBUG
  if (Pal.ClassID()!=ClassColorArray && Pal.Colors()!=256) ErrorExit("MergeTablePP8::Construct()  need a 256 color palette.");
  #endif
  palette=&Pal;
  const int Colors=Pal.Colors();
  // create tables
  palindex_to_rgb4_4_mul=new DWORD[Colors*18]; // 18 tables
  grb4_to_palindex=new UBYTE[1<<12];
  // compute the array palette index -> rgb4_4
  int Mul1TableStart=Colors; 
  for (int PalIndex=0;PalIndex<Colors;PalIndex++)
    {
    ColorRef RGB=Pal.GetColor(PalIndex);
    palindex_to_rgb4_4_mul[Mul1TableStart+PalIndex]=((RGB8Red(RGB)&0xf0)<<12) | ((RGB8Green(RGB)&0xf0)<<4) | ((RGB8Blue(RGB)&0xf0)>>4);
    }
  // now 16 times the same table, but multiplied with the factors 0 - 16
  for (int i=0;i<Colors;i++) palindex_to_rgb4_4_mul[i]=0;
  for (int Table=2;Table<18;Table++)
    {
    int Factor=FactorTable[Table];
    int MulTableStart=Colors*Table;
    for (PalIndex=0;PalIndex<Colors;PalIndex++)
      palindex_to_rgb4_4_mul[MulTableStart+PalIndex]=palindex_to_rgb4_4_mul[Mul1TableStart+PalIndex]*Factor;
    }
  // one table for grb4 -> palette index
  int GRB4=0; // using green red blue order is faster than rgb
  for (int g=0;g<16;g++)
    for (int r=0;r<16;r++)
      for (int b=0;b<16;b++)
        {
        grb4_to_palindex[GRB4]=(UBYTE)Pal.GetPaletteIndex(RGB8((UBYTE)(r<<4),(UBYTE)(g<<4),(UBYTE)(b<<4)));
        GRB4++;
        }
  };

#ifdef I_HAVE_GOT_LOTS_OF_TIME
// This function works for all types of bitmaps, but is extremely slow.
// Args are self-explaining, except Gradient:
// Gradient=0        - show only bitmap Src1
//         =256      - show only bitmap Src2
//         >0 & <256 - merge the differently weighted bitmaps together 
MergeRect(Bitmap &Dst,int dxs,int dy,Bitmap &Src1,int x1s,int y1,Bitmap &Src2,const Rect &SrcRect2,int Gradient)
  {
  #ifdef DEBUG
  // check if rects are inside the bitmap.

  #endif
  const int Gradient2=256-Gradient;
  const int RectWidth=SrcRect2.Width();
  const int RectHeight=SrcRect2.Height();
  int LinesToGo=RectHeight;
  int y2=SrcRect2.Top;
  while (LinesToGo--)
    {
    int PixelsToGo=RectWidth;
    int dx=dxs;
    int x1=x1s;
    int x2=SrcRect2.Left;
    while (PixelsToGo--)
      {
      ColorRef c1=Src1.GetPixelRGB(x1,y1);
      ColorRef c2=Src2.GetPixelRGB(x2,y2);
      ColorRef c=RGB8(
       (RGB8Red(c1)*Gradient+RGB8Red(c2)*Gradient2)>>8,
       (RGB8Green(c1)*Gradient+RGB8Green(c2)*Gradient2)>>8,
       (RGB8Blue(c1)*Gradient+RGB8Blue(c2)*Gradient2)>>8);
      Dst.SetPixelRGB(dx,dy,c);
      dx++;
      x1++;
      x2++;
      }
    dy++;
    y1++;
    y2++;
    }
  }
#endif

// Same as above but accelerated for PackedPixel8 bitmaps with the same
// palette and a precompute merging table.
void MergeRect(Bitmap &Dst,int dxs,int dy,Bitmap &Src1,int x1s,int y1,Bitmap &Src2,const Rect &SrcRect2,int Gradient,MergeTablePP8 &PaletteInfo)
  {
  #ifdef DEBUG
  // same palettes
  if (Dst.Palette()!=Src1.Palette() || Src1.Palette!=Src2.Palette()) ErrorExit("MergeRect() {PackedPixel8}  Bitmaps must use the same palette.");
  if (Dst.ClassID()!=PackedPixel8) ErrorExit("MergeRect() {PackedPixel8}  Need PackedPixel8 bitmaps.");
  if (Dst.Palette()!=PaletteInfo.palette) ErrorExit("MergeRect() {PackedPixel8}  MergeTable is not computed for this palette.");
  #endif
  Gradient>>=4; // we use only 16 merge states, not 256.
  const int RectWidth=SrcRect2.Width();
  const int RectHeight=SrcRect2.Height();
  UBYTE *SrcPtr1=Src1.Bits()+y1*Src1.WidthBytes()+x1s;
  UBYTE *SrcPtr2=Src2.Bits()+SrcRect2.Top*Src2.WidthBytes()+SrcRect2.Left;
  UBYTE *DstPtr = Dst.Bits()+dy* Dst.WidthBytes()+dxs;
  int SrcDiff1=Src1.Width()-RectWidth;
  int SrcDiff2=Src2.Width()-RectWidth;
  int  DstDiff= Dst.Width()-RectWidth;
  if (Gradient>8)  // swap bitmaps (bitmap1 gradient is always smaller)
    {
    Swap(SrcDiff1,SrcDiff2);
    Swap(SrcPtr1,SrcPtr2);
    Gradient=16-Gradient;
    }
  DWORD *MulTable=PaletteInfo.palindex_to_rgb4_4_mul+Gradient*256;
  int LinesToGo=RectHeight;
  while (LinesToGo--)
    {
    int PixelsToGo=RectWidth;
    while (PixelsToGo--)
      {
      DWORD c1,c2,c,t;
      c1=*SrcPtr1;
      c2=*SrcPtr2;
      c1=MulTable[c1];
      c2=MulTable[c2+9*256];
      c=(c1+c2)>>4;
      c&=0xf0f0f;
      t=(c&0xffff)|(c>>12);
      *DstPtr=PaletteInfo.grb4_to_palindex[t];
      SrcPtr1++;
      SrcPtr2++;
      DstPtr++;
      }
    SrcPtr1+=SrcDiff1;
    SrcPtr2+=SrcDiff2;
    DstPtr +=DstDiff;
    }
  }

