/*  Copyright (C) 1993   Marc Stern  (internet: stern@mble.philips.be)  */

#include "strings.h"
#include "tools.h"
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>




/***
 *
 *  Function   :   getparam
 *
 *  Topics     :   Read one paramater/topic in setup file.
 *
 *  Parameters :    in    FILE *file           input file
 *                  in    const char *keyword  keyword to match
 *                  out   char *value          value associated to keyword
 *                  in    topicPos             topic position in file
 *
 *  Values     :    topicPos   -1        whole file
 *                              0        from beginning of file to first topic
 *                              other    line after topic
 *
 *  Return code:     0     OK
 *                  -1     keyword not found
 *                  -2     file problem
 *
 *  Precond    :   Input file must be opened with read access.
 *
 */

static int near getparam( FILE *file, const char *keyword, char *value, long topicPos )
{
 char buffer[256];
 int length, loop;
 long inipos;

 length = strlen( keyword );

 inipos = ftell( file );
 for ( loop = 0; loop < 2; loop++ )
    {
     while ( fgets(buffer, 255, file) )
        {
         if ( *buffer == '#' ) continue;

         strreduce( buffer );

         if ( *keyword == '[' )
            {
             char *ptr = strstr( buffer, keyword );
             if ( ptr && *(ptr + length) == ']'  )
                return 0;
             continue;
            }

         if ( ! strncmp(buffer, keyword, length) )
            switch( buffer[length] )
            {
             case '=':
                 strcpy( value, buffer + length + 1);
	         return 0;

             case '\0':
                 *value = '\0';
	         return 0;
            }

         if ( *buffer == '[' && (topicPos >= 0) ) break;  /* new topic */

         if ( loop && (ftell(file) >= inipos) ) return -1;

	}  /*  while ( fgets )  */

     if ( fseek(file, max(topicPos, 0), SEEK_SET) ) return -2;    

    }  /* for ( loop )  */

 return -1;
}



/***
 *
 *  Function   :   getsetup
 *
 *  Topics     :   Read paramaters in setup file (like Windows init files).
 *
 *  Decisions  :   Expected file format is a list of lines of the form
 *                                         
 *                   [topic1] [topic2] ... [topicN]  (optional)
 *                    or
 *		     <keyword>=<value>
 *
 *                 - If a [topic] is given, keyword is searched only between
 *                   that [topic] line and the next one.
 *
 *                 - several [topic] names may be given on the same line
 *                   to be aliases for the same section.
 *
 *                 - All letters are treated (and returned) as uppercases.
 *                 - Blanks and tabs are skipped.
 *
 *                 - <value> can be enclosed between double quotes
 *		     to preserve lowercases and blanks.
 *
 * 		   - 2 consecutive double quotes allow to keep
 *                   a double quote in <value>.
 *
 *                 - A number sign (#) in first column forces the line
 *                   to be ignored.
 *
 *                 - The input formats in parameter 'format' are C standard
 *                   ones ( %d, %s, %f,...).
 *                   Values to be stored are C standard
 *                   ones ( &myInt, myString, &myFloat,...).
 *
 *                 - Special format added: "%b" will translate
 *                   True/False, Yes/No, On/Off, 1/0 into integer 1/0
 *
 *                 - Special format added: "%S" will result into copying
 *                   all the input line (with spaces in it if any).
 *                   It will not be processed by scanf-like processing.
 *
 *  Parameters :    in    FILE *file        input file
 *                  in    char *format      "[topic1] keyword1=format keyword2=format [topic2] ..."
 *                  out   pointer list      values to be stored
 *
 *
 *  Optimization:   Keywords (or topics) are searched from the current
 *                  position of the file; if not found, search is
 *                  redone from the [topic] line (or the begin of
 *                  the file if no [topic] asked).
 *                  That means that it is very more efficient
 *                  (less disk access) to sort parameters
 *                  in 'format' in the same order as encountered
 *                  in the file.
 *
 *  Return code:    number of found parameters
 *                  -1     syntax error
 *                  -2     file error
 *                  -3     not enough memory
 *
 *  Precond    :   Input file must be opened with read access.
 *
 */

int getsetup( FILE *file, const char *format, ... )
{
 va_list list;
 int status, count = 0;
 char par_value[255], *topic = NULL, *fmt, *keyword, *ptr;
 void *param;
 long topicPos = -1;

 va_start( list, format );
 fmt = strdup( format );
 if ( ! fmt ) return -3;

 for ( keyword = strtok(fmt, "="); keyword; keyword = strtok(NULL, "=") )
    {
     param = va_arg( list, void* );

     strreduce( keyword );
     if ( *keyword == '[' )    /* search for a [topic] line */
        {
         char *ptr = strchr( keyword, ']' );
         if ( ptr )
            {
             topic = keyword;
             *ptr++ = '\0';
             keyword = ptr;

             /* search topic entry */
             status = getparam( file, topic, NULL, -1 );
             switch( status )
             {
              case  0:
                 topicPos = ftell( file );
                 if ( topicPos < 0 )
                    {
                     free( fmt );
                     return -2;
                    }
                 break;

              case -1:
                 if ( *keyword != '*' )  /* first section if topic not found */
                    topic = NULL;
                 else
                    {
                     topicPos = 0;
                     if ( fseek(file, 0, SEEK_SET) )
                        {
                         free( fmt );
                         return -2;
                        }
                    }
                 break;

              default:
                 free( fmt );
                 return status;

             }  /*  switch( status )  */

             if ( *keyword == '*' )  /* first section if topic not found */
                keyword++;

            }  /*  if ( ptr )  */

        } /*  if ( *keyword == '[' ) */

     if ( topicPos >= 0 && ! topic ) continue;

     status = getparam( file, keyword, par_value, topicPos );
     format = strtok( NULL, " \t\r\n" );
     switch( status )
     {
      case  0: break;

      case -1: continue;

      default: free( fmt );
               return status;
     }

     count++;

     /* handle boolean (%b) format */
     ptr = strstr( format, "%b" );
     if ( ptr )
        {
         if ( ! *par_value ||
              strisequal(par_value, "TRUE") ||
              strisequal(par_value, "YES") ||
              strisequal(par_value, "ON") ||
              !strncmp(par_value, "ENABLE", 6) ||
              strisequal(par_value, "1")
            ) *(int*)param = 1;
         else *(int*)param = 0;

         continue;
        }  /*  if ( ptr )  */


     /* Handle %S to get spaces inside strings */
     ptr = strstr( format, "%S" );
     if ( ptr )
        {
         strcpy( ((char*)param), par_value );
         continue;
        }

     if ( sscanf(par_value, format, param) != 1 )
        {
         free( fmt );
         return -1;
        }

    }   /*  for (;;)  */

 va_end( list );

 free( fmt );
 return count;
}
 

#ifdef MAIN

void main( void )
{
 FILE *inFile;
 char buffer[10][50];
 int flag;

 inFile = fopen( "setup.dat", "rt" );
 if ( ! inFile ) return;

 memset( buffer, '\0', sizeof(buffer) );
 fseek( inFile, 200, SEEK_SET );
 getsetup( inFile,
            "flag=%b "
            "[notexist]*TYPE=%S TEST=%S "
            "[topic3] OBJ_NAME=%S ORG_DATE=%S "
            "[topic1] OBJ_NAME=%S ORG_DATE=%S ",
            &flag,
            buffer[0], buffer[1],
            buffer[2], buffer[3],
            buffer[4], buffer[5]
          );

 fclose( inFile );
}     

#endif                    
