/* LP4 Support Functions
 * Copyright (C) 1994 Tale Software
 * All rights reserved.
 */

#include <alloc.h>    /* for malloc */
#include <fcntl.h>    /* access modes for open */
#include <io.h>	      /* for I/O */
#include <string.h>   /* for LP4 header identification */

#include "lp4const.h" /* LP4 constants */
#include "tgdconst.h" /* common tgdlp4 data types and structures */

/* define errors */
#define LP4ERR_NOFILE 1
#define LP4ERR_BADFILE 2
#define LP4ERR_NOMEM 3
#define LP4ERR_EOF 4

/* module identificator:
 * it is inmoral to change or remove the following line
 */
const char lp4suppID[]="\r\nTale LP4 Support Library V1.5b\r\n";

/* LP4 bitmap descriptor structure */
typedef struct { cmhdr huge *cmdescr;
		 ubyte huge *cmap;
		 bmhdr huge *bmdescr;
		 ubyte huge *bmap;
		 bitmap huge *tgdbmdescr;
	       } bcdescr;

/* LP4 font descriptor structure */
typedef struct { cmhdr huge *cmdescr;
		 ubyte huge *cmap;
		 fmhdr huge *fmdescr;
		 ubyte huge *bmap;
		 fontmap huge *tgdfmdescr;
	       } fcdescr;

/* LP4 sprite descriptor structure */
typedef struct { cmhdr huge *cmdescr;
		 ubyte huge *cmap;
		 smhdr huge *smdescr;
		 ubyte huge *bmap;
		 spritemap huge *tgdsmdescr;
	       } scdescr;

/*
 * procedures for internal use
 */

ubyte checklp4(const char far *filename,int *readhandle)
{ char fileID[6];
  lp4hdr lp4ver_rev;

  /* open LP4 file for reading */
  *readhandle=open(filename,O_RDONLY|O_BINARY);
  if (*readhandle==-1) return(LP4ERR_NOFILE);

  /* is it really a LP4 file? */
  if (_read(*readhandle,&fileID,6)!=6) return(LP4ERR_BADFILE);
  if (strncmp(fileID,lp4ID,6)) return(LP4ERR_BADFILE);

  /* read version, revision */
  _read(*readhandle,&lp4ver_rev,2);
  if (lp4ver_rev.version!=LP4_VERSION||lp4ver_rev.revision!=LP4_REVISION)
     return(LP4ERR_BADFILE);

  return(0);
}

byte findlp4chunk(const char far *chunkID,ubyte chunkno,int readhandle)
{ ubyte filecnkno=0;
  byte chunkfnd=FALSE;
  lp4cnkhdr searchcnk;

  while (_read(readhandle,&searchcnk,6))
  { /* specified chunk ID found?... */
    if (strncmp(searchcnk.cnkname,chunkID,2))
       { lseek(readhandle,searchcnk.cnklen,SEEK_CUR); continue; };
    /* ...yes! */
    filecnkno+=1;

    /* correct chunk number found? */
    if (filecnkno==chunkno) { chunkfnd=TRUE; break; };

    lseek(readhandle,searchcnk.cnklen,SEEK_CUR);
  };
  return(chunkfnd);
}

byte getcolchunks(lp4cnkhdr far *colcnkhdr,cmhdr huge **colmdescr,ubyte huge **colmap,int readhandle,ubyte far *loaderror)
{
  /* color header found? */
  if (!strncmp(colcnkhdr->cnkname,chID,2)&&!*colmdescr)
  { *colmdescr=farmalloc(colcnkhdr->cnklen);
    if (!*colmdescr) { *loaderror=LP4ERR_NOMEM; return(FALSE); };
    _read(readhandle,*colmdescr,(uword)colcnkhdr->cnklen);
    return(TRUE); /* color header WAS found */
  };

  /* color map found? */
  if (!strncmp(colcnkhdr->cnkname,cmID,2)&&!*colmap)
  { *colmap=farmalloc(colcnkhdr->cnklen);
    if (!*colmap) { *loaderror=LP4ERR_NOMEM; return(FALSE); };
    _read(readhandle,*colmap,(uword)colcnkhdr->cnklen);
    return(TRUE); /* color map WAS found */
  };

  return(FALSE); /* no color chunks were found */
}

void bmhdr2bm(ubyte huge *rawbm,bmhdr huge *srcbh,bitmap huge *destbm)
{ ubyte planecnt;

  destbm->xsize=srcbh->xsize;
  destbm->ysize=srcbh->ysize;
  destbm->bytespline=srcbh->bytespline;
  for (planecnt=0;planecnt<4;planecnt+=1)
  { destbm->plane[planecnt]=rawbm+srcbh->planeoffset[planecnt]; };
}

/*
 * procedures for public use
 */

void freebc(bcdescr huge *bcinfo)
{
  if (!bcinfo) return;
  if (bcinfo->cmdescr) farfree(bcinfo->cmdescr);
  if (bcinfo->cmap) farfree(bcinfo->cmap);
  if (bcinfo->bmdescr) farfree(bcinfo->bmdescr);
  if (bcinfo->bmap) farfree(bcinfo->bmap);
  if (bcinfo->tgdbmdescr) farfree(bcinfo->tgdbmdescr);
  farfree(bcinfo);
}

bcdescr huge *readbc(const char far *filename,ubyte bcnumber,ubyte far *loaderr)
{ ubyte huge *memlocptr;
  int rdhandle;
  udword memadd=0;
  bcdescr huge *bcinfo=0;

  lp4cnkhdr bccnkhdr;

  /* open LP4 file and check LP4 header */
  *loaderr=checklp4(filename,&rdhandle);
  if (*loaderr) goto bcreadend;

  /* allocate memory for the bcdescr structure */
  bcinfo=farmalloc(sizeof(bcdescr));
  if (!bcinfo) { *loaderr=LP4ERR_NOMEM; goto bcreadend; };

  bcinfo->cmdescr=0; bcinfo->cmap=0;
  bcinfo->bmdescr=0; bcinfo->bmap=0;
  bcinfo->tgdbmdescr=0;

  /* find desired bitmap chunk */
  if (!findlp4chunk(bcID,bcnumber,rdhandle)) { *loaderr=LP4ERR_EOF; goto bcreadend; };

  /* read chunk header */
  while (_read(rdhandle,&bccnkhdr,6))
  { /* color chunks found? */
    if (getcolchunks(&bccnkhdr,&(bcinfo->cmdescr),&(bcinfo->cmap),rdhandle,loaderr)) continue;
    if (*loaderr) break;

    /* bitmap header found? */
    if (!strncmp(bccnkhdr.cnkname,bhID,2)&&!(bcinfo->bmdescr))
    { bcinfo->bmdescr=farmalloc(bccnkhdr.cnklen);
      if (!(bcinfo->bmdescr)) { *loaderr=LP4ERR_NOMEM; break; };
      _read(rdhandle,bcinfo->bmdescr,(uword)bccnkhdr.cnklen);
      continue;
    };

    /* bit map found? */
    if (!strncmp(bccnkhdr.cnkname,bmID,2)&&!(bcinfo->bmap)&&(bcinfo->bmdescr))
    { memlocptr=farmalloc(bcinfo->bmdescr->maxbitmaplen);
      if (!memlocptr) { *loaderr=LP4ERR_NOMEM; break; };
      bcinfo->bmap=memlocptr;
      /* reading the bitmap is somewhat complexer since we transfer
       * it in 32K blocks to memory.
       */
      for (;bccnkhdr.cnklen>32767;memadd+=32767,bccnkhdr.cnklen-=32767)
      { _read(rdhandle,memlocptr+memadd,32767); };
      _read(rdhandle,memlocptr+memadd,(uword)bccnkhdr.cnklen);
      continue;
    };

    /* overread unknown or double or misaligned chunk */
    lseek(rdhandle,bccnkhdr.cnklen,SEEK_CUR);
  };

  /* any error occured while reading chunks? */
  if (*loaderr) goto bcreadend;

  /* create tgdlp4 bitmap structure */
  if (!bcinfo->bmap) goto bcreadend;
  bcinfo->tgdbmdescr=farmalloc(sizeof(bitmap));
  if (!bcinfo->tgdbmdescr) { *loaderr=LP4ERR_NOMEM; goto bcreadend; };
  /* create tgdlp4 bitmap structure out of bitmap header */
  bmhdr2bm(bcinfo->bmap,bcinfo->bmdescr,bcinfo->tgdbmdescr);

bcreadend:
  if (rdhandle!=-1) close(rdhandle);
  if (*loaderr && bcinfo) { freebc(bcinfo); return(0); };
  return(bcinfo);
}

void freefc(fcdescr huge *fcinfo)
{
  if (!fcinfo) return;
  if (fcinfo->cmdescr) farfree(fcinfo->cmdescr);
  if (fcinfo->cmap) farfree(fcinfo->cmap);
  if (fcinfo->fmdescr) farfree(fcinfo->fmdescr);
  if (fcinfo->bmap) farfree(fcinfo->bmap);
  if (fcinfo->tgdfmdescr) farfree(fcinfo->tgdfmdescr);
  farfree(fcinfo);
}

fcdescr huge *readfc(const char far *filename,ubyte fcnumber,ubyte far *loaderr)
{ ubyte huge *memlocptr,huge *fntpsrcptr,huge *fntpdstptr;
  int rdhandle;
  udword memadd=0;
  fcdescr huge *fcinfo=0;

  lp4cnkhdr fccnkhdr;
  fmhdr huge *srcfmhdr;
  fontmap huge *destfm;

  /* open LP4 file and check LP4 header */
  *loaderr=checklp4(filename,&rdhandle);
  if (*loaderr) goto fcreadend;

  /* allocate memory for the fcdescr structure */
  fcinfo=farmalloc(sizeof(fcdescr));
  if (!fcinfo) { *loaderr=LP4ERR_NOMEM; goto fcreadend; };

  fcinfo->cmdescr=0; fcinfo->cmap=0;
  fcinfo->fmdescr=0; fcinfo->bmap=0;
  fcinfo->tgdfmdescr=0;

  /* find desired fontmap chunk */
  if (!findlp4chunk(fcID,fcnumber,rdhandle)) { *loaderr=LP4ERR_EOF; goto fcreadend; };

  /* read chunk header */
  while (_read(rdhandle,&fccnkhdr,6))
  { /* color chunks found? */
    if (getcolchunks(&fccnkhdr,&(fcinfo->cmdescr),&(fcinfo->cmap),rdhandle,loaderr)) continue;
    if (*loaderr) break;

    /* fontmap header found? */
    if (!strncmp(fccnkhdr.cnkname,fhID,2)&&!(fcinfo->fmdescr))
    { fcinfo->fmdescr=farmalloc(fccnkhdr.cnklen);
      if (!(fcinfo->fmdescr)) { *loaderr=LP4ERR_NOMEM; break; };
      _read(rdhandle,fcinfo->fmdescr,(uword)fccnkhdr.cnklen);
      continue;
    };

    /* bit map found? */
    if (!strncmp(fccnkhdr.cnkname,bmID,2)&&!(fcinfo->bmap)&&(fcinfo->fmdescr))
    { memlocptr=farmalloc(fcinfo->fmdescr->fontbitmap.maxbitmaplen);
      if (!memlocptr) { *loaderr=LP4ERR_NOMEM; break; };
      fcinfo->bmap=memlocptr;
      /* reading the bitmap is somewhat complexer since we transfer
       * it in 32K blocks to memory.
       */
      for (;fccnkhdr.cnklen>32767;memadd+=32767,fccnkhdr.cnklen-=32767)
      { _read(rdhandle,memlocptr+memadd,32767); };
      _read(rdhandle,memlocptr+memadd,(uword)fccnkhdr.cnklen);
      continue;
    };

    /* overread unknown or double or misaligned chunk */
    lseek(rdhandle,fccnkhdr.cnklen,SEEK_CUR);
  };

  /* any error occured while reading chunks? */
  if (*loaderr) goto fcreadend;

  /* create tgdlp4 fontmap structure */
  if (!fcinfo->bmap) goto fcreadend;
  /* note: fontmap structure of proportional font is different from
   * fixed font fontmap structure.
   */
  if (fcinfo->fmdescr->fonttype==LP4_FONT_PROP)
     fcinfo->tgdfmdescr=farmalloc(sizeof(fontmap)+(((fcinfo->fmdescr->stopchr-fcinfo->fmdescr->startchr)+1)<<1));
  else fcinfo->tgdfmdescr=farmalloc(sizeof(fontmap));
  if (!fcinfo->tgdfmdescr) { *loaderr=LP4ERR_NOMEM; goto fcreadend; };

  /* init fontmap structure */
  srcfmhdr=fcinfo->fmdescr; destfm=fcinfo->tgdfmdescr;
  destfm->fonttype=srcfmhdr->fonttype;
  destfm->startchr=srcfmhdr->startchr;
  destfm->stopchr=srcfmhdr->stopchr;
  destfm->chrxsize=srcfmhdr->chrxsize;
  destfm->chrysize=srcfmhdr->chrysize;
  destfm->spacingx=srcfmhdr->spacingx;
  destfm->spacingy=srcfmhdr->spacingy;

  /* create tgdlp4 bitmap structure out of fontmap header */
  bmhdr2bm(fcinfo->bmap,&(srcfmhdr->fontbitmap),&(destfm->fontbitmap));

  /* copy proportional font information (if required) */
  if (destfm->fonttype==LP4_FONT_PROP)
  { fntpsrcptr=(void *)srcfmhdr;
    fntpdstptr=(void *)destfm;
    memcpy(fntpdstptr+sizeof(fontmap),fntpsrcptr+sizeof(fmhdr),((destfm->stopchr-destfm->startchr)+1)<<1);
  };

fcreadend:
  if (rdhandle!=-1) close(rdhandle);
  if (*loaderr && fcinfo) { freefc(fcinfo); return(0); };
  return(fcinfo);
}

void freesc(scdescr huge *scinfo)
{
  if (!scinfo) return;
  if (scinfo->cmdescr) farfree(scinfo->cmdescr);
  if (scinfo->cmap) farfree(scinfo->cmap);
  if (scinfo->smdescr) farfree(scinfo->smdescr);
  if (scinfo->bmap) farfree(scinfo->bmap);
  if (scinfo->tgdsmdescr) farfree(scinfo->tgdsmdescr);
  farfree(scinfo);
}

scdescr huge *readsc(const char far *filename,ubyte scnumber,ubyte far *loaderr)
{ ubyte huge *memlocptr;
  int rdhandle;
  udword memadd=0;
  scdescr huge *scinfo=0;

  lp4cnkhdr sccnkhdr;

  /* open LP4 file and check LP4 header */
  *loaderr=checklp4(filename,&rdhandle);
  if (*loaderr) goto screadend;

  /* allocate memory for the scdescr structure */
  scinfo=farmalloc(sizeof(scdescr));
  if (!scinfo) { *loaderr=LP4ERR_NOMEM; goto screadend; };

  scinfo->cmdescr=0; scinfo->cmap=0;
  scinfo->smdescr=0; scinfo->bmap=0;
  scinfo->tgdsmdescr=0;

  /* find desired spritemap chunk */
  if (!findlp4chunk(scID,scnumber,rdhandle)) { *loaderr=LP4ERR_EOF; goto screadend; };

  /* read chunk header */
  while (_read(rdhandle,&sccnkhdr,6))
  { /* color chunks found? */
    if (getcolchunks(&sccnkhdr,&(scinfo->cmdescr),&(scinfo->cmap),rdhandle,loaderr)) continue;
    if (*loaderr) break;

    /* spritemap header found? */
    if (!strncmp(sccnkhdr.cnkname,shID,2)&&!(scinfo->smdescr))
    { scinfo->smdescr=farmalloc(sccnkhdr.cnklen);
      if (!(scinfo->smdescr)) { *loaderr=LP4ERR_NOMEM; break; };
      _read(rdhandle,scinfo->smdescr,(uword)sccnkhdr.cnklen);
      continue;
    };

    /* bit map found? */
    if (!strncmp(sccnkhdr.cnkname,bmID,2)&&!(scinfo->bmap)&&(scinfo->smdescr))
    { memlocptr=farmalloc(scinfo->smdescr->spritebitmap.maxbitmaplen);
      if (!memlocptr) { *loaderr=LP4ERR_NOMEM; break; };
      scinfo->bmap=memlocptr;
      /* reading the bitmap is somewhat complexer since we transfer
       * it in 32K blocks to memory.
       */
      for (;sccnkhdr.cnklen>32767;memadd+=32767,sccnkhdr.cnklen-=32767)
      { _read(rdhandle,memlocptr+memadd,32767); };
      _read(rdhandle,memlocptr+memadd,(uword)sccnkhdr.cnklen);
      continue;
    };

    /* overread unknown or double or misaligned chunk */
    lseek(rdhandle,sccnkhdr.cnklen,SEEK_CUR);
  };

  /* any error occured while reading chunks? */
  if (*loaderr) goto screadend;

  /* create tgdlp4 spritemap structure */
  if (!scinfo->bmap) goto screadend;
  scinfo->tgdsmdescr=farmalloc(sizeof(spritemap));
  if (!scinfo->tgdsmdescr) { *loaderr=LP4ERR_NOMEM; goto screadend; };

  /* init spritemap structure */
  scinfo->tgdsmdescr->animseqno=scinfo->smdescr->animseqno;
  scinfo->tgdsmdescr->animseqsize=scinfo->smdescr->animseqsize;

  /* create tgdlp4 bitmap structure out of spritemap header */
  bmhdr2bm(scinfo->bmap,&(scinfo->smdescr->spritebitmap),&(scinfo->tgdsmdescr->spritebitmap));

screadend:
  if (rdhandle!=-1) close(rdhandle);
  if (*loaderr && scinfo) { freesc(scinfo); return(0); };
  return(scinfo);
}
