/***************************************************************************
*   emstest3.c                                                             *
*   MODULE:  EMSIF                                                         *
*   OS:      DOS                                                           *
*   VERSION: 1.2                                                           *
*   DATE:    12/01/91                                                      *
*                                                                          *
*   Copyright (c) 1991 James W. Birdsall. All Rights Reserved.             *
*                                                                          *
*   Requires emsif.h, testutil.h, and emstest.h to compile.                *
*   Compiles under Borland C++ 2.0, TC 2.0, or MSC 6.0A.                   *
*                                                                          *
*   Regression test and example for EMSIF. See EMSTEST.C for more info.    *
*                                                                          *
***************************************************************************/

/*
** system includes <>
*/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <string.h>

/* include appropriate file according to compiler */
#ifdef __TURBOC__

#include <alloc.h>

#else

#include <malloc.h>

#endif


/*
** custom includes ""
*/

#include "testutil.h"

#include "emsif.h"
#include "emstest.h"


/*
** local #defines
*/

#define UCF         unsigned char far
#define UCH         unsigned char huge

/*
** Define LMALLOC() and LFREE() to the appropriate functions, according to
** compiler.
*/
#ifdef __TURBOC__

#define LMALLOC(kbytes)         farmalloc((((unsigned long) kbytes) * 1024L))
#define LFREE(ptr)              farfree((ptr))

#else

#define LMALLOC(kbytes)         halloc((kbytes), 1024)
#define LFREE(ptr)              hfree((ptr))

#endif

/*
** This macro is the same as TRIPLECHECK(), defined in EMSTEST.H, except that
** it calls functions which can handle a far pointer.
*/
#define LTRIPLECHECK(fu, st, ex, fr1, fr2, fr3)                         \
                         lfailcheck((fu), (st), (fr1), (fr2), (fr3));   \
                         lweirdretchk((fu), (st), (fr1), (fr2), (fr3)); \
                         lweirdcodechk((fu), (ex), (fr1), (fr2), (fr3))

/*
** Checks to see if a region of memory (possibly longer than 64K) is still
** incrementing word values, handles cleanup and exit if not.
*/
#define WORDCHECK(buf, len, msg, start)                         \
    if (lfarincwordcheck((buf), (len), (start)) != 0) {         \
        printf("Copy corrupted %s.\n", (msg)); EMMfree(handle); \
        LFREE(testbuf); exit(3); }

/*
** Check source and destination buffers, respectively.
*/
#define SRCWORDCHECK(buf, len)    WORDCHECK(buf, len, "source buffer", 0)
#define CPYWORDCHECK(buf, len)    WORDCHECK(buf, len, "copied bytes", 0)

/*
** Checks to see if a region of memory (possibly longer than 64K) is still
** filled with a given value, handles cleanup and exit if not.
*/
#define MEMCHECK(buf, len, val)                           \
    if (lfarmemcheck((UCF *) (buf), (len), (val)) != 0) { \
        printf("Copy corrupted destination.\n");          \
        EMMfree(handle); LFREE(testbuf); exit(3); }

/*
** Checks buffer for nonzero values.
*/
#define ZEROCHECK(buf, len)       MEMCHECK(buf, len, '\0')


/*
** misc: copyright strings, version macros, etc.
*/

/*
** typedefs
*/

/*
** global variables
*/

/* see EMSTEST.C for info */
extern int testno;
extern unsigned char far *frameptr[];
extern char *gblmsg;


/*
** static globals
*/

/*
** function prototypes
*/

static void do_nlongcopy_tests(void);
static void do_ilongcopy_tests(void);

static void lfailcheck(char *function, int status, void far *tofree1,
                                                     int tofree2, int tofree3);
static void lnofailcheck(char *function, int status, void far *tofree1,
                                                     int tofree2, int tofree3);
static void lweirdretchk(char *function, int status, void far *tofree1,
                                                     int tofree2, int tofree3);
static void lweirdcodechk(char *function, int expected, void far *tofree1,
                                                     int tofree2, int tofree3);


/*
** functions
*/


/***************************************************************************
*   FUNCTION: DO_LONGCOPY_TESTS                                            *
*                                                                          *
*   DESCRIPTION:                                                           *
*                                                                          *
*       Central dispatching function for long copy tests.                  *
*                                                                          *
*   ENTRY:                                                                 *
*                                                                          *
*       Void.                                                              *
*                                                                          *
*   EXIT:                                                                  *
*                                                                          *
*       Void.                                                              *
*                                                                          *
*   CONSTRAINTS/SIDE EFFECTS:                                              *
*                                                                          *
***************************************************************************/
void do_longcopy_tests(void)
{
    do_nlongcopy_tests();
    do_ilongcopy_tests();
    return;
} /* end of do_longcopy_tests() */


/***************************************************************************
*   FUNCTION: DO_NLONGCOPY_TESTS  (STATIC)                                 *
*                                                                          *
*   DESCRIPTION:                                                           *
*                                                                          *
*       Tests EMSIF functions EMMcopyto() and EMMcopyfrom() with copies    *
*       longer than one EMS page ( > 16384 bytes).                         *
*                                                                          *
*   ENTRY:                                                                 *
*                                                                          *
*       Void.                                                              *
*                                                                          *
*   EXIT:                                                                  *
*                                                                          *
*       Void.                                                              *
*                                                                          *
*   CONSTRAINTS/SIDE EFFECTS:                                              *
*                                                                          *
***************************************************************************/
static void do_nlongcopy_tests(void)
{
    unsigned char far *testbuf;
    int handle;
    int status;
    unsigned long ticks, totticks;
    int loop;
    unsigned long loop2;

    /* allocate memory to test against */
    testbuf = (unsigned char far *) LMALLOC(80);
    if (testbuf == (unsigned char far *) NULL)
    {
        printf("Cannot allocate test memory. Aborting.\n");
        exit(1);
    }

    /* now allocate some EMS to test with */
    handle = test_EMMalloc(81920L);

    /* fill test buffer with incrementing word pattern */
    lfarincwordfill(testbuf, 81920L, 0);

    /* fill EMS with a different pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try a copy off end */
    TESTHEADER();
    printf("Calling EMMcopyto() with offset off end of EMS block.\n");
    printf("Should fail.\n");
    status = EMMcopyto(2L, testbuf, handle, 90000L);
    lnofailcheck("EMMcopyto()", status, testbuf, handle, 0);
    lweirdretchk("EMMcopyto()", status, testbuf, handle, 0);
    lweirdcodechk("EMMcopyto()", EMM_BADOFFSET, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        ZEROCHECK(frameptr[0], 16384L);
    }
    TESTTAILER();

    /* and another */
    TESTHEADER();
    printf("Calling EMMcopyto() with length that will run off end of block.\n");
    printf("Should fail.\n");
    status = EMMcopyto(30000L, testbuf, handle, 55000L);
    lnofailcheck("EMMcopyto()", status, testbuf, handle, 0);
    lweirdretchk("EMMcopyto()", status, testbuf, handle, 0);
    lweirdcodechk("EMMcopyto()", EMM_BADOFFSET, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        ZEROCHECK(frameptr[0], 16384L);
    }
    TESTTAILER();

    /* now try copy across page boundary */
    TESTHEADER();
    printf("Calling EMMcopyto() with copy across page boundary.\n");
    printf("Should succeed.\n");
    status = EMMcopyto(10000L, testbuf, handle, 10000L);
    LTRIPLECHECK("EMMcopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    test_EMMmappage(0, handle, 0);
    test_EMMmappage(1, handle, 1);
    CPYWORDCHECK((frameptr[0] + 10000), 10000L);
    ZEROCHECK(frameptr[0], 10000L);
    ZEROCHECK((frameptr[0] + 10000 + 10000), ((32768L - 10000L) - 10000L));
    for (loop = 2; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        ZEROCHECK(frameptr[0], 16384L);
    }
    printf("EMMcopyto() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try copy across several page boundaries */
    TESTHEADER();
    printf("Calling EMMcopyto() with copy across several page boundaries.\n");
    printf("Should succeed.\n");
    status = EMMcopyto(50000L, testbuf, handle, 10000L);
    LTRIPLECHECK("EMMcopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 4; loop++)
    {
        test_EMMmappage(loop, handle, loop);
    }
    ZEROCHECK(frameptr[0], 10000L);
    CPYWORDCHECK((frameptr[0] + 10000), 50000L);
    ZEROCHECK((frameptr[0] + 10000U + 50000U), ((65536L - 10000L) - 50000L));
    test_EMMmappage(0, handle, 4);
    ZEROCHECK(frameptr[0], 16384L);
    printf("EMMcopyto() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try copy > 64K */
    TESTHEADER();
    printf("Calling EMMcopyto() with copy length > 64K.\n");
    printf("Should succeed.\n");
    ticks = get_tick();
    status = EMMcopyto(81920L, testbuf, handle, 0L);
    ticks = get_tick() - ticks;
    LTRIPLECHECK("EMMcopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "copied bytes", (8192U * loop));
    }
    printf("EMMcopyto() succeeded. 81920 bytes took %lu ticks.\n", ticks);
    TESTTAILER();

    /* restore destination pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* do same copy by longwords */
    TESTHEADER();
    printf("Calling EMMcopyto() to copy 81920 bytes by longwords.\n");
    printf("Should succeed, slowly.\n");
    totticks = 0L;
    for (loop2 = 0L; loop2 < 81920L; loop2 += 4L)
    {
        ticks = get_tick();
        status = EMMcopyto(4L, (UCF *)(((UCH *) testbuf) + loop2),
                                                                handle, loop2);
        totticks += (get_tick() - ticks);
        LTRIPLECHECK("EMMcopyto()", status, 0, testbuf, handle, 0);
    }
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "copied bytes", (8192U * loop));
    }
    printf("EMMcopyto() succeeded. 81920 bytes by longwords took %lu ticks.\n",
                                                                     totticks);
    TESTTAILER();


    /* fill test buffer with zeros */
    LFMEMSET(testbuf, '\0', 81920L);

    /* fill EMS with a different pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        lfarincwordfill(frameptr[0], 16384L, (8192U * loop));
    }

    /* try a copy off end */
    TESTHEADER();
    printf("Calling EMMcopyfrom() with offset off end of EMS block.\n");
    printf("Should fail.\n");
    status = EMMcopyfrom(2L, handle, 90000L, testbuf);
    lnofailcheck("EMMcopyfrom()", status, testbuf, handle, 0);
    lweirdretchk("EMMcopyfrom()", status, testbuf, handle, 0);
    lweirdcodechk("EMMcopyfrom()", EMM_BADOFFSET, testbuf, handle, 0);
    ZEROCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    TESTTAILER();

    /* and another */
    TESTHEADER();
    printf("Calling EMMcopyfrom() with length that will run off end of block.\n");
    printf("Should fail.\n");
    status = EMMcopyfrom(30000L, handle, 55000L, testbuf);
    lnofailcheck("EMMcopyfrom()", status, testbuf, handle, 0);
    lweirdretchk("EMMcopyfrom()", status, testbuf, handle, 0);
    lweirdcodechk("EMMcopyfrom()", EMM_BADOFFSET, testbuf, handle, 0);
    ZEROCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    TESTTAILER();

    /* now try copy across page boundary */
    TESTHEADER();
    printf("Calling EMMcopyfrom() with copy across page boundary.\n");
    printf("Should succeed.\n");
    status = EMMcopyfrom(10000L, handle, 0L, (testbuf + 10000));
    LTRIPLECHECK("EMMcopyfrom()", status, 0, testbuf, handle, 0);
    ZEROCHECK(testbuf, 10000L);
    CPYWORDCHECK((testbuf + 10000), 10000L);
    ZEROCHECK((testbuf + 10000 + 10000), ((81920L - 10000L) - 10000L));
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    printf("EMMcopyfrom() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    LFMEMSET(testbuf, '\0', 81920L);

    /* try copy across several page boundaries */
    TESTHEADER();
    printf("Calling EMMcopyfrom() with copy across several page boundaries.\n");
    printf("Should succeed.\n");
    status = EMMcopyfrom(50000L, handle, 0L, (testbuf + 10000));
    LTRIPLECHECK("EMMcopyfrom()", status, 0, testbuf, handle, 0);
    ZEROCHECK(testbuf, 10000L);
    CPYWORDCHECK((testbuf + 10000), 50000L);
    ZEROCHECK((testbuf + 10000U + 50000U), ((81920L - 10000L) - 50000L));
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    printf("EMMcopyfrom() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    LFMEMSET(testbuf, '\0', 81920L);

    /* try copy > 64K */
    TESTHEADER();
    printf("Calling EMMcopyfrom() with copy length > 64K.\n");
    printf("Should succeed.\n");
    ticks = get_tick();
    status = EMMcopyfrom(81920L, handle, 0L, testbuf);
    ticks = get_tick() - ticks;
    LTRIPLECHECK("EMMcopyfrom()", status, 0, testbuf, handle, 0);
    CPYWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    printf("EMMcopyfrom() succeeded. 81920 bytes took %lu ticks.\n", ticks);
    TESTTAILER();

    /* restore destination pattern */
    LFMEMSET(testbuf, '\0', 81920L);

    /* do same copy by longwords */
    TESTHEADER();
    printf("Calling EMMcopyfrom() to copy 81920 bytes by longwords.\n");
    printf("Should succeed, slowly.\n");
    totticks = 0L;
    for (loop2 = 0L; loop2 < 81920L; loop2 += 4L)
    {
        ticks = get_tick();
        status = EMMcopyfrom(4L, handle, loop2,
                                           (UCF *)(((UCH *) testbuf) + loop2));
        totticks += (get_tick() - ticks);
        LTRIPLECHECK("EMMcopyfrom()", status, 0, testbuf, handle, 0);
    }
    CPYWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    printf("EMMcopyfrom() succeeded. 81920 bytes by longwords took %lu ticks.\n",
                                                                     totticks);
    TESTTAILER();


    /* clean up */
    test_EMMfree(handle);
    LFREE(testbuf);

    return;
} /* end of do_nlongcopy_tests() */


/***************************************************************************
*   FUNCTION: DO_ILONGCOPY_TESTS  (STATIC)                                 *
*                                                                          *
*   DESCRIPTION:                                                           *
*                                                                          *
*       Tests EMSIF functions _EMMicopyto() and _EMMicopyfrom() with       *
*       copies longer than one EMS page ( > 16384 bytes).                  *
*                                                                          *
*   ENTRY:                                                                 *
*                                                                          *
*       Void.                                                              *
*                                                                          *
*   EXIT:                                                                  *
*                                                                          *
*       Void.                                                              *
*                                                                          *
*   CONSTRAINTS/SIDE EFFECTS:                                              *
*                                                                          *
***************************************************************************/
void do_ilongcopy_tests(void)
{
    unsigned char far *testbuf;
    int handle;
    int status;
    unsigned long ticks, totticks;
    int loop;

    /* allocate memory to test against */
    testbuf = (unsigned char far *) LMALLOC(80);
    if (testbuf == (unsigned char far *) NULL)
    {
        printf("Cannot allocate test memory. Aborting.\n");
        exit(1);
    }

    /* now allocate some EMS to test with */
    handle = test_EMMalloc(81920L);

    /* fill test buffer with incrementing word pattern */
    lfarincwordfill(testbuf, 81920L, 0);

    /* fill EMS with a different pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try a copy off end */
    TESTHEADER();
    printf("Calling _EMMicopyto() with offset off end of EMS block.\n");
    printf("Should fail.\n");
    status = EMMicopyto(2L, 2, 2, testbuf, handle, 90000L);
    lnofailcheck("_EMMicopyto()", status, testbuf, handle, 0);
    lweirdretchk("_EMMicopyto()", status, testbuf, handle, 0);
    lweirdcodechk("_EMMicopyto()", EMM_BADOFFSET, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        ZEROCHECK(frameptr[0], 16384L);
    }
    TESTTAILER();

    /* and another */
    TESTHEADER();
    printf("Calling _EMMicopyto() with length that will run off end of block.\n");
    printf("Should fail.\n");
    status = EMMicopyto(10000L, 2, 2, testbuf, handle, 55000L);
    lnofailcheck("_EMMicopyto()", status, testbuf, handle, 0);
    lweirdretchk("_EMMicopyto()", status, testbuf, handle, 0);
    lweirdcodechk("_EMMicopyto()", EMM_BADOFFSET, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        ZEROCHECK(frameptr[0], 16384L);
    }
    TESTTAILER();

    /* now try copy across page boundary, aligned */
    TESTHEADER();
    printf(
     "Calling _EMMicopyto() with copy across page boundary, element aligned.\n");
    printf("Should succeed.\n");
    status = EMMicopyto(2500L, 2, 2, testbuf, handle, 10000L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    status = EMMicopyto(2500L, 2, 2, (testbuf + 2), handle, 10002L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    test_EMMmappage(0, handle, 0);
    test_EMMmappage(1, handle, 1);
    CPYWORDCHECK((frameptr[0] + 10000), 10000L);
    ZEROCHECK(frameptr[0], 10000L);
    ZEROCHECK((frameptr[0] + 10000 + 10000), ((32768L - 10000L) - 10000L));
    for (loop = 2; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        ZEROCHECK(frameptr[0], 16384L);
    }
    printf("_EMMicopyto() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try copy across several page boundaries, element not aligned */
    TESTHEADER();
    printf("Calling _EMMicopyto() with copy across page boundary, %s\n",
                                                       "element not aligned.");
    printf("Should succeed.\n");
    status = EMMicopyto(5000L, 5, 5, testbuf, handle, 10000L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    status = EMMicopyto(5000L, 5, 5, (testbuf + 5), handle, 10005L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 4; loop++)
    {
        test_EMMmappage(loop, handle, loop);
    }
    ZEROCHECK(frameptr[0], 10000L);
    CPYWORDCHECK((frameptr[0] + 10000), 50000L);
    ZEROCHECK((frameptr[0] + 10000U + 50000U), ((65536L - 10000L) - 50000L));
    test_EMMmappage(0, handle, 4);
    ZEROCHECK(frameptr[0], 16384L);
    printf("_EMMicopyto() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* do full copy by longwords */
    TESTHEADER();
    printf(
       "Calling _EMMicopyto() to copy 81920 bytes by longwords, two passes.\n");
    printf("Should succeed.\n");
    ticks = get_tick();
    status = EMMicopyto(10240L, 4, 4, testbuf, handle, 0L);
    totticks = get_tick() - ticks;
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    ticks = get_tick();
    status = EMMicopyto(10240L, 4, 4, (testbuf + 4), handle, 4L);
    totticks += (get_tick() - ticks);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "copied bytes", (8192U * loop));
    }
    printf("_EMMicopyto() succeeded. 81920 bytes by longwords took %lu ticks.\n",
                                                                     totticks);
    TESTTAILER();


    /* restore destination pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try copy with maximum skip */
    TESTHEADER();
    printf("Calling _EMMicopyto() with skip of 32768.\n");
    printf("Should succeed.\n");
    status = EMMicopyto(2L, 8192, 32768U, testbuf, handle, 0L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    status = EMMicopyto(2L, 8192, 32768U, (testbuf + 8192), handle, 8192L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920);
    test_EMMmappage(0, handle, 0);
    WORDCHECK(frameptr[0], 16384L, "copied bytes", 0);
    test_EMMmappage(0, handle, 1);
    ZEROCHECK(frameptr[0], 16384);
    test_EMMmappage(0, handle, 2);
    ZEROCHECK(frameptr[0], 8192);
    WORDCHECK((frameptr[0] + 8192), 8192L, "copied bytes", (4096 * 5));
    test_EMMmappage(0, handle, 3);
    WORDCHECK(frameptr[0], 8192L, "copied bytes", (8192U * 3));
    ZEROCHECK((frameptr[0] + 8192), 8192);
    test_EMMmappage(0, handle, 4);
    ZEROCHECK(frameptr[0], 16384);
    printf("_EMMicopyto() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try copy with maximum element size */
    TESTHEADER();
    printf("Calling _EMMicopyto() with element size of 16384.\n");
    printf("Should succeed.\n");
    status = EMMicopyto(3L, 16384, 16384, testbuf, handle, 0L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920);
    test_EMMmappage(0, handle, 0);
    WORDCHECK(frameptr[0], 16384L, "copied bytes", 0);
    test_EMMmappage(0, handle, 1);
    ZEROCHECK(frameptr[0], 16384);
    test_EMMmappage(0, handle, 2);
    WORDCHECK(frameptr[0], 16384L, "copied bytes", (8192U * 2));
    test_EMMmappage(0, handle, 3);
    ZEROCHECK(frameptr[0], 16384);
    test_EMMmappage(0, handle, 4);
    WORDCHECK(frameptr[0], 16384L, "copied bytes", (8192U * 4U));
    status = EMMicopyto(2L, 16384, 16384, (testbuf + 16384), handle, 16384L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "copied bytes", (8192U * loop));
    }
    status = EMMicopyto(2L, 16384, 16384, (testbuf + 8192), handle, 8192L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "copied bytes", (8192U * loop));
    }
    printf("_EMMicopyto() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try copy with max skip and max element */
    TESTHEADER();
    printf("Calling _EMMicopyto() with skip 32768, element size 16384.\n");
    printf("Should succeed.\n");
    status = EMMicopyto(2L, 16384, 32768U, testbuf, handle, 0L);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    test_EMMmappage(0, handle, 0);
    WORDCHECK(frameptr[0], 16384L, "copied bytes", 0);
    test_EMMmappage(0, handle, 1);
    ZEROCHECK(frameptr[0], 16384);
    test_EMMmappage(0, handle, 2);
    ZEROCHECK(frameptr[0], 16384);
    test_EMMmappage(0, handle, 3);
    WORDCHECK(frameptr[0], 16384L, "copied bytes", (8192U * 3));
    test_EMMmappage(0, handle, 4);
    ZEROCHECK(frameptr[0], 16384);
    printf("_EMMicopyto() succeeded.\n");
    TESTTAILER();


    /* fill test buffer with zeros */
    LFMEMSET(testbuf, '\0', 81920L);

    /* fill EMS with a different pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        lfarincwordfill(frameptr[0], 16384L, (8192U * loop));
    }

    /* try a copy off end */
    TESTHEADER();
    printf("Calling _EMMicopyfrom() with offset off end of EMS block.\n");
    printf("Should fail.\n");
    status = EMMicopyfrom(2L, 2, 2, handle, 90000L, testbuf);
    lnofailcheck("_EMMicopyfrom()", status, testbuf, handle, 0);
    lweirdretchk("_EMMicopyfrom()", status, testbuf, handle, 0);
    lweirdcodechk("_EMMicopyfrom()", EMM_BADOFFSET, testbuf, handle, 0);
    ZEROCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    TESTTAILER();

    /* and another */
    TESTHEADER();
    printf("Calling _EMMicopyfrom() with length that will run off end of block.\n");
    printf("Should fail.\n");
    status = EMMicopyfrom(10000L, 2, 2, handle, 55000L, testbuf);
    lnofailcheck("_EMMicopyfrom()", status, testbuf, handle, 0);
    lweirdretchk("_EMMicopyfrom()", status, testbuf, handle, 0);
    lweirdcodechk("_EMMicopyfrom()", EMM_BADOFFSET, testbuf, handle, 0);
    ZEROCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    TESTTAILER();

    /* now try copy across page boundary, aligned */
    TESTHEADER();
    printf("Calling _EMMicopyfrom() with copy across page boundary, %s\n",
                                                           "element aligned.");
    printf("Should succeed.\n");
    status = EMMicopyfrom(2500L, 2, 2, handle, 0L, (testbuf + 10000));
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    status = EMMicopyfrom(2500L, 2, 2, handle, 2L, (testbuf + 10002));
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    ZEROCHECK(testbuf, 10000L);
    CPYWORDCHECK((testbuf + 10000), 10000L);
    ZEROCHECK((testbuf + 10000 + 10000), ((81920L - 10000L) - 10000L));
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    printf("_EMMicopyfrom() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    LFMEMSET(testbuf, '\0', 81920L);

    /* try copy across several page boundaries, element not aligned */
    TESTHEADER();
    printf("Calling _EMMicopyfrom() with copy across page boundary, %s\n",
                                                       "element not aligned.");
    printf("Should succeed.\n");
    status = EMMicopyfrom(5000L, 5, 5, handle, 0L, (testbuf + 10000));
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    status = EMMicopyfrom(5000L, 5, 5, handle, 5L, (testbuf + 10005));
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    ZEROCHECK(testbuf, 10000L);
    CPYWORDCHECK((testbuf + 10000), 50000L);
    ZEROCHECK((testbuf + 10000U + 50000U), ((81920L - 10000L) - 50000L));
    printf("_EMMicopyfrom() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    LFMEMSET(testbuf, '\0', 81920L);

    /* do full copy by longwords */
    TESTHEADER();
    printf(
     "Calling _EMMicopyfrom() to copy 81920 bytes by longwords, two passes.\n");
    printf("Should succeed.\n");
    ticks = get_tick();
    status = EMMicopyfrom(10240L, 4, 4, handle, 0L, testbuf);
    totticks = get_tick() - ticks;
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    ticks = get_tick();
    status = EMMicopyfrom(10240L, 4, 4, handle, 4L, (testbuf + 4));
    totticks += (get_tick() - ticks);
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    CPYWORDCHECK(testbuf, 81920L);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    printf("_EMMicopyfrom() succeeded. 81920 bytes by longwords took %lu ticks.\n",
                                                                     totticks);
    TESTTAILER();

    /* restore destination pattern */
    LFMEMSET(testbuf, '\0', 81920L);

    /* try copy with maximum skip */
    TESTHEADER();
    printf("Calling _EMMicopyfrom() with skip of 32768.\n");
    printf("Should succeed.\n");
    status = EMMicopyfrom(2L, 8192, 32768U, handle, 0L, testbuf);
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    status = EMMicopyfrom(2L, 8192, 32768U, handle, 8192L, (testbuf + 8192));
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    WORDCHECK(testbuf, 16384L, "copied bytes", 0);
    ZEROCHECK((testbuf + 16384), (16384L + 8192L));
    WORDCHECK((testbuf + (32768U + 8192U)), 16384L, "copied bytes", (4096*5));
    ZEROCHECK((testbuf + (8192U * 7U)), (16384 + 8192));
    printf("_EMMicopyfrom() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    LFMEMSET(testbuf, '\0', 81920L);

    /* try copy with maximum element size */
    TESTHEADER();
    printf("Calling _EMMicopyfrom() with element size of 16384.\n");
    printf("Should succeed.\n");
    status = EMMicopyfrom(3L, 16384, 16384, handle, 0L, testbuf);
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    WORDCHECK(testbuf, 16384L, "copied bytes", 0);
    ZEROCHECK((testbuf + 16384), 16384);
    WORDCHECK((testbuf + 32768U), 16384L, "copied bytes", (8192U * 2));
    ZEROCHECK((testbuf + (16384U * 3U)), 16384);
    WORDCHECK(MK_FP((FP_SEG(testbuf) + 0x1000), FP_OFF(testbuf)), 16384L, "copied bytes", (8192U * 4));
    status = EMMicopyfrom(2L, 16384, 16384, handle, 16384L, (testbuf + 16384));
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    WORDCHECK(testbuf, 81920L, "copied bytes", 0);
    status = EMMicopyfrom(2L, 16384, 16384, handle, 8192L, (testbuf + 8192));
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    WORDCHECK(testbuf, 81920L, "copied bytes", 0);
    printf("_EMMicopyfrom() succeeded.\n");
    TESTTAILER();

    /* restore destination pattern */
    LFMEMSET(testbuf, '\0', 81920L);

    /* try copy with max skip and max element */
    TESTHEADER();
    printf("Calling _EMMicopyfrom() with skip 32768, element size 16384.\n");
    printf("Should succeed.\n");
    status = EMMicopyfrom(2L, 16384, 32768U, handle, 0L, testbuf);
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        WORDCHECK(frameptr[0], 16384L, "source buffer", (8192U * loop));
    }
    WORDCHECK(testbuf, 16384L, "copied bytes", 0);
    ZEROCHECK((testbuf + 16384), 32768L);
    WORDCHECK((testbuf + (16384U * 3U)), 16384L, "copied bytes", (8192U * 3));
    ZEROCHECK(MK_FP((FP_SEG(testbuf) + 0x1000), FP_OFF(testbuf)), 16384);
    printf("_EMMicopyfrom() succeeded.\n");
    TESTTAILER();


    /* fill test buffer with incrementing word pattern */
    lfarincwordfill(testbuf, 81920L, 0);

    /* fill EMS with a different pattern */
    for (loop = 0; loop < 5; loop++)
    {
        test_EMMmappage(0, handle, loop);
        FMEMSET(frameptr[0], 0, 16384);
    }

    /* try interleaved copy and back */
    TESTHEADER();
    printf("Calling _EMMicopyto() with source and dest skip different.\n");
    printf("Then calling _EMMicopyfrom() to restore interleave.\n");
    printf("Should succeed.\n");
    status = _EMMicopyto(20480L, 2, 2, testbuf, handle, 0L, 0);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    status = _EMMicopyto(20480L, 2, 2, (testbuf + 2), handle, 40960L, 0);
    LTRIPLECHECK("_EMMicopyto()", status, 0, testbuf, handle, 0);
    SRCWORDCHECK(testbuf, 81920L);
    status = _EMMicopyfrom(20480L, 2, 0, handle, 0L, testbuf, 2);
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    status = _EMMicopyfrom(20480L, 2, 0, handle, 40960L, (testbuf + 2), 2);
    LTRIPLECHECK("_EMMicopyfrom()", status, 0, testbuf, handle, 0);
    CPYWORDCHECK(testbuf, 81920);
    TESTTAILER();


    /* clean up */
    test_EMMfree(handle);
    LFREE(testbuf);

    return;
} /* end of do_ilongcopy_tests() */


/*
** The following group of functions is essentially the same as the
** similarly-named functions in EMSTEST.C, but can handle a far pointer.
*/


/***************************************************************************
*   FUNCTION: LWEIRDRETCHK  (STATIC)                                       *
*                                                                          *
*   DESCRIPTION:                                                           *
*                                                                          *
*       This function checks to see if the status value passed to it is    *
*       either 0 or EMMOOPS, and assumes something has gone wrong if it    *
*       is not. If something has gone wrong, does some clean up before     *
*       exiting.                                                           *
*                                                                          *
*   ENTRY:                                                                 *
*                                                                          *
*       function - name of function which may have goofed                  *
*       status   - status value to check                                   *
*       tofree1  - conventional memory block to be freed on exit. Not      *
*                  freed if NULL.                                          *
*       tofree2  - EMS handle to be freed on exit. Not freed if 0.         *
*       tofree2  - another EMS handle to be freed on exit. Not freed if 0. *
*                                                                          *
*   EXIT:                                                                  *
*                                                                          *
*       Void, or may not return.                                           *
*                                                                          *
*   CONSTRAINTS/SIDE EFFECTS:                                              *
*                                                                          *
***************************************************************************/
static void lweirdretchk(char *function, int status, void far *tofree1,
                                                      int tofree2, int tofree3)
{
    if ((status != EMMOOPS) && (status != 0))
    {
        printf("%s returned weird value %d, code 0x%X.\n", function, status,
                                                     (unsigned int) _EMMerror);
        if (tofree1 != (void *) NULL)
        {
            LFREE(tofree1);
        }
        if (tofree2 != 0)
        {
            EMMfree(tofree2);
        }
        if (tofree3 != 0)
        {
            EMMfree(tofree3);
        }
        exit(3);
    }

    return;
} /* end of lweirdretchk() */


/***************************************************************************
*   FUNCTION: LWEIRDCODECHK  (STATIC)                                      *
*                                                                          *
*   DESCRIPTION:                                                           *
*                                                                          *
*       This function checks to see if the EMSIF error code value matches  *
*       the expected value, and assumes something has gone wrong if it     *
*       does not. If something has gone wrong, does some clean up before   *
*       exiting.                                                           *
*                                                                          *
*   ENTRY:                                                                 *
*                                                                          *
*       function - name of function which may have goofed                  *
*       expected - expected value of _EMMerror                             *
*       tofree1  - conventional memory block to be freed on exit. Not      *
*                  freed if NULL.                                          *
*       tofree2  - EMS handle to be freed on exit. Not freed if 0.         *
*       tofree2  - another EMS handle to be freed on exit. Not freed if 0. *
*                                                                          *
*   EXIT:                                                                  *
*                                                                          *
*       Void, or may not return.                                           *
*                                                                          *
*   CONSTRAINTS/SIDE EFFECTS:                                              *
*                                                                          *
***************************************************************************/
static void lweirdcodechk(char *function, int expected, void far *tofree1,
                                                      int tofree2, int tofree3)
{
    if ((int) _EMMerror != expected)
    {
        printf("%s returned unexpected code 0x%X.\n", function,
                                                     (unsigned int) _EMMerror);
        if (tofree1 != (void *) NULL)
        {
            LFREE(tofree1);
        }
        if (tofree2 != 0)
        {
            EMMfree(tofree2);
        }
        if (tofree3 != 0)
        {
            EMMfree(tofree3);
        }
        exit(3);
    }

    return;
} /* end of lweirdcodechk() */


/***************************************************************************
*   FUNCTION: LFAILCHECK  (STATIC)                                         *
*                                                                          *
*   DESCRIPTION:                                                           *
*                                                                          *
*       This function checks to see if the status value passed to it is    *
*       EMMOOPS and exits if it is. failcheck() is used when a function    *
*       is expected to succeed. Does some clean up before exiting.         *
*                                                                          *
*   ENTRY:                                                                 *
*                                                                          *
*       function - name of function which may have goofed                  *
*       status   - status value to be checked                              *
*       tofree1  - conventional memory block to be freed on exit. Not      *
*                  freed if NULL.                                          *
*       tofree2  - EMS handle to be freed on exit. Not freed if 0.         *
*       tofree2  - another EMS handle to be freed on exit. Not freed if 0. *
*                                                                          *
*   EXIT:                                                                  *
*                                                                          *
*       Void, or may not return.                                           *
*                                                                          *
*   CONSTRAINTS/SIDE EFFECTS:                                              *
*                                                                          *
***************************************************************************/
static void lfailcheck(char *function, int status, void far *tofree1,
                                                      int tofree2, int tofree3)
{
    if (status == EMMOOPS)
    {
        printf("%s failed, code 0x%X.\n", function, (unsigned int) _EMMerror);
        if (tofree1 != (void *) NULL)
        {
            LFREE(tofree1);
        }
        if (tofree2 != 0)
        {
            EMMfree(tofree2);
        }
        if (tofree3 != 0)
        {
            EMMfree(tofree3);
        }
        exit(3);
    }

    return;
} /* end of lfailcheck() */


/***************************************************************************
*   FUNCTION: LNOFAILCHECK  (STATIC)                                       *
*                                                                          *
*   DESCRIPTION:                                                           *
*                                                                          *
*       This function checks to see if the status value passed to it is    *
*       0 and exits if it is. nofailcheck() is used when a function is     *
*       expected to fail. Does some clean up before exiting.               *
*                                                                          *
*   ENTRY:                                                                 *
*                                                                          *
*       function - name of function which may have goofed                  *
*       status   - status value to be checked                              *
*       tofree1  - conventional memory block to be freed on exit. Not      *
*                  freed if NULL.                                          *
*       tofree2  - EMS handle to be freed on exit. Not freed if 0.         *
*       tofree2  - another EMS handle to be freed on exit. Not freed if 0. *
*                                                                          *
*   EXIT:                                                                  *
*                                                                          *
*       Void, or may not return.                                           *
*                                                                          *
*   CONSTRAINTS/SIDE EFFECTS:                                              *
*                                                                          *
***************************************************************************/
static void lnofailcheck(char *function, int status, void far *tofree1,
                                                      int tofree2, int tofree3)
{
    if (status == 0)
    {
        printf("%s did not fail.\n", function);
        if (tofree1 != (void *) NULL)
        {
            LFREE(tofree1);
        }
        if (tofree2 != 0)
        {
            EMMfree(tofree2);
        }
        if (tofree3 != 0)
        {
            EMMfree(tofree3);
        }
        exit(3);
    }

    return;
} /* end of lnofailcheck() */

