/*==============================================================================================*/
/*   font3d.cc                                                                     Font3D       */
/*----------------------------------------------------------------------------------------------*/
/*                                                                                              */
/*   Copyright (c) 1994, 1995 by Todd A. Prater                                Version 1.50     */
/*   All rights reserved.                                                                       */
/*                                                                                              */
/*==============================================================================================*/

#include <stdlib.h>
#include <iostream.h>
#include <stdio.h>
#include <string.h>

#include "config.h"
#include "geometry.h"
#include "truetype.h"
#include "font3d.h"

#define MAX_FILENAME_SIZE 1024



void PrintGreeting(void)
{
   cout<<"-------------------------------------------------------------------"<<endl;
   cout<<"  Font3D Version 1.50                                              "<<endl;
   cout<<"  3-D Text Object Creation Utility                                 "<<endl;
   cout<<endl;
   cout<<"  Copyright 1994, 1995 by Todd A. Prater.  All rights reserved.    "<<endl;
   cout<<"-------------------------------------------------------------------"<<endl;
}


void PrintPixarCopyright(void)
{
   cout<<"     The RenderMan Interface Procedures and RIB Protocol are:      "<<endl;
   cout<<"                   Copyright 1988, 1989, Pixar.                    "<<endl;
   cout<<"                       All rights reserved.                        "<<endl;
   cout<<"           RenderMan is a registered trademark of Pixar.           "<<endl;
   cout<<"-------------------------------------------------------------------"<<endl;
}


/*==============================================================================================*/
/*  main()                                                                                      */
/*==============================================================================================*/

int main(int argc, char* argv[])
{
   int           errorCode;
   int           errorPos;
   int           success;
   Font3DOptions options;
   USHORT        pid,sid,lid;

   int           numFileArgs;
   char**        fileArgs;

   char          fileName[MAX_FILENAME_SIZE];

   PrintGreeting();

   success = ReadOptions("font3d.def", numFileArgs, fileArgs);

   if (success==ERR_OutOfMemory)
   {
      cout<<endl<<"   ERROR: Out of memory."<<endl;
      exit(1);
   }
   else if (success!=ERR_UnableToOpenFile)
   {
      success = ParseOptions(numFileArgs,fileArgs,options,errorCode,errorPos);

      if (!success)
      {
         cout<<endl;
         cout<<"   ERROR: Invalid option \""<<fileArgs[errorPos]<<"\""<<endl;
         cout<<"          in configuration file \""<<"font3d.def"<<"\""<<endl<<endl;
         exit(1);
      }

      for (int i=0;i<numFileArgs;i++)
         delete fileArgs[i];
      delete fileArgs;
   }

   if (argc>1)
   {
      success = ParseOptions(argc-1,&argv[1],options,errorCode,errorPos);

      if (!success)
      {
         cout<<endl;
         cout<<"   ERROR: Invalid command line option \""
             <<argv[errorPos+1]<<"\""<<endl<<endl;;
         exit(1);
      }
   }

   if (options.fontFileName==NULL)
   {
       cout<<endl;
       cout<<"   ERROR: No Font File Specified."<<endl<<endl;;
       exit (1);
   }

   if (options.mapType==PID_Microsoft)
   {
      pid = PID_Microsoft;
      sid = SID_MS_UGL;
      lid = LID_MS_USEnglish;
   }
   else
   {
      pid = PID_Macintosh;
      sid = SID_MAC_Roman;
      lid = LID_MAC_English;
   }

   TTFont UserFont(options.fontFileName,options.fontPathName,pid,sid,lid);

   switch(UserFont.LastError())
   {
      case ERR_NoError:
         break;

      case ERR_OutOfMemory:
         cout<<endl; 
         cout<<"   ERROR: Out of memory."<<endl<<endl;
         exit(1);
         break;

      case ERR_UnableToOpenFile:
         cout<<endl;
         cout<<"   ERROR: Unable to open font file \""<<options.fontFileName
             <<"\"."<<endl<<endl;
         exit(1);
         break;

      case ERR_TableNotFound:
         cout<<endl;
         cout<<"   ERROR: A required part of this TrueType font is missing."<<endl<<endl;;
         exit(1);
         break;

      case ERR_UnknownCmapFormat:
         cout<<endl;
         cout<<"   ERROR: An invalid or unsupported character encoding method"<<endl;
         cout<<"          was specified."<<endl<<endl;
         exit(1);
         break;

      case ERR_UnknownKernFormat:
         break;

      default:
         cout<<endl;
         cout<<"   ERROR: Unable to read font file."<<endl;
         exit(1);
   }


   if (options.outputPathName==NULL)
      strcpy(fileName, options.outputFileName);
   else
      sprintf(fileName,"%s/%s",options.outputPathName,options.outputFileName);
   ofstream outputfile(fileName);
   if (!outputfile)
   {
      cout<<endl;
      cout<<"   ERROR: Unable to open \""<<options.outputFileName<<"\" for output."<<endl<<endl;
      exit(1);
   }

   if (options.outputFormat==RIB)
      PrintPixarCopyright();

   cout<<endl;

   cout<<"   Font File:     "<<options.fontFileName<<" ("<<UserFont.FullName()<<")"<<endl;
   cout<<"   Output File:   "<<options.outputFileName<<endl;
   cout<<"   Object Name:   "<<options.objectName<<endl;
   cout<<endl;

   cout<<"   Please Wait..."<<endl;

   success = OutputObjects(outputfile,UserFont,options);

   switch (success)
   {
      case ERR_NoError:
         cout<<"   Output File Successfully Created!"<<endl<<endl;
         break;
      case ERR_OutOfMemory:
         cout<<"   ERROR:  Out Of Memory."<<endl;
         cout<<"           Output file is probably incomplete."<<endl<<endl;
         break;
      case ERR_NoPolyFound:
         cout<<"   ERROR:  Unable to Triangulate."<<endl;
         cout<<"           Output file is probably incomplete."<<endl<<endl;
         break;
      default:
         cout<<"   ERROR:  Fatal Error."<<endl;
         cout<<"           Output file is probably incomplete."<<endl<<endl;
   }

   return 0;

}

