
/*
 * The Real SoundTracker - track editor
 *
 * Copyright (C) 1998-1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <gdk/gdkkeysyms.h>

#include "i18n.h"
#include "track-editor.h"
#include "gui.h"
#include "st-subs.h"
#include "keys.h"
#include "audio.h"
#include "main.h"
#include "gui-settings.h"
#include "sample-editor.h"
#include "gui-subs.h"
#include "preferences.h"

Tracker *tracker;
GtkWidget *vscrollbar;

static GtkWidget *hscrollbar;

static XMPattern *pattern_buffer = NULL;
static XMNote *track_buffer = NULL;
static int track_buffer_length;

static XMPattern block_buffer;
static int block_start_ch = -1, block_start_row = -1;

/* this array contains -1 if the note is not running, or the channel number where
   it is being played. this is necessary to handle the key on/off situation. */
static int note_running[96];

static int update_freq = 30;
static int gtktimer = -1;

/* jazz edit stuff */
static GtkWidget *jazzbox;
static GtkToggleButton *jazztoggles[32];
static int jazz_enabled = FALSE;

static void vscrollbar_changed(GtkAdjustment *adj);
static void hscrollbar_changed(GtkAdjustment *adj);
static void update_vscrollbar(Tracker *t, int patpos, int patlen, int disprows);
static void update_hscrollbar(Tracker *t, int leftchan, int numchans, int dispchans);
static gboolean track_editor_handle_column_input(Tracker *t, int gdkkey);

void
tracker_page_create (GtkNotebook *nb)
{
    GtkWidget *vbox, *hbox, *table, *thing;
    int i;

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(vbox);

    jazzbox = hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    thing = gtk_label_new(_("Jazz Edit:"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    table = gtk_table_new(1, 32, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    for(i = 0; i < 32; i++) {
	char buf[10];
	sprintf(buf, "%02d", i+1);
	thing = gtk_toggle_button_new_with_label(buf);
	jazztoggles[i] = GTK_TOGGLE_BUTTON(thing);
	gtk_widget_show(thing);
	gtk_table_attach(GTK_TABLE(table), thing, i, i + 1, 0, 1, GTK_FILL, 0, 0, 0);
    }

    table = gtk_table_new(2, 2, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    thing = tracker_new();
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 1, 0, 1);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "patpos", GTK_SIGNAL_FUNC(update_vscrollbar), NULL);
    gtk_signal_connect(GTK_OBJECT(thing), "xpanning", GTK_SIGNAL_FUNC(update_hscrollbar), NULL);
//    gtk_widget_set_events(thing, gtk_widget_get_events (thing) | GDK_BUTTON_PRESS_MASK);
//    gtk_signal_connect(GTK_OBJECT(thing), "button_press_event", GTK_SIGNAL_FUNC(tracker_popup_menu), NULL);
    tracker = TRACKER(thing);

//    tracker_create_popup_menu();

    hscrollbar = gtk_hscrollbar_new(NULL);
    gtk_table_attach(GTK_TABLE(table), hscrollbar, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);
    gtk_widget_show(hscrollbar);

    vscrollbar = gtk_vscrollbar_new(NULL);
    gtk_table_attach(GTK_TABLE(table), vscrollbar, 1, 2, 0, 1, 0, GTK_FILL, 0, 0);
    gtk_widget_show(vscrollbar);

    for(i = 0; i < sizeof(note_running) / sizeof(note_running[0]); i++) {
	note_running[i] = -1;
    }

    gtk_notebook_append_page(nb, vbox, gtk_label_new(_("Tracker")));
    gtk_container_border_width(GTK_CONTAINER(vbox), 10);

    memset(&block_buffer, 0, sizeof(block_buffer));
}

static void update_vscrollbar(Tracker *t, int patpos, int patlen, int disprows)
{
    gui_update_range_adjustment(GTK_RANGE(vscrollbar), patpos, patlen + disprows - 1, disprows, vscrollbar_changed);
}

static void update_hscrollbar(Tracker *t, int leftchan, int numchans, int dispchans)
{
    gui_update_range_adjustment(GTK_RANGE(hscrollbar), leftchan, numchans, dispchans, hscrollbar_changed);
}

static void vscrollbar_changed(GtkAdjustment *adj)
{
    if(gui_playing_mode != PLAYING_SONG && gui_playing_mode != PLAYING_PATTERN) {
	gtk_signal_handler_block_by_func(GTK_OBJECT(tracker), GTK_SIGNAL_FUNC(update_vscrollbar), NULL);
	tracker_set_patpos(TRACKER(tracker), adj->value);
	gtk_signal_handler_unblock_by_func(GTK_OBJECT(tracker), GTK_SIGNAL_FUNC(update_vscrollbar), NULL);
    }
}

static void hscrollbar_changed(GtkAdjustment *adj)
{
    tracker_set_xpanning(TRACKER(tracker), adj->value);
}

void
track_editor_toggle_jazz_edit (void)
{
    if(!jazz_enabled) {
	gtk_widget_show(jazzbox);
	jazz_enabled = TRUE;
    } else {
	gtk_widget_hide(jazzbox);
	jazz_enabled = FALSE;
    }
}

void
track_editor_do_the_note_key (int notekeymeaning,
			      gboolean pressed,
			      guint32 xkeysym,
			      int modifiers)
{
    int j, n;
    int note;

    note = notekeymeaning + 12 * gui_get_current_octave_value() + 1;

    if(!(note >= 0 && note < 96))
	return;

    if(pressed) {
	if(note_running[note] == -1) {
	    gui_play_note(tracker->cursor_ch, note);
	    note_running[note] = tracker->cursor_ch;
	    if(jazz_enabled) {
		for(j = 0, n = (tracker->cursor_ch + 1) % xm->num_channels; j < xm->num_channels - 1; j++, n = (n+1) % xm->num_channels) {
		    if(jazztoggles[n]->active)
			break;
		}
		tracker_step_cursor_channel(tracker, n - tracker->cursor_ch);
	    }
	}
    } else {
	if(note_running[note] != -1) {
	    if(keys_is_key_pressed(xkeysym, modifiers)) {
		/* this is just an auto-repeat fake keyoff. pooh.
		   in reality this key is still being held down */
		return;
	    }
	    for(j = 0, n = 0; j < sizeof(note_running) / sizeof(note_running[0]); j++) {
		n += (note_running[j] == note_running[note]);
	    }
	    if(n == 1) {
	        // only do the keyoff if all previous keys are off.
		gui_play_note_keyoff(note_running[note]);
	    }
	    note_running[note] = -1;
	}
    }
}

gboolean
track_editor_handle_keys (int shift,
			  int ctrl,
			  int alt,
			  guint32 keyval,
			  gboolean pressed)
{
    int i, m;
    Tracker *t = tracker;
    gboolean handled = FALSE;

    if(t->cursor_item == 0) {
	m = i = keys_get_key_meaning(keyval, ENCODE_MODIFIERS(shift, ctrl, alt));
	if(i != -1) {
	    switch(KEYS_MEANING_TYPE(i)) {
	    case KEYS_MEANING_NOTE:
		i += 12 * gui_get_current_octave_value() + 1;
		if(i < 96) {
		    if(pressed && GTK_TOGGLE_BUTTON(editing_toggle)->active) {
			XMNote *note = &t->curpattern->channels[t->cursor_ch][t->patpos];
			note->note = i;
			note->instrument = gui_get_current_instrument();
			tracker_redraw_current_row(t);
			tracker_step_cursor_row(t, gui_get_current_jump_value());
			xm->modified = 1;
		    }
		    track_editor_do_the_note_key(m, pressed, keyval, ENCODE_MODIFIERS(shift, ctrl, alt));
		}
		break;
	    case KEYS_MEANING_KEYOFF:
		if(pressed && GTK_TOGGLE_BUTTON(editing_toggle)->active) {
		    XMNote *note = &t->curpattern->channels[t->cursor_ch][t->patpos];
		    note->note = 97;
		    note->instrument = 0;
		    tracker_redraw_current_row(t);
		    tracker_step_cursor_row(t, gui_get_current_jump_value());
		    xm->modified = 1;
		}
		break;
	    }
	    return TRUE;
	}
    }

    if(!pressed)
	return FALSE;

    switch (keyval) {
    case GDK_Up:
	if(GUI_ENABLED && !ctrl && !shift && !alt) {
	    tracker_set_patpos(t, t->patpos > 0 ? t->patpos - 1 : t->curpattern->length - 1);
	    handled = TRUE;
	}
	break;
    case GDK_Down:
	if(GUI_ENABLED && !ctrl && !shift && !alt) {
	    tracker_set_patpos(t, t->patpos < t->curpattern->length - 1 ? t->patpos + 1 : 0);
	    handled = TRUE;
	}
	break;
    case GDK_Page_Up:
	if(!GUI_ENABLED)
	    break;
	if(t->patpos >= 16)
	    tracker_set_patpos(t, t->patpos - 16);
	else
	    tracker_set_patpos(t, 0);
	handled = TRUE;
	break;
    case GDK_Page_Down:
	if(!GUI_ENABLED)
	    break;
	if(t->patpos < t->curpattern->length - 16)
	    tracker_set_patpos(t, t->patpos + 16);
	else
	    tracker_set_patpos(t, t->curpattern->length - 1);
	handled = TRUE;
	break;
    case GDK_F9:
    case GDK_Home:
	if(GUI_ENABLED) {
	    tracker_set_patpos(t, 0);
	    handled = TRUE;
	}
	break;
    case GDK_F10:
	if(GUI_ENABLED) {
	    tracker_set_patpos(t, t->curpattern->length/4);
	    handled = TRUE;
	}
	break;
    case GDK_F11:
	if(GUI_ENABLED) {
	    tracker_set_patpos(t, t->curpattern->length/2);
	    handled = TRUE;
	}
	break;
    case GDK_F12:
	if(GUI_ENABLED) {
	    tracker_set_patpos(t, 3*t->curpattern->length/4);
	    handled = TRUE;
	}
	break;
    case GDK_End:
	if(GUI_ENABLED) {
	    tracker_set_patpos(t, t->curpattern->length - 1);
	    handled = TRUE;
	}
	break;
    case GDK_Left:
	if(!shift && !ctrl && !alt) {
	    /* cursor left */
	    tracker_step_cursor_item(t, -1);
	    handled = TRUE;
	}
	break;
    case GDK_Right:
	if(!shift && !ctrl && !alt) {
	    /* cursor right */
	    tracker_step_cursor_item(t, 1);
	    handled = TRUE;
	}
	break;
    case GDK_Tab:
    case GDK_ISO_Left_Tab:
	tracker_step_cursor_channel(t, shift ? -1 : 1);
	handled = TRUE;
	break;
    case GDK_Shift_R:
	/* record pattern */
	break;
    case GDK_Delete:
    case GDK_BackSpace:
	if(GTK_TOGGLE_BUTTON(editing_toggle)->active) {
	    XMNote *note = &t->curpattern->channels[t->cursor_ch][t->patpos];
	    note->note = 0;
	    note->instrument = 0;
	    tracker_redraw_current_row(t);
	    tracker_step_cursor_row(t, gui_get_current_jump_value());
	    xm->modified = 1;
	    handled = TRUE;
	}
	break;
    default:
	if(!ctrl) {
	    if(GTK_TOGGLE_BUTTON(editing_toggle)->active) {
		handled = track_editor_handle_column_input(t, keyval);
	    }
	}
	break;
    }

    return handled;
}

void
track_editor_copy_pattern (Tracker *t)
{
    XMPattern *p = t->curpattern;

    if(pattern_buffer) {
	st_free_pattern_channels(pattern_buffer);
	free(pattern_buffer);
    }
    pattern_buffer = st_dup_pattern(p);
    tracker_redraw(t);
}

void
track_editor_cut_pattern (Tracker *t)
{
    XMPattern *p = t->curpattern;

    if(pattern_buffer) {
	st_free_pattern_channels(pattern_buffer);
	free(pattern_buffer);
    }
    pattern_buffer = st_dup_pattern(p);
    st_clear_pattern(p);
    xm->modified = 1;
    tracker_redraw(t);
}

void
track_editor_paste_pattern (Tracker *t)
{
    XMPattern *p = t->curpattern;
    int i;

    if(!pattern_buffer)
	return;
    for(i = 0; i < 32; i++) {
	free(p->channels[i]);
	p->channels[i] = st_dup_track(pattern_buffer->channels[i], pattern_buffer->length);
    }
    p->alloc_length = pattern_buffer->length;
    if(p->length != pattern_buffer->length) {
	p->length = pattern_buffer->length;
	gui_update_pattern_data();
	tracker_reset(t);
    } else {
	tracker_redraw(t);
    }
    xm->modified = 1;
}

void
track_editor_copy_track (Tracker *t)
{
    int l = t->curpattern->length;
    XMNote *n = t->curpattern->channels[t->cursor_ch];

    if(track_buffer) {
	free(track_buffer);
    }
    track_buffer_length = l;
    track_buffer = st_dup_track(n, l);
    tracker_redraw(t);
}

void
track_editor_cut_track (Tracker *t)
{
    int l = t->curpattern->length;
    XMNote *n = t->curpattern->channels[t->cursor_ch];

    if(track_buffer) {
	free(track_buffer);
    }
    track_buffer_length = l;
    track_buffer = st_dup_track(n, l);
    st_clear_track(n, l);
    xm->modified = 1;
    tracker_redraw(t);
}

void
track_editor_paste_track (Tracker *t)
{
    int l = t->curpattern->length;
    XMNote *n = t->curpattern->channels[t->cursor_ch];
    int i;

    if(!track_buffer)
	return;
    i = track_buffer_length;
    if(l < i)
	i = l;
    while(i--)
	n[i] = track_buffer[i];
    xm->modified = 1;
    tracker_redraw(t);
}

void
track_editor_delete_track (Tracker *t)
{
    st_pattern_delete_track(t->curpattern, t->cursor_ch);
    xm->modified = 1;
    tracker_redraw(t);
}

void
track_editor_insert_track (Tracker *t)
{
    st_pattern_insert_track(t->curpattern, t->cursor_ch);
    xm->modified = 1;
    tracker_redraw(t);
}

void
track_editor_mark_selection (Tracker *t)
{
    block_start_ch = t->cursor_ch;
    block_start_row = t->patpos;
}

static void
track_editor_copy_cut_selection_common (Tracker *t,
					gboolean cut)
{
    int i;
    int height, width;

    if(block_start_ch == -1 || block_start_row >= t->curpattern->length || block_start_ch >= xm->num_channels)
	return;

    width = t->cursor_ch - block_start_ch + 1;
    if(width <= 0) {
	width += xm->num_channels;
    }
    height = t->patpos - block_start_row + 1;
    if(height <= 0) {
	height += t->curpattern->length;
    }

    printf("%d %d\n", width, height);
    block_buffer.alloc_length = block_buffer.length = height;

    for(i = 0; i < 32; i++) {
	free(block_buffer.channels[i]);
	block_buffer.channels[i] = NULL;
    }

    for(i = 0; i < width; i++) {
	block_buffer.channels[i] = st_dup_track_wrap(t->curpattern->channels[(block_start_ch + i) % xm->num_channels],
						     t->curpattern->length,
						     block_start_row,
						     height);
	if(cut) {
	    st_clear_track_wrap(t->curpattern->channels[(block_start_ch + i) % xm->num_channels],
				t->curpattern->length,
				block_start_row,
				height);
	}
    }
}

void
track_editor_copy_selection (Tracker *t)
{
    track_editor_copy_cut_selection_common(t, FALSE);
}

void
track_editor_cut_selection (Tracker *t)
{
    track_editor_copy_cut_selection_common(t, TRUE);
    xm->modified = 1;
    tracker_redraw(t);
}

void
track_editor_paste_selection (Tracker *t)
{
    int i;

    if(block_buffer.length > t->curpattern->length)
	return;

    for(i = 0; i < 32; i++) {
	st_paste_track_into_track_wrap(block_buffer.channels[i],
				       t->curpattern->channels[(t->cursor_ch + i) % xm->num_channels],
				       t->curpattern->length,
				       t->patpos,
				       block_buffer.length);
    }

    xm->modified = 1;
    tracker_set_patpos(t, (t->patpos + block_buffer.length) % t->curpattern->length);
    tracker_redraw(t);
}

static void
track_editor_handle_semidec_column_input (Tracker *t,
					  int exp,
					  gint8 *modpt,
					  int n)
{
    switch(exp) {
    case 0:
	if(n < 0 || n > 9)
	    return;
	*modpt = (*modpt / 10) * 10 + n;
	break;
    case 1:
	if(n < 0 || n > 26)
	    return;
	*modpt = (*modpt % 10) + 10 * n;
	break;
    }

    tracker_redraw_current_row(t);
    if(gui_settings.advance_cursor_in_fx_columns)
	tracker_step_cursor_row(t, 1);
    else
	tracker_step_cursor_item(t, 1);
    xm->modified = 1;
}

static void
track_editor_handle_hex_column_input (Tracker *t,
				      int exp,
				      gint8 *modpt,
				      int n)
{
    int s;

    if(n < 0 || n > 15)
	return;

    exp *= 4;
    s = *modpt & (0xf0 >> exp);
    s |= n << exp;
    *modpt = s;
    tracker_redraw_current_row(t);
    if(gui_settings.advance_cursor_in_fx_columns)
	tracker_step_cursor_row(t, 1);
    else
	tracker_step_cursor_item(t, 1);
    xm->modified = 1;
}

static gboolean
track_editor_handle_column_input (Tracker *t,
				  int gdkkey)
{
    int n;
    XMNote *note = &t->curpattern->channels[t->cursor_ch][t->patpos];

    if(t->cursor_item == 5) {
	/* Effect column (not the parameter) */
	switch(gdkkey) {
	case '0' ... '9':
	    n = gdkkey - '0';
	    break;
	case 'a' ... 'z': case 'A' ... 'Z':
	    gdkkey = tolower(gdkkey);
	    n = gdkkey - 'a' + 10;
	    break;
	default:
	    return FALSE;
	}
	note->fxtype = n;
	tracker_redraw_current_row(t);
	if(gui_settings.advance_cursor_in_fx_columns)
	    tracker_step_cursor_row(t, 1);
	else
	    tracker_step_cursor_item(t, 1);
	xm->modified = 1;
	return TRUE;
    }

    gdkkey = tolower(gdkkey);
    n = gdkkey - '0' - (gdkkey >= 'a') * ('a' - '9' - 1);

    switch(t->cursor_item) {
    case 1: case 2: /* instrument column */
	if(gui_settings.tracker_hexmode)
	    track_editor_handle_hex_column_input(t, 2 - t->cursor_item, &note->instrument, n);
	else
	    track_editor_handle_semidec_column_input(t, 2 - t->cursor_item, &note->instrument, n);
	break;
    case 3: case 4: /* volume column */
	track_editor_handle_hex_column_input(t, 4 - t->cursor_item, &note->volume, n);
	break;
    case 6: case 7: /* effect parameter */
	track_editor_handle_hex_column_input(t, 7 - t->cursor_item, &note->fxparam, n);
	break;
    default:
	return FALSE;
    }

    return TRUE;
}

gint
tracker_timeout (gpointer data)
{
    double display_songtime;
    audio_player_pos *p;

    g_assert(current_driver_object);

    display_songtime = current_driver->get_play_time(current_driver_object);

    p = time_buffer_get(audio_playerpos_tb, display_songtime);
    if(p) {
	gui_update_player_pos(p->time, p->songpos, p->patpos);
    }

    // Not quite the right place for this, but anyway...
    gui_clipping_indicator_update(display_songtime);
    sample_editor_update_mixer_position(display_songtime);

    return TRUE;
}

void
tracker_start_updating (void)
{
    if(gtktimer != -1)
	return;

    gtktimer = gtk_timeout_add(1000/update_freq, tracker_timeout, NULL);
}

void
tracker_stop_updating (void)
{
    if(gtktimer == -1)
	return;

    gtk_timeout_remove(gtktimer);
    gtktimer = -1;
    gui_clipping_indicator_update(-1.0);
    sample_editor_update_mixer_position(-1.0);
}

void
tracker_set_update_freq (int freq)
{
    update_freq = freq;
    if(gtktimer != -1) {
	tracker_stop_updating();
	tracker_start_updating();
    }
}

void
track_editor_load_config (void)
{
    char buf[256];
    FILE *f;
    int i, j;

    sprintf(buf, "%s/jazz", prefs_get_prefsdir());

    f = fopen(buf, "rb");
    if(f) {
	for(i = 0; i < 32; i++) {
	    sprintf(buf, "jazz-toggle-%d", i);
	    prefs_get_int(f, buf, &j);
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(jazztoggles[i]), j);
	}

	fclose(f);
    }
}

void
track_editor_save_config (void)
{
    char buf[256];
    FILE *f;
    int i;

    prefs_check_prefs_dir();
    sprintf(buf, "%s/jazz", prefs_get_prefsdir());

    f = fopen(buf, "wb");
    if(!f)
	return;

    for(i = 0; i < 32; i++) {
	sprintf(buf, "jazz-toggle-%d", i);
	prefs_put_int(f, buf, GTK_TOGGLE_BUTTON(jazztoggles[i])->active);
    }

    fclose(f);
}


