
/*
 * The Real SoundTracker - sample editor
 *
 * Copyright (C) 1998-1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <stdio.h>
#include <string.h>

#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>

#ifndef NO_AUDIOFILE
#include <audiofile.h>
#endif

#include <glib.h>
#include <gtk/gtk.h>
#ifdef USE_GNOME
#include <gnome.h>
#endif

#include "i18n.h"
#include "sample-editor.h"
#include "xm.h"
#include "st-subs.h"
#include "gui.h"
#include "gui-subs.h"
#include "instrument-editor.h"
#include "sample-display.h"
#include "endian-conv.h"
#include "keys.h"
#include "track-editor.h"
#include "errors.h"
#include "time-buffer.h"
#include "audio.h"
#include "mixer.h"
#include "module-info.h"
#include "file-operations.h"

// == GUI variables

static STSample *current_sample = NULL;

static GtkWidget *spin_volume, *spin_panning, *spin_finetune, *spin_relnote;
static GtkWidget *savebutton;
static SampleDisplay *sampledisplay;
static GtkWidget *sample_editor_hscrollbar;
static GtkWidget *loopradio[3], *resolution_radio[2];
static GtkWidget *spin_loopstart, *spin_loopend, *spin_selstart, *spin_selend;
static GtkWidget *box_loop, *box_params, *box_sel;
static GtkWidget *label_slength;

// = Volume ramping dialog

static GtkWidget *volrampwindow = NULL;
static GtkWidget *sample_editor_volramp_spin_w[2];

// = Sampler variables

static SampleDisplay *monitorscope;
static GtkWidget *cancelbutton, *okbutton, *startsamplingbutton;

st_in_driver *sampling_driver = NULL;
void *sampling_driver_object = NULL;

static GtkWidget *samplingwindow = NULL;

struct recordbuf {
    struct recordbuf *next;
    int length;
    gint16 data[0];
};

static struct recordbuf *recordbufs, *current;
static const int recordbuflen = 10000;
static int currentoffs;
static int recordedlen, sampling;

// = Editing operations variables

static void *copybuffer = NULL;
static int copybufferlen;
static int copybuffertype;


static void sample_editor_ok_clicked(void);
static void sample_editor_start_sampling_clicked(void);

static void sample_editor_spin_volume_changed(GtkSpinButton *spin);
static void sample_editor_spin_panning_changed(GtkSpinButton *spin);
static void sample_editor_spin_finetune_changed(GtkSpinButton *spin);
static void sample_editor_spin_relnote_changed(GtkSpinButton *spin);
static void sample_editor_loop_changed(void);
static void sample_editor_spin_selection_changed(void);
static void sample_editor_display_loop_changed(SampleDisplay *, int start, int end);
static void sample_editor_display_selection_changed(SampleDisplay *, int start, int end);
static void sample_editor_display_window_changed(SampleDisplay *, int start, int end);
static void sample_editor_reset_selection_clicked(void);
static void sample_editor_clear_clicked(void);
static void sample_editor_show_all_clicked(void);
static void sample_editor_zoom_in_clicked(void);
static void sample_editor_zoom_out_clicked(void);
static void sample_editor_loopradio_changed(void);
static void sample_editor_resolution_changed(void);
static void sample_editor_monitor_clicked(void);
static void sample_editor_cut_clicked(void);
static void sample_editor_remove_clicked(void);
static void sample_editor_copy_clicked(void);
static void sample_editor_paste_clicked(void);
static void sample_editor_zoom_to_selection_clicked(void);

static void sample_editor_load_wav(void);
static void sample_editor_save_wav(void);

static void sample_editor_open_volume_ramp_dialog(void);
static void sample_editor_close_volume_ramp_dialog(void);
static void sample_editor_perform_ramp(GtkWidget *w, gpointer data);

static void
sample_editor_lock_sample (void)
{
    g_mutex_lock(current_sample->sample.lock);
}

static void
sample_editor_unlock_sample (void)
{
    if(gui_playing_mode) {
	mixer->updatesample(&current_sample->sample);
    }
    g_mutex_unlock(current_sample->sample.lock);
}

void
sample_editor_page_create (GtkNotebook *nb)
{
    GtkWidget *box, *thing, *hbox, *vbox;
    static const char *looplabels[] = {
	N_("No loop"),
	N_("Amiga"),
	N_("PingPong"),
	NULL
    };
    static const char *resolutionlabels[] = {
	N_("8 bits"),
	N_("16 bits"),
	NULL
    };

    box = gtk_vbox_new(FALSE, 2);
    gtk_container_border_width(GTK_CONTAINER(box), 10);
    gtk_notebook_append_page(nb, box, gtk_label_new(_("Sample Editor")));
    gtk_widget_show(box);

    thing = sample_display_new(TRUE);
    gtk_box_pack_start(GTK_BOX(box), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "loop_changed",
		       GTK_SIGNAL_FUNC(sample_editor_display_loop_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(thing), "selection_changed",
		       GTK_SIGNAL_FUNC(sample_editor_display_selection_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(thing), "window_changed",
		       GTK_SIGNAL_FUNC(sample_editor_display_window_changed), NULL);
    sampledisplay = SAMPLE_DISPLAY(thing);
    sample_display_enable_zero_line(SAMPLE_DISPLAY(thing), TRUE);

    sample_editor_hscrollbar = gtk_hscrollbar_new(NULL);
    gtk_widget_show(sample_editor_hscrollbar);
    gtk_box_pack_start(GTK_BOX(box), sample_editor_hscrollbar, FALSE, TRUE, 0);

    hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, TRUE, 0);
    gtk_widget_show(hbox);

    box_loop = vbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, TRUE, 0);

    make_radio_group((const char**)looplabels, vbox, loopradio, FALSE, FALSE, sample_editor_loopradio_changed);
    gui_put_labelled_spin_button(vbox, _("Start"), 0, 0, &spin_loopstart, sample_editor_loop_changed, NULL);
    gui_put_labelled_spin_button(vbox, _("End"), 0, 0, &spin_loopend, sample_editor_loop_changed, NULL);
    sample_editor_loopradio_changed();

    thing = gtk_vseparator_new();
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);

    box_params = vbox = gtk_vbox_new(TRUE, 2);
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, TRUE, 0);

    gui_put_labelled_spin_button(vbox, _("Volume"), 0, 64, &spin_volume, sample_editor_spin_volume_changed, NULL);
    gui_put_labelled_spin_button(vbox, _("Panning"), -128, 127, &spin_panning, sample_editor_spin_panning_changed, NULL);
    gui_put_labelled_spin_button(vbox, _("Finetune"), -128, 127, &spin_finetune, sample_editor_spin_finetune_changed, NULL);
    make_radio_group((const char**)resolutionlabels, vbox, resolution_radio, FALSE, FALSE, sample_editor_resolution_changed);

    thing = gtk_vseparator_new();
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);

    box_sel = vbox = gtk_vbox_new(TRUE, 2);
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, TRUE, 0);

    gui_put_labelled_spin_button(vbox, _("SelStart"), 0, 0, &spin_selstart, sample_editor_spin_selection_changed, NULL);
    gui_put_labelled_spin_button(vbox, _("SelEnd"), 0, 0, &spin_selend, sample_editor_spin_selection_changed, NULL);
    gtk_widget_set_sensitive(spin_selstart, 0);
    gtk_widget_set_sensitive(spin_selend, 0);
    thing = gtk_button_new_with_label(_("Reset Sel"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_reset_selection_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    gui_put_labelled_spin_button(vbox, _("RelNote"), -128, 127, &spin_relnote, sample_editor_spin_relnote_changed, NULL);
    label_slength = thing = gtk_label_new(_("Length: 0"));
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(vbox), thing, FALSE, TRUE, 0);

    thing = gtk_vseparator_new();
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

#ifndef NO_AUDIOFILE
    fileops_dialogs[DIALOG_LOAD_SAMPLE] = file_selection_create(_("Load Sample.."), sample_editor_load_wav);
    fileops_dialogs[DIALOG_SAVE_SAMPLE] = file_selection_create(_("Save WAV.."), sample_editor_save_wav);
#endif

    thing = gtk_button_new_with_label(_("Load Sample"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(fileops_open_dialog), (void*)DIALOG_LOAD_SAMPLE);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
#ifdef NO_AUDIOFILE
    gtk_widget_set_sensitive(thing, 0);
#endif

    thing = gtk_button_new_with_label(_("Save WAV"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(fileops_open_dialog), (void*)DIALOG_SAVE_SAMPLE);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    savebutton = thing;
#ifdef NO_AUDIOFILE
    gtk_widget_set_sensitive(thing, 0);
#endif

    thing = gtk_button_new_with_label(_("Clear"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_clear_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Monitor"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_monitor_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Volume Ramp"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_open_volume_ramp_dialog), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

    thing = gtk_button_new_with_label(_("Zoom to selection"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_zoom_to_selection_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Show all"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_show_all_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Zoom in (+50%)"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_zoom_in_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Zoom out (-50%)"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_zoom_out_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Resample"));
/*    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_monitor_clicked), NULL);
*/    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_widget_set_sensitive(thing, 0);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

    thing = gtk_button_new_with_label(_("Cut"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_cut_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Remove"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_remove_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Copy"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_copy_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Paste"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_paste_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label(_("Filter"));
/*    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_filter_clicked), NULL);
*/    gtk_box_pack_start(GTK_BOX(vbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_widget_set_sensitive(thing, 0);
}

static void
sample_editor_block_loop_spins (int block)
{
    void (*func) (GtkObject*, GtkSignalFunc, gpointer);

    func = block ? gtk_signal_handler_block_by_func : gtk_signal_handler_unblock_by_func;

    func(GTK_OBJECT(spin_loopstart), sample_editor_loop_changed, NULL);
    func(GTK_OBJECT(spin_loopend), sample_editor_loop_changed, NULL);
}

static void
sample_editor_block_sel_spins (int block)
{
    void (*func) (GtkObject*, GtkSignalFunc, gpointer);

    func = block ? gtk_signal_handler_block_by_func : gtk_signal_handler_unblock_by_func;

    func(GTK_OBJECT(spin_selstart), sample_editor_spin_selection_changed, NULL);
    func(GTK_OBJECT(spin_selend), sample_editor_spin_selection_changed, NULL);
}

static void
sample_editor_blocked_set_loop_spins (int start,
				      int end)
{
    sample_editor_block_loop_spins(1);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_loopstart), start);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_loopend), end);
    sample_editor_block_loop_spins(0);
}

static void
sample_editor_blocked_set_sel_spins (int start,
				     int end)
{
    sample_editor_block_sel_spins(1);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_selstart), start);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_selend), end);
    sample_editor_block_sel_spins(0);
}

static void
sample_editor_blocked_set_display_loop (int start,
					int end)
{
    gtk_signal_handler_block_by_func(GTK_OBJECT(sampledisplay), GTK_SIGNAL_FUNC(sample_editor_display_loop_changed), NULL);
    sample_display_set_loop(sampledisplay, start, end);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(sampledisplay), GTK_SIGNAL_FUNC(sample_editor_display_loop_changed), NULL);
}

void
sample_editor_update (void)
{
    STSample *sts = current_sample;
    st_mixer_sample_info *s;
    char buf[20];
    int m = xm_get_modified();

    if(!sts || !sts->sample.data) {
	gtk_widget_set_sensitive(box_loop, 0);
	gtk_widget_set_sensitive(box_params, 0);
	gtk_widget_set_sensitive(box_sel, 0);
	gtk_widget_hide(fileops_dialogs[DIALOG_SAVE_SAMPLE]);
	gtk_widget_set_sensitive(savebutton, 0);
	sample_display_set_data_16(sampledisplay, NULL, 0, FALSE);
	return;
    }

    s = &sts->sample;

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_volume), sts->volume);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_panning), sts->panning - 128);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_finetune), sts->finetune);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_relnote), sts->relnote);
    gtk_entry_set_text(GTK_ENTRY(gui_cursmpl_name), sts->name);

    if(s->type == ST_MIXER_SAMPLE_TYPE_16)
	sample_display_set_data_16(sampledisplay, s->data, s->length, FALSE);
    else
	sample_display_set_data_8(sampledisplay, s->data, s->length, FALSE);

    if(s->looptype != ST_MIXER_SAMPLE_LOOPTYPE_NONE)
	sample_editor_blocked_set_display_loop(s->loopstart, s->loopend);

    gtk_widget_set_sensitive(box_loop, 1);
    gtk_widget_set_sensitive(box_params, 1);
    gtk_widget_set_sensitive(box_sel, 1);
#ifndef NO_AUDIOFILE
    gtk_widget_set_sensitive(savebutton, 1);
#endif

    sprintf(buf, _("Length: %d"), s->length);
    gtk_label_set(GTK_LABEL(label_slength), buf);

    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(resolution_radio[(s->type / 8) - 1]), TRUE);

    sample_editor_block_loop_spins(1);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(loopradio[s->looptype]), TRUE);
    gui_update_spin_adjustment(GTK_SPIN_BUTTON(spin_loopstart), 0, s->length - 1);
    gui_update_spin_adjustment(GTK_SPIN_BUTTON(spin_loopend), 1, s->length);
    sample_editor_block_loop_spins(0);

    sample_editor_blocked_set_loop_spins(s->loopstart, s->loopend);

    sample_editor_block_sel_spins(1);
    gui_update_spin_adjustment(GTK_SPIN_BUTTON(spin_selstart), -1, s->length - 1);
    gui_update_spin_adjustment(GTK_SPIN_BUTTON(spin_selend), 1, s->length);
    sample_editor_block_sel_spins(0);
    sample_editor_blocked_set_sel_spins(-1, 0);

    xm_set_modified(m);
}

gboolean
sample_editor_handle_keys (int shift,
			   int ctrl,
			   int alt,
			   guint32 keyval,
			   gboolean pressed)
{
    int i;

    if(!pressed)
	return FALSE;

    i = keys_get_key_meaning(keyval, ENCODE_MODIFIERS(shift, ctrl, alt));
    if(i != -1 && KEYS_MEANING_TYPE(i) == KEYS_MEANING_NOTE) {
	i += 12 * gui_get_current_octave_value() + 1;
	if(i < 96 && current_sample != NULL) {
	    gui_play_note_full(tracker->cursor_ch, i, current_sample, 0);
	}
	return TRUE;
    }

    return FALSE;
}

void
sample_editor_set_sample (STSample *s)
{
    current_sample = s;
    sample_editor_update();
}

void
sample_editor_update_mixer_position (double songtime)
{
    audio_mixer_position *p;
    int i;

    if(songtime >= 0.0 && current_sample && (p = time_buffer_get(audio_mixer_position_tb, songtime))) {
	for(i = 0; i < sizeof(p->dump) / sizeof(p->dump[0]); i++) {
	    if(p->dump[i].current_sample == &current_sample->sample) {
		sample_display_set_mixer_position(sampledisplay, p->dump[i].current_position);
		return;
	    }
	}
    }

    sample_display_set_mixer_position(sampledisplay, -1);
}

static void
sample_editor_spin_volume_changed (GtkSpinButton *spin)
{
    g_return_if_fail(current_sample != NULL);

    current_sample->volume = gtk_spin_button_get_value_as_int(spin);
    xm_set_modified(1);
}

static void
sample_editor_spin_panning_changed (GtkSpinButton *spin)
{
    g_return_if_fail(current_sample != NULL);

    current_sample->panning = gtk_spin_button_get_value_as_int(spin) + 128;
    xm_set_modified(1);
}

static void
sample_editor_spin_finetune_changed (GtkSpinButton *spin)
{
    g_return_if_fail(current_sample != NULL);

    current_sample->finetune = gtk_spin_button_get_value_as_int(spin);
    xm_set_modified(1);
}

static void
sample_editor_spin_relnote_changed (GtkSpinButton *spin)
{
    g_return_if_fail(current_sample != NULL);

    current_sample->relnote = gtk_spin_button_get_value_as_int(spin);
    xm_set_modified(1);
}

static void
sample_editor_loop_changed ()
{
    int s = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_loopstart)),
	e = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_loopend));

    g_return_if_fail(current_sample != NULL);
    g_return_if_fail(current_sample->sample.data != NULL);
    g_return_if_fail(current_sample->sample.looptype != ST_MIXER_SAMPLE_LOOPTYPE_NONE);

    if(s != current_sample->sample.loopstart || e != current_sample->sample.loopend) {
	if(e <= s) {
	    e = s + 1;
	    sample_editor_blocked_set_loop_spins(s, e);
	}

	sample_editor_lock_sample();
	current_sample->sample.loopend = e;
	current_sample->sample.loopstart = s;
	sample_editor_unlock_sample();

	sample_editor_blocked_set_display_loop(s, e);
    }

    xm_set_modified(1);
}

static void
sample_editor_spin_selection_changed (void)
{

}

static void
sample_editor_display_loop_changed (SampleDisplay *sample_display,
				    int start,
				    int end)
{
    g_return_if_fail(current_sample != NULL);
    g_return_if_fail(current_sample->sample.data != NULL);
    g_return_if_fail(current_sample->sample.looptype != ST_MIXER_SAMPLE_LOOPTYPE_NONE);
    g_return_if_fail(start < end);

    if(start != current_sample->sample.loopstart || end != current_sample->sample.loopend) {
	sample_editor_blocked_set_loop_spins(start, end);

	sample_editor_lock_sample();
	current_sample->sample.loopend = end;
	current_sample->sample.loopstart = start;
	sample_editor_unlock_sample();
    }

    xm_set_modified(1);
}

static void
sample_editor_reset_selection_clicked (void)
{
    sample_display_set_selection(sampledisplay, -1, 1);
}

static void
sample_editor_display_selection_changed (SampleDisplay *sample_display,
					 int start,
					 int end)
{
    g_return_if_fail(current_sample != NULL);
    g_return_if_fail(current_sample->sample.data != NULL);
    g_return_if_fail(start < end);

    sample_editor_blocked_set_sel_spins(start, end);
}

static void
sample_editor_hscrollbar_changed (GtkAdjustment *adj)
{
    sample_display_set_window(sampledisplay,
			      adj->value,
			      adj->value + sampledisplay->win_length);
}

static void
sample_editor_display_window_changed (SampleDisplay *sample_display,
				      int start,
				      int end)
{
    if(current_sample == NULL)
	return;

    gui_update_range_adjustment(GTK_RANGE(sample_editor_hscrollbar),
				start,
				current_sample->sample.length,
				end - start,
				sample_editor_hscrollbar_changed);
}

static void
sample_editor_loopradio_changed (void)
{
    int n = find_current_toggle(loopradio, 3);

    gtk_widget_set_sensitive(spin_loopstart, n != 0);
    gtk_widget_set_sensitive(spin_loopend, n != 0);
    
    if(current_sample != NULL) {
	if(current_sample->sample.looptype != n) {
	    sample_editor_lock_sample();
	    current_sample->sample.looptype = n;
	    sample_editor_unlock_sample();
	}

	if(n != ST_MIXER_SAMPLE_LOOPTYPE_NONE) {
	    sample_editor_blocked_set_display_loop(current_sample->sample.loopstart, current_sample->sample.loopend);
	} else {
	    sample_editor_blocked_set_display_loop(-1, 1);
	}
    }

    xm_set_modified(1);
}

static void
sample_editor_convert_sample (void *src,
			      void *dst,
			      int srcformat,
			      int dstformat,
			      int count)
{
    gint16 *d16;
    gint8 *d8;

    if(srcformat == dstformat) {
	memcpy(dst, src, count * (srcformat / 8));
    } else {
	if(dstformat == ST_MIXER_SAMPLE_TYPE_16) {
	    /* convert to 16 bit */
	    d16 = dst;
	    d8 = src;
	    while(count--)
		*d16++ = (*d8++ << 8);
	} else {
	    /* convert to 8 bit */
	    d8 = dst;
	    d16 = src;
	    while(count--)
		*d8++ = (*d16++ >> 8);
	}
    }
}

static void
sample_editor_resolution_changed (void)
{
    void *newdata;
    STSample *sts = current_sample;
    st_mixer_sample_info *s;
    int n = find_current_toggle(resolution_radio, 2);

    if(!sts)
	return;

    s = &sts->sample;
    if((s->type / 8) - 1 != n) {
	sample_editor_lock_sample();

	newdata = malloc(s->length * (n + 1));
	if(!newdata)
	    return;

	gui_play_stop();

	sample_editor_convert_sample(s->data,
				     newdata,
				     s->type,
				     (n + 1) * 8,
				     s->length);

	free(s->data);

	s->data = newdata;
	s->type = (n + 1) * 8;

	sample_editor_unlock_sample();
	sample_editor_set_sample(sts);
    }

    xm_set_modified(1);
}

static void
sample_editor_clear_clicked (void)
{
    STInstrument *instr;

    sample_editor_lock_sample();

    st_clean_sample(current_sample, NULL);

    instr = instrument_editor_get_instrument();
    if(st_instrument_num_samples(instr) == 0) {
	instrument_editor_clear_current_instrument();
    } else {
	instrument_editor_update();
	sample_editor_update();
	modinfo_update_all();
    }

    sample_editor_unlock_sample();

    xm_set_modified(1);
}

static void
sample_editor_show_all_clicked (void)
{
    if(current_sample == NULL || current_sample->sample.data == NULL)
	return;
    sample_display_set_window(sampledisplay, 0, current_sample->sample.length);
}

static void
sample_editor_zoom_in_clicked (void)
{
    int ns = sampledisplay->win_start,
	ne = sampledisplay->win_start + sampledisplay->win_length;
    int l;
    
    if(current_sample == NULL || current_sample->sample.data == NULL)
	return;

    l = sampledisplay->win_length / 4;

    ns += l;
    ne -= l;

    if(ne <= ns)
	ne = ns + 1;

    sample_display_set_window(sampledisplay, ns, ne);
}

static void
sample_editor_zoom_out_clicked (void)
{
    int ns = sampledisplay->win_start,
	ne = sampledisplay->win_start + sampledisplay->win_length;
    int l;
    
    if(current_sample == NULL || current_sample->sample.data == NULL)
	return;

    l = sampledisplay->win_length / 2;

    if(ns > l)
	ns -= l;
    else
	ns = 0;

    if(ne <= current_sample->sample.length - l)
	ne += l;
    else
	ne = current_sample->sample.length;

    sample_display_set_window(sampledisplay, ns, ne);
}

static void
sample_editor_zoom_to_selection_clicked (void)
{
    if(current_sample == NULL || current_sample->sample.data == NULL || sampledisplay->sel_start == -1)
	return;
    sample_display_set_window(sampledisplay, sampledisplay->sel_start, sampledisplay->sel_end);
}

static void
sample_editor_copy_cut_common (gboolean copy,
			       gboolean spliceout)
{
    int cutlen, newlen;
    void *newsample;
    STSample *oldsample = current_sample;
    int ss = sampledisplay->sel_start, se, mult;
    
    if(oldsample == NULL || ss == -1)
	return;
    
    se = sampledisplay->sel_end;
    mult = mixer_sample_word_length(&oldsample->sample);

    cutlen = se - ss;
    newlen = oldsample->sample.length - cutlen;

    if(copy) {
	if(copybuffer) {
	    free(copybuffer);
	    copybuffer = NULL;
	}
	copybufferlen = cutlen;
	copybuffertype = oldsample->sample.type;
	copybuffer = malloc(copybufferlen * mult);
	if(!copybuffer) {
	    error_error("Out of memory for copybuffer.\n");
	} else {
	    memcpy(copybuffer,
		   oldsample->sample.data + ss * mult,
		   cutlen * mult);
	}
    }

    if(!spliceout)
	return;

    if(newlen == 0) {
	sample_editor_clear_clicked();
	return;
    }

    newsample = malloc(newlen * mult);
    if(!newsample)
	return;

    sample_editor_lock_sample();

    memcpy(newsample,
	   oldsample->sample.data,
	   ss * mult);
    memcpy(newsample + ss * mult,
	   oldsample->sample.data + se * mult,
	   (oldsample->sample.length - se) * mult);

    free(oldsample->sample.data);

    oldsample->sample.data = newsample;
    oldsample->sample.length = newlen;

    st_sample_fix_loop(oldsample);
    sample_editor_unlock_sample();
    sample_editor_set_sample(oldsample);
    xm_set_modified(1);
}

static void
sample_editor_cut_clicked (void)
{
    sample_editor_copy_cut_common(TRUE, TRUE);
}

static void
sample_editor_remove_clicked (void)
{
    sample_editor_copy_cut_common(FALSE, TRUE);
}

static void
sample_editor_copy_clicked (void)
{
    sample_editor_copy_cut_common(TRUE, FALSE);
}

static void
sample_editor_init_sample (const char *samplename)
{
    STInstrument *instr;

    st_clean_sample(current_sample, NULL);
    
    instr = instrument_editor_get_instrument();
    if(st_instrument_num_samples(instr) == 0) {
	st_clean_instrument(instr, samplename);
    }
	
    st_clean_sample(current_sample, samplename);
    
    current_sample->volume = 64;
    current_sample->finetune = 0;
    current_sample->panning = 128;
    current_sample->relnote = 0;
}

static void
sample_editor_paste_clicked (void)
{
    void *newsample;
    STSample *oldsample = current_sample;
    int ss = sampledisplay->sel_start, mult, newlen;
    int update_ie = 0;

    if(oldsample == NULL || copybuffer == NULL)
	return;

    if(!oldsample->sample.data) {
	/* pasting into empty sample */
	sample_editor_lock_sample();
	sample_editor_init_sample(_("<just pasted>"));
	current_sample->sample.type = copybuffertype;
	sample_editor_unlock_sample();
	ss = 0;
	update_ie = 1;
    } else {
	if(ss == -1)
	    return;
    }

    mult = mixer_sample_word_length(&oldsample->sample);

    newlen = oldsample->sample.length + copybufferlen;

    newsample = malloc(newlen * mult);
    if(!newsample)
	return;

    sample_editor_lock_sample();

    memcpy(newsample,
	   oldsample->sample.data,
	   ss * mult);
    sample_editor_convert_sample(copybuffer,
				 newsample + ss * mult,
				 copybuffertype,
				 oldsample->sample.type,
				 copybufferlen);
    memcpy(newsample + (ss + copybufferlen) * mult,
	   oldsample->sample.data + ss * mult,
	   (oldsample->sample.length - ss) * mult);

    free(oldsample->sample.data);

    oldsample->sample.data = newsample;
    oldsample->sample.length = newlen;

    sample_editor_unlock_sample();
    sample_editor_update();
    if(update_ie)
	instrument_editor_update();
    xm_set_modified(1);
}

static void
sample_editor_modify_wav_sample (STSample *s)
{
    if(s->sample.type == ST_MIXER_SAMPLE_TYPE_16) {
	byteswap_16_array(s->sample.data, s->sample.length);
    } else {
	gint8 *data = s->sample.data;
	int length = s->sample.length;
	while(length) {
	    *data = *data++ + 128;
	    length--;
	}
    }
}

#ifndef NO_AUDIOFILE
static void
sample_editor_load_wav (void)
{
    const gchar *fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_LOAD_SAMPLE]));
    void *sbuf;
    const char *samplename;
    AFfilehandle file;
    AFframecount frameCount;
    int sampleFormat, sampleWidth, channelCount;
    int len;

    gtk_widget_hide(fileops_dialogs[DIALOG_LOAD_SAMPLE]);

    g_return_if_fail(current_sample != NULL);

    file = afOpenFile(fn, "r", NULL);
    if(!file) {
	error_error(_("Can't read sample"));
	return;
    }

    frameCount = afGetFrameCount(file, AF_DEFAULT_TRACK);
    if(frameCount > mixer->max_sample_length) {
	error_warning(_("Sample is too long for current mixer module. Loading anyway."));
    }

    channelCount = afGetChannels(file, AF_DEFAULT_TRACK);
    afGetSampleFormat(file, AF_DEFAULT_TRACK, &sampleFormat, &sampleWidth);

    /* I think audiofile-0.1.7 does this automatically, but I'm not sure */
#if defined(i386) || defined(alpha)
    afSetVirtualByteOrder(file, AF_DEFAULT_TRACK, AF_BYTEORDER_LITTLEENDIAN);
#else
    afSetVirtualByteOrder(file, AF_DEFAULT_TRACK, AF_BYTEORDER_BIGENDIAN);
#endif

    if((sampleWidth != 16 && sampleWidth != 8) || channelCount > 1) {
	error_error(_("Can only handle mono 8 and 16 bit samples"));
	goto errnobuf;
    }

    len = frameCount * (sampleWidth/8) * channelCount;
    if(!(sbuf = malloc(len))) {
	error_error("Out of memory for sample data.");
	goto errnobuf;
    }

    if(frameCount != afReadFrames(file, AF_DEFAULT_TRACK, sbuf, frameCount)) {
	error_error(_("Read error."));
	goto errnodata;
    }

    samplename = strrchr(fn, '/');
    if(!samplename)
	samplename = fn;
    else
	samplename++;

    sample_editor_lock_sample();
    sample_editor_init_sample(samplename);
    current_sample->sample.data = sbuf;
    current_sample->sample.type = sampleWidth;
    current_sample->sample.length = len / mixer_sample_word_length(&current_sample->sample);

    if(sampleWidth == 8)
	sample_editor_modify_wav_sample(current_sample);

    sample_editor_unlock_sample();

    instrument_editor_update();
    sample_editor_update();
    xm_set_modified(1);
    afCloseFile(file);
    return;

  errnodata:
    free(sbuf);
  errnobuf:
    afCloseFile(file);
}

static void
sample_editor_save_wav (void)
{
    const gchar *fn;
    AFfilehandle outfile;
    AFfilesetup outfilesetup;

    gtk_widget_hide(fileops_dialogs[DIALOG_SAVE_SAMPLE]);

    g_return_if_fail(current_sample != NULL);
    g_return_if_fail(current_sample->sample.data != NULL);

    fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_SAMPLE]));

    outfilesetup = afNewFileSetup();
    afInitFileFormat(outfilesetup, AF_FILE_WAVE);
    afInitChannels(outfilesetup, AF_DEFAULT_TRACK, 1);
    afInitSampleFormat(outfilesetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, current_sample->sample.type);
    outfile = afOpenFile(fn, "w", outfilesetup);
    afFreeFileSetup(outfilesetup);

    if(!outfile) {
	error_error(_("Can't open file for writing."));
	return;
    }

    sample_editor_lock_sample();

    if(current_sample->sample.type == 8)
	sample_editor_modify_wav_sample(current_sample);

    afWriteFrames(outfile, AF_DEFAULT_TRACK, current_sample->sample.data, current_sample->sample.length);
    afCloseFile(outfile);

    if(current_sample->sample.type == 8)
	sample_editor_modify_wav_sample(current_sample);

    sample_editor_unlock_sample();
}
#endif /* NO_AUDIOFILE */

/* ============================ Sampling functions coming up -------- */

GtkWidget*
sample_editor_create_sampling_widgets (void)
{
    GtkWidget *box, *thing, *box2;

    box = gtk_vbox_new(FALSE, 2);

    thing = sample_display_new(FALSE);
    gtk_box_pack_start(GTK_BOX(box), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    monitorscope = SAMPLE_DISPLAY(thing);
    
    box2 = gtk_hbox_new(TRUE, 4);
    gtk_box_pack_start(GTK_BOX(box), box2, FALSE, TRUE, 0);
    gtk_widget_show(box2);

    thing = gtk_button_new_with_label(_("OK"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_ok_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(box2), thing, TRUE, TRUE, 0);
    gtk_widget_set_sensitive(thing, 0);
    gtk_widget_show(thing);
    okbutton = thing;

    thing = gtk_button_new_with_label(_("Start sampling"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_start_sampling_clicked), NULL);
    gtk_box_pack_start(GTK_BOX(box2), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    startsamplingbutton = thing;

    thing = gtk_button_new_with_label(_("Cancel"));
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(sample_editor_stop_sampling), NULL);
    gtk_box_pack_start(GTK_BOX(box2), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    cancelbutton = thing;

    return box;
}

static void
sampler_page_enable_widgets (int enable)
{
    gtk_widget_set_sensitive(okbutton, !enable);
    gtk_widget_set_sensitive(startsamplingbutton, enable);
}

static void
sample_editor_monitor_clicked (void)
{
    GtkWidget *mainbox, *thing;

    if(samplingwindow != NULL) {
	gdk_window_raise(samplingwindow->window);
	return;
    }
    
#ifdef USE_GNOME
    samplingwindow = gnome_app_new("SoundTracker", _("Sampling Window"));
#else
    samplingwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(samplingwindow), _("Sampling Window"));
#endif
    gtk_signal_connect (GTK_OBJECT (samplingwindow), "delete_event",
			GTK_SIGNAL_FUNC (sample_editor_stop_sampling), NULL);

    mainbox = gtk_vbox_new(FALSE, 2);
    gtk_container_border_width(GTK_CONTAINER(mainbox), 4);
#ifdef USE_GNOME
    gnome_app_set_contents(GNOME_APP(samplingwindow), mainbox);
#else
    gtk_container_add(GTK_CONTAINER(samplingwindow), mainbox);
#endif
    gtk_widget_show(mainbox);

    thing = sample_editor_create_sampling_widgets();
    gtk_box_pack_start(GTK_BOX(mainbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);

    sampler_page_enable_widgets(TRUE);

    recordbufs = NULL;
    sampling = 0;
    recordedlen = 0;
    currentoffs = recordbuflen;
    current = NULL;

    gtk_widget_show (samplingwindow);

    if(!sampling_driver->common.open(sampling_driver_object)) {
	sample_editor_stop_sampling();
    }
}

void
sample_editor_sampled (void *dest,
		       guint32 count,
		       int mixfreq,
		       int mixformat)
{
    int x;

    g_assert(mixformat == ST_MIXER_FORMAT_S16_LE);

    sample_display_set_data_16(monitorscope, dest, count, FALSE);

    while(sampling && count > 0) {
	if(currentoffs == recordbuflen) {
	    struct recordbuf *newbuf = malloc(sizeof(struct recordbuf) + recordbuflen * 2);
	    if(!newbuf) {
		error_error("Out of memory while sampling!");
		sampling = 0;
		break;
	    }
	    newbuf->next = NULL;
	    newbuf->length = 0;
	    currentoffs = 0;
	    if(!recordbufs)
		recordbufs = newbuf;
	    else
		current->next = newbuf;
	    current = newbuf;
	}

	x = MIN(count, recordbuflen - currentoffs);
	memcpy(current->data + currentoffs, dest, x * 2);
	dest += x * 2;
	count -= x;
	current->length += x;
	currentoffs += x;
	recordedlen += x;
    }
}

void
sample_editor_stop_sampling (void)
{
    struct recordbuf *r, *r2;

    if(!samplingwindow) {
	return;
    }

    sampling_driver->common.release(sampling_driver_object);

    gtk_widget_destroy(samplingwindow);
    samplingwindow = NULL;

    /* clear the recorded sample */
    for(r = recordbufs; r; r = r2) {
	r2 = r->next;
	free(r);
    }
}

static void
sample_editor_ok_clicked (void)
{
    STInstrument *instr;
    struct recordbuf *r, *r2;
    gint16 *sbuf;
    char *samplename = _("<just sampled>");

    sampling_driver->common.release(sampling_driver_object);

    gtk_widget_destroy(samplingwindow);
    samplingwindow = NULL;

    g_return_if_fail(current_sample != NULL);

    sample_editor_lock_sample();

    st_clean_sample(current_sample, NULL);

    instr = instrument_editor_get_instrument();
    if(st_instrument_num_samples(instr) == 0)
	st_clean_instrument(instr, samplename);

    st_clean_sample(current_sample, samplename);
    
    sbuf = malloc(recordedlen * 2);
    current_sample->sample.data = sbuf;
    
    for(r = recordbufs; r; r = r2) {
	r2 = r->next;
	memcpy(sbuf, r->data, r->length * 2);
	sbuf += r->length;
	free(r);
    }

    if(recordedlen > mixer->max_sample_length) {
	error_warning(_("Recorded sample is too long for current mixer module. Using it anyway."));
    }

    current_sample->sample.length = recordedlen;
    current_sample->volume = 64;
    current_sample->finetune = 0;
    current_sample->panning = 128;
    current_sample->relnote = 0;
    current_sample->sample.type = 16;

    sample_editor_unlock_sample();

    instrument_editor_update();
    sample_editor_update();
    xm_set_modified(1);
}

static void
sample_editor_start_sampling_clicked (void)
{
    sampler_page_enable_widgets(FALSE);
    sampling = 1;
}

/* ==================== VOLUME RAMPING DIALOG =================== */

static void
sample_editor_open_volume_ramp_dialog (void)
{
    GtkWidget *mainbox, *box1, *thing;
    int i;
    const char *labels1[] = {
	_("Normalize"),
	_("Execute"),
	_("Close")
    };

    if(volrampwindow != NULL) {
	gdk_window_raise(volrampwindow->window);
	return;
    }
    
#ifdef USE_GNOME
    volrampwindow = gnome_app_new("SoundTracker", _("Volume Ramping"));
#else
    volrampwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(volrampwindow), _("Volume Ramping"));
#endif
    gtk_signal_connect (GTK_OBJECT (volrampwindow), "delete_event",
			GTK_SIGNAL_FUNC (sample_editor_close_volume_ramp_dialog), NULL);

//    gtk_window_set_modal(GTK_WINDOW(volrampwindow), TRUE);
    gtk_window_set_transient_for(GTK_WINDOW(volrampwindow), GTK_WINDOW(mainwindow));

    mainbox = gtk_vbox_new(FALSE, 2);
    gtk_container_border_width(GTK_CONTAINER(mainbox), 4);
#ifdef USE_GNOME
    gnome_app_set_contents(GNOME_APP(volrampwindow), mainbox);
#else
    gtk_container_add(GTK_CONTAINER(volrampwindow), mainbox);
#endif
    gtk_widget_show(mainbox);

    thing = gtk_label_new(_("Perform linear volume fade on Selection"));
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);

    thing = gtk_hseparator_new();
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);

    box1 = gtk_hbox_new(FALSE, 4);
    gtk_widget_show(box1);
    gtk_box_pack_start(GTK_BOX(mainbox), box1, FALSE, TRUE, 0);

    gui_put_labelled_spin_button(box1, _("Left [%]:"), 0, 1000, &sample_editor_volramp_spin_w[0], NULL, NULL);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sample_editor_volramp_spin_w[0]), 100);

    add_empty_hbox(box1);

    gui_put_labelled_spin_button(box1, _("Right [%]:"), 0, 1000, &sample_editor_volramp_spin_w[1], NULL, NULL);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sample_editor_volramp_spin_w[1]), 100);

    box1 = gtk_hbox_new(FALSE, 4);
    gtk_widget_show(box1);
    gtk_box_pack_start(GTK_BOX(mainbox), box1, FALSE, TRUE, 0);

    for(i = 0; i < 3; i++) {
	thing = gtk_button_new_with_label(labels1[i]);
	gtk_widget_show(thing);
	gtk_box_pack_start(GTK_BOX(box1), thing, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(thing), "clicked",
			   GTK_SIGNAL_FUNC(sample_editor_perform_ramp), (gpointer)i);
    }

    gtk_widget_show (volrampwindow);
}

static void
sample_editor_close_volume_ramp_dialog (void)
{
    gtk_widget_destroy(volrampwindow);
    volrampwindow = NULL;
}

static void
sample_editor_perform_ramp (GtkWidget *w,
			    gpointer data)
{
    int action = (int)data;
    double left, right;
    int ss = sampledisplay->sel_start, se = sampledisplay->sel_end;
    int mult, i;

    if(action == 2 || !current_sample || ss == -1) {
	sample_editor_close_volume_ramp_dialog();
	return;
    }

    mult = mixer_sample_word_length(&current_sample->sample);

    if(action == 0) {
	// Find maximum amplitude
	int m;
	if(mult == 1) {
	    gint8 *p = current_sample->sample.data;
	    int q;
	    p += ss;
	    for(i = 0, m = 0; i < se - ss; i++) {
		q = *p++;
		q = ABS(q);
		if(q > m)
		    m = q;
	    }
	    left = right = (double)0x7f / m;
	} else {
	    gint16 *p = current_sample->sample.data;
	    int q;
	    p += ss;
	    for(i = 0, m = 0; i < se - ss; i++) {
		q = *p++;
		q = ABS(q);
		if(q > m)
		    m = q;
	    }
	    left = right = (double)0x7fff / m;
	}
    } else {
	left = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(sample_editor_volramp_spin_w[0])) / 100;
	right = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(sample_editor_volramp_spin_w[1])) / 100;
    }

    // Now perform the actual operation
    sample_editor_lock_sample();

    if(mult == 1) {
	gint8 *p = current_sample->sample.data;
	p += ss;
	for(i = 0; i < se - ss; i++) {
	    double q = *p;
	    q *= left + i * (right - left) / (se - ss);
	    *p++ = CLAMP((int)q, -128, +127);
	}
    } else {
	gint16 *p = current_sample->sample.data;
	p += ss;
	for(i = 0; i < se - ss; i++) {
	    double q = *p;
	    q *= left + i * (right - left) / (se - ss);
	    *p++ = CLAMP((int)q, -32768, +32767);
	}
    }

    xm_set_modified(1);
    sample_editor_unlock_sample();
    sample_editor_update();
    sample_display_set_selection(sampledisplay, ss, se);
}
