
/*
 * The Real SoundTracker - Audio configuration dialog
 *
 * Copyright (C) 1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <gtk/gtk.h>
#ifdef USE_GNOME
#include <gnome.h>
#endif

#include "i18n.h"
#include "audioconfig.h"
#include "gui-subs.h"
#include "audio.h"
#include "sample-editor.h"
#include "driver.h"
#include "preferences.h"
#include "sample-editor.h"
#include "gui.h"

GList *drivers[2] = { NULL, NULL };

static GtkWidget *configwindow = NULL;
static GtkWidget *cw_clist, *cw_hbox;
static int cw_currentobject = -1;
static GtkWidget *driverwidget = NULL;

typedef struct audio_object {
    const char *title;
    const char *shorttitle;
    int type;
    void **driver_object;
    void **driver;
} audio_object;

static audio_object audio_objects[] = {
    { N_("Playback Output"),
      "playback",
      DRIVER_OUTPUT,
      &playback_driver_object,
      (void**)&playback_driver
    },
    { N_("Editing Output"),
      "editing",
      DRIVER_OUTPUT,
      &editing_driver_object,
      (void**)&editing_driver
    },
    { N_("Sampling"),
      "sampling",
      DRIVER_INPUT,
      &sampling_driver_object,
      (void**)&sampling_driver
    }
};

#define NUM_AUDIO_OBJECTS (sizeof(audio_objects) / sizeof(audio_objects[0]))

static void
audioconfig_clist_select (GtkCList *list,
			  gint row,
			  gint column)
{
    if(cw_currentobject != -1) {
	audio_object *object = &audio_objects[cw_currentobject];
	st_driver *old_driver = *object->driver;
	void *old_driver_object = *object->driver_object;
	st_driver *new_driver = g_list_nth_data(drivers[object->type], row);
	GtkWidget *new_driverwidget;

	if(new_driver != old_driver) {
	    // stop playing and sampling here
	    sample_editor_stop_sampling();
	    gui_play_stop();

	    // get new driver object
	    *object->driver_object = new_driver->new();
	    *object->driver = new_driver;
	}

	new_driverwidget = new_driver->getwidget(*object->driver_object);

	if(new_driverwidget != driverwidget) {
	    if(driverwidget) {
		gtk_container_remove(GTK_CONTAINER(cw_hbox), driverwidget);
	    }
	    driverwidget = new_driverwidget;
	    gtk_widget_show(driverwidget);
	    /* we don't want the widget to be destroyed upon removal... */
	    gtk_object_ref(GTK_OBJECT(driverwidget));
	    gtk_box_pack_start(GTK_BOX(cw_hbox), driverwidget, TRUE, FALSE, 0);
	}

	if(new_driver != old_driver) {
	    // free old driver object
	    old_driver->destroy(old_driver_object);
	}
    } else {
	// The CList is being updated
    }
}

static void
audioconfig_object_changed (void *a,
			    void *b)
{
    unsigned n = (int)b;
    GList *l;
    gchar *insertbuf[2] = { NULL };
    int i, active = -1;

    g_assert(n < NUM_AUDIO_OBJECTS);

    if(n == cw_currentobject)
	return;

    cw_currentobject = -1; // disable clist select callback

    gtk_clist_freeze(GTK_CLIST(cw_clist));
    gtk_clist_clear(GTK_CLIST(cw_clist));
    for(i = 0, l = drivers[audio_objects[n].type]; l; i++, l = l->next) {
	insertbuf[0] = *((gchar **)l->data);
	if(l->data == *audio_objects[n].driver)
	    active = i;
	gtk_clist_append(GTK_CLIST(cw_clist), insertbuf);
    }
    gtk_clist_thaw(GTK_CLIST(cw_clist));

    g_assert(active != -1);

    // Now update the GUI
    cw_currentobject = n;
    if(driverwidget) {
	gtk_container_remove(GTK_CONTAINER(cw_hbox), driverwidget);
	driverwidget = NULL;
    }
    gtk_clist_select_row(GTK_CLIST(cw_clist), active, 0);
}

static void
audioconfig_close_requested (void)
{
    if(driverwidget) {
	gtk_container_remove(GTK_CONTAINER(cw_hbox), driverwidget);
	gtk_widget_hide(driverwidget);
	driverwidget = NULL;
    }
    gtk_widget_destroy(configwindow);
    configwindow = NULL;
}

void
audioconfig_dialog (void)
{
    GtkWidget *mainbox, *thing, *box1;
    OptionMenuItem menu1[NUM_AUDIO_OBJECTS];
    static gchar *listtitles[1];
    int i;

    listtitles[0] = gettext("Driver Module");

    if(configwindow != NULL) {
	gdk_window_raise(configwindow->window);
	return;
    }
    
#ifdef USE_GNOME
    configwindow = gnome_app_new("SoundTracker", _("Audio Configuration"));
#else
    configwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(configwindow), _("Audio Configuration"));
#endif
    gtk_signal_connect (GTK_OBJECT (configwindow), "delete_event",
			GTK_SIGNAL_FUNC (audioconfig_close_requested), NULL);

    mainbox = gtk_vbox_new(FALSE, 2);
    gtk_container_border_width(GTK_CONTAINER(mainbox), 4);
#ifdef USE_GNOME
    gnome_app_set_contents(GNOME_APP(configwindow), mainbox);
#else
    gtk_container_add(GTK_CONTAINER(configwindow), mainbox);
#endif
    gtk_widget_show(mainbox);


    // Key Group Selector
    for(i = 0; i < NUM_AUDIO_OBJECTS; i++) {
	menu1[i].name = gettext(audio_objects[i].title);
	menu1[i].func = audioconfig_object_changed;
    }
    thing = build_option_menu(menu1, NUM_AUDIO_OBJECTS, 0);
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, FALSE, 0);

    cw_hbox = box1 = gtk_hbox_new(FALSE, 4);
    gtk_widget_show(box1);
    gtk_box_pack_start(GTK_BOX(mainbox), box1, TRUE, TRUE, 0);

    // List at the left side of the window
    thing = get_clist_in_scrolled_window(1, listtitles, box1);
    gtk_clist_set_selection_mode(GTK_CLIST(thing), GTK_SELECTION_BROWSE);
    gtk_clist_column_titles_passive(GTK_CLIST(thing));
    gtk_clist_set_column_justification(GTK_CLIST(thing), 0, GTK_JUSTIFY_LEFT);
    gtk_widget_set_usize(thing, 200, 50);
    gtk_signal_connect_after(GTK_OBJECT(thing), "select_row",
			     GTK_SIGNAL_FUNC(audioconfig_clist_select), NULL);
    cw_clist = thing;

    audioconfig_object_changed(NULL, (void*)0);

    gtk_widget_show (configwindow);
}

void
audioconfig_load_config (void)
{
    char buf[256];
    FILE *f;
    GList *l;
    int i;

    sprintf(buf, "%s/audio-objects", prefs_get_prefsdir());

    f = fopen(buf, "rb");
    if(f) {
	for(i = 0; i < NUM_AUDIO_OBJECTS; i++) {
	    if(prefs_get_string(f, audio_objects[i].shorttitle, buf)) {
		for(l = drivers[audio_objects[i].type]; l; l = l->next) {
		    if(!strcmp(*((gchar **)l->data), buf)) {
			*audio_objects[i].driver = l->data;
			break;
		    }
		}
	    }
	}
    }

    for(i = 0; i < NUM_AUDIO_OBJECTS; i++) {
	st_driver *d = *audio_objects[i].driver;

	if(!d) {
	    // set default driver if none has been configured
	    d = *audio_objects[i].driver = drivers[audio_objects[i].type]->data;
	}

	// create driver instance
	*audio_objects[i].driver_object = d->new();

	// load settings
	if(d->loadsettings) {
	    sprintf(buf, "%s/audio-object-%s", prefs_get_prefsdir(), audio_objects[i].shorttitle);
	    f = fopen(buf, "rb");
	    if(f) {
		d->loadsettings(*audio_objects[i].driver_object, f);
		fclose(f);
	    } else {
		// set default values
		// will use module defaults for now
	    }
	}
    }
}

void
audioconfig_save_config (void)
{
    char buf[256];
    FILE *f;
    int i;

    prefs_check_prefs_dir();
    sprintf(buf, "%s/audio-objects", prefs_get_prefsdir());

    f = fopen(buf, "wb");
    if(!f)
	return;

    // Write the driver module names
    for(i = 0; i < NUM_AUDIO_OBJECTS; i++) {
	prefs_put_string(f, audio_objects[i].shorttitle,
			 ((st_driver*)*(audio_objects[i].driver))->name);
    }

    fclose(f);

    /* Write the driver module's configurations in extra files */
    for(i = 0; i < NUM_AUDIO_OBJECTS; i++) {
	gboolean (*savesettings)(void *, FILE *) = ((st_driver*)*(audio_objects[i].driver))->savesettings;

	if(savesettings) {
	    sprintf(buf, "%s/audio-object-%s", prefs_get_prefsdir(), audio_objects[i].shorttitle);
	    f = fopen(buf, "wb");
	    if(f) {
		savesettings(*audio_objects[i].driver_object, f);
		fclose(f);
	    }
	}
    }

    return;
}
