;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;      hexmem.asm - memory dump in hex
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;
;   1.0 - First internal version
;   1.1 - GOTO function implemented
;   1.2 - Various bugs fixed
;   1.3 - ALPHA mode implemented
;

#include "ti-85.h"

.org

MEMOFFSET   = TEXT_MEM    ; Word
FLAGS       = TEXT_MEM+2  ; Byte

    .db "MemMaster v1.5",0

    ROM_CALL(CLEARLCD)

    ld  de, 0
    ld  (MEMOFFSET), de
    
Main:
    ld  de, (MEMOFFSET)
    ld  a, (FLAGS)
    
    bit 0, a
    CALL_NZ(DumpMemPageA) 
    CALL_Z(DumpMemPage)

KeyLoop:
    halt
    call GET_KEY
    or  a
    JUMP_Z(KeyLoop)

    cp K_EXIT
    JUMP_Z(Quit)

    cp K_F1
    JUMP_Z(Goto)

    cp K_F2
    JUMP_Z(ToggleAlpha)

    cp K_F5
    JUMP_Z(Info)

    cp K_UP
    JUMP_Z(BackRow)

    cp K_DOWN
    JUMP_Z(NextRow)

    cp K_MINUS    
    JUMP_Z(BackPage)

    cp K_PLUS
    JUMP_Z(NextPage)
    
    jr Main

Quit:
    ret

;; ***** INFO PAGE *****
Info:
    ROM_CALL(CLEARLCD)
    
    ld  hl, (PROGRAM_ADDR)
    ld  de, InfoString
    add hl, de
    
    ld  a, 5
    ld  ($800C), a
    ld  a, 0
    ld  ($800D), a
    ROM_CALL(D_ZT_STR)
InfoWait:
    call get_key
    or a
    jr z,InfoWait
                 
    JUMP_(Main)

;; ***** GOTO ADDR *****
Goto:
    ld  a, 2
    ld  ($800C), a
    ld  a, 4
    ld  ($800D), a
    
    ld  hl, (PROGRAM_ADDR)
    ld  de, GotoString
    add hl,de
    ROM_CALL(D_ZT_STR)

    ld  a, 3
    ld  ($800C), a
    ld  a, 4
    ld  ($800D), a
    ROM_CALL(D_ZT_STR)
                 
    ld  a, 4
    ld  ($800C), a
    ld  a, 4
    ld  ($800D), a
    ROM_CALL(D_ZT_STR)
                 
    ld  hl, $0B03
    CALL_(Input16Hex)
    jr  c, Canceled         ; Input was CANCELED

    ld  (MEMOFFSET), de
Canceled:
    JUMP_(Main)

;; ***** BACK ROW *****
BackRow:
    ld  a, (FLAGS)
    bit 0, a
    jr  z, NoAlphaBR

    ld  de, 16
    ld  hl, (MEMOFFSET)
    sbc hl, de    
    ld  (MEMOFFSET), hl
    JUMP_(Main)
NoAlphaBR: 
    ld  de, 8
    ld  hl, (MEMOFFSET)
    sbc hl, de    
    ld  (MEMOFFSET), hl
    JUMP_(Main)

;; ***** NEXT ROW *****
NextRow:
    ld  a, (FLAGS)
    bit 0, a
    jr  z, NoAlphaNR

    ld  de, 16
    ld  hl, (MEMOFFSET)
    add hl, de
    ld  (MEMOFFSET), hl
    JUMP_(Main)
NoAlphaNR:
    ld  de, 8
    ld  hl, (MEMOFFSET)
    add hl, de
    ld  (MEMOFFSET), hl
    JUMP_(Main)

;; ***** BACK PAGE *****
BackPage:
    ld  a, (FLAGS)
    bit 0, a
    jr  z, NoAlphaBP
    
    ld  de, 128
    ld  hl, (MEMOFFSET)
    sbc hl, de    
    ld  (MEMOFFSET), hl
    JUMP_(Main)
NoAlphaBP:
    ld  de, 64
    ld  hl, (MEMOFFSET)
    sbc hl, de    
    ld  (MEMOFFSET), hl
    JUMP_(Main)

;; ***** NEXT PAGE *****
NextPage:
    ld  a, (FLAGS)
    bit 0, a
    jr  z, NoAlphaNP

    ld  de, 128
    ld  hl, (MEMOFFSET)
    add hl, de    
    ld  (MEMOFFSET), hl
    JUMP_(Main)
NoAlphaNP:
    ld  de, 64
    ld  hl, (MEMOFFSET)
    add hl, de    
    ld  (MEMOFFSET), hl
    JUMP_(Main)

;; ***** TOGGLE ALPHA *****
ToggleAlpha:
    ROM_CALL(CLEARLCD)    
    ld  a, (FLAGS)
    ld  b, 1
    xor b
    ld  (FLAGS), a
    JUMP_(Main)

;       Dump Memory Page - Hex
;       """"""""""""""""""""""
; Inputs:   DE - memory offset
;
; Traps:    Modifyes textmemory if the 
;           textwrite bit are setup that way
;           
; Outputs:  Writes out an 8x8 matrix of memory data
;           in hex.

DumpMemPage:
    push af
    push bc
    push de
    push hl
    
    ld  b,  8           ; Eight rows of memory

WrtMemRow:
    push bc

    ld  h, $00
    ld  a, 8
    sub b
    ld  l, a
    ld  ($800C), hl

    ld  a, d
    CALL_(Disp8Hex)
    ld  a, e
    CALL_(Disp8Hex)

    ld  a, ':'
    ROM_CALL(TX_CHARPUT)

    ld  b,  8
WrtMemCol:
    push bc

    ld  a, (de)         ; Move data at offset into acc.
    
    CALL_(Disp8Hex)     ; Write!

    inc de              ; Increase offset

    pop bc
    djnz WrtMemCol

    pop bc
    djnz WrtMemRow

    pop hl
    pop de
    pop bc
    pop af
    ret

;       Dump Memory Page - Alpha
;       """"""""""""""""""""""""
; Inputs:   DE - memory offset
;
; Traps:    Modifyes textmemory if the 
;           textwrite bit are setup that way
;           
; Outputs:  Writes out an 8x8 matrix of memory data
;           in hex.

DumpMemPageA:
    push af
    push bc
    push de
    push hl
    
    ld  b,  8           ; Eight rows of memory

WrtMemRowA:
    push bc

    ld  h, $00
    ld  a, 8
    sub b
    ld  l, a
    ld  ($800C), hl

    ld  a, d
    CALL_(Disp8Hex)
    ld  a, e
    CALL_(Disp8Hex)
    
    ld  a,  ':'
    ROM_CALL(TX_CHARPUT)

    ld  b,  16
WrtMemColA:
    push bc

    ld  a, (de)         ; Move data at offset into acc.
    
    cp  $E1
    jr  nc, CtrlChar

    ROM_CALL(TR_CHARPUT)
    jr  OkCharPut
CtrlChar:
    ld  a, $20
    ROM_CALL(TR_CHARPUT)

OkCharPut:
    ld  a, ($800D)
    inc a
    ld  ($800D), a
    
    inc de              ; Increase offset

    pop bc
    djnz WrtMemColA

    pop bc
    djnz WrtMemRowA

    pop hl
    pop de
    pop bc
    pop af
    ret


;         Write 8bit Hex Number         
;         """""""""""""""""""""         
;  Inputs:  a - number to display  
;           + memory and bits for normal
;             output
;                                  
;  Traps:   None        
;                                  
;  Outputs: $800C updated with new    
;           cursor pos. 
;           C contains number printed           
Disp8Hex:    
    push    af
    push    bc
    push    de
    push    hl    
    
    ld  c, a                ; keycode to acc.
   
    and $F0                 ;
    srl a                   ; Mask out 
    srl a                   ; upper bits
    srl a                   ; only
    srl a                   ;
    
    add a, $30              ; add 30 [ numerical offset ] 
    ld  b, a                ;

    cp  $3A                 ;
    jr  c, Low16            ; if(a<$3A) goto 16Low (i.e value is numerical)

    add a, 7                ; add 8 skip to alphas
    ld  b, a                ;

Low16:
    ld  a, b                ; PRINT FIRST NUMBER
    ROM_CALL(TR_CHARPUT)    ;
    ld  a, ($800D)
    inc a
    ld  ($800D), a

Hex8:
    ld  a, c                ; restore number
    
    and $0F
    
    add a, $30              ; $30 = numbase
    ld  b, a                ; add

    cp  $3A                 ; 
    jr  c, Low8             ; if(a<$3A) goto 8Low

    add a, 7                ;
    ld  b, a                ; add alpha offset

Low8:
    ld  a, b                ; PRINT SECOND NUMBER
    ROM_CALL(TR_CHARPUT)    ; 
    ld  a, ($800D)
    inc a
    ld  ($800D), a

    pop hl
    pop de
    pop bc
    pop af

    ret


;       Input 16 Bit Hex Number
;       """""""""""""""""""""""
; Inputs:   hl - Cursor position for output
;              + Memory bits for normal output
;
; Traps:    None
;
; Outputs:  de - 16bit number from user
;              - cursor pos updated with next column&line
;              - flags except ZERO are unaffected
;                (Zero flag set if CANCELED, res if all is OK)


Input16Hex:
    push af
    push bc
    push hl

    ld  bc, $0000
    ld  de, $0000
    
    ld  (CURSOR_ROW), hl
    
    ld  a, d
    CALL_(Disp8Hex)
    ld  a, e   
    CALL_(Disp8Hex)


GotoGetK:
    ld  (CURSOR_ROW), hl
    
    CALL_(GetLetter)
    cp  127
    jr  z, Cancel
    cp  128
    jr  z, Exit
        
    ld  b, a

    bit 0, c
    jr  nz, LoNibble

    ld  a, b
    add a,a
    add a,a
    add a,a
    add a,a
    ld  b, a

LoNibble:
    bit 1, c
    jr  nz, LoByte

    ld  a, d 
    or  b
    ld  d, a
    jr  Show

LoByte:
    ld  a, e
    or  b
    ld  e, a
   
Show:
    inc c
    bit 2, c
    jr  nz, Exit

    ld  a, d
    CALL_(Disp8Hex)
    ld  a, e   
    CALL_(Disp8Hex)

    jr  GotoGetK

Cancel:
    pop hl
    pop bc
    pop af
    scf
    ret

Exit:
    pop hl
    pop bc
    pop af
    or a
    ret 

;+-------
;| Get One Letter & Translate
;| This routine and LetterTable from Magnus Hagander's Texanoid Game
;+-------
GetLetter:
   push hl
   push de
   push bc
   ld   hl,(PROGRAM_ADDR)
   ld   de,LetterTable-1 ;Point to one before, since we always inc once
   add  hl,de                   ;HL now points right

ConvertLoop:
   push hl
   CALL Get_Key                 ;ACC holds key
   pop  hl

   cp   0
   jr   z,ConvertLoop

   push hl                      ;save away...
   ld   b,a
   ld   c,0

ConversionLoop:
   inc  hl
   djnz ConversionLoop          ;After this, we have the correct offset...
   ld   a,(HL)
   pop  hl
   cp   255
   jr   z,ConvertLoop           ;Invalid key

Shazam:

   pop  bc
   pop  de
   pop  hl

   ret                          ;Value is in acc

LetterTable:                                ; For converting keys into letters
   .db 255,255,255,255
   .db 255,255,255,255
   .db 128,255,255,255
   .db 255,14 ,255,255
   .db 255,3  ,6  ,9 
   .db 255,13 ,255,255
   .db 255,2  ,5  ,8
   .db 255,12 ,255,255
   .db 0  ,1  ,4  ,7  
   .db 255,11 ,255,255
   .db 255,255,255,255
   .db 15 ,10 ,255,255
   .db 255,255,255,255
   .db 255,255,127,255

GotoString:
   .db "+-----------+",0
   .db $7c," Goto:     ",$7c,0
   .db "+-----------+",0

InfoString:
   .db  " F1     - Goto       "
   .db  " F2     - Alpha/Hex  "
   .db  " EXIT   - Quit       "
   .db  " UP/DWN - Move row   "
   .db  " +/-    - Move page  ",0
.end
