/*===========================================================================*/
/*                                                                           */
/* File    : WINHELP.C                                                       */
/*                                                                           */
/* Purpose : WinHelp() function which can read Microsoft QuickHelp files     */
/*                                                                           */
/* 22-Feb-93 Wolfgang Lorenz 100112,220                                      */
/* - removing menu and button line                                           */
/* - adding colon commands: title, freeze, previous and next                 */
/* - adding !B command                                                       */
/* - changing the processing of vertical scroll bar                          */
/* - changing the default colors to those in QuickHelp                       */
/* - adding the WM_ERASEBKGND message for non-white backgrounds              */
/* - adding HELP_CONTEXT via topic string "#xxx"                             */
/* - adding a "last topic" call to WinHelp() by VK_BACK                      */
/* - left indenting the screen one space                                     */
/* - making the processing of <Tab> more like QuickHelp                      */
/* - letting the marking of words recognize umlauts                          */
/* - translating messages into German                                        */
/* - adding demonstration files HELPTEST.CPP and HELPTEST.HLP                */
/*                                                                           */
/* (C) Copyright 1991-1993 Marc Adler/Magma Systems     All Rights Reserved  */
/*===========================================================================*/

#define LEFT_INDENT 1
#define ERROR_BEEP
// #define GERMAN

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <io.h>
#include <malloc.h>
#include <memory.h>
#include <dos.h>
#include "window.h"
#include "winhelp.h"
#include "winternl.h"

#undef isalnum
#define isalnum IsIntlAlNum
int IsIntlAlNum(int iChar) {
  if (iChar < 128) {
    if (iChar >= 'a' && iChar <= 'z') return 1;
    if (iChar >= 'A' && iChar <= 'Z') return 1;
    if (iChar >= '0' && iChar <= '9') return 1;
    if (iChar == '_' || iChar == '-' || iChar == '#') return 1;
  } else {
    if (iChar <= 154) return 1;
    if (iChar >= 160 && iChar <= 165) return 1;
    if (iChar >= 224 && iChar <= 235) return 1;
  }
  return 0;
}
static char acError[] =
  #if GERMAN
    "Fehler";
  #else
    "Error";
  #endif
#define HELPERR_NO_ACTION -3

extern WINDOW *FAR PASCAL WID_TO_WIN(HWND);


/*
  Defined constants
*/
#define MAXHANDLES    1024
#define MAXFILES      32
#define PATHLEN       128
#define MAXHELPWIDTH  76

#ifdef MEWEL_GUI
#define CY_BUTTON_FROM_BOTTOM  4
#define CY_BUTTONROW           32
#else
#define CY_BUTTON_FROM_BOTTOM  0
#define CY_BUTTONROW           0
#endif

/*
  Structures
*/
struct _HelpFile
{
  nc   ncInitial;
  nc   ncCurr;
  char szTopic[256];
  int  nLines;
  int  iTopLine;
  char **pLines;
  PB   pUncompressedTopic;
  HWND hwnd;
  BOOL bIsDlg;
  hotspot hs;       // line and column of selected link in topic
  int  ifreeze;     // # of top lines not to scroll
  char acPrev[40];  // previous topic or ""
  char acNext[40];  // next topic or ""
  short	MaxWidth;   // Maximum width data displayed
  short maxLines;
  short	xChr;       // Average char width
  short yChr;
  short	xWnd;       // Width of the child window
  short yWnd;       // Height of the child window
  short HrzScrPos;  // Horizontal scrollbar thumb
  short	HrzScrMax;  // Horizontal scroll maximum
  HWND	hOldWnd;    // Window that has focus coming in
};


/*
  Variables
*/
HWND hwndHelp;
static HWND hwndStatus = NULLHWND;
static int  LastKeyPressed = 0;
static struct _HelpFile hf;
static char szTopicToFind[256];
static char szCurrHelpFile[128];  /* The name of the current help file */
static int  iMonochrome;
static unsigned char PhysAttrs[9][2] = {
#if defined(MEWEL_GUI)
  { 0xF0, 0x0F },			      /* 0: normal text	      */
  { 0x0F, 0xF0 },			      /* 1: bold	      */
  { 0xF1, 0xF0 },			      /* 2: italics	      */
  { 0x09, 0xF0 },			      /* 3: bold italics      */
  { 0xF4, 0x8F },			      /* 4: underline	      */
  { 0x0C, 0x70 },			      /* 5: bold ul	      */
  { 0x75, 0x70 },			      /* 6: italics ul	      */
  { 0x0D, 0x70 },			      /* 7: bold italics ul   */
  { 0xF7, 0xF7 },                             /* 8: hot spot          */
#else
  { 0x07, 0x07 },                             /* 0: normal text       */
  { 0x0F, 0x0F },                             /* 1: bold              */
  { 0x02, 0x0F },                             /* 2: italics           */
  { 0x03, 0x07 },                             /* 3: bold italics      */
  { 0x0C, 0x0F },                             /* 4: underline         */
  { 0x07, 0x0F },                             /* 5: bold ul           */
  { 0x3F, 0x07 },                             /* 6: italics ul        */
  { 0x00, 0x00 },                             /* 7: bold italics ul   */
  { 0x70, 0x70 },                             /* 8: hot spot          */
#endif
};

/*
  Prototypes
*/
static HWND PASCAL _HelpCreateSearchDialog(HWND);
static HWND PASCAL WinCreateHelpWindow(HWND);
static long PASCAL HelpWndProc(HWND, WORD, WORD, LONG);
static int  PASCAL TopicSearchDialogProc(HWND, WORD, WORD, DWORD);
static nc   PASCAL HelpOpenFile(LPCSTR);
static VOID PASCAL CloseHelpFile(nc);
static VOID PASCAL freeLinePointers(void);
static nc   PASCAL HelpFindContext(nc, char *);
static int  PASCAL HelpDisplayContext(HWND, nc);
static VOID PASCAL HelpRefreshWindow(HWND);
static VOID PASCAL GetWordUnderCursor(int, int, char *);
static int  PASCAL DoOpenHelpFile(LPCSTR);
static int  PASCAL HelpProcessKeys(HWND, int);
static int  PASCAL HelpDisplayLineWithAttrs(HWND, HDC, int, int, char *);
static int  PASCAL HelpTraverseLinks(HWND, int);
static BOOL PASCAL HelpDisplayCurrHotlink(HWND, BOOL);
static BOOL PASCAL HelpInitClass(void);



/****************************************************************************/
/*                                                                          */
/* Function : WinHelp()                                                     */
/*                                                                          */
/* Purpose  : Public entry point to the help engine.                        */
/*                                                                          */
/* Returns  :                                                               */
/*                                                                          */
/****************************************************************************/
int FAR PASCAL WinHelp(HWND hwndParent, LPCSTR pcFile, WORD wCmd, DWORD dData) {
  char acTopic[80];
  nc ncCurr;

  // Don't recurse if a WH_MSGFILTER has been set
  if (hf.hwnd || wCmd == HELP_QUIT) return FALSE;

  // Make sure the help class is registered
  if (!HelpInitClass()) return FALSE;

  // If monochrome mode we have to use the alternate colors
  iMonochrome = VID_IN_MONO_MODE() ? 1 : 0;

  // Open the correct file if last topic command
  if (wCmd == VK_BACK && szCurrHelpFile[0]) pcFile = szCurrHelpFile;

  // Open the help file
  if (!pcFile || !DoOpenHelpFile(pcFile) || !hf.ncInitial) {
    char ac[128];
    wsprintf(ac,
      #if GERMAN
        "Kann die Hilfedatei %s nicht finden",
      #else
        "Can't find file %s",
      #endif
      pcFile);
    #ifdef ERROR_BEEP
      MessageBeep(MB_ICONEXCLAMATION);
    #endif
    MessageBox(hwndHelp, ac, acError, MB_ICONEXCLAMATION | MB_OK);
    return FALSE;
  }

  // Parse the help command
  switch (wCmd) {
  case VK_BACK:
    // Display the last topic <Alt+F1>
    if (hf.ncCurr) {
      ncCurr = hf.ncCurr & 0x0000FFFFL | hf.ncInitial & 0xFFFF0000L;
      goto LastTopic;
    }
    // fall through
  case HELP_INDEX:
    // Display contents
    strcpy(acTopic, "h.contents");
    break;
  case HELP_HELPONHELP :
    // Display help on using help
    strcpy(acTopic, "QuickHelp");
    break;
  case HELP_KEY:
    // Display topic for keyword in dData
    lstrcpy(acTopic, (LPSTR)dData);
    break;
  case HELP_CONTEXT:
    // Display topic in dData
    wsprintf(acTopic, "#%lu", dData);
    break;
  default:
    // The following can't translate into the MS Help Advisor
    CloseHelpFile(hf.ncInitial);
    return FALSE;
  }

  // Search for the passed context
  ncCurr = HelpFindContext(hf.ncInitial, acTopic);
  switch (ncCurr) {
  case HELPERR_CONTEXTNOTFOUND:
    {
      char ac[128];
      CloseHelpFile(hf.ncInitial);
      wsprintf(ac,
        #if GERMAN
          "Kann den Begriff \"%s\" nicht finden",
        #else
          "Can't find topic \"%s\"",
        #endif
        acTopic);
      #ifdef ERROR_BEEP
        MessageBeep(MB_ICONEXCLAMATION);
      #endif
      MessageBox(hwndHelp, ac, acError, MB_ICONEXCLAMATION | MB_OK);
    }
  case HELPERR_NO_ACTION:
    return FALSE;
  }
  lstrcpy(hf.szTopic, acTopic);

LastTopic:

  // Create the help window
  hf.bIsDlg = TRUE;
  hf.hwnd = DialogCreate(hwndParent, 0, 0,
    GetSystemMetrics(SM_CYSCREEN), GetSystemMetrics(SM_CXSCREEN),
    #if GERMAN
      "Hilfe",
    #else
      "Help",
    #endif
    SYSTEM_COLOR,
    WS_POPUP | WS_BORDER | WS_CAPTION | WS_SYSMENU | WS_VSCROLL,
    NULL, 0, (LPSTR)"Help", 0);

  // Show the selected topic
  LastKeyPressed = wCmd;
  HelpDisplayContext(hf.hwnd, ncCurr);
  LastKeyPressed = 0;

  // Execute help window as dialog
  hwndHelp = hf.hwnd;
  _DialogBox(hf.hwnd);
  hwndHelp = NULL;

  return TRUE;
}


static BOOL PASCAL HelpInitClass(void)
{
  WNDCLASS wc;
  static   BOOL bHelpRegistered = FALSE;

  if (bHelpRegistered)
    return TRUE;

  memset((char *) &wc, 0, sizeof(wc));
  wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
  wc.hIcon         = (HICON) NULL;
  wc.lpszMenuName  = NULLHWND;
  wc.lpszClassName = "Help";
  wc.hbrBackground = (HBRUSH) GetStockObject(WHITE_BRUSH);
  wc.hInstance     = 0;
  wc.style         = CS_HREDRAW | CS_VREDRAW;
  wc.lpfnWndProc   = HelpWndProc;

  if (!RegisterClass((LPWNDCLASS) &wc))
    return FALSE;
  return (bHelpRegistered = TRUE);
}


typedef struct tagPBInfo
{
  char *text;
  int  x, y, width, height;
  int  id;
} PBINFO;


/*
  HelpWndProc - window procedure for the main window
*/
static long PASCAL HelpWndProc(HWND hwnd, WORD message, WORD wParam, LONG lParam) {
  nc   ncCurr;
  hotspot hsCurr;

  switch (message) {
  case WM_CREATE:
    {
      /*
        Setup stuff we need for horizontal scrolling.
      */
      HDC        hdc;
      TEXTMETRIC tm;
      RECT       r;

      hdc = GetDC(hwnd);
      GetTextMetrics(hdc, &tm);
      hf.xChr = tm.tmAveCharWidth;
      hf.yChr = tm.tmHeight /*+ tm.tmExternalLeading*/;
      ReleaseDC(hwnd, hdc);
      hf.MaxWidth = MAXHELPWIDTH * hf.xChr;
      GetClientRect(hwnd, &r);
      hf.maxLines = (r.bottom - CY_BUTTONROW) / hf.yChr;
      hf.pLines = NULL;
    }
    break;

  case WM_ERASEBKGND:
    // erase with background of normal attr
    {
      RECT rc;
      HBRUSH hbrOld = SelectObject((HDC)wParam, CreateSolidBrush(AttrToRGB(PhysAttrs[0][iMonochrome] >> 4)));
      GetUpdateRect(hwnd, &rc, FALSE);
      PatBlt((HDC)wParam, rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top, PATCOPY);
      DeleteObject(SelectObject((HDC)wParam, hbrOld));
    }
    return TRUE;

  case WM_PAINT:
    HelpRefreshWindow(hwnd);
    if (hf.bIsDlg) break;
    return 0;

  case WM_KEYDOWN:
  case WM_SYSKEYDOWN:
  case WM_CHAR:
    switch (wParam) {
    case VK_RETURN:
    case VK_SPACE:
      hsCurr = hf.hs;
      if (!HelpXRef(hf.pUncompressedTopic, &hsCurr)) break;
get_xref:
      if (hsCurr.pXref[0]) lstrcpy(szTopicToFind, hsCurr.pXref);
      else {
        ncCurr = HelpNc(hsCurr.pXref, hf.ncInitial);
        if (ncCurr) {
          HelpSzContext(szTopicToFind, ncCurr);
          strcpy(hf.szTopic, szTopicToFind);
          HelpDisplayContext(hwnd, ncCurr);
        } else MessageBeep(0);
        break;
      }
do_search:
      switch (ncCurr = HelpFindContext(hf.ncInitial, szTopicToFind)) {
      case HELPERR_CONTEXTNOTFOUND:
        {
          char ac[128];
          wsprintf(ac,
            #if GERMAN
              "Kann den Begriff \"%s\" nicht finden",
            #else
              "Cannot find topic \"%s\"",
            #endif
            szTopicToFind);
          #ifdef ERROR_BEEP
            MessageBeep(MB_ICONEXCLAMATION);
          #endif
          MessageBox(hwndHelp, ac, acError, MB_ICONEXCLAMATION | MB_OK);
        }
      case HELPERR_NO_ACTION:
        break;
      default:
        strcpy(hf.szTopic, szTopicToFind);
#if DEBUG
        sprintf(szFile, "Topic [%s]  Context %d", szTopicToFind, ncCurr);
        MessageBox(hwnd, "The topic has been found", szFile, MB_OK);
#endif
        HelpDisplayContext(hwnd, ncCurr);
      }
      break;

    case VK_SH_F1:
      strcpy(szTopicToFind, "h.contents");
      goto do_search;

    case VK_F1:
      strcpy(szTopicToFind, "QuickHelp");
      goto do_search;

    case VK_ESC:
    case VK_CTRL_F4:
    case VK_ALT_F4:
    case VK_ALT_X:
      PostMessage(hwnd, WM_CLOSE, 0, 0);
      break;
    case VK_ALT_F1:
    case VK_CTRL_B:
      HelpProcessKeys(hwnd, VK_BACK);
      break;
    case VK_CTRL_P:
    case '<':
    case '-':
      if (hf.acPrev[0]) ncCurr = HelpFindContext(hf.ncInitial, hf.acPrev);
      else ncCurr = HelpNcPrev(hf.ncCurr);
      if (ncCurr) HelpDisplayContext(hwnd, ncCurr);
      break;
    case VK_CTRL_N:
    case '>':
    case '+':
      if (hf.acNext[0]) ncCurr = HelpFindContext(hf.ncInitial, hf.acNext);
      else ncCurr = HelpNcNext(hf.ncCurr);
      if (ncCurr) HelpDisplayContext(hwnd, ncCurr);
      break;
    default :
      HelpProcessKeys(hwnd, wParam);
    }
    break;

  case WM_LBUTTONDOWN:
  case WM_RBUTTONDOWN:
    {
      int iCol = (LOWORD(lParam) - LEFT_INDENT) / hf.xChr + 1;
      int iLine = HIWORD(lParam) / hf.yChr;

      if (iLine < hf.ifreeze) iLine++;
      else iLine += hf.iTopLine;
      /*
        HelpXRef fills the hsCurr structure with hot-link info. hsCurr.ecol
        will contain the ending column of the hotlink topic. If hsCurr.pXref[0]
        is 0, then there is a special 3-byte structure which is used
        to determine the hot link info.
      */
      hsCurr.col = iCol;
      hsCurr.line = iLine;
      if ((HelpXRef(hf.pUncompressedTopic, &hsCurr))) {
        goto get_xref;
      } else {
        GetWordUnderCursor(iLine, iCol, szTopicToFind);
        if (*szTopicToFind) goto do_search;
      }
    }
    break;

  case WM_VSCROLL:
    {
      int key;

      /* The user touched the listbox's vertical scroll bar */
      key = 0;
      switch (wParam) {
      case SB_LINEUP:
        key = VK_UP;
        break;
      case SB_LINEDOWN:
        key = VK_DOWN;
        break;
      case SB_PAGEUP:
        key = VK_PGUP;
        break;
      case SB_PAGEDOWN:
        key = VK_PGDN;
        break;
      case SB_THUMBTRACK:
        hf.iTopLine = LOWORD(lParam) + 1;
        SetScrollPos(HIWORD(lParam), SB_CTL, LOWORD(lParam), FALSE);
        InvalidateRect(hwnd, NULL, TRUE);
        UpdateWindow(hwnd);
        if (hf.bIsDlg) break;
        return TRUE;
      }
      if (key) HelpProcessKeys(hwnd, key);
    }
    break;

  case WM_HSCROLL:
    {
      int   HrzScrInc;     // Horizontal scrolling incr (+/-)
      RECT  HelpRect;      // Client area of help window

      switch (wParam) {
      case SB_LINEUP:
        HrzScrInc = -1;
        break;

      case SB_LINEDOWN:
        HrzScrInc = 1;
        break;

      case SB_PAGEUP:
        HrzScrInc = -8;
        break;

      case SB_PAGEDOWN:
        HrzScrInc = 8;
        break;

      case SB_THUMBTRACK:
        HrzScrInc = LOWORD(lParam) - hf.HrzScrPos;
        break;

      default:                      // Default - don't scroll
        HrzScrInc = 0;
      }  //  switch (wParam)

      if (HrzScrInc = max(-hf.HrzScrPos,
                      min(HrzScrInc, hf.HrzScrMax - hf.HrzScrPos)))
      {
        hf.HrzScrPos += HrzScrInc;
        GetClientRect(hwnd, (LPRECT) &HelpRect);
        UpdateWindow(hwnd);
        ScrollWindow(hwnd, -hf.xChr * HrzScrInc, 0,
                     (LPRECT) &HelpRect, NULL);
        SetScrollPos(hwnd, SB_HORZ, hf.HrzScrPos, TRUE);
      }

      if (hf.bIsDlg) break;
      return 0;
    }

  case WM_SIZE:
    {
      RECT rClient;
      int iScroll;

      hf.xWnd = LOWORD(lParam);      // Get new width
      hf.yWnd = HIWORD(lParam);
      hf.HrzScrMax =             // Calc horz scroll range
        max(0, (hf.MaxWidth + LEFT_INDENT * 2 - hf.xWnd) / hf.xChr);
      hf.HrzScrPos =              //  and new horz scroll pos
        min(hf.HrzScrPos, hf.HrzScrMax);
      SetScrollRange(hwnd, SB_HORZ, 0, hf.HrzScrMax, FALSE);
      SetScrollPos(hwnd, SB_HORZ, hf.HrzScrPos, TRUE);

      GetClientRect(hwnd, &rClient);
      hf.maxLines = (rClient.bottom - CY_BUTTONROW) / hf.yChr;

      // Align end of topic to window bottom
      iScroll = hf.iTopLine + rClient.bottom - rClient.top - hf.nLines;
      if (iScroll > 0) {
        hf.iTopLine -= iScroll;
        if (hf.iTopLine < 1) hf.iTopLine = 1;
        InvalidateRect(hwnd, &rClient, TRUE);
      }
    }
    break;

  case WM_CLOSE:
    CloseHelpFile(hf.ncInitial);
    if (hf.bIsDlg) EndDialog(hwnd, TRUE);
    else DestroyWindow(hwnd);
    return TRUE;

  case WM_DESTROY:
    if (hf.ncInitial) CloseHelpFile(hf.ncInitial);
    hf.hwnd = NULLHWND;
    return TRUE;

  default:
    /* Call the default window procedure for the main window */
    return DefWindowProc(hwnd, message, wParam, lParam);
  }

  if (hf.bIsDlg) return TRUE;
  else return FALSE;
}


/****************************************************************************/
/*                                                                          */
/* Function :                                                               */
/*                                                                          */
/* Purpose  :                                                               */
/*                                                                          */
/* Returns  :                                                               */
/*                                                                          */
/****************************************************************************/

static int PASCAL DoOpenHelpFile(LPCSTR szFile) {
  nc ncCurr;
  if ((ncCurr = HelpOpenFile(szFile)) <= HELPERR_MAX) {
    char ac[128];
    wsprintf(ac,
      #if GERMAN
        "Kann die Hilfedatei %s nicht ffnen",
      #else
        "Can't open file %s",
      #endif
      szFile);
    #ifdef ERROR_BEEP
      MessageBeep(MB_ICONEXCLAMATION);
    #endif
    MessageBox(hwndHelp, ac, acError, MB_ICONEXCLAMATION | MB_OK);
    return FALSE;
  } else {
    hf.ncInitial = ncCurr;
#ifdef DEBUG
    MessageBox(hwndHelp, "The help file is open!", "Open", MB_OK);
#endif
    return TRUE;
  }
}



static int PASCAL HelpProcessKeys(HWND hwnd, int key)
{
  int iHeight;
  nc  ncCurr;
  RECT rClient;

  if (hf.ncCurr == 0)
    return FALSE;

  GetClientRect(hwnd, (LPRECT) &rClient);
  rClient.bottom -= CY_BUTTONROW;
  iHeight = RECT_HEIGHT(rClient) / hf.yChr;
  rClient.top += hf.ifreeze;

  switch (key)
  {
    case VK_UP      :
      if (hf.iTopLine > 1)
      {
        hf.iTopLine--;
        rClient.bottom = rClient.top + hf.yChr;
        InvalidateRect(hwnd, (LPRECT) &rClient, TRUE);
        break;
      }
      return TRUE;

    case VK_DOWN    :
      if (hf.iTopLine <= hf.nLines - iHeight)
      {
        hf.iTopLine++;
        rClient.top = rClient.bottom - hf.yChr;
        InvalidateRect(hwnd, (LPRECT) &rClient, TRUE);
        break;
      }
      return TRUE;

    case VK_LEFT    :
      SendMessage(hwnd, WM_HSCROLL, SB_PAGEUP, 0L);
      break;

    case VK_RIGHT   :
      SendMessage(hwnd, WM_HSCROLL, SB_PAGEDOWN, 0L);
      break;

    case VK_BACK:
      if ((ncCurr = HelpNcBack()) != 0) {
        LastKeyPressed = VK_BACK;
        HelpDisplayContext(hwnd, ncCurr);
        LastKeyPressed = 0;
        return TRUE;
      }
      #ifdef ERROR_BEEP
        MessageBeep(MB_ICONEXCLAMATION);
      #endif
      MessageBox(hwndHelp,
        #if GERMAN
          "Kann nicht weiter zurckgehen",
        #else
          "No additional history available",
        #endif
        acError, MB_ICONEXCLAMATION | MB_OK);
      break;

    case VK_TAB     :
    case VK_BACKTAB :
      HelpTraverseLinks(hwnd, key);
      return TRUE;

    case VK_HOME    :
      hf.iTopLine = 1;
      break;

    case VK_END     :
      hf.iTopLine = max(hf.nLines - iHeight + 1, 1);
      break;

    case VK_PGUP    :
      hf.iTopLine = max(hf.iTopLine - iHeight + 3, 1);
      break;

    case VK_PGDN    :
      hf.iTopLine = min(hf.iTopLine + iHeight - 3, max(hf.nLines - iHeight + 1, 1));
      break;

    default :
      if (isalnum(key)) HelpTraverseLinks(hwnd, key);
      else {
        MessageBeep(0);
        return FALSE;
      }
      break;
  }

  InvalidateRect(hwnd, (LPRECT) NULL, TRUE);
  return TRUE;
}

static nc PASCAL HelpOpenFile(LPCSTR pszHelpFile)
{
  nc   ncHelp;
  char szFile[128];

  if (hf.ncInitial);
    CloseHelpFile(hf.ncInitial);

  strcpy(szFile, pszHelpFile);
  if (!strchr(szFile, '.'))
    strcat(szFile, ".hlp");

  ncHelp = HelpOpen((char far *) szFile);
  if (ncHelp)
    strcpy(szCurrHelpFile, strupr(szFile));
  return ncHelp;
}


static VOID PASCAL CloseHelpFile(nc ncHelp)
{
  if (!ncHelp)
    return;

  /*
    Close the help file.
  */
  HelpClose(ncHelp);

  /*
    Get rid of the array of line pointers
  */
  freeLinePointers();

  /*
    Get rid of the buffer for the uncompressed topic
  */
  if (hf.pUncompressedTopic)
  {
    free(hf.pUncompressedTopic);
    hf.pUncompressedTopic = NULL;
  }

  /*
    Erase the help window
  */
  hf.ncInitial = 0;
  InvalidateRect(hf.hwnd, (LPRECT) NULL, TRUE);

  _heapmin();
}


static VOID PASCAL freeLinePointers(void)
{
  /*
    Get rid of the array of line pointers
  */
  if (hf.pLines)
  {
    int i;
    for (i = 0;  i < hf.nLines;  i++)
      if (hf.pLines[i])
        free(hf.pLines[i]);
    free(hf.pLines);
    hf.pLines = NULL;
  }
}


static nc PASCAL HelpFindContext(nc ncHelp, char *szContext)
{
  nc    ncCurr;
  char *pBang;
  char *pLastSlash;

  if (ncHelp == 0) return 0;
  if (!szContext || !*szContext) return (nc)HELPERR_CONTEXTNOTFOUND;

  // Check for commands
  if (*szContext == '!') {
    if (!strcmp(szContext + 1, "B")) {
      // Back
      PostMessage(hf.hwnd, WM_KEYDOWN, VK_BACK, 0);
      return (nc)HELPERR_NO_ACTION;
    } else {
      // Unknown command
      char ac[128];
      wsprintf(ac,
        #if GERMAN
          "Unbekannter Befehl \"%s\"",
        #else
          "Unknown command \"%s\"",
        #endif
        szContext);
      #ifdef ERROR_BEEP
        MessageBeep(MB_ICONEXCLAMATION);
      #endif
      MessageBox(hwndHelp, ac, acError, MB_ICONEXCLAMATION | MB_OK);
      return (nc)HELPERR_NO_ACTION;
    }
  }

  /*
    See if we have a topic of the form "filename!topic"
  */
  if ((pBang = strchr(szContext, '!')) != NULL)
  {
    *pBang = '\0';
    /*
      See if the desired file name is the same as the current file name.
      If the current file name has a path attached, just compare the
      file name portions.
    */
    if (!strcmp(strupr(szContext), szCurrHelpFile)   ||
        (pLastSlash = strrchr(szCurrHelpFile, '\\')) &&
         !strcmp(szContext, pLastSlash+1))
    {
      /*
        We are looking in the current help file. Get rid of the filename.
      */
      strcpy(szContext, pBang+1);
      *pBang = '!';
    }
    else
    {
      /*
        Search for the file
      */
      if ((ncCurr = HelpOpenFile(szContext)) <= HELPERR_MAX)
      {
        *pBang = '!';
        return (nc) HELPERR_CONTEXTNOTFOUND;
      }
      else
      {
        hf.ncInitial = ncHelp = ncCurr;
      }

      strcpy(szContext, pBang+1);
#if 0
      MessageBox(hwndHelp, "HelpComp does not support multiple helpfiles.",
                 "Error", MB_OK);
      return HELPERR_CONTEXTNOTFOUND;
#endif
    }
  }

  /* Find a context id for the keyword */
  if ((ncCurr = HelpNc((char far *) szContext, ncHelp)) == 0)
    return (nc) HELPERR_CONTEXTNOTFOUND;
  else
    return ncCurr;
}


static int PASCAL HelpDisplayContext(HWND hwnd, nc ncCurr)
{
  char   szBuf[256];
  WORD   nBytes;
  int    iLine;
  int    iHeight;
  RECT   rClient;
  PB     pCompressedTopic;

  if (!hwnd || !IsWindow(hwnd))
    return FALSE;

  GetClientRect(hwnd, (LPRECT) &rClient);
  rClient.bottom -= CY_BUTTONROW;
  iHeight = RECT_HEIGHT(rClient) / hf.yChr;

#if DEBUG
  sprintf(szBuf, "HelpDipslayContext() - ncCurr %d", ncCurr);
  MessageBox(hwnd, szBuf, NULL, MB_OK);
#endif

  HelpShrink();

  /*
    Get the number of bytes in the compressed topic
  */
  if ((nBytes = HelpNcCb(ncCurr)) == 0)
  {
#if DEBUG
    MessageBox(hwnd, "HelpNcCb() returns 0", NULL, MB_OK);
#endif
    return FALSE;
  }

#if DEBUG
  sprintf(szBuf, "HelpNcCb() - nBytes %d", nBytes);
  MessageBox(hwnd, szBuf, NULL, MB_OK);
#endif

  /*
    Allocate memory for the compressed topic
  */
  if ((pCompressedTopic = calloc(1, nBytes)) == NULL)
  {
    #ifdef ERROR_BEEP
      MessageBeep(MB_ICONEXCLAMATION);
    #endif
    MessageBox(hwndHelp,
      #if GERMAN
        "Nicht genug Speicherplatz vorhanden",
      #else
        "Not enough memory available",
      #endif
      acError, MB_ICONEXCLAMATION | MB_OK);
    return FALSE;
  }

  /*
    Read the compressed topic into the buffer
  */
  nBytes = HelpLook(ncCurr, pCompressedTopic);
#if DEBUG
  sprintf(szBuf, "HelpLook() - nBytes %d", nBytes);
  MessageBox(hwnd, szBuf, NULL, MB_OK);
#endif

  /*
    Decompress the topic
  */
  if (hf.pUncompressedTopic)   /* a topic allocated from before? */
  {
    free(hf.pUncompressedTopic);
    hf.pUncompressedTopic = NULL;
  }
  if ((hf.pUncompressedTopic = calloc(1, nBytes)) == NULL)
  {
    #ifdef ERROR_BEEP
      MessageBeep(MB_ICONEXCLAMATION);
    #endif
    MessageBox(hwndHelp,
      #if GERMAN
        "Nicht genug Speicherplatz vorhanden",
      #else
        "Not enough memory available",
      #endif
      acError, MB_ICONEXCLAMATION | MB_OK);
    free(pCompressedTopic);
    return FALSE;
  }

  /**************************************************************************
   *  Save backtrace info if we are not backtracing right now (e.g. if user
   *  didn't press "Back".  Also, free up all memory from the last topic
   *  displayed (if any).
   *************************************************************************/
  if (hf.ncCurr && (LastKeyPressed != VK_BACK))
    HelpNcRecord(hf.ncCurr);

  freeLinePointers();

  /**************************************************************************
  *  Setup to display new topic.  Allocate array of strings to hold the new
  *  topic (we'll paint to the display out of this array...).
  *************************************************************************/
  HelpDecomp(pCompressedTopic, hf.pUncompressedTopic, ncCurr);
  free(pCompressedTopic);

  /*
    Fetch the number of lines in this topic and calculate the number of
    pages in the topic. Set the scrollbar range between 1 and the
    number of pages.
  */
  hf.nLines = HelpcLines(hf.pUncompressedTopic);
  SetScrollRange(hwnd, SB_VERT, 0, max(hf.nLines - iHeight, 0), FALSE);

  // Get window title, previous and next topic
  {
    WORD wLine;
    BOOL fTitle = FALSE;
    hf.ifreeze = 0;
    hf.acPrev[0] = hf.acNext[0] = 0;
    HelpCtl(hf.pUncompressedTopic, TRUE);
    for (wLine = 1; wLine < (WORD)hf.nLines; wLine++) {
      char acLine[81];
      WORD wLength;
      wLength = HelpGetLine(wLine, 80, acLine, hf.pUncompressedTopic);
      if (!wLength || acLine[0] != *(char *)hf.pUncompressedTopic) break;
      acLine[wLength] = 0;
      switch(acLine[1]) {
      case 'n':
        fTitle = TRUE;
        SetWindowText(hf.hwnd, acLine + 2);
        break;
      case 'z':
        hf.ifreeze = atoi(acLine + 2);
        break;
      case '<':
        strncpy(hf.acPrev, acLine + 2, 40);
        break;
      case '>':
        strncpy(hf.acNext, acLine + 2, 40);
        break;
      }
    }
    if (!fTitle) SetWindowText(hf.hwnd,
      #if GERMAN
        "Hilfe");
      #else
        "Help");
      #endif
    HelpCtl(hf.pUncompressedTopic, FALSE);
  }

  /*
    Save backtrace info if we are showing a new topic
  */
  hf.ncCurr = ncCurr;
  hf.hs.line = hf.iTopLine = 1;
  hf.hs.col = hf.hs.ecol = 0;

  /*
    Allocate a new array of lines
  */
  hf.pLines = (char **) calloc(1, hf.nLines * sizeof(char *));

  for (iLine = 1;  ;  iLine++)
  {
    nBytes = HelpGetLine(iLine, sizeof(szBuf)-1, szBuf, hf.pUncompressedTopic);
    if (nBytes > 0) {
      szBuf[nBytes] = '\0';
    } else {
      break;
    }
    hf.pLines[iLine-1] = strdup(szBuf);
  }

  InvalidateRect(hwnd, (LPRECT) NULL, TRUE);
  return TRUE;
}


static BOOL PASCAL HelpDisplayCurrHotlink(HWND hwnd, BOOL fOn) {
  int iLine;
  if (!hwnd || !IsWindow(hwnd)) return FALSE;
  if (!hf.hs.col) return FALSE;
  if ((int)hf.hs.line <= hf.ifreeze) iLine = hf.hs.line - 1;
  else {
    iLine = hf.hs.line - hf.iTopLine;
    if (iLine < hf.ifreeze) return FALSE;
  }
  if (iLine < 0 || iLine >= hf.yWnd) return FALSE;
  if (fOn) {
    HDC hdc;
    int iAttr;
    int iLength;
    iLength = 5;
    hdc = GetDC(hwnd);
    iAttr = PhysAttrs[HELPCLR_HOTSPOT][iMonochrome];
    SetTextColor(hdc, AttrToRGB(GET_FOREGROUND(iAttr)));
    SetBkColor(hdc, AttrToRGB(GET_BACKGROUND(iAttr)));
    TextOut(hdc,
            (hf.hs.col - 1) * hf.xChr + LEFT_INDENT - hf.HrzScrPos,
            iLine * hf.yChr,
            hf.pLines[hf.hs.line - 1] + hf.hs.col - 1,
            hf.hs.ecol - hf.hs.col + 1);
    ReleaseDC(hwnd, hdc);
  } else HelpDisplayLineWithAttrs(hwnd, NULL, hf.hs.line, iLine, hf.pLines[hf.hs.line - 1]);
  return TRUE;
}


/*
  GetWordUnderCursor()
    Given an [x,y] coordinate, return the word which occupies that
  position.
*/
static VOID PASCAL GetWordUnderCursor(int y, int x, char *szBuf)
{
  BYTE *s;
  BYTE *pBOL;

  *szBuf = '\0';

  /*
    Make sure that we are not going past the last line
  */
  if (y > hf.nLines) return;

  /*
    Make sure we are pointing in a column past the end of the line, or
    else we get a GP fault.
  */
  pBOL = hf.pLines[y-1];
  if (x > (int)strlen(pBOL)) return;

  /*
    Move back through the non-blank characters until we find the
    start of the word.
  */
  s = pBOL + x - 1;
  if (isalnum(*s)) {
    while (--s >= pBOL && isalnum(*s));
    s++;
  }

  /*
    Copy characters until we reach the end of the word
  */
  while (isalnum(*s)) *szBuf++ = (char)*s++;
  *szBuf = '\0';
}


static int PASCAL HelpTraverseLinks(HWND hwnd, int key) {
  int iLead;
  hotspot hsFound;
  int iScroll;
  if (!hwnd || !IsWindow(hwnd)) return FALSE;
  switch (key) {
  case VK_TAB:
    iLead = 0;
    break;
  case VK_BACKTAB:
    iLead = -1;
    break;
  default:
    if (key >= 'a' && key <= 'z') iLead = toupper(key);
    else if (key >= 'A' && key <= 'Z') iLead = -(key);
  }
  hsFound = hf.hs;
  if (iLead >= 0) { // Search forward
    hsFound.col = hsFound.ecol + 1;
    if (!HelpHlNext(iLead, hf.pUncompressedTopic, &hsFound)) {
      hsFound.line = hsFound.col = 1;
      if (!HelpHlNext(iLead, hf.pUncompressedTopic, &hsFound)) return FALSE;
    }
  } else { // Search backward
    if (!HelpHlNext(iLead, hf.pUncompressedTopic, &hsFound)) {
      hsFound.line = hf.nLines;
      hsFound.col = strlen(hf.pLines[hf.nLines - 1]) + 1;
      if (!HelpHlNext(iLead, hf.pUncompressedTopic, &hsFound)) return FALSE;
    }
  }
  iScroll = 0;
  if ((int)hsFound.line > hf.ifreeze) { // Adjust window
    int iLine = hsFound.line - hf.iTopLine;
    if (iLine < hf.ifreeze) iScroll = iLine - hf.ifreeze;
    else if (iLine >= hf.yWnd) iScroll = iLine + 1 - hf.yWnd;
  }
  if (iScroll) {
    hf.iTopLine += iScroll;
    hf.hs = hsFound;
    InvalidateRect(hwnd, NULL, TRUE);
  } else {
    HelpDisplayCurrHotlink(hwnd, FALSE);
    hf.hs = hsFound;
    HelpDisplayCurrHotlink(hwnd, TRUE);
  }
  return TRUE;
}


/****************************************************************************/
/*                                                                          */
/*                        DISPLAY FUNCTIONS                                 */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                                                                          */
/* Function : HelpRefreshWindow()                                           */
/*                                                                          */
/* Purpose  : Draws the text within the client area of the help window.     */
/*                                                                          */
/* Returns  :                                                               */
/*                                                                          */
/****************************************************************************/
static VOID PASCAL HelpRefreshWindow(HWND hwnd)
{
  PAINTSTRUCT ps;
  HDC     hDC;
  LPRECT  lprcUpdate;
  int     iLine;
  int     iHeight;
  RECT    rClient;

  if (!hwnd)
    return;

  /*
    Get the coordinates of the displayable client area.
  */
  GetClientRect(hwnd, (LPRECT) &rClient);
  rClient.bottom -= CY_BUTTONROW;

  /*
    Determine the area which has to be refreshed
  */
  hDC = BeginPaint(hwnd, &ps);
  lprcUpdate = &ps.rcPaint;
  if (lprcUpdate->bottom > rClient.bottom) lprcUpdate->bottom = rClient.bottom;

  /*
    Get the number of lines which can be displayed in the help window's
    client area (minus the row of buttons).
  */
  iHeight = RECT_HEIGHT(rClient) / hf.yChr;

  /*
    Do the actual painting.
  */
  if (hf.pLines != NULL) {
    int i;
    for (i = lprcUpdate->top / hf.yChr; i <= lprcUpdate->bottom / hf.yChr; i++) {
      if (i < hf.ifreeze) iLine = i + 1;
      else iLine = hf.iTopLine + i;
      if (iLine > hf.nLines) break;
      if (!HelpDisplayLineWithAttrs(hwnd, hDC, iLine, i, hf.pLines[iLine - 1])) break;
    }
  }
  EndPaint(hwnd, &ps);

  HelpDisplayCurrHotlink(hwnd, TRUE);

  /*
    Refresh the scrollbars
  */
  SetScrollRange(hwnd, SB_VERT, 0, max(hf.nLines - iHeight, 0), FALSE);
  SetScrollPos(hwnd, SB_VERT, hf.iTopLine - 1, TRUE);
}


/****************************************************************************/
/*                                                                          */
/* Function : HelpDisplayLineWithAttrs                                      */
/*                                                                          */
/* Purpose  : Displays a single line of help text at the passed coordinates.*/
/*            The coordinates are really indices into the help text.        */
/*                                                                          */
/* Returns  :                                                               */
/*                                                                          */
/****************************************************************************/
static int PASCAL 
HelpDisplayLineWithAttrs(HWND hwnd, HDC hDC, int iLine, int row, char *pString)
{
  HDC  hDCOrig;
  lineattr la[80], *pAttrs;
  int  iCol = 0;
  char *pOrigStr = pString;
  WORD attr;


  if (!hwnd || iLine <= 0)
    return FALSE;

  if ((hDCOrig = hDC) == NULL)
    hDC = GetDC(hwnd);

  iCol = -hf.HrzScrPos;

  /*
    Get the line's color-attribute array.
  */

  if (HelpGetLineAttr(iLine, sizeof(la) * 80, (lineattr far *) la, 
                      hf.pUncompressedTopic) == 0)
  {
    if (hDCOrig == NULL)
      ReleaseDC(hwnd, hDC);
    return FALSE;
  }

  /*
    Go through the attribute array for this line and output a run
    of text which has the same attribute.
  */
  pAttrs = la;
  while (*pString) {
    WORD wLength = strlen(pString);
    if (pAttrs->cb < wLength) {
      wLength = pAttrs->cb;
    }
    attr = PhysAttrs[pAttrs->attr][iMonochrome];
    SetTextColor(hDC, AttrToRGB(GET_FOREGROUND(attr)));
    SetBkColor(hDC, AttrToRGB(GET_BACKGROUND(attr)));
    TextOut(hDC, iCol * hf.xChr + LEFT_INDENT, row * hf.yChr, pString, wLength);
    iCol += wLength;
    pString += wLength;
    pAttrs++;
  }

  if (hDCOrig == NULL)
    ReleaseDC(hwnd, hDC);
  return TRUE;
}


int PASCAL HelpQueryColor(WORD iSysColor)
{
#ifdef OS2
  extern int AdapterType;
  BOOL bMono = (BOOL) (AdapterType == 1);
#else
  BOOL bMono = VID_IN_MONO_MODE() ? 1 : 0;
#endif

  if ((int) iSysColor >= HELPCLR_FIRST && (int) iSysColor <= HELPCLR_LAST)
    return bMono ? PhysAttrs[iSysColor][1] : PhysAttrs[iSysColor][0];
  else
    return 0;
}


int PASCAL HelpSetColor(WORD iSysColor, WORD attr)
{
#ifdef OS2
  extern int AdapterType;
  BOOL bMono = (BOOL) (AdapterType == 1);
#else
  BOOL bMono = VID_IN_MONO_MODE() ? 1 : 0;
#endif

  if ((int) iSysColor >= HELPCLR_FIRST && (int) iSysColor <= HELPCLR_LAST)
    return bMono ? (PhysAttrs[iSysColor][1] = (BYTE) attr) 
                 : (PhysAttrs[iSysColor][0] = (BYTE) attr);
  else
    return 0;
}



/*===========================================================================*/
/*                                                                           */
/*               DOS SUPPORT ROUTINES FOR THE MS HELP ENGINE                 */
/*                     Memory Allocation & File Handling                     */
/*                                                                           */
/*===========================================================================*/

typedef struct _hfInfo
{
  char *szFileName;
} hfINFO;
hfINFO hfInfo[MAXFILES] = {0};  /* file name pointer table    */

static int  fdLastOpened = -1;
static int  hFileLastOpened = -1;

/*
 * Function Prototypes
 */
extern mh   PASCAL far HelpAlloc(WORD);
extern VOID PASCAL far HelpDealloc(mh);
extern char far *PASCAL far HelpLock(mh);
extern VOID PASCAL far HelpUnlock(mh);
extern int  PASCAL far OpenFileOnPath(char far *, int);
extern DWORD PASCAL far ReadHelpFile(int, DWORD, uchar far *, WORD);
extern VOID PASCAL far HelpCloseFile(int);
extern char *PASCAL    _SearchHelpfiles(char far *, char *);

/*===========================================================================*/
/*                                                                           */
/*                    FILE ACCESS ROUTINES                                   */
/*                                                                           */
/*===========================================================================*/

int PASCAL far OpenFileOnPath(char far *lpszPathName, int mode)
{
  char far *pColon;
  char szRetName[128], szEnvVar[65];
  char npszPathName[128];
  int  fd, i;

  /*
    If the path name starts with $xxx:, then use use the envvar xxx for search
  */
  if (*lpszPathName == '$' && (pColon = lstrchr(lpszPathName,':')) != NULL)
  {
    *pColon = '\0';
    lstrcpy(szEnvVar, lpszPathName);
    *pColon++ = ':';
    lpszPathName = pColon;
  }
  else
    strcpy(szEnvVar, "PATH");

  lstrcpy(npszPathName, lpszPathName);  /* for medium model... */

  if (_DosSearchPath(szEnvVar, npszPathName, szRetName) == NULL &&
      _SearchHelpfiles(npszPathName, szRetName) == NULL)
    return 0;

  if ((fd = _lopen(szRetName, OF_READ | OF_SHARE_DENY_NONE)) < 0)
    return 0;
  close(fd);

  for (i = 0;  i < MAXFILES;  i++)
    if (hfInfo[i].szFileName == NULL)
    {
      strcpy(hfInfo[i].szFileName = calloc(1, strlen(szRetName)+1),
             szRetName);
      return i+1;
    }

  return 0;
}


VOID PASCAL far HelpCloseFile(int hFile)
{
  hfINFO *pHFI = &hfInfo[--hFile];

  if (!pHFI)
    return;

  /*
    Close the file, free any memory allocated to the help file table entry,
    and flag that slot in the help file table as being available.
  */
  free(pHFI->szFileName);
  pHFI->szFileName = NULL;

  /*
    If the file which we're closing is the last file accessed, then the
    file is still open. Close it.
  */
  if (hFileLastOpened == hFile)
  {
    hFileLastOpened = -1;
    if (fdLastOpened != -1)
      close(fdLastOpened);
  }
}


DWORD PASCAL far ReadHelpFile(int hFile,DWORD fPos,uchar far *pBuf,WORD nBytes)
{
  hfINFO *pHFI = &hfInfo[--hFile];
  WORD rc = 0;

  if (!pHFI)
    return rc;

  /*
    If we are opening a different file than the last one accessed, then
    close the last one accessed.
  */
  if (hFile != hFileLastOpened)
  {
    if (fdLastOpened != -1)
      close(fdLastOpened);
    fdLastOpened = _lopen(pHFI->szFileName, OF_READ | OF_SHARE_DENY_NONE);
    hFileLastOpened = hFile;
  }

  if (fdLastOpened < 0)
    return rc;


  /*
    If we specified NULL for the buffer, then we want to find out the size
    of the file.
  */
  if (pBuf == NULL)
  {
    rc = (WORD) lseek(fdLastOpened, 0L, 2);
    return rc;
  }

  /*
    Seek to the desired position and read 'em
  */
  if (lseek(fdLastOpened, fPos, 0) < 0)
    return rc;
  nBytes = _lread(fdLastOpened, pBuf, nBytes);
  return nBytes;
}


#ifdef OS2
VOID far PASCAL LOADDS HelpShrink(VOID) {}
#endif


char *PASCAL _SearchHelpfiles(szhfName, szRetFileName)
  char far *szhfName;
  char *szRetFileName;
{
  char *pDir, *pSemi, *pEnd;
  char *pchWild;
  char szName[80];

  /*
    Get the path specified in the environment var
  */
  if ((pDir = getenv("HELPFILES")) == NULL)
    return NULL;

  for (;;)
  {
    /*
      Is there a semi-colon. If so, cut it off before copying.
    */
    if ((pSemi = strchr(pDir, ';')) != NULL)
      *pSemi = '\0';

    /*
      See if we have something like :
        c:\msc6\help\*.hlp
    */
    if ((pchWild = strchr(pDir, '*')) && (pEnd = strrchr(pDir, '\\')))
    {
      *pEnd = '\0';
    }

    /*
      Copy the path, a backslash, and the filename
    */
    strcpy(szName, pDir);
    pEnd = szName + strlen(szName);
    if (pEnd[-1] != '\\')    /* append the final backslash */
      *pEnd++ = '\\';
    lstrcpy(pEnd, szhfName);

    /*
      Check for existence
    */
    if (access(szName, 0) == 0)
      return strcpy(szRetFileName, szName);

    /*
      Not there. Try to move on to the next path.
    */
    if (pSemi == NULL)
      return NULL;
    else
      pDir = pSemi + 1;   /* go past the semicolon */
  }
}


/*===========================================================================*/
/*                                                                           */
/*                    MEMORY ALLOCATION ROUTINES                             */
/*                                                                           */
/*===========================================================================*/
mh PASCAL FAR HelpAlloc(WORD size)
{
  mh   hMem;

  if (!size)     // Anything requested?
    return 0;    // No, return no handle

  /*
    If the first allocation doesn't succeed, call HelpShrink()
    and try again.
  */
  if ((hMem = GlobalAlloc(GMEM_MOVEABLE, (DWORD) size)) != NULL)
    return hMem;
  HelpShrink();
  if ((hMem = GlobalAlloc(GMEM_MOVEABLE, (DWORD) size)) != NULL)
    return hMem;
  return 0;
}

VOID PASCAL FAR HelpDealloc(mh hMem)
{
  if (hMem)
    GlobalFree(hMem);
}

LPSTR PASCAL FAR HelpLock(mh hMem)
{
  if (hMem)
    return GlobalLock(hMem);
  return (LPSTR) NULL;
}

VOID PASCAL FAR HelpUnlock(mh hMem)
{
  GlobalUnlock(hMem);
}
