// Verwaltung der Link-Fenster

#include "pc64.h"

#define CHRIS 0

const gwBufSize = 1024;

static flag fMove;
static char acSource[80];
static char acDest[80];
static CDevice* pSource;
static CDevice* pDest;
static HWND hwndListBox;
static int hFile;

// Abbruchprozedur fr Dialogboxen mit Abbruchknopf
static HWND hwndDialog;
static long lTotal;
static long lPos;
flag fAbort;
static flag LinkWndAbortProc() {
  SetDlgItemProgress(hwndDialog, IDT_PROGRESS, lPos, lTotal);
  return fAbort;
}

// Fehlermeldung anhand von Bus-Flags setzen
static void SetBusError(word wFlags, char* pcError) {
  if (fAbort) {
    pcError[0] = 0;
  } else if (wFlags & ENODEVICE) {
    strcpy(pcError, acDeviceNotPresent);
  } else if (wFlags & EOUTPUT) {
    strcpy(pcError, acOutput);
  } else if (wFlags & EINPUT) {
    strcpy(pcError, acInput);
  } else {
    strcpy(pcError, acUnknownBusError);
  }
}

// Gre der markierten Dateien auffrischen
static void UpdateMarked(LINKWND* plw) {
  if (!plw->pDevice) {
    return;
  }
  long lSize = 0;
  if (SendMessage(plw->hwndListBox, LB_GETSELCOUNT, 0, 0) >= 0) {
    word wItem = (word)SendMessage(plw->hwndListBox, LB_GETCOUNT, 0, 0);
    while (wItem--) {
      if (SendMessage(plw->hwndListBox, LB_GETSEL, wItem, 0) > 0) {
        lSize += SendMessage(plw->hwndListBox, LB_GETITEMDATA, wItem, 0);
      }
    }
  }
  char ac[80];
  GetWindowText(plw->hwnd, ac, 80);
  assert(strcmp(ac, acUnknown));
  char* pc = strend(ac);
  if (!strcmp(pc - 8, " Blocks)")) {
    pc -= 9;
    while (*pc != ' ') {
      pc--;
    }
    assert(pc >= ac);
  }
  if (lSize > 0) {
    wsprintf(pc, " (%lu Blocks)", lSize / 254);
  } else {
    *pc = 0;
  }
  SetWindowText(plw->hwnd, ac);
}

// Neues Link-Fenster anlegen
HWND CreateLinkWnd(WNDPOS* pwp) {
  LINKWND* plw = (LINKWND*)malloc(sizeof LINKWND);
  assert(plw);
  memset(plw, 0, sizeof LINKWND);
  MDICREATESTRUCT mdic;
  mdic.szClass = "PC64Link";
  mdic.szTitle = acUnknown;
  mdic.hOwner = hInst;
  if (pwp) {
    mdic.x = (muldiv(pwp->iLeft, iFrameWidth, 5000) + 1) / 2;
    mdic.y = (muldiv(pwp->iTop, iFrameHeight, 5000) + 1) / 2;
    mdic.cx = (muldiv(pwp->iRight, iFrameWidth, 5000) + 1) / 2 - mdic.x;
    mdic.cy = (muldiv(pwp->iBottom, iFrameHeight, 5000) + 1) / 2 - mdic.y;
    mdic.style = pwp->dStyle;
  } else {
    mdic.x = mdic.y = mdic.cx = mdic.cy = CW_USEDEFAULT;
    mdic.style = 0;
  }
  lSetWindowLong = (long)plw;
  HWND hwnd = (HWND)SendMessage(hwndClient, WM_MDICREATE, 0, (LONG)(LPSTR)&mdic);
  assert(hwnd);
  assert(plw->hwndDirTree);
  assert(plw->hwndListBox);
  if (pwp) {
    SetWindowText(plw->hwndDirTree, pwp->acName);
  } else {
    SetWindowText(plw->hwndDirTree, ::acDir);
  }
  return hwnd;
}

void DeleteNames(LINKWND* plw) {
  assert(plw);
  assert(plw->hwndDirTree);
  assert(plw->hwndListBox);
  if (plw->pDevice) {
    SendMessage(plw->hwndListBox, LB_RESETCONTENT, 0, 0);
    SetWindowText(plw->hwnd, acUnknown);
    delete plw->pDevice;
    plw->pDevice = NULL;
  }
}

// Verzeichnis in ListBox einlesen
flag ReadNames(LINKWND* plw) {
  assert(plw);
  assert(plw->hwnd);
  assert(plw->hwndDirTree);
  assert(plw->hwndListBox);
  DeleteNames(plw);
  int i;
  word w;
  CDevice* pDevice;
  char acName[80];
  int iEntries;
  GetWindowText(plw->hwndDirTree, acName, 80);
  assert(acName[0]);
  flag fReturn = FALSE;
  char acError[80];
  iRetries = 5;
  if (!SetDir(&plw->pDevice, acName)) {
    assert(plw->pDevice);
    plw->pDevice->GetError(acError, 80);
    goto Stop;
  }
  pDevice = plw->pDevice;
  assert(pDevice);
  if (pDevice->GetType() == DEVFLOPPY) {
    SetWindowLong(plw->hwndListBox, GWL_STYLE, GetWindowLong(plw->hwndListBox, GWL_STYLE) | LBS_SORT);
  } else {
    SetWindowLong(plw->hwndListBox, GWL_STYLE, GetWindowLong(plw->hwndListBox, GWL_STYLE) & ~LBS_SORT);
  }
  iRetries = 50;
  w = pDevice->Open(0, "$", 1);
  if (w) {
    SetBusError(w, acError);
    goto Stop;
  }
  if (pDevice->GetError(acError, 80)) {
    goto Stop;
  }
  for (i = 0; i < 8; i++) {
    w = pDevice->Get(0);
    if (w >= 256) {
      SetBusError(w, acError);
      goto Stop;
    }
  }
  for (i = 0; i < 16; i++) {
    w = pDevice->Get(0);
    if (w >= 256) {
      SetBusError(w, acError);
      goto Stop;
    }
    acName[i] = (char)w;
  }
  while (i-- && acName[i] == ' ');
  acName[i + 1] = 0;
  SetWindowText(plw->hwnd, acName);
  for (i = 0; i < 8; i++) {
    w = pDevice->Get(0);
    if (w >= 256) {
      SetBusError(w, acError);
      goto Stop;
    }
  }
  switch (pDevice->GetType()) {
  case DEVFLOPPY:
    iEntries = 32767;
    break;
  case DEVIMAGE:
    iEntries = 150;
    break;
  case DEVLINK:
    iEntries = 1000;
    break;
  }
  while (--iEntries >= 0) {
    iRetries = 50;
    char acLine[40];
    for (i = 0; i < 2; i++) {
      w = pDevice->Get(0);
      if (w >= 256) {
        SetBusError(w, acError);
        goto Stop;
      }
    }
    word wBlocks = pDevice->Get(0);
    if (wBlocks >= 256) {
      SetBusError(wBlocks, acError);
      goto Stop;
    }
    w = pDevice->Get(0);
    if (w >= 256) {
      SetBusError(w, acError);
      goto Stop;
    }
    wBlocks |= w << 8;
    while ((w = pDevice->Get(0)) != '"') {
      if (w >= 256) {
        SetBusError(w, acError);
        goto Stop;
      }
      if (w != ' ') {
        w = pDevice->Close(0);
        if (w >= 256) {
          SetBusError(w, acError);
          goto Stop;
        }
        if (pDevice->GetError(acError, 80) == 0) {
          fReturn = TRUE;
        }
        goto Stop;
      }
    }
    for (i = 0; i < 17; i++) {
      w = pDevice->Get(0);
      if (w >= 256) {
        SetBusError(w, acError);
        goto Stop;
      }
      switch ((char)w) {
      case 0:
      case '\t':
      case '\n':
      case ':':
      case ',':
        acLine[i] = '?';
        break;
      default:
        acLine[i] = (char)w;
      }
    }
    while (i && acLine[i - 1] == ' ') {
      i--;
    }
    if (i > 1 && acLine[i - 1] == '"' && !memchr(acLine, '"', i - 1)) {
      i--;
    }
    while ((w = pDevice->Get(0)) == ' ');
    if (w >= 256) {
      SetBusError(w, acError);
      goto Stop;
    }
    if (memchr("PSUDR", (char)w, 5)) {
      if ((char)w != 'P') {
        acLine[i++] = ',';
        acLine[i++] = (char)w;
      }
      acLine[i] = 0;
      assert(i <= 18);
      i = (int)SendMessage(plw->hwndListBox, LB_ADDSTRING, 0, (LONG)(LPSTR)acLine);
      if (i >= 0) {
        SendMessage(plw->hwndListBox, LB_SETITEMDATA, i, wBlocks * 254L);
      }
    }
    int iTimeout = 100;
    while ((w = pDevice->Get(0)) != 0) {
      if (w >= 256) {
        SetBusError(w, acError);
        goto Stop;
      }
      iTimeout--;
      if (!iTimeout) {
        goto DirDestroyed;
      }
    }
  }
DirDestroyed:
  #if GERMAN
    strcpy(acError, "Inhaltsverzeichnis ist zerstrt!");
  #else
    strcpy(acError, "Directory is destroyed!");
  #endif
Stop:
  if (!fReturn) {
    if (atoi(acError) != 97) {
      ErrorBox(NULL, acError);
    }
    if (SendMessage(plw->hwndListBox, LB_GETCOUNT, 0, 0)) {
      fReturn = TRUE;
      assert(pDevice == plw->pDevice);
      pDevice->Reset();
    } else {
      DeleteNames(plw);
    }
  }
  return fReturn;
}

// Link-Fenster schlieen
extern "C" int FAR PASCAL GetInternalWindowPos(HWND hwnd, LPRECT prc, LPPOINT ppt);
flag CloseLinkWnd(HWND hwnd, int hFile) {
  assert(hwnd);
  if (hFile > 0) {
    LINKWND* plw = (LINKWND*)GetWindowLong(hwnd, 0);
    assert(plw);
    WNDPOS wp;
    memset(&wp, 0, sizeof WNDPOS);
    if (GetWindowText(plw->hwndDirTree, wp.acName, 80)) {
      wp.fIsLinkWnd = TRUE;
      RECT rc;
      POINT pt;
      int iShow = GetInternalWindowPos(hwnd, &rc, &pt);
      ScreenToClient(hwndClient, (LPPOINT)&rc.left);
      ScreenToClient(hwndClient, (LPPOINT)&rc.right);
      wp.iLeft = muldiv(rc.left, 10000, iFrameWidth);
      wp.iTop = muldiv(rc.top, 10000, iFrameHeight);
      wp.iRight = muldiv(rc.right, 10000, iFrameWidth);
      wp.iBottom = muldiv(rc.bottom, 10000, iFrameHeight);
      if (iShow == SW_SHOWMAXIMIZED) wp.dStyle = WS_MAXIMIZE;
      else if (iShow == SW_SHOWMINIMIZED) wp.dStyle = WS_MINIMIZE;
      _lwrite(hFile, &wp, sizeof WNDPOS);
    }
  }
  return TRUE;
}

// nderungen in allen Fenstern aktualisieren
static void FileChange(char* pcDir, char* pcOld, char* pcNew, long lSize) {
  assert(pcDir);
  assert(strchr(pcDir, ':'));
  // ",P" lschen
  char* pcOldType = strchr(pcOld, ',');
  assert(!pcOld || pcOldType);
  if (pcOldType && pcOldType[1] == 'P') {
    *pcOldType = 0;
  }
  char* pcNewType = strchr(pcNew, ',');
  assert(!pcNew || pcNewType);
  if (pcNewType && pcNewType[1] == 'P') {
    *pcNewType = 0;
  }
  assert(hwndClient);
  for (HWND hwnd = GetWindow(hwndClient, GW_CHILD); hwnd; hwnd = GetWindow(hwnd, GW_HWNDNEXT)) {
    if (GetWindowLong(hwnd, GWL_WNDPROC) == (LONG)LinkWndProc) {
      LINKWND* plw = (LINKWND*)GetWindowLong(hwnd, 0);
      assert(plw);
      assert(plw->hwnd == hwnd);
      if (plw->pDevice) {
        char acDir[80];
        plw->pDevice->GetDir(acDir, 80);
        if (!strcmp(acDir, pcDir)) {
          assert(plw->hwndListBox);
          int iFocus = (int)SendMessage(plw->hwndListBox, LB_GETCARETINDEX, 0, 0);
          int i = (int)SendMessage(plw->hwndListBox, LB_GETCOUNT, 0, 0);
          while (i--) {
            long lBlocks = SendMessage(plw->hwndListBox, LB_GETITEMDATA, i, 0);
            assert(!(lBlocks & 0xC0000000));
            if (SendMessage(plw->hwndListBox, LB_GETSEL, i, 0) > 0) {
              lBlocks |= 0x80000000L;
            }
            if (i == iFocus) {
              lBlocks |= 0x40000000L;
            }
            SendMessage(plw->hwndListBox, LB_SETITEMDATA, i, lBlocks);
          }
          if (pcOld) {
            int iIndex = (int)SendMessage(plw->hwndListBox, LB_FINDSTRINGEXACT, -1, (LONG)(LPSTR)pcOld);
            if (iIndex >= 0) {
              lSize = SendMessage(plw->hwndListBox, LB_GETITEMDATA, iIndex, 0);
              SendMessage(plw->hwndListBox, LB_DELETESTRING, iIndex, 0);
            } else {
              lSize = 0;
            }
          }
          if (pcNew && SendMessage(plw->hwndListBox, LB_FINDSTRINGEXACT, -1, (LONG)(LPSTR)pcNew) < 0) {
            int iIndex = (int)SendMessage(plw->hwndListBox, LB_ADDSTRING, 0, (LONG)(LPSTR)pcNew);
            assert(iIndex >= 0);
            SendMessage(plw->hwndListBox, LB_SETITEMDATA, iIndex, lSize);
          }
          i = (int)SendMessage(plw->hwndListBox, LB_GETCOUNT, 0, 0);
          if (iFocus < 0 || iFocus >= i) {
            iFocus = i - 1;
          }
          while (i--) {
            long lSize = SendMessage(plw->hwndListBox, LB_GETITEMDATA, i, 0);
            if (lSize & 0x80000000L) {
              SendMessage(plw->hwndListBox, LB_SETSEL, TRUE, i);
            }
            if (lSize & 0x40000000L) {
              SendMessage(plw->hwndListBox, LB_SETCARETINDEX, i, TRUE);
              iFocus = -1;
            }
            SendMessage(plw->hwndListBox, LB_SETITEMDATA, i, lSize & 0x3FFFFFFFL);
          }
          if (iFocus >= 0) {
            SendMessage(plw->hwndListBox, LB_SETCARETINDEX, iFocus, TRUE);
          }
          UpdateMarked(plw);
        }
      }
    }
  }
  // ",P" wiederherstellen
  if (pcOldType) {
    *pcOldType = ',';
  }
  if (pcNewType) {
    *pcNewType = ',';
  }
}

// Datei umbenennen
BOOL FAR PASCAL RenameDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  static HWND hwndListBox;
  static int i;
  static char acOld[19];
  static char* pc;
  LINKWND* plw;
  switch (wMsg) {
  case WM_INITDIALOG:
    CenterWindow(hwnd);
    SendDlgItemMessage(hwnd, IDE_NEWNAME, EM_LIMITTEXT, 16, 0);
    assert(hwndChild);
    assert(GetWindowLong(hwndChild, GWL_WNDPROC) == (LONG)LinkWndProc);
    plw = (LINKWND*)GetWindowLong(hwndChild, 0);
    assert(plw);
    hwndListBox = plw->hwndListBox;
    assert(hwndListBox);
    if (SendMessage(hwndListBox, LB_GETSELCOUNT, 0, 0) > 0) {
      i = 0;
    NextFile:
      int iCount = (int)SendMessage(hwndListBox, LB_GETCOUNT, 0, 0);
      assert(iCount >= 0);
      while (SendMessage(hwndListBox, LB_GETSEL, i, 0) <= 0) {
        if (++i >= iCount) {
          EndDialog(hwnd, TRUE);
          return TRUE;
        }
      }
    } else {
      i = (int)SendMessage(hwndListBox, LB_GETCARETINDEX, 0, 0);
      assert(i >= 0);
    }
    SendMessage(hwndListBox, LB_GETTEXT, i, (LONG)(LPSTR)acOld);
    assert(acOld[0]);
    if (!strchr(acOld, ',')) {
      strcat(acOld, ",P");
    }
    assert(strlen(acOld) <= 18);
    pc = strchr(acOld, ',');
    assert(pc);
    *pc = 0;
    SetDlgItemText(hwnd, IDT_OLDNAME, acOld);
    if (wMsg != WM_INITDIALOG) {
      SetFocus(NULL);
    }
    SetDlgItemText(hwnd, IDE_NEWNAME, acOld);
    if (wMsg != WM_INITDIALOG) {
      SetFocus(GetDlgItem(hwnd, IDE_NEWNAME));
    }
    return TRUE;
  case WM_COMMAND:
    switch (wParam) {
    case IDOK:
      {
        LINKWND* plw = (LINKWND*)GetWindowLong(hwndChild, 0);
        assert(plw);
        assert(plw->hwnd == hwndChild);
        assert(plw->pDevice);
        char acNew[19];
        GetDlgItemText(hwnd, IDE_NEWNAME, acNew, 19);
        assert(strlen(acNew) <= 16);
        char* pcType = strchr(acNew, ',');
        if (pcType) {
          *pcType = 0;
        }
        char ac[80];
        if (strcmp(acOld, acNew)) {
          iRetries = 50;
          word wLength = wsprintf(ac, "R:%s=%s", acNew, acOld);
          assert(wLength <= 40);
          word wError = plw->pDevice->Command(ac, wLength);
          assert(!wError);
          wsprintf(ac, "%s\n\n", acNew);
          if (plw->pDevice->GetError(strend(ac), 41)) {
            ErrorBox(NULL, ac);
            return TRUE;
          }
        }
        assert(strlen(ac) <= 80);
        SendMessage(hwndListBox, LB_SETSEL, FALSE, i);
        UpdateMarked(plw);
        *pc = ',';
        assert(pc[1] && strchr("PSUDR", pc[1]));
        strcat(acNew, pc);
        assert(strlen(acNew) <= 18);
        plw->pDevice->GetDir(ac, 80);
        FileChange(ac, acOld, acNew, 0);
      }
      goto NextFile;
    case IDCANCEL:
      EndDialog(hwnd, FALSE);
      return TRUE;
    }
  }
  return FALSE;
}

// Summe der Blocks aller markierter Dateien holen
static long GetTaged(HWND hwndListBox) {
  long lReturn;
  if (SendMessage(hwndListBox, LB_GETSELCOUNT, 0, 0) > 0) {
    lReturn = 0;
    int i = (int)SendMessage(hwndListBox, LB_GETCOUNT, 0, 0);
    while (i-- > 0) {
      if (SendMessage(hwndListBox, LB_GETSEL, i, 0) > 0) {
        long l = SendMessage(hwndListBox, LB_GETITEMDATA, i, 0);
        assert(l >= 0);
        lReturn += l;
      }
    }
  } else {
    int i = (int)SendMessage(hwndListBox, LB_GETCARETINDEX, 0, 0);
    assert(i >= 0);
    lReturn = SendMessage(hwndListBox, LB_GETITEMDATA, i, 0);
  }
  return lReturn;
}

// berschreiben auswhlen lassen
static BOOL FAR PASCAL OverwriteDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  switch (wMsg) {
  case WM_INITDIALOG:
    {
      #if CHRIS
        int iBlocks = pSource->GetBlocks(2);
        if (iBlocks != -1) {
          SetDlgItemInt(hwnd, BLOCKS_ALT, iBlocks, TRUE);
        }
        dword dw = pSource->GetDateAndTime(2);
        if (dw != 0) {
          char ac[10];
          wsprintf(ac, "%02d.%02d.%02d", HIWORD(dw) & 31, (HIWORD(dw) >> 5) & 15, (1980 + (HIWORD(dw) >> 9)) % 100);
          SetDlgItemText(hwnd, DATUM_ALT, ac);
        }
        char* pc = acDest + strlen(acDest) - 2;
        if (*pc == ',') {
          *pc = 0;
          pDest->Open(2, acDest, pc - acDest);
          iBlocks = pDest->GetBlocks(2);
          if (iBlocks != -1) {
            SetDlgItemInt(hwnd, BLOCKS_NEU, iBlocks, TRUE);
          }
          dw = pDest->GetDateAndTime(2);
          if (dw != 0) {
            char ac[10];
            wsprintf(ac, "%02d.%02d.%02d", HIWORD(dw) & 31, (HIWORD(dw) >> 5) & 15, (1980 + (HIWORD(dw) >> 9)) % 100);
            SetDlgItemText(hwnd, DATUM_NEU, ac);
          }
          pDest->Close(2);
          *pc = ',';
          strcpy(acRunName, (LPSTR)lParam);
          char* pc = strchr(acRunName, ',');
          if (pc != NULL) {
            *pc = 0;
          }
        }
      #else
        HWND hwndIcon = CreateWindow("Static", IDI_QUESTION, WS_CHILD | WS_VISIBLE | SS_ICON, 1, 1, 8, 4, hwnd, 0, hInst, NULL);
        assert(hwndIcon);
        WinSetAttr(hwndIcon, MAKE_ATTR(BLACK, GREEN), FALSE);
      #endif
      SetDlgItemText(hwnd, IDT_FILENAME, (LPSTR)lParam);
      CenterWindow(hwnd);
      hwndMsgFilter = hwnd;
    }
    return TRUE;
  case WM_CHAR:
    {
      static word awSysKey[26] = {
        VK_ALT_A, VK_ALT_B, VK_ALT_C, VK_ALT_D, VK_ALT_E, VK_ALT_F, VK_ALT_G,
        VK_ALT_H, VK_ALT_I, VK_ALT_J, VK_ALT_K, VK_ALT_L, VK_ALT_M, VK_ALT_N,
        VK_ALT_O, VK_ALT_P, VK_ALT_Q, VK_ALT_R, VK_ALT_S, VK_ALT_T, VK_ALT_U,
        VK_ALT_V, VK_ALT_W, VK_ALT_X, VK_ALT_Y, VK_ALT_Z
      };
      int i = toupper(wParam) - 'A';
      if (i >= 0 && i < 26) {
        PostMessage(hwnd, WM_SYSKEYDOWN, awSysKey[i], 0);
        return TRUE;
      }
      return FALSE;
    }
  case WM_COMMAND:
    #if CHRIS
      if (wParam == TESTEN_ALT || wParam == TESTEN_NEU) {
        CDevice* pDevice;
        if (wParam == TESTEN_ALT) {
          pDevice = pSource;
        } else {
          pDevice = pDest;
        }
        assert(pDevice);
        iRetries = 50;
        pDevice->Open(0, acRunName, strlen(acRunName));
        char acError[40];
        if (pDevice->GetError(acError, 40)) {
          ErrorBox(NULL, acError);
          return TRUE;
        }
        lRunStart = (byte)pDevice->Get(0);
        lRunStart |= pDevice->Get(0) << 8;
        pDevice->Close(0);
        if (pDevice->GetError(acError, 40)) {
          ErrorBox(NULL, acError);
          return FALSE;
        }
        pDevice->GetDir(acRunDir, 80);
        HWND hwndEmulator = RunC64Program();
        SendMessage(hwndEmulator, WM_CLOSE, 0, 0);
        SetFocus(hwnd);
        SetFocus(GetDlgItem(hwnd, wParam));
        return TRUE;
      }
    #endif
    hwndMsgFilter = NULL;
    EndDialog(hwnd, wParam);
    return TRUE;
  }
  return FALSE;
}

// Dateien kopieren/verschieben
static BOOL FAR PASCAL CopyingDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  word wCount;
  word wEnd;
  switch (wMsg) {
  case WM_INITDIALOG:
    SetWindowText(hwnd, fMove ? acMoving : acCopying);
    SetDlgItemProgress(hwnd, IDT_PROGRESS, 0, 1);
    CenterWindow(hwnd);
    PostMessage(hwnd, WM_TIMER, 0, 0);
    return TRUE;
  case WM_TIMER:
    {
      word wRecord;
      word wIndex;
      int i;
      fAbort = FALSE;
      hwndDialog = hwnd;
      lPos = 0;
      pfnLinkAbortProc = LinkWndAbortProc;
      word wOverwrite = 0;
      assert(pSource);
      assert(pDest);
      assert(hwndListBox);
      int iFocus = -1;
      if (!SendMessage(hwndListBox, LB_GETSELCOUNT, 0, 0)) {
        iFocus = (int)SendMessage(hwndListBox, LB_GETCARETINDEX, 0, 0);
        assert(iFocus >= 0);
        SendMessage(hwndListBox, LB_SETSEL, TRUE, iFocus);
        UpdateMarked((LINKWND*)GetWindowLong(hwndChild, 0));
      }
      char ac[80];
      byte* pbBuffer = (byte*)malloc(gwBufSize);
      assert(pbBuffer);
      flag fReturn = FALSE;
      pSource->GetDir(ac, 80);
      if (!pSource->SetDir(ac)) {
        if (!fAbort) {
          pSource->GetError(ac, 80);
          ErrorBox(NULL, ac);
        }
        goto Stop;
      }
      i = 0;
      while (i < (int)SendMessage(hwndListBox, LB_GETCOUNT, 0, 0)) {
        if (!SendMessage(hwndListBox, LB_GETSEL, i, 0)) {
          i++;
          continue;
        }
        char acName[19];
        word w = (word)SendMessage(hwndListBox, LB_GETTEXT, i, (LONG)(LPSTR)acName);
        assert(w);
        flag fRel = FALSE;
        if (!strchr(acName, ',')) {
          strcpy(acName + w, ",P");
          w += 2;
        } else {
          assert(acName[w - 2] == ',');
          fRel = acName[w - 1] == 'R';
        }
        assert(w < 19);
        SetDlgItemText(hwnd, IDT_FILENAME, acName);
        long lSize = SendMessage(hwndListBox, LB_GETITEMDATA, i, 0);
        assert(lSize >= 0);
        long lNext = lPos + lSize;
        strcpy(ac, acName);
        strcat(ac, ",R");
        w = pSource->Open(2, ac, strlen(ac));
        if (w) {
          SetBusError(w, ac);
          goto Stop;
        }
        if (pSource->GetError(ac, 80)) {
          goto Stop;
        }
        if (fRel) {
          wRecord = pSource->GetRelLength(2);
          if (!wRecord) {
            ac[0] = 0;
            goto Stop;
          }
          if (pSource->GetError(ac, 80)) {
            goto Stop;
          }
          wIndex = 1;
          strcpy(acDest, acName);
          char* pc = strend(acDest) - 2;
          assert(*pc == ',');
          wsprintf(pc, ",L,%c", wRecord);
          pDest->Open(2, acName, strlen(acName));
          w = pDest->GetError(ac, 80);
          if (!w) {
            pDest->Close(2);
            goto FileExists;
          }
        } else {
          wRecord = gwBufSize;
          wsprintf(acDest, "%s,W", acName);
        }
      ReOpen:
        w = pDest->Open(2, acDest, strlen(acDest));
        if (w) {
          SetBusError(w, ac);
          goto Stop;
        }
        w = pDest->GetError(ac, 80);
        if (w == 63) {
        FileExists:
          switch (wOverwrite) {
          default:
            #if CHRIS
              wOverwrite = DialogBoxParam(hInst, "Chris", hwnd, (FARPROC)OverwriteDlgProc, (LONG)(LPSTR)acName);
              if (wOverwrite == IDNO) {
                pSource->Close(2);
                wsprintf(ac, "S:%s", acName);
                char* pc = strchr(ac, ',');
                assert(pc);
                *pc = 0;
                w = pSource->Command(ac, strlen(ac));
                pSource->GetDir(ac, 80);
                FileChange(ac, acName, NULL, 0);
              }
            #else
              wOverwrite = DialogBoxParam(hInst, "Overwrite", hwnd, (FARPROC)OverwriteDlgProc, (LONG)(LPSTR)acName);
            #endif
            switch (wOverwrite) {
            case IDCANCEL:
              fReturn = TRUE;
              goto Stop;
            case IDNO:
            case IDB_OVERWRITENONE:
              SendMessage(hwndListBox, LB_SETSEL, FALSE, i);
              UpdateMarked((LINKWND*)GetWindowLong(hwndChild, 0));
              goto Skip;
            }
          case IDB_OVERWRITEALL:
            {
              wsprintf(ac, "S:%s", acName);
              char* pc = strchr(ac, ',');
              assert(pc);
              *pc = 0;
              w = pDest->Command(ac, strlen(ac));
              if (w) {
                SetBusError(w, ac);
                goto Stop;
              }
              w = pDest->GetError(ac, 80);
              if (w == 1) {
                FileChange(acDestDir, acName, NULL, 0);
                goto ReOpen;
              }
            }
            goto Stop;
          case IDB_OVERWRITENONE:
            SendMessage(hwndListBox, LB_SETSEL, FALSE, i);
            UpdateMarked((LINKWND*)GetWindowLong(hwndChild, 0));
            goto Skip;
          }
        }
        if (w) {
          goto Stop;
        }
        do {
          if (fRel) {
            ac[0] = 'P';
            ac[1] = 2;
            *(word*)(ac + 2) = wIndex++;
            ac[4] = 1;
            pSource->Command(ac, 5);
            pDest->Command(ac, 5);
            w = pSource->GetError(ac, 40);
            if (w != 0) {
              if (w == 50) {
                goto EndOfFile;
              }
              goto Stop;
            }
            w = pDest->GetError(ac, 40);
            if (w != 0 && w != 50) {
              goto Stop;
            }
          }
          wCount = wRecord;
          wEnd = pSource->Read(2, pbBuffer, &wCount);
          if (wEnd & ~ENDOFFILE) {
            SetBusError(wEnd, ac);
            goto Stop;
          }
          if (pSource->GetError(ac, 80)) {
            goto Stop;
          }
          w = pDest->Write(2, pbBuffer, &wCount);
          if (w) {
            SetBusError(w, ac);
            goto Stop;
          }
          if (pDest->GetError(ac, 80)) {
            pDest->Close(2);
            char acDel[80];
            pDest->Command(acDel, wsprintf(acDel, "S:%s", acName) - 2);
            pDest->GetError(acDel, 80);
            goto Stop;
          }
          lPos += wRecord;
          if (lPos > lNext) {
            lPos = lNext;
          }
          SetDlgItemProgress(hwndDialog, IDT_PROGRESS, lPos, lTotal);
          if (fAbort) {
            pDest->Close(2);
            pDest->Command(ac, wsprintf(ac, "S:%s", acName) - 2);
            fReturn = TRUE;
            goto Stop;
          }
        } while (fRel || !(wEnd & ENDOFFILE));
      EndOfFile:
        {
          dword dwDateAndTime = pSource->GetDateAndTime(2);
          if (dwDateAndTime != 0) {
            pDest->SetDateAndTime(2, dwDateAndTime);
          }
        }
        w = pSource->Close(2);
        if (w) {
          SetBusError(w, ac);
          goto Stop;
        }
        w = pDest->Close(2);
        if (w) {
          SetBusError(w, ac);
          goto Stop;
        }
        if (fMove) {
          w = pSource->Command(ac, wsprintf(ac, "S:%s", acName) - 2);
          if (w) {
            SetBusError(w, ac);
            goto Stop;
          }
          w = pSource->GetError(ac, 80);
          if (w != 1) {
            goto Stop;
          }
          FileChange(acSource, acName, NULL, 0);
        } else {
          SendMessage(hwndListBox, LB_SETSEL, FALSE, i);
          UpdateMarked((LINKWND*)GetWindowLong(hwndChild, 0));
        }
        FileChange(acDestDir, NULL, acName, lSize);
      Skip:
        lPos = lNext;
        SetDlgItemProgress(hwndDialog, IDT_PROGRESS, lPos, lTotal);
        if (fAbort) {
          break;
        }
      }
      fReturn = TRUE;
    Stop:
      free(pbBuffer);
      pfnLinkAbortProc = DefAbortProc;
      if (iFocus >= 0) {
        SendMessage(hwndListBox, LB_SETSEL, FALSE, iFocus);
        UpdateMarked((LINKWND*)GetWindowLong(hwndChild, 0));
      }
      if (!fReturn && !fAbort && ac[0]) {
        ErrorBox(NULL, ac);
      }
      EndDialog(hwnd, fReturn);
    }
    return TRUE;
  case WM_COMMAND:
    fAbort = TRUE;
    return TRUE;
  }
  return FALSE;
}

// Zielverzeichnis auswhlen lassen
BOOL FAR PASCAL CopyMoveDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  switch (wMsg) {
  case WM_INITDIALOG:
    {
      fMove = (flag)lParam;
      assert(hwndChild);
      assert(GetWindowLong(hwndChild, GWL_WNDPROC) == (LONG)LinkWndProc);
      assert(!GetWindow(hwndChild, GW_HWNDPREV));
      LINKWND* plw = (LINKWND*)GetWindowLong(hwndChild, 0);
      assert(plw);
      assert(plw->hwndListBox);
      hwndListBox = plw->hwndListBox;
      lTotal = GetTaged(hwndListBox);
      assert(plw->pDevice);
      pSource = plw->pDevice;
      pSource->GetDir(acSource, 80);
      assert(acSource[0]);
      if (!acDestDir[0] || !strcmp(acDestDir, acSource)) {
        strcpy(acDestDir, ".");
        HWND hwnd = GetWindow(hwndChild, GW_HWNDNEXT);
        while (hwnd) {
          if (GetWindowLong(hwnd, GWL_WNDPROC) == (LONG)LinkWndProc) {
            plw = (LINKWND*)GetWindowLong(hwnd, 0);
            assert(plw);
            char ac[80];
            GetWindowText(plw->hwndDirTree, ac, 80);
            assert(ac[0]);
            if (strcmp(ac, acSource)) {
              strcpy(acDestDir, ac);
              break;
            }
          }
          hwnd = GetWindow(hwnd, GW_HWNDNEXT);
        }
      }
      SetDlgItemText(hwnd, fMove ? IDDT_MOVE : IDDT_COPY, acDestDir);
      CenterWindow(hwnd);
    }
    return TRUE;
  case WM_COMMAND:
    switch (wParam) {
    case IDOK:
      {
        GetDlgItemText(hwnd, fMove ? IDDT_MOVE : IDDT_COPY, acDestDir, 80);
        assert(acDestDir[0]);
        assert(strlen(acDestDir) < 80);
        if (!strcmp(acDestDir, acSource)) {
          ErrorBox(NULL, acSameDest);
          break;
        }
        pDest = NULL;
        flag fOK = SetDir(&pDest, acDestDir);
        assert(pDest);
        if (!fOK) {
          char ac[40];
          pDest->GetError(ac, 40);
          delete pDest;
          pDest = NULL;
          ErrorBox(NULL, ac);
          break;
        }
        EndDialog(hwnd, DialogBox(hInst, "Delete", hwnd, (FARPROC)CopyingDlgProc));
        delete pDest;
        pDest = NULL;
        pSource->Reset();
        return TRUE;
      }
    case IDCANCEL:
      EndDialog(hwnd, FALSE);
      return TRUE;
    }
  }
  return FALSE;
}

// Dateien lschen
BOOL FAR PASCAL DeleteDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  switch (wMsg) {
  case WM_INITDIALOG:
    CenterWindow(hwnd);
    PostMessage(hwnd, WM_TIMER, 0, 0);
    return TRUE;
  case WM_TIMER:
    {
      fAbort = FALSE;
      hwndDialog = hwnd;
      lPos = 0;
      pfnLinkAbortProc = LinkWndAbortProc;
      assert(hwndChild);
      assert(GetWindowLong(hwndChild, GWL_WNDPROC) == (LONG)LinkWndProc);
      LINKWND* plw = (LINKWND*)GetWindowLong(hwndChild, 0);
      assert(plw);
      assert(plw->hwnd == hwndChild);
      assert(plw->pDevice);
      int iFocus = -1;
      lTotal = SendMessage(plw->hwndListBox, LB_GETSELCOUNT, 0, 0);
      if (!lTotal) {
        iFocus = (int)SendMessage(plw->hwndListBox, LB_GETCARETINDEX, 0, 0);
        assert(iFocus >= 0);
        SendMessage(plw->hwndListBox, LB_SETSEL, TRUE, iFocus);
        UpdateMarked(plw);
        lTotal = 1;
      }
      int i;
      flag fReturn = FALSE;
      char ac[80];
      plw->pDevice->GetDir(ac, 80);
      if (!plw->pDevice->SetDir(ac)) {
        if (!fAbort) {
          plw->pDevice->GetError(ac, 80);
          ErrorBox(NULL, ac);
        }
        goto Stop;
      }
      for (i = 0; i < (int)SendMessage(plw->hwndListBox, LB_GETCOUNT, 0, 0); i++) {
        while (SendMessage(plw->hwndListBox, LB_GETSEL, i, 0)) {
          char acName[40];
          SendMessage(plw->hwndListBox, LB_GETTEXT, i, (LONG)(LPSTR)acName);
          if (!strchr(acName, ',')) {
            strcat(acName, ",P");
          }
          word wLength = wsprintf(ac, "S:%s", acName) - 2;
          assert(wLength);
          assert(wLength < 40);
          assert(ac[wLength] == ',');
          ac[wLength] = 0;
          SetDlgItemText(hwnd, IDT_FILENAME, ac + 2);
          SetDlgItemProgress(hwndDialog, IDT_PROGRESS, lPos++, lTotal);
          if (fAbort) {
            goto Stop;
          }
          word wError = plw->pDevice->Command(ac, wLength);
          if (fAbort) {
            goto Stop;
          }
          assert(!wError);
          wsprintf(ac, "%s\n\n", acName);
          if (plw->pDevice->GetError(strend(ac), 41) != 1 || !strstr(ac, ",01,")) {
            ErrorBox(NULL, ac);
            goto Stop;
          }
          plw->pDevice->GetDir(ac, 80);
          FileChange(ac, acName, NULL, 0);
        }
      }
      SetDlgItemProgress(hwndDialog, IDT_PROGRESS, 1, 1);
      fReturn = TRUE;
    Stop:
      if (iFocus >= 0 && !fReturn) {
        SendMessage(plw->hwndListBox, LB_SETSEL, FALSE, iFocus);
        UpdateMarked(plw);
      }
      pfnLinkAbortProc = DefAbortProc;
      EndDialog(hwnd, fReturn);
    }
    return TRUE;
  case WM_COMMAND:
    fAbort = TRUE;
    return TRUE;
  }
  return FALSE;
}

// C64-Header hinzufgen
flag LinkImport() {
  assert(IsWindow(hwndChild));
  assert(GetWindowLong(hwndChild, GWL_WNDPROC) == (LONG)LinkWndProc);
  LINKWND* plw = (LINKWND*)GetWindowLong(hwndChild, 0);
  assert(plw);
  assert(plw->hwnd == hwndChild);
  assert(plw->pDevice);
  pDest = plw->pDevice;
  OPENFILENAME ofn;
  memset(&ofn, 0, sizeof OPENFILENAME);
  ofn.lStructSize = sizeof OPENFILENAME;
  ofn.hwndOwner = hwndChild;
  ofn.hInstance = hInst;
  ofn.lpstrFilter = acImportFilter;
  ofn.nFilterIndex = 1;
  ofn.lpstrFile = acSource;
  ofn.nMaxFile = 80;
  ofn.lpstrTitle = acImport;
  ofn.Flags = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_PATHMUSTEXIST;
  acSource[0] = 0;
  if (!::acImportDir[0]) {
    strcpy(::acImportDir, ::acDir);
  }
  _chdrive(::acImportDir[0] - '@');
  _chdir(::acImportDir);
  if (!GetOpenFileName(&ofn)) {
    return FALSE;
  }
  _getcwd(::acImportDir, 80);
  int hFile = _lopen(acSource, OF_READ | OF_SHARE_DENY_NONE);
  if (hFile <= 0) {
    return ErrorBox(acSource, NULL);
  }
  char acComp[8];
  if (_lread(hFile, acComp, 8) == 8 && !strcmp(acComp, "C64File")) {
    _lclose(hFile);
    return ErrorBox(acSource, acIsAlreadyC64);
  }
  _lclose(hFile);
  return DialogBox(hInst, "Import", hwndFrame, (FARPROC)ImportDlgProc);
}

BOOL FAR PASCAL ImportDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  static flag fCopying;
  switch (wMsg) {
  case WM_INITDIALOG:
    {
      SetDlgItemProgress(hwnd, IDT_PROGRESS, 0, 1);
      char* pc = strrchr(acSource, '\\');
      assert(pc);
      strcpy(acDest, pc + 1);
      pc = strchr(acDest, '.');
      int iRecord = atoi(pc + 1);
      char acRecord[10];
      wsprintf(acRecord, ".%d", iRecord);
      int iType = 0;
      if (pc) {
        if (!strcmp(pc, ".PRG") || !strcmp(pc, ".RUN")) {
          *pc = 0;
        } else if (!strcmp(pc, ".SEQ")) {
          iType = 1;
          *pc = 0;
        } else if (!strcmp(pc, ".USR")) {
          iType = 2;
          *pc = 0;
        } else if (!strcmp(pc, ".DEL")) {
          iType = 3;
          *pc = 0;
        } else if (!strcmp(pc, ".REL") || !strcmp(pc, acRecord)) {
          iType = 4;
          *pc = 0;
        }
      }
      assert(strlen(acDest) < 19);
      SetDlgItemText(hwnd, IDE_NEWNAME, acDest);
      SendDlgItemMessage(hwnd, IDE_NEWNAME, EM_LIMITTEXT, 16, 0);
      CheckRadioButton(hwnd, IDB_TYPE, IDB_TYPE + 4, IDB_TYPE + iType);
      if (iRecord > 0 && iRecord < 256) {
        SetDlgItemInt(hwnd, IDE_RECORD, iRecord, TRUE);
      }
    }
    fCopying = FALSE;
    CenterWindow(hwnd);
    return TRUE;
  case WM_COMMAND:
    switch (wParam) {
    case IDOK:
      {
        int iType;
        for (iType = 0; iType < 5; iType++) {
          if (IsDlgButtonChecked(hwnd, IDB_TYPE + iType)) {
            break;
          }
        }
        assert(iType < 5);
        word wRecord;
        if (iType == 4) {
          wRecord = GetDlgItemInt(hwnd, IDE_RECORD, NULL, FALSE);
          if (wRecord < 1 || wRecord > 255) {
            ErrorBox(NULL, acWrongRecordLen);
            SetFocus(GetDlgItem(hwnd, IDE_RECORD));
            return TRUE;
          }
        }
        int iCount;
        int hFile;
        char acError[80];
        word wIndex;
        word w;
        fCopying = TRUE;
        fAbort = FALSE;
        hwndDialog = hwnd;
        lPos = 0;
        pfnLinkAbortProc = LinkWndAbortProc;
        GetDlgItemText(hwnd, IDE_NEWNAME, acDest, 17);
        assert(strlen(acDest) < 17);
        char* pc = strchr(acDest, ',');
        if (!pc) {
          pc = strend(acDest);
        }
        *pc++ = ',';
        *pc++ = "PSUDL"[iType];
        *pc++ = ',';
        if (iType == 4) {
          *pc++ = (char)wRecord;
        } else {
          *pc++ = 'W';
        }
        *pc++ = 0;
        char* pcBuffer = (char*)malloc(gwBufSize);
        assert(pcBuffer);
        if (iType == 4) {
          pc = strend(acDest) - 2;
          assert(*pc == ',');
          *pc = 0;
          pDest->Open(2, acDest, strlen(acDest));
          *pc = ',';
          w = pDest->GetError(acError, 40);
          if (!w) {
            pDest->Close(2);
            goto FileExists;
          }
        }
      ReOpen:
        w = pDest->Open(2, acDest, strlen(acDest));
        assert(w == 0);
        w = pDest->GetError(acError, 40);
        if (w) {
          if (w == 63) {
          FileExists:
            w = MessageBox(hwndFrame, acOverwrite, acConfirm, MB_ICONQUESTION | MB_YESNO);
            if (w == IDYES) {
              wsprintf(acError, "S:%s", acDest);
              char* pc = strchr(acError, ',');
              assert(pc);
              pDest->Command(acError, pc - acError);
              w = pDest->GetError(acError, 40);
              if (w == 1) {
                pDest->GetDir(acError, 80);
                pc = strend(acDest) - 2;
                assert(*pc == ',');
                *pc = 0;
                FileChange(acError, acDest, NULL, 0);
                *pc = ',';
                goto ReOpen;
              }
              ErrorBox(NULL, acError);
            }
            goto Stop;
          } else {
            ErrorBox(NULL, acError);
            goto Stop;
          }
        }
        hFile = _lopen(acSource, OF_READ | OF_SHARE_DENY_NONE);
        if (hFile <= 0) {
          pDest->Close(2);
          ErrorBox(acSource, NULL);
          goto Stop;
        }
        lTotal = _filelength(hFile);
        if (!lTotal) {
          lTotal = 1;
        }
        if (iType == 4) {
          wIndex = 1;
        } else {
          wRecord = gwBufSize;
        }
        for (;;) {
          iCount = _lread(hFile, pcBuffer, wRecord);
          if (!iCount) {
            break;
          }
          if (iCount < 0) {
            ErrorBox(acSource, NULL);
            _lclose(hFile);
            pDest->Close(2);
            goto Stop;
          }
          if (iType == 4) {
            acError[0] = 'P';
            acError[1] = 2;
            *(word*)(acError + 2) = wIndex++;
            acError[4] = 1;
            pDest->Command(acError, 5);
            w = pDest->GetError(acError, 40);
            if (w != 0 && w != 50) {
              ErrorBox(NULL, acError);
              _lclose(hFile);
              pDest->Close(2);
              goto Stop;
            }
            while (iCount > 1 && pcBuffer[iCount - 1] == -1) {
              iCount--;
            }
          }
          w = pDest->Write(2, (byte*)pcBuffer, (word*)&iCount);
          assert(w == 0);
          w = pDest->GetError(acError, 40);
          if (w) {
            ErrorBox(NULL, acError);
            _lclose(hFile);
            pDest->Close(2);
            goto Stop;
          }
          lPos += wRecord;
          if (lPos > lTotal) {
            lPos = lTotal;
          }
          SetDlgItemProgress(hwndDialog, IDT_PROGRESS, lPos, lTotal);
        }
        SetDlgItemProgress(hwndDialog, IDT_PROGRESS, 1, 1);
        _lclose(hFile);
        w = pDest->Close(2);
        assert(w == 0);
        pDest->GetDir(acError, 80);
        pc = strend(acDest) - 2;
        assert(*pc == ',');
        *pc = 0;
        if (iType == 4) {
          assert(pc[-1] == 'L');
          pc[-1] = 'R';
        }
        FileChange(acError, NULL, acDest, lTotal);
        EndDialog(hwnd, TRUE);
      Stop:
        free(pcBuffer);
        pfnLinkAbortProc = DefAbortProc;
        fCopying = FALSE;
      }
      return TRUE;
    case IDCANCEL:
      if (fCopying) {
        fAbort = TRUE;
      } else {
        EndDialog(hwnd, FALSE);
      }
      return TRUE;
    }
  }
  return FALSE;
}

// C64-Header entfernen
static int iType;
static word wRecord;
flag LinkExport() {
  assert(IsWindow(hwndChild));
  assert(GetWindowLong(hwndChild, GWL_WNDPROC) == (LONG)LinkWndProc);
  LINKWND* plw = (LINKWND*)GetWindowLong(hwndChild, 0);
  assert(plw->hwnd == hwndChild);
  assert(plw->hwndDirTree);
  assert(plw->hwndListBox);
  assert(plw->pDevice);
  pSource = plw->pDevice;
  int iFocus = -1;
  if (!SendMessage(plw->hwndListBox, LB_GETSELCOUNT, 0, 0)) {
    iFocus = (int)SendMessage(plw->hwndListBox, LB_GETCARETINDEX, 0, 0);
    if (iFocus >= 0) {
      SendMessage(plw->hwndListBox, LB_SETSEL, TRUE, iFocus);
      UpdateMarked(plw);
    }
  }
  char* pcBuffer = (char*)malloc(gwBufSize);
  assert(pcBuffer);
  flag fReturn = FALSE;
  for (int i = 0; i < (int)SendMessage(plw->hwndListBox, LB_GETCOUNT, 0, 0); i++) {
    while (SendMessage(plw->hwndListBox, LB_GETSEL, i, 0)) {
      char acDefExt[5];
      SendMessage(plw->hwndListBox, LB_GETTEXT, i, (LONG)(LPSTR)acSource);
      if (!strchr(acSource, ',')) {
        strcat(acSource, ",P");
      }
      assert(strlen(acSource) < 19);
      char* pc = strchr(acSource, ',');
      assert(pc);
      *pc = 0;
      char* pcExt = strrchr(acSource, '.');
      if (pcExt && strlen(pcExt) == 4) {
        strcpy(acDefExt, pcExt);
        *pcExt = 0;
        ReduceName(acSource, acDest);
        *pcExt = '.';
      } else {
        memset(acDefExt, 0, sizeof acDefExt);
        ReduceName(acSource, acDest);
      }
      *pc++ = ',';
      switch (*pc) {
      case 'P':
        if (!acDefExt[0]) {
          strcpy(acDefExt, ".PRG");
        }
        iType = 0;
        break;
      case 'S':
        if (!acDefExt[0]) {
          strcpy(acDefExt, ".SEQ");
        }
        iType = 1;
        break;
      case 'U':
        if (!acDefExt[0]) {
          strcpy(acDefExt, ".USR");
        }
        iType = 2;
        break;
      case 'D':
        if (!acDefExt[0]) {
          strcpy(acDefExt, ".DEL");
        }
        iType = 3;
        break;
      case 'R':
        assert(plw->pDevice);
        assert(pc[-1] == ',');
        pc[-1] = 0;
        plw->pDevice->Open(2, acSource, strlen(acSource));
        wRecord = plw->pDevice->GetRelLength(2);
        plw->pDevice->Close(2);
        if (!wRecord) {
          goto Stop;
        }
        wsprintf(acDefExt, ".%d", wRecord);
        iType = 4;
        break;
      }
      strcat(acDest, acDefExt);
      OPENFILENAME ofn;
      memset(&ofn, 0, sizeof OPENFILENAME);
      ofn.lStructSize = sizeof OPENFILENAME;
      ofn.hwndOwner = hwndChild;
      ofn.hInstance = hInst;
      ofn.lpstrFilter = acImportFilter;
      ofn.nFilterIndex = 1;
      ofn.lpstrFile = acDest;
      ofn.nMaxFile = 80;
      ofn.lpstrTitle = acSource;
      ofn.Flags = OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY | OFN_PATHMUSTEXIST;
      ofn.lpstrDefExt = acDefExt + 1;
      if (!::acImportDir[0]) {
        strcpy(::acImportDir, ::acDir);
      }
      _chdrive(::acImportDir[0] - '@');
      _chdir(::acImportDir);
      if (!GetSaveFileName(&ofn)) {
        goto Stop;
      }
      _getcwd(::acImportDir, 80);
      hFile = _lcreat(acDest, 0);
      if (hFile <= 0) {
        ErrorBox(acDest, NULL);
        goto Stop;
      }
      lTotal = SendMessage(plw->hwndListBox, LB_GETITEMDATA, i, 0);
      if (!DialogBox(hInst, "Exporting", hwndFrame, (FARPROC)ExportingDlgProc)) {
        _lclose(hFile);
        remove(acDest);
        goto Stop;
      }
      if (_lclose(hFile) != 0) {
        ErrorBox(acDest, NULL);
        goto Stop;
      }
      SendMessage(plw->hwndListBox, LB_SETSEL, FALSE, i);
      UpdateMarked(plw);
    }
  }
  fReturn = TRUE;
Stop:
  if (iFocus >= 0) {
    SendMessage(plw->hwndListBox, LB_SETSEL, FALSE, iFocus);
    UpdateMarked(plw);
  }
  free(pcBuffer);
  return fReturn;
}

BOOL FAR PASCAL ExportingDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  switch (wMsg) {
  case WM_INITDIALOG:
    SetDlgItemText(hwnd, IDT_OLDNAME, acSource);
    SetDlgItemProgress(hwnd, IDT_PROGRESS, 0, 1);
    CenterWindow(hwnd);
    PostMessage(hwnd, WM_TIMER, 0, 0);
    return TRUE;
  case WM_TIMER:
    {
      word wIndex;
      word wCount;
      word wEnd;
      fAbort = FALSE;
      hwndDialog = hwnd;
      if (!lTotal) {
        lTotal = 1;
      }
      lPos = 0;
      flag fReturn = FALSE;
      char acError[40];
      byte* pbBuffer = (byte*)malloc(gwBufSize);
      assert(pbBuffer);
      pfnLinkAbortProc = LinkWndAbortProc;
      word w = pSource->Open(2, acSource, strlen(acSource));
      if (w) {
        SetBusError(w, acError);
        ErrorBox(NULL, acError);
        goto Stop;
      }
      if (pSource->GetError(acError, 40)) {
        ErrorBox(NULL, acError);
        goto Stop;
      }
      if (iType == 4) {
        // wRecord ist bereits gesetzt
        wIndex = 1;
      } else {
        wRecord = gwBufSize;
      }
      do {
        if (iType == 4) {
          acError[0] = 'P';
          acError[1] = 2;
          *(word*)(acError + 2) = wIndex++;
          acError[4] = 1;
          pSource->Command(acError, 5);
          w = pSource->GetError(acError, 40);
          if (w != 0) {
            if (w == 50) {
              goto EndOfFile;
            }
            ErrorBox(NULL, acError);
            goto Stop;
          }
          memset(pbBuffer, 255, wRecord);
        }
        wCount = wRecord;
        wEnd = pSource->Read(2, pbBuffer, &wCount);
        if (iType == 4) {
          wCount = wRecord;
        }
        if (wEnd & ~ENDOFFILE) {
          SetBusError(wEnd, acError);
          ErrorBox(NULL, acError);
          goto Stop;
        }
        if (pSource->GetError(acError, 40)) {
          ErrorBox(NULL, acError);
          goto Stop;
        }
        w = _lwrite(hFile, pbBuffer, wCount);
        if (w == (word)-1) {
          ErrorBox(acDest, NULL);
          goto Stop;
        }
        if (w != wCount) {
          ErrorBox(acDest, apcDOSError[39]);
          goto Stop;
        }
        lPos += wRecord;
        if (lPos > lTotal) {
          lPos = lTotal;
        }
        SetDlgItemProgress(hwndDialog, IDT_PROGRESS, lPos, lTotal);
        if (fAbort) {
          goto Stop;
        }
      } while (iType == 4 || !(wEnd & ENDOFFILE));
    EndOfFile:
      w = pSource->Close(2);
      if (w) {
        SetBusError(w, acError);
        ErrorBox(NULL, acError);
        goto Stop;
      }
      if (pSource->GetError(acError, 40)) {
        ErrorBox(NULL, acError);
        goto Stop;
      }
      SetDlgItemProgress(hwndDialog, IDT_PROGRESS, 1, 1);
      fReturn = TRUE;
    Stop:
      free(pbBuffer);
      pfnLinkAbortProc = DefAbortProc;
      if (!fReturn) {
        pSource->Reset();
      }
      EndDialog(hwnd, fReturn);
    }
    return TRUE;
  case WM_COMMAND:
    fAbort = TRUE;
    return TRUE;
  }
  return FALSE;
}

LONG FAR PASCAL LinkWndProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  LINKWND* plw = (LINKWND*)GetWindowLong(hwnd, 0);
  HWND hwndFocus = GetFocus();
  if (plw && (hwndFocus == plw->hwndDirTree || hwndFocus == plw->hwndListBox)) {
    plw->hwndFocus = hwndFocus;
  }
  switch (wMsg) {
  case WM_CREATE:
    SetWindowLong(hwnd, 0, lSetWindowLong);
    plw = (LINKWND*)lSetWindowLong;
    plw->hwnd = hwnd;
    plw->hwndDirTree = CreateWindow("DirTree", NULL,
      WS_CHILD | DTS_SORT | DTS_NOTIFY,
      0, 0, 0, 0, hwnd, 1, hInst, NULL);
    assert(plw->hwndDirTree);
    WinSetAttr(plw->hwndDirTree, def.awColor[DTC_DIRTREEFORE] | def.awColor[DTC_DIRTREEBACK] << 4, FALSE);
    plw->hwndFocus = plw->hwndDirTree;
    plw->hwndListBox = CreateWindow("ListBox", NULL,
      WS_CHILD | WS_HSCROLL | LBS_NOTIFY | LBS_MULTICOLUMN | LBS_MULTIPLESEL,
      0, 0, 0, 0, hwnd, 2, hInst, NULL);
    assert(plw->hwndListBox);
    WinSetAttr(plw->hwndListBox, def.awColor[DTC_LISTBOXFORE] | def.awColor[DTC_LISTBOXBACK] << 4, FALSE);
    SendMessage(plw->hwndListBox, LB_SETCOLUMNWIDTH, 20, 0L);
    wChildren++;
    fDoArrange = TRUE;
    break;
  case WM_PAINT:
    assert(plw);
    {
      PAINTSTRUCT ps;
      HDC hdc = BeginPaint(hwnd, &ps);
      MoveTo(hdc, plw->iSplit, 0);
      LineTo(hdc, plw->iSplit, 1000);
      EndPaint(hwnd, &ps);
    }
    break;
  case WM_SIZE:
    assert(plw);
    if (!plw) {
      goto Default;
    }
    plw->iSplit = LOWORD(lParam) / 3;
    if (plw->iSplit < 8) {
      plw->iSplit = 8;
    } else {
      if (plw->iSplit > 20) {
        plw->iSplit = 20;
      }
    }
    assert(plw->hwndDirTree);
    MoveWindow(plw->hwndDirTree, 0, 0, plw->iSplit, HIWORD(lParam), TRUE);
    assert(plw->hwndListBox);
    MoveWindow(plw->hwndListBox, plw->iSplit + 1, 0, LOWORD(lParam) - plw->iSplit - 1, HIWORD(lParam), FALSE);
    SendMessage(plw->hwndListBox, LB_SETCARETINDEX, (word)SendMessage(plw->hwndListBox, LB_GETCARETINDEX, 0, 0), 0);
    goto Default;
  case WM_MDIACTIVATE:
    if (wParam) {
      hwndChild = hwnd;
      hwndPassKeys = hwnd;
    } else {
      assert(hwndChild == hwnd);
      hwndChild = NULL;
      assert(hwndPassKeys == hwnd);
      hwndPassKeys = NULL;
    }
    goto Default;
  case WM_SETFOCUS:
    assert(plw);
    assert(plw->hwndFocus);
    SetFocus(plw->hwndFocus);
    break;
  case WM_PASSKEYS:
    {
      assert(plw);
      assert(hwnd == plw->hwnd);
      if (hwndFocus != plw->hwndDirTree && hwndFocus != plw->hwndListBox) {
        return FALSE;
      }
      switch (wParam) {
      case VK_F1:
        Help(hwndFrame, hwndFocus == plw->hwndDirTree ? IDH_DIRTREE : IDH_LISTBOX);
        return TRUE;
      case VK_TAB:
      case VK_SH_TAB:
        SetFocus(hwndFocus ^ plw->hwndDirTree ^ plw->hwndListBox);
        return TRUE;
      case VK_RETURN:
        if (hwndFocus == plw->hwndListBox) {
          LinkExecute();
          return TRUE;
        }
        break;
      case VK_INSERT:
      case ' ':
        if (hwndFocus == plw->hwndListBox) {
          word wIndex = (word)SendMessage(plw->hwndListBox, LB_GETCARETINDEX, 0, 0);
          flag fSelect = (flag)SendMessage(plw->hwndListBox, LB_GETSEL, wIndex, 0);
          SendMessage(plw->hwndListBox, LB_SETSEL, !fSelect, wIndex);
          SendMessage(plw->hwndListBox, LB_SETCARETINDEX, wIndex + 1, 0);
          UpdateMarked(plw);
          return TRUE;
        }
        break;
      case '+':
      case '-':
      case '*':
        if (hwndFocus == plw->hwndListBox) {
          word wCount = (word)SendMessage(plw->hwndListBox, LB_GETCOUNT, 0, 0);
          for (word w = 0; w < wCount; w++) {
            flag fSelect;
            if (wParam == '*') {
              fSelect = !(flag)SendMessage(plw->hwndListBox, LB_GETSEL, w, 0);
            } else {
              fSelect = wParam == '+';
            }
            SendMessage(plw->hwndListBox, LB_SETSEL, fSelect, w);
          }
          UpdateMarked(plw);
          return TRUE;
        }
        break;
      case VK_F5:
        PostMessage(hwndFrame, WM_COMMAND, IDM_LINKREFRESH, 0);
        return TRUE;
      default:
        if (wParam < 256 && hwndFocus == plw->hwndListBox) {
          int iFocus = (int)SendMessage(plw->hwndListBox, LB_GETCARETINDEX, 0, 0);
          if (iFocus >= 0) {
            int iCount = (int)SendMessage(plw->hwndListBox, LB_GETCOUNT, 0, 0);
            if (iCount > 0) {
              assert(iFocus < iCount);
              char c = (char)(LONG)AnsiUpper((LPSTR)(LONG)wParam);
              int i = iFocus;
              for (;;) {
                if (++i >= iCount) {
                  i = 0;
                }
                if (i == iFocus) {
                  break;
                }
                char ac[19];
                SendMessage(plw->hwndListBox, LB_GETTEXT, i, (LONG)(LPSTR)ac);
                if (!strchr(ac, ',')) {
                  strcat(ac, ",P");
                }
                assert(strlen(ac) < 19);
                if (ac[0] == c) {
                  SendMessage(plw->hwndListBox, LB_SETCARETINDEX, i, 0);
                  break;
                }
              }
            }
          }
          return TRUE;
        }
      }
    }
    break;
  case WM_COMMAND:
    switch (wParam) {
    case 1: // DirTree
      switch (HIWORD(lParam)) {
      case DTN_SETFOCUS:
        assert(plw);
        plw->hwndFocus = plw->hwndDirTree;
        break;
      case DTN_SELCHANGE:
        assert(plw);
        DeleteNames(plw);
        plw->hwndFocus = plw->hwndDirTree;
        if (GetFocus() == plw->hwndListBox) {
          SetFocus(plw->hwndDirTree);
        }
        break;
      }
      break;
    case 2: // ListBox
      switch (HIWORD(lParam)) {
      case LBN_SETFOCUS:
        assert(plw);
        plw->hwndFocus = plw->hwndListBox;
        if (!plw->pDevice) {
          if (!ReadNames(plw)) {
            plw->hwndFocus = plw->hwndDirTree;
            SetFocus(plw->hwndDirTree);
          }
        }
        break;
      case LBN_SELCHANGE:
        UpdateMarked(plw);
        break;
      case LBN_DBLCLK:
        LinkExecute();
        break;
      }
      break;
    }
    break;
  case WM_SYSCOMMAND:
    switch (wParam & 0xFFF0) {
    case SC_MINIMIZE:
    case SC_RESTORE:
      fDoArrange = TRUE;
    }
    goto Default;
  case WM_DESTROY:
    assert(plw);
    if (plw->pDevice) {
      delete plw->pDevice;
    }
    free(plw);
    if (--wChildren) {
      fDoArrange = TRUE;
    }
    goto Default;
  default:
  Default:
    return DefMDIChildProc(hwnd, wMsg, wParam, lParam);
  }
  return 0;
}

// Fehler am Ende des Diskettenkopierens anzeigen
static void ShowErrors(HWND hwndParent) {
  if (pSource->GetType() == DEVIMAGE && pDest->GetType() == DEVIMAGE) {
    return;
  }
  char ac[768];
  ac[0] = 0;
  for (byte b = 2; b < 16; b++) {
    flag fErrorCode = FALSE;
    for (int i = 0; i < 683; i++) {
      if (gpbError[i] == b) {
        if (!ac[0]) {
          #if GERMAN
            strcpy(ac, "Auf der Quelldiskette befinden sich folgende Lesefehler:\n");
          #else
            strcpy(ac, "Source disk contains the following read errors:\n");
          #endif
        }
        int iTrack = 1;
        int iSector = i;
        int iSectors;
        for (;;) {
          if (iTrack < 18) {
            iSectors = 21;
          } else if (iTrack < 25) {
            iSectors = 19;
          } else if (iTrack < 31) {
            iSectors = 18;
          } else {
            iSectors = 17;
          }
          if (iSector < iSectors) {
            break;
          }
          iSector -= iSectors;
          iTrack++;
        }
        if (fErrorCode) {
          strcat(ac, " ");
        } else {
          switch (b) {
          case 2:
            strcat(ac, "\n20,READ ERROR:");
            break;
          case 3:
            strcat(ac, "\n21,READ ERROR:");
            break;
          case 4:
            strcat(ac, "\n22,READ ERROR:");
            break;
          case 5:
            strcat(ac, "\n23,READ ERROR:");
            break;
          case 6:
            strcat(ac, "\n24,READ ERROR:");
            break;
          case 9:
            strcat(ac, "\n27,READ ERROR:");
            break;
          case 11:
            strcat(ac, "\n29,DISK ID MISMATCH:");
            break;
          case 15:
            strcat(ac, "\n74,DRIVE NOT READY:");
            break;
          default:
            sprintf(strend(ac), "\nUnknown job error code $%02X:", b);
            break;
          }
          fErrorCode = TRUE;
        }
        sprintf(strend(ac), " %d,%d", iTrack, iSector);
        for (int iCount = 1; iSector + iCount < iSectors; iCount++) {
          if (gpbError[i + iCount] != b) {
            break;
          }
        }
        if (--iCount) {
          sprintf(strend(ac), "-%d", iSector + iCount);
          i += iCount;
        }
      }
    }
  }
  if (ac[0]) {
    if (pDest->GetType() == DEVLINK) {
      #if GERMAN
        sprintf(strend(ac), "\n\nDa diese Fehler nicht mitkopiert wurden, funktioniert eventuell der Kopierschutz nicht mehr!");
      #else
        sprintf(strend(ac), "\n\nBecause these errors are not copied, maybe a copy protection doesn't work anymore!");
      #endif
      MessageBeep(MB_ICONEXCLAMATION);
      #if GERMAN
        MessageBox(hwndParent, ac, "Warnung", MB_ICONEXCLAMATION | MB_OK);
      #else
        MessageBox(hwndParent, ac, "Warning", MB_ICONEXCLAMATION | MB_OK);
      #endif
    } else {
      #if GERMAN
        sprintf(strend(ac), "\n\nDie Fehler wurden auf das Diskettenimage kopiert.");
      #else
        sprintf(strend(ac), "\n\nThe errors have been copied to the disk image.");
      #endif
      MessageBeep(MB_ICONINFORMATION);
      #if GERMAN
        MessageBox(hwndParent, ac, "Hinweis", MB_ICONINFORMATION | MB_OK);
      #else
        MessageBox(hwndParent, ac, "Info", MB_ICONINFORMATION | MB_OK);
      #endif
    }
  }
}

// Diskettenimage kopieren
static BOOL FAR PASCAL CopyingImageDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  switch (wMsg) {
  case WM_INITDIALOG:
    SetWindowText(hwnd, acCopying);
    SetDlgItemProgress(hwnd, IDT_PROGRESS, 0, 1);
    CenterWindow(hwnd);
    PostMessage(hwnd, WM_TIMER, 0, 0);
    return TRUE;
  case WM_TIMER:
    {
      word wTrack;
      fAbort = FALSE;
      hwndDialog = hwnd;
      lPos = 0;
      lTotal = 683L * 256;
      pfnLinkAbortProc = LinkWndAbortProc;
      assert(pSource);
      assert(pDest);
      flag fReturn = FALSE;
      byte* pbBuffer = (byte*)malloc(21 * 256);
      assert(pbBuffer);
      gpbError = (byte*)malloc(683);
      assert(gpbError);
      memset(gpbError, 1, 683);
      if (!pSource->BeginTrack(FALSE)) {
        goto Error0;
      }
      if (!pDest->BeginTrack(TRUE)) {
        goto Error1;
      }
      for (wTrack = 1; wTrack <= 35; wTrack++) {
        word wBytes = pSource->ReadNextTrack(pbBuffer);
        if (!wBytes) {
          goto Error2;
        }
        if (!pDest->WriteNextTrack(pbBuffer, wBytes)) {
          goto Error2;
        }
        lPos += wBytes;
        SetDlgItemProgress(hwndDialog, IDT_PROGRESS, lPos, lTotal);
        if (fAbort) {
          goto Error2;
        }
      }
      fReturn = TRUE;
    Error2:
      pfnLinkAbortProc = DefAbortProc;
      iRetries = 50;
      pDest->EndTrack();
    Error1:
      pfnLinkAbortProc = DefAbortProc;
      iRetries = 50;
      pSource->EndTrack();
    Error0:
      if (fReturn) {
        ShowErrors(hwnd);
      }
      pfnLinkAbortProc = DefAbortProc;
      free(gpbError);
      gpbError = NULL;
      free(pbBuffer);
      EndDialog(hwnd, fReturn);
    }
    return TRUE;
  case WM_COMMAND:
    fAbort = TRUE;
    return TRUE;
  }
  return FALSE;
}

BOOL FAR PASCAL CopyImageDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  static char acName[13];
  switch (wMsg) {
  case WM_INITDIALOG:
    {
      CenterWindow(hwnd);
      LINKWND* plw = (LINKWND*)GetWindowLong(hwndChild, 0);
      assert(plw);
      assert(plw->hwnd == hwndChild);
      GetWindowText(plw->hwndDirTree, acSource, 80);
      pSource = NULL;
      SetDir(&pSource, acSource);
      assert(pSource);
      assert(pSource->GetType() != DEVFLOPPY);
      SetDlgItemText(hwnd, IDDT_COPY, acDest);
      if (pSource->GetType() == DEVIMAGE) {
        char* pc = strrchr(acSource, '\\');
        assert(pc);
        strcpy(acName, pc + 1);
        pc = strrchr(acName, '.');
        assert(pc);
        *pc = 0;
      } else if (acName[0] == 0) {
        char acPath[80];
        strcpy(acPath, acDest);
        char* pcPath = strend(acPath);
        if (pcPath[-1] != '\\') {
          *pcPath++ = '\\';
        }
        for (int i = 1; i < 100; i++) {
          wsprintf(acName, "DISK%02d", i);
          strcpy(pcPath, acName);
          strcat(pcPath, ".D64");
          if (_access(acPath, 0) != 0) {
            break;
          }
        }
      }
      SetDlgItemText(hwnd, IDE_DOSNAME, acName);
      return TRUE;
    }
  case WM_COMMAND:
    switch (wParam) {
    case IDDT_COPY:
      if (HIWORD(lParam) == DTN_SELCHANGE) {
        GetDlgItemText(hwnd, IDDT_COPY, acDest, 80);
        flag fDisable = IsLink(acDest) || IsImage(acDest);
        EnableWindow(GetDlgItem(hwnd, IDE_DOSNAME), !fDisable);
        EnableWindow(GetDlgItem(hwnd, IDT_EXTENSION), !fDisable);
        return TRUE;
      }
      return FALSE;
    case IDOK:
      {
        assert(pSource);
        assert(pSource->GetType() != DEVFLOPPY);
        GetDlgItemText(hwnd, IDDT_COPY, acDest, 80);
        GetDlgItemText(hwnd, IDE_DOSNAME, acName, 9);
        acName[8] = 0;
        char* pcEnd = strend(acDest);
        flag fRemove = FALSE;
        if (IsLink(acDest)) {
          if (!strcmp(acSource, acDest)) {
            ErrorBox(NULL, acSameDest);
            return TRUE;
          }
        } else {
          if (IsImage(acDest)) {
          Prompt:
            if (!strcmp(acSource, acDest)) {
              ErrorBox(acDest, acSameDest);
              *pcEnd = 0;
              return TRUE;
            }
            char ac[160];
            wsprintf(ac, acOverwriteImage, acDest);
            MessageBeep(MB_ICONQUESTION);
            if (MessageBox(hwndFrame, ac, acConfirm, MB_ICONQUESTION | MB_OKCANCEL) != IDOK) {
              *pcEnd = 0;
              return TRUE;
            }
          } else {
            if (pcEnd[-1] != '\\') {
              strcpy(pcEnd, "\\");
            }
            strcat(pcEnd, acName);
            strcat(pcEnd, ".D64");
            int hFile = _open(acDest, _O_BINARY | _O_CREAT | _O_EXCL | _O_RDWR, _S_IREAD | _S_IWRITE);
            if (hFile < 0) {
              if (errno == EEXIST) {
                goto Prompt;
              }
              ErrorBox(acDest, NULL);
              *pcEnd = NULL;
              return TRUE;
            }
            char* pbBuffer = (char*)malloc(16384);
            assert(pbBuffer);
            memset(pbBuffer, 1, 16384);
            for (int i = 0; i < 16384; i += 256) {
              pbBuffer[i] = 0x4B;
            }
            SetError(EWRITE);
            long l = 174848;
            while (l) {
              int i = 16384;
              if ((long)i > l) {
                i = (int)l;
              }
              if (_lwrite(hFile, pbBuffer, i) != i) {
                free(pbBuffer);
                ErrorBox(acDest, NULL);
                _close(hFile);
                remove(acDest);
                *pcEnd = 0;
                return TRUE;
              }
              l -= i;
            }
            free(pbBuffer);
            _close(hFile);
            fRemove = TRUE;
          }
        }
        pDest = NULL;
        SetDir(&pDest, acDest);
        assert(pDest);
        assert(pDest->GetType() != DEVFLOPPY);
        flag fOK = DialogBox(hInst, "Delete", hwnd, (FARPROC)CopyingImageDlgProc);
        delete pDest;
        if (fOK) {
          UpdateAllDirTrees(NULL, NULL);
          int i = strlen(acName);
          if (acName[i - 1] >= '0' && acName[i - 1] <= '9') {
            for (;;) {
              i--;
              if (i < 0 || !isdigit(acName[i])) {
                if (strlen(acName) < 8) {
                  i++;
                  memmove(acName + i + 1, acName + i, strlen(acName + i) + 1);
                }
                if (i < 0) {
                  break;
                }
                acName[i] = '0';
              }
              if (i < 0) {
                break;
              }
              if (acName[i] != '9') {
                acName[i]++;
                break;
              }
              acName[i] = '0';
            }
          }
        } else {
          if (fRemove) {
            remove(acDest);
          }
        }
        *pcEnd = 0;
      }
    case IDCANCEL:
      delete pSource;
      pSource = NULL;
      EndDialog(hwnd, 0);
      return TRUE;
    }
  }
  return FALSE;
}
