// Emulation der Commodore Floppy 1541 auf Festplatte

#include "pc64.h"

// Fehlermeldung setzen
static flag fQuiet;
flag CFloppy::SetError(ferr e) {
  if (wRunDebug & DEV && !fQuiet) {
    char sError[40];
    eError = e;
    GetError(sError, 40);
    WARNING (DEV) "%s!", sError);
  }
  eError = e;
  return FALSE;
}

// Fehlercode von DOS nach C64 wandeln
flag CFloppy::DOSError(int hFile) {
  switch (errno) {
  case EBADF:
  case EACCES:
    SetError(WriteProt);
    break;
  case EEXIST:
    SetError(Exists);
    break;
  case EMFILE:
    SetError(NoChannel);
    break;
  case ENOENT:
    SetError(NotFound);
    break;
  case ENOSPC:
    SetError(NoSpace);
    break;
  default:
    SetError(Unknown);
  }
  if (hFile > 0) {
    _lclose(hFile);
  }
  return FALSE;
}

// Auf grogeschriebenen Vokal prfen
static inline flag isvocal(char c) {
  return memchr("AEIOU", c, 5) != NULL;
}

// Dateinamen von 16 Zeichen auf 8 Zeichen reduzieren
flag ReduceName(const char* sC64Name, char* pDOSName) {
  int iStart;
  // Lnge berprfen
  int iLen = strlen(sC64Name);
  if (iLen > 16) {
    return FALSE;
  }
  // Bei einem Jokerzeichen alle Dateien durchsuchen
  if (strpbrk(sC64Name, "*?")) {
    strcpy(pDOSName, "*");
    return TRUE;
  }
  // In Puffer umkopieren
  char sBuf[16 + 1];
  memset(sBuf, 0, 16);
  strcpy(sBuf, sC64Name);
  // Gltige Zeichen holen
  for (int i = 0; i <= 15; i++) {
    switch (sBuf[i]) {
    case ' ':
    case '-':
      // Leerzeichen durch Unterstrich ersetzen
      sBuf[i] = '_';
      break;
    default:
      // Kleinbuchstaben gro machen
      if (islower((byte)sBuf[i])) {
        sBuf[i] -= 32;
        break;
      }
      // Grobuchstaben und Ziffern sind OK
      if (isalnum((byte)sBuf[i])) {
        break;
      }
      // Ungltige Zeichen entfernen
      if (sBuf[i]) {
        sBuf[i] = 0;
        iLen--;
      }
    }
  }
  // Namen auf 8 Zeichen bringen
  if (iLen <= 8) {
    goto Copy;
  }
  // Unterstriche entfernen
  for (i = 15; i >= 0; i--) {
    if (sBuf[i] == '_') {
      sBuf[i] = 0;
      if (--iLen <= 8) {
        goto Copy;
      }
    }
  }
  // Ersten Nicht-Vokal suchen
  for (iStart = 0; iStart < 15; iStart++) {
    if (sBuf[iStart] && !isvocal(sBuf[iStart])) {
      break;
    }
  }
  // Vokale entfernen
  for (i = 15; i >= iStart; i--) {
    if (isvocal(sBuf[i])) {
      sBuf[i] = 0;
      if (--iLen <= 8) {
        goto Copy;
      }
    }
  }
  // Konsonanten entfernen
  for (i = 15; i >= 0; i--) {
    if (isalpha(sBuf[i])) {
      sBuf[i] = 0;
      if (--iLen <= 8) {
        goto Copy;
      }
    }
  }
  // briggebliebene Ziffern entfernen
  for (i = 0; i <= 15; i++) {
    if (sBuf[i]) {
      sBuf[i] = 0;
      if (--iLen <= 8) {
        goto Copy;
      }
    }
  }
Copy:
  // Dummy-Namen bei Lnge = 0 erfinden
  if (!iLen) {
    strcpy(pDOSName, "_");
    return TRUE;
  }
  // Dateinamen umkopieren
  char* p = pDOSName;
  for (i = 0; i <= 15; i++) {
    if (sBuf[i]) {
      *p++ = sBuf[i];
    }
  }
  *p = 0;
  // DOS-Namen auf Gert prfen
  int hFile = _open(pDOSName, _O_BINARY | _O_RDONLY);
  if (hFile == -1) {
    return TRUE;
  }
  // Wenn es ein Gertename ist, dann '_' als letztes Zeichen anhngen
  if (isatty(hFile)) {
    if (iLen < 8) {
      strcat(pDOSName, "_");
    } else if (pDOSName[7] != '_') {
      pDOSName[7] = '_';
    } else {
      pDOSName[7] = 'X';
    }
  }
  // Zur Probe geffnetes Gert wird nicht mehr gebraucht
  _close(hFile);
  return TRUE;
}

// Dateinamen analysieren und Parameter setzen
flag CFloppy::SetName(channel* pch, char* pcName) {
  // Struktur mit Standardwerten initialisieren
  pch->sPattern[0] = 0;
  pch->fDirectory = (flag)(*pcName == '$');
  pch->eType = (pch == ch + 0 || pch == ch + 1) ? PRG : ANY;
  pch->bRecord = 0;
  eMode = (pch == ch + 1) ? eWRITE : eREAD;
  fReplace = FALSE;
  // Maximal 41 Zeichen sind als Dateiname zulssig
  if (strlen(pcName) > 41) {
    return SetError(TooLong);
  }
  // Bei Inhaltsverzeichnissen keine Beschrnkung des Dateityps
  if (pch->fDirectory) {
    pch->eType = ANY;
    // Nachgestellte Unit-Nummer 0 oder 1 entfernen
    switch (*++pcName) {
    case '1':
      return SetError(NotReady);
    case '0':
      pcName++;
    }
    // Wenn Spezifikation fehlt, dann Verzeichnis aller Dateien holen
    if (!*pcName) {
      strcpy(pch->sPattern, "*");
      return TRUE;
    }
  }
  // Auf Replace und Unit prfen
  const char* p = strchr(pcName, ':');
  if (p) {
    if (*pcName == '@') {
      fReplace = TRUE;
    }
    if (p > pcName && p[-1] == '1') {
      return SetError(NotReady);
    }
    p++;
  } else {
    p = pcName;
  }
  // Dateinamen bertragen
  char* pDest = pch->sPattern;
  fWildcards = FALSE;
  for (;;) {
    char cSeparator;
    switch (*p) {
    case 0:
      goto EndOfName;
    case '=':
    case ',':
      cSeparator = *p;
      // Ende bei "Name,S,W" oder "$Name=P", danach Modifizierer auswerten
      for (;;) {
        switch (*++p) {
        case 'P':
          pch->eType = PRG;
          break;
        case 'S':
          pch->eType = SEQ;
          break;
        case 'U':
          pch->eType = USR;
          break;
        case 'D':
          pch->eType = DEL;
          break;
        case 'R':
          if (cSeparator != '=') {
            eMode = eREAD;
            break;
          }
        case 'L':
          pch->eType = REL;
          while (*p++ != ',') {
            if (!*p) {
              goto EndOfName;
            }
          }
          pch->bRecord = (byte)*p;
          goto EndOfName;
        case 'M':
          eMode = eREAD;
          break;
        case 'W':
          eMode = eWRITE;
          break;
        case 'A':
          eMode = eAPPEND;
          break;
        }
        while (*p != ',') {
          if (!*++p) {
            goto EndOfName;
          }
        }
      }
    case '?':
    case '*':
      // Jokerzeichen gefunden
      fWildcards = TRUE;
    default:
      // Nchstes Zeichen bertragen, dabei auf maximal 16 beschrnken
      if (pDest < pch->sPattern + 16) {
        *pDest++ = *p;
      }
    }
    p++;
  }
EndOfName:
  *pDest = 0;
  // Leere Dateinamen sind nur bei Verzeichnissen erlaubt
  if (pDest == pch->sPattern && !pch->fDirectory) {
    return SetError(NoName);
  }
  // Ein Verzeichnis darf nur gelesen werden
  if (pch->fDirectory && eMode != eREAD) {
    return SetError(WrongType);
  }
  // Beim Schreiben sind keine Jokerzeichen erlaubt
  if (eMode == eWRITE && fWildcards) {
    return SetError(Joker);
  }
  // Ersetzen mit "@:" nur beim Schreiben zulssig
  if (fReplace && eMode != eWRITE) {
    return SetError(WrongType);
  }
  return TRUE;
}

// Nchste passende Datei suchen und ffnen
flag CFloppy::FindFile(channel* pc, find_t* pFind, word* pwFind) {
  int i;
  // Solange noch nicht alle Dateien im Verzeichnis abgearbeitet wurden
  while (!*pwFind) {
    // Aktuelle Parameter sichern und nchsten Eintrag suchen
    strcpy(pName, pFind->name);
    long lSize = pFind->size;
    *pwFind = (word)_dos_findnext(pFind);
    // Ist die Erweiterung 3 Zeichen lang mit zwei Ziffern hintendran?
    char* p = strchr(pName, '.');
    if (!p || strlen(p) != 4 || !isdigit(p[2]) || !isdigit(p[3])) {
      continue;
    }
    // Dateityp anhand des ersten Buchstabens feststellen
    switch (p[1]) {
    case 'P':
      eType = PRG;
      break;
    case 'S':
      eType = SEQ;
      break;
    case 'U':
      eType = USR;
      break;
    case 'D':
      eType = DEL;
      break;
    case 'R':
      eType = REL;
      break;
    default:
      continue;
    }
    // Gefundene Datei ffnen und auf gltigen C64-Header prfen
    flag fReadOnly = FALSE;
    int hFile = _lopen(sPath, READ_WRITE | OF_SHARE_DENY_NONE);
    if (hFile == -1) {
      fReadOnly = TRUE;
      hFile = _lopen(sPath, READ | OF_SHARE_DENY_NONE);
      if (hFile == -1) {
        return DOSError(hFile);
      }
    }
    int iCount = _lread(hFile, (char*)&hdr, sizeof hdr);
    if (iCount == -1) {
      return DOSError(hFile);
    }
    if (iCount != sizeof hdr || strcmp(hdr.sTag, "C64File")) {
      goto NextFile;
    }
    // Namensvergleich mit der 16-Zeichen-Maske durchfhren
    for (i = 0; i <= 15; i++) switch (pc->sPattern[i]) {
    case 0:
      if (hdr.sName[i]) {
        goto NextFile;
      }
    case '*':
      goto Found;
    case '?':
      if (!hdr.sName[i]) {
        goto NextFile;
      }
      break;
    default:
      if (hdr.sName[i] != pc->sPattern[i]) {
        goto NextFile;
      }
    }
  Found:
    // Dateitypen vergleichen
    if (eType != pc->eType && pc->eType != ANY) {
      if (pc->fDirectory) {
        goto NextFile;
      }
      _lclose(hFile);
      return SetError(WrongType);
    }
    // Anzahl der Blocks bestimmen
    lSize = (lSize - sizeof hdr + 253) / 254;
    if (lSize < 65535L) {
      wBlocks = (word)lSize;
    } else {
      wBlocks = 65535U;
    }
    // Zeiger je nach Dateimodus positionieren und zurck
    switch (eMode) {
    case eAPPEND:
      if (fReadOnly) {
        _lclose(hFile);
        return SetError(WriteProt);
      }
      lseek(hFile, 0L, SEEK_END);
    case eREAD:
      pc->hFile = hFile;
      return TRUE;
    case eWRITE:
      if (!fReplace) {
        _lclose(hFile);
        return SetError(Exists);
      }
      if (fReadOnly || chsize(hFile, sizeof hdr)) {
        _lclose(hFile);
        return SetError(WriteProt);
      }
      wBlocks = 0;
      pc->hFile = hFile;
      return TRUE;
    }
  NextFile:
    // Das war die falsche Datei, weiter versuchen
    _lclose(hFile);
  }
  if (*pwFind != 0x12) {
    // Es gab einen Fehler
    return SetError(NotReady);
  }
  // Datei nicht gefunden, dann vielleicht neu anlegen?
  if (eMode != eWRITE && pc->bRecord == 0) {
    if (pc->fDirectory) {
      return FALSE;
    } else {
      return SetError(NotFound);
    }
  }
  // Erweiterung an den Namen anhngen
  char* p = strchr(pName, '.');
  if (!p) {
    p = pName + strlen(pName);
  }
  eType = pc->eType;
  switch (pc->eType) {
  case PRG:
    strcpy(p, ".P00");
    break;
  case USR:
    strcpy(p, ".U00");
    break;
  case DEL:
    strcpy(p, ".D00");
    break;
  case REL:
    strcpy(p, ".R00");
    break;
  default:
    strcpy(p, ".S00");
    eType = SEQ;
    break;
  }
  // Eine freie Ziffernkombination von 00 bis 99 suchen
  for (;;) {
    if (access(sPath, 0) == -1) {
      break;
    }
    if (++p[3] > '9') {
      p[3] = '0';
      if (++p[2] > '9') {
        return SetError(NoSpace);
      }
    }
  }
  // Neue Datei erzeugen und C64-Header schreiben
  int hFile = _lcreat(sPath, 0);
  if (hFile == -1) {
    return DOSError(hFile);
  }
  memset(&hdr, 0, sizeof hdr);
  strcpy(hdr.sTag, "C64File");
  strcpy(hdr.sName, pc->sPattern);
  hdr.bRecord = pc->bRecord;
  if (_lwrite(hFile,(char*)&hdr, sizeof hdr) != sizeof hdr) {
    return DOSError(hFile);
  }
  // Datei wurde erfolgreich angelegt
  pc->hFile = hFile;
  return TRUE;
}

// C64-Datei als DOS-Datei ffnen
flag CFloppy::ChOpen(channel* pch) {
  assert(pch);
  assert(pch->sPattern[0]);
  NoError();
  if (!ReduceName(pch->sPattern, pName)) {
    SetError(TooLong);
    return FALSE;
  }
  strcat(pName, ".*");
  // Datei suchen und ffnen
  find_t Temp;
  word wTemp = _dos_findfirst(sPath, 0x20, &Temp);
  return FindFile(pch, &Temp, &wTemp);
}

// Dateityp im Klartext zurckgeben
const char* CFloppy::TypeStr(ftype eType) {
  switch (eType) {
  case PRG:
    return "PRG";
  case SEQ:
    return "SEQ";
  case USR:
    return "USR";
  case DEL:
    return "DEL";
  case REL:
    return "REL";
  default:
    return "ANY";
  }
}

// Relative Dateien vom Ende bis zur neuen Position mit 255 auffllen
void FillRecord(int hFile) {
  long lPos = _tell(hFile);
  assert(lPos != -1);
  long lEnd = _lseek(hFile, 0, SEEK_END);
  assert(lEnd != -1);
  if (lEnd < lPos) {
    char* pcBuffer = (char*)malloc(1024);
    assert(pcBuffer);
    memset(pcBuffer, 255, 1024);
    do {
      long lSize = lPos - lEnd;
      assert(lSize >= 0);
      if (lSize > 1024) {
        lSize = 1024;
      }
      _lwrite(hFile, pcBuffer, (int)lSize);
      lEnd += lSize;
    } while (lEnd < lPos);
    free(pcBuffer);
    assert(lEnd == lPos);
  } else {
    _lseek(hFile, lPos, SEEK_SET);
  }
}

// Pufferinhalt auf Diskette schreiben
word CFloppy::Flush(channel* pc) {
  assert(pc->hFile > 0);
  int iCount = pc->wOut - pc->wIn;
  if (iCount > 0) {
    if (pc->bRecord) {
      FillRecord(pc->hFile);
    }
    if (_lwrite(pc->hFile, pc->pbBuffer + pc->wIn, iCount) != iCount) {
      DOSError(-1);
      return EOUTPUT;
    }
    pc->wIn = pc->wOut;
  }
  return 0;
}

// Konstruktor
CFloppy::CFloppy() {
  // Aktuelles Verzeichnis holen
  if (getcwd(sPath, 64)) {
    pName = sPath + strlen(sPath);
    if (pName[-1] != '\\') * pName++ = '\\';
  } else {
    sPath[0] = 0;
    pName = sPath;
  }
  // Es sind keine Dateien offen
  for (int i = 0; i <= 14; i++) {
    memset(ch + i, 0, sizeof channel);
  }
  // Rest der Initialisierung vornehmen
  Reset();
}

// Destruktor
CFloppy::~CFloppy() {
  Stop();
}

// Typ zurckgeben
word CFloppy::GetType() {
  return DEVFLOPPY;
}

// Neues Verzeichnis setzen
flag CFloppy::SetDir(char* pDir) {
  // Laufwerk hinzufgen und ".\" sowie "..\" auflsen
  char acDir[80];
  _fullpath(acDir, pDir, 80);
  strupr(pDir);
  // Verzeichnis mit '\*' abschlieen
  int i = strlen(acDir);
  if (acDir[i - 1] != '\\') {
    strcpy(acDir + i, "\\*");
    i++;
  } else {
    strcpy(acDir + i, "*");
  }
  // Prfen, ob Verzeichnis existiert
  find_t Dummy;
  word w = _dos_findfirst(acDir, 0, &Dummy);
  // Neues Verzeichnis setzen
  acDir[i] = 0;
  strcpy(sPath, acDir);
  pName = sPath + i;
  return w == 0 || w == 18;
}

// Aktuelles Verzeichnis auslesen
void CFloppy::GetDir(char* pDir, word wMax) {
  // Verzeichnislnge ohne abschlieenden '\' holen
  word wLen = (word)(pName - sPath);
  if (wLen > 3) {
    wLen--;
  }
  if (wLen <= wMax) {
    // Zeichenkette entweder normal umkopieren
    memcpy(pDir, sPath, wLen);
    pDir[wLen] = 0;
  } else {
    // oder bei berlnge ab erstem Verzeichnis krzen -> "X:\...\DIR"
    memcpy(pDir, sPath, 3);
    memcpy(pDir + 3, "...", 3);
    word w = wLen - wMax + 6;
    while (sPath[w] != '\\') {
      if (++w >= wLen) {
        w = wLen - wMax + 6;
        break;
      }
    }
    memcpy(pDir + 6, sPath + w, wLen - w);
    pDir[6 + wLen - w] = 0;
  }
}

// Dateigre holen
long CFloppy::GetFileSize(word wChannel) {
  int hFile = ch[wChannel].hFile;
  if (hFile > 0) return _filelength(hFile) - sizeof hdr;
  else return 0;
}

// Floppy zurcksetzen
word CFloppy::Reset() {
  // DOS-Dateien schlieen und Speicher freigeben
  Stop();
  // Keine C64-Datei geffnet
  for (int i = 0; i <= 14; i++) {
    ch[i].sPattern[0] = 0;
    ch[i].fDirectory = FALSE;
  }
  // Kein Inhaltsverzeichnis geffnet
  wFind = 0xFFFF;
  // Einschaltmeldung setzen
  eError = ResetMsg;
  return 0;
}

// Fehler als Text im VC1541-Format holen
word CFloppy::GetError(char* pBuffer, word wMax) {
  const char* sText;
  char sMsg[80];
  switch (eError) {
  case OK:
    sText = " OK";
    break;
  case Scratched:
    sprintf(sMsg, "01, FILES SCRATCHED,%02u,00", wScratched);
    goto Copy;
  case NoHdr:
  case NoSync:
  case WrongHdr:
  case WrongData:
  case CheckSum:
  case CheckHdr:
    sText = "READ ERROR";
    break;
  case Verify:
  case WriteErr:
    sText = "WRITE ERROR";
    break;
  case WriteProt:
    sText = "WRITE PROTECT ON";
    break;
  case WrongDisk:
    sText = "DISK ID MISMATCH";
    break;
  case NoCmd:
  case WrongCmd:
  case TooLong:
  case Joker:
  case NoName:
    sText = "SYNTAX ERROR";
    break;
  case CantExec:
  case NotFound:
    sText = "FILE NOT FOUND";
    break;
  case NoRecord:
    sText = "RECORD NOT PRESENT";
    break;
  case Overflow:
    sText = "OVERFLOW IN RECORD";
    break;
  case TooLarge:
    sText = "FILE TOO LARGE";
    break;
  case NotClosed:
    sText = "WRITE FILE OPEN";
    break;
  case NotOpen:
    sText = "FILE NOT OPEN";
    break;
  case Exists:
    sText = "FILE EXISTS";
    break;
  case WrongType:
    sText = "FILE TYPE MISMATCH";
    break;
  case NoBlock:
    sText = "NO BLOCK";
    break;
  case NoSector:
  case WrongBAM:
    sText = "ILLEGAL TRACK OR SECTOR";
    break;
  case NoChannel:
    sText = "NO CHANNEL";
    break;
  case WrongDir:
    sText = "DIR ERROR";
    break;
  case NoSpace:
    sText = "DISK FULL";
    break;
  case ResetMsg:
    sText = "PC64 VERSION " VERSION BETA;
    break;
  case NotReady:
    sText = "DRIVE NOT READY";
    break;
  default:
    sText = "UNKNOWN ERROR";
    break;
  }
  sprintf(sMsg, "%02u,%s,00,00", eError, sText);
Copy:
  strncpy(pBuffer, sMsg, wMax);
  word wReturn = (word)eError;
  NoError();
  return wReturn;
}

// Zustand des Laufwerks in Image speichern
flag CFloppy::Save(int hFile) {
  if(_lwrite(hFile, &eError, sizeof ferr) != sizeof ferr) return FALSE;
  if(_lwrite(hFile, &wScratched, sizeof word) != sizeof word) return FALSE;
  if(_lwrite(hFile, ch, sizeof ch) != sizeof ch) return FALSE;
  return TRUE;
}

// Zustand des Laufwerks aus Image zurckholen
flag CFloppy::Load(int hFile) {
  if(_lread(hFile, &eError, sizeof ferr) != sizeof ferr) return FALSE;
  if(_lread(hFile, &wScratched, sizeof word) != sizeof word) return FALSE;
  if(_lread(hFile, ch, sizeof ch) != sizeof ch) return FALSE;
  wFind = 0xFFFF;
  return TRUE;
}

// Offene C64-Dateien auch unter DOS ffnen
word CFloppy::Start() {
  for (int i = 0; i <= 14; i++) {
    if (ch[i].sPattern[0] && !ch[i].fDirectory) {
      // Name ins DOS-Format wandeln
      if (!ReduceName(ch[i].sPattern, pName)) {
        SetError(TooLong);
      }
      strcat(sPath, ".*");
      // Fehler beim Schreiben und Esc/F5 vermeiden
      // eMode wird hier nicht gebraucht, Modus interessiert nur beim ffnen
      eMode = eREAD;
      // Datei suchen und ffnen
      find_t Temp;
      word wTemp = _dos_findfirst(sPath, 0x20, &Temp);
      if (FindFile(ch + i, &Temp, &wTemp)) {
        // Puffer bereitstellen
        ch[i].wIn = ch[i].wOut = 0;
        if ((ch[i].pbBuffer = (byte*)malloc(1024)) == NULL) {
          goto Error;
        }
        // Auf gesicherte Position fahren
        lseek(ch[i].hFile, ch[i].lPos, 0);
      } else {
      Error:
        // Datei nicht mehr geffnet
        if (ch[i].hFile) {
          _lclose(ch[i].hFile);
        }
        ch[i].sPattern[0] = 0;
      }
    }
  }
  return 0;
}

// Offene DOS-Dateien schlieen
word CFloppy::Stop() {
  for (int i = 0; i <= 14; i++) {
    if (ch[i].sPattern[0] && ch[i].hFile > 0) {
      // Ungesicherte Daten schreiben
      Flush(ch + i);
      // Zugehrigen Puffer freigeben
      free(ch[i].pbBuffer);
      ch[i].pbBuffer = NULL;
      // Position innerhalb der Datei holen
      ch[i].lPos = tell(ch[i].hFile) - ch[i].wIn + ch[i].wOut;
      // Datei unter DOS schlieen
      if (_lclose(ch[i].hFile) == -1) {
        DOSError(-1);
      }
    }
    // Datei nicht geffnet
    ch[i].hFile = 0;
  }
  return 0;
}

// Eine Datei oder ein Verzeichnis ffnen
word CFloppy::Open(word wChannel, char* pcName, word wLength) {
  // Nur Kanle von 0 bis 14 sind zugelassen
  if (wChannel > 14) {
    SetError(NoChannel);
    return 0;
  }
  // Eventuell offene Datei oder Verzeichnis schlieen
  if (ch[wChannel].hFile) {
    Close(wChannel);
  }
  ch[wChannel].fDirectory = FALSE;
  // Dateinamen analysieren
  NoError();
  if (!SetName(ch + wChannel, pcName)) {
    return 0;
  }
  if (ch[wChannel].fDirectory) {
    iDir = iDirLen = 0;
    // Ist eine Diskette eingelegt?
    strcpy(pName, "*.*");
    wFind = _dos_findfirst(sPath, 0, &Find);
    if (wFind && wFind != 18) {
      SetError(NotReady);
      wFind = 0xFFFF;
      return 0;
    }
    // Als nchstes Diskettennamen lesen
    wFind |= 0x8000;
    return 0;
  }
  // Kein Verzeichnis ffnen, dann Name ins DOS-Format wandeln
  if (!ReduceName(ch[wChannel].sPattern, pName)) {
    SetError(TooLong);
    return 0;
  }
  strcat(sPath, ".*");
  // Datei suchen und ffnen
  find_t Temp;
  word wTemp = _dos_findfirst(sPath, 0x20, &Temp);
  if (FindFile(ch + wChannel, &Temp, &wTemp)) {
    // Dateityp bertragen
    ch[wChannel].eType = eType;
    ch[wChannel].bRecord = hdr.bRecord;
    // Puffer bereitstellen
    ch[wChannel].wIn = ch[wChannel].wOut = 0;
    if ((ch[wChannel].pbBuffer = (byte*)malloc(1024)) == NULL) {
      _lclose(ch[wChannel].hFile);
      SetError(NoSpace);
    }
  }
  return 0;
}

// Ein Zeichen in die Datei schreiben
word CFloppy::Put(word wChannel, byte bValue) {
  // Prfen, ob die Datei geffnet ist
  channel* pc = ch + wChannel;
  if (wChannel > 14 || !pc->hFile) {
    return EOUTPUT;
  }
  // Zeichen in den Puffer schreiben
  pc->pbBuffer[pc->wOut++] = bValue;
  // Wenn Puffer voll, dann auf Diskette schreiben
  if (pc->wOut < 1024) {
    return 0;
  }
  word wReturn = Flush(pc);
  pc->wIn = pc->wOut = 0;
  return wReturn;
}

// Ende Schreiben
word CFloppy::Unlisten() {
  return 0;
}

// Ein Zeichen aus der Datei lesen
word CFloppy::Get(word wChannel) {
  char* q;
  // Zulssige Kanle liegen zwischen 0 und 14
  channel* pc = ch + wChannel;
  if (wChannel > 14) return EINPUT;
  // Verzeichnis wird gelesen
  if (ch[wChannel].fDirectory) {
    if (iDir >= iDirLen) {
      char* p = (char*)abDir;
      switch (wFind >> 8) {
      // Inverse Kopfzeile erzeugen
      case 0x80:
        // Anfangsadresse
        *(word*)p = 0x0401;
        p += 2;
        // Verbindung zur Folgezeile
        *(word*)p = 0x0101;
        p += 2;
        // Zeilennummer
        *(word*)p = 0;
        p += 2;
        // Aktuelles Verzeichnis holen
        char sDir[17];
        GetDir(sDir, 16);
        strupr(sDir);
        // Alle '\' durch '/' ersetzen
        for (q = sDir; *q; q++) {
          if (*q == '\\') {
            *q = '/';
          }
        }
        p += sprintf(p, "\022\"%-16.16s\" E%X.%02X", sDir, HEXVER >> 8, HEXVER & 0x00FF) + 1;
        // Kopfzeilenkennung lschen
        wFind &= 0xFF;
        break;
      // Normaler Verzeichniseintrag
      default:
        // Nchste Datei suchen
        if (FindFile(ch + wChannel, &Find, &wFind)) {
          // Nur der Name wird gebraucht
          _lclose(ch[wChannel].hFile);
          ch[wChannel].hFile = 0;
          // Verbindung zur Folgezeile
          *(word*)p = 0x0101;
          p += 2;
          // Zeilennummer = Dateigre
          *(word*)p = wBlocks;
          p += 2;
          // Auf 4 Stellen auffllen
          q = p + 27;
          if (wBlocks < 1000) {
            *p++ = ' ';
            if (wBlocks < 100) {
              *p++ = ' ';
              if (wBlocks < 10) *p++ = ' ';
            }
          }
          if (!strchr(hdr.sName, '"')) {
            strcat(hdr.sName, "\"");
          }
          p += sprintf(p, "\"%-17.17s %s", hdr.sName, TypeStr(eType));
          while (p < q) {
            *p++ = ' ';
          }
          *p++ = 0;
        // Kein Eintrag mehr vorhanden
        } else {
          // Verbindung zur Folgezeile
          *(word*)p = 0x0101;
          p += 2;
          // Anzahl der freien Blocks holen
          diskfree_t df;
          _dos_getdiskfree(sPath[0] - '@', &df);
          dword dBlocks = (dword)df.avail_clusters * df.sectors_per_cluster * df.bytes_per_sector / 254;
          if (dBlocks < 65535L) *(word*)p = (word)dBlocks;
          else *(word*)p = 65535U;
          p += 2;
          p += sprintf(p, "BLOCKS FREE.             ");
          // Verzeichnis zuende
          *(word*)p = 0;
          p += 2;
          // Keine Daten mehr vorhanden
          wFind = 0xFE00;
        }
        break;
      case 0xFE:
        // Ende erreicht
        wFind = 0xFFFF;
        // Abschlieende 0 von BYTES free.
        return ENDOFFILE;
      case 0xFF:
        // Ende berschritten
        return EINPUT;
      }
      // Zeiger auf Anfang
      iDir = 0;
      // Lnge der Zeile
      iDirLen = (byte*)p - abDir;
    }
    // Zeichen aus Puffer holen
    return abDir[iDir++];
  }
  // Normale Datei lesen
  if (!pc->hFile) {
    return EINPUT;
  }
  // Puffer auffllen, falls leer
  if (pc->wOut >= pc->wIn) {
    int iSize = 1024;
    if (pc->bRecord) {
      iSize = pc->bRecord;
    }
    int iRead = _lread(pc->hFile, pc->pbBuffer, iSize);
    switch (iRead) {
    case -1:
      DOSError(-1);
    case 0:
      if (pc->bRecord) {
        return 255 | ENDOFFILE;
      } else {
        return EINPUT;
      }
    }
    pc->wOut = 0;
    pc->wIn = (word)iRead;
  }
  word wReturn = pc->pbBuffer[pc->wOut++];
  if (pc->wOut >= pc->wIn && _eof(pc->hFile)) {
    wReturn |= ENDOFFILE;
  }
  return wReturn;
}

// Block schreiben
word CFloppy::Write(word wChannel, byte* pbBuffer, word* pwCount) {
  assert(wChannel <= 14);
  assert(ch[wChannel].sPattern[0]);
  assert(ch[wChannel].hFile >= 0);
  assert(ch[wChannel].wIn == ch[wChannel].wOut);
  if (ch[wChannel].bRecord) {
    FillRecord(ch[wChannel].hFile);
  }
  word w = _lwrite(ch[wChannel].hFile, pbBuffer, *pwCount);
  if (w != *pwCount) {
    if (w == (word)-1) {
      DOSError(0);
      *pwCount = 0;
    } else {
      SetError(NoSpace);
      *pwCount = w;
    }
  }
  return 0;
}

// Block lesen
word CFloppy::Read(word wChannel, byte* pbBuffer, word* pwCount) {
  assert(wChannel <= 14);
  assert(ch[wChannel].sPattern[0]);
  assert(ch[wChannel].hFile >= 0);
  assert(ch[wChannel].wIn == ch[wChannel].wOut);
  word w = _lread(ch[wChannel].hFile, pbBuffer, *pwCount);
  if (w == (word)-1) {
    DOSError(0);
    *pwCount = 0;
  } else if (w != *pwCount) {
    *pwCount = w;
    return ENDOFFILE;
  }
  return 0;
}

// Datei schlieen
word CFloppy::Close(word wChannel) {
  // Verzeichnis schlieen
  if (ch[wChannel].fDirectory) {
    ch[wChannel].fDirectory = FALSE;
    ch[wChannel].sPattern[0] = 0;
    return 0;
  }
  // Ist auf diesem Kanal berhaupt eine Datei offen?
  if (!ch[wChannel].hFile) {
    if (!eError) {
      SetError(NotOpen);
    }
    return 0;
  }
  // Bei normaler Datei Puffer schreiben und schlieen
  Flush(ch + wChannel);
  free(ch[wChannel].pbBuffer);
  ch[wChannel].pbBuffer = NULL;
  if (_lclose(ch[wChannel].hFile) == -1) DOSError(-1);
  ch[wChannel].hFile = 0;
  ch[wChannel].sPattern[0] = 0;
  return 0;
}

// Befehl absetzen
word CFloppy::Command(char* sCommand, word wLength) {
  // Kein Befehl, dann ohne Aktion zurck
  if (!wLength) {
    return 0;
  }
  // Auf berlauf im Kommandopuffer prfen
  if (wLength > 41) {
    return SetError(TooLong);
  }
  // Doppelpunkt suchen
  char* pc = (char*)memchr(sCommand, ':', wLength);
  // Laufwerk 1 gibt es nicht
  if (pc && pc[-1] == '1') {
    return SetError(NotReady);
  }
  switch (*sCommand) {
  case 'R':
    // Datei umbenennen
    {
      // Abschlieende 13 aus Namen herausfiltern
      if (sCommand[wLength - 1] == 13) {
        wLength--;
      }
      if (!pc) {
        return SetError(WrongCmd);
      }
      // Lnge an Zeichenkette hinter Doppelpunkt anpassen
      wLength -= ++pc - sCommand;
      // Gleichheitszeichen suchen
      char* pcEqual = (char*)memchr(pc, '=', wLength);
      if (!pcEqual) {
        return SetError(NoCmd);
      }
      // Neuen Dateinamen bertragen
      char acNew[50];
      assert(pcEqual - pc < 50);
      memcpy(acNew, pc, pcEqual - pc);
      acNew[pcEqual - pc] = 0;
      // Lnge an Zeichenkette hinter Doppelpunkt anpassen
      wLength -= ++pcEqual - pc;
      // Alten Dateinamen bertragen
      char acOld[80];
      assert(wLength < 80);
      memcpy(acOld, pcEqual, wLength);
      acOld[wLength] = 0;
      // Alte Datei ffnen
      channel chOld;
      if (!SetName(&chOld, acOld)) {
        return 0;
      }
      if (chOld.fDirectory || chOld.eType != ANY || eMode != eREAD || fWildcards) {
        return SetError(NoCmd);
      }
      if (!ChOpen(&chOld)) {
        return 0;
      }
      assert(eType != ANY);
      chOld.eType = eType;
      assert(chOld.hFile > 0);
      strcpy(acOld, sPath);
      assert(strlen(acOld) < 80);
      // Neue Datei anlegen
      channel chNew;
      if (!SetName(&chNew, acNew)) {
        return 0;
      }
      if (chNew.fDirectory || chNew.eType != ANY || eMode != eREAD || fWildcards) {
        return SetError(NoCmd);
      }
      chNew.eType = chOld.eType;
      chNew.bRecord = chOld.bRecord;
      eMode = eWRITE;
      assert(!fReplace);
      if (!ChOpen(&chNew)) {
        _lclose(chOld.hFile);
        return 0;
      }
      // Neue Datei schlieen und wieder lschen
      _lclose(chNew.hFile);
      remove(sPath);
      // Alte Datei schlieen wegen SHARE.EXE
      _lclose(chOld.hFile);
      // Alte Datei umbenennen
      if (rename(acOld, sPath)) {
        return DOSError(chOld.hFile);
      }
      // Neuen Dateinamen setzen
      chOld.hFile = _lopen(sPath, READ_WRITE);
      if (chOld.hFile == -1) {
        DOSError(-1);
        rename(sPath, acOld);
        return FALSE;
      }
      memset(acNew, 0, 16);
      strcpy(acNew, chNew.sPattern);
      if (_llseek(chOld.hFile, 8, 0) == -1) {
        DOSError(chOld.hFile);
        rename(sPath, acOld);
        return FALSE;
      }
      if (_lwrite(chOld.hFile, acNew, 16) != 16) {
        DOSError(chOld.hFile);
        rename(sPath, acOld);
        return FALSE;
      }
      // Datei wieder schlieen
      _lclose(chOld.hFile);
      assert(eError == OK);
    }
    return 0;
  case 'S':
    // Dateien lschen
    {
      if (!pc) {
        return SetError(WrongCmd);
      }
      char* pcEnd = sCommand + wLength;
      // Abschlieende 13 aus Namen herausfiltern
      if (pcEnd[-1] == 13) {
        pcEnd--;
      }
      // Anzahl der gelschten Dateien zurcksetzen
      wScratched = 0;
      // Keine DEV Fehlermeldungen ausgeben
      fQuiet = TRUE;
      while (++pc < pcEnd) {
        // Nchstes Komma oder Befehlsende suchen
        char* pcComma = (char*)memchr(pc, ',', pcEnd - pc);
        if (!pcComma) {
          pcComma = pcEnd;
        }
        // Dateimaske umkopieren
        char ac[40];
        assert(pcComma - pc < 40);
        memcpy(ac, pc, pcComma - pc);
        ac[pcComma - pc] = 0;
        pc = pcComma;
        // Dateien suchen
        channel ch;
        if (SetName(&ch, ac) && ReduceName(ch.sPattern, pName)) {
          strcat(pName, ".*");
          // Datei suchen und ffnen
          find_t Temp;
          word wTemp = _dos_findfirst(sPath, 0x20, &Temp);
          for (;;) {
            if (FindFile(&ch, &Temp, &wTemp)) {
              // Datei schlieen und lschen
              _lclose(ch.hFile);
              if (remove(sPath) == 0 || remove(sPath) == 0) {
                wScratched++;
              }
            } else if (eError != WrongType) {
              break;
            }
          }
        }
      }
      // Fehlermeldungen wieder zulassen
      fQuiet = FALSE;
      // Anzahl gelschter Dateien zurckgeben
      eError = Scratched;
    }
    return 0;
  case 'P':
    // Positionieren in relativen Dateien
    {
      if (wLength < 2) {
        return SetError(NoChannel);
      }
      word wChannel = (byte)sCommand[1] & 15;
      channel* pCh = ch + wChannel;
      if (wChannel < 2 || wChannel > 14 || !pCh->hFile || pCh->eType != REL) {
        return SetError(NoChannel);
      }
      Flush(pCh);
      pCh->wIn = pCh->wOut = 0;
      long lPos = 0;
      byte bPos = 0;
      switch (wLength) {
      default:
        bPos = (byte)sCommand[4];
      case 4:
        lPos |= (word)sCommand[3] << 8;
      case 3:
        lPos |= (byte)sCommand[2];
      }
      if (lPos) {
        lPos--;
      }
      lPos *= pCh->bRecord;
      if (bPos) {
        bPos--;
      }
      lPos += sizeof(header) + bPos;
      _llseek(pCh->hFile, lPos, 0);
      if (lPos >= _filelength(pCh->hFile)) {
        return SetError(NoRecord);
      }
    }
    return 0;
  case 'I':
    // Laufwerk initialisieren, alle Dateien schlieen
    Reset();
    NoError();
    return 0;
  case 'U':
    switch (sCommand[1]) {
    case 'I':
    case 'J':
      // Laufwerk initialisieren, alle Dateien schlieen
      Reset();
      return 0;
    }
    break;
  }
  // Unbekannter Befehl
  return SetError(NoCmd);
}

// Satzlnge einer relativen Datei liefern
byte CFloppy::GetRelLength(word wChannel) {
  // Zulssige Kanle liegen zwischen 0 und 14
  if (wChannel > 14) {
    return 0;
  }
  // Datei mu offen sein
  if (!ch[wChannel].hFile) {
    return 0;
  }
  // Satzlnge zurckgeben oder 0 bei sequentiellen Dateien
  return ch[wChannel].bRecord;
}

// Disketten kopieren
flag CFloppy::BeginTrack(flag fWrite) {
  return FALSE;
}

word CFloppy::ReadNextTrack(byte* pbBuffer) {
  return 0;
}

flag CFloppy::WriteNextTrack(byte* pbBuffer, word wLength) {
  return FALSE;
}

flag CFloppy::EndTrack() {
  return FALSE;
}

// Gre holen
int CFloppy::GetBlocks(word wChannel) {
  long l = _filelength(ch[wChannel].hFile);
  if (l == -1) {
    return -1;
  }
  return (int)((l - 26 + 253) / 254);
}

// Datum holen
dword CFloppy::GetDateAndTime(word wChannel) {
  unsigned uDate;
  unsigned uTime;
  if (_dos_getftime(ch[wChannel].hFile, &uDate, &uTime) == -1) {
    return 0;
  }
  return ((dword)uDate << 16) | uTime;
}

// Datum setzen
void CFloppy::SetDateAndTime(word wChannel, dword dwDateAndTime) {
  _dos_setftime(ch[wChannel].hFile, HIWORD(dwDateAndTime), LOWORD(dwDateAndTime));
}
