/*
 * Picket Fence by Jari Komppa aka Sol/Trauma, 1998
 * Version 1.4
 *
 * If you use this source, you must include the following
 * line into your program or in its documentation:
 * "Certified with Picket Fence v1.4"
 * Of course if your program causes picket fence crash,
 * you should NOT include the line.
 *
 * For usage and licence, see pfence.txt
 */

/*
 * Required libraries:
 */

#include <stdlib.h> /* malloc, free */
#include <stdio.h>  /* printf */
#include <string.h> /* memset/memcpy */

/*
 * Compilation options:
 */

/*
 * Set mode 3 before crash report (uses watcom specific stuff)
 */
#define SET_MODE3

/*
 * Checks watcom's memory structure (the one just before allocated area)
 */
//#define WATCOM_MEMTHING

/*
 * Skips memory integrity cheks; for memory leak testing (faster)
 */
//#define NO_CHECKMEM

/*
 * Track freed memory for writes; hogs up memory FAST:
 */
//#define KEEP_MEM

/*
 * Fills freshly allocated memory with trash; normally 0:
 */
//#define RANDOM_MEM

/*
 * Width of the fence, in bytes:
 */
#define FENCE_WID 16

/*
 * Crash on the specified alloc call:
 */
#define SPECIFIED -1

/*
 * Crash if freeing previously non-allocated or freed memory:
 * (doesn't crash on free(0))
 */
#define NONALLOC_FREE

/*
 * Global variables:
 */

int pfence_total_alloced=0;
int pfence_alloc_count=0;

/*
 * Memory-tracking linked list & functions: (static)
 */

struct memstruc {
  struct memstruc * next, * prev;
  char * ofs;
  int size;
  int count;
  int row;
  int watcom_memvalue;
  char * filename;
};

static struct memstruc * membase=NULL;
static struct memstruc * memtop=NULL;

static struct memstruc * newmem(void)
{
  struct memstruc * temp, * help;
  temp=(struct memstruc*)malloc(sizeof(struct memstruc));
  temp->next=temp->prev=NULL;
  temp->ofs=NULL;
  if (membase==NULL) {
    membase=memtop=temp;
    return membase;
  }
  help=membase;
  while (help->next!=NULL) help=help->next;
  help->next=temp;
  temp->prev=help;
  memtop->next=temp;
  memtop=temp;
  return temp;
}

static void killmem(struct memstruc * item)
{
  struct memstruc * temp;
  if (membase==item) {
    if (item->next) {
      temp=item->next;
      free(item);
      membase=temp;
      membase->prev=NULL;
      return;
    }
    free(item);
    membase=memtop=NULL;
    return;
  }
  if (item->next) {
    item->next->prev=item->prev;
    item->prev->next=item->next;
    free(item);
    return;
  }
  memtop=item->prev;
  item->prev->next=NULL;
  free(item);
}

#ifdef SET_MODE3
void setmode3(void);
#pragma aux setmode3=\
"mov ax,3"\
"int 10h"\
modify [ax];
#endif

/*
 * The crash-function. Place your breakpoint at printf-row.
 *
 * This function must have a printf or other meaningful function
 * call where the breakpoint can be set or watcom eliminates the
 * whole function.
 */

static void crash(int reason, int alloc, int row, char * fname)
{
  /* reasons:
   * 1 - malloc <=0
   * 2 - out of memory
   * 3 - under trash
   * 4 - over trash
   * 5 - free() call on unallocated non-null pointer
   * 6 - trashed 'freed' memory
   * 7 - crash on specified alloc number
   */
#ifdef SET_MODE3
  setmode3();
#endif
  printf("Picket Fence crash!\n"
         "Reason %d - alloc %d\n"
         "Allocated at Row %d of %s\n",reason,alloc,row,fname);
  switch (reason) {
  case 1: printf("Allocation of 0 or less bytes\n"); break;
  case 2: printf("Out of memory\n"); break;
  case 3: printf("Under trash\n"); break;
  case 4: printf("Over trash\n"); break;
  case 5: printf("free() call on unallocated non-null pointer\n"); break;
  case 6: printf("Trashed 'freed' memory\n"); break;
  case 7: printf("Crash on specified alloc number\n"); break;
  default: printf("Unknown crash reason\n");
  }
  exit(reason);
}

/*
 * Function that checks the memory tree for overwrites.
 * Automatically called on free and alloc; call manually
 * if you don't do much of either, or wish to pinpoint the
 * violating call.
 */

void pfence_checkmemtree(void)
{
#ifndef NO_CHECKMEM
  int a;
  struct memstruc * item;
  item=membase;
  while (item!=NULL) {
#ifdef WATCOM_MEMTHING
    if (item->watcom_memvalue!=*((int*)item->ofs-1))
      crash(3,item->count,item->row,item->filename);
#endif
    if (item->size>0) {
      for (a=0;a<FENCE_WID;a++) {
        if (*(item->ofs+a)!=(a^0xaa)) crash(3,item->count,item->row,item->filename);
        if (*(item->ofs+item->size+FENCE_WID+a)!=(a^0xaa)) crash(4,item->count,item->row,item->filename);
      }
    } else {
      for (a=0;a<(-item->size);a++)
        if(*(item->ofs+a)!=(a^0xaa)) crash(6,item->count,item->row,item->filename);
    }
    item=item->next;
  }
#endif
}

/*
 * Memory allocation function
 */

void * pfence_alloc(int size,int row,char*fname)
{
  int a;
  struct memstruc * item;
  pfence_alloc_count++;
  if (pfence_alloc_count==SPECIFIED) crash(7,pfence_alloc_count,row,fname);
  pfence_total_alloced+=size;
  pfence_checkmemtree();
  if (size<=0) crash(1,pfence_alloc_count,row,fname);
  item=newmem();
  item->ofs=(char*)malloc(size+FENCE_WID+FENCE_WID);
  item->size=size;
  item->row=row;
  item->filename=strdup(fname);
  item->count=pfence_alloc_count;
#ifdef WATCOM_MEMTHING
  item->watcom_memvalue=*((int*)item->ofs-1);
#endif

  if (item->ofs==NULL) crash(2,pfence_alloc_count,row,fname);
#ifdef RANDOM_MEM
  for (a=0;a<size;a++) {
    *(item->ofs+FENCE_WID+a)=(a^0xaa);
  }
#else
  memset(item->ofs+FENCE_WID,0,size);
#endif
  for (a=0;a<FENCE_WID;a++) {
    *(item->ofs+a)=(a^0xaa);
    *(item->ofs+size+FENCE_WID+a)=(a^0xaa);
  }
  return (item->ofs+FENCE_WID);
}

/*
 * Memory freeing function
 */

void pfence_free(void * ptr)
{
  struct memstruc * item;
  int a;
  pfence_checkmemtree();
  item=membase;
  if (ptr==NULL) return;
  ptr=(char*)ptr-FENCE_WID;
  while (item!=NULL) {
    if (item->ofs==ptr) {
      if (item->size<0) {
#ifdef NONALLOC_FREE
        crash(5,item->count,item->row,item->filename);
#else
        return;
#endif
      }
      pfence_total_alloced-=item->size;
      for (a=0;a<(item->size+FENCE_WID+FENCE_WID);a++)
        *(item->ofs+a)=(a^0xaa);
#ifndef KEEP_MEM
      free(item->ofs);
      free(item->filename);
      killmem(item);
#else /* KEEP_MEM */
      item->size=-(item->size+FENCE_WID+FENCE_WID);
#endif /* KEEP_MEM */
      return;
    }
    item=item->next;
  }
#ifdef NONALLOC_FREE
  crash(5,pfence_alloc_count,0,"<not allocated>");
#endif
}

/*
 * Memory re-allocating function
 */

void * pfence_realloc(void * oldmem, int newsize,int row,char *fname)
{
  struct memstruc * old;
  void * temp;
  int a;
  if (oldmem==NULL)
    return pfence_alloc(newsize,row,fname);
  if (newsize==0) {
    pfence_free(oldmem);
    return NULL;
  }
  pfence_alloc_count++;
  if (newsize<0) crash(1,pfence_alloc_count,row,fname);
  old=membase;
  while (old!=NULL && old->ofs!=((char*)oldmem-FENCE_WID)) old=old->next;
  if (!old)
    return pfence_alloc(newsize,row,fname);
  temp=pfence_alloc(newsize,row,fname);
  a=newsize;
  if (a<old->size) a=old->size;
  memcpy(temp,old->ofs+FENCE_WID,a);
  pfence_free(oldmem);
  return temp;
}
