;=============================================================================
; txtfx.asm - Texture Effect Demostration.
;                                                    File created: 11/03/93
; Copyright (c) 1993, Carlos Hasan                  Last modified: 11/04/93
;
; Description:
;   This file implements a texture algorithm that maps repeteadly into
; screen a picture of 128x128x256 using the tweaked 80x200x256 mode.
;
; Portability:
;   Requires Turbo Assembler 3.2 or better to be assembled.
;   Dependent on the IBM PC 286 and the VGA graphics card.
;=============================================================================

        ideal
        model   small,pascal
        jumps
        p286
        
        dosseg                  ; Used for Standalone DOS programs.
        stack   1024            ; Sets the Stack with 1KB of space.
        ends                    ; Must close stack segment!

;===================== External Data =========================================

        global  PicRaw:far      ; 128x128x256 Image.
        global  PicPal:far      ; RGB Image Palette.
        global  TextFx:proc

;===================== Demo Equates ==========================================

        dataseg

MAXWIDTH        equ     80      ; Screen Dimensions.
MAXHEIGHT       equ     200
BMPHEIGHT       equ     128     ; Image Bitmap Dimensions.
BMPWIDTH        equ     128
BMPMASK         equ     3FFFh   ; Image Position Mask.
TIMEOUT         equ     1800    ; Max Number of Frames for TimeOut.
ROTSPEED        equ     2       ; Rotation Speed Factor.
AMPSPEED        equ     4       ; Amplify Speed Factor.
FADESPEED       equ     6       ; Fadeout Speed Factor.

        include "SinCos.Inc"    ; Sinus/Cosinus Table.

;======================== Demo Routines ======================================

        codeseg

;-----------------------------------------------------------------------------
; SetGraphMode - Sets the tweaked 80x200x256 VGA mode.
;-----------------------------------------------------------------------------

proc    SetGraphMode
        mov     ax,13h          ; Sets VGA 320x200x256 mode.
        int     10h
        mov     dx,3C4h         ; Disable Chain-Four.
        mov     al,04h
        out     dx,al
        inc     dx
        in      al,dx
        and     al,0F7h
        out     dx,al
        mov     dx,3C4h         ; Enable All Planes.
        mov     ax,0F02h
        out     dx,ax
        mov     ax,0A000h       ; Clear Video Memory
        mov     es,ax
        xor     di,di
        xor     ax,ax
        mov     cx,8000h
        cld
        rep     stosw
        mov     dx,3D4h         ; Normal Word Addressing.
        mov     al,14h
        out     dx,al
        inc     dx
        in      al,dx
        and     al,0BFh
        out     dx,al
        dec     dx              ; Address Output Byte Mode.
        mov     al,17h
        out     dx,al
        inc     dx
        in      al,dx
        or      al,40h
        out     dx,al
        ret
endp    SetGraphMode


;-----------------------------------------------------------------------------
; SetTextMode - Restores 80x25x16 text mode.
;-----------------------------------------------------------------------------

proc    SetTextMode
        mov     ax,03h
        int     10h
        ret
endp    SetTextMode

;-----------------------------------------------------------------------------
; WaitVR - Waits the Next Vertical Retrace.
;-----------------------------------------------------------------------------

proc    WaitVR
        mov     dx,3DAh
WaitNotVRT:
        in      al,dx
        test    al,8
        jne     WaitNotVRT
WaitEndVRT:
        in      al,dx
        test    al,8
        je      WaitEndVRT
        ret
endp    WaitVR

;-----------------------------------------------------------------------------
; SetPalette - Sets the current VGA color palette.
; In:
;  DS:SI - Palette Address.
;-----------------------------------------------------------------------------

proc    SetPalette
        call    WaitVR
        mov     dx,3C8h
        xor     al,al
        out     dx,al
        inc     dx
        mov     cx,768
        cld
        rep     outsb
        ret
endp    SetPalette

;-----------------------------------------------------------------------------
; FadeOut - Fades out the RGB Palette.
; In:
;  DS:SI = Palette Address.
;-----------------------------------------------------------------------------

proc    FadeOut
        push    si
        push    ds
        mov     cx,768
        xor     bx,bx
        mov     di,si
DecPal: sub     [byte ptr di],FADESPEED
        jnc     NotDec
        inc     bx
        mov     [byte ptr di],0
NotDec: inc     di
        loop    DecPal
        push    bx
        call    SetPalette
        pop     bx
        pop     ds
        pop     si
        cmp     bx,768
        jb      FadeOut
        ret
endp    FadeOut

;-----------------------------------------------------------------------------
; RotImage - Puts on the screen the rotated image.
; In:
;  DeltaX,DeltaY = Rotation Parameters.
;  Image         = Source 128x128x256 Image Bitmap.
;-----------------------------------------------------------------------------

proc    RotImage DeltaX:word,DeltaY:word,Image:word
local   AddX:word:MAXWIDTH,AddY:word:MAXHEIGHT

        push    ds
        push    es

        shl     [DeltaX],1      ; Makes the Horiz Array
        shl     [DeltaY],1      ; of increments for the
        mov     cx,MAXWIDTH     ; Rotation.
        xor     bl,bl
        xor     dl,dl
        lea     di,[AddX]
MakeAddX:
        xor     bh,bh
        xor     dh,dh
        add     bx,[DeltaX]
        add     dx,[DeltaY]
        mov     al,dh
        cbw
        imul    si,ax,BMPWIDTH
        mov     al,bh
        cbw
        add     ax,si
        mov     [ss:di],ax
        add     di,2
        loop    MakeAddX

        sar     [DeltaX],1      ; Makes the Vert Array
        sar     [DeltaY],1      ; of increments for the
        mov     cx,MAXHEIGHT    ; Rotation.
        xor     bl,bl
        xor     dl,dl
        lea     di,[AddY]
MakeAddY:
        xor     bh,bh
        xor     dh,dh
        sub     bx,[DeltaY]
        add     dx,[DeltaX]
        mov     al,dh
        cbw
        imul    si,ax,BMPWIDTH
        mov     al,bh
        cbw
        add     ax,si
        mov     [ss:di],ax
        add     di,2
        loop    MakeAddY

        mov     ax,0A000h       ; Start drawing at the top left
        mov     es,ax           ; screen corner.
        xor     di,di
        mov     ds,[Image]      ; Load Source Image Address.
        xor     si,si
        mov     dx,BMPMASK      ; Load Image Position Mask.
        lea     bx,[AddY]
        mov     cx,MAXHEIGHT
        cld
DrawScanLines:
        push    si
        OFF=0                    ; This Loop have been unrolled
        REPT MAXWIDTH/2          ; for fastest drawing.
        and      si,dx           ; Also, the drawwing is done 
        mov      al,[si]         ; putting two pixels at once.
        add      si,[ss:AddX+OFF]
        and      si,dx
        mov      ah,[si]
        add      si,[ss:AddX+OFF+2]
        stosw
        OFF=OFF+4
        ENDM
        pop     si
        add     si,[ss:bx]       ; Next Scan Line. 
        add     bx,2
        dec     cx
        jne     DrawScanLines
        pop     es
        pop     ds
        ret
endp    RotImage

;-----------------------------------------------------------------------------
; TextFx - Performs the Demostration.
; In:
;  DS      = Data Segment.
;  Image   = Source Image Segment.
;  Palette = Image RGB Palette.
;-----------------------------------------------------------------------------

proc    TextFx Image:word,Palette:dword
        pusha
        push    ds
        push    es
        call    SetGraphMode    ; Init Graphics Mode.
        push    ds
        lds     si,[Palette]    ; Sets the RGB Palette.
        call    SetPalette
        pop     ds

        xor     bx,bx           ; Init Rotation Angle.
        mov     dx,32           ; Init Amplify Factor.
TextFxLoop:
        add     bl,ROTSPEED     ; Increase Rot Angle.
        add     dx,AMPSPEED     ; Increase Amplify Factor.
        mov     cl,[SinTable+bx]
        mov     ch,[CosTable+bx]
        push    bx
        push    dx
        mov     al,cl           ; Amplify Rotation Parameters
        cbw                     ; using the Amplify Factor.
        imul    dx
        mov     al,ah
        mov     ah,dl
        mov     bx,ax
        pop     dx
        push    dx
        mov     al,ch
        cbw
        imul    dx
        mov     al,ah
        mov     ah,dl
        call    RotImage,ax,bx,[Image]
        mov     ah,1            ; Any Key Pressed?
        int     16h
        pop     dx
        pop     bx
        jz      CheckTimeOut
        mov     ah,0
        int     16h
        jmp     TextExit
CheckTimeout:
        cmp     dx,TIMEOUT      ; Enough Frames Showed?
        jb      TextFxLoop
TextExit:
        lds     si,[Palette]    ; Fade out the Palette.
        call    FadeOut
        call    SetTextMode     ; Restores 80x25x16 Text Mode.
        pop     es
        pop     ds
        popa
        ret
endp    TextFx

;-----------------------------------------------------------------------------
; Start - Startup code called from DOS.
; In:
;  ES = Program Segment Prefix.
;-----------------------------------------------------------------------------

proc    Start
        mov     ax,@Data        ; Setup Data Segment.
        mov     ds,ax           ; Do the Demostration.
        call    TextFx,SEG PicRaw,SEG PicPal,OFFSET PicPal
        mov     ax,4C00h
        int     21h             ; Exit to DOS.
endp    Start

        end     Start
