/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      Text drawing routines.
 *
 *      See readme.txt for copyright information.
 */


#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>

#include "allegro.h"
#include "internal.h"



static FONT_8x8 _font_8x8 =                     /* the default 8x8 font */
{
   {
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },   /* ' ' */
      { 0x18, 0x3C, 0x3C, 0x18, 0x18, 0x00, 0x18, 0x00 },   /* '!' */
      { 0x6C, 0x6C, 0x6C, 0x00, 0x00, 0x00, 0x00, 0x00 },   /* '"' */
      { 0x6C, 0x6C, 0xFE, 0x6C, 0xFE, 0x6C, 0x6C, 0x00 },   /* '#' */
      { 0x18, 0x7E, 0xC0, 0x7C, 0x06, 0xFC, 0x18, 0x00 },   /* '$' */
      { 0x00, 0xC6, 0xCC, 0x18, 0x30, 0x66, 0xC6, 0x00 },   /* '%' */
      { 0x38, 0x6C, 0x38, 0x76, 0xDC, 0xCC, 0x76, 0x00 },   /* '&' */
      { 0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00 },   /* ''' */
      { 0x18, 0x30, 0x60, 0x60, 0x60, 0x30, 0x18, 0x00 },   /* '(' */
      { 0x60, 0x30, 0x18, 0x18, 0x18, 0x30, 0x60, 0x00 },   /* ')' */
      { 0x00, 0x66, 0x3C, 0xFF, 0x3C, 0x66, 0x00, 0x00 },   /* '*' */
      { 0x00, 0x18, 0x18, 0x7E, 0x18, 0x18, 0x00, 0x00 },   /* '+' */
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30 },   /* ',' */
      { 0x00, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00 },   /* '-' */
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00 },   /* '.' */
      { 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0x80, 0x00 },   /* '/' */
      { 0x7C, 0xCE, 0xDE, 0xF6, 0xE6, 0xC6, 0x7C, 0x00 },   /* '0' */
      { 0x30, 0x70, 0x30, 0x30, 0x30, 0x30, 0xFC, 0x00 },   /* '1' */
      { 0x78, 0xCC, 0x0C, 0x38, 0x60, 0xCC, 0xFC, 0x00 },   /* '2' */
      { 0x78, 0xCC, 0x0C, 0x38, 0x0C, 0xCC, 0x78, 0x00 },   /* '3' */
      { 0x1C, 0x3C, 0x6C, 0xCC, 0xFE, 0x0C, 0x1E, 0x00 },   /* '4' */
      { 0xFC, 0xC0, 0xF8, 0x0C, 0x0C, 0xCC, 0x78, 0x00 },   /* '5' */
      { 0x38, 0x60, 0xC0, 0xF8, 0xCC, 0xCC, 0x78, 0x00 },   /* '6' */
      { 0xFC, 0xCC, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x00 },   /* '7' */
      { 0x78, 0xCC, 0xCC, 0x78, 0xCC, 0xCC, 0x78, 0x00 },   /* '8' */
      { 0x78, 0xCC, 0xCC, 0x7C, 0x0C, 0x18, 0x70, 0x00 },   /* '9' */
      { 0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x00 },   /* ':' */
      { 0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x30 },   /* ';' */
      { 0x18, 0x30, 0x60, 0xC0, 0x60, 0x30, 0x18, 0x00 },   /* '<' */
      { 0x00, 0x00, 0x7E, 0x00, 0x7E, 0x00, 0x00, 0x00 },   /* '=' */
      { 0x60, 0x30, 0x18, 0x0C, 0x18, 0x30, 0x60, 0x00 },   /* '>' */
      { 0x3C, 0x66, 0x0C, 0x18, 0x18, 0x00, 0x18, 0x00 },   /* '?' */
      { 0x7C, 0xC6, 0xDE, 0xDE, 0xDC, 0xC0, 0x7C, 0x00 },   /* '@' */
      { 0x30, 0x78, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0x00 },   /* 'A' */
      { 0xFC, 0x66, 0x66, 0x7C, 0x66, 0x66, 0xFC, 0x00 },   /* 'B' */
      { 0x3C, 0x66, 0xC0, 0xC0, 0xC0, 0x66, 0x3C, 0x00 },   /* 'C' */
      { 0xF8, 0x6C, 0x66, 0x66, 0x66, 0x6C, 0xF8, 0x00 },   /* 'D' */
      { 0xFE, 0x62, 0x68, 0x78, 0x68, 0x62, 0xFE, 0x00 },   /* 'E' */
      { 0xFE, 0x62, 0x68, 0x78, 0x68, 0x60, 0xF0, 0x00 },   /* 'F' */
      { 0x3C, 0x66, 0xC0, 0xC0, 0xCE, 0x66, 0x3A, 0x00 },   /* 'G' */
      { 0xCC, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0xCC, 0x00 },   /* 'H' */
      { 0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00 },   /* 'I' */
      { 0x1E, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78, 0x00 },   /* 'J' */
      { 0xE6, 0x66, 0x6C, 0x78, 0x6C, 0x66, 0xE6, 0x00 },   /* 'K' */
      { 0xF0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xFE, 0x00 },   /* 'L' */
      { 0xC6, 0xEE, 0xFE, 0xFE, 0xD6, 0xC6, 0xC6, 0x00 },   /* 'M' */
      { 0xC6, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0xC6, 0x00 },   /* 'N' */
      { 0x38, 0x6C, 0xC6, 0xC6, 0xC6, 0x6C, 0x38, 0x00 },   /* 'O' */
      { 0xFC, 0x66, 0x66, 0x7C, 0x60, 0x60, 0xF0, 0x00 },   /* 'P' */
      { 0x7C, 0xC6, 0xC6, 0xC6, 0xD6, 0x7C, 0x0E, 0x00 },   /* 'Q' */
      { 0xFC, 0x66, 0x66, 0x7C, 0x6C, 0x66, 0xE6, 0x00 },   /* 'R' */
      { 0x7C, 0xC6, 0xE0, 0x78, 0x0E, 0xC6, 0x7C, 0x00 },   /* 'S' */
      { 0xFC, 0xB4, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00 },   /* 'T' */
      { 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xFC, 0x00 },   /* 'U' */
      { 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00 },   /* 'V' */
      { 0xC6, 0xC6, 0xC6, 0xC6, 0xD6, 0xFE, 0x6C, 0x00 },   /* 'W' */
      { 0xC6, 0xC6, 0x6C, 0x38, 0x6C, 0xC6, 0xC6, 0x00 },   /* 'X' */
      { 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x30, 0x78, 0x00 },   /* 'Y' */
      { 0xFE, 0xC6, 0x8C, 0x18, 0x32, 0x66, 0xFE, 0x00 },   /* 'Z' */
      { 0x78, 0x60, 0x60, 0x60, 0x60, 0x60, 0x78, 0x00 },   /* '[' */
      { 0xC0, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x02, 0x00 },   /* '\' */
      { 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00 },   /* ']' */
      { 0x10, 0x38, 0x6C, 0xC6, 0x00, 0x00, 0x00, 0x00 },   /* '^' */
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF },   /* '_' */
      { 0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00 },   /* '`' */
      { 0x00, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x76, 0x00 },   /* 'a' */
      { 0xE0, 0x60, 0x60, 0x7C, 0x66, 0x66, 0xDC, 0x00 },   /* 'b' */
      { 0x00, 0x00, 0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x00 },   /* 'c' */
      { 0x1C, 0x0C, 0x0C, 0x7C, 0xCC, 0xCC, 0x76, 0x00 },   /* 'd' */
      { 0x00, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00 },   /* 'e' */
      { 0x38, 0x6C, 0x64, 0xF0, 0x60, 0x60, 0xF0, 0x00 },   /* 'f' */
      { 0x00, 0x00, 0x76, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8 },   /* 'g' */
      { 0xE0, 0x60, 0x6C, 0x76, 0x66, 0x66, 0xE6, 0x00 },   /* 'h' */
      { 0x30, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00 },   /* 'i' */
      { 0x0C, 0x00, 0x1C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78 },   /* 'j' */
      { 0xE0, 0x60, 0x66, 0x6C, 0x78, 0x6C, 0xE6, 0x00 },   /* 'k' */
      { 0x70, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00 },   /* 'l' */
      { 0x00, 0x00, 0xCC, 0xFE, 0xFE, 0xD6, 0xD6, 0x00 },   /* 'm' */
      { 0x00, 0x00, 0xB8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00 },   /* 'n' */
      { 0x00, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00 },   /* 'o' */
      { 0x00, 0x00, 0xDC, 0x66, 0x66, 0x7C, 0x60, 0xF0 },   /* 'p' */
      { 0x00, 0x00, 0x76, 0xCC, 0xCC, 0x7C, 0x0C, 0x1E },   /* 'q' */
      { 0x00, 0x00, 0xDC, 0x76, 0x62, 0x60, 0xF0, 0x00 },   /* 'r' */
      { 0x00, 0x00, 0x7C, 0xC0, 0x70, 0x1C, 0xF8, 0x00 },   /* 's' */
      { 0x10, 0x30, 0xFC, 0x30, 0x30, 0x34, 0x18, 0x00 },   /* 't' */
      { 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00 },   /* 'u' */
      { 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00 },   /* 'v' */
      { 0x00, 0x00, 0xC6, 0xC6, 0xD6, 0xFE, 0x6C, 0x00 },   /* 'w' */
      { 0x00, 0x00, 0xC6, 0x6C, 0x38, 0x6C, 0xC6, 0x00 },   /* 'x' */
      { 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8 },   /* 'y' */
      { 0x00, 0x00, 0xFC, 0x98, 0x30, 0x64, 0xFC, 0x00 },   /* 'z' */
      { 0x1C, 0x30, 0x30, 0xE0, 0x30, 0x30, 0x1C, 0x00 },   /* '{' */
      { 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00 },   /* '|' */
      { 0xE0, 0x30, 0x30, 0x1C, 0x30, 0x30, 0xE0, 0x00 },   /* '}' */
      { 0x76, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },   /* '~' */
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },   /* '' */
      { 0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x18, 0x0C, 0x78 },   /* '' */
      { 0x00, 0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00 },   /* '' */
      { 0x1C, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00 },   /* '' */
      { 0x7E, 0xC3, 0x3C, 0x06, 0x3E, 0x66, 0x3F, 0x00 },   /* '' */
      { 0xCC, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },   /* '' */
      { 0xE0, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },   /* '' */
      { 0x30, 0x30, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },   /* '' */
      { 0x00, 0x00, 0x78, 0xC0, 0xC0, 0x78, 0x0C, 0x38 },   /* '' */
      { 0x7E, 0xC3, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00 },   /* '' */
      { 0xCC, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00 },   /* '' */
      { 0xE0, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00 },   /* '' */
      { 0xCC, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00 },   /* '' */
      { 0x7C, 0xC6, 0x38, 0x18, 0x18, 0x18, 0x3C, 0x00 },   /* '' */
      { 0xE0, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00 },   /* '' */
      { 0xC6, 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00 },   /* '' */
      { 0x30, 0x30, 0x00, 0x78, 0xCC, 0xFC, 0xCC, 0x00 },   /* '' */
      { 0x1C, 0x00, 0xFC, 0x60, 0x78, 0x60, 0xFC, 0x00 },   /* '' */
      { 0x00, 0x00, 0x7F, 0x0C, 0x7F, 0xCC, 0x7F, 0x00 },   /* '' */
      { 0x3E, 0x6C, 0xCC, 0xFE, 0xCC, 0xCC, 0xCE, 0x00 },   /* '' */
      { 0x78, 0xCC, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },   /* '' */
      { 0x00, 0xCC, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },   /* '' */
      { 0x00, 0xE0, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },   /* '' */
      { 0x78, 0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00 },   /* '' */
      { 0x00, 0xE0, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00 },   /* '' */
      { 0x00, 0xCC, 0x00, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8 },   /* '' */
      { 0xC3, 0x18, 0x3C, 0x66, 0x66, 0x3C, 0x18, 0x00 },   /* '' */
      { 0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00 },   /* '' */
      { 0x18, 0x18, 0x7E, 0xC0, 0xC0, 0x7E, 0x18, 0x18 },   /* '' */
      { 0x38, 0x6C, 0x64, 0xF0, 0x60, 0xE6, 0xFC, 0x00 },   /* '' */
      { 0xCC, 0xCC, 0x78, 0xFC, 0x30, 0xFC, 0x30, 0x30 },   /* '' */
      { 0xF8, 0xCC, 0xCC, 0xFA, 0xC6, 0xCF, 0xC6, 0xC7 },   /* '' */
      { 0x0E, 0x1B, 0x18, 0x3C, 0x18, 0x18, 0xD8, 0x70 },   /* '' */
      { 0x1C, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },   /* '' */
      { 0x38, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00 },   /* '' */
      { 0x00, 0x1C, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },   /* '' */
      { 0x00, 0x1C, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00 },   /* '' */
      { 0x00, 0xF8, 0x00, 0xF8, 0xCC, 0xCC, 0xCC, 0x00 },   /* '' */
      { 0xFC, 0x00, 0xCC, 0xEC, 0xFC, 0xDC, 0xCC, 0x00 },   /* '' */
      { 0x3C, 0x6C, 0x6C, 0x3E, 0x00, 0x7E, 0x00, 0x00 },   /* '' */
      { 0x38, 0x6C, 0x6C, 0x38, 0x00, 0x7C, 0x00, 0x00 },   /* '' */
      { 0x30, 0x00, 0x30, 0x60, 0xC0, 0xCC, 0x78, 0x00 },   /* '' */
      { 0x00, 0x00, 0x00, 0xFC, 0xC0, 0xC0, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x00, 0xFC, 0x0C, 0x0C, 0x00, 0x00 },   /* '' */
      { 0xC3, 0xC6, 0xCC, 0xDE, 0x33, 0x66, 0xCC, 0x0F },   /* '' */
      { 0xC3, 0xC6, 0xCC, 0xDB, 0x37, 0x6F, 0xCF, 0x03 },   /* '' */
      { 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00 },   /* '' */
      { 0x00, 0x33, 0x66, 0xCC, 0x66, 0x33, 0x00, 0x00 },   /* '' */
      { 0x00, 0xCC, 0x66, 0x33, 0x66, 0xCC, 0x00, 0x00 },   /* '' */
      { 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88 },   /* '' */
      { 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA },   /* '' */
      { 0xDB, 0x77, 0xDB, 0xEE, 0xDB, 0x77, 0xDB, 0xEE },   /* '' */
      { 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18 },   /* '' */
      { 0x18, 0x18, 0x18, 0x18, 0xF8, 0x18, 0x18, 0x18 },   /* '' */
      { 0x18, 0x18, 0xF8, 0x18, 0xF8, 0x18, 0x18, 0x18 },   /* '' */
      { 0x36, 0x36, 0x36, 0x36, 0xF6, 0x36, 0x36, 0x36 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0xFE, 0x36, 0x36, 0x36 },   /* '' */
      { 0x00, 0x00, 0xF8, 0x18, 0xF8, 0x18, 0x18, 0x18 },   /* '' */
      { 0x36, 0x36, 0xF6, 0x06, 0xF6, 0x36, 0x36, 0x36 },   /* '' */
      { 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36 },   /* '' */
      { 0x00, 0x00, 0xFE, 0x06, 0xF6, 0x36, 0x36, 0x36 },   /* '' */
      { 0x36, 0x36, 0xF6, 0x06, 0xFE, 0x00, 0x00, 0x00 },   /* '' */
      { 0x36, 0x36, 0x36, 0x36, 0xFE, 0x00, 0x00, 0x00 },   /* '' */
      { 0x18, 0x18, 0xF8, 0x18, 0xF8, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0xF8, 0x18, 0x18, 0x18 },   /* '' */
      { 0x18, 0x18, 0x18, 0x18, 0x1F, 0x00, 0x00, 0x00 },   /* '' */
      { 0x18, 0x18, 0x18, 0x18, 0xFF, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0xFF, 0x18, 0x18, 0x18 },   /* '' */
      { 0x18, 0x18, 0x18, 0x18, 0x1F, 0x18, 0x18, 0x18 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00 },   /* '' */
      { 0x18, 0x18, 0x18, 0x18, 0xFF, 0x18, 0x18, 0x18 },   /* '' */
      { 0x18, 0x18, 0x1F, 0x18, 0x1F, 0x18, 0x18, 0x18 },   /* '' */
      { 0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36 },   /* '' */
      { 0x36, 0x36, 0x37, 0x30, 0x3F, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x3F, 0x30, 0x37, 0x36, 0x36, 0x36 },   /* '' */
      { 0x36, 0x36, 0xF7, 0x00, 0xFF, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0xFF, 0x00, 0xF7, 0x36, 0x36, 0x36 },   /* '' */
      { 0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36 },   /* '' */
      { 0x00, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0x00, 0x00 },   /* '' */
      { 0x36, 0x36, 0xF7, 0x00, 0xF7, 0x36, 0x36, 0x36 },   /* '' */
      { 0x18, 0x18, 0xFF, 0x00, 0xFF, 0x00, 0x00, 0x00 },   /* '' */
      { 0x36, 0x36, 0x36, 0x36, 0xFF, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0xFF, 0x00, 0xFF, 0x18, 0x18, 0x18 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0xFF, 0x36, 0x36, 0x36 },   /* '' */
      { 0x36, 0x36, 0x36, 0x36, 0x3F, 0x00, 0x00, 0x00 },   /* '' */
      { 0x18, 0x18, 0x1F, 0x18, 0x1F, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x1F, 0x18, 0x1F, 0x18, 0x18, 0x18 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0x3F, 0x36, 0x36, 0x36 },   /* '' */
      { 0x36, 0x36, 0x36, 0x36, 0xFF, 0x36, 0x36, 0x36 },   /* '' */
      { 0x18, 0x18, 0xFF, 0x18, 0xFF, 0x18, 0x18, 0x18 },   /* '' */
      { 0x18, 0x18, 0x18, 0x18, 0xF8, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0x1F, 0x18, 0x18, 0x18 },   /* '' */
      { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF },   /* '' */
      { 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0 },   /* '' */
      { 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F },   /* '' */
      { 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x76, 0xDC, 0xC8, 0xDC, 0x76, 0x00 },   /* '' */
      { 0x00, 0x78, 0xCC, 0xF8, 0xCC, 0xF8, 0xC0, 0xC0 },   /* '' */
      { 0x00, 0xFC, 0xCC, 0xC0, 0xC0, 0xC0, 0xC0, 0x00 },   /* '' */
      { 0x00, 0xFE, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0x00 },   /* '' */
      { 0xFC, 0xCC, 0x60, 0x30, 0x60, 0xCC, 0xFC, 0x00 },   /* '' */
      { 0x00, 0x00, 0x7E, 0xD8, 0xD8, 0xD8, 0x70, 0x00 },   /* '' */
      { 0x00, 0x66, 0x66, 0x66, 0x66, 0x7C, 0x60, 0xC0 },   /* '' */
      { 0x00, 0x76, 0xDC, 0x18, 0x18, 0x18, 0x18, 0x00 },   /* '' */
      { 0xFC, 0x30, 0x78, 0xCC, 0xCC, 0x78, 0x30, 0xFC },   /* '' */
      { 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0x6C, 0x38, 0x00 },   /* '' */
      { 0x38, 0x6C, 0xC6, 0xC6, 0x6C, 0x6C, 0xEE, 0x00 },   /* '' */
      { 0x1C, 0x30, 0x18, 0x7C, 0xCC, 0xCC, 0x78, 0x00 },   /* '' */
      { 0x00, 0x00, 0x7E, 0xDB, 0xDB, 0x7E, 0x00, 0x00 },   /* '' */
      { 0x06, 0x0C, 0x7E, 0xDB, 0xDB, 0x7E, 0x60, 0xC0 },   /* '' */
      { 0x38, 0x60, 0xC0, 0xF8, 0xC0, 0x60, 0x38, 0x00 },   /* '' */
      { 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x00 },   /* '' */
      { 0x00, 0xFC, 0x00, 0xFC, 0x00, 0xFC, 0x00, 0x00 },   /* '' */
      { 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0xFC, 0x00 },   /* '' */
      { 0x60, 0x30, 0x18, 0x30, 0x60, 0x00, 0xFC, 0x00 },   /* '' */
      { 0x18, 0x30, 0x60, 0x30, 0x18, 0x00, 0xFC, 0x00 },   /* '' */
      { 0x0E, 0x1B, 0x1B, 0x18, 0x18, 0x18, 0x18, 0x18 },   /* '' */
      { 0x18, 0x18, 0x18, 0x18, 0x18, 0xD8, 0xD8, 0x70 },   /* '' */
      { 0x30, 0x30, 0x00, 0xFC, 0x00, 0x30, 0x30, 0x00 },   /* '' */
      { 0x00, 0x76, 0xDC, 0x00, 0x76, 0xDC, 0x00, 0x00 },   /* '' */
      { 0x38, 0x6C, 0x6C, 0x38, 0x00, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00 },   /* '' */
      { 0x0F, 0x0C, 0x0C, 0x0C, 0xEC, 0x6C, 0x3C, 0x1C },   /* '' */
      { 0x78, 0x6C, 0x6C, 0x6C, 0x6C, 0x00, 0x00, 0x00 },   /* '' */
      { 0x70, 0x18, 0x30, 0x60, 0x78, 0x00, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x3C, 0x3C, 0x3C, 0x3C, 0x00, 0x00 },   /* '' */
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },   /* '' */
   }
};


static FONT _font =
{
   8,
   {
      &_font_8x8
   }
};


FONT *font = &_font;

int _textmode = 0;



/* text_mode:
 *  Sets the mode in which text will be drawn. If mode is positive, text
 *  output will be opaque and the background will be set to mode. If mode
 *  is negative, text will be drawn transparently (ie. the background will
 *  not be altered). The default is a mode of zero.
 */
void text_mode(int mode)
{
   if (mode < 0)
      _textmode = -1;
   else
      _textmode = mode;
}



/* blit_character:
 *  Helper routine for opaque multicolor output of proportional fonts.
 */
static void blit_character(BITMAP *bmp, BITMAP *sprite, int x, int y)
{
   blit(sprite, bmp, 0, 0, x, y, sprite->w, sprite->h);
}



/* textout:
 *  Writes the null terminated string str onto bmp at position x, y, using 
 *  the current text mode and the specified font and foreground color.
 *  If color is -1 and a proportional font is in use, it will be drawn
 *  using the colors from the original font bitmap (the one imported into
 *  the grabber program), which allows multicolored text output.
 */
void textout(BITMAP *bmp, FONT *f, unsigned char *str, int x, int y, int color)
{
   FONT_PROP *fp;
   BITMAP *b;
   int c;
   void (*putter)();

   if (f->height == 8) {
      bmp->vtable->textout_fixed(bmp, f->dat.dat_8x8, 3, str, x, y, color);
      return;
   }
   else if (f->height == 16) {
      bmp->vtable->textout_fixed(bmp, f->dat.dat_8x16, 4, str, x, y, color);
      return;
   }

   fp = f->dat.dat_prop;

   if (color < 0) {
      if (_textmode < 0)
	 putter = bmp->vtable->draw_sprite;  /* masked multicolor output */
      else
	 putter = blit_character;            /* opaque multicolor output */
   }
   else
      putter = bmp->vtable->draw_character;  /* single color output */

   while (*str) {
      c = (int)*str - ' ';
      if ((c < 0) || (c >= FONT_SIZE))
	 c = 0;
      b = fp->dat[c];
      if (b) {
	 putter(bmp, b, x, y, color);
	 x += b->w;
	 if (x >= bmp->cr)
	    return;
      }
      str++;
   }
}



/* textout_centre:
 *  Like textout(), but uses the x coordinate as the centre rather than 
 *  the left of the string.
 */
void textout_centre(BITMAP *bmp, FONT *f, unsigned char *str, int x, int y, int color)
{
   int len;

   len = text_length(f, str);
   textout(bmp, f, str, x - len/2, y, color);
}



/* text_length:
 *  Calculates the length of a string in a particular font.
 */
int text_length(FONT *f, unsigned char *str)
{
   FONT_PROP *fp;
   int c;
   int len;

   if (f->height > 0)
      return strlen(str) * 8;

   fp = f->dat.dat_prop;
   len = 0;

   while (*str) {
      c = (int)*str - ' ';
      if ((c < 0) || (c >= FONT_SIZE))
	 c = 0;
      if (fp->dat[c])
	 len += fp->dat[c]->w;
      str++;
   }

   return len;
}



/* text_height:
 *  Returns the height of a character in the specified font.
 */
int text_height(FONT *f)
{
   return (f->height > 0) ? f->height : f->dat.dat_prop->dat[0]->h;
}



/* destroy_font:
 *  Frees the memory being used by a font structure.
 */
void destroy_font(FONT *f)
{
   FONT_PROP *fp;
   int c;

   if (f) {
      if (f->height == 8) {
	 /* free 8x8 font */
	 if (f->dat.dat_8x8)
	    free(f->dat.dat_8x8);
      }
      else if (f->height == 16) {
	 /* free 8x16 font */
	 if (f->dat.dat_8x16)
	    free(f->dat.dat_8x16);
      }
      else {
	 /* free proportional font */
	 fp = f->dat.dat_prop;
	 if (fp) {
	    for (c=0; c<FONT_SIZE; c++) {
	       if (fp->dat[c])
		  destroy_bitmap(fp->dat[c]);
	    }
	   free(fp); 
	 }
      }
      free(f);
   }
}

