unit WPlace; { Helpers for "sticky windows". }

interface

uses
  Windows;

type
  TWinPlace = class(TObject)
  public
    constructor Create;
    procedure Save(KeyName: string; Window: HWND);
    procedure Restore(KeyName: string; Window: HWND; var OK: Boolean);
  private
    WP: TWindowPlacement;
    procedure WriteToRegistry(KeyName: string);
    procedure ReadFromRegistry(KeyName: string);
  end;


implementation

uses
  Registry,
  Classes;   { for Rect function }

constructor TWinPlace.Create;
begin
  inherited Create;

  WP.length := SizeOf(TWindowPlacement);
end;

procedure TWinPlace.Save(KeyName: string; Window: HWND);
begin
  GetWindowPlacement(Window, @WP);
  WriteToRegistry(KeyName);
end;

procedure TWinPlace.Restore(KeyName: string; Window: HWND; var OK: Boolean);
var
  TempRect, ScreenRect: TRect;
begin
  ReadFromRegistry(KeyName);

  ScreenRect := Rect(0, 0, GetSystemMetrics(SM_CXSCREEN), GetSystemMetrics(SM_CYSCREEN));
  if not IntersectRect(TempRect, WP.rcNormalPosition, ScreenRect) then
    OK := False
  else
  begin
    SetWindowPlacement(Window, @WP); 
    OK := True
  end;
end;

procedure TWinPlace.WriteToRegistry(KeyName: string);
var
  R: TRegistry;
begin
  R := TRegistry.Create;
  try
    R.OpenKey(KeyName + '\Window', True);
    try
      R.WriteInteger('Flags', WP.flags);
      R.WriteInteger('ShowCmd', WP.showCmd);
      R.WriteInteger('MinPosX', WP.ptMinPosition.x);
      R.WriteInteger('MinPosY', WP.ptMinPosition.y);
      R.WriteInteger('MaxPosX', WP.ptMaxPosition.x);
      R.WriteInteger('MaxPosY', WP.ptMaxPosition.y);
      R.WriteInteger('Left', WP.rcNormalPosition.left);
      R.WriteInteger('Right', WP.rcNormalPosition.right);
      R.WriteInteger('Top', WP.rcNormalPosition.top);
      R.WriteInteger('Bottom', WP.rcNormalPosition.bottom);
    finally
      R.CloseKey;
    end;
  finally
    R.Free;
  end;
end;

procedure TWinPlace.ReadFromRegistry(KeyName: string);
var
  R: TRegistry;
begin
  R := TRegistry.Create;
  try
    if not R.OpenKey(KeyName + '\Window', False) then
      Exit;
    try
      WP.flags := R.ReadInteger('Flags');
      WP.showCmd := R.ReadInteger('ShowCmd');
      WP.ptMinPosition.x := R.ReadInteger('MinPosX');
      WP.ptMinPosition.y := R.ReadInteger('MinPosY');
      WP.ptMaxPosition.x := R.ReadInteger('MaxPosX');
      WP.ptMaxPosition.y := R.ReadInteger('MaxPosY');
      WP.rcNormalPosition.left := R.ReadInteger('Left');
      WP.rcNormalPosition.right := R.ReadInteger('Right');
      WP.rcNormalPosition.top := R.ReadInteger('Top');
      WP.rcNormalPosition.bottom := R.ReadInteger('Bottom');
    finally
      R.CloseKey;
    end;
  finally
    R.Free;
  end;
end;

end.

