unit Xyhist;

{unit xyhist requires that STATISTICS is defined in units xygraph and xydata}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics,
  Controls, Forms, Dialogs, ExtCtrls, DsgnIntf, Printers, xyData, xygraph;

const
  maxbuckets = 300;     {This is a quite arbitrary number
                          - it can be heaps more if you want}

type
  PHistData = ^THistData;
  THistData = record
      freq:double;
      next:PHistData;
    end;

  THistogram = class;

{------------ THAxis, THDimensions, THAppearance ----------------}

  THAxis = class(TPersistent)
  private
    FGraph: THistogram;
    FM: Double;
    FMin, FMax, FStep: Double;
    FAutoSizing, FAutoStepping, FIsXAxis: Boolean;
    FAxisTitle: ShortString;
    FLabelDec: Integer;
    FMinScale: Double;
    procedure SetLabelDec(v: Integer);
    procedure SetAutoSizing(v: Boolean);
    procedure SetAutoStepping(v: Boolean);
    procedure SetMax(v: Double);
    procedure SetMin(v: Double);
    procedure SetStep(v: Double);
    procedure SetMinDiff(v: Double);
    procedure SetAxisTitle(v: ShortString);
  protected
    function  GetStep: Double;
    procedure SetMinMax;
    function  DoResize: Boolean;
    function  GetFirstTick: Double;
    function  GetNextTick(tick: Double): Double;
    procedure AdjustLabelDecs;
    function  LabelString(tick: Double): ShortString;
  public
    function  CheckScale: Boolean;
  published
    property Title: ShortString read FAxisTitle write SetAxisTitle;
    property LabelDecimals: Integer read FLabelDec write SetLabelDec default 1;
    property Max: Double read FMax write SetMax;
    property Min: Double read FMin write SetMin;
    property StepSize: Double read FStep write SetStep;
    property MinScaleLength: Double read FMinScale write SetMinDiff;
   { these 2 properties must come last to override the other properties'
      effects on the values at load time: }
    property AutoSizing: Boolean read FAutoSizing write SetAutoSizing;
    property AutoStepping: Boolean read FAutoStepping write SetAutoStepping;
  end {class THAxis};

  THDimensions = class(TPersistent)
  private
    FGraph: THistogram;
    FLeft, FRight, FTop, FBottom: Word;
    FTMLength: Word;
    FXAxisTitleDistance: Integer;
    FYAxisTitleDistance: Integer;
    FXAxisLabelDistance: Integer;
    FYAxisLabelDistance: Integer;
    FGraphTitleDistance: Integer;
    FScalePct: Word;
    FXOffsetPct, FYOffsetPct: Word;
    procedure SetMargBottom(v: Word);
    procedure SetMargTop(v: Word);
    procedure SetMargLeft(v: Word);
    procedure SetMargRight(v: Word);
    procedure SetTickLength(v: Word);
    procedure SetXAxisTitleDistance(v: Integer);
    procedure SetYAxisTitleDistance(v: Integer);
    procedure SetXAxisLabelDistance(v: Integer);
    procedure SetYAxisLabelDistance(v: Integer);
    procedure SetGraphTitleDistance(v: Integer);
    procedure SetScale(v: Word);
    procedure SetXOffset(v: Word);
    procedure SetYOffset(v: Word);
  published
    property BottomMargin: Word read FBottom write SetMargBottom default 40;
    property LeftMargin: Word read FLeft write SetMargLeft default 40;
    property RightMargin: Word read FRight write SetMargRight default 15;
    property TopMargin: Word read FTop write SetMargTop default 30;
    property TickLength: Word read FTMLength write SetTickLength default 4;
    property XAxisTitleOffset: Integer read FXAxisTitleDistance write SetXAxisTitleDistance default 4;
    property XAxisLabelOffset: Integer read FXAxisLabelDistance write SetXAxisLabelDistance default 2;
    property YAxisTitleOffset: Integer read FYAxisTitleDistance write SetYAxisTitleDistance default 4;
    property YAxisLabelOffset: Integer read FYAxisLabelDistance write SetYAxisLabelDistance default 2;
    property GraphTitleOffset: Integer read FGraphTitleDistance write SetGraphTitleDistance default 7;
   {set print offsets as integer percent (<=100) of page width, height}
    property PrintXOffsetPct: Word read FXOffsetPct write SetXOffset default 5;
    property PrintYOffsetPct: Word read FYOffsetPct write SetYOffset default 20;
   {set print scale as integer percent (<=100) of full page}
    property PrintScalePct: Word read FScalePct write SetScale default 90;
  end {THDimensions};

  THAppearance = class(TPersistent)
  private
    FGraph: THistogram;
    FTickMarks, FGridlines, FShowMarks: Boolean;
    FLabelGraph: Boolean;
    FGridColor, FAxesColor, FBkgdColor: TColor;
    FGridStyle: TPenStyle;
    FBarBrush:TBrush;
    FBarPen:TPen;
    FBkgdWhenPrint,FPlotNormal:Boolean;
    FCaption: ShortString;
    FErrorCaption: ShortString;
    FGraphTitle: ShortString;
    FMinSteps,FMaxSteps, FBarWidth: Word;
    procedure SetAxesColor(v: TColor);
    procedure SetBkgdColor(v: TColor);
    procedure SetErrorCaption(v: ShortString);
    procedure SetLabelGraph(v: Boolean);
    procedure SetShowMarks(v: Boolean);
    procedure SetTickMarks(v: Boolean);
    procedure SetGraphTitle(v: ShortString);
    procedure SetMinSteps(v: Word);
    procedure SetMaxSteps(v: Word);
    procedure SetGridColor(v: TColor);
    procedure SetGridLines(v: Boolean);
    procedure SetGridStyle(v: TPenStyle);
    procedure SetBarBrush(v: TBrush);
    procedure SetBarPen(v: TPen);
    procedure SetBarWidth(v:word);
    procedure SetPlotNormal(v:boolean);

  published
    property AxesColor: TColor read FAxesColor write SetAxescolor default clBlack;
    property BackgroundColor: TColor read FBkgdColor write SetBkgdColor;
    property PrintBkgndColor: Boolean read FBkgdWhenPrint write FBkgdWhenPrint;
    property ErrorCaption: ShortString read FErrorCaption write SetErrorCaption;
    property ShowGraphLabels: Boolean read FLabelGraph write SetLabelGraph default True;
    property ShowMarks: Boolean read FShowMarks write SetShowMarks;
    property ShowTicks: Boolean read FTickMarks write SetTickmarks default True;
    property GridColor: TColor read FGridColor write SetGridcolor default clSilver;
    property ShowGridLines: Boolean read FGridlines write SetGridlines default True;
    property GridStyle: TPenStyle read FGridStyle write SetGridStyle default psDot;
    property GraphTitle: ShortString read FGraphTitle write SetGraphTitle;
    property MinSteps: Word read FMinSteps write SetMinSteps default 5;
    property MaxSteps: Word read FMaxSteps write SetMaxSteps default 50;
    property BarBrush:TBrush read FBarBrush write SetBarBrush;
    property BarPen:TPen read FBarPen write setBarPen;
    property BarWidth:word read FBarWidth write SetBarWidth;
    property PlotNormal:boolean read FPlotNormal write FPlotNormal;
  end {THAppearance};


{-------------------------- THistogram ------------------------------*}

  THistogram = class(TCustomPanel)
  private
    { Private declarations }
    FCanvas: TCanvas;
    FMarks: pMark;
    FFirstTime: Boolean;

    Fdata:PHistData;
    FXStart,FXStep,FMean,FSD:double;
    FXcount:word;

  {Graph property data:}
    FScale: Double;
    FXOffset, FYOffset: Word;

    FDimensions: THDimensions;
    FAppearance: THAppearance;
    FXAxis, FYAxis: THAxis;

   {Events:}
    FOnRescale, FOnScaleError: TNotifyEvent;
    FOnPaintEnd: TPaintEvent;

    FDataSeries:TDataSeries;
    FBuckets:word;
    FDataIsX:boolean;
    procedure WhenSeriesDataChanges(Sender:TObject; TheMessage:TDSChangeType);
    procedure SetBuckets(v:word);
    procedure readhistogram;
  protected
    procedure ClipGraph;
    procedure UnclipGraph;
    function  DoResizeX: Boolean;
    function  DoResizeY: Boolean;
    function  DoResize: Boolean;
    procedure CalcXMetrics;
    procedure CalcYMetrics;
    procedure CalcMetrics;
    function  fx(v: Double): Integer;
    function  fy(v: Double): Integer;
    procedure DrawLineSegment(x1, y1, x2, y2: Integer);
    procedure DrawYGridlines;
    procedure DrawXTickMarks;
    procedure DrawYTickMarks;
    procedure DrawXLabels;
    procedure DrawYLabels;
    procedure DrawGraphTitle;
    procedure DrawMarks(when: Boolean);
    procedure DrawXAxis;
    procedure DrawYAxis;
    procedure DrawAxes;
    procedure ColorBackground;
    procedure drawbars;
    procedure drawnormal;
    procedure PaintGraph;
    procedure Paint; override;
    procedure DoRescaleEvent;

  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;

   { data routines }
   procedure adddata(xstart, xstep, Mean, SD:double; p:pHistData);
   procedure mycleardata;
   procedure cleardata;
   property data:pHistData read FData;

   procedure HookDataSeries(ds:TDataSeries; ifhookx:boolean);
   procedure unhookDataSeries(wantclear:boolean);
   property Buckets:word read FBuckets write SetBuckets;

   { Marks }
    procedure ClearMarks;
    procedure Addmark(id: Integer; x1,y1, x2,y2: Double;
                   c: Tcolor; name: ShortString;
                   marktype:et_marktype; status:boolean);
    function  DeleteMark(id: Integer): Boolean;

   { PrintOnPage allows more than one graph to be printed on a page; calls
    to it must be preceded by BeginDoc amd followed by EndDoc.
    Print prints only the current graph on its own sheet.
    The graph is printed without change in aspect ratio.
    The size of the printed graph is controlled by PrintScalePct,
    which makes the dimension of the printed graph a percentage of the page
    dimension (on the axis with the larger Screen/Page ratio)
    The location of the upper left corner of the graph is controlled by
    PrintX(Y)OffsetPct, which locate the corner as percentages of the
    page height and width}
    procedure PrintOnPage;
    procedure Print;

  { A convenient way to graph a function; automatically fills graph; handles
    singularities in the function.
    F must be far; parms may be of any type (usu. array of Double) -- no error checks;
    the graph of the function will be drawn from x=x1 to x=x2;
    if x1=x2, will be set to XMin, XMax; if larger, will be clipped to XMin, XMax
    if steps=0, will choose "enough" to be smooth }
    procedure DrawFunction(F: PlotFunction; var parms; x1, x2: Double;
                   color: TColor; style: TPenStyle; steps: Word);

    function  CheckScales: Boolean;
    procedure Debug(i: Integer; fn: String);

  published
    property XAxis: THAxis read FXAxis write FXAxis;
    property YAxis: THAxis read FYAxis write FYAxis;
    property Dimensions: THDimensions read FDimensions write FDimensions;
    property Appearance: THAppearance read FAppearance write FAppearance;

    property OnRescale: TNotifyEvent read FOnRescale write FOnRescale;
    property OnScaleError: TNotifyEvent read FOnScaleError write FOnScaleError;
    property OnPaintEnd: TPaintEvent read FOnPaintEnd write FOnPaintEnd;

    property Align;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderStyle default bsSingle;
    property BorderWidth;
    property Ctl3D;
    property Cursor;
    property Color;
    property Font;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
  end {class THistogram};


procedure Register;

implementation

const
  Loge = 0.4342944818;
  maxgaus = {0.398942} 0.28;

var
  gausfact:double;

{--------------------------------------------------------------------
    #2a. THAxis  - scaling routines
 --------------------------------------------------------------------}

{ step size chosen in a 1,2,5,10 squence depending not only on the
  characteristic, but also the mantissa, of the range}
function THAxis.GetStep: Double;
var
  w, t, B: Double; minSteps: Word;
begin
  minSteps := FGraph.FAppearance.MinSteps;
  w := FMax - FMin;
  if w <= 0 then raise Exception.Create('GetStep entered with bad range');
  t := ln(w)*Loge;
  if t < 0 then t := t - 1;
  B := exp( trunc(t * 1.001) / Loge );

  if         w/B >= minSteps then Result := B
  else if  2*w/B >= minSteps then Result := B/2
  else if  5*w/B >= minSteps then Result := B/5
  else if 10*w/B >= minSteps then Result := B/10
  else if 20*w/B >= minSteps then Result := B/20
  else if 50*w/B >= minSteps then Result := B/50
  else                            Result := B/100;
  {sufficient for maxSteps <= 125}
end {GetStep};


{--------- DoResize helper functions -------}
procedure THAxis.SetMinMax;
var p:pHistdata;
begin
 if Not FIsXAxis then
   begin
   FMin := 0;
   FMax := 0;
   p := FGraph.FData;
   while p <> nil do
     begin
     if p^.freq > FMax then FMax := p^.freq;
     p := p^.next;
     end;
   if Fmax = 0 then FMax := 0.3;
   end
 else
   with Fgraph do
     begin
     FMin := FXStart - FXStep;
     FMax := FXStart + (FXcount+1) * FXStep;
     end;
end;

function THAxis.DoResize: Boolean;
begin {DoResize}
  Result := false;
  if not FIsXAxis and (fGraph.FData = nil) then
  begin
    FMin := 0;
    FMax := 0.3;
    Exit;
  end {not datafound};

 {data: pick intelligent min/max/tick values:}
  if FAutoSizing then
  begin
   SetMinMax;
   if FAutoStepping then FStep := GetStep;
   Result := true;
  end;
end {DoResize};

function OneSigFigDecs(v: Double): Integer;
{eg: OneSigFigDecs(0.1)->1; (100)-> -2}
var
  t: Double;
begin
  t := ln(v*1.01)*Loge;
  if t < 0 then
    Result  := -trunc(t) + 1
  else
    Result  := -trunc(t);
end;


procedure  THAxis.AdjustLabelDecs;
begin
  FLabelDec := OneSigFigDecs(FStep*1.01);
  if FLabelDec < 0 then
    FLabelDec := 0;
end;

function THAxis.GetFirstTick: Double;
begin
  Result := FMin + 0.01*FStep;
  Result := trunc( Result / FStep ) * FStep;
  if (FMin < 0) then Result := Result - FStep;
  if (FMin > Result + 0.01*FStep) then Result := Result + FStep;
  AdjustLabelDecs;
end;

function THAxis.GetNextTick;
begin
  Result := tick + FStep;
end;

function THAxis.CheckScale: Boolean;
begin
  Result := (FMin < FMax);
end;


{--------------------------------------------------------------------
    #2b. THAxis - property servers
 --------------------------------------------------------------------}

procedure THAxis.SetAxisTitle(v: ShortString);
begin
  FAxisTitle := v;
  FGraph.Paint;
end;

procedure THAxis.SetLabelDec(v: Integer);
begin
  FLabelDec := v;
  FGraph.Paint;
end;

procedure THAxis.SetAutoSizing(v: Boolean);
begin
 if not FIsXAxis then
   begin
   FAutoSizing := v;
   FGraph.Paint;
   end;
end;

procedure THAxis.SetAutoStepping(v: Boolean);
begin
 if not FIsXAxis then
   begin
   FAutoStepping := v;
   FGraph.Paint;
   end;
end;

procedure THAxis.SetMax(v: Double);
begin
  if FIsXAxis then exit;
  FAutoSizing := False;
  if FMax <> v then
  begin
    FMax := v;
    FGraph.DoRescaleEvent;
    FGraph.Paint;
  end;
end;

procedure THAxis.SetMin(v: Double);
begin
  if FIsXAxis then exit;
  FAutoSizing := False;
  if FMin <> v then
  begin
    FMin := v;
    FGraph.DoRescaleEvent;
    FGraph.Paint;
  end;
end;

procedure THAxis.SetStep(v: Double);
begin
  if FIsXAxis then exit;
  FAutoStepping := False;
  if FStep <> v then
  begin
    FStep := v;
    FGraph.DoRescaleEvent;
    FGraph.Paint;
  end;
end;

procedure THAxis.SetMinDiff(v: Double);
var holdmin: Double;
begin
  if FIsXAxis then exit;
  holdmin := FMinScale;
  FMinScale := v;
  if ((FMax - FMin < FMinScale) or (FMax - FMin = holdmin))
    then FGraph.Paint;
end;

{-------------------------------------------------------------------
    #3. THDimensions
--------------------------------------------------------------------}

procedure THDimensions.SetMargBottom(v: Word);
begin
  FBottom := v;
  FGraph.Paint;
end;

procedure THDimensions.SetMargTop(v: Word);
begin
  FTop := v;
  FGraph.Paint;
end;

procedure THDimensions.SetMargLeft(v: Word);
begin
  FLeft := v;
  FGraph.Paint;
end;

procedure THDimensions.SetMargRight(v: Word);
begin
  FRight := v;
  FGraph.Paint;
end;

procedure THDimensions.SetTickLength(v: Word);
begin
  FTMLength := v;
  if v = 0 then FGraph.FAppearance.FTickMarks := False;
  FGraph.Paint;
end;

procedure THDimensions.SetGraphTitleDistance(v: Integer);
begin
  FGraphTitleDistance := v;
  FGraph.Paint;
end;

procedure THDimensions.SetXAxisTitleDistance(v: Integer);
begin
  FXAxisTitleDistance := v;
  FGraph.Paint;
end;

procedure THDimensions.SetYAxisTitleDistance(v: Integer);
begin
  FYAxisTitleDistance := v;
  FGraph.Paint;
end;

procedure THDimensions.SetXAxisLabelDistance(v: Integer);
begin
  FXAxisLabelDistance := v;
  FGraph.Paint;
end;

procedure THDimensions.SetYAxisLabelDistance(v: Integer);
begin
  FYAxisLabelDistance := v;
  FGraph.Paint;
end;

procedure THDimensions.SetScale(v: Word);
begin
  if v < 20 then v := 20 else if v > 100 then v := 100;
  FScalePct := v;
end;

procedure THDimensions.SetXOffset(v: Word);
begin
  if v < 0 then v := 0 else if v > 100 then v := 100;
  FXOffsetPct := v;
end;

procedure THDimensions.SetYOffset(v: Word);
begin
  if v < 0 then v := 0 else if v > 100 then v := 100;
  FYOffsetPct := v;
end;


{-------------------------------------------------------------------
    #4. THAppearance
---------------------------------------------------------------------}
procedure THAppearance.SetLabelGraph(v: Boolean);
begin
  FLabelGraph := v;
  FGraph.Paint;
end;

procedure THAppearance.SetGridlines(v: Boolean);
begin
  FGridlines := v;
  FGraph.Paint;
end;

procedure THAppearance.SetTickmarks(v: Boolean);
begin
  FTickMarks := v;
  FGraph.Paint;
end;

procedure THAppearance.SetShowMarks;
begin
  if v <> FShowMarks then
  begin
    FShowMarks := v;
    FGraph.Paint;
  end;
end;

procedure THAppearance.SetGridcolor(v: TColor);
begin
  FGridColor := v;
  FGraph.Paint;
end;

procedure THAppearance.SetAxescolor(v: TColor);
begin
  FAxesColor := v;
  FGraph.Paint;
end;

procedure THAppearance.SetBarBrush(v: TBrush);
begin
  FBarBrush := v;
  FGraph.Paint;
end;

procedure THAppearance.SetBarPen(v: TPen);
begin
  FBarPen := v;
  FGraph.Paint;
end;

procedure THAppearance.SetBarwidth;
begin
  if v <= 100 then FBarwidth := v;
  FGraph.Paint;
end;

procedure THAppearance.SetBkgdcolor(v: TColor);
begin
  FBkgdColor := v;
  FGraph.Paint;
end;

procedure THAppearance.SetGridStyle(v: TPenStyle);
begin
  FGridStyle := v;
  FGraph.Paint;
end;

procedure THAppearance.SetErrorCaption;
begin
  FErrorCaption := v;
  if not FGraph.CheckScales then FGraph.Paint;
end;

procedure THAppearance.SetGraphTitle;
begin
  FGraphTitle := v;
  FGraph.Paint;
end;

procedure THAppearance.SetMinSteps;
begin
  if (v > 0) and (v < FMaxSteps/2.5) then
    FMinSteps := v;
  FGraph.Paint;
end;

procedure THAppearance.SetMaxSteps;
begin
  if (v > FMinSteps*2.5) and (v <= 90) then
    FMaxSteps := v;
  FGraph.Paint;
end;

procedure THAppearance.SetPlotNormal;
begin
  FPlotNormal := v;
  FGraph.paint;
end;

{---------------------------------------------------------------------
    #5a. THistogram  - administration
 ---------------------------------------------------------------------}

constructor THistogram.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCanvas := Canvas;
  FMarks := nil;
  FFirstTime := True;
  FScale := 1.0;
  FXOffset := 0;
  FYOffset := 0;
  FDataSeries := nil;
  FBuckets := 15;

  Fdata := nil;
  FXStart := -2;
  FXStep := 0.5;
  FXcount := 9;
  FMean := 0;
  FSD := 1;

{property defaults}
  Width := 200;
  Height := 200;
  Font.Color := clBlack;
  Font.Height := - 9;
  Font.Name := 'Arial';
  Font.Style := [];
  BevelInner := bvNone;
  BevelOuter := bvNone;
  BorderStyle := bsSingle;
  Color := clWhite;

  FXAxis := THAxis.Create;
  FXAxis.FGraph := self;
  with FXAxis do
  begin
    FIsXAxis := true;
    FMinScale := 0;
    FMin := 0;
    FMax := 5;
    FStep := FMax / 5;
    FAutoSizing := true;
    FAutoStepping := true;
    FAxisTitle := 'X axis';
    FLabelDec := 1;
  end;

  FYAxis := THAxis.Create;
  FYAxis.FGraph := self;
  with FYAxis do
  begin
    FIsXAxis := false;
    FMinScale := defMinYScale;
    FMin := 0;
    FMax := 1.0;
    FStep := FMax / 5;
    FAutoSizing := False;
    FAutoStepping := False;
    FAxisTitle := 'Y axis';
    FLabelDec := 1;
  end;

  FDimensions := THDimensions.Create;
  FDimensions.FGraph := self;
  with FDimensions do
  begin
    FBottom := 40;
    FLeft := 40;
    FRight := 15;
    FTop := 30;
    FTMLength := 4;
    FGraphTitleDistance := 7;
    FXAxisTitleDistance := 4;
    FXAxisLabelDistance := 2;
    FYAxisTitleDistance := 4;
    FYAxisLabelDistance := 2;
    FScalePct := 90;
    FXOffsetPct := 5;
    FYOffsetPct := 20;
  end;

  FAppearance := THAppearance.Create;
  FAppearance.FGraph := self;
  with FAppearance do
  begin
    FLabelGraph := True;
    FShowMarks := False;
    FTickMarks := True;
    FErrorCaption := 'MAX/MIN VALUES INCONSISTENT';
    FAxesColor := clBlack;
    FGridColor := clSilver;
    FGridlines := True;
    FGridStyle := psdot;
    FBkgdColor := color;
    FGraphTitle := 'xyHistogram 0.00';
    FMinSteps := 5;
    FMaxSteps := 50;
    FBarBrush := TBrush.create;
    FBarpen := TPen.create;
    FBarWidth := 100;
  end;

end {Create};


destructor THistogram.Destroy;
var om: pMark;
    p1,p2:PhistData;
begin
  p1 := FData;
  while p1 <> nil do
    begin
    p2 := p1;
    p1 := p1^.next;
    dispose(p2);
    end;
  Fdata := nil;
  FXAxis.Free;
  FYAxis.Free;
  FDimensions.free;
  FAppearance.FBarBrush.free;
  FAppearance.FBarpen.free;
  FAppearance.free;

  while FMarks <> nil do
  begin
    om := FMarks;
    FMarks := FMarks^.next;
    Dispose(om);
  end;
  inherited Destroy;
end;

{---------------------------------------------------------------------
    #5c. THistogram  - metrics routines
 ---------------------------------------------------------------------}

procedure THistogram.DoRescaleEvent;
begin
  if CheckScales and Assigned(FOnRescale) then FOnRescale(self);
end;

procedure THistogram.CalcXMetrics;
begin
  with FXAxis do
  begin
    if (FMin>=FMax) then raise Exception.Create('CalcXMetrics: XMin>=XMax');
    FM := (Width - FDimensions.FLeft - FDimensions.FRight) / (FMax - FMin);
  end;
end;

procedure THistogram.CalcYMetrics;
begin
  with FYAxis do
  begin
    if (FMin>=FMax) then raise Exception.Create('CalcYMetrics: YMin>=YMax');
    FM := (Height - FDimensions.FTop - FDimensions.FBottom) / (FMax - FMin);
  end;
end;

procedure THistogram.CalcMetrics;
begin
  CalcXMetrics;
  CalcYMetrics;
end;

function THistogram.fx(v: Double): Integer;
var w: Double;
begin
  with FXAxis do w :=(v - FMin) * FM;
  if abs(w) > 20000 then
    Result := 20000
  else
    Result := Round(w) + FDimensions.FLeft + FXOffset;
{20000  is large compared with device pixel size; this avoids range
 error if some points are far outside the axis range}
end;

function THistogram.fy(v: Double): Integer;
var w: Double;
begin
  with FYAxis do w := (FMax - v) * FM;
  if abs(w) > 20000 then
    Result := 20000
  else
    Result := Round(w) + FDimensions.FTop + FYOffset;
end;

{---------------------- graph scaling ------------------------------------}
function THistogram.DoResizeX: Boolean;
begin
  Result := FXAxis.DoResize;
end;

function THistogram.DoResizeY: Boolean;
begin
  Result := FYAxis.DoResize;
end;

function THistogram.DoResize: Boolean;
var b: Boolean;
begin
  b := DoResizeY;    {force evaluation of both functions - need side effects!}
  Result := DoResizeX or b;
end;

{---------------------------------------------------------------------
    #5d. THistogram  - painting routines
 ---------------------------------------------------------------------}

procedure THistogram.ClipGraph;
var
  ClipRgn: HRgn;
begin
   ClipRgn := CreateRectRgn(FDimensions.FLeft + FXOffset,
                            FDimensions.FTop + FYOffset,
                            Width - FDimensions.FRight + 1 + FXOffset,
                            Height - FDimensions.FBottom + 1 + FYOffset);
   SelectClipRgn(FCanvas.Handle, ClipRgn);
   DeleteObject(ClipRgn);
end;

procedure THistogram.UnclipGraph;
var
  ClipRgn: HRgn;
begin
  { note for confused readers: the shortcut to unclip that is valid for
       screen metrics is not valid for printers :) }
   ClipRgn := CreateRectRgn(FXOffset,
                            FYOffset,
                            Width + 1 + FXOffset,
                            Height + 1 + FYOffset);
   SelectClipRgn(FCanvas.Handle, ClipRgn);
   DeleteObject(ClipRgn);
end;

procedure THistogram.DrawLineSegment(x1, y1, x2, y2: Integer);
begin
  FCanvas.MoveTo(x1, y1);
  FCanvas.LineTo(x2, y2);
end;

procedure THistogram.DrawYGridlines;
var
  tick, maxTick: Double;
  tx1, tx2, ty1: Word;
  obc:TColor;
begin
  FCanvas.Pen.Color := FAppearance.FGridColor;
  FCanvas.Pen.Style := FAppearance.FGridStyle;
  obc := SetBkColor(FCanvas.handle, FAppearance.FBkgdColor);
  with FYAxis do
    maxTick := FMax + 0.001*(FMax-FMin); { rounding errors might exclude last point }
  with FXAxis do
  begin
    tx1 := fx(FMin);
    tx2 := fx(FMax);
  end;
  with FYAxis do
  begin
    tick := GetFirstTick;
   {don't draw the first grid line on top of axis:}
    if tick < FMin + 0.01*FStep then
      tick := GetNextTick(tick);
    while tick < maxTick do
    begin
      ty1 := fy(tick);
      DrawLineSegment(tx1, ty1, tx2, ty1);
      tick := GetNextTick(tick);
    end;
  end;
  SetBkColor(FCanvas.handle, obc);
end {DrawYGridlines};

procedure THistogram.DrawXTickMarks;
var
  tx, ty1, ty2, i: Word;
begin
  FCanvas.Pen.Color := FAppearance.FAxesColor;
  FCanvas.Pen.Style := psSolid;
  ty1 := fy(FYAxis.FMin);
  ty2 := ty1 + FDimensions.FTMLength;
  for i := 0 to FXCount do
    begin
    tx := fx(FXStart + i * FXStep);
    DrawLineSegment(tx, ty1, tx, ty2);
    end;
end {DrawXTickMarks};

procedure THistogram.DrawYTickMarks;
var
  tick, maxTick: Double;
  tx1, tx2, ty: Word;
begin
  FCanvas.Pen.Color := FAppearance.FAxesColor;
  FCanvas.Pen.Style := psSolid;
  with FYAxis do
    maxTick := FMax + 0.001*(FMax-FMin);
  with FXAxis do
    tx1 := fx(FMin);
  tx2 := tx1 - FDimensions.FTMLength;
  with FYAxis do
  begin
    tick := GetFirstTick;
    while tick < maxTick do
    begin
      ty := fy(tick);
      DrawLineSegment(tx1, ty, tx2, ty);
      tick := GetNextTick(tick);
    end;
  end;
end {DrawYTickMarks};

function  THAxis.LabelString(tick: Double): ShortString;
begin
 if (abs(tick) < (FMax - FMin)*0.001 {zero}) or
        ((abs(tick) > 0.000999) and (abs(tick) < 9999)) then
   Result := FloatToStrF(tick, ffFixed, 5, FLabelDec)
 else {very small or very large}
   Result := FloatToStrF(tick, ffExponent, 2, 0)
end;

procedure THistogram.DrawXLabels;
var
  tick: Double;
  tx, ty, i: Word;
  lblStr: ShortString;
begin
{ X-axis labels }
  ty := fy(FYAxis.FMin) + FDimensions.FTMLength + FDimensions.FXAxisLabelDistance;
  for i := 0 to FXCount do
    begin
    tick := FXStart + i * FXStep;
    lblStr := FXaxis.LabelString(tick);
    FCanvas.TextOut(fx(tick) - FCanvas.TextWidth(lblStr) div 2, ty,lblStr);
    end;
{X-axis title}
  with FDimensions do
  begin
    tx := FLeft + (Width - FLeft - FRight) div 2 + FXOffset;
    ty := fy(FYAxis.FMin) + FTMLength + FXAxisTitleDistance;
  end;
  FCanvas.TextOut(tx - FCanvas.TextWidth(FXAxis.Title) div 2,
                  ty + FCanvas.TextHeight(FXAxis.Title),
                  FXAxis.Title);
end {DrawXLabels};

procedure THistogram.DrawYLabels;
var
  tick, maxTick: Double;
  tx, ty: Integer;
  lblStr: ShortString;
  Angle:integer;
  LogRec: TLOGFONT;              { Storage area for font information }
  OldFont,
  NewFont: HFONT;
  drawIt: Boolean;
begin
{ Y-axis Labels }
  with FYAxis do
    maxTick := FMax + 0.001*(FMax-FMin);   { rounding errors might exclude last point }
  tx := fx(FXAxis.FMin) - FDimensions.FTMLength - FDimensions.FYAxisLabelDistance;
  with FYAxis do
  begin
    tick := GetFirstTick;
    drawIt := true;
    while tick < maxTick do
    begin
      lblStr := LabelString(tick);
      if drawIt or (maxTick - tick < FStep) then
        FCanvas.TextOut(tx - FCanvas.TextWidth(lblStr),
                        fy(tick) - FCanvas.TextHeight(lblStr) div 2,
                        lblStr);
      tick := GetNextTick(tick);
    end;
  end;
{Y-axis title}
  Angle := 90;
  { Get the current font information. We only want to modify the angle }
  GetObject(FCanvas.Font.Handle, SizeOf(LogRec), @LogRec);
  LogRec.lfEscapement := Angle * 10;
  LogRec.lfOrientation := Angle * 10; {see win32 api?}
  LogRec.lfOutPrecision := OUT_TT_ONLY_PRECIS;
       {doesn't seem to work in win95?? not tested in NT. Use a truetype font for
        the font property instead for vertical text}
  NewFont := CreateFontIndirect(LogRec);
  OldFont := SelectObject(FCanvas.Handle, NewFont); {Save old font}
  with FDimensions do
  begin
    tx := fx(FXAxis.FMin) - FTMLength - FYAxisTitleDistance;
    ty := FTop + (Height - FTop - FBottom) div 2 + FYOffset;
  end;
  with FCanvas do
  begin
    tx := tx - TextWidth(lblStr) - TextHeight(FYAxis.Title);
    if tx < 0 then tx := 0;
    TextOut(tx, ty + (TextWidth(FYAxis.Title) div 2), FYAxis.Title);
  end;
  NewFont := SelectObject(FCanvas.Handle, OldFont); {Restore oldfont}
  DeleteObject(NewFont);
end {DrawYLabels};

procedure THistogram.DrawGraphTitle;
var
  tx: Word;
begin
  with FDimensions, FAppearance do
  begin
    tx := FLeft + (Width - FLeft - FRight) div 2 + FXOffset;
    with FCanvas do
      TextOut(tx - TextWidth(FGraphTitle) div 2,
              FTop - FGraphTitleDistance - TextHeight(FGraphTitle) + FYOffset,
              FGraphTitle);
  end;
end {DrawGraphTitle};

procedure THistogram.DrawMarks;

  function inrange(v,min,max:double):boolean;
  begin
   result := (v >= min) and (v <= max);
  end;

  procedure MarkFontOut(x,y:integer;angle:double;
               s:ShortString; offset:integer; color:TColor);
  var LogRec  : TLOGFONT;     {* Storage area for font information *}
      OldFont,
      NewFont: HFONT;
      xoffst,yoffst:integer;
      c:Tcolor;
  begin
    { Get the current font information. We only want to modify the angle }
    c := FCanvas.Font.Color;
    FCanvas.Font.Color := color;
    GetObject(Font.Handle,SizeOf(LogRec),@LogRec);
    LogRec.lfEscapement := trunc(angle *10 * 180 / pi);
    LogRec.lfOrientation := trunc(angle *10 * 180 / pi);    {see win32 api?}
    LogRec.lfOutPrecision := OUT_TT_ONLY_PRECIS;  {doesn't work under win95??}
    NewFont := CreateFontIndirect(LogRec);
    OldFont := SelectObject(Canvas.Handle,NewFont);  {Save old font}
  { offsets }
    xoffst := Round(FScale*(sin(angle) * offset));
    yoffst := Round(FScale*(cos(angle) * offset));
    FCanvas.TextOut(x - xoffst, y - yoffst, s);
    NewFont := SelectObject(Canvas.Handle,OldFont); {Restore oldfont}
    DeleteObject(NewFont);
    FCanvas.font.color := c;
  end {markFontOut};

  procedure drawxmark(p:Pmark);
  begin
    Fcanvas.pen.color := p^.color;
    DrawLineSegment(fx(p^.x1), fy(FYAxis.FMin), fx(p^.x1), fy(FYAxis.FMax));
    MarkFontOut(fx(p^.x1),fy(FYAxis.FMin) - 3, pi/2, p^.caption, FCanvas.font.size + 5,
                p^.color);
  end;

  procedure drawymark(p:Pmark);
  var c:Tcolor;
  begin
    Fcanvas.pen.color := p^.color;
    DrawLineSegment(fx(FXAxis.Fmin), fy(P^.y1), fx(FXAxis.Fmax), fy(p^.y1));
    c := FCanvas.font.color;
    FCanvas.font.color := p^.color;
    FCanvas.TextOut(FDimensions.FLeft + Round(FScale*10) + FXOffset,
                    fy(p^.y1) - Round(FScale*2) - FCanvas.TextHeight(p^.caption),
                    p^.caption);
    FCanvas.font.color := c;
  end;

  procedure drawboxmark(p:Pmark);
  var d:integer;
      c:Tcolor;
  begin
    d := trunc(FScale*p^.x2);
    FCanvas.pen.color := p^.color;
    FCanvas.brush.style := bsclear;
    FCanvas.rectangle(fx(p^.x1) - d, fy(p^.y1) - d,
                      fx(p^.x1) + d, fy(p^.y1) + d);
    c := FCanvas.font.color;
    FCanvas.font.color := p^.color;
    FCanvas.TextOut(fx(p^.x1) - d, fy(p^.y1) + d, p^.caption);
    FCanvas.font.color := c;
  end;

  procedure drawLineMark(p:Pmark);

    procedure swap(var n1,n2:integer);
    var t:integer;
    begin
      t := n1;
      n1 := n2;
      n2 := t;
    end;

  var minx,miny,maxy,maxx:integer;
      angle:double;
  begin {drawLineMark}
    FCanvas.pen.color := p^.color;
    minx := fx(p^.x1);
    miny := fy(p^.y1);
    maxx := fx(p^.x2);
    maxy := fy(p^.y2);
    if maxx < minx then
    begin
      swap(minx,maxx);
      swap(miny,maxy);
    end;
    DrawLineSegment(minx, miny, maxx, maxy);
    if maxx - minx = 0 then
      angle := pi/2
    else
      angle := - arctan((maxy-miny)/(maxx-minx));
      MarkFontOut(minx,miny,angle,p^.caption, FCanvas.font.size + 5, p^.color);
  end {drawLineMark};

var p: pmark;
begin {DrawMarks}
  ClipGraph;
  p := fmarks;
  while p <> nil do
  begin
    if p^.status = when then
      case p^.marktype of
      mtXMark: drawxmark(p);
      mtYMark: drawymark(p);
      mtPoint: drawboxmark(p);
      mtLine : drawLineMark(p);
      end;
    p := p^.next;
  end;
 UnclipGraph;
end {DrawMarks};

procedure THistogram.ColorBackground;
begin
  if (FAppearance.FBkgdColor <> Color) then
  begin
    FCanvas.brush.color := FAppearance.FBkgdColor;
    FCanvas.brush.style := bsSolid;
    FCanvas.pen.color := FAppearance.FBkgdColor;
    FCanvas.pen.style := psSolid;
    FCanvas.rectangle(fx(FXAxis.FMin), fy(FYAxis.FMax),
                     fx(FXAxis.FMax), fy(FYAxis.FMin));
    FCanvas.brush.color := Color;
  end;
end {ColorBackground};

procedure THistogram.DrawBars;
var i:word;
    p:phistdata;
begin
 if FData = nil then exit;
 FCanvas.Brush := FAppearance.FBarBrush;
 FCanvas.Pen := FAppearance.FBarpen;
 p := FData;
 for i := 0 to FXCount do
   begin
   if p = nil then raise exception.create('internal error in bar plotting');
   FCanvas.rectangle(fx(FXStart + (i-FAppearance.FBarWidth/200) * FXstep), fy(p^.freq),
                  fx(FXStart + (i+FAppearance.FBarWidth/200) * FXstep), fy(FYAxis.FMin));
   p := p^.next;
   end;
end;

procedure THistogram.DrawXAxis;
begin
  with FAppearance, FXAxis do
  begin
    if FTickMarks then DrawXTickMarks;
    if FLabelGraph then DrawXLabels;
    FCanvas.Pen.Color := FAxesColor;
    FCanvas.Pen.Style := psSolid;
    DrawLineSegment(fx(FXAxis.FMin), fy(FYAxis.FMin), fx(FXAxis.FMax), fy(FYAxis.FMin));
  end;
end {DrawXAxis};

procedure THistogram.DrawYAxis;
var holdYStep: Double;
begin
  with FAppearance, FYAxis do
  begin
    holdYStep := FStep;
    if ((FStep = 0) or ((FMax - FMin) / FStep > FMaxSteps)) then
      FStep := GetStep;
    if FGridlines then DrawYGridlines;
    if FTickMarks then DrawYTickMarks;
    if FLabelGraph then DrawYLabels;
    FStep := holdYStep;
    FCanvas.Pen.Color := FAxesColor;
    FCanvas.Pen.Style := psSolid;
    DrawLineSegment(fx(FXAxis.FMin), fy(FYAxis.FMax), fx(FXAxis.FMin), fy(FYAxis.FMin));
  end;
end {DrawYAxis};

procedure THistogram.DrawAxes;
begin
  if (FCanvas <> Printer.Canvas) or FAppearance.FBkgdWhenPrint then
    ColorBackground;
  DrawXAxis;
  DrawYAxis;
  if FAppearance.FLabelGraph then DrawGraphTitle;
end;

function NormalDistribution(x: Double; var parms): Double; far;
var denom:real;
    info: array[0..1] of Double absolute parms;
begin
 {if abs((x - mean) / sd) > no_sds then result := 0}
 denom := ((x - info[0]) /info[1]) * ((x - info[0]) / info[1]) / 2;
 result := (gausfact / exp(denom))*maxgaus;
end;

procedure THistogram.DrawFunction(F: PlotFunction; var parms;
                x1, x2: Double;
                color: TColor; style: TPenStyle; steps: Word);
var
  x, step: Double;
  j: Word;
  started: Boolean;
  holdColor: TColor;
  holdStyle: TPenStyle;
begin
  holdColor := FCanvas.Pen.Color;
  holdStyle := FCanvas.Pen.Style;
  FCanvas.Pen.Color := color;
  FCanvas.Pen.Style := style;
  with FXAxis do
    if x1 = x2 then
    begin
      x1 := FMin;
      x2 := FMax;
    end
    else
    begin
      if x1 < FMin then
        x1 := FMin;
      if x2 > FMax then
        x2 := FMax;
    end;
  if (steps <= 0) or (steps > 3500){best likely printer resolution?} then
  begin
  {this can be made more intelligent, but this is usually smooth enough:}
    step := (x2 - x1) / 100;
    steps := 100;
  end
  else
    step := (x2 - x1) / steps;

  x := x1;
  started := false;
  j := 0;
  try
   ClipGraph;
    while j < steps do begin
      while (not started) and (j < steps) do
        try
          FCanvas.MoveTo( fx(x), fy(F(x, parms)) );
          started := true;
        except
          x := x + step;
          Inc(j);
        end;
      while (started) and (j < steps) do
      try
        x := x + step;
        FCanvas.LineTo( fx(x), fy(F(x, parms)) );
        Inc(j);
      except
        Started := false; {skip out-of-support x's}
      end;
    end;
  finally
    FCanvas.Pen.Color := holdColor;
    FCanvas.Pen.Style := holdStyle;
    UnclipGraph;
  end;
end;

procedure THistogram.DrawNormal;
var info: array[0..1] of Double;
begin
 info[0] := FMean;
 info[1] := FSD;
 if (Fmean = 0) and (FSD = 1) then
   drawfunction(NormalDistribution, info, 0, 0, clblack, psSolid, 0)
 else
  drawfunction(NormalDistribution, info, 0, 0, clblack, psSolid, 0);
end;

function THistogram.CheckScales: Boolean;
begin
  Result := FXAxis.CheckScale and FYAxis.CheckScale;
end;

procedure THistogram.PaintGraph;
begin
  CalcMetrics;
  DrawAxes;
  {Printing: linewidth for series, points thinner than for axes}
  if FCanvas.Pen.Width >= 3 then
    FCanvas.Pen.Width := (FCanvas.Pen.Width + 1) div 2;
  if FAppearance.FShowMarks then DrawMarks(drawbefore);
  drawbars;
  if FAppearance.FPlotNormal then DrawNormal;
  if FAppearance.FShowMarks then DrawMarks(drawafter);
end;


procedure THistogram.Paint;
var
  XOK, YOK, xResized, yResized, saveGridLines: Boolean;
  holdMin, holdMax: Double;
begin
  inherited Paint;

  if FFirstTime then
  begin
    FFirstTime := False;
    Caption := Name;
    FAppearance.FCaption := Caption;
  end;

  xResized := false;
  yResized := false;
  XOK := FXAxis.CheckScale;
  YOK := FYAxis.CheckScale;
  with FAppearance do
  begin
    with FXAxis do
    begin
      xResized := DoResizeX;
      XOK := CheckScale;
    end;
    with FYAxis do
    begin
      if FAutoSizing then
      begin
        yResized := DoResizeY;
        YOK := CheckScale;
      end
      else if YOK and FAutoStepping then
        FStep := GetStep
    end;

    if XOK and YOK then
    begin
      if xResized or yResized then
        DoRescaleEvent;
      if (csDesigning in ComponentState) then
      begin
        Caption := FCaption;
        CalcMetrics;
        DrawAxes;
      end
      else
      begin
        Caption := '';
        PaintGraph;
        If Assigned(FOnPaintEnd) then FOnPaintEnd(self, FCanvas);
      end
    end
    else  {at least one axis is bad:}
    begin
      saveGridLines := FGridLines;
      FGridLines := false;
      DrawGraphTitle;
      if not XOK and not YOK then
        Caption := 'BOTH AXES: ' + FErrorCaption
      else if XOK then
      begin
        CalcXMetrics;
       { set arbitrary FMin,FMax values to allow CalcYMetrics
         and fy(FMin,FMax) in x-axis ticks, labels:}
        with FYAxis do
        begin
          holdMin := FMin;
          holdMax := FMax;
          FMin := 1;
          FMax := 10;
          CalcYMetrics; {needed to place axis at correct "x=0" - not at pixel 0}
          DrawXAxis;
          FMin := holdMin;
          FMax := holdMax;
        end;
        Caption := 'Y-AXIS: ' + FErrorCaption;
      end
      else {YOK}
      begin
        CalcYMetrics;
       { set arbitrary FXMin,Max values to allow fx(FXMin,Max)
         in y-axis ticks, labels:}
        with FXAxis do
        begin
          holdMin := FMin;
          holdMax := FMax;
          FMin := 1;
          FMax := 10; {don't need CalcXMetrics, because y-axis is at pixel 0}
          DrawYAxis;
          FMin := holdMin;
          FMax := holdMax;
        end;
        Caption := 'X-AXIS: ' + FErrorCaption;
      end;
      FGridLines := saveGridLines;
    end;
  end;
end {Paint};


procedure THistogram.Print;
begin
  if not CheckScales then Exit;
  Printer.BeginDoc;
  PrintOnPage;
  Printer.EndDoc;
end;

procedure THistogram.PrintOnPage;
var
  scaleX, scaleY: Double;
  saveWidth, saveHeight,
  saveMargLeft, saveMargRight, saveMargTop, saveMargBottom,
  saveTMLength, saveXLabelDistance, saveYLabelDistance,
  saveXAxisDistance, saveYAxisDistance, saveGraphTitleDistance: Integer;
  videoFontColor: TColor;
  videoFontSize: Integer;
  videoFontName: ShortString;
  videoFontStyles: TFontStyles;
  videoPenWidth: Integer;
  VertPixPerInch: Integer;
begin
  if not CheckScales then Exit;
  saveWidth := Width;
  saveHeight := Height;
  with FDimensions do
  begin
    saveMargLeft := FLeft;
    saveMargRight := FRight;
    saveMargTop := FTop;
    saveMargBottom := FBottom;
    saveTMLength := FTMLength;
    saveXLabelDistance := FXAxisLabelDistance;
    saveYLabelDistance := FYAxisLabelDistance;
    saveXAxisDistance := FXAxisTitleDistance;
    saveYAxisDistance := FYAxisTitleDistance;
    saveGraphTitleDistance := FGraphTitleDistance;
  end;
  videoFontName := Font.Name;
  videoFontColor := Font.Color;
  videoFontStyles := Font.Style;
  videoFontSize := Font.Size;
  videoPenWidth := FCanvas.Pen.Width;
 {scale & offsets}
  scaleX := Printer.PageWidth / ClientWidth;
  scaleY := Printer.PageHeight / ClientHeight;
  with FDimensions do
  begin
    if scaleX < scaleY then
    begin
      FScale := FScalePct/100 * scaleX;
      if FScalePct + FXOffsetPct > 100 then
        FXOffsetPct := (100 - FScalePct) div 2;
      if FScalePct + FYOffsetPct > 100*(scaleY/scaleX) then
        FYOffsetPct := (Round(100*(scaleY/scaleX)) - FScalePct) div 2;
    end
    else
    begin
      FScale := FScalePct * scaleY / 100;
      if FScalePct + FYOffsetPct > 100 then
        FYOffsetPct := (100 - FScalePct) div 2;
      if FScalePct + FXOffsetPct > 100*(scaleX/scaleY) then
        FXOffsetPct := (Round(100*(scaleX/scaleY)) - FScalePct) div 2;
    end;
    FXOffset := Round(FXOffsetPct * (Printer.PageWidth / 100));
    FYOffset := Round(FYOffsetPct * (Printer.PageHeight / 100));
  end;
  Width := Round(FScale*Width);
  Height := Round(FScale*Height);
  with FDimensions do
  begin
    FLeft := Round(FScale*FLeft);
    FRight := Round(FScale*FRight);
    FTop := Round(FScale*FTop);
    FBottom := Round(FScale*FBottom);
    FTMLength := Round(FScale*FTMLength);
    FXAxisLabelDistance := Round(FScale*FXAxisLabelDistance);
    FYAxisLabelDistance := Round(FScale*FYAxisLabelDistance);
    FXAxisTitleDistance := Round(FScale*FXAxisTitleDistance);
    FYAxisTitleDistance := Round(FScale*FYAxisTitleDistance);
    FGraphTitleDistance := Round(FScale*FGraphTitleDistance);
  end;
 {set up printer canvas}
  FCanvas := Printer.Canvas;
  with FCanvas do   {for now there are no special printer font properties}
  begin
    Font.Name := videoFontName;
    Font.Color := videoFontColor;
    Font.Style := videoFontStyles;
       {magic of scaling fonts to printer while it's running???}
    VertPixPerInch := GetDeviceCaps(Printer.Handle, LogPixelsY);
    Font.Size := Round((FScale / VertPixPerInch) * videoFontSize * Canvas.Font.PixelsPerInch);
    Pen.Width := Round(videoPenWidth * FScale);
  end;

 {print and restore canvas}
  try
    PaintGraph;
    If assigned(FOnPaintEnd) then FOnPaintEnd(self, FCanvas);
  finally
    FCanvas := Canvas;
    FScale := 1.0;
    FXOffset := 0;
    FYOffset := 0;

    Width := saveWidth;
    Height := saveHeight;
    with FDimensions do
    begin
      FLeft := saveMargLeft;
      FRight := saveMargRight;
      FTop := saveMargTop;
      FBottom := saveMargBottom;
      FTMLength := saveTMLength;
      FXAxisLabelDistance := saveXLabelDistance;
      FYAxisLabelDistance := saveYLabelDistance;
      FXAxisTitleDistance := saveXAxisDistance;
      FYAxisTitleDistance := saveYAxisDistance;
      FGraphTitleDistance := saveGraphTitleDistance;
    end;
  end;
end {PrintOnPage};


{---------------------------------------------------------------------
    #5e. THistogram  - property servers
 ---------------------------------------------------------------------}

{---------------------------------------------------------------------
    #5g. THistogram  - data manipulation
 ---------------------------------------------------------------------}

procedure THistogram.adddata;
var p1:PhistData;
begin
  mycleardata;
  if p <> nil then
    begin
    Fxstart := xstart;
    FXstep := xstep;
    FData := p;
    p1 := FData;
    FXCount := 0;
    while p1 <> nil do
      begin
      inc(FXCount);
      p1 := p1^.next;
      end;
    dec(FXCount);
    end;
  paint;
end;

procedure THistogram.MyClearData;
var p1,p2:PhistData;
begin
  p1 := FData;
  while p1 <> nil do
    begin
    p2 := p1;
    p1 := p1^.next;
    dispose(p2);    
    end;
  Fdata := nil;
  FXStart := -2;
  FXStep := 0.5;
  FXcount := 9;
  FMean := 0;
  FSD := 1;
end;

procedure THistogram.Cleardata;
begin
  MyClearData;
  Paint;
end;

procedure THistogram.readhistogram;
var min,max,diff,mean,sd:double;
    i,total:integer;
    bins:array [0..maxbuckets-1] of integer;
    p,p1,p2:phistdata;
    pp:pgpoint;
  procedure readxvalues;
  begin
    with FDataSeries do
      begin
      min := XMinVal;
      max := XMaxVal;
      mean := XStats.Mean;
      SD := XStats.SD;
      pp := FDataSeries.Data;
      end;
    diff := max - min;
    if diff = 0 then exit;
    while pp <> nil do
      begin
      inc(bins[trunc(((pp^.xv - min) / diff)*Fbuckets)]);
      pp := pp^.next;
      end;
  end;
  procedure readyvalues;
  begin
    with FDataSeries do
      begin
      min := YMinVal;
      max := YMaxVal;
      pp := Data;
      mean := YStats.Mean;
      SD := YStats.SD;
      end;
     diff := max - min;
    if diff = 0 then exit;
    while pp <> nil do
      begin
      inc(bins[trunc(((pp^.yv - min) / diff)*Fbuckets)]);
      pp := pp^.next;
      end;
  end;
begin
  cleardata;
  if FDataSeries = nil then exit;
  for i := 0 to FBuckets - 1 do bins[i] := 0;
  total := FDataSeries.PointNumber;
  if total = 0 then exit;
  if FDataIsX then readxvalues else readyvalues;
  if diff = 0 then exit;
  FXStep := diff / FBuckets;
  FXStart := min + FXStep / 2;
  new(p);
  p^.next := nil;
  p1 := p;
  p^.freq := bins[0]/total;
  for i := 2 to FBuckets do
   begin
   new(p2);
   p2^.freq := bins[i-1]/total;
   p2^.next := nil;
   p1^.next := p2;
   p1 := p2;
   end;
  adddata(FXStart, FXStep, Mean, SD, p);
end;

procedure THistogram.WhenSeriesDataChanges;
begin
  case TheMessage of
   dsUnLoad:UnHookDataSeries(true);
   dsDataChange,dsClearUpdates: readhistogram;
  end;
  {don't care: dsRegressionChange, dsZeroChange
   not consulted: dsChangeLook }
end;

procedure THistogram.SetBuckets(v:word);
begin
  if V < maxbuckets then
   begin
   FBuckets := v;
   if FDataSeries <> nil then readhistogram;
   end;
end;

procedure THistogram.HookDataSeries(ds:TDataSeries; ifhookx:boolean);
begin
  ds.hooks[self] := WhenSeriesDataChanges;
  FDataSeries := ds;
  FDataIsX := ifhookx;
  readhistogram;
end;

procedure THistogram.UnHookDataSeries;
begin
 FDataSeries.hooks[self] := nil;
 FDataSeries := nil;
 if wantclear and not application.terminated then cleardata;
end;
{---------------------------------------------------------------------
    #5h. THistogram  - Marks routines
 ---------------------------------------------------------------------}

procedure THistogram.ClearMarks;
var om: pMark;
begin
  while FMarks <> nil do
  begin
    om := FMarks;
    FMarks := FMarks^.next;
    Dispose(om);
  end;
  Paint;
end;

procedure THistogram.addmark;
var tm,p,p1:pmark;
begin
 p := fmarks;
 while (p <> nil) and (p^.id <> id) do p := p^.next;
 if p <> nil then
 begin
   p^.id := id;
   p^.color := c;
   p^.x1 := x1;
   p^.y1 := y1;
   p^.x2 := x2;
   p^.y2 := y2;
   p^.caption := name;
   p^.marktype := marktype;
   p^.status := status;
 end
 else
 begin
   new(tm);
   tm^.id := id;
   tm^.color := c;
   tm^.x1 := x1;
   tm^.y1 := y1;
   tm^.x2 := x2;
   tm^.y2 := y2;
   tm^.caption := name;
   tm^.marktype := marktype;
   tm^.status := status;
   if (fmarks = nil) or (fmarks^.id > id) then
   begin
     tm^.next := Fmarks;
     Fmarks := tm;
   end
   else
   begin
    p := fmarks^.next;
    p1 := fmarks;
    while (p <> nil) and (p^.id > id) do
    begin
      p1 := p;
      p := p^.next;
    end;
    tm^.next := p;
    p1^.next := tm;
   end;
 end;
 paint;
end;


function THistogram.DeleteMark;
var p, p1: pMark;
begin
  Result := False;
  p := FMarks;
  while p <> nil do
    if p^.id = id then
    begin
      Result := True;
      if p = FMarks then FMarks := p^.next else
      begin
        p1 := FMarks;
        while p1 <> p^.next do p1 := p1^.next;
        p1^.next := p^.next;
        Dispose(p);
      end;
    end
    else p := p^.next;
  if Result then Paint;
end;

{---------------------------------------------------------------------
    #5i. THistogram  - debug routine
 ---------------------------------------------------------------------}

procedure THistogram.Debug;
{var f: system.Text;
  p: TSeries;
  p1: pgPoint;}
begin
(*
  if not findseries(i) then p := nil else p := series[i];
  AssignFile(f, fn);
  Rewrite(f);
  if p <> nil then
  with p do
  begin
    Write(f, 'Series ', p.FSeriesIndex);
    if next = nil then WriteLn(f, ',  last series!') else WriteLn(f);
    if FData = nil then WriteLn(f, '  no data') else WriteLn(f, '  has data');
    if active then WriteLn(f, '  active') else WriteLn(f, '  not active');
    if Changed then WriteLn(f, '  changed') else WriteLn(f, '  not changed');
    if DrawLine then WriteLn(f, '  has line') else WriteLn(f, '  no line');
    if DrawPoints then WriteLn(f, '  has points') else WriteLn(f, '  no points');
    if FillPoints then WriteLn(f, '  filled points') else WriteLn(f, '  empty points');
    WriteLn(f, '  min/max, (x/y), 0: ', xvMin:0:2, ',', xvMax:0:2, ',',
            yvMin:0:2, ',', yvMax:0:2, ',', InternalZero:0:2);
 {$IFDEF STATISTICS}
    if NeedRegreCalc then WriteLn(f, '  need regrcalc') else WriteLn(f, 'regr done');
    case RegType of
      rg_None: WriteLn(f, '  regression: None');
      rg_passingBablok: WriteLn(f, '  regression: Passing-Bablok');
      rg_Linear: WriteLn(f, '  regression: linear');
      rg_RunningAverage: WriteLn(f, '  regression: running average');
      rg_Spline: WriteLn(f, '  regresssion: spline,', RegControl1:0, ',', RegControl2:0:3);
      rg_DWLS: Writeln(f, '  regression: DWLS,', RegControl1:0, ',', RegControl2:0:3);
      rg_quadratic: writeln(f, '  regression: quadratic');
    end;
 {$ENDIF}
    p1 := FData;
    while p1 <> nil do
    begin
      WriteLn(f, p1^.i:0, ',', p1^.xv:0:4, ',', p1^.yv:0:4, ',', p1^.rv:0:4);
      p1 := p1^.next;
    end;
  end
  else WriteLn(f, 'No series ', i:0, ' exists');
  CloseFile(f);*)
end;

{---------------------------------------------------------------------
    #6. THistogram  - register routine
 ---------------------------------------------------------------------}

procedure Register;
begin
  RegisterComponents('TxyGraph', [THistogram]);
end;

begin
  gausfact := 1 / sqrt(2 * pi);
end.

