(* TxyGraph  version 1.0 *)

unit XyGraph;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls;

const numseries    = 6;
      maxpoints    = 600;
(* data is stored internally as an array, for speed and simplicity.
if more points are needed, increase these values. in delphi 16 the
maximum value for maxpoints is about 4000. There is no such limit
in Delphi32. *)
      defminYScale = 0.01;
      defminXScale = 1;

{$IFDEF WINDOWS} type shortstring = string[255]; {$ENDIF}

type

    ELoggingError = class(EMathError);

(* possibilities for point shapes: *)
    et_pointshape = (ps_square, ps_circle, ps_diamond, ps_cross);


(* each point is stored as an x,y coordinate *)
     tgpoint = record
         xv,yv:double;
     end;

(* this data tracks the data and settings for each series *)
type srecord = record
                  active, changed, hasline, haspoints,
                    pointsfilled, autozero, initialised:boolean;
                  xvmin,xvmax,yvmin,yvmax,zeroOffset:double;
                  linecolour,pointcolour:tcolor;
                  pshape:et_pointshape;
                  pointsize,pmax:word; (* !!! pmax tracks last data point *)
        end;

SeriesPointerSize = array [1..maxpoints] of tgpoint;

type
  TxyGraph = class(TCustomPanel)
  private
    { Private declarations }
    series: ARRAY[1..numseries] of ^SeriesPointerSize;
    svals :array [1..numseries] of srecord;

    (* graph settings:
    1. Fplotting / property plotting:
        when Fplotting is true, the graph will update every time the data is
        changed. When Fplotting is false, the caption is shown.
    2. FX/YAutosizing / property AutoX/YSizing:
        when these are set to true, the values for FX/YMin/Max are recalculated
        everytime the graph is drawn. If any have changed, the graph will be
        wiped. The graph is redrawn. Setting the properties Xmin etc sets
        autosizing to off. Note that the x and y axes are separate for Autosizing
    3. FAutoStepping / property AutoStepping
        when this is true the values for XStep / YStep are automatically re-
        calculated when the min/max values change. XStep and Ystep are the
        position of the tick marks and gridlines. Setting XStep / YStep properties
        sets autostepping to false. X&Y are tied together.
    4. Gridlines / Tickmarks: whether these are drawn.
    5. TMlength - size of tickmarks. setting tmLength to 0 overrides the
       TickMarks setting.   *)

    Fxmin, FXmax, FXStep, FYmin, FYmax, FYstep:double; {graph control}
    FMinXScale, FMinYScale:double;
    FXAutoSizing, FYAutoSizing, FAutoStepping, Fplotting, FTickMarks,
    FGridlines, FAllowedOut, FXlogging, FYlogging :boolean;
    FMargLeft,FMargRight,FMargtop,FMargbottom, FTMLength:word;
    xm,ym:real;
    FLabel:Shortstring;
    FOnRescale,FOnNumericalError:TNotifyEvent;
    FGridColour,FAxesColour:TColor;
    FGridStyle:TPenStyle;

    FLabelGraph : boolean;
    XLabelDec, YLabelDec : word;
    FYlabelDistance, FXlabelDistance,
            FYAxisDistance, FXAxisDistance:integer;
    FXAxisTitle, FYAxisTitle, FGraphTitle   : shortstring;

    procedure SetPlotting(v:boolean);
    procedure SetYAutosizing(v:boolean);
    procedure SetXAutosizing(v:boolean);
    procedure SetAutostepping(v:boolean);
    procedure SetAllowedOut(v:boolean);
    procedure SetXmin(v:double);
    procedure SetXmax(v:double);
    procedure SetXstep(v:double);
    procedure SetYmin(v:double);
    procedure SetYmax(v:double);
    procedure SetYstep(v:double);
    procedure SetMinXScale(v:double);
    procedure SetMinYScale(v:double);
    procedure setMargLeft(v:word);
    procedure setMargright(v:word);
    procedure setMargtop(v:word);
    procedure setMargbottom(v:word);

    procedure setXLabelDec(v:word);
    procedure setYLabelDec(v:word);
    procedure setLabelGraph(v:boolean);
    procedure setXAxisTitle(v:shortstring);
    procedure setYAxisTitle(v:shortstring);
    procedure setYlabelDistance(v:integer);
    procedure setXlabelDistance(v:integer);
    procedure setYAxisDistance(v:integer);
    procedure setXAxisDistance(v:integer);
    procedure setXlogging(v:boolean);
    procedure setYlogging(v:boolean);
    procedure setGraphTitle(v:shortstring);
    procedure SetGridlines(v:boolean);
    procedure setTickmarks(v:boolean);
    procedure setTMLength(v:word);
    procedure setLabel(v:shortstring);
    procedure setGridColour(v:Tcolor);
    procedure setAxesColour(v:TColor);
    procedure setGridStyle(v:TPenStyle);

  protected
    { Protected declarations }
    constructor Create(AOwner: TComponent); override;
    destructor destroy;
  (* painting the graph *)
    function resize:boolean;
    procedure extractzero(i:integer);
    procedure getminmax(i:integer);
    procedure calcmetrics;
    procedure recalcsteps;
    function  fx(v:double):integer;
    function  fy(v:double):integer;
    procedure drawgridlines;
    procedure drawtickmarks;
    procedure drawlabels;

    procedure drawaxes;
    procedure drawline(i:integer);
    procedure drawpoints(i:integer);
    procedure drawseries(i:integer);
    procedure Paint; override;

  (* the Rescale event - notify when the graph rescales *)
    procedure DoRescaleEvent;

  (* data utilities *)
    function findx(sN:integer; x:double; var p:word):boolean;
    function datafound:boolean;
    procedure OpenSeries(sN:integer);
    procedure CloseSeries(sN:integer);
    (* these procedures must be called before a series is used,
       but they are called as required by setseries *)

  public

  (* data routines *)
    procedure clear(sN:integer);
        (* destroys data for selected series, if i = 0, all series cleared *)
    function add(sN:integer; x, y:double):boolean;
        (* adds data to a series. returns true if the data was successfully  added *)
    function get(sN:integer; var x,y:double):boolean;
        (* get a data point. y contains the data. returns true if a data  point was
           found *)
    function delete(sN:integer; x:double):boolean;
        (* deletes  data for a selected x point. if sN = 0, all series checked.
           true if a point was found to delete *)
    procedure setseries(sN:integer; makeactive, initialise, autozero:boolean;
                         zerovalue:double);
    function getseries(sn:integer; var zerovalue:double):boolean;
       (* turns a series on or off. if turning the series on, you can choose to
          keep the current data or get rid of it. getseries checks status. note that
          status (active in svals) is basically ignored if no data is present *)
    procedure setseriesline(sn:integer; status:boolean; colour:Tcolor);
    function  getseriesline(sn:integer; var colour:TColor):boolean;
    procedure setseriespoints(sn:integer; status:boolean; size:integer;
                              shape:et_pointshape; filled:boolean; colour:Tcolor);
    function getseriespoints(sn:integer; var size:integer; var shape:et_pointshape;
                              var filled:boolean; var colour:Tcolor):boolean;

(* for mouse events *)
    function getmousexy(sN,x,y:integer; var t:integer;
        var xbyscale,ybyscale,xbypoint,ybypoint:double):boolean;
    function getmousex(x:integer):double;
    function getmousey(y:integer):double;
    function getmousevals(sN,x:integer; var t:integer;
        var xval,yval:double):boolean;

  published
    property Xmin:double read FXmin write setXmin;
    property Xmax:double read FXmax write setXmax;
    property Xstep:double read FXstep write setXstep;
    property XminDiff:double read FminXscale write SetMinXScale;
    property Ymin:double read FYmin write setYmin;
    property YMax:double read FYmax write setYmax;
    property Ystep:double read FYstep write setYstep;
    property YminDiff:double read FminYscale write SetMinYScale;
    property XlogScale:boolean read FXLogging write setXlogging;
    property YlogScale:boolean read FYLogging write setYlogging;
    (* these next 3 properties must come last to override the other properties
       effects on the values at load time *)
    property AutoYSizing:boolean read FYAutoSizing write SetYAutosizing;
    property AutoXSizing:boolean read FXAutoSizing write SetXAutosizing;
    property AutoStepping:boolean read FAutoStepping write SetAutostepping;
    property PlotOutOfScale:boolean read FAllowedOut write SetAllowedOut;
    property marginleft:word read FMargLeft write setMargLeft;
    property marginright:word read FMargright write setMargright;
    property margintop:word read FMargtop write setMargtop;
    property marginbottom:word read FMargbottom write setMargbottom;
    property Gridlines:boolean read FGridlines write SetGridlines;
    property Text:shortstring read FLabel write SetLabel;
    property TickMarks:boolean read FTickMarks write setTickmarks;
    property TMLength:word read FTMLength write setTMLength;
    property GridColour:TColor read FGridColour write setGridColour;
    property AxesColour:Tcolor read FAxesColour write setAxesColour;
    property GridStyle:TPenStyle read FGridStyle write setGridStyle;

    property XLabelDecimals:word read XLabelDec write setXLabelDec;
    property YLabelDecimals:word read YLabelDec write setYLabelDec;
    property LabelGraph:boolean read FLabelGraph write setLabelGraph;
    property XAxisTitle : shortstring read FXAxisTitle write setXAxisTitle;
    property YAxisTitle : shortstring read FYAxisTitle write setYAxisTitle;
    property YlabelDistance:integer read FYlabelDistance write setYlabelDistance;
    property XlabelDistance:integer read FXlabelDistance write setXlabelDistance;
    property YAxisTitleDist:integer read FYAxisDistance write setYAxisDistance;
    property XAxisTitleDist:integer read FXAxisDistance write setXAxisDistance;
    property GraphTitle : shortstring read FGraphTitle write setGraphTitle;
    property plotting:boolean read FPlotting write setPlotting;

    property Align;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderStyle;
    property BorderWidth;
    property Ctl3D;
    property cursor;
    property Color;
    property Font;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnReScale:TNotifyEvent read FOnRescale write FOnRescale;
    property OnNumericalError:TNotifyEvent read FOnNumericalError
                 write FOnNumericalError;
    property OnResize;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('xyGraph', [TxyGraph]);
end;

(*------------------------- administrative crap ------------------------------*)

constructor TxyGraph.Create(AOwner: TComponent);
var i:integer;
begin
  inherited create(aowner);
  FMinXscale := defMinXscale;
  FMinYScale := defMinYscale;
  for i := 1 to numseries do
   with svals[i] do
    begin
     initialised := false;
     active := false;
     changed := false;
     xvmin := 0;
     xvmax := FminXscale;
     yvmin := 0;
     yvmax := FminYscale;
     linecolour := rgb(0,0,0);
     pointcolour := rgb(0,0,0);
     pmax := 0;
     hasline := true;
     haspoints := true;
     pointsize := 5;
    end;
  Fxmin := 0;
  FXmax := FminXScale;
  FXStep := FminXScale / 10;
  FYmin := 0;
  FYmax := FMinYscale;
  FYstep := FMinYscale / 10;
  FXAutoSizing := true;
  FYAutoSizing := true;
  FXlogging := false;
  FYlogging := false;
  FAutoStepping := true;
  Fplotting := false;
  FMargLeft := 10;
  FMargRight := 10;
  FMargtop := 10;
  FMargbottom := 10;
  FGridLines := false;
  FTickMarks := true;
  FTMLength := 4;
  Caption := '(Graph)';
  FLabel := '(Graph)';
end;

destructor TxyGraph.Destroy;
var i:Integer;
begin
  for i := 1 to numseries do
   if svals[i].initialised then closeseries(i);
  inherited destroy;
end;

(*---------------------------- Graph Metrics ------------------------------*)

procedure TxyGraph.DoRescaleEvent;
begin
  if Fplotting and Assigned(FOnRescale) then FOnRescale(self);
end;

procedure TxyGraph.extractzero(i:integer);
var t:double;
    j:integer;
begin
 if svals[i].autozero then
  begin
   t := series[i]^[1].yv;
   for j := 2 to svals[i].pmax do
    if series[i]^[j].yv < t then t := series[i]^[j].yv;
   svals[i].zeroOffset := t;
  end;
end;

procedure TxyGraph.getminmax(i:integer);
var j:word;
begin
 with svals[i] do
  if pmax <> 0 then
   begin
    extractzero(i);
    yvmin := series[i]^[1].yv - zeroOffset;
    yvmax := series[i]^[1].yv - zeroOffset;
    xvmin := series[i]^[1].xv;
    xvmax := series[i]^[1].xv;
    for j := 2 to pmax do
     begin
      if yvmin > series[i]^[j].yv - zeroOffset
              then yvmin := series[i]^[j].yv - zeroOffset;
      if yvmax < series[i]^[j].yv - zeroOffset
              then yvmax := series[i]^[j].yv - zeroOffset;
      if xvmin > series[i]^[j].xv then xvmin := series[i]^[j].xv;
      if xvmax < series[i]^[j].xv then xvmax := series[i]^[j].xv;
     end;
   end;
 svals[i].changed := false;
end;

function getstep(w:double):double;
var i,t:double;
    f,j:integer;
begin
(* get factor of ten *)
 i := ln(w) / 2.30258;
 if i < 0 then i := i - 1;
 f := trunc(i);
 t := 1;
 if f > 0
  then for j := 1 to f do t := t * 10
  else for j := -1 downto f do t := t / 10;
 result := 0.2 * t;
end;

procedure TxyGraph.recalcsteps;
begin
  If FAutoStepping then
   begin
    if FXlogging then FXStep := 10 else FXStep := getstep(FXmax - FXmin);
    if FYlogging then FYStep := 10 else FyStep := getstep(Fymax - Fymin);
   end;
end;

function TxyGraph.resize;
var i:integer;
    oxmax,oxmin,oymax,oymin:double;
    yresize,xresize,init:boolean;
begin
 if not datafound then
   begin
    if FXAutoSizing then
     begin
      FXmin := 0;
      FXmax := FXstep * 10;
      if FXmax - FXMin < FminXScale then FXmax := FXmin + FMinXScale;
      result := true;
     end;
    if FYAutoSizing then
     begin
      FYmin := 0;
      FYmax := FYstep * 10;
      if FYmax - FYMin < FminYScale then FYmax := FYmin + FMinYScale;
      result := true;
     end;
   end
  else
 begin
  if (FXAutosizing or FYAutosizing) then
    for i := 1 to numseries do if svals[i].changed then getminmax(i);
  if FYAutoSizing then
   begin
    init := false;
    for i := 1 to numseries do if svals[i].active then
     begin
      if not init then
        begin
         oYmin := svals[i].yvmin;
         oYmax := svals[i].yvmax;
         init := true;
        end
       else
        begin
         if svals[i].yvmin < oYmin then oYmin := svals[i].yvmin;
         if svals[i].yvmax > oYmax then oYmax := svals[i].yvmax;
        end;
     end;
    if oYmax - oYmin < FminYscale then oYMax := oYmin + FminYscale;
    yresize := (oymax <> FYmax) or (oymin <> FYmin);
    if yresize then begin FYmax := oymax; FYmin := oymin; end;
   end else yresize := false;
  if FXAutosizing then
   begin
    init := false;
    for i := 1 to numseries do if svals[i].active then
     begin
      if not init then
        begin
         oXmin := svals[i].xvmin;
         oXmax := svals[i].xvmax;
         init := true;
        end
       else
        begin
         if svals[i].xvmin < oXmin then oXmin := svals[i].xvmin;
         if svals[i].xvmax > oXmax then oXmax := svals[i].xvmax;
        end;
     end;
    if oXmax - oXmin < FminXscale then oXMax := oXmin + FminXscale;
    xresize := (oxmax <> Fxmax) or (oxmin <> Fxmin);
    if xresize then begin Fxmax := oxmax; Fxmin := oxmin; end;
   end else xresize := false;
  result := (yresize or xresize);
  if result then recalcsteps;
 end;
 if result then DoRescaleEvent;
 calcmetrics;
end;

procedure TxyGraph.calcmetrics;
begin
  (* logging - can't log a negative number. Log will not be called for any
  number smaller then min. so if min > 0 we are ok.*)
  if (FXLogging and (FXmin <= 0)) or (FYlogging and (FYmin <= 0)) then
   raise ELoggingError.Create('Attempt to log scale a non-positive number');
  (* get an y = mx + c equation for x and y;  *)
  if FXLogging
   then xm := (width - FMargLeft - FMargRight) / (ln(FXmax) - ln(FXmin))
   else xm := (width - FMargLeft - FMargRight) / (FXmax - FXmin);
  if FYlogging
   then ym := (height - FMargTop - FMargBottom) / (ln(FYmax) - ln(FYmin))
   else ym := (height - FMargTop - FMargBottom) / (FYmax - FYmin);
end;

function TxyGraph.fx(v:double):integer;
begin
  if FXlogging
   then result := trunc((ln(v) - ln(FXmin)) * xm + FMargLeft)
   else result := trunc((v - FXmin) * xm + FMargLeft);
end;

function TxyGraph.fy(v:double):integer;
begin
  if FYlogging
   then result := trunc((ln(FYmax) - ln(v)) * ym + FMargTop)
   else result := trunc((FYmax - v) * ym + FMargTop);
end;

(* reverse the metrics! *)
function TxyGraph.getmousexy(sN,x,y:integer; var t:integer; var xbyscale,ybyscale,
                          xbypoint,ybypoint:double):boolean;
begin
 xbyscale := getmousex(x);
 ybyscale := getmousey(y);
 result := getmousevals(sN,x,t,xbypoint,ybypoint);
end;

(* getmousex/y return the value as determined by the scales *)
function TxyGraph.getmousex(x:integer):double;
begin
  if FXlogging
   then result := Exp(FXmin + ((x - FMargLeft) / xm))
   else result := FXmin + ((x - FMargLeft) / xm);
end;

function TxyGraph.getmousey(y:integer):double;
begin
  if FYLogging
   then result := exp(FYMax - ((y - FMargtop) / ym))
   else result := FYMax - ((y - FMargtop) / ym);
end;

function average(d1,d2:double):double;
begin
  result := (d1 + d2) / 2;
end;

(* getmousevals returns the value of the data point matching the
mouse point best, or false if the mouse is outside the plot area *)
function TxyGraph.getmousevals(sN,x:integer; var t:integer;
                            var xval,yval:double):boolean;
var temp:double;
    i,m:integer;
begin
 temp := getmousex(x);
 result := (svals[sN].pmax > 0) and ((temp <= FXmax) and (temp >= FXmin));
 if result then
  begin
    if temp <= series[sN]^[1].xv then m := 1 else
      begin
       i := 1;
       while (i < svals[sN].pmax) and (temp > series[sN]^[i].xv) do inc(i);
       if i = svals[sN].pmax then m := i else
        if temp = series[sN]^[i].xv then m := i else
         if temp > average(series[sN]^[i].xv, series[sN]^[i - 1].xv)
          then m := i else m := i - 1;
      end;
    t := m;
    xval := series[sN]^[m].xv;
    yval := series[sN]^[m].yv;
  end;
end;

(*---------------------------- Graph Drawing -------------------------------*)

function getfirsttick(step, min:double;logging:boolean):double;
var t:double;
begin
 if logging then
   begin
    result := 1;
    while result < min do result := result * step;
    while result > min do result := result / step;
    if result < min then result := result * Step;
   end
  else
   begin
    result := 0;
    while result < min do result := result + step;
    while result > min do result := result - step;
    if result < min then result := result + Step;
   end;
end;

function getnexttick(step,last:double; logging:boolean):double;
begin
 if logging then result := last * step else result := last + step;
end;


procedure TxyGraph.drawgridlines;
var count:double;
begin
 canvas.pen.color := FGridColour;
 canvas.pen.style := FGridStyle;
 count := getfirsttick(FXStep,FXmin,FXlogging);
 while count < Fxmax do
  begin
    canvas.moveto(fx(count),fy(ymin));
    canvas.lineto(fx(count),fy(ymax));
    count := getnexttick(FXStep,count,FXLogging);
  end;
 count := getfirsttick(FYStep,FYmin,FYlogging);
 while count < Fymax do
  begin
    canvas.moveto(fx(xmin),fy(count));
    canvas.lineto(fx(xmax),fy(count));
    count := getnexttick(FYStep,count,FYLogging);
  end;
end;

procedure TxyGraph.drawtickmarks;
var count:double;
begin
 canvas.pen.color := FAxesColour;
 canvas.pen.style := pssolid;
 count := getfirsttick(FXStep,FXmin,FXlogging);
 while count < Fxmax do
  begin
    canvas.moveto(fx(count),fy(ymin));
    canvas.lineto(fx(count),fy(ymin) + FTMLength);
    count := getnexttick(FXStep,count,FXlogging);
  end;
 count := getfirsttick(FYstep, FYmin,FYlogging);
 while count < Fymax do
  begin
    canvas.moveto(fx(xmin),fy(count));
    canvas.lineto(fx(xmin) - TMLength,fy(count));
    count := getnexttick(FyStep,count,FYlogging);
  end;
end;

procedure TxyGraph.DrawLabels;
var count   :double;
    Value   : string;
    Angle,
    YAxisTextWidth   : integer;
    LogRec  : TLOGFONT;     {* Storage area for font information *}
    OldFont,
    NewFont: HFONT;

begin
 count := getfirsttick(FXStep,FXmin,FXlogging);
 while count < Fxmax do
  begin
    Value := FloatToStrF(Count,FFFixed,7,XLabelDec);  {Added by PB}
    Canvas.TextOut(fx(count)-(Canvas.TextWidth(Value) div 2),
                  fy(ymin) + FTMLength + FXLabelDistance, Value);
    count := getnexttick(FXStep,count,FXlogging);
  end;
 count := getfirsttick(FYstep,FYMin,FYlogging);
 while count < Fymax do
  begin
    Value := FloatToStrF(Count,FFFixed,7, YLabelDec);  {Added by PB}
    Canvas.TextOut(fx(xmin) - FTMLength -Canvas.TextWidth(Value) - FYLabelDistance,
                 fy(count)-(Canvas.TextHeight(Value) div 2), Value);
    count := getnexttick(FyStep,count,FYlogging);
  end;

  {Graph}
  Canvas.TextOut(MarginLeft + ((Width-MarginLeft-MarginRight) div 2) - (Canvas.TextWidth(FGraphTitle) div 2),
                 MarginTop-Canvas.TextHeight(FGraphTitle), FGraphTitle);

  {Xaxis}
  Canvas.TextOut(MarginLeft + ((Width-MarginLeft-MarginRight) div 2) - (Canvas.TextWidth(FXAxisTitle) div 2),
                  fy(ymin) + FTMLength + FXAxisDistance +(Canvas.TextHeight(FXAxisTitle)), FXAxisTitle);

  {Now for the Y axis}
  Angle := 90;
  {* Get the current font information. We only want to modify the angle *}
  GetObject(Font.Handle,SizeOf(LogRec),@LogRec);
  LogRec.lfEscapement := Angle*10;
  LogRec.lfOrientation := Angle*10;    {see win32 api?}
  LogRec.lfOutPrecision := OUT_TT_ONLY_PRECIS;
  NewFont := CreateFontIndirect(LogRec);
  OldFont := SelectObject(Canvas.Handle,NewFont);  {Save old font}
  Canvas.TextOut(fx(xmin) - TMLength-Canvas.TextWidth(Value)-Canvas.TextHeight(FYAxisTitle)-FYAxisDistance,
                 MarginTop +  ((Height-MarginTop-MarginBottom) div 2) + (Canvas.TextWidth(FYAxisTitle) div 2)
                 , FYAxisTitle);

  NewFont := SelectObject(Canvas.Handle,OldFont); {Restore oldfont}
  DeleteObject(NewFont);
end;

procedure TxyGraph.drawaxes;
begin
 if FGridlines then drawgridlines;
 if FTickMarks then drawtickmarks;
 IF FLabelGraph then drawlabels;

 canvas.pen.color := FAxesColour;
 canvas.pen.style := psSolid;
 canvas.moveto(fx(Fxmin), fy(Fymax));
 canvas.lineto(fx(Fxmin), fy(Fymin));
 canvas.lineto(fx(Fxmax), fy(Fymin));
end;

procedure TxyGraph.drawline(i:integer);
function interpolate(x1,y1,x2,y2,midx:double):double;
var m,c:double;
begin
  result := y1 + ((midx - x1) / (x2 - x1)) * (y2 - y1);
end;

procedure maybedraw(x1,y1,x2,y2:double);
var x1in,x2in,y1in,y2in:boolean;
begin
 x1in := (x1 <= Fxmax) and (x1 >= FXmin);
 x2in := (x2 <= Fxmax) and (x2 >= FXmin);
 y1in := (y1 <= Fymax) and (y1 >= Fymin);
 y2in := (y2 <= Fymax) and (y2 >= Fymin);
 if x1in and x2in and y1in and y2in then
  begin
   canvas.moveto(fx(x1),fy(y1));
   canvas.lineto(fx(x2),fy(y2));
  end else
 if ( (not x1in) and (not x2in) ) or ( (not y1in) and (not y2in) )
   or (x2 = FXMin) or (x1 = FXMax) then
  {don't do anything ! } else
  begin
   if not x1in then
     begin
      x1 := FXMin;
      y1 := interpolate(x1,y1,x2,y2,FXmin);
     end;
   if not x2in then
     begin
      x2 := FXmax;
      y2 := interpolate(x1,y1,x2,y2,FXMax);
     end;
   if not x1in or not x2in then maybedraw(x1,y1,x2,y2) else
       (* this recursive call to maybedraw can only happen once *)
    begin
     (* x's are ok. One of the y's is out *)
     if not y1in then
      begin
        if y1 < FYmin then
          begin
           x1 := interpolate(y1,x1,y2,x2,FYmin);
           y1 := FYmin;
          end
         else
          begin
           x1 := interpolate(y1,x1,y2,x2,FYmax);
           y1 := FYmax;
          end;
      end;
     if not y2in then
      begin
        if y2 < FYmin then
          begin
           x2 := interpolate(y1,x1,y2,x2,FYmin);
           y2 := FYMin;
          end
         else
          begin
           x2 := interpolate(y1,x1,y2,x2,FYmax);
           y2 := FYmax;
          end;
      end;
     canvas.moveto(fx(x1),fy(y1));
     canvas.lineto(fx(x2),fy(y2));
    end;
  end;
end;

var j,max:integer;
begin
 if svals[i].pmax > 0 then
  begin
   canvas.pen.color := svals[i].linecolour;
   canvas.pen.style := pssolid;
   if FAllowedOut then
     begin
      canvas.moveto(fx(series[i]^[1].xv),fy(series[i]^[1].yv - svals[i].zeroOffset));
      for j := 2 to svals[i].pmax do
        canvas.lineto(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset))
     end
    else for j := 2 to svals[i].pmax do
     maybedraw(series[i]^[j-1].xv,series[i]^[j-1].yv - svals[i].zeroOffset,
                    series[i]^[j].xv, series[i]^[j].yv - svals[i].zeroOffset);
  end;
end;

procedure TxyGraph.drawpoints(i:integer);
var s1size,s2size:integer;

procedure makemetrics;
begin
 s1size := svals[i].pointsize div 2;
 s2size := s1size + svals[i].pointsize mod 2;
end;

procedure plotsquare(x,y:integer);
begin
 canvas.rectangle(x - s1size, y - s1size, x + s2size, y + s2size);
end;

procedure plotcircle(x,y:integer);
begin
 canvas.ellipse(x - s1size, y - s1size, x + s2size, y + s2size);
end;

procedure plotdiamond(x,y:integer);
begin
  Canvas.Polygon([Point(x, y - s1size), Point(x + s2size, y),
    Point(x, y + s2size), Point(x - s1size, y)]);
end;

procedure plotcross(x,y:integer);
begin
 canvas.moveto(x-s1size,y);
 canvas.lineto(x+s1size,y);
 canvas.moveto(x,y-s1size);
 canvas.lineto(x,y+s2size);
end;

var j:integer;
    max:double;
begin
if svals[i].pmax > 0 then
 begin
  canvas.pen.color := svals[i].pointcolour;
  if svals[i].pointsfilled
   then canvas.brush.color := svals[i].pointcolour
   else canvas.brush.color := color;
  makemetrics;
  if FAllowedout then
    begin
     case svals[i].pshape of (* only test this once! SPEED ---> *)
       ps_square: for j := 1 to svals[i].pmax do
            plotsquare(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset));
       ps_circle: for j := 1 to svals[i].pmax do
            plotcircle(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset));
       ps_diamond:for j := 1 to svals[i].pmax do
            plotdiamond(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset));
       ps_cross:  for j := 1 to svals[i].pmax do
            plotcross(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset));
      end;
    end
   else
    begin
     max := svals[i].pmax;
     j := 1;
     while (j <= max) and (series[i]^[j].xv < FXmin) do inc(j);
     case svals[i].pshape of
       ps_square:while (j <= max) and (series[i]^[j].xv <= FXMax) do
         begin
          if (series[i]^[j].yv - svals[i].zeroOffset >= FYmin) and (series[i]^[j].yv - svals[i].zeroOffset <= FYmax) then
            plotsquare(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset));
          inc(j);
         end;
       ps_circle:while (j <= max) and (series[i]^[j].xv <= FXMax) do
         begin
          if (series[i]^[j].yv - svals[i].zeroOffset >= FYmin) and (series[i]^[j].yv - svals[i].zeroOffset <= FYmax) then
            plotcircle(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset));
          inc(j);
         end;
        ps_diamond:while (j <= max) and (series[i]^[j].xv <= FXMax) do
         begin
          if (series[i]^[j].yv - svals[i].zeroOffset >= FYmin) and (series[i]^[j].yv - svals[i].zeroOffset <= FYmax) then
            plotdiamond(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset));
          inc(j);
         end;
       ps_cross:while (j <= max) and (series[i]^[j].xv <= FXMax) do
         begin
          if (series[i]^[j].yv - svals[i].zeroOffset >= FYmin) and (series[i]^[j].yv - svals[i].zeroOffset <= FYmax) then
            plotcross(fx(series[i]^[j].xv),fy(series[i]^[j].yv - svals[i].zeroOffset));
          inc(j);
         end;
      end;
    end
  end;
end;

procedure TxyGraph.drawseries(i:integer);
begin
 if svals[i].hasline then drawline(i);
 if svals[i].haspoints then drawpoints(i);
end;

procedure TxyGraph.Paint;
var i:integer;
begin
 inherited paint;
 if Fplotting then
  begin
   try
    resize;
    drawaxes;
    for i := 1 to numseries do
     if svals[i].active and (svals[i].pmax > 0) then drawseries(i);
   except
    on ELoggingError do
     begin
      Fplotting := false;
      if Assigned(FOnNumericalError)
       then FOnNumericalError(self)
       {else raise; ?? not sure if this is a good idea or not?}
     end;
   end;
  end;
end;

(*---------------------------- Graph appearance  ----------------------------*)
procedure TxyGraph.SetPlotting(v:boolean);
begin
  if v then caption := '' else caption := FLabel;
  Fplotting := v;
  paint;
end;

procedure TxyGraph.SetXAutosizing(v:boolean);
begin
  FXAutoSizing := v;
  if v and FPlotting then paint;
end;

procedure TxyGraph.SetYAutosizing(v:boolean);
begin
  FYAutoSizing := v;
  if v and FPlotting then paint;
end;

procedure TxyGraph.SetAutostepping(v:boolean);
begin
  FAutoStepping := v;
  if v and FPlotting then paint;
end;

procedure TxyGraph.SetAllowedOut;
begin
  FAllowedOut := v;
  if FPLotting then paint;
end;

procedure TxyGraph.setMargLeft(v:word);
begin
  FMargLeft := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setMargright(v:word);
begin
  FMargright := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setMargtop(v:word);
begin
  FMargtop := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setMargbottom(v:word);
begin
  FMargbottom := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setXLabelDec(v:word);
begin
  XLabelDec := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setYLabelDec(v:word);
begin
  YLabelDec := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setYlabelDistance;
begin
  FYlabelDistance := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setXlabelDistance;
begin
  FXlabelDistance := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setXlogging;
begin
 FXLogging := v;
 if fplotting then paint;
end;

procedure TxyGraph.setYlogging;
begin
 FYLogging := v;
 if fplotting then paint;
end;

procedure TxyGraph.setYAxisDistance;
begin
  FYAxisDistance := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setXAxisDistance;
begin
  FXAxisDistance := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setLabelGraph(v:boolean);
begin
  FLabelGraph := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setXAxisTitle;
begin
  FXAxisTitle := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setYAxisTitle;
begin
  FYAxisTitle := v;
  if Fplotting then paint;
end;

procedure TxyGraph.setGraphTitle;
begin
  FGraphTitle := v;
  if Fplotting then paint;
end;


procedure TxyGraph.SetXmin(v:double);
var change:boolean;
begin
  FXAutosizing := false;
  if FXmin <> v then
   begin
    FXmin := v;
    DoRescaleEvent;
    If FPlotting then paint;
   end;
end;

procedure TxyGraph.SetminXScale(v:double);
var omin:double;
begin
 omin := FMinXscale;
 FMinXscale := v;
 if Fplotting and ((FXmax - FXMin < FMinXscale) or (FXmax - FXMin = omin))
    then paint;
end;

procedure TxyGraph.SetminYScale(v:double);
var omin:double;
begin
 omin := FMinYscale;
 FMinYscale := v;
 if Fplotting and ((FYmax - FYMin < FMinYscale) or (FYmax - FYmin = omin))
    then paint;
end;

procedure TxyGraph.SetXmax(v:double);
begin
  FXAutosizing := false;
  if FXmax <> v then
   begin
    FXmax := v;
    DoRescaleEvent;
    If FPlotting then paint;
   end;
end;

procedure TxyGraph.SetXstep(v:double);
begin
  FAutostepping := false;
  if FXstep <> v then
   begin
    FXStep := v;
    DoRescaleEvent;
    If FPlotting then paint;
   end;
end;

procedure TxyGraph.SetYmin(v:double);
begin
  FYAutosizing := false;
  if FYmin <> v then
   begin
    FYmin := v;
    DoRescaleEvent;
    If FPlotting then paint;
   end;
end;

procedure TxyGraph.SetYmax(v:double);
begin
 FYAutoSizing := false;
 if FYmax <> v then
  begin
   FYmax := v;
    DoRescaleEvent;
   If FPlotting then paint;
  end;
end;

procedure TxyGraph.SetYstep(v:double);
begin
  FAutostepping := false;
  if FYstep <> v then
   begin
    FYstep := v;
    DoRescaleEvent;
    If FPlotting then paint;
   end;
end;

procedure TxyGraph.SetGridlines(v:boolean);
begin
 FGridlines := v;
 if Fplotting then paint;
end;

procedure TxyGraph.setTickmarks(v:boolean);
begin
 FTickmarks := v;
 if Fplotting and (FTMLength > 0) then paint;
end;

procedure TxyGraph.setTMLength(v:word);
var ov:word;
begin
 ov := FTMlength;
 FTMLength := v;
 if v = 0 then FTickMarks := false;
 if FPLotting and FTickmarks then paint;
end;

procedure TxyGraph.setGridColour(v:Tcolor);
begin
 FGridColour := v;
 if Fplotting then paint;
end;

procedure TxyGraph.setAxesColour(v:TColor);
begin
 FAxesColour := v;
 if FPLotting then paint;
end;

procedure TxyGraph.setGridStyle(v:TPenStyle);
begin
 FGridStyle := v;
 if Fplotting then paint;
end;

procedure TxyGraph.SetLabel;
begin
 FLabel := v;
 if not FPlotting then Caption := v;
end;

(*------------------------ series settings ---------------------------------*)
procedure TxyGraph.setseries;
begin
 if sN in [1..numseries] then
  begin
   svals[sN].active := makeactive;
   if makeactive then
    begin
     if not svals[sN].initialised then openseries(sN);
     if initialise then
       begin
       svals[sN].pmax := 0;
       svals[sN].changed := true;
      end;
     svals[sN].autoZero := autozero;
     if not autoZero then svals[sN].zeroOffset := zerovalue;
    end;
   if FPlotting then paint;
  end;
end;

procedure TxyGraph.OpenSeries(sN:integer);
begin
  new(series[sN]);
  svals[sN].initialised := true;
end;

procedure TxyGraph.CloseSeries(sN:integer);
begin
  dispose(series[sN]);
  svals[sN].initialised := false;
end;

function TxyGraph.getseries;
begin
 if sN in [1..numseries]
  then
   begin
    result := svals[sN].active;
    zerovalue := svals[sN].zeroOffset;
   end
  else result := false;
end;

procedure TxyGraph.setseriesline;
begin
 if sN in [1..numseries] then
  begin
   svals[sN].hasline := status;
   svals[sN].linecolour := colour;
   if FPlotting then paint;
  end;
end;

function TxyGraph.getseriesline;
begin
 if sN in [1..numseries] then
  begin
   colour := svals[sN].linecolour;
   result := svals[sN].hasline;
  end else result := false;
end;

procedure TxyGraph.setseriespoints;
begin
 if sN in [1..numseries] then
  begin
   svals[sN].haspoints := status;
   svals[sN].pointsize := size;
   svals[sN].pointcolour := colour;
   svals[sN].pshape := shape;
   svals[sN].pointsfilled := filled;
   if FPlotting then paint;
  end;
end;

function TxyGraph.getseriespoints;
begin
 if sN in [1..numseries] then
  begin
   result := svals[sN].haspoints;
   colour := svals[sN].pointcolour;
   size := svals[sN].pointsize;
   shape := svals[sN].pshape;
   filled := svals[sN].pointsfilled;
  end else result := false;
end;

(*------------------------- data manipulation --------------------------*)

function TxyGraph.datafound;
var i:integer;
begin
 result := false;
 for i := 1 to numseries do
   result := result or (svals[i].pmax > 0);
end;

procedure TxyGraph.clear;
var j,b,e:integer;
begin
 if sN in [0..numseries] then
  begin
   b := sN; e := sN;
   if sN = 0 then begin b := 1; e := numseries; end;
   for j := b to e do
    begin
     svals[j].active := false;
     svals[j].pmax := 0;
     svals[j].changed := true;
     closeseries(j);
    end;
   if fplotting then paint;
  end;
end;

function TxyGraph.findx;
var i:integer;
begin
 (* the list is always sorted. If list is empty or the time is greater than
  the last on the list, index := last + 1 *)
 if svals[sN].pmax = 0 then
   begin
    result := false;
    p := 1;
   end
  else
   begin
    (* search backwards, most insertions will be at the end! *)
    if series[sN]^[svals[sN].pmax].xv < x then
      begin
       result := false;
       p := svals[sN].pmax + 1;
      end
     else
      begin
       i := svals[sN].pmax;
       while (series[sN]^[i].xv > x) and (i > 1) do dec(i);
       result := (series[sN]^[i].xv = x);
       p := i;
      end;
   end;
end;

function TxyGraph.add;
var pval,i:word;
    isnew:boolean;
begin
 if (sN in [1..numseries]) and (svals[sN].active = true)
       and (svals[sN].pmax < maxpoints) then
  begin
   result := true;
   isnew := not findx(sn,x,pval);
   if isnew then
    begin
     for i := svals[sN].pmax downto pval do series[sN]^[i + 1] := series[sN]^[i];
     inc(svals[sN].pmax);
    end;
   series[sN]^[pval].yv := y;
   series[sN]^[pval].xv := x;
   if svals[sN].pmax = 1 then svals[sN].changed := true;
   if svals[sN].AutoZero and (y < svals[sN].zeroOffset)
    then
     begin
      svals[sN].zeroOffset := y;
      svals[sN].changed := true;
     end;
   if y - svals[sN].zeroOffset < svals[sN].yvmin
           then svals[sN].yvmin := y - svals[sN].zeroOffset;
   if y - svals[sN].zeroOffset > svals[sN].yvmax
           then svals[sN].yvmax := y - svals[sN].zeroOffset;
   if x < svals[sN].xvmin then svals[sN].xvmin := x;
   if x > svals[sN].xvmax then svals[sN].xvmax := x;
   if FPlotting and svals[sn].active then paint;
  end else result := false;
end;

function TxyGraph.get;
var t:word;
begin
 result := findx(sn,x,t);
 if result then y := series[sN]^[t].yv
end;

function TxyGraph.delete;
var i,j,k,b,e:word;
    found:boolean;
begin
 if sN in [0..numseries] then
  begin
   result := false;
   b := sN; e := sN;
   if sN = 0 then begin b := 1; e := numseries; end;
   for k := b to e do
    begin
     found := findx(k, x, i);
     result := result or found; (* any deletions returns true *);
     if found then
      begin
       for j := i to svals[k].pmax - 1 do
         series[k]^[j] := series[k]^[j + 1];
       dec(svals[k].pmax);
       svals[k].changed := true;
      end;
    end;
   if FPlotting and result then paint;
  end else result := false;
end;

end.

