{*******************************************************************************
*                                                                              *
*  DCListView                                                                  *
*                                                                              *
*  TDCListView                                                                 *
*  A Descendant of Delphi's TListView that accepts files dropped by windows    *
*  has OnFileDrop event for responding to a file drop.                         *
*                                                                              *
*  TFileDropEvent                                                              *
*  Passes a String List containing fully qualified paths of files dropped and  *
*  the screen point at which they were dropped.                                *
*                                                                              *
*  Copyright  1996, Richard L. Chase                                          *
*  Made available for free use and modification by all                         *
*  Comments, Suggestions or criticism to: dchase@tiac.net or CIS - 72420,2703  *
*                                                                              *
*******************************************************************************}

unit dclstview;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ShellApi;

type

  TFileDropEvent = procedure(Files: Tstrings; X, Y: Integer) of object;

  TDCListView = class(TListView)
  private
    FAcceptFiles: Boolean;
    FOnFileDrop: TFileDropEvent;
    procedure WMDROPFILES(var Message: TWMDropFiles); message WM_DROPFILES;
    procedure setAcceptFiles(Accept: Boolean);
  protected
    { Protected declarations }
    procedure Loaded; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property AcceptFiles: Boolean read FAcceptFiles write setAcceptFiles;
    property OnFileDrop: TFileDropEvent read FOnFileDrop write FOnFileDrop;
  end;

procedure Register;

implementation

procedure TDCListView.setAcceptFiles(Accept: Boolean);
begin
  if not (csDesigning in ComponentState) then
    DragAcceptFiles(Handle, Accept);
  FAcceptFiles := Accept;
end;

constructor TDCListView.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAcceptFiles := False;
end;

procedure TDCListView.Loaded;
begin
  inherited Loaded;
  if not (csDesigning in ComponentState) then
    DragAcceptFiles(Handle, FAcceptFiles);
end;

procedure TDCListView.WMDROPFILES(var Message: TWMDropFiles);
var
  i, DropCount, BufSize: integer;
  FileName: pChar;
  FileList: TStrings;
  Point: TPoint;
begin
  BufSize := 0;
  DropCount := DragQueryFile(Message.Drop, $FFFFFFFF, nil, BufSize);
  FileList := TStringList.Create;
  try
  for i := 0 to DropCount - 1 do begin
    BufSize := DragQueryFile(Message.Drop, i, nil, BufSize) + 1;
    FileName := StrAlloc(BufSize + 1);
    try
      DragQueryFile(Message.Drop, i, FileName, BufSize);
      FileList.Add(FileName);
      DragQueryPoint(Message.Drop, Point);
    finally
      StrDispose(FileName);
    end;
  end;
  DragFinish(Message.Drop);
  if Assigned(FOnFileDrop) then
    FOnFileDrop(FileList, Point.X, Point.Y)
  finally
    FileList.Free;
  end;
end;

procedure Register;
begin
  RegisterComponents('Dick Chase', [TDCListView]);
end;

end.
