unit LOForm;

interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  StdCtrls, ExtCtrls, Spin, SysUtils, LOPanel, Askyn, Dialogs {DDDDebug, ShowMetr} ;

type

  { Array of all possible relationships }
  TRelationshipNames = array [TRelationship] of string;

  { Array of all possible edges }
  TEdgeNames = array [TEdge] of string;

  { Match string and edge code }
  TEdgePair = record
    Name : string[10];
    Edge : TEdge;
  end;

  { Array of edges possible for constraint }
  TEdgeForConstraintArray = array [0..2] of TEdgePair;

  { Array of arrays of edges possible for constraint }
  TEdgeForConstraintItems = array [TWhichConstraint] of TEdgeForConstraintArray;

const

  { Names of all possible relationships }
  RelationshipNames : TRelationshipNames =
    ('lmAsIs', 'lmPercentOf', 'lmAbove', 'lmBelow', 'lmSameAs', 'lmAbsolute');

  { Names of all possible edges }
  EdgeNames : TEdgeNames =
    ( 'lmLeft', 'lmTop', 'lmRight', 'lmBottom', 'lmCenter', 'lmWidth', 'lmHeight' );

  { Possible edge items for any of constraints }
  EdgeForConstraintItems : TEdgeForConstraintItems =
    ( ( ( Name : 'lmLeft'; Edge : lmLeft ),
        ( Name : 'lmRight'; Edge : lmRight ),
        ( Name : 'lmCenter'; Edge : lmCenter ) ),
      ( ( Name : 'lmTop'; Edge : lmTop ),
        ( Name : 'lmBottom'; Edge : lmBottom ),
        ( Name : 'lmCenter'; Edge : lmCenter ) ),
      ( ( Name : 'lmRight'; Edge : lmRight ),
        ( Name : 'lmCenter'; Edge : lmCenter ),
        ( Name : 'lmWidth'; Edge : lmWidth ) ),
      ( ( Name : 'lmBottom'; Edge : lmBottom ),
        ( Name : 'lmCenter'; Edge : lmCenter ),
        ( Name : 'lmHeight'; Edge : lmHeight ) ) );

type

  { types to store a dynamic array of constraints }
  TLotOfMetrics = array[0..64535 div sizeof(TLayoutMetrics)] of TLayoutMetrics;
  PLotOfMetrics = ^TLotOfMetrics;

  TStringConstraints = array[TWhichConstraint] of string;

  TLayoutForm = class(TForm)
    OKBtn: TBitBtn;
    CancelBtn: TBitBtn;
    HelpBtn: TBitBtn;
    MainBevel: TBevel;
    ControlListBox: TListBox;
    ControlLabel: TLabel;
    EdgeListBox: TListBox;
    EdgeLabel: TLabel;
    RelationListBox: TListBox;
    RelControlLabel: TLabel;
    RelControlListBox: TListBox;
    RelEdgeListBox: TListBox;
    RelEdgeLabel: TLabel;
    RelationLabel: TLabel;
    ValueLabel: TLabel;
    ValueEdit: TSpinEdit;
    ConstraintBtn: TButton;
    ConstraintsPanel: TPanel;
    ConstraintsLabel: TLabel;
    ConstraintListBox: TListBox;
    XRestoreBtn: TBitBtn;
    YRestoreBtn: TBitBtn;
    WidthRestoreBtn: TBitBtn;
    HeightRestoreBtn: TBitBtn;
    ExtremalsPanel: TPanel;
    MinWLabel: TLabel;
    MinWEdit: TSpinEdit;
    MaxWEdit: TSpinEdit;
    MaxWLabel: TLabel;
    MinHEdit: TSpinEdit;
    MinHLabel: TLabel;
    MaxHEdit: TSpinEdit;
    MaxHLabel: TLabel;
    ConfirmCheckBox: TCheckBox;
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure ControlListBoxClick(Sender: TObject);
    procedure ConstraintListBoxClick(Sender: TObject);
    procedure RelationListBoxClick(Sender: TObject);
    procedure ConstraintBtnClick(Sender: TObject);
    procedure RestoreBtnClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure CloseBtnClick(Sender: TObject);
    procedure HelpBtnClick(Sender: TObject);
    procedure ExtremalEditChange(Sender: TObject);

  private
    { Private declarations }
    FModified : boolean;
    FMetrics : PLotOfMetrics;
    FUndoMetrics : TLayoutMetrics;
    FMetricsCount : integer;
    FLayoutPanel : TLayoutPanel;
    FStringConstraints : TStringConstraints;
    FNumMake : integer;
    FCloseBtn : TObject;

    function GetMetrics( Ind : integer ) : TLayoutMetrics;
    procedure SetMetrics( Ind : integer; Metr : TLayoutMetrics );
    procedure ReadMetricsFromPanel;
    procedure FillConstraintListBox;
    procedure FillExtremalsEdits;
    function MakeStringConstraint( var constr : TLayoutConstraint ) : string;
    function GetConstraintPtr( NConstraint : integer ) : PLayoutConstraint;
    function GetRelControlNum( var Control : TControl ) : integer;

  protected
    property Metrics[ Ind : integer ] : TLayoutMetrics read GetMetrics write SetMetrics;

  public
    property Modified : boolean read FModified write FModified;
    { Public declarations }
    constructor Create( AOwner : TComponent; Panel : TLayoutPanel );


  end;

var Confirm: boolean;

implementation

{$R *.DFM}

{--------------------------------------- TLayoutForm.Create }
{ Initialize fields }
constructor TLayoutForm.Create( AOwner : TComponent; Panel : TLayoutPanel );
begin
  inherited Create( AOwner );

  { If there are no controls raise an exception }
  if Panel.ControlCount < 1 then
    raise Exception.Create( 'There are no controls to layout !' );

  FMetrics := nil;
  FMetricsCount := 0;
  FLayoutPanel := Panel;
  FNumMake := 0;
  FCloseBtn := nil;
  FModified := False;
  ReadMetricsFromPanel;
  ControlListBox.ItemIndex := 0;
  ControlListBoxClick(ControlListBox);
  ConfirmCheckBox.Checked := Confirm;

{DDDDebug
  ShowMetricsForm.ShowMetrics(FLayoutPanel, true);}
end; { TLayoutForm.Create }

{--------------------------------------- TLayoutForm.FormDestroy }
{ Clean up }
procedure TLayoutForm.FormDestroy(Sender: TObject);
begin
  { Free previously allocated memory }
  if FMetrics <> nil then
    FreeMem( FMetrics, FMetricsCount * SizeOf( TLayoutMetrics ) );
end; { TLayoutForm.FormDestroy }

{--------------------------------------- TLayoutForm.ReadMetricsFromPanel }
{ Create list for metrics }
procedure TLayoutForm.ReadMetricsFromPanel;
var
  i : integer;
  Metr: TLayoutMetrics;
begin
  { Free previously allocated memory }
  if FMetrics <> nil then
    FreeMem( FMetrics, FMetricsCount * SizeOf( TLayoutMetrics ) );

  { Allocate memory }
  FMetricsCount := FLayoutPanel.ControlCount;
  GetMem( FMetrics, FMetricsCount * SizeOf( TLayoutMetrics ) );

  { Clear control listboxes }
  ControlListBox.Clear;
  RelControlListBox.Clear;

  { Set listboxes }
  for i := 0 to FMetricsCount-1 do
  begin
    { Fill layout metrics (default if they were not assigned yet) }
    InitLayoutMetrics( Metr );
    FLayoutPanel.GetChildLayoutMetrics( FLayoutPanel.Controls[i], Metr );
    AssignLayoutMetrics( FMetrics^[i], Metr );

    { Set controls listbox values }
    ControlListBox.Items.Add( FLayoutPanel.Controls[i].Name + ': ' + FLayoutPanel.Controls[i].ClassName  );
    RelControlListBox.Items.Add( FLayoutPanel.Controls[i].Name + ': ' + FLayoutPanel.Controls[i].ClassName );
  end; {for}

  RelControlListBox.Items.Add( 'lmParent' );
end; { TLayoutForm.ReadMetricsFromPanel }

{--------------------------------------- TLayoutForm.GetMetrics }
{ Get layout metrics by index }
function TLayoutForm.GetMetrics( Ind : integer ) : TLayoutMetrics;
begin
  if ( FMetrics = nil ) or
     ( Ind < 0 ) or
     ( Ind >= FMetricsCount ) then
    InitLayoutMetrics( Result )
  else
    AssignLayoutMetrics( Result, FMetrics^[Ind] );
end; { TLayoutForm.GetMetrics }

{--------------------------------------- TLayoutForm.SetMetrics }
{ Set layout metrics by index }
procedure TLayoutForm.SetMetrics( Ind : integer; Metr : TLayoutMetrics );
begin
  if ( FMetrics <> nil ) and
     ( Ind >= 0 ) and
     ( Ind < FMetricsCount ) then
    AssignLayoutMetrics( FMetrics^[Ind], Metr );
end; { TLayoutForm.SetMetrics }

{--------------------------------------- TLayoutForm.FormShow }
{ Set listbox sizes }
procedure TLayoutForm.FormShow(Sender: TObject);
begin
  RelationListBox.Height := RelationListBox.ItemHeight * (Ord(High(TRelationship)) + 1) + 4;
  ConstraintListBox.Height := RelationListBox.ItemHeight * (Ord(High(TWhichConstraint)) + 1) + 4;
  EdgeListBox.Height := EdgeListBox.ItemHeight * (Ord(High(TEdge)) + 1) + 4;
  RelEdgeListBox.Height := RelEdgeListBox.ItemHeight * (Ord(High(TEdge)) + 1) + 4;
end; { TLayoutForm.FormShow }

{--------------------------------------- TLayoutForm.FillConstraintListBox }
{ Fill constraint list box using current Control settings }
procedure TLayoutForm.FillConstraintListBox;
var
  Metr : TLayoutMetrics;
  N : integer;
begin
  N := ControlListBox.ItemIndex;
  AssignLayoutMetrics( Metr, Metrics[ N ] );
  ConstraintListBox.Clear;
  ConstraintListBox.Items.Add( MakeStringConstraint( Metr.X ) );
  ConstraintListBox.Items.Add( MakeStringConstraint( Metr.Y ) );
  ConstraintListBox.Items.Add( MakeStringConstraint( Metr.Width ) );
  ConstraintListBox.Items.Add( MakeStringConstraint( Metr.Height ) );
  ConstraintListBox.ItemIndex := 0;
end; { TLayoutForm.FillConstraintListBox }

{--------------------------------------- TLayoutForm.FillExtremalsEdits }
{ Fill constraint list box using current Control settings }
procedure TLayoutForm.FillExtremalsEdits;
var
  Metr : TLayoutMetrics;
  N : integer;
begin
  N := ControlListBox.ItemIndex;
  AssignLayoutMetrics( Metr, Metrics[ N ] );
  MinWEdit.Value := Metr.MinW;
  MaxWEdit.Value := Metr.MaxW;
  MinHEdit.Value := Metr.MinH;
  MaxHEdit.Value := Metr.MaxH;
end; { TLayoutForm.FillExtremalsEdits }

{--------------------------------------- TLayoutForm.GetRelControlNum }
{ Get number of control }
function TLayoutForm.GetRelControlNum( var Control : TControl ) : integer;
var
  i: integer;
begin
  if Control = lmParent then {lmParent - last item in list box}
    Result := RelControlListBox.Items.Count-1
  else
    Result := FLayoutPanel.IsChildControl( Control );
end; { TLayoutForm.GetRelControlNum }

{--------------------------------------- TLayoutForm.MakeStringConstraint }
{ Make string by constraint }
function TLayoutForm.MakeStringConstraint( var constr : TLayoutConstraint ) : string;
var
  N, i : integer;

begin

  { Set control name - get current conrol from list box and delete
    everything after colon }
  Result := ControlListBox.Items[ControlListBox.ItemIndex];
  N := Pos( ':', Result );

  { We cannot use SetLength or Result[0] because it seems
    impossible to check current Delphi compilator version }
  if N > 0 then
    Delete( Result, N, Length( Result) );

  {Add my edge and relationship}
  Result := Result + ', ' +
            EdgeNames[constr.MyEdge] + ', ' +
            RelationshipNames[constr.Relationship];

  { if relationship is AsIs everything is done }
  if constr.Relationship = lmAsIs then
    Exit;

  { Get number of relative control in listbox only if
    relationship is not Absolute }
  if constr.Relationship <> lmAbsolute then
  begin
    Result := Result + ', ';

    N := GetRelControlNum( constr.RelWin );

    if N >= 0 then
    begin
      Result := Result + RelControlListBox.Items[N];
      N := Pos( ':', Result );

      if N > 0 then
        Delete( Result, N, Length( Result) );
    end; {if N >= 0}

    {Add relative edge}
    Result := Result + ', ' + EdgeNames[constr.OtherEdge];
  end {if constr.Relationship <> lmAbsolute}
  else
    Result := Result + ',,';

  { Add value only if relationship is not SameAs }
  if constr.Relationship <> lmSameAs then
    Result := Result + ', ' + IntToStr( constr.Value );
end; { TLayoutForm.MakeStringConstraint }

{--------------------------------------- TLayoutForm.ControlListBoxClick }
{ Current control changed  }
procedure TLayoutForm.ControlListBoxClick(Sender: TObject);
begin
  XRestoreBtn.Enabled := False;
  YRestoreBtn.Enabled := False;
  WidthRestoreBtn.Enabled := False;
  HeightRestoreBtn.Enabled := False;
  FillConstraintListBox;
  FillExtremalsEdits;
  ConstraintListBoxClick(ConstraintListBox);
end; { TLayoutForm.ControlListBoxClick }

{--------------------------------------- TLayoutForm.GetConstraintPtr }
{ Get current constraint pointer using list boxes indexes  }
function TLayoutForm.GetConstraintPtr( NConstraint : integer ) : PLayoutConstraint;
var
  NControl : integer;
begin
  NControl := ControlListBox.ItemIndex;

  { Get pointer to current constraint }
  Result := PLayoutConstraint( PChar( FMetrics )+NControl*SizeOf( TLayoutMetrics )+
                               NConstraint*SizeOf( TLayoutConstraint ) );
end; { TLayoutForm.GetConstraintPtr }

{--------------------------------------- TLayoutForm.ConstraintListBoxClick }
{ Current constraint changed  }
procedure TLayoutForm.ConstraintListBoxClick(Sender: TObject);
var
  NConstraint, i : integer;
  ConstrPtr : PLayoutConstraint;
begin
  NConstraint := ConstraintListBox.ItemIndex;

  { Get pointer to current constraint }
  ConstrPtr := GetConstraintPtr( NConstraint );

  EdgeListBox.Clear;

  for i := 0 to 2 do
  begin
    EdgeListBox.Items.Add( EdgeForConstraintItems[TWhichConstraint(NConstraint)][i].Name );
    if EdgeForConstraintItems[TWhichConstraint(NConstraint)][i].Edge = ConstrPtr^.MyEdge then
      EdgeListBox.ItemIndex := i;
  end; {for}

  RelationListBox.ItemIndex := integer(ConstrPtr^.Relationship);
  RelEdgeListBox.ItemIndex := integer(ConstrPtr^.OtherEdge);

  RelControlListBox.ItemIndex := GetRelControlNum( ConstrPtr^.RelWin );

  { Set Value }
  ValueEdit.Value := ConstrPtr^.Value;
  RelationListBoxClick( RelationListBox );
end; { TLayoutForm.ConstraintListBoxClick }

{--------------------------------------- TLayoutForm.RelationListBoxClick }
{ Current relation changed  }
procedure TLayoutForm.RelationListBoxClick(Sender: TObject);
begin
  ValueEdit.Enabled := true;
  ValueLabel.Enabled := true;

  case TRelationship(RelationListBox.ItemIndex) of
    lmAsIs, lmSameAs :           {Value doesn't matter}
    begin
      ValueEdit.Enabled := false;
      ValueLabel.Enabled := false;
    end; {lmAsIs, lmSameAs :}

    lmPercentOf :
    begin
      ValueEdit.MaxValue := High(longint);
      ValueEdit.Increment := 10;
    end; {lmPercentOf :}

    else
    begin
      ValueEdit.MaxValue := 0;
      ValueEdit.Increment := 1;
    end; {else}
  end; { case }
end; { TLayoutForm.RelationListBoxClick }

{--------------------------------------- TLayoutForm.ConstraintBtnClick }
{ Make constraint }
procedure TLayoutForm.ConstraintBtnClick(Sender: TObject);
var
  NConstraint : integer;
  ConstrPtr : PLayoutConstraint;
begin
  NConstraint := ConstraintListBox.ItemIndex;

  { Get pointer to current constraint }
  ConstrPtr := GetConstraintPtr( NConstraint );

  case TWhichConstraint( NConstraint ) of
    XConstraint :
    begin
       AssignConstraint( FUndoMetrics.X, ConstrPtr^ );
       XRestoreBtn.Enabled := true;
    end; {XConstraint}
    YConstraint :
    begin
       AssignConstraint( FUndoMetrics.Y, ConstrPtr^ );
       YRestoreBtn.Enabled := true;
    end; {YConstraint}
    WidthConstraint :
    begin
       AssignConstraint( FUndoMetrics.Width, ConstrPtr^ );
       WidthRestoreBtn.Enabled := true;
    end; {XConstraint}
    HeightConstraint :
    begin
       AssignConstraint( FUndoMetrics.Height, ConstrPtr^ );
       HeightRestoreBtn.Enabled := true;
    end; {HeightConstraint}
  end; {case}

  { Set constraint values }
  SetConstraint( ConstrPtr^,
                 EdgeForConstraintItems[TWhichConstraint(NConstraint)]
                                       [EdgeListBox.ItemIndex].Edge,
                 TRelationship( RelationListBox.ItemIndex ),
                 lmParent,
                 TEdge( RelEdgeListBox.ItemIndex ),
                 ValueEdit.Value );

  { Cut superfluous constraints }
  case TRelationship( RelationListBox.ItemIndex ) of
    lmAsIs :
    begin
      ConstrPtr^.OtherEdge := lmLeft;
      ConstrPtr^.Value := 0;
    end; {lmAsIs}
    lmSameAs :
      ConstrPtr^.Value := 0;
    lmAbsolute :
      ConstrPtr^.OtherEdge := lmLeft;
  end; {case}

  { Set other control if required }
  if ( TRelationship( RelationListBox.ItemIndex ) <> lmAsIs ) and
     ( TRelationship( RelationListBox.ItemIndex ) <> lmAbsolute ) and
     ( RelControlListBox.ItemIndex < FLayoutPanel.ControlCount ) then

    ConstrPtr^.RelWin := FLayoutPanel.Controls[ RelControlListBox.ItemIndex ];

  FillConstraintListBox;
  ConstraintListBox.ItemIndex := NConstraint;
  Inc(FNumMake);
end; { TLayoutForm.ConstraintBtnClick }

{--------------------------------------- TLayoutForm.RestoreBtnClick }
{ One of four restore buttons clicked  }
procedure TLayoutForm.RestoreBtnClick(Sender: TObject);
var
  NConstraint : integer;
  Btn : TButton;
  ConstrPtr : PLayoutConstraint;
begin
  if not ( Sender is TButton ) then
    Exit;

  Btn := Sender as TButton;

  if Btn = XRestoreBtn then
  begin
    NConstraint := ord(XConstraint);
    ConstrPtr := GetConstraintPtr( NConstraint );
    AssignConstraint( ConstrPtr^, FUndoMetrics.X );
  end {if Btn = XRestoreBtn}
  else
  if Btn = YRestoreBtn then
  begin
    NConstraint := ord(YConstraint);
    ConstrPtr := GetConstraintPtr( NConstraint );
    AssignConstraint( ConstrPtr^, FUndoMetrics.Y );
  end {if Btn = YRestoreBtn}
  else
  if Btn = WidthRestoreBtn then
  begin
    NConstraint := ord(WidthConstraint);
    ConstrPtr := GetConstraintPtr( NConstraint );
    AssignConstraint( ConstrPtr^, FUndoMetrics.Width );
  end {if Btn = WidthRestoreBtn}
  else
  begin
    NConstraint := ord(HeightConstraint);
    ConstrPtr := GetConstraintPtr( NConstraint );
    AssignConstraint( ConstrPtr^, FUndoMetrics.Height );
  end; {if Btn = HeightRestoreBtn}

  Btn.Enabled := false;
  NConstraint := ConstraintListBox.ItemIndex;
  FillConstraintListBox;
  ConstraintListBox.ItemIndex := NConstraint;
  Dec(FNumMake);
end; { TLayoutForm.RestoreBtnClick }

{--------------------------------------- TLayoutForm.FormClose }
{ Make all necessary cleanup
  Check which button was press (OK or Cancel) and
  set new constraints if required }
procedure TLayoutForm.FormClose(Sender: TObject; var Action: TCloseAction);
var
  i : integer;
  Change : boolean;
begin
  Confirm := ConfirmCheckBox.Checked;
  Action := caFree;

  { if noting was changed then there is noting to do and to ask }
  if ( not Modified ) and ( FNumMake = 0 ) then
    Exit;

  { Exit if Cancel button is pressed and confirm isn't required }
  if (not Confirm) and (FCloseBtn = CancelBtn) then
    Exit;

  Change := True; { Assume change is required }

  if Confirm then
  begin
    AskynForm := TAskynForm.Create(Application);

    { Consider Cancel and OK buttons separately }
    if FCloseBtn = CancelBtn then
      AskynForm.AskLabel.Caption := 'Discard changes ?'
    else
      AskynForm.AskLabel.Caption := 'Set new metrics ?';

    if AskynForm.ShowModal = idYes then
      Change := FCloseBtn = OKBtn
    else
    begin
      Change := False;
      Action := caNone;
    end; {else}

    AskynForm.Free;
  end; {if Confirm}

  if Change then
  begin
    { Set metrics for layout panel }
{DDDDebug
    ShowMetricsForm.ShowMetrics(FLayoutPanel, true);}
    DontCareAboutRelMetrics := True; {provide the right order of metrics}

    for i := 0 to FLayoutPanel.ControlCount-1 do
      FLayoutPanel.SetChildLayoutMetrics( FLayoutPanel.Controls[i], FMetrics^[i] );

    DontCareAboutRelMetrics := False;
    FLayoutPanel.Layout;
    Modified := True;

{DDDDebug
    ShowMetricsForm.ShowMetrics(FLayoutPanel, true);}
  end { if Change  }

end; { TLayoutForm.FormClose }

{--------------------------------------- TLayoutForm.CloseBtnClick }
{ Set pointer to pressed button }
procedure TLayoutForm.CloseBtnClick(Sender: TObject);
begin
  FCloseBtn := Sender;
end; { TLayoutForm.CloseBtnClick }

{--------------------------------------- TLayoutForm.ExtremalEditChange }
{ Extremal width or height changed }
procedure TLayoutForm.ExtremalEditChange(Sender: TObject);
var
  N : integer;
  Edit : TSpinEdit;
  V : ^Integer;
begin
  N := ControlListBox.ItemIndex;
  Edit := Sender as TSpinEdit;

  if Edit = MinWEdit then
    V := @FMetrics^[ N ].MinW
  else
  if Edit = MaxWEdit then
    V := @FMetrics^[ N ].MaxW
  else
  if Edit = MinHEdit then
    V := @FMetrics^[ N ].MinH
  else
  if Edit = MaxHEdit then
    V := @FMetrics^[ N ].MaxH
  else
    Exit;

  { Not less then 0 (MinValue) }
  if Edit.Value < 0 then
  begin
    Edit.Value := 0;
    Exit;
  end; {if}

  if V^ <> Edit.Value then
  begin
    V^ := Edit.Value;
    Modified := True;
  end; {if}

end; { TLayoutForm.ExtremalEditChange }

{--------------------------------------- TLayoutForm.HelpBtnClick }
{ Help is not implemented }
procedure TLayoutForm.HelpBtnClick(Sender: TObject);
begin
  MessageDlg( 'This is TLayoutPanel component Beta v. 1.31'#10#13 +
              'Help is not implemented yet'#10#13#10#13 +
              'See enclosed README.TXT file or contact me'#10#13 +
              'E-mail: anna@xperts1.rtc.neva.ru',
              mtInformation, [mbOk], 0 );
end; { TLayoutForm.HelpBtnClick }

initialization

Confirm := True;

end.
