
unit Capi;

interface

type
  Short = Word;
  DWord = Longint;

const
  { Info masks }
  imChargingInformation = $00000001;
  imDate                = $00000002;
  imDisplay             = $00000004;
  imUserUserInformation = $00000008;
  imCause               = $00000010;
  imStatusOfCalledParty = $00000020;

  imInfoMask_All        = $0000003F;

  { Serviced EAZ mask }
  semGlobalCall         = $0001;

  semServicedEaz_All    = $03FF;       { 10 lower bits }

  { Serviced SI mask }
  simVideophone         = $0001;       { bit  0  Videophone }
  simTelephony          = $0002;       { bit  1  Telephony }
  simAbServices         = $0004;       { bit  2  a/b services }
  simX21Services        = $0008;       { bit  3  X.21 services }
  simFaxG4              = $0010;       { bit  4  Fax (Group 4) }
  simVideotex64kbps     = $0020;       { bit  5  Videotex (64 kbit/s) }
                                       { bit  6 }
  simData64kbps         = $0080;       { bit  7  Data transmission (64 kbit/s) }
  simX25Services        = $0100;       { bit  8  X.25 services }
  simTeletex64          = $0200;       { bit  9  Teletex 64 }
  simMixedMode          = $0400;       { bit 10  Mixed mode }
                                       { bit 11 }
                                       { bit 12 }
  simRemoteControl      = $2000;       { bit 13  Remote control }
  simGraphicTelService  = $4000;       { bit 14  Graphic telephone service }
  simVideotex           = $8000;       { bit 15  Videotex (new standard) }

  simServicedSi_All     = $E7BF;

  { Commands }
  cmd_LISTEN            = $05;
  cmd_MANUFACTURER      = $FF;

  { Subcommands }
  subcmd_REQ            = $00;
  subcmd_CONF           = $01;
  subcmd_IND            = $02;
  subcmd_RESP           = $03;

type
  PCapiMessage = ^TCapiMessage;
  TCapiMessage =
    record
      Length : Word;
      APPL_ID : Word;
      Command,
      SubCommand : Byte;
      MessageNumber : Word;
      Data : record end;
    end;

  TCapi_Listen_Req =
    record
      Length : Word;
      APPL_ID : Word;
      Command,
      SubCommand : Byte;
      MessageNumber : Word;
      Controller : Byte;
      InfoMask : DWord;
      ServicedEazMask : Word;
      ServicedSiMask : Word;
    end;

  TCapi_Listen_Conf =
    record
      Length : Word;
      APPL_ID : Word;
      Command,
      SubCommand : Byte;
      MessageNumber : Word;
      Controller : Byte;
      Info : Word;
    end;

  TCapiSignalProc = procedure;

{ Common ISDN API (CAPI) functions }

function API_REGISTER (Buffer : Pointer;
                       MaxMessages,
                       MaxConnections,
                       MaxBChanPackets,
                       MaxBChanDataSize : Short) : Integer;
function API_RELEASE (Appl_Id : Integer) : Integer;
function API_GET_MESSAGE (Appl_Id : Integer; var Msg : PCapiMessage) : Integer;
function API_PUT_MESSAGE (Appl_Id  : Integer; Msg : PCapiMessage) : Integer;
function API_SET_SIGNAL (Appl_Id  : Integer; B : TCapiSignalProc) : Integer;
function API_GET_MANUFACTURER (Identification : PChar) : Integer;
function API_GET_VERSION (VersionString : PChar) : Integer;
function API_MANUFACTURER (Appl_Id  : Integer) : Integer;
function API_INSTALLED : Integer;
function API_GET_SERIAL_NUMBER (A : PChar) : Integer;
function API_GET_ADDRESSMODE : Integer;

{ CAPI utility functions }

function CapiBufferSizeNeeded (MaxMessages,
                               MaxConnections,
                               MaxBChanPackets,
                               MaxBChanDataSize : Longint) : Longint;

function CapiMk_Listen_Req (Appl_Id : Integer;
                            Controller : Byte;
                            InfoMask : DWord;
                            ServicedEazMask,
                            ServicedSiMask : Word) : TCapi_Listen_Req;

procedure CapiMk_Manufacturer_Req (AAppl_Id : Integer;
                                   ASubCommand : Byte;
                                   const Elements : array of Byte;
                                   var Message);

function CapiMessageToString (var Msg) : string;

implementation

uses
  SysUtils;

{ Externals }

const
  DllName = 'capi.dll';

function API_REGISTER; external DllName;
function API_RELEASE; external DllName;
function API_GET_MESSAGE; external DllName;
function API_PUT_MESSAGE; external DllName;
function API_SET_SIGNAL; external DllName;
function API_GET_MANUFACTURER; external DllName;
function API_GET_VERSION; external DllName;
function API_MANUFACTURER; external DllName;
function API_INSTALLED; external DllName;
function API_GET_SERIAL_NUMBER; external DllName;
function API_GET_ADDRESSMODE; external DllName;

{ Publics }

function CapiBufferSizeNeeded (MaxMessages,
                               MaxConnections,
                               MaxBChanPackets,
                               MaxBChanDataSize : Longint) : Longint;
begin
  Result:=(180 * MaxMessages) +
          (MaxConnections * MaxBChanPackets * MaxBChanDataSize);
end;

function CapiMk_Listen_Req (Appl_Id : Integer;
                            Controller : Byte;
                            InfoMask : DWord;
                            ServicedEazMask,
                            ServicedSiMask : Word) : TCapi_Listen_Req;
begin
  FillChar(Result, SizeOf(Result), 0);
  Result.Length:=SizeOf(Result);
  Result.Appl_Id:=Appl_Id;
  Result.Command:=cmd_LISTEN;
  Result.Controller:=Controller;
  Result.InfoMask:=InfoMask;
  Result.ServicedEazMask:=ServicedEazMask;
  Result.ServicedSiMask:=ServicedSiMask;
end;

procedure CapiMk_Manufacturer_Req (AAppl_Id : Integer;
                                   ASubCommand : Byte;
                                   const Elements : array of Byte;
                                   var Message);
var
  I : Integer;
  P : ^Byte;
begin
  with TCapiMessage(Message) do
    begin
      Length:=SizeOf(TCapiMessage) + High(Elements) - Low(Elements) + 1;
      Appl_Id:=AAppl_Id;
      Command:=cmd_MANUFACTURER;
      SubCommand:=ASubCommand;
      P:=@Data;
    end;
  for I:=Low(Elements) to High(Elements) do
    begin
      P^:=Elements[I];
      Inc(P);
    end;
end;

function CapiMessageToString (var Msg) : string;
var
  Len : Word;
  P : ^Byte;
begin
  with TCapiMessage(Msg) do
    begin
      Len:=0;
      case Command of
        cmd_LISTEN :
          case SubCommand of
            subcmd_REQ :
              with TCapi_Listen_Req(Msg) do
                Result:=Format(
                  'LISTEN_REQ (Ctl=%d InfoMask=%08x EazMask=%04x SiMask=%04x)',
                  [Controller, InfoMask, ServicedEazMask, ServicedSiMask]);
            subcmd_CONF :
              with TCapi_Listen_Conf(Msg) do
                Result:=Format(
                  'LISTEN_CONF (Ctl=%d Info=%04x)',
                  [Controller, Info]);
          end;
        cmd_MANUFACTURER :
          begin
            Len:=Length - SizeOf(TCapiMessage);
            case SubCommand of
              subcmd_REQ :
                Result:='MANUFACTURER_REQ';
              subcmd_CONF :
                Result:='MANUFACTURER_CONF';
              subcmd_IND :
                Result:='MANUFACTURER_IND';
              subcmd_RESP :
                Result:='MANUFACTURER_RESP';
            end;
          end;
      else
        begin
          Result:='Unknown message';
          Len:=Length - SizeOf(TCapiMessage);
        end;
      end;

      if Len > 0 then
        begin
          AppendStr(Result, ' (' + IntToStr(Len) + ' bytes');
          if Len > 0 then
            begin
              AppendStr(Result, ', Data=');
              P:=@Data;
              while Len > 0 do
                begin
                  AppendStr(Result, Format('%.2x ', [P^]));
                  Inc(P);
                  Dec(Len);
                end;
            end;
          AppendStr(Result, ')');
        end;
      Result:=Format('Len=%d APPL_ID=%.2x Cmd=%.2x/%.2x M#=%.4x ',
        [Length, Appl_Id, Command, SubCommand, MessageNumber]) + Result;
    end;
end;

end.
