/* MainWnd.c -- window procedure for the main window */

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <mem.h>
#include "statbar.h"
#include "taquin.h"
#include "resource.h"
#include "commands.h"

BOOL Taquin_OnCreate( HWND hwnd, LPCREATESTRUCT lpCreateStruct );
void Taquin_OnSize( HWND hwnd, UINT state, int cx, int cy );
void Taquin_OnPaint( HWND hwnd );
void Taquin_OnCommand( HWND hwnd, int id, HWND hwndCtl, UINT codeNotify );
void Taquin_OnLButtonDown( HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags );
void Taquin_OnKey( HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags );
void Taquin_OnClose( HWND hwnd );
void Taquin_OnDestroy( HWND hwnd );
HWND InitStatusBar( HWND hwndParent );

LOGFONT g_lfTileFont;                  /* logical font for drawing tile numbers */
char    g_szHelpFileName[_MAX_PATH];   /* name of our help file */
int     g_iLastStatusMessage;          /* ID of last status message string */
BOOL    g_bSettingsChanged;            /* TRUE if any setting has changed */

/* Menu command handler IDs and handler function pointers: */
COMMAND acmdMenuHandlers[] =
{
	{ IDM_GAME_NEW,  Taquin_OnCmd_GameNew },
	{ IDM_GAME_EXIT, Taquin_OnCmd_GameExit },
	{ IDM_OPTIONS_SOUND, Taquin_OnCmd_OptionsSound },
	{ IDM_OPTIONS_FONT, Taquin_OnCmd_OptionsFont },
	{ IDM_HELP_CONTENTS, Taquin_OnCmd_HelpContents },
	{ IDM_HELP_SEARCH, Taquin_OnCmd_HelpSearch },
	{ IDM_HELP_HELPONHELP, Taquin_OnCmd_HelpHelpOnHelp },
	{ IDM_HELP_ABOUT, Taquin_OnCmd_HelpAbout }
};

LRESULT CALLBACK _export Taquin_WndProc ( HWND hwnd, UINT uiMsg, WPARAM wParam, LPARAM lParam )
{
    LRESULT lResult = 0L;

    switch ( uiMsg )
    {
        case WM_CREATE:
            lResult = HANDLE_WM_CREATE( hwnd, wParam, lParam, Taquin_OnCreate );
            break;

        case WM_SIZE:
            HANDLE_WM_SIZE( hwnd, wParam, lParam, Taquin_OnSize );
            break;

        case WM_PAINT:
            lResult = HANDLE_WM_PAINT( hwnd, wParam, lParam, Taquin_OnPaint );
            break;

        case WM_COMMAND:
            lResult = HANDLE_WM_COMMAND( hwnd, wParam, lParam, Taquin_OnCommand );
            break;

        case WM_LBUTTONDOWN:
            lResult = HANDLE_WM_LBUTTONDOWN( hwnd, wParam, lParam, Taquin_OnLButtonDown );
            break;

        case WM_KEYDOWN:
            lResult = HANDLE_WM_KEYDOWN( hwnd, wParam, lParam, Taquin_OnKey );
            break;

        case WM_CLOSE:
            HANDLE_WM_CLOSE( hwnd, wParam, lParam, Taquin_OnClose );
            break;

        case WM_DESTROY:
            lResult = HANDLE_WM_DESTROY( hwnd, wParam, lParam, Taquin_OnDestroy );
            break;

        default:
            lResult = DefWindowProc( hwnd, uiMsg, wParam, lParam );
            break;
    }

    return lResult;
}

BOOL Taquin_OnCreate ( HWND hwnd, LPCREATESTRUCT lpCreateStruct )
{
    HDC hdc;
    HMENU hmenu;
    RECT rcClient;
	char szMessage[256];

    LoadString( g_hInst, IDS_HELPFILENAME, 
		g_szHelpFileName, sizeof(g_szHelpFileName) );

    GetClientRect( hwnd, &rcClient );

    g_hwndStatusBar = InitStatusBar( hwnd );
    if ( g_hwndStatusBar == NULL )
    {
        return FALSE;
    }

    GameData_Init();
    GameData_SetGameState( GS_IDLING );

    Settings_Read();
    g_bSettingsChanged = FALSE;

    hmenu = GetMenu( hwnd );
    CheckMenuItem( hmenu, IDM_OPTIONS_SOUND,
        Settings_IsSoundEnabled() ? MF_CHECKED : MF_UNCHECKED );

    _fmemset( &g_lfTileFont, 0, sizeof(LOGFONT) );

    /* Set default font for tiles. */
    hdc = GetDC( hwnd );
    g_lfTileFont.lfHeight = -MulDiv(
        Settings_GetFontSize(),
        GetDeviceCaps( hdc, LOGPIXELSY ),
        72 );
    Settings_GetFontName( g_lfTileFont.lfFaceName );
    g_lfTileFont.lfWeight = Settings_IsFontBold()   ? FW_BOLD : FW_NORMAL;
    g_lfTileFont.lfItalic = Settings_IsFontItalic() ? 1 : 0;
    ReleaseDC( hwnd, hdc );

	lstrcpy( szMessage, GetString( g_hInst, IDS_STAT_PRESSTOPLAY ) );
    StatusBar_SetText( szMessage );
    g_iLastStatusMessage = IDS_STAT_PRESSTOPLAY;

    return TRUE;
}

void Taquin_OnSize ( HWND hwnd, UINT state, int cx, int cy )
{
    int xStatus, yStatus, cxStatus, cyStatus;
    RECT rectStatus;

    GetWindowRect( g_hwndStatusBar, &rectStatus );
    cyStatus = rectStatus.bottom - rectStatus.top;

	/* Keep the status bar aligned to the bottom of
	   the client area. */
    xStatus = 0;
    yStatus = cy - cyStatus;
    cxStatus = cx;

    MoveWindow( g_hwndStatusBar,
		xStatus, yStatus,
		cxStatus, cyStatus, TRUE );
}

void Taquin_OnPaint ( HWND hwnd )
{
    HDC hdc;
    PAINTSTRUCT ps;
    int iRow, iColumn;
    RECT rcPaint, rcDest;

    hdc = BeginPaint( hwnd, &ps );

    for ( iRow = 0;  iRow < NUMROWS;  iRow++ )
    {
        for ( iColumn = 0;  iColumn < NUMCOLUMNS;  iColumn++ )
        {
            SetRect( &rcPaint,
				TILEWIDTH * iColumn, TILEHEIGHT * iRow,
				TILEWIDTH * (iColumn + 1), TILEHEIGHT * (iRow + 1)
            );
            if ( IntersectRect( &rcDest, &rcPaint, &(ps.rcPaint) ) )
            {
                PaintTile( hdc, iRow, iColumn, &g_lfTileFont );
            }
        }
    }

    EndPaint( hwnd, &ps );
}

void Taquin_OnCommand ( HWND hwnd, int id, HWND hwndCtl, UINT codeNotify )
{
	int i;

	for ( i = 0;  i < sizeof(acmdMenuHandlers) / sizeof(acmdMenuHandlers[0]);  i++ )
	{
		if ( id == acmdMenuHandlers[i].id )
		{
			(*acmdMenuHandlers[i].pfnHandler)( hwnd, hwndCtl, codeNotify );
		}
	}
}

void Taquin_OnLButtonDown ( HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags )
{
	char szMessage[256];
    int iMouseRow, iMouseColumn;
    RECT rectInvalid;
    int iBlankRow, iBlankColumn;

    /* If the user has not started a new game, do nothing. */
    if ( GameData_GetGameState() == GS_IDLING )
    {
        return;
    }

    /* If the last attempt to move was blocked, restore
       the "Ready" message. */
    if ( g_iLastStatusMessage == IDS_STAT_BLOCKED )
    {
		lstrcpy( szMessage, GetString( g_hInst, IDS_STAT_READY ) );
        StatusBar_SetText( szMessage );
        g_iLastStatusMessage = IDS_STAT_READY;
    }

	iMouseColumn = x / TILEWIDTH;
	iMouseRow    = y / TILEHEIGHT;

    iBlankRow    = GameData_GetBlankRow();
    iBlankColumn = GameData_GetBlankColumn();

    if ( iMouseRow < 0 || iMouseColumn < 0 ||
         iMouseRow >= NUMROWS || iMouseColumn >= NUMCOLUMNS ||
        (iMouseRow != iBlankRow && iMouseColumn != iBlankColumn) ||
        (iMouseRow == iBlankRow && iMouseColumn == iBlankColumn) )
    {
        if ( GameData_GetGameState() != GS_SCRAMBLING )
        {
			lstrcpy( szMessage, GetString( g_hInst, IDS_STAT_BLOCKED ) );
            StatusBar_SetText( szMessage );
            g_iLastStatusMessage = IDS_STAT_BLOCKED;
            PlayGameSound( GSND_BLOCKED, SND_ASYNC | SND_NODEFAULT );
        }
        return;
    }

    if ( iMouseRow == iBlankRow )
    {
        int iColumn;

        if ( iMouseColumn < iBlankColumn )
            for ( iColumn = iBlankColumn;  iColumn > iMouseColumn;  iColumn-- )
                GameData_SetCell( iBlankRow, iColumn, GameData_GetCell( iBlankRow, iColumn - 1 ) );
        else
            for ( iColumn = iBlankColumn;  iColumn < iMouseColumn;  iColumn++ )
                GameData_SetCell( iBlankRow, iColumn, GameData_GetCell( iBlankRow, iColumn + 1 ) );
    }
    else
    {
        int iRow;

        if ( iMouseRow < iBlankRow )
            for ( iRow = iBlankRow;  iRow > iMouseRow;  iRow-- )
                GameData_SetCell( iRow, iBlankColumn, GameData_GetCell( iRow - 1, iBlankColumn ) );
        else
            for ( iRow = iBlankRow;  iRow < iMouseRow;  iRow++ )
                GameData_SetCell( iRow, iBlankColumn, GameData_GetCell( iRow + 1, iBlankColumn ) );
    }

    /* Calculate the invalid rectangle based on the position
       of the blank cell and the clicked cell. This is done so
       as to optimize the painting code. */
    SetRect( &rectInvalid,
        TILEWIDTH * min(iMouseColumn, iBlankColumn),
        TILEHEIGHT * min(iMouseRow, iBlankRow),
        TILEWIDTH * (max(iMouseColumn, iBlankColumn) + 1),
        TILEHEIGHT * (max(iMouseRow, iBlankRow) + 1) );

    GameData_SetBlankCell( iMouseRow, iMouseColumn );
    /* GameData_SetCell( GameData_GetBlankRow(), GameData_GetBlankColumn(), EMPTY ); */

    /* "Invalidate" the client area, passing the rectangle that
       needs to be painted. */
    InvalidateRect( hwnd, &rectInvalid, TRUE );

    if ( GameData_GetGameState() == GS_SCRAMBLING )
    {
        return;
    }
    PlayGameSound( GSND_MOVE, SND_ASYNC | SND_NODEFAULT );

    /* Already tested GS_IDLING and GS_SCRAMBLING, must be GS_PLAYING. */
    if ( GameData_IsSolved() )
    {
        GameData_SetCell( GameData_GetBlankRow(), GameData_GetBlankColumn(), EMPTY );
        InvalidateRect( hwnd, NULL, TRUE );
        UpdateWindow( hwnd );

        PlayGameSound( GSND_SOLVED, SND_ASYNC | SND_NODEFAULT );
        ShowMessage( g_hInst, hwnd, IDS_MSG_SOLVED, IDS_WINDOWTITLE, MB_ICONEXCLAMATION | MB_OK );
        GameData_SetGameState( GS_IDLING );

        if ( ShowMessage( g_hInst, hwnd, IDS_MSG_ANOTHER, IDS_WINDOWTITLE, MB_ICONQUESTION | MB_YESNO ) == IDYES )
            SendMessage( hwnd, WM_COMMAND, IDM_GAME_NEW, 0L );
        else
            SendMessage( hwnd, WM_CLOSE, 0, 0L );
    }
}

void Taquin_OnKey ( HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags )
{
    int iMouseRow, iMouseColumn;

    iMouseRow    = GameData_GetBlankRow();
    iMouseColumn = GameData_GetBlankColumn();

    switch ( vk )
    {
        case VK_LEFT:
            iMouseColumn++;
            break;

        case VK_RIGHT:
            iMouseColumn--;
            break;

        case VK_UP:
            iMouseRow++;
            break;

        case VK_DOWN:
            iMouseRow--;
            break;

        default:
            return;
    }

    /* Send the updated mouse position as a WM_LBUTTONDOWN message. */
    SendMessage( hwnd, WM_LBUTTONDOWN, 0,
		MAKELPARAM( iMouseColumn * TILEWIDTH, iMouseRow * TILEHEIGHT )
    );
}

void Taquin_OnClose ( HWND hwnd )
{
    if ( GameData_GetGameState() != GS_PLAYING )
    {
        DestroyWindow( hwnd );
    }
    else
    {
        int rc = ShowMessage( g_hInst, hwnd, IDS_MSG_REALLYQUIT,
                IDS_WINDOWTITLE, MB_ICONQUESTION | MB_YESNO );
        if ( rc == IDYES )
        {
            DestroyWindow( hwnd );
        }
    }
}

void Taquin_OnDestroy ( HWND hwnd )
{
    DestroyWindow( g_hwndStatusBar );
    Settings_Write();
    WinHelp( hwnd, g_szHelpFileName, HELP_QUIT, 0L );   /* cancel WinHelp */
    PostQuitMessage( 0 );
}

HWND InitStatusBar ( HWND hwndParent )
{
    HWND hwndStatusBar;
    RECT rcClient;

    GetClientRect( hwndParent, &rcClient );

    hwndStatusBar = CreateWindow( "Taquin16StatusBar", "",
        WS_CHILDWINDOW | WS_VISIBLE,
        0, rcClient.bottom - StatusBar_GetHeight(),
        rcClient.right - rcClient.left, StatusBar_GetHeight(),
        hwndParent,              /* my parent's window handle */
        (HMENU) ID_STATUSBAR,    /* my child window ID */
        g_hInst,
        NULL
    );
    return hwndStatusBar;
}
