/***************************************************************************
 *
 * MODULE:	RMX Communications API
 * SOURCE:	$Source$
 * OVERVIEW:	This file contains a named pipe based implementation of
 *              the RMX Communications API. 
 *
 * Copyright (c) 1995 Johan Wikman
 *
 * $Log$
 *
 ***************************************************************************** 
 *
 * Permission to use, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided  
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 *****************************************************************************/

#define INCL_NOCOMMON
#define INCL_NOPMAPI
#define INCL_DOSDATETIME
#define INCL_DOSERRORS
#define INCL_DOSFILEMGR
#define INCL_DOSMEMMGR
#define INCL_DOSMISC
#define INCL_DOSMODULEMGR
#define INCL_DOSNMPIPES
#define INCL_DOSPROCESS
#include <rmxcomms.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <os2.h>


/****************************************************************************
 * CONSTANTS & DEFINES
 ****************************************************************************/
// We want a named pipe that is blocking, whose data is read and
// written as a byte stream (no pipe-managed messages), and of which
// an unlimited number of instances with the same name can be created.

const ULONG NP_OPEN_FLAGS   = NP_NOWRITEBEHIND       |
                              NP_NOINHERIT           |
                              NP_ACCESS_DUPLEX;

const ULONG NP_PIPE_FLAGS   = NP_WAIT                |
                              NP_TYPE_BYTE           |
		              NP_READMODE_BYTE       |
		              NP_UNLIMITED_INSTANCES;

const ULONG NP_MODE_FLAGS   = OPEN_FLAGS_WRITE_THROUGH |
                              OPEN_FLAGS_FAIL_ON_ERROR |
			      OPEN_FLAGS_NO_CACHE      |
			      OPEN_FLAGS_SEQUENTIAL    |
                              OPEN_SHARE_DENYNONE      |
			      OPEN_ACCESS_READWRITE    |
			      OPEN_FLAGS_NOINHERIT;



// We choose 4K as the size of the message buffers. A bigger size
// would give better throughput if we were transfering large amounts
// of data. As RMX only transfers large numbers of small messages, big
// message buffers are only a waste of resources.

const ULONG SIZE_PAGE    = 4096;
const ULONG SIZE_PIPE    = SIZE_PAGE;


// These are related to the reading of service entries in rmxpipe.dat. 
// If you change SIZE_SERVICE and SIZE_NAME, make sure to update the
// format string as well.

const ULONG SIZE_LINE    = 256;
const ULONG SIZE_SERVICE = 32;
const ULONG SIZE_NAME    = 128;
const CHAR acFormat[]    = "%32s%128s%128s";


// The name of the services file

const CHAR acServicesName[] = "rmxpipe.dat";


// AcUniqueName is used as a template for creating unique names and 
// SIZE_UNIQUEPART is the max size of the unique part of a unique name.

const CHAR  acUniqueName[]  = "\\PIPE\\RMXPIPE\\UNIQUE\\";

const ULONG SIZE_UNIQUEPART = 14;
const ULONG SIZE_UNIQUE     = sizeof(acUniqueName) + SIZE_UNIQUEPART;



/****************************************************************************
 * MODULE TYPES
 ****************************************************************************/

const ULONG SIZE_BUFFER = SIZE_PIPE - sizeof(HPIPE);

struct CONNECTION
{
  HPIPE hPipe;   // The actual handle of the named pipe.
  BYTE  bytes[SIZE_BUFFER];
};

typedef CONNECTION* PCONNECTION;


/****************************************************************************
 * MODULE VARIABLES
 ****************************************************************************/
// AcServicesPath is used for storing the full path of the initialization
// file, acServicesName is the actual name.

static CHAR acServicesPath[CCHMAXPATH];


/****************************************************************************
 * MODULE FUNCTIONS
 ****************************************************************************/

static BOOL   GetServiceName(PCSZ pcszService, 
			     PSZ  pszName);
static VOID   GetUniqueName (PSZ  pszName);


/****************************************************************************
 * DLL INITIALIZATION ROUTINE
 ****************************************************************************
 * 
 * FUNCTION: ULONG DLLMAIN(ULONG terminating, HMODULE hmod);
 *
 * INPUT:
 *    terminating: 0, the DLL is being initialized.
 *                 1, the DLL is being terminated.
 *    hmod:        The module handle of this DLL. 
 *
 * RETURN:
 *    0: initialization/termination failed.
 *    1: initialization/termination succeeded.
 *
 * OVERVIEW:
 *    The initialization routine determines the name of the file
 *    containing the names of services. 
 *
 *****************************************************************************/

DLL_PREAMBLE;

ULONG DLLMAIN(ULONG terminating, HMODULE hmod)
{
  if (!terminating)
    {
      // The file, where the names of different services are kept, must
      // be in the same directory as this DLL.
      
      if (DosQueryModuleName(hmod, sizeof(acServicesPath),
			     acServicesPath))
	return 0;
      
      // First we search for the last '\'

      int
	i = strlen(acServicesPath);
      
      while (acServicesPath[i] != '\\')
	i--;

      i++;

      // Once found, we set the character following it to 0

      acServicesPath[i] = 0;
      
      // and catanate the actual file name.
      
      strcat(acServicesPath, acServicesName);
    }
  
  return 1;
}


/****************************************************************************
 * EXPORTED FUNCTIONS
 ****************************************************************************
 * 
 * FUNCTION: ULONG RmxClose(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The handle of the connection to be closed.
 *
 * RETURN:
 *    NO_ERROR
 *
 *    DosClose
 *
 * OVERVIEW:
 *    This function closes the connection (pipe) and releases the
 *    memory associated with the handle.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxClose(HCONNECTION hConn)
{
  CONNECTION
    *pconn = (PCONNECTION) hConn;

  ULONG
    rc = DosClose(pconn->hPipe);
    
  DosFreeMem(pconn);

  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxConnect(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The handle of the connection to be connected.
 *
 * RETURN:
 *    NO_ERROR
 *
 *    DosConnectNPipe \ ERROR_INTERRUPT
 *
 * OVERVIEW:
 *    This function sets the connection into listening state and
 *    blocks until a client opens the connection.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxConnect(HCONNECTION hConn)
{
  CONNECTION
    *pconn = (PCONNECTION) hConn;

  ULONG
    rc = 0;

  // Loop until we get a proper exit code.
  
  do
    rc = DosConnectNPipe(pconn->hPipe);
  while (rc == ERROR_INTERRUPT);
  
  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxCreate(PCSZ pcszName, HCONNECTION* phConn);
 *
 * INPUT:
 *    pcszName: The name of the connection to be created.
 *
 * OUTPUT:
 *    phConn:   Points to a variable where the connection handle will
 *              be returned.
 *
 * RETURN:
 *    NO_ERROR
 *
 *    DosCreateNPipe
 *    DosAllocMem
 *
 * OVERVIEW:
 *    This function creates a connection (pipe) with the given name
 *    and returns the handle of the connection. 
 * 
 *    The pipe created is: writethrough
 *                         not inherited by child processes
 *                         read/write
 *                         blocking
 *                     
 *    Additionally an unlimited amount of connections can be created
 *    with the same name.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxCreate(PCSZ pcszName, HCONNECTION* phConn)
{
  ULONG 
    rc;
  HPIPE  
    hPipe;
  
  rc = DosCreateNPipe(pcszName,
		      &hPipe,
		      NP_OPEN_FLAGS,
		      NP_PIPE_FLAGS,
		      SIZE_PIPE,
		      SIZE_PIPE,
		      0);
 
  if (rc == NO_ERROR)
    {
      PVOID
	pvMemory;
      
      // Make sure we get the memory or a proper error code.
      
      do
	{
	  rc = DosAllocMem(&pvMemory, sizeof(CONNECTION), 
			   PAG_READ | PAG_WRITE | PAG_COMMIT);
	}
      while (rc == ERROR_INTERRUPT);
      
      if (rc == NO_ERROR)
	{
	  CONNECTION
	    *pconn = (PCONNECTION) pvMemory;
	  
	  memset(pconn, sizeof(CONNECTION), 0);
	  
	  pconn->hPipe = hPipe;
	  *phConn = (HCONNECTION) pconn;
	}
    }

  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxCreateUnique(ULONG*       pulSize, 
 *                                 PSZ          pszName, 
 *                                 HCONNECTION* phConn);
 *
 * INPUT/OUTPUT:
 *    pulSize: If NULL on input, the used unique name will not be
 *             returned in pszName. Otherwise should point to a
 *             variable that specifies the size of the buffer pointed
 *             to by pszName. If the function fails because the buffer
 *             is too small, the variable pointed to will contain the
 *             needed size.
 *
 * OUTPUT:
 *    pszName: If pulSize is non-NULL the buffer pointed to by pszName
 *             will on return contain the name used when the
 *             connection was created. 
 *    phConn:  Points to a variable where the connection handle will
 *             be returned.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_PARAM_TOO_SMALL
 *
 *    RmxCreate
 *
 * OVERVIEW:
 *    This function creates a connection (pipe) with a unique name
 *    and returns the handle of the connection. 
 *
 *****************************************************************************/

ULONG RMXENTRY RmxCreateUnique(ULONG*       pulSize, 
			       PSZ          pszName, 
			       HCONNECTION* phConn)
{
  if (pulSize && (*pulSize < SIZE_UNIQUE))
    {
      *pulSize = SIZE_UNIQUE;
      return ERROR_PARAM_TOO_SMALL;
    }
  
  CHAR
    achUniqueName[SIZE_UNIQUE];
  
  ULONG
    rc;

  do
    {
      GetUniqueName(achUniqueName);
      
      rc = RmxCreate(achUniqueName, phConn);
    }
  while (rc == ERROR_PIPE_BUSY);
  
  if ((rc == NO_ERROR) && pulSize)
    strcpy(pszName, achUniqueName);

  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxDisConnect(HCONNECTION hConn);
 *
 * INPUT:
 *    hConn: The handle of the connection to be disconnected.
 *
 * RETURN:
 *    NO_ERROR
 *
 *    DosDisConnectNPipe
 *
 * OVERVIEW:
 *    This function acknowledges that a client has closed a connection.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxDisConnect(HCONNECTION hConn)
{
  CONNECTION
    *pconn = (PCONNECTION) hConn;

  ULONG
    rc = DosDisConnectNPipe(pconn->hPipe);

  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxGetServiceName(PCSZ pcszService, ULONG* pulSize,
 *				     PSZ pszName);
 *
 * INPUT:
 *    pcszService: The service whose name is queried.
 *
 * INPUT/OUTPUT:
 *    pulSize:     Points to a variable that upon call should contain
 *                 the size of pszName. On return will contain the
 *                 length of pszName. If the function fails because
 *                 the buffer is too small, the variable will contain
 *                 the required size of the buffer.
 *
 * OUTPUT:
 *    pszName:     Points to a buffer where the name of the service
 *                 will be written.  
 *
 * RETURN:
 *    NO_ERROR
 *    RMXERR_UNKNOWN_SERVICE
 *    RMXERR_ENLARGE_BUFFER
 *
 * OVERVIEW:
 *    This function scans the file rmxpipe.dat, which should reside in
 *    the same directory as this DLL, for the given service entry. If
 *    the service is found, its name is returned.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxGetServiceName(PCSZ pcszService, ULONG* pulSize, PSZ pszName)
{	
  CHAR
    acName[CCHMAXPATH];
  
  if (!GetServiceName(pcszService, acName))
    return RMXERR_UNKNOWN_SERVICE;
  
  ULONG
    ulSize = strlen(acName) + 1;
  
  if (*pulSize < ulSize)
    {
      *pulSize = ulSize;
      return RMXERR_ENLARGE_BUFFER;
    }
  
  *pulSize = ulSize;
  strcpy(pszName, acName);

  return NO_ERROR;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxOpen(PCSZ         pcszHost
 *                         PCSZ         pcszPort, 
 *                         HCONNECTION* phConn);
 *
 * INPUT:
 *    pcszHost: The name of the host computer. If it is NULL then the
 *              local host will be used. 
 *    pcszPort: The port name to be opened.
 *
 * OUTPUT:
 *    phConn:   Points to a variable where the connection handle will
 *              be returned.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_PATH_NOT_FOUND
 *
 *    DosAllocMem
 *    DosOpen
 *    DosSetNPHState
 *
 * OVERVIEW:
 *    This function opens an existing connection (named pipe) and
 *    returns its handle. The pipe is always opened in byte and
 *    blocking mode.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxOpen(PCSZ         pcszHost,
		       PCSZ         pcszPort, 
		       HCONNECTION* phConn)
{
  // A NULL host is ok, an empty is not.
  
  if (pcszHost && (*pcszHost == 0))
    return ERROR_PATH_NOT_FOUND;
  
  // We now need to concatanate the host and port names. As we will be
  // needing a buffer anyway, we can allocate it right away.

  PVOID
    pvMemory;
	  
  // Make sure we get the memory or a proper error code.
	
  ULONG
    rc;
  
  do
    {
      rc = DosAllocMem(&pvMemory, sizeof(CONNECTION), 
		       PAG_READ | PAG_WRITE | PAG_COMMIT);
    }
  while (rc == ERROR_INTERRUPT);
  
  if (rc != NO_ERROR)
    return rc;
  
  CONNECTION
    *pconn = (PCONNECTION) pvMemory;
  PSZ
    pszName = (PSZ) pconn;
  
  pszName[0] = 0;
  
  if (pcszHost)
    strcpy(pszName, pcszHost);
  
  strcat(pszName, pcszPort);
  
  // Ok, then we can open the actual named pipe.
  
  HPIPE
    hPipe;

  // The following must be done in a loop. Even if we increase the
  // number of handles, another thread might get in between and
  // grab the new handle.

  do
    {
      ULONG
	ulActionTaken; // Not used.

      rc = DosOpen(pszName,
		   &hPipe,
		   &ulActionTaken,
		   0L,
		   FILE_NORMAL,
		   FILE_OPEN,
		   NP_MODE_FLAGS,
		   0L);
      
      if (rc == ERROR_TOO_MANY_OPEN_FILES)
	{
	  LONG
	    lReqCount = 1; // We need one additional handle
	  ULONG
	    ulCurMaxFH;    // Not used.

	  rc = DosSetRelMaxFH(&lReqCount, &ulCurMaxFH);
	}
    }
  while (rc == ERROR_TOO_MANY_OPEN_FILES);


  if (rc == NO_ERROR)
    {
      // The connection must be blocking.
      
      rc = DosSetNPHState(hPipe, NP_WAIT | NP_READMODE_BYTE);

      if (rc == NO_ERROR)
	{
	  memset(pconn, sizeof(CONNECTION), 0);
  
	  pconn->hPipe = hPipe;
	  *phConn = (HCONNECTION) pconn;
	}
    }
  
  if (rc != NO_ERROR)
    DosFreeMem(pconn);

  return rc;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxRead(HCONNECTION hConn, PBYTE pbBuffer, 
 *                         ULONG ulSize, ULONG* pulBytesRead);
 *
 * INPUT:
 *    hConn:        The handle of the connection.
 *    pbBuffer:     Pointer to buffer where the data will be returned.
 *    ulSize:       The size of the buffer.
 *
 * OUTPUT:
 *    pulBytesRead: Points to a variable where the number of read
 *                  bytes (or the required buffer size) will be returned.
 *
 * RETURN:
 *    NO_ERROR
 *    RMXERR_ENLARGE_BUFFER
 *
 *    DosRead
 *
 * OVERVIEW:
 *    This function returns the next message from the connection. The
 *    caller need not know in advance how many bytes are arriving. If
 *    the buffer is too small, the caller is instructed to enlarge it.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxRead(HCONNECTION hConn, 
		       PBYTE       pbBuffer,
		       ULONG       ulSize,
		       ULONG*      pulBytesRead)
{
  CONNECTION
    *pconn = (PCONNECTION) hConn;
  HPIPE
    hPipe = pconn->hPipe;

  // The length of the message is encoded in the first four bytes.

  ULONG
    *pulLength = (ULONG*) pconn->bytes;

  ULONG
    rc = NO_ERROR;

  // If the lenght is zero we havn't read anything yet. If it is
  // non-zero we have read the length of the message and concluded
  // that the user's buffer is too small and have instructed him to
  // enlarge it.

  if (*pulLength == 0)
    {
      ULONG
	ulBytesLeft = sizeof(ULONG),
        ulBytesRead = 0;

      do
	{
	  PBYTE
	    pbData = ((PBYTE) pulLength) + ulBytesRead;
	  
	  rc = DosRead(hPipe, pbData, ulBytesLeft, &ulBytesRead);
      
	  if ((rc != NO_ERROR) || (ulBytesRead == 0))
	    {
	      *pulLength = 0;
	      *pulBytesRead = 0;
	      return rc;
	    }
	  
	  ulBytesLeft -= ulBytesRead;
	}
      while (ulBytesLeft != 0);

      if (*pulLength > ulSize)
	{
	  *pulBytesRead = *pulLength;
	  return RMXERR_ENLARGE_BUFFER;
	}
    }
  else
    {
      ULONG
	ulTotalSize = *pulLength;
      
      if (ulTotalSize > ulSize)
	{
	  // If the buffer is still too small so the caller must enlarge it.
	  
	  *pulBytesRead = ulTotalSize;
	  return RMXERR_ENLARGE_BUFFER;
	}
    }
  
  // Now that we know how many bytes we are expecting we can read the
  // actual message.

  ULONG
    ulBytesLeft = *pulLength,
    ulBytesRead = 0,
    ulTotalBytesRead = 0;

  do
    {
      PBYTE
	pbData = pbBuffer + ulTotalBytesRead;
      
      rc = DosRead(hPipe, pbData, ulBytesLeft, &ulBytesRead);
      
      if ((rc != NO_ERROR) || (ulBytesRead == 0))
	{
	  *pulLength = 0;
	  *pulBytesRead = 0;
	  return rc;
	}
      
      ulBytesLeft -= ulBytesRead;
      ulTotalBytesRead += ulBytesRead;
    }
  while (ulBytesLeft != 0);

  *pulBytesRead = *pulLength;

  // We set the count to zero as an indication that all data
  // belonging to the current message has been read.

  *pulLength = 0;
  
  return NO_ERROR;
}


/****************************************************************************
 * 
 * FUNCTION: ULONG RmxWrite(HCONNECTION hConn, PCBYTE pcbBuffer, 
 *                          ULONG ulBytesToWrite);
 *
 * INPUT:
 *    hConn:          The handle of the connection.
 *    pcbBuffer:      Pointer to buffer containing data.
 *    ulBytesToWrite: Number of bytes to write.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_WRITE_FAULT
 *  
 *    DosWrite
 *
 * OVERVIEW:
 *    This function writes the specified amount of bytes to the
 *    connection. If the buffer is too large for one packet, it is
 *    transparently split into separate packets.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxWrite(HCONNECTION hConn, 
			PCBYTE      pbBuffer,	
			ULONG       ulBytesToWrite)
{
  if (ulBytesToWrite == 0)
    return NO_ERROR;
  
  PCONNECTION
    pconn = (PCONNECTION) hConn;
  HPIPE
    hPipe = pconn->hPipe;

  // The first four bytes of the message is the actual size of the
  // data. If this code was ever ported to PowerPC the different
  // endianess wuld have to be taken into account here.
  
  BYTE
    *pbData = pconn->bytes;

  *((ULONG*) pbData) = ulBytesToWrite;

  ULONG
    ulBytesInBuffer = SIZE_BUFFER - sizeof(ULONG);
  ULONG
    ulBytesToCopy   = ulBytesInBuffer < ulBytesToWrite ?
                      ulBytesInBuffer : ulBytesToWrite;

  // The first packet of a message (along with the size info) is
  // always copied to the buffer associated with the connection
  // handle, and only then is the buffer sent. This arrangement
  // guarantees that when the message is less than SIZE_BUFFER,
  // everything will be sent in one single packet. If the size was
  // sent separately, a message - no matter how small - would always
  // require two system calls. Copying bytes around is a lot cheaper
  // than system calls.

  memcpy(pbData + sizeof(ULONG), pbBuffer, ulBytesToCopy);
  
  // The adding of sizeof(ULONG) in the line below is required for
  // compensating for the first 4 bytes that actually are not part of
  // the message. I don't quite like this mixing of signed and
  // unsigned variables.
  
  ULONG
    ulBytesLeft   = ulBytesToWrite + sizeof(ULONG),
    ulBytesToSend = ulBytesToCopy  + sizeof(ULONG); 
  LONG
    lBytesSent = -sizeof(ULONG);

  ULONG
    rc = NO_ERROR;
  
  do
    {
      ULONG
	ulBytesSent;
      
      rc = DosWrite(hPipe, pbData, ulBytesToSend, &ulBytesSent);
      
      if (rc == NO_ERROR)
	{
	  if (ulBytesToSend != ulBytesSent)
	    rc = ERROR_WRITE_FAULT;
	  else
	    {
	      lBytesSent  += ulBytesToSend;
	      ulBytesLeft -= ulBytesToSend;

	      if (ulBytesLeft != 0)
		{
		  pbData = (PSZ)pbBuffer + lBytesSent;
		  
		  ulBytesToSend = SIZE_PIPE < ulBytesLeft ?
		                  SIZE_PIPE : ulBytesLeft;
		}
	    }
	}
    }
  while ((ulBytesLeft != 0) && (rc == NO_ERROR));

  *((ULONG*) pconn->bytes) = 0;

  return rc;
}



/****************************************************************************
 * MODULE FUNCTIONS
 ****************************************************************************
 * 
 * FUNCTION: BOOL GetServiceName(PCSZ pcszService, PSZ pszName);
 *
 * INPUT:
 *    pcszService: The service.
 *
 * OUTPUT:
 *    pszName:     Points to the buffer where the name will be returned.
 *
 * RETURN:
 *    TRUE:  The service was found.
 *    FALSE: Otherwise
 *
 * OVERVIEW:
 *    This function searches the service file for the specified
 *    service. If would be faster to read the entire file on startup
 *    but that would require an application to be restarted before it
 *    would see any changes.
 *
 *****************************************************************************/

static BOOL GetServiceName(PCSZ pcszService, PSZ pszName)
{
  // It would be possible to read the entire file at startup and keep
  // the entries in memory, but then changes could not be made
  // dynamically. 
  
  FILE
    *file = fopen(acServicesPath, "rt");
  
  if (!file)
    return FALSE;

  BOOL
    bFound = FALSE;
  CHAR
    acLine[SIZE_LINE];

  while (!bFound && fgets(acLine, SIZE_LINE, file))
    {
      // Remove comments.
      
      PSZ
	psz = acLine;
      
      while (*psz)
	if (*psz == '#')
	  *psz = 0;
	else
	  psz++;
      
      CHAR
	acService[SIZE_SERVICE],
        acName   [SIZE_NAME];

      // The second acName is only there in order to allow us to
      // detect an illegal line.
      
      INT
	iCount = sscanf(acLine, acFormat, acService, acName, acName);

      if (iCount == 2)
	{
	  if (stricmp(acService, pcszService) == 0)
	    {
	      strcpy(pszName, "\\PIPE");
	      strcat(pszName, acName);

	      bFound = TRUE;
	    }
	}
    }

  fclose(file);
  
  return bFound;
}


/****************************************************************************
 * 
 * FUNCTION: VOID GetUniqueName(PSZ pszName);
 *
 * OUTPUT:
 *    pszName:     Points to a buffer where the unique name will be written
 *
 * OVERVIEW:
 *    This function generates a unique name from the PID, the thread
 *    ordinal and the current time.
 *
 *****************************************************************************/

static VOID GetUniqueName(PSZ pszName)
{
  // The following system calls never fails.

  PPIB
    ppib;
  PTIB
    ptib;
  
  DosGetInfoBlocks(&ptib, &ppib);
  
  DATETIME
    dateTime;
  
  DosGetDateTime(&dateTime);
  
  USHORT
    pid        = (USHORT) ppib->pib_ulpid,
    tord       = (USHORT) ptib->tib_ordinal,
    seconds    = (USHORT) dateTime.seconds,	// <= 59
    hundredths = (USHORT) dateTime.hundredths;  // <= 99

  sprintf(pszName, "%s%hu%hu%hu%hu", 
	acUniqueName, pid, tord, seconds, hundredths);
}



