#define INCL_DOSSEMAPHORES
#define INCL_GPIBITMAPS
#define INCL_GPIPRIMITIVES
#define INCL_WINRECTANGLES
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "cmnspr.h"

static USHORT accessSem(PCMNHANDLE pchHandle,USHORT usAction);
static HPS getMemHps(HAB habAnchor);
static BOOL clipBltPoints(HAB habAnchor,PPOINTL pptlArray,PSIZEL pszlPlay);
static BOOL drawSpriteAt(HPS hpsDraw,
                         HCSSPRITE hsSprite,
                         PSIZEL pszlPlay,
                         PPOINTL pptlSprite,
                         PRECTL prclSprite);
static BOOL drawBackAt(HPS hpsDraw,
                       HCSPLAYGROUND hpgPlay,
                       PRECTL prclDest,
                       PSIZEL pszlDest,
                       PRECTL prclSrc);

USHORT EXPENTRY queryHandle(PVOID pvHandle)
//-------------------------------------------------------------------------
// This function returns the type of the specified handle
//
// Input:  lhHandle - specifies the handle to query
// Returns:  QH_* constant
//-------------------------------------------------------------------------
{
   ULONG ulSig;

   if (pvHandle==NULLHANDLE) {
      return QH_ERROR;
   } /* endif */

   ulSig=*((PULONG)pvHandle);

   switch (ulSig) {
   case SIG_CSPLAYGROUND:
      return QH_HCSPLAYGROUND;
   case SIG_CSSPRITE:
      return QH_HCSSPRITE;
   default:
      return QH_ERROR;
   } /* endswitch */
}

static USHORT accessSem(PCMNHANDLE pchHandle,USHORT usAction)
//-------------------------------------------------------------------------
// This function provides semaphore access for mutual
// exclusion of private data access.
//
// Input:  pchHandle - points to the handle header
//         usAction - specifies the action to perform:
//            ACCSEM_SET - requests access to the handle
//            ACCSEM_CLEAR - relinquishes access to the handle
//            ACCSEM_ALREADYSET - not used
//            ACCSEM_NOTSET - not used
// Returns:  ACCSEM_ERROR if an error occurred, else the action to take
//           on the next call to this function
//-------------------------------------------------------------------------
{
   switch (usAction) {
   case ACCSEM_SET:
      if ((pchHandle->ulStatus & HSTATUS_INLIBRARY)!=0) {
         return ACCSEM_ALREADYSET;
      } /* endif */

      DosRequestMutexSem(pchHandle->hsmAccess,SEM_INDEFINITE_WAIT);
      pchHandle->ulStatus|=HSTATUS_INLIBRARY;
      return ACCSEM_CLEAR;
   case ACCSEM_CLEAR:
      if ((pchHandle->ulStatus & HSTATUS_INLIBRARY)==0) {
         return ACCSEM_NOTSET;
      } /* endif */

      DosReleaseMutexSem(pchHandle->hsmAccess);
      pchHandle->ulStatus&=~HSTATUS_INLIBRARY;
      return ACCSEM_SET;
   case ACCSEM_ALREADYSET:
      return ACCSEM_NOTSET;
   case ACCSEM_NOTSET:
      return ACCSEM_ALREADYSET;
   default:
      return ACCSEM_ERROR;
   } /* endswitch */
}

static HPS getMemHps(HAB habAnchor)
//-------------------------------------------------------------------------
// This function creates an HPS associated with a memory HDC.  The
// HDC handle can be retrieved using the GpiQueryDevice() function.
//
// Input:  habAnchor - anchor block of the calling thread.
// Returns:  HPS handle if successful, NULLHANDLE otherwise
//-------------------------------------------------------------------------
{
   HDC hdcMem;
   SIZEL szlHps;
   HPS hpsMem;

   hdcMem=DevOpenDC(habAnchor,OD_MEMORY,"*",0,NULL,NULLHANDLE);
   if (hdcMem==NULLHANDLE) {
      return NULLHANDLE;
   } /* endif */

   szlHps.cx=0;
   szlHps.cy=0;

   hpsMem=GpiCreatePS(habAnchor,
                      hdcMem,
                      &szlHps,
                      PU_PELS|GPIT_MICRO|GPIA_ASSOC);
   if (hpsMem==NULLHANDLE) {
      DevCloseDC(hdcMem);
   } /* endif */

   return hpsMem;
}

static BOOL clipBltPoints(HAB habAnchor,PPOINTL pptlArray,PSIZEL pszlPlay)
//-------------------------------------------------------------------------
// This function clips the first two points in pptlArray to a rectangle
// of size pszlPlay.  The last two points in pptlArray are then adjusted
// by the amount clipped.
//
// It is assumed that the first two points refer to a coordinate space
// of size pszlPlay and that the two rectangles formed by the first and
// last pair of points in pptlArray are of the same size.
//
// Input:  habAnchor - anchor block of the calling thread.
//         pptlArray - points to array of 4 points for GpiBitBlt()
//         pszlPlay - points to the size of the playground to clip to
// Output:  pptlArray - points to adjusted array
// Returns:  TRUE if at least one pel was *not* clipped, FALSE if all
//           points fell outside of the clipping region.
//-------------------------------------------------------------------------
{
   RECTL rclPlay;
   RECTL rclDest;
   RECTL rclInter;
   RECTL rclDelta;

   rclPlay.xLeft=0;
   rclPlay.yBottom=0;
   rclPlay.xRight=pszlPlay->cx-1;
   rclPlay.yTop=pszlPlay->cy-1;

   rclDest.xLeft=pptlArray[0].x;
   rclDest.yBottom=pptlArray[0].y;
   rclDest.xRight=pptlArray[1].x;
   rclDest.yTop=pptlArray[1].y;

   WinIntersectRect(habAnchor,&rclInter,&rclPlay,&rclDest);

   //----------------------------------------------------------------------
   // If the result is an empty rectangle, return FALSE to indicate so.
   //----------------------------------------------------------------------
   if (WinIsRectEmpty(habAnchor,&rclInter)) {
      return FALSE;
   } /* endif */

   rclDelta.xLeft=rclDest.xLeft-rclInter.xLeft;
   rclDelta.yBottom=rclDest.yBottom-rclInter.yBottom;
   rclDelta.xRight=rclDest.xRight-rclInter.xRight;
   rclDelta.yTop=rclDest.yTop-rclInter.yTop;

   pptlArray[0].x-=rclDelta.xLeft;
   pptlArray[0].y-=rclDelta.yBottom;
   pptlArray[1].x-=rclDelta.xRight;
   pptlArray[1].y-=rclDelta.yTop;
   pptlArray[2].x-=rclDelta.xLeft;
   pptlArray[2].y-=rclDelta.yBottom;
   pptlArray[3].x-=rclDelta.xRight;
   pptlArray[3].y-=rclDelta.yTop;
   return TRUE;
}

static BOOL drawSpriteAt(HPS hpsDraw,
                         HCSSPRITE hsSprite,
                         PSIZEL pszlPlay,
                         PPOINTL pptlSprite,
                         PRECTL prclSprite)
//-------------------------------------------------------------------------
// This function draws the sprite at the specified position.  It is assumed
// that the background has already been drawn into hpsDraw before this
// function is called.
//
// Input:  hpsDraw - handle of the presentation space to draw in
//         hsSprite - handle of the sprite to draw
//         pszlPlay - points to the size of hpsDraw.  If NULL, the size
//                    of the playground is used.
//         pptlSprite - points to the point specifying the position.  If
//                      NULL, the sprite's current position is used.
//         prclSprite - points to the rectangle within the sprite to draw,
//                      i.e. a rectangle bounded by (0,0)-(cx,cy) which
//                      is added to the sprite's position to determine
//                      how much of the sprite is to be drawn.  If NULL,
//                      the entire sprite is drawn.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   SIZEL szlUsePlay;
   POINTL ptlUseSpr;
   RECTL rclUseSpr;
   POINTL aptlPoints[4];

   if ((!hsSprite->hpgPlay->bUpdate) || (!hsSprite->bVisible)) {
      return TRUE;
   } /* endif */

   //----------------------------------------------------------------------
   // Initialize the local variables with either what was passed in or
   // the defaults as noted above in the function prologue
   //----------------------------------------------------------------------
   if (pszlPlay==NULL) {
      CmnSprQueryPlaygroundSize(hsSprite->hpgPlay,&szlUsePlay);
   } else {
      szlUsePlay=*pszlPlay;
   } /* endif */

   if (pptlSprite==NULL) {
      ptlUseSpr=hsSprite->ptlPos;
   } else {
      ptlUseSpr=*pptlSprite;
   } /* endif */

   if (prclSprite==NULL) {
      rclUseSpr.xLeft=0;
      rclUseSpr.yBottom=0;
      rclUseSpr.xRight=hsSprite->bmihMask.cx;
      rclUseSpr.yTop=hsSprite->bmihMask.cy;
   } else {
      rclUseSpr=*prclSprite;
   } /* endif */

   rclUseSpr.xRight-=rclUseSpr.xLeft;
   rclUseSpr.yTop-=rclUseSpr.yBottom;

   aptlPoints[0].x=ptlUseSpr.x+rclUseSpr.xLeft;
   aptlPoints[0].y=ptlUseSpr.y+rclUseSpr.yBottom;
   aptlPoints[1].x=aptlPoints[0].x+rclUseSpr.xRight-1;
   aptlPoints[1].y=aptlPoints[0].y+rclUseSpr.yTop-1;
   aptlPoints[2].x=rclUseSpr.xLeft;
   aptlPoints[2].y=rclUseSpr.yBottom;
   aptlPoints[3].x=aptlPoints[2].x+rclUseSpr.xRight;
   aptlPoints[3].y=aptlPoints[2].y+rclUseSpr.yTop;

   if (clipBltPoints(hsSprite->habAnchor,aptlPoints,&szlUsePlay)) {
      //-------------------------------------------------------------------
      // Blit the mask and then the bitmap
      //-------------------------------------------------------------------
      GpiWCBitBlt(hpsDraw,
                  hsSprite->hbmMask,
                  4,
                  aptlPoints,
                  ROP_SRCAND,
                  BBO_IGNORE);

      GpiWCBitBlt(hpsDraw,
                  hsSprite->hbmBitmap,
                  4,
                  aptlPoints,
                  ROP_SRCPAINT,
                  BBO_IGNORE);
   } /* endif */

   return TRUE;
}

static BOOL drawBackAt(HPS hpsDraw,
                       HCSPLAYGROUND hpgPlay,
                       PRECTL prclDest,
                       PSIZEL pszlDest,
                       PRECTL prclSrc)
//-------------------------------------------------------------------------
// This function draws the background in the specified presentation space.
//
// Input:  hpsDraw - handle of the presentation space to draw in
//         hpgPlay - handle of the playground containing the background
//         prclDest - points to the destination rectangle.  If NULL, the
//                    value of prclSrc is used.
//         pszlDest - points to the size of hpsDraw.  If NULL, the size of
//                    the playground is used.
//         prclSrc - points to the source rectangle.  If NULL, the entire
//                   background is painted.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   RECTL rclUseSrc;
   RECTL rclUseDest;
   SIZEL szlUse;
   POINTL aptlPoints[4];

   if (!hpgPlay->bUpdate) {
      return TRUE;
   } /* endif */

   if (prclSrc==NULL) {
      rclUseSrc.xLeft=0;
      rclUseSrc.yBottom=0;
      rclUseSrc.xRight=hpgPlay->bmihBack.cx;
      rclUseSrc.yTop=hpgPlay->bmihBack.cy;
   } else {
      rclUseSrc=*prclSrc;
   } /* endif */

   if (prclDest==NULL) {
      rclUseDest=rclUseSrc;
      rclUseDest.xRight--;
      rclUseDest.yTop--;
   } else {
      rclUseDest=*prclDest;
   } /* endif */

   if (pszlDest==NULL) {
      szlUse.cx=hpgPlay->bmihBack.cx;
      szlUse.cy=hpgPlay->bmihBack.cy;
   } else {
      szlUse=*pszlDest;
   } /* endif */

   aptlPoints[0].x=rclUseDest.xLeft;
   aptlPoints[0].y=rclUseDest.yBottom;
   aptlPoints[1].x=rclUseDest.xRight;
   aptlPoints[1].y=rclUseDest.yTop;
   aptlPoints[2].x=rclUseSrc.xLeft;
   aptlPoints[2].y=rclUseSrc.yBottom;
   aptlPoints[3].x=rclUseSrc.xRight;
   aptlPoints[3].y=rclUseSrc.yTop;

   if (clipBltPoints(hpgPlay->habAnchor,aptlPoints,&szlUse)) {
      //-------------------------------------------------------------------
      // If there is a background bitmap, blit it, otherwise black out the
      // area.
      //-------------------------------------------------------------------
      if (hpgPlay->hbmBack!=NULLHANDLE) {
         GpiWCBitBlt(hpsDraw,
                     hpgPlay->hbmBack,
                     4,
                     aptlPoints,
                     ROP_SRCCOPY,
                     BBO_IGNORE);
      } else {
         //----------------------------------------------------------------
         // WinFillRect() excludes the top and right of the rectangle
         //----------------------------------------------------------------
         rclUseDest.xRight++;
         rclUseDest.yTop++;
         WinFillRect(hpsDraw,&rclUseDest,hpgPlay->lBackColor);
      } /* endif */
   } /* endif */

   return TRUE;
}

SPRERROR EXPENTRY CmnSprCreatePlayground(HAB habAnchor,
                                         PHCSPLAYGROUND phpgPlay)
//-------------------------------------------------------------------------
// This function creates a playground to which sprites can be added.
//
// Input:  habAnchor - anchor block of the calling thread.
// Output:  phpgPlay - points to the variable with the HCSPLAYGROUND handle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   BITMAPINFOHEADER2 bmihInfo;
   LONG lValue;

   if (DosAllocMem((PPVOID)phpgPlay,
                   sizeof(CSPLAYGROUND),
                   PAG_READ|PAG_WRITE|OBJ_TILE|PAG_COMMIT)) {
      *phpgPlay=NULL;
      return SPR_ERR_NOMEMORY;
   } /* endif */

   (*phpgPlay)->ulSig=SIG_CSPLAYGROUND;
   (*phpgPlay)->ulStatus=0;

   if (DosCreateMutexSem(NULL,&(*phpgPlay)->hsmAccess,0,FALSE)) {
      DosFreeMem(*phpgPlay);
      return SPR_ERR_RESOURCE;
   } /* endif */

   (*phpgPlay)->habAnchor=habAnchor;

   (*phpgPlay)->hpsWork=getMemHps(habAnchor);
   if ((*phpgPlay)->hpsWork==NULLHANDLE) {
      DosCloseMutexSem((*phpgPlay)->hsmAccess);
      DosFreeMem(*phpgPlay);
      *phpgPlay=NULL;
      return SPR_ERR_RESOURCE;
   } /* endif */

   (*phpgPlay)->hdcWork=GpiQueryDevice((*phpgPlay)->hpsWork);

   bmihInfo.cbFix=16;
   bmihInfo.cx=MAX_SPRITE_CX*2;
   bmihInfo.cy=MAX_SPRITE_CY*2;
   bmihInfo.cPlanes=1;

   DevQueryCaps((*phpgPlay)->hdcWork,CAPS_COLOR_BITCOUNT,1,&lValue);
   bmihInfo.cBitCount=lValue;

   (*phpgPlay)->hbmWork=GpiCreateBitmap((*phpgPlay)->hpsWork,
                                        &bmihInfo,
                                        0,
                                        NULL,
                                        NULL);
   if ((*phpgPlay)->hbmWork==NULLHANDLE) {
      GpiDestroyPS((*phpgPlay)->hpsWork);
      DevCloseDC((*phpgPlay)->hdcWork);
      DosCloseMutexSem((*phpgPlay)->hsmAccess);
      DosFreeMem(*phpgPlay);
      *phpgPlay=NULL;
      return SPR_ERR_RESOURCE;
   } /* endif */

   GpiSetBitmap((*phpgPlay)->hpsWork,(*phpgPlay)->hbmWork);

   (*phpgPlay)->lBackColor=CLR_BLACK;
   (*phpgPlay)->bUpdate=TRUE;

   (*phpgPlay)->hbmBack=NULLHANDLE;
   (*phpgPlay)->ulNumMembers=0;
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprDestroyPlayground(HCSPLAYGROUND hpgPlay)
//-------------------------------------------------------------------------
// This function destroys the playground including any sprites that are
// still members of it.  All resources consumed by the playground,
// including the back bitmap, are returned to the system.
//
// Input:  hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;
   ULONG ulIndex;
   HCSSPRITE hsSprite;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      GpiDeleteBitmap(hpgPlay->hbmBack);
   } /* endif */

   for (ulIndex=0; ulIndex<hpgPlay->ulNumMembers; ulIndex++) {
      hsSprite=hpgPlay->ahsSprites[ulIndex];
      CmnSprRemoveSprite(hpgPlay,hsSprite);
      CmnSprDestroySprite(hsSprite);
   } /* endfor */

   GpiSetBitmap(hpgPlay->hpsWork,NULLHANDLE);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      GpiDeleteBitmap(hpgPlay->hbmBack);
   } /* endif */

   GpiDestroyPS(hpgPlay->hpsWork);
   DevCloseDC(hpgPlay->hdcWork);

   accessSem((PCMNHANDLE)hpgPlay,usSemAction);

   DosCloseMutexSem(hpgPlay->hsmAccess);
   DosFreeMem(hpgPlay);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprAddSprite(HCSPLAYGROUND hpgPlay,HCSSPRITE hsSprite)
//-------------------------------------------------------------------------
// This function labels a sprite as a "member" of the specified playground.
// Doing so allows the application to control the sprite's position,
// visibility, etc. on a drawing surface.
//
// Input:  hpgPlay - handle to the playground
//         hsSprite - handle to the sprite to add
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } else
   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);

   if (hsSprite->hpgPlay!=NULL) {
      accessSem((PCMNHANDLE)hpgPlay,usSemAction);
      return SPR_ERR_HASPLAYGROUND;
   } else
   if (hpgPlay->ulNumMembers==MAX_SPRITES) {
      accessSem((PCMNHANDLE)hpgPlay,usSemAction);
      return SPR_ERR_PLAYGROUNDFULL;
   } /* endif */

   hpgPlay->ahsSprites[hpgPlay->ulNumMembers]=hsSprite;
   hpgPlay->ulNumMembers++;

   hsSprite->hpgPlay=hpgPlay;

   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprRemoveSprite(HCSPLAYGROUND hpgPlay,HCSSPRITE hsSprite)
//-------------------------------------------------------------------------
// This function removes the sprite from the membership list of the
// specified playground.  The sprite can then be added to another
// playground, or this one at a later time.
//
// Since there is a limited number of sprites that can be members of
// a playground, this function can be used to temporarily remove unused
// sprites from a playground so that others can be used.
//
// Input:  hpgPlay - handle to the playground
//         hsSprite - handle to the sprite to remove
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;
   ULONG ulIndex;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } else
   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);

   for (ulIndex=0; ulIndex<hpgPlay->ulNumMembers; ulIndex++) {
      if (hpgPlay->ahsSprites[ulIndex]==hsSprite) {
         break;
      } /* endif */
   } /* endfor */

   if (ulIndex==hpgPlay->ulNumMembers) {
      accessSem((PCMNHANDLE)hpgPlay,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   //----------------------------------------------------------------------
   // Adjust the member array by moving all of the sprites after the one
   // being removed to the slot just before there current position.  Then,
   // decrement the number of members and we're done.
   //----------------------------------------------------------------------
   hpgPlay->ulNumMembers--;

   while (ulIndex<hpgPlay->ulNumMembers) {
      hpgPlay->ahsSprites[ulIndex]=hpgPlay->ahsSprites[ulIndex+1];
      ulIndex++;
   } /* endwhile */

   hpgPlay->ahsSprites[ulIndex]=NULL;
   hsSprite->hpgPlay=NULL;

   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprSetPlaygroundBack(HCSPLAYGROUND hpgPlay,
                                          HBITMAP hbmNew,
                                          HBITMAP *phbmOld)
//-------------------------------------------------------------------------
// This function sets the background bitmap of the playground.
//
// Note that, once this function is called, the bitmap is managed by
// the sprite subsystem.  The bitmap should *NOT* be deleted by the
// application unless the bitmap is "unset" from the playground (by
// calling this function again with a different handle).
//
// Input:  hpgPlay - handle to the playground
//         hbmNew - handle to the new bitmap to used as the background
// Output:  phbmOld - points to the handle to the old background bitmap.
//          This can be NULL, meaning that the application isn't interested
//          in receiving this value.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);

   if (phbmOld!=NULL) {
      *phbmOld=hpgPlay->hbmBack;
   } /* endif */

   hpgPlay->hbmBack=hbmNew;

   //----------------------------------------------------------------------
   // We're only interested in the cx and cy fields
   //----------------------------------------------------------------------
   hpgPlay->bmihBack.cbFix=16;
   GpiQueryBitmapInfoHeader(hpgPlay->hbmBack,&hpgPlay->bmihBack);

   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQueryPlaygroundBack(HCSPLAYGROUND hpgPlay,
                                            HBITMAP *phbmBack)
//-------------------------------------------------------------------------
// This function returns the handle of the background bitmap currently in
// use.
//
// Input:  hpgPlay - handle to the playground
// Output:  phbmBack - points to the handle to the background bitmap.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);
   *phbmBack=hpgPlay->hbmBack;
   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprSetPlaygroundSize(HCSPLAYGROUND hpgPlay,
                                          PSIZEL pszlSize)
//-------------------------------------------------------------------------
// This function sets the playground size for playgrounds that do not have
// a bitmap set as the background.
//
// Input:  hpgPlay - handle to the playground
//         pszlSize - points to the size of the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      accessSem((PCMNHANDLE)hpgPlay,usSemAction);
      return SPR_ERR_HASBACKGROUND;
   } /* endif */

   hpgPlay->bmihBack.cx=pszlSize->cx;
   hpgPlay->bmihBack.cy=pszlSize->cy;

   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQueryPlaygroundSize(HCSPLAYGROUND hpgPlay,
                                            PSIZEL pszlSize)
//-------------------------------------------------------------------------
// This function returns the size of the playground.  For playgrounds with
// bitmaps set as the background, the returned value is the size of the
// bitmap.  Otherwise, the returned value is that which was specified on
// the last call to CmnSprSetPlaygroundSize().
//
// Input:  hpgPlay - handle to the playground
//         pszlSize - points to the variable to receive the size of the
//                    playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);
   pszlSize->cx=hpgPlay->bmihBack.cx;
   pszlSize->cy=hpgPlay->bmihBack.cy;
   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprSetPlaygroundColor(HCSPLAYGROUND hpgPlay,
                                           LONG lBackColor)
//-------------------------------------------------------------------------
// This function sets the new background color of the playground and is
// only valid if the playground doesn't have a bitmap.
//
// Input:  hpgPlay - handle to the playground
//         lBackColor - specifies the new background color
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      accessSem((PCMNHANDLE)hpgPlay,usSemAction);
      return SPR_ERR_HASBACKGROUND;
   } /* endif */

   hpgPlay->lBackColor=lBackColor;
   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQueryPlaygroundColor(HCSPLAYGROUND hpgPlay,
                                             PLONG plBackColor)
//-------------------------------------------------------------------------
// This function returns the background color of the playground and is
// only valid if the playground doesn't have a bitmap.
//
// Input:  hpgPlay - handle to the playground
//         plBackColor - points to the variable to receive the background
//                       color
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      accessSem((PCMNHANDLE)hpgPlay,usSemAction);
      return SPR_ERR_HASBACKGROUND;
   } /* endif */

   *plBackColor=hpgPlay->lBackColor;
   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprSetUpdateFlag(HCSPLAYGROUND hpgPlay,BOOL bUpdate)
//-------------------------------------------------------------------------
// This function sets the update flag for the playground.  If FALSE, no
// drawing actually takes place in any of the functions requiring an HPS,
// and the value of the HPS handle may be NULLHANDLE.  If TRUE, updating
// is reenabled, but you should still call CmnSprDrawPlayground() to refresh
// the screen with the current contents.
//
// Input:  hpgPlay - handle to the playground
//         bUpdate - specifies the new update flag
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);
   hpgPlay->bUpdate=bUpdate;
   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQueryUpdateFlag(HCSPLAYGROUND hpgPlay,PBOOL pbUpdate)
//-------------------------------------------------------------------------
// This function returns the setting of the update flag.  See the notes
// for CmnSprSetUpdateFlag() for more information about this setting.
//
// Input:  hpgPlay - handle to the playground
//         pbUpdate - points to the variable to receive the update flag
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);
   *pbUpdate=hpgPlay->bUpdate;
   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprDrawPlayground(HPS hpsDraw,HCSPLAYGROUND hpgPlay)
//-------------------------------------------------------------------------
// This function redraws the playground and all sprites belonging to the
// playground.
//
// Input:  hpsDraw - handle to the HPS to draw the playground in
//         hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;
   ULONG ulIndex;

   if (queryHandle(hpgPlay)!=QH_HCSPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hpgPlay,ACCSEM_SET);

   if (hpgPlay->bUpdate) {
      drawBackAt(hpsDraw,hpgPlay,NULL,NULL,NULL);

      for (ulIndex=0; ulIndex<hpgPlay->ulNumMembers; ulIndex++) {
         CmnSprDrawSprite(hpsDraw,hpgPlay->ahsSprites[ulIndex]);
      } /* endfor */
   } /* endif */

   accessSem((PCMNHANDLE)hpgPlay,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprCreateSprite(HAB habAnchor,
                                     HBITMAP hbmBitmap,
                                     PHCSSPRITE phsSprite)
//-------------------------------------------------------------------------
// This function creates a sprite from the specified bitmap.  The sprite
// cannot be moved, shown, etc., however, until it is associated with a
// playground.
//
// The color black is used as the transparency color.  If you need to use
// black in the bitmap without it becoming transparent, use the next
// closest color.  <grin>
//
// New sprites are initialized as being at position (0,0) and hidden.
//
// Note that, once this function is called, the bitmap is managed by
// the sprite subsystem.  The bitmap should *NOT* be deleted by the
// application or else unpredictable results will occur.
//
// Input:  habAnchor - anchor block of the calling thread.
//         hbmBitmap - handle to the bitmap
// Output:  phsSprite - points to the sprite handle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   HPS hpsMem;
   HDC hdcMem;
   POINTL aptlPoints[4];

   if (DosAllocMem((PPVOID)phsSprite,
                   sizeof(CSSPRITE),
                   PAG_READ|PAG_WRITE|OBJ_TILE|PAG_COMMIT)) {
      *phsSprite=NULL;
      return SPR_ERR_NOMEMORY;
   } /* endif */

   (*phsSprite)->ulSig=SIG_CSSPRITE;
   (*phsSprite)->ulStatus=0;

   if (DosCreateMutexSem(NULL,&(*phsSprite)->hsmAccess,0,FALSE)) {
      DosFreeMem(*phsSprite);
      return SPR_ERR_RESOURCE;
   } /* endif */

   (*phsSprite)->habAnchor=habAnchor;

   (*phsSprite)->hbmBitmap=hbmBitmap;

   (*phsSprite)->ptlPos.x=0;
   (*phsSprite)->ptlPos.y=0;
   (*phsSprite)->bVisible=FALSE;

   (*phsSprite)->bmihBitmap.cbFix=16;
   GpiQueryBitmapInfoHeader((*phsSprite)->hbmBitmap,&(*phsSprite)->bmihBitmap);

   if (((*phsSprite)->bmihBitmap.cx>MAX_SPRITE_CX) ||
       ((*phsSprite)->bmihBitmap.cy>MAX_SPRITE_CY)) {
      DosCloseMutexSem((*phsSprite)->hsmAccess);
      DosFreeMem(*phsSprite);
      return SPR_ERR_BADHANDLE;
   } /* endif */

   //----------------------------------------------------------------------
   // Get an OD_MEMORY HDC and HPS to create the mask in.  Since we will
   // save the bitmap handle, but don't give a $%#@ about the HDC/HPS, they
   // can be local variables.
   //----------------------------------------------------------------------
   hpsMem=getMemHps(habAnchor);
   if (hpsMem==NULLHANDLE) {
      DosCloseMutexSem((*phsSprite)->hsmAccess);
      DosFreeMem(*phsSprite);
      *phsSprite=NULL;
      return SPR_ERR_RESOURCE;
   } /* endif */

   hdcMem=GpiQueryDevice(hpsMem);

   (*phsSprite)->bmihMask=(*phsSprite)->bmihBitmap;
   (*phsSprite)->bmihMask.cPlanes=1;
   (*phsSprite)->bmihMask.cBitCount=1;

   (*phsSprite)->hbmMask=GpiCreateBitmap(hpsMem,
                                         &(*phsSprite)->bmihMask,
                                         0,
                                         NULL,
                                         NULL);
   if ((*phsSprite)->hbmMask==NULLHANDLE) {
      GpiDestroyPS(hpsMem);
      DevCloseDC(hdcMem);
      DosCloseMutexSem((*phsSprite)->hsmAccess);
      DosFreeMem(*phsSprite);
      *phsSprite=NULL;
      return SPR_ERR_RESOURCE;
   } /* endif */

   GpiSetBitmap(hpsMem,(*phsSprite)->hbmMask);

   aptlPoints[0].x=0;
   aptlPoints[0].y=0;
   aptlPoints[1].x=aptlPoints[0].x+(*phsSprite)->bmihMask.cx-1;
   aptlPoints[1].y=aptlPoints[0].y+(*phsSprite)->bmihMask.cy-1;
   aptlPoints[2].x=0;
   aptlPoints[2].y=0;
   aptlPoints[3].x=aptlPoints[2].x+(*phsSprite)->bmihBitmap.cx;
   aptlPoints[3].y=aptlPoints[2].y+(*phsSprite)->bmihBitmap.cy;

   //----------------------------------------------------------------------
   // Set the foreground to white and the background to black so that this
   // works.  The resulting behavior in the GpiWCBitBlt() call is
   // inconsistent with the docs, so I don't know what to think.
   //----------------------------------------------------------------------
   GpiSetColor(hpsMem,CLR_WHITE);
   GpiSetBackColor(hpsMem,CLR_BLACK);

   GpiWCBitBlt(hpsMem,
               (*phsSprite)->hbmBitmap,
               4,
               aptlPoints,
               ROP_SRCCOPY,
               BBO_IGNORE);
   GpiSetBitmap(hpsMem,NULLHANDLE);
   GpiDestroyPS(hpsMem);
   DevCloseDC(hdcMem);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprDestroySprite(HCSSPRITE hsSprite)
//-------------------------------------------------------------------------
// This function destroys the sprite and returns all resources to the
// system.
//
// Input:  hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay!=NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASPLAYGROUND;
   } /* endif */

   GpiDeleteBitmap(hsSprite->hbmBitmap);
   GpiDeleteBitmap(hsSprite->hbmMask);

   accessSem((PCMNHANDLE)hsSprite,usSemAction);

   DosCloseMutexSem(hsSprite->hsmAccess);
   DosFreeMem(hsSprite);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprSetSpritePosition(HPS hpsDraw,
                                          HCSSPRITE hsSprite,
                                          PPOINTL pptlNew)
//-------------------------------------------------------------------------
// This function changes the position of the sprite.  This function is
// optimized so that, if the rectangle bounding the sprite at the new
// position overlaps the old, only one "bit blit" to the specified HPS
// is done, eliminating flicker.
//
// Input:  hpsDraw - handle to the HPS to draw the sprite in once it is
//                   moved
//         hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;
   SIZEL szlPlay;
   SIZEL szlWork;
   RECTL rclOld;
   RECTL rclNew;
   RECTL rclUnion;
   RECTL rclSrc;
   RECTL rclDest;
   ULONG ulIndex;
   HCSSPRITE hsLayer;
   RECTL rclLayer;
   RECTL rclInter;
   POINTL ptlWork;
   POINTL aptlPoints[4];

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   if ((hsSprite->bVisible) && (hsSprite->hpgPlay->bUpdate)) {
      szlWork.cx=MAX_SPRITE_CX*2;
      szlWork.cy=MAX_SPRITE_CY*2;

      CmnSprQueryPlaygroundSize(hsSprite->hpgPlay,&szlPlay);

      CmnSprQuerySpriteRect(hsSprite,&rclOld);
      hsSprite->ptlPos=*pptlNew;
      CmnSprQuerySpriteRect(hsSprite,&rclNew);

      WinUnionRect(hsSprite->habAnchor,&rclUnion,&rclOld,&rclNew);

      if ((rclUnion.xRight-rclUnion.xLeft>MAX_SPRITE_CX*2) ||
          (rclUnion.yTop-rclUnion.yBottom>MAX_SPRITE_CY*2)) {

         rclSrc.xLeft=rclOld.xLeft;
         rclSrc.yBottom=rclOld.yBottom;
         rclSrc.xRight=rclSrc.xLeft+hsSprite->bmihBitmap.cx;
         rclSrc.yTop=rclSrc.yBottom+hsSprite->bmihBitmap.cy;

         drawBackAt(hpsDraw,hsSprite->hpgPlay,NULL,NULL,&rclSrc);

         CmnSprDrawSprite(hpsDraw,hsSprite);
      } else {
         rclSrc=rclUnion;
         rclSrc.xRight++;
         rclSrc.yTop++;

         rclDest.xLeft=0;
         rclDest.yBottom=0;
         rclDest.xRight=rclUnion.xRight-rclUnion.xLeft;
         rclDest.yTop=rclUnion.yTop-rclUnion.yBottom;

         drawBackAt(hsSprite->hpgPlay->hpsWork,
                    hsSprite->hpgPlay,
                    &rclDest,
                    &szlWork,
                    &rclSrc);

         for (ulIndex=0; ulIndex<hsSprite->hpgPlay->ulNumMembers; ulIndex++) {
            hsLayer=hsSprite->hpgPlay->ahsSprites[ulIndex];

            CmnSprQuerySpriteRect(hsLayer,&rclLayer);

            WinIntersectRect(hsSprite->hpgPlay->habAnchor,
                             &rclInter,
                             &rclUnion,
                             &rclLayer);

            if (!WinIsRectEmpty(hsSprite->hpgPlay->habAnchor,&rclInter)) {
               ptlWork.x=hsLayer->ptlPos.x-rclUnion.xLeft;
               ptlWork.y=hsLayer->ptlPos.y-rclUnion.yBottom;

               rclInter.xLeft-=rclLayer.xLeft;
               rclInter.yBottom-=rclLayer.yBottom;
               rclInter.xRight-=rclLayer.xLeft;
               rclInter.yTop-=rclLayer.yBottom;

               rclInter.xRight++;
               rclInter.yTop++;

               drawSpriteAt(hsSprite->hpgPlay->hpsWork,
                            hsLayer,
                            &szlWork,
                            &ptlWork,
                            &rclInter);
            } /* endif */
         } /* endfor */

         //----------------------------------------------------------------
         // GpiBitBlt is non-inclusive on source AND target
         //----------------------------------------------------------------
         aptlPoints[0].x=rclUnion.xLeft;
         aptlPoints[0].y=rclUnion.yBottom;
         aptlPoints[1].x=rclUnion.xRight+1;
         aptlPoints[1].y=rclUnion.yTop+1;
         aptlPoints[2].x=0;
         aptlPoints[2].y=0;
         aptlPoints[3].x=rclUnion.xRight-rclUnion.xLeft+1;
         aptlPoints[3].y=rclUnion.yTop-rclUnion.yBottom+1;

         if (clipBltPoints(hsSprite->habAnchor,aptlPoints,&szlPlay)) {
            GpiBitBlt(hpsDraw,
                      hsSprite->hpgPlay->hpsWork,
                      4,
                      aptlPoints,
                      ROP_SRCCOPY,
                      BBO_IGNORE);
         } /* endif */
      } /* endif */
   } else {
      hsSprite->ptlPos=*pptlNew;
   } /* endif */

   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQuerySpritePosition(HCSSPRITE hsSprite,
                                            PPOINTL pptlPos)
//-------------------------------------------------------------------------
// This function returns the current position of the sprite.  Note that
// a sprite has a current position even if it is hidden.
//
// Input:  hsSprite - handle to the sprite
// Output:  pptlPos - points to the current position
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   *pptlPos=hsSprite->ptlPos;
   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQuerySpriteSize(HCSSPRITE hsSprite,PSIZEL pszlSize)
//-------------------------------------------------------------------------
// This function returns the current size of the sprite.
//
// Input:  hsSprite - handle to the sprite
// Output:  pszlSize - points to the current size
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);
   pszlSize->cx=hsSprite->bmihBitmap.cx;
   pszlSize->cy=hsSprite->bmihBitmap.cy;
   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQuerySpriteRect(HCSSPRITE hsSprite,PRECTL prclRect)
//-------------------------------------------------------------------------
// This function returns the bounding rectangle of the sprite at its
// current position.
//
// Input:  hsSprite - handle to the sprite
// Output:  prclRect - points to the current bounding rectangle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   prclRect->xLeft=hsSprite->ptlPos.x;
   prclRect->yBottom=hsSprite->ptlPos.y;
   prclRect->xRight=prclRect->xLeft+hsSprite->bmihBitmap.cx-1;
   prclRect->yTop=prclRect->yBottom+hsSprite->bmihBitmap.cy-1;

   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprSetSpriteVisibility(HPS hpsDraw,
                                            HCSSPRITE hsSprite,
                                            BOOL bVisible)
//-------------------------------------------------------------------------
// This function shows or hides a sprite.
//
// Input:  hpsDraw - handle to the HPS to draw in once the sprite is
//                   shown or hidden
//         hsSprite - handle to the sprite
//         bVisible - new visibility state
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;
   RECTL rclSprite;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   if (hsSprite->bVisible!=bVisible) {
      hsSprite->bVisible=bVisible;

      if (hsSprite->hpgPlay->bUpdate) {
         if (hsSprite->bVisible) {
            CmnSprDrawSprite(hpsDraw,hsSprite);
         } else {
            CmnSprQuerySpriteRect(hsSprite,&rclSprite);
            rclSprite.xRight++;
            rclSprite.yTop++;

            drawBackAt(hpsDraw,hsSprite->hpgPlay,NULL,NULL,&rclSprite);
         } /* endif */
      } /* endif */
   } /* endif */

   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQuerySpriteVisibility(HCSSPRITE hsSprite,
                                              PBOOL pbVisible)
//-------------------------------------------------------------------------
// This function returns the visibility state of the sprite
//
// Input:  hsSprite - handle to the sprite
// Output:  pbVisible - points to the visibility state
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   *pbVisible=hsSprite->bVisible;
   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprDrawSprite(HPS hpsDraw,HCSSPRITE hsSprite)
//-------------------------------------------------------------------------
// This function draws a sprite
//
// Input:  hpsDraw - handle to the HPS to draw the sprite in
//         hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;
   RECTL rclSprite;
   ULONG ulIndex;
   HCSSPRITE hsLayer;
   RECTL rclLayer;
   RECTL rclInter;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   if ((!hsSprite->bVisible) || (!hsSprite->hpgPlay->bUpdate)) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_NOERROR;
   } /* endif */

   CmnSprQuerySpriteRect(hsSprite,&rclSprite);

   rclSprite.xRight++;
   rclSprite.yTop++;

   drawBackAt(hpsDraw,hsSprite->hpgPlay,NULL,NULL,&rclSprite);

   rclSprite.xRight--;
   rclSprite.yTop--;

   for (ulIndex=0; ulIndex<hsSprite->hpgPlay->ulNumMembers; ulIndex++) {
      hsLayer=hsSprite->hpgPlay->ahsSprites[ulIndex];

      CmnSprQuerySpriteRect(hsLayer,&rclLayer);

      WinIntersectRect(hsSprite->hpgPlay->habAnchor,
                       &rclInter,
                       &rclSprite,
                       &rclLayer);

      if (!WinIsRectEmpty(hsSprite->hpgPlay->habAnchor,&rclInter)) {
         rclInter.xLeft-=rclLayer.xLeft;
         rclInter.yBottom-=rclLayer.yBottom;
         rclInter.xRight-=rclLayer.xLeft;
         rclInter.yTop-=rclLayer.yBottom;

         rclInter.xRight++;
         rclInter.yTop++;

         drawSpriteAt(hpsDraw,hsLayer,NULL,NULL,&rclInter);
      } /* endif */
   } /* endfor */

   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprSetLayering(HPS hpsDraw,HCSSPRITE hsSprite,LONG lPos)
//-------------------------------------------------------------------------
// This function sets the z-order of a sprite.
//
// Input:  hpsDraw - handle to the HPS to make the updates in.
//         hsSprite - handle to the sprite whose z-order is to be changed
//         lPos - an absolute layer number (0 to MAX_SPRITES-1) or an
//                SSL_* constant.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;
   SPRERROR seError;
   LONG lOld;
   LONG lNew;
   LONG lOldDx;
   LONG lNewDx;
   HCSSPRITE ahsCopy[MAX_SPRITES];
   ULONG ulIndex;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   seError=CmnSprQueryLayering(hsSprite,&lOld);
   if (seError!=SPR_ERR_NOERROR) {
      return seError;
   } /* endif */

   lNew=lOld;

   switch (lPos) {
   case SSL_TOP:
      lNew=hsSprite->hpgPlay->ulNumMembers-1;
      break;
   case SSL_BOTTOM:
      lNew=0;
      break;
   case SSL_UP:
      lNew=lOld+1;
      break;
   case SSL_DOWN:
      lNew=lOld-1;
      break;
   default:
      lNew=lPos;
   } /* endswitch */

   if ((lNew<0) || (lNew>hsSprite->hpgPlay->ulNumMembers-1)) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_BADLAYER;
   } /* endif */

   if ((lNew==lOld) || (hsSprite->hpgPlay->ulNumMembers==1)) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_NOERROR;
   } /* endif */

   memset(ahsCopy,0,sizeof(ahsCopy));
   ahsCopy[lNew]=hsSprite->hpgPlay->ahsSprites[lOld];

   lOldDx=0;
   lNewDx=0;

   for (ulIndex=0; ulIndex<hsSprite->hpgPlay->ulNumMembers-1; ulIndex++) {
      if (ulIndex==lOld) {
         lOldDx=1;
      } else
      if (ulIndex==lNew) {
         lNewDx=1;
      } /* endif */

      ahsCopy[ulIndex+lNewDx]=hsSprite->hpgPlay->ahsSprites[ulIndex+lOldDx];
   } /* endfor */

   memcpy(hsSprite->hpgPlay->ahsSprites,ahsCopy,sizeof(ahsCopy));

   CmnSprDrawSprite(hpsDraw,hsSprite);

   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY CmnSprQueryLayering(HCSSPRITE hsSprite,PLONG plPos)
//-------------------------------------------------------------------------
// This function sets the z-order of a sprite.
//
// Input:  hsSprite - handle to the sprite whose z-order is to be queried
//         plPos - points to the variable to receive the layer number.
// Output:  plPos - points to the layer number (0 to MAX_SPRITES-1).
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usSemAction;
   ULONG ulIndex;

   if (queryHandle(hsSprite)!=QH_HCSSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usSemAction=accessSem((PCMNHANDLE)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PCMNHANDLE)hsSprite,usSemAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   for (ulIndex=0; ulIndex<hsSprite->hpgPlay->ulNumMembers; ulIndex++) {
      if (hsSprite->hpgPlay->ahsSprites[ulIndex]==hsSprite) {
         *plPos=ulIndex;
         return SPR_ERR_NOERROR;
      } /* endif */
   } /* endfor */

   accessSem((PCMNHANDLE)hsSprite,usSemAction);
   return SPR_ERR_NOTFOUND;
}
