/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
/* Except for image_save_pcx, see copyright information below. */
#include <stdio.h>
#include <stdlib.h>
#include <jlib.h>

typedef struct PCX_HEADER {
	UBYTE manufacturer;
	UBYTE version;
	UBYTE encoding;
	UBYTE bits_per_pixel;
	SHORT xmin, ymin;
	SHORT xmax, ymax;
	SHORT hres;
	SHORT vres;
	UBYTE palette16[48];
	UBYTE reserved;
	UBYTE color_planes;
	SHORT bytes_per_line;
	SHORT palette_type;
	UBYTE filler[58];
} pcx_header;


/*+------------------------------------------------------------------------+ */
/*|Load a .PCX image file from a file name.                                | */
/*+------------------------------------------------------------------------+ */
image *image_load_pcx(char *filename)
{
	FILE *in;
	image *foo;

	JLIB_ENTER("image_load_pcx");

	if ((in = fopen(filename, "rb")) == NULL) {
		JLIB_SPRINTF("%s File Open Error.", filename);
		JLIB_LEAVE;
		return (image *) NULL;
	}

	foo = image_load_pcx_fp(in);

	fclose(in);

	JLIB_LEAVE;
	return foo;
}


/*+------------------------------------------------------------------------+ */
/*|Load a .PCX image file from a file pointer.                             | */
/*+------------------------------------------------------------------------+ */
image *image_load_pcx_fp(FILE * fp)
{
	image *img;
	int i, c;
	pcx_header pcxhead;

	JLIB_ENTER("image_load_pcx_fp");

	if (fp == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));

	/* Allocate header record */
	if ((img = (image *) malloc(sizeof(image))) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* Allocate palette */
	IMG_PALETTE(img) = pal_init();

	jio_read_elementary_type(fp, &pcxhead.manufacturer, sizeof(UBYTE));
	jio_read_elementary_type(fp, &pcxhead.version, sizeof(UBYTE));
	jio_read_elementary_type(fp, &pcxhead.encoding, sizeof(UBYTE));
	jio_read_elementary_type(fp, &pcxhead.bits_per_pixel, sizeof(UBYTE));

	jio_read_elementary_type(fp, &pcxhead.xmin, sizeof(SHORT));
	jio_read_elementary_type(fp, &pcxhead.ymin, sizeof(SHORT));
	jio_read_elementary_type(fp, &pcxhead.xmax, sizeof(SHORT));
	jio_read_elementary_type(fp, &pcxhead.ymax, sizeof(SHORT));
	jio_read_elementary_type(fp, &pcxhead.hres, sizeof(SHORT));
	jio_read_elementary_type(fp, &pcxhead.vres, sizeof(SHORT));

	for (i = 0; i < 48; i++)
		jio_read_elementary_type(fp, &pcxhead.palette16[i], sizeof(UBYTE));

	jio_read_elementary_type(fp, &pcxhead.reserved, sizeof(UBYTE));
	jio_read_elementary_type(fp, &pcxhead.color_planes, sizeof(UBYTE));
	jio_read_elementary_type(fp, &pcxhead.bytes_per_line, sizeof(SHORT));
	jio_read_elementary_type(fp, &pcxhead.palette_type, sizeof(SHORT));

	for (i = 0; i < 58; i++)
		jio_read_elementary_type(fp, &pcxhead.filler[i], sizeof(UBYTE));

	/* set width and height */
	IMG_WIDTH(img) = (int) (pcxhead.xmax - pcxhead.xmin) + 1;
	IMG_HEIGHT(img) = (int) (pcxhead.ymax - pcxhead.ymin) + 1;

	JLIB_SPRINTF("PCX Width: %d.", IMG_WIDTH(img));
	JLIB_SPRINTF("PCX Height: %d.", IMG_HEIGHT(img));

	image_setup(img);

	/* get the RLE data into the buffer */
	for (i = 0; i < IMG_SIZE(img);) {
		unsigned int x;

		c = fgetc(fp) & 0xff;
		if ((c & 0xc0) == 0xc0) {
			x = c & 0x3f;
			c = fgetc(fp);

			while (x--) {
				img->data[i] = (UBYTE) (c & 0xff);
				i++;
			}
		}
		else {
			img->data[i] = (UBYTE) (c & 0xff);
			i++;
		}
	}

	/* Get the check marker for 256-colour files */
	/*  fseek (fp, -769L, 2); */
	c = fgetc(fp);

	if (c != 0x0C) {
		JLIB_PRINT_DEBUG_INFO("Checksum incorrect loading pcx file.");
		free(IMG_DATA_PTR(img));
		free(IMG_PALETTE(img));
		free(img);

		JLIB_LEAVE;
		return (image *) NULL;
	}

	/* Read in the 256 colour palette */
	if ((c = fread(IMG_PALETTE(img), 1, 768, fp)) != 768) {
		JLIB_PRINT_DEBUG_INFO("Palette read failed");
		free(IMG_DATA_PTR(img));
		free(IMG_PALETTE(img));
		free(img);

		JLIB_LEAVE;
		return (image *) NULL;
	}

	JLIB_LEAVE;
	return img;
}


/* Save a JLib buffer to a pcx file
 * (c) by Lennart Steinke
 *
 * This is "mailware": if you're using this stuff, send me a postcard, or email me.
 * Lennart Steinke Hofener Str. 16 74366 Kirchheim/N. GERMANY
*/

static UBYTE create_rle(UBYTE * data, UBYTE max, UBYTE * rle, UBYTE * value);

/*+------------------------------------------------------------------------+ */
/*|Save a buffer to a .PCX image file.                                     | */
/*+------------------------------------------------------------------------+ */
int image_save_pcx(char *filename, buffer_rec * buff, UBYTE * pal)
{
	FILE *f;
	int retval;

	JLIB_ENTER("image_save_pcx");

	/* Open output file */
	f = fopen(filename, "wb");

	if (f == NULL) {
		JLIB_SPRINTF("%s File Open Error.", filename);
		JLIB_LEAVE;
		return 0;
	}
	
	retval = image_save_pcx_fp(f, buff, pal);

	fclose(f);

	JLIB_LEAVE;
	return retval;
}


/*+------------------------------------------------------------------------+ */
/*|Save a buffer as a .PCX to a file pointer.                              | */
/*+------------------------------------------------------------------------+ */
int image_save_pcx_fp(FILE *f, buffer_rec * buff, UBYTE * pal)
{
	pcx_header header;
	int x, y, i;
	UBYTE *data, rle, value, amount;
	USHORT c = 0, d = 0;

	JLIB_ENTER("image_save_pcx_fp");

	/* Simple check */
	if ((buff == NULL) || (pal == NULL) || (f == NULL)) {
		JLIB_PRINT_DEBUG_INFO("PCX write failed with NULL parameter(s)");
		JLIB_LEAVE;
		return 0;
	}

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_palette(pal);
#endif

	/* Fill in header info, then write header */
	header.manufacturer = 10;
	header.version = 5;
	header.encoding = 1;
	header.bits_per_pixel = 8;
	header.xmin = 0;
	header.ymin = 0;
	header.xmax = (SHORT) B_MAX_X(buff);
	header.ymax = (SHORT) B_MAX_Y(buff);
	header.hres = (SHORT) B_X_SIZE(buff);
	header.vres = (SHORT) B_Y_SIZE(buff);
	header.reserved = 0;
	header.color_planes = 1;
	header.bytes_per_line = (SHORT) B_X_SIZE(buff);
	header.palette_type = 1;

	jio_write_elementary_type(f, &header.manufacturer, sizeof(UBYTE));
	jio_write_elementary_type(f, &header.version, sizeof(UBYTE));
	jio_write_elementary_type(f, &header.encoding, sizeof(UBYTE));
	jio_write_elementary_type(f, &header.bits_per_pixel, sizeof(UBYTE));

	jio_write_elementary_type(f, &header.xmin, sizeof(SHORT));
	jio_write_elementary_type(f, &header.ymin, sizeof(SHORT));
	jio_write_elementary_type(f, &header.xmax, sizeof(SHORT));
	jio_write_elementary_type(f, &header.ymax, sizeof(SHORT));
	jio_write_elementary_type(f, &header.hres, sizeof(SHORT));
	jio_write_elementary_type(f, &header.vres, sizeof(SHORT));

	for (i = 0; i < 48; i++) 
		jio_write_elementary_type(f, &header.palette16[i], sizeof(UBYTE));

	jio_write_elementary_type(f, &header.reserved, sizeof(UBYTE));
	jio_write_elementary_type(f, &header.color_planes, sizeof(UBYTE));
	jio_write_elementary_type(f, &header.bytes_per_line, sizeof(SHORT));
	jio_write_elementary_type(f, &header.palette_type, sizeof(SHORT));

	for (i = 0; i < 58; i++)
		jio_write_elementary_type(f, &header.filler[i], sizeof(UBYTE));

	/* Write data */
	data = B_BUFF_PTR(buff);

	for (y = 0; y < header.vres; y++) {
		for (x = 0; x < header.hres;) {
			if ((x + 63) < header.hres)
				amount = create_rle(data, 63, &rle, &value);
			else
				amount = create_rle(data, (UBYTE) (header.hres - x), &rle, &value);

			data += amount;
			x += amount;

			if (rle) {
				jio_write_elementary_type(f, &rle, sizeof(UBYTE));
				d++;
			}
			jio_write_elementary_type(f, &value, sizeof(UBYTE));
			c++;
		}

	}


	/* Write the palette */
	value = 0x0c;		/* Palette recognition byte */
	jio_write_elementary_type(f, &value, sizeof(UBYTE));

	for (i = 0; i < 768; i++)
		jio_write_elementary_type(f, &pal[i], sizeof(UBYTE));

	JLIB_LEAVE;
	return 1;
}


static UBYTE create_rle(UBYTE * data, UBYTE max, UBYTE * rle, UBYTE * value)
{
	UBYTE last = 0, amount = 0;

	JLIB_ENTER("create_rle");

	last = *data;
	data++;
	amount++;

	while ((last == (*data)) && (amount < max)) {
		data++;
		amount++;
	}

	if (amount == 1) {
		if (last < 192)
			*rle = 0;
		else
			*rle = 193;	/* RLE run length 1 */
	}
	else
		*rle = (UBYTE) (192 + amount);

	*value = last;

	JLIB_LEAVE;
	return amount;
}