/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details.  */
#include <jlib.h>


/*+------------------------------------------------------------------------+ */
/*|Draw a sprite in a buffer without clipping                              | */
/*+------------------------------------------------------------------------+ */
void buff_draw_spriteNC(sprite_system * s, int snum, buffer_rec * buff)
{
	int frame, bwidth, height;
	UBYTE *pattern, *data, *out;

	JLIB_ENTER("buff_draw_spriteNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_sprite(s, snum);
#endif

	frame = s->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
	jlib_check_frame(s, frame);
#endif

	height = s->sprite_data[frame]->height;
	pattern = s->sprite_data[frame]->pattern;
	data = s->sprite_data[frame]->data;
	bwidth = B_X_SIZE(buff) - s->sprite_data[frame]->width;
	out = B_OFFSET(buff, s->sprites[snum]->y) + s->sprites[snum]->x;

	while (height--) {
		int width = *pattern++;
		while (width--) {
			UBYTE datum = *pattern++;
			SPR_STENCIL_COPY(out, data, datum);
			out += (datum & 15);
			data += (datum & 15);
		}
		out += bwidth;
	}

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Draw a sprite in a buffer with clipping                               | */
/*+----------------------------------------------------------------------+ */
void buff_draw_sprite(sprite_system * s, int snum, buffer_rec * buff)
{
	int width, height, frame, x, y, lskip, rskip, bwidth, clip = 0;
	UBYTE *pattern, *data, *out;

	JLIB_ENTER("buff_draw_sprite");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_sprite(s, snum);
#endif

	frame = s->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
	jlib_check_frame(s, frame);
#endif

	x = s->sprites[snum]->x;
	y = s->sprites[snum]->y;
	width = s->sprite_data[frame]->width;
	height = s->sprite_data[frame]->height;

	/* Clip X */
	if (x + width > B_CX2(buff)) {
		rskip = x + width - B_CX2(buff);
		clip = C_RIGHT;
	}
	else {
#ifndef JLIB_PRODUCTION
		rskip = 0;			/* stop gcc warning */
#endif
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}
	}

	if (x < B_CX1(buff)) {
		lskip = B_CX1(buff) - x;
		x = B_CX1(buff);
		clip |= C_LEFT;
	}
	else {
#ifndef JLIB_PRODUCTION
		lskip = 0;			/* stop gcc warning */
#endif
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}
	}

	pattern = s->sprite_data[frame]->pattern;
	data = s->sprite_data[frame]->data;

	/* Clip Y */
	if (y + height > B_CY2(buff))
		height = B_CY2(buff) - y + 1;
	else
		if (y + height < B_CY1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (y < B_CY1(buff)) {
		int remain = B_CY1(buff) - y;
		height -= remain;

		/* skip RLE pattern */
		do {
			pattern += *pattern + 1;
			data += width;
		} while (--remain);

		y = B_CY1(buff);
	}
	else
		if (y > B_CY2(buff)) {
			JLIB_LEAVE;
			return;
		}

	out = B_OFFSET(buff, y) + x;
	bwidth = B_X_SIZE(buff) - width;

	if (clip & C_LEFT) {
		data += lskip;
		bwidth += lskip;
		width = width - lskip;
	}
	if (clip & C_RIGHT) {
		bwidth += rskip;
		width = width - rskip;
	}

	switch (clip & 0x3) {
	case 0:				/* no x clipping */
		while (height--) {
			int w = *pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				SPR_STENCIL_COPY(out, data, datum);
				out += (datum & 15);
				data += (datum & 15);
			}
			out += bwidth;
		}
		break;
	case C_LEFT:			/* left only */
		while (height--) {
			int to_skip = lskip;
			int w = *pattern;
			while (to_skip > 0) {
				pattern++;
				to_skip -= (*pattern & 15);
				w--;
			}
			if (*pattern & 16) {
				to_skip = -to_skip;
				MEM_COPY_SHORT(out,data,to_skip)
				out += to_skip;
				data += to_skip;
			}
			else {
				out -= to_skip;
				data -= to_skip;
			}
			pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				SPR_STENCIL_COPY(out, data, datum);
				out += (datum & 15);
				data += (datum & 15);
			}
			out += bwidth;
			data += lskip;
		}
		break;
	case C_RIGHT:			/* right only */
		while (height--) {
			int drawn = width;
			int w = *pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				drawn -= datum & 15;
				if (drawn < 0) {
					drawn += (datum & 15);
					if (datum & 16)
						MEM_COPY_SHORT(out,data,(drawn + 1));
					out += drawn;
					data += drawn;
					pattern += w;
					w = 0;
				}
				else {
					SPR_STENCIL_COPY(out, data, datum);
					out += (datum & 15);
					data += (datum & 15);
				}
			}
			out += bwidth;
			data += rskip;
		}
		break;
	default:			/* left and right */
		while (height--) {
			int to_skip = lskip;
			int drawn = width;
			int w = *pattern;
			while (to_skip > 0) {
				pattern++;
				to_skip -= (*pattern & 15);
				w--;
			}
			to_skip = -to_skip;
			if (to_skip < drawn) {
				if (*pattern & 16)
					MEM_COPY_SHORT(out,data,to_skip)
				out += to_skip;
				data += to_skip;
				drawn -= to_skip;
				pattern++;
				while (w--) {
					UBYTE datum = *pattern++;
					drawn -= datum & 15;
					if (drawn < 0) {
						drawn += (datum & 15);
						if (datum & 16)
							MEM_COPY_SHORT(out,data,(drawn + 1));
						out += drawn;
						data += drawn;
						pattern += w;
						w = 0;
					}
					else {
						SPR_STENCIL_COPY(out, data, datum);
						out += (datum & 15);
						data += (datum & 15);
					}
				}
			}
			else {
				if (*pattern & 16)
					MEM_COPY_SHORT(out,data,drawn);
				out += drawn;
				data += drawn;
				pattern = pattern + w + 1;
			}
			out += bwidth;
			data = data + lskip + rskip;
		}
		break;
	}
	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Save the overwritten buffer under a sprite without clipping.           | */
/*+-----------------------------------------------------------------------+ */
void buff_save_spriteNC(sprite_system * s, int snum, buffer_rec * buff)
{
	int frame, bwidth, height;
	UBYTE *pattern, *buffer, *out;

	JLIB_ENTER("buff_save_spriteNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_sprite(s, snum);
#endif

	frame = s->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
	jlib_check_frame(s, frame);
#endif

	height = s->sprite_data[frame]->height;
	pattern = s->sprite_data[frame]->pattern;
	bwidth = B_X_SIZE(buff) - s->sprite_data[frame]->width;
	out = B_OFFSET(buff, s->sprites[snum]->y) + s->sprites[snum]->x;
	buffer = s->sprites[snum]->buffer;

	while (height--) {
		int width = *pattern++;
		while (width--) {
			UBYTE datum = *pattern++;
			SPR_STENCIL_COPY(buffer, out, datum);
			out += (datum & 15);
			if (datum & 16)
				buffer += (datum & 15);
		}
		out += bwidth;
	}

	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Save the overwritten buffer under a sprite with clipping               | */
/*+-----------------------------------------------------------------------+ */
void buff_save_sprite(sprite_system * s, int snum, buffer_rec * buff)
{
	int width, height, frame, x, y, lskip, rskip, bwidth, clip = 0;
	UBYTE *pattern, *buffer, *out;

	JLIB_ENTER("buff_save_sprite");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_sprite(s, snum);
#endif

	frame = s->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
	jlib_check_frame(s, frame);
#endif

	x = s->sprites[snum]->x;
	y = s->sprites[snum]->y;
	width = s->sprite_data[frame]->width;
	height = s->sprite_data[frame]->height;

	/* Clip X */
	if (x + width > B_CX2(buff)) {
		rskip = x + width - B_CX2(buff);
		clip = C_RIGHT;
	}
	else {
#ifndef JLIB_PRODUCTION
		rskip = 0;			/* stop gcc warning */
#endif
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}
	}

	if (x < B_CX1(buff)) {
		lskip = B_CX1(buff) - x;
		x = B_CX1(buff);
		clip |= C_LEFT;
	}
	else {
#ifndef JLIB_PRODUCTION
		lskip = 0;			/* stop gcc warning */
#endif
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}
	}

	pattern = s->sprite_data[frame]->pattern;
	buffer = s->sprites[snum]->buffer;

	/* Clip Y */
	if (y + height > B_CY2(buff))
		height = B_CY2(buff) - y + 1;
	else
		if (y + height < B_CY1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (y < B_CY1(buff)) {
		int remain = B_CY1(buff) - y;
		height -= remain;

		/* skip RLE pattern */
		do {
			pattern = pattern + *pattern + 1;
		} while (--remain);

		y = B_CY1(buff);
	}
	else
		if (y > B_CY2(buff)) {
			JLIB_LEAVE;
			return;
		}

	out = B_OFFSET(buff, y) + x;
	bwidth = B_X_SIZE(buff) - width;

	if (clip & C_LEFT) {
		bwidth += lskip;
		width = width - lskip;
	}
	if (clip & C_RIGHT) {
		bwidth += rskip;
		width = width - rskip;
	}

	switch (clip & 0x3) {
	case 0:				/* no x clipping */
		while (height--) {
			int w = *pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				SPR_STENCIL_COPY(buffer, out, datum);
				out += (datum & 15);
				if (datum & 16)
					buffer += (datum & 15);
			}
			out += bwidth;
		}
		break;
	case C_LEFT:			/* left only */
		while (height--) {
			int to_skip = lskip;
			int w = *pattern;
			while (to_skip > 0) {
				pattern++;
				to_skip -= (*pattern & 15);
				w--;
			}
			if (*pattern & 16) {
				to_skip = -to_skip;
				MEM_COPY_SHORT(buffer,out,to_skip);
				out += to_skip;
				buffer += to_skip;
			}
			else {
				out -= to_skip;
			}
			pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				SPR_STENCIL_COPY(buffer, out, datum);
				out += (datum & 15);
				if (datum & 16)
					buffer += (datum & 15);
			}
			out += bwidth;
		}
		break;
	case C_RIGHT:			/* right only */
		while (height--) {
			int drawn = width;
			int w = *pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				drawn -= datum & 15;
				if (drawn < 0) {
					drawn += (datum & 15);
					if (datum & 16) {
						MEM_COPY_SHORT(buffer, out,(drawn + 1));
						buffer += drawn;
					}
					out += drawn;
					pattern += w;
					w = 0;
				}
				else {
					SPR_STENCIL_COPY(buffer, out, datum);
					if (datum & 16)
						buffer += (datum & 15);
					out += (datum & 15);
				}
			}
			out += bwidth;
		}
		break;
	default:			/* left and right */
		while (height--) {
			int to_skip = lskip;
			int drawn = width;
			int w = *pattern;
			while (to_skip > 0) {
				pattern++;
				to_skip -= (*pattern & 15);
				w--;
			}
			to_skip = -to_skip;
			if (to_skip < drawn) {
				if (*pattern & 16) {
					MEM_COPY_SHORT(buffer,out,to_skip);
					buffer += to_skip;
				}
				out += to_skip;
				drawn -= to_skip;
				pattern++;
				while (w--) {
					UBYTE datum = *pattern++;
					drawn -= datum & 15;
					if (drawn < 0) {
						drawn += (datum & 15);
						if (datum & 16) {
							MEM_COPY_SHORT(buffer,out,(drawn + 1));
							buffer += drawn;
						}
						out += drawn;
						pattern += w;
						w = 0;
					}
					else {
						SPR_STENCIL_COPY(buffer, out, datum);
						out += (datum & 15);
						if (datum & 16)
							buffer += (datum & 15);
					}
				}
			}
			else {
				if (*pattern & 16) {
					MEM_COPY_SHORT(buffer,out,drawn);
					buffer += drawn;
				}
				out += drawn;
				pattern = pattern + w + 1;
			}
			out += bwidth;
		}
		break;
	}
	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Restore the buffer under a sprite without clipping                     | */
/*+-----------------------------------------------------------------------+ */
void buff_rest_spriteNC(sprite_system * s, int snum, buffer_rec * buff)
{
	int frame, bwidth, height;
	UBYTE *pattern, *buffer, *out;

	JLIB_ENTER("buff_rest_spriteNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_sprite(s, snum);
#endif

	frame = s->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
	jlib_check_frame(s, frame);
#endif

	height = s->sprite_data[frame]->height;
	pattern = s->sprite_data[frame]->pattern;
	buffer = s->sprites[snum]->buffer;
	bwidth = B_X_SIZE(buff) - s->sprite_data[frame]->width;
	out = B_OFFSET(buff, s->sprites[snum]->y) + s->sprites[snum]->x;

	while (height--) {
		int width = *pattern++;
		while (width--) {
			UBYTE datum = *pattern++;
			SPR_STENCIL_COPY(out, buffer, datum);
			out += (datum & 15);
			if (datum & 16)
				buffer += (datum & 15);
		}
		out += bwidth;
	}

	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Restore the overwritten buffer under a sprite with clipping            | */
/*+-----------------------------------------------------------------------+ */
void buff_rest_sprite(sprite_system * s, int snum, buffer_rec * buff)
{
	int width, height, frame, x, y, lskip, rskip, bwidth, clip = 0;
	UBYTE *pattern, *buffer, *out;

	JLIB_ENTER("buff_rest_sprite");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_sprite(s, snum);
#endif

	frame = s->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
	jlib_check_frame(s, frame);
#endif

	x = s->sprites[snum]->x;
	y = s->sprites[snum]->y;
	width = s->sprite_data[frame]->width;
	height = s->sprite_data[frame]->height;

	/* Clip X */
	if (x + width > B_CX2(buff)) {
		rskip = x + width - B_CX2(buff);
		clip = C_RIGHT;
	}
	else {
#ifndef JLIB_PRODUCTION
		rskip = 0;			/* stop gcc warning */
#endif
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}
	}

	if (x < B_CX1(buff)) {
		lskip = B_CX1(buff) - x;
		x = B_CX1(buff);
		clip |= C_LEFT;
	}
	else {
#ifndef JLIB_PRODUCTION
		lskip = 0;			/* stop gcc warning */
#endif
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}
	}

	pattern = s->sprite_data[frame]->pattern;
	buffer = s->sprites[snum]->buffer;

	/* Clip Y */
	if (y + height > B_CY2(buff))
		height = B_CY2(buff) - y + 1;
	else
		if (y + height < B_CY1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (y < B_CY1(buff)) {
		int remain = B_CY1(buff) - y;
		height -= remain;

		/* skip RLE pattern */
		do {
			pattern += *pattern + 1;
		} while (--remain);

		y = B_CY1(buff);
	}
	else
		if (y > B_CY2(buff)) {
			JLIB_LEAVE;
			return;
		}

	out = B_OFFSET(buff, y) + x;
	bwidth = B_X_SIZE(buff) - width;

	if (clip & C_LEFT) {
		bwidth += lskip;
		width = width - lskip;
	}
	if (clip & C_RIGHT) {
		bwidth += rskip;
		width = width - rskip;
	}

	switch (clip & 0x3) {
	case 0:				/* no x clipping */
		while (height--) {
			int w = *pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				SPR_STENCIL_COPY(out, buffer, datum);
				out += (datum & 15);
				if (datum & 16)
					buffer += (datum & 15);
			}
			out += bwidth;
		}
		break;
	case C_LEFT:			/* left only */
		while (height--) {
			int to_skip = lskip;
			int w = *pattern;
			while (to_skip > 0) {
				pattern++;
				to_skip -= (*pattern & 15);
				w--;
			}
			if (*pattern & 16) {
				to_skip = -to_skip;
				MEM_COPY_SHORT(out, buffer, to_skip);
				out += to_skip;
				buffer += to_skip;
			}
			else {
				out -= to_skip;
			}
			pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				SPR_STENCIL_COPY(out, buffer, datum);
				out += (datum & 15);
				if (datum & 16)
					buffer += (datum & 15);
			}
			out += bwidth;
		}
		break;
	case C_RIGHT:			/* right only */
		while (height--) {
			int drawn = width;
			int w = *pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				drawn -= datum & 15;
				if (drawn < 0) {
					drawn += (datum & 15);
					if (datum & 16) {
						MEM_COPY_SHORT(out, buffer, (drawn + 1));
						buffer += drawn;
					}
					out += drawn;
					pattern += w;
					w = 0;
				}
				else {
					SPR_STENCIL_COPY(out, buffer, datum);
					if (datum & 16)
						buffer += (datum & 15);
					out += (datum & 15);
				}
			}
			out += bwidth;
		}
		break;
	default:			/* left and right */
		while (height--) {
			int to_skip = lskip;
			int drawn = width;
			int w = *pattern;
			while (to_skip > 0) {
				pattern++;
				to_skip -= (*pattern & 15);
				w--;
			}
			to_skip = -to_skip;
			if (to_skip < drawn) {
				if (*pattern & 16) {
					MEM_COPY_SHORT(out, buffer, to_skip);
					buffer += to_skip;
				}
				out += to_skip;
				drawn -= to_skip;
				pattern++;
				while (w--) {
					UBYTE datum = *pattern++;
					drawn -= datum & 15;
					if (drawn < 0) {
						drawn += (datum & 15);
						if (datum & 16) {
							MEM_COPY_SHORT(out, buffer, (drawn + 1));
							buffer += drawn;
						}
						out += drawn;
						pattern += w;
						w = 0;
					}
					else {
						SPR_STENCIL_COPY(out, buffer, datum);
						out += (datum & 15);
						if (datum & 16)
							buffer += (datum & 15);
					}
				}
			}
			else {
				if (*pattern & 16) {
					MEM_COPY_SHORT(out, buffer, drawn);
					buffer += drawn;
				}
				out += drawn;
				pattern = pattern + w + 1;
			}
			out += bwidth;
		}
		break;
	}
	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Stamp a sprite frame on buff without clipping                          | */
/*+-----------------------------------------------------------------------+ */
void buff_stamp_spriteNC(sprite_system * s, int frame, buffer_rec * buff, int x, int y)
{
	int height, width;
	UBYTE *data, *out;

	JLIB_ENTER("buff_stamp_spriteNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_frame(s,frame);
#endif

	height = s->sprite_data[frame]->height;
	width = s->sprite_data[frame]->width;
	data = s->sprite_data[frame]->data;
	out = B_OFFSET(buff, y) + x;
	
	if (width <= 16)
		while (height--) {
			MEM_COPY_SHORT(out,data,width);
			out += B_X_SIZE(buff);
			data += width;
		}
	else
		while (height--) {
			MEM_COPY_LONG(out,data,width);
			out += B_X_SIZE(buff);
			data += width;
		}
		
	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Stamp a sprite frame on buff with clipping                             | */
/*+-----------------------------------------------------------------------+ */
void buff_stamp_sprite(sprite_system * s, int frame, buffer_rec * buff, int x, int y)
{
	int width, height, cwidth ;
	UBYTE *data, *out;

	JLIB_ENTER("buff_stamp_sprite");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_frame(s, frame);
#endif

	cwidth = width = s->sprite_data[frame]->width;
	height = s->sprite_data[frame]->height;
	data = s->sprite_data[frame]->data;
	out = B_OFFSET(buff, y) + x;

	/* Clip X */
	if (x + width > B_CX2(buff)) {
		int cut = B_CX2(buff) - x - width + 1;
		width += cut;
	}
	else
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (x < B_CX1(buff)) {
		int cut = B_CX1(buff) - x;
		width -= cut;
		data += cut;
		out += cut;
	}
	else
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}


	/* Clip Y */
	if (y + height > B_CY2(buff))
		height = B_CY2(buff) - y + 1;
	else
		if (y + height < B_CY1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (y < B_CY1(buff)) {
		int remain = B_CY1(buff) - y;
		height -= remain;
		out += (B_OFFSET(buff,remain) - B_BUFF_PTR(buff));
		data += remain * cwidth;
	}
	else
		if (y > B_CY2(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (width <= 16)
		while (height--) {
			MEM_COPY_SHORT(out,data,width);
			out += B_X_SIZE(buff);
			data += cwidth;
		}
	else
		while (height--) {
			MEM_COPY_LONG(out,data,width);
			out += B_X_SIZE(buff);
			data += cwidth;
		}

	JLIB_LEAVE;
}



/*+-----------------------------------------------------------------------+ */
/*|Stencil a sprite frame on buff without clipping                        | */
/*+-----------------------------------------------------------------------+ */
void buff_stencil_spriteNC(sprite_system * s, int frame, buffer_rec * buff, int x, int y)
{
	int bwidth, height;
	UBYTE *pattern, *data, *out;

	JLIB_ENTER("buff_stencil_spriteNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_frame(s, frame);
#endif

	height = s->sprite_data[frame]->height;
	pattern = s->sprite_data[frame]->pattern;
	data = s->sprite_data[frame]->data;
	bwidth = B_X_SIZE(buff) - s->sprite_data[frame]->width;
	out = B_OFFSET(buff, y) + x;

	while (height--) {
		int width = *pattern++;
		while (width--) {
			UBYTE datum = *pattern++;
			SPR_STENCIL_COPY(out, data, datum);
			out += (datum & 15);
			data += (datum & 15);
		}
		out += bwidth;
	}

	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Stencil a sprite frame on buff with clipping                           | */
/*+-----------------------------------------------------------------------+ */
void buff_stencil_sprite(sprite_system * s, int frame, buffer_rec * buff, int x, int y)
{
	int width, height, lskip, rskip, bwidth, clip = 0;
	UBYTE *pattern, *data, *out;

	JLIB_ENTER("buff_draw_sprite");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
	jlib_check_frame(s, frame);
#endif

	width = s->sprite_data[frame]->width;
	height = s->sprite_data[frame]->height;

	/* Clip X */
	if (x + width > B_CX2(buff)) {
		rskip = x + width - B_CX2(buff);
		clip = C_RIGHT;
	}
	else {
#ifndef JLIB_PRODUCTION
		rskip = 0;			/* stop gcc warning */
#endif
		if (x + width < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}
	}

	if (x < B_CX1(buff)) {
		lskip = B_CX1(buff) - x;
		x = B_CX1(buff);
		clip |= C_LEFT;
	}
	else {
#ifndef JLIB_PRODUCTION
		lskip = 0;			/* stop gcc warning */
#endif
		if ((x > B_CX2(buff))) {
			JLIB_LEAVE;
			return;
		}
	}

	pattern = s->sprite_data[frame]->pattern;
	data = s->sprite_data[frame]->data;

	/* Clip Y */
	if (y + height > B_CY2(buff))
		height = B_CY2(buff) - y + 1;
	else
		if (y + height < B_CY1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (y < B_CY1(buff)) {
		int remain = B_CY1(buff) - y;
		height -= remain;

		/* skip RLE pattern */
		do {
			pattern += *pattern + 1;
			data += width;
		} while (--remain);

		y = B_CY1(buff);
	}
	else
		if (y > B_CY2(buff)) {
			JLIB_LEAVE;
			return;
		}

	out = B_OFFSET(buff, y) + x;
	bwidth = B_X_SIZE(buff) - width;

	if (clip & C_LEFT) {
		data += lskip;
		bwidth += lskip;
		width = width - lskip;
	}
	if (clip & C_RIGHT) {
		bwidth += rskip;
		width = width - rskip;
	}

	switch (clip & 0x3) {
	case 0:				/* no x clipping */
		while (height--) {
			int w = *pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				SPR_STENCIL_COPY(out, data, datum);
				out += (datum & 15);
				data += (datum & 15);
			}
			out += bwidth;
		}
		break;
	case C_LEFT:			/* left only */
		while (height--) {
			int to_skip = lskip;
			int w = *pattern;
			while (to_skip > 0) {
				pattern++;
				to_skip -= (*pattern & 15);
				w--;
			}
			if (*pattern & 16) {
				to_skip = -to_skip;
				MEM_COPY_SHORT(out,data,to_skip)
				out += to_skip;
				data += to_skip;
			}
			else {
				out -= to_skip;
				data -= to_skip;
			}
			pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				SPR_STENCIL_COPY(out, data, datum);
				out += (datum & 15);
				data += (datum & 15);
			}
			out += bwidth;
			data += lskip;
		}
		break;
	case C_RIGHT:			/* right only */
		while (height--) {
			int drawn = width;
			int w = *pattern++;
			while (w--) {
				UBYTE datum = *pattern++;
				drawn -= datum & 15;
				if (drawn < 0) {
					drawn += (datum & 15);
					if (datum & 16)
						MEM_COPY_SHORT(out,data,(drawn + 1));
					out += drawn;
					data += drawn;
					pattern += w;
					w = 0;
				}
				else {
					SPR_STENCIL_COPY(out, data, datum);
					out += (datum & 15);
					data += (datum & 15);
				}
			}
			out += bwidth;
			data += rskip;
		}
		break;
	default:			/* left and right */
		while (height--) {
			int to_skip = lskip;
			int drawn = width;
			int w = *pattern;
			while (to_skip > 0) {
				pattern++;
				to_skip -= (*pattern & 15);
				w--;
			}
			to_skip = -to_skip;
			if (to_skip < drawn) {
				if (*pattern & 16)
					MEM_COPY_SHORT(out,data,to_skip)
				out += to_skip;
				data += to_skip;
				drawn -= to_skip;
				pattern++;
				while (w--) {
					UBYTE datum = *pattern++;
					drawn -= datum & 15;
					if (drawn < 0) {
						drawn += (datum & 15);
						if (datum & 16)
							MEM_COPY_SHORT(out,data,(drawn + 1));
						out += drawn;
						data += drawn;
						pattern += w;
						w = 0;
					}
					else {
						SPR_STENCIL_COPY(out, data, datum);
						out += (datum & 15);
						data += (datum & 15);
					}
				}
			}
			else {
				if (*pattern & 16)
					MEM_COPY_SHORT(out,data,drawn);
				out += drawn;
				data += drawn;
				pattern = pattern + w + 1;
			}
			out += bwidth;
			data = data + lskip + rskip;
		}
		break;
	}
	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Draw every active sprite                                               | */
/*+-----------------------------------------------------------------------+ */
JINLINE void buff_draw_all_sprites(sprite_system * s, buffer_rec * buff)
{
	int count;

	JLIB_ENTER("buff_draw_all_sprites");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
#endif

	for (count = 0; count < s->no_sprites; count++)
		if (SPR_IS_ON(s, count))
			buff_draw_sprite(s, count, buff);

	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Save every active sprite                                               | */
/*+-----------------------------------------------------------------------+ */
JINLINE void buff_save_all_sprites(sprite_system * s, buffer_rec * buff)
{
	int count;

	JLIB_ENTER("buff_save_all_sprites");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
#endif

	for (count = 0; count < s->no_sprites; count++)
		if (SPR_IS_ON(s, count))
			buff_save_sprite(s, count, buff);

	JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+ */
/*|Restore every active sprite                                            | */
/*+-----------------------------------------------------------------------+ */
JINLINE void buff_rest_all_sprites(sprite_system * s, buffer_rec * buff)
{
	int count;

	JLIB_ENTER("buff_rest_all_sprites");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_sprite_system(s);
#endif

	for (count = 0; count < s->no_sprites; count++)
		if (SPR_IS_ON(s, count))
			buff_rest_sprite(s, count, buff);

	JLIB_LEAVE;
}