/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <jlib.h>

void traceNC(int x1, int y1, int x2, int y2);
void trace(buffer_rec *buff,int x1, int y1, int x2, int y2, int *low, int *high);

/* Use Bresenham's line algorithm to trace an edge of the polygon. */
void traceNC(int x1, int y1, int x2, int y2)
{
	int dx, dy, xf, yf, a, b, t, i;

	if (x1 == x2 && y1 == y2)
		return;

	if (x2 > x1) {
		dx = x2 - x1;
		xf = 1;
	}
	else {
		dx = x1 - x2;
		xf = -1;
	}

	if (y2 > y1) {
		dy = y2 - y1;
		yf = 1;
	}
	else {
		dy = y1 - y2;
		yf = -1;
	}

	if (dx > dy) {
		a = dy + dy;
		t = a - dx;
		b = t - dx;
		for (i = 0; i <= dx; i++) {
			if (x1 < left[y1])
				left[y1] = x1;
			if (x1 > right[y1])
				right[y1] = x1;
			x1 += xf;
			if (t < 0)
				t += a;
			else {
				t += b;
				y1 += yf;
			}
		}
	}
	else {
		a = dx + dx;
		t = a - dy;
		b = t - dy;
		for (i = 0; i <= dy; i++) {
			if (x1 < left[y1])
				left[y1] = x1;
			if (x1 > right[y1])
				right[y1] = x1;
			y1 += yf;
			if (t < 0)
				t += a;
			else {
				t += b;
				x1 += xf;
			}
		}
	}
}


void trace(buffer_rec *buff,int x1, int y1, int x2, int y2, int *low, int *high)
{
	int dx, dy, xf, yf, a, b, t, i, delta;
	int max_x, min_x, max_y, min_y;
	
	if (x1 == x2 && y1 == y2)
		return;

	min_x = B_CX1(buff);
	max_x = B_CX2(buff);
	min_y = B_CY1(buff);
	max_y = B_CY2(buff);

	if (x1 > x2) {
		JLIB_SWAP(x1, x2);
		JLIB_SWAP(y1, y2);
	}

	if ((x2 < min_x) || (x1 > max_x)) {
		if (y1 > y2)
			JLIB_SWAP(y1, y2);

		if ((y2 < min_y) || (y1 > max_y))
			return;

		if (y1 < min_y)
			y1 = min_y;
			
		if (y2 > max_y)
			y2 = max_y;
			
		if (x2 < min_x)
			/* line is totally off lhs */
			do {
				if (left[y2] > min_x)
					left[y2] = min_x;
				y2--;
			} while (y2 >= y1);
		else
			/* line is totally off rhs */
			do {
				if (right[y2] < max_x)
					right[y2] = max_x;
				y2--;
			} while (y2 >= y1);
		return;
	}

	if (y1 < y2) {
		if ((y2 < min_y) || (y1 > max_y))
			return;

		if (y1 < min_y) {
			delta = (x2 - x1) * (min_y - y1) / (y2 - y1);
			y1 = min_y;
			if ((x1 += delta) > max_x) {
				/* line is totally off rhs */
				if (y2 > max_y)
					y2 = max_y;
				do {
					if (right[y1] < max_x)
						right[y1] = max_x;
					y1++;
				} while (y1 <= y2);
				return;
			}
		}
		if (y2 > max_y) {
			delta = (x2 - x1) * (y2 - max_y) / (y2 - y1);
			y2 = max_y;
			if ((x2 -= delta) < min_x) {
				/* line is totally off lhs */
				do {
					if (left[y2] > min_x)
						left[y2] = min_x;
					y2--;
				} while (y2 >= y1);
				return;
			}
		}
		if (x1 < min_x) {
			delta = (y2 - y1) * (min_x - x1) / (x2 - x1);

			while(delta--){
				if (left[y1] > min_x)
					left[y1] = min_x;
				y1++;
			}
			x1 = min_x;
		}
		if (x2 > max_x) {
			delta = (y2 - y1) * (x2 - max_x) / (x2 - x1);

			while(delta--){
				if (right[y2] < max_x)
					right[y2] = max_x;
				y2--;
			}
			x2 = max_x;
		}
	}
	else {
		if ((y1 < min_y) || (y2 > max_y))
			return;
		if (y2 < min_y) {
			delta = (x2 - x1) * (min_y - y2) / (y1 - y2);
			y2 = min_y;
			if ((x2 -= delta) < min_x) {
				if (y1 > max_y)
					y1 = max_y;
				/* line is totally off lhs */
				do {
					if (left[y1] > min_x)
						left[y1] = min_x;
					y1--;
				} while (y1 >= y2);
				return;
			}
		}
		if (y1 > max_y) {
			delta = (x2 - x1) * (y1 - max_y) / (y1 - y2);
			y1 = max_y;
			if ((x1 += delta) > max_x) {
				/* line is totally off rhs */
				if (y1 > max_y)
					y1 = max_y;
				do {
					if (right[y2] < max_x)
						right[y2] = max_x;
					y2++;
				} while (y2 <= y1);
				return;
			}
		}
		if (x1 < min_x) {
			delta = (y1 - y2) * (min_x - x1) / (x2 - x1);

			while(delta--){
				if (left[y1] > min_x)
					left[y1] = min_x;
				y1--;
			}
			x1 = min_x;
		}
		if (x2 > max_x) {
			delta = (y1 - y2) * (x2 - max_x) / (x2 - x1);

			while(delta--){
				if (right[y2] < max_x)
					right[y2] = max_x;
				y2++;
			}
			x2 = max_x;
		}
	}

	dx = x2 - x1;
	xf = 1;

	if (y2 > y1) {
		dy = y2 - y1;
		yf = 1;
	}
	else {
		dy = y1 - y2;
		yf = -1;
	}

	if (dx > dy) {
		a = dy + dy;
		t = a - dx;
		b = t - dx;
		for (i = 0; i <= dx; i++) {
			if (x1 < left[y1])
				left[y1] = x1;
			if (x1 > right[y1])
				right[y1] = x1;
			x1 += xf;
			if (t < 0)
				t += a;
			else {
				t += b;
				y1 += yf;
			}
		}

	}
	else {
		a = dx + dx;
		t = a - dy;
		b = t - dy;
		for (i = 0; i <= dy; i++) {
			if (x1 < left[y1])
				left[y1] = x1;
			if (x1 > right[y1])
				right[y1] = x1;
			y1 += yf;
			if (t < 0)
				t += a;
			else {
				t += b;
				x1 += xf;
			}
		}
	}
}
	

/*+------------------------------------------------------------------------+ */
/*|Draw a filled polygon of n vertices without clipping.                   | */
/*+------------------------------------------------------------------------+ */
void buff_convex_polyNC(buffer_rec * buff, int n, int *x, int *y, UBYTE col)
{
	int i, iy;
	int low = MAX_YRES, high = 0;

	JLIB_ENTER("buff_convex_polyNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	/* set highest and lowest points to visit */
	for (i = 0; i < n; i++) {
		if (y[i] > high)
			high = y[i];
		if (y[i] < low)
			low = y[i];
	}

	/* reset the minumum amount of the edge tables */
	for (iy = low; iy <= high; iy++) {
		left[iy] = MAX_XRES + 1;
		right[iy] = - 1;
	}

	/* define edges */
	traceNC(x[n - 1], y[n - 1], x[0], y[0]);

	for (i = 0; i < n - 1; i++)
		traceNC(x[i], y[i], x[i + 1], y[i + 1]);

	/* fill horizontal spans of pixels from left[] to right[] */
	for (iy = low; iy <= high; iy++)
		buff_draw_h_lineNC(buff, left[iy], iy, right[iy], col);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|Draw a filled polygon of n vertices with clipping.                      | */
/*+------------------------------------------------------------------------+ */
void buff_convex_poly(buffer_rec * buff, int n, int *x, int *y, UBYTE col)
{
	int i, x1, y1, x2, y2, clipped = 0;

	JLIB_ENTER("buff_convex_poly");

#ifndef JLIB_PRODUCTION
	if ((x == NULL) || (y == NULL))
		jlib_exit(jlib_msg(JLIB_ENULL));
	jlib_check_buffer(buff);
#endif

	x2 = B_CX1(buff) - 1;
	x1 = B_CX2(buff) + 1;
	y2 = B_CY1(buff) - 1;
	y1 = B_CY2(buff) + 1;
	
	for (i = 0; i < n; i++) {
		if (x[i] > x2)
			x2 = x[i];
		if (x[i] < x1)
			x1 = x[i];
		if (y[i] > y2)
			y2 = y[i];
		if (y[i] < y1)
			y1 = y[i];
	}

	if (x2 > B_CX2(buff))
		clipped = 1;
	else
		if (x2 < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (y2 > B_CY2(buff)) {
		y2 = B_CY2(buff);
		clipped = 1;
	}
	else
		if (y2 < B_CY1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (x1 < B_CX1(buff))
		clipped = 1;
	else
		if (x1 > B_CX2(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (y1 < B_CY1(buff)) {
		clipped = 1;
		y1 = B_CY1(buff);
	}
	else
		if (y1 > B_CY2(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (!clipped) {
		buff_convex_polyNC(buff,n,x,y,col);
		JLIB_LEAVE;
		return;
	}

	/* reset the edge tables */
	for (i = y1; i <= y2; i++) {
		left[i] = x2 + 1;
		right[i] = x1 - 1;
	}

	/* define edges */
	trace(buff, x[n - 1], y[n - 1], x[0], y[0], &y1, &y2);

	for (i = 0; i < n - 1; i++)
		trace(buff, x[i], y[i], x[i + 1], y[i + 1], &y1, &y2);

	/* fill spans */
	for (i = y1; i <= y2; i++)
		if (left[i] <= right[i])
			buff_draw_h_line(buff, left[i], i, right[i], col);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|Draw a concave/convex hollow polygon of n vertices without clipping.    | */
/*+------------------------------------------------------------------------+ */
void buff_hollow_polyNC(buffer_rec * buff, int n, int *x, int *y, UBYTE col)
{
	int i;

	JLIB_ENTER("buff_hollow_polyNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	for (i = 1; i < n; i++)
		buff_draw_lineNC(buff, x[i], y[i], x[i - 1], y[i - 1], col);

	buff_draw_lineNC(buff, x[0], y[0], x[n - 1], y[n - 1], col);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|Draw a concave/convex hollow polygon of n vertices with clipping.       | */
/*+------------------------------------------------------------------------+ */
void buff_hollow_poly(buffer_rec * buff, int n, int *x, int *y, UBYTE col)
{
	int i;

	JLIB_ENTER("buff_hollow_poly");

#ifndef JLIB_PRODUCTION
	if ((x == NULL) || (y == NULL) || (n < 1))
		jlib_exit(jlib_msg(JLIB_ENULL));
	jlib_check_buffer(buff);
#endif

	for (i = 1; i < n; i++)
		buff_draw_line(buff, x[i], y[i], x[i - 1], y[i - 1], col);

	buff_draw_line(buff, x[0], y[0], x[n - 1], y[n - 1], col);

	JLIB_LEAVE;
}