/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <stdlib.h>
#include <jlib.h>


/*+------------------------------------------------------------------------+ */
/*|initialise an offscreen buffer that won't be used to blit to the screen.| */
/*+------------------------------------------------------------------------+ */
buffer_rec *noscreen_buff_init(int width, int height)
{
	buffer_rec *buff;
	UBYTE *ptr, **index;

	JLIB_ENTER("noscreen_buff_init");

	/* allocate memory for the buffer record */
	buff = (buffer_rec *) malloc(sizeof(buffer_rec));

	if (buff == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* Set up size and clipping info */
	B_X_SIZE(buff) = width;
	B_Y_SIZE(buff) = height;
	B_CX1(buff) = B_CY1(buff) = 0;
	B_CX2(buff) = B_MAX_X(buff);
	B_CY2(buff) = B_MAX_Y(buff);
	
	/* allocate memory for the buffer itself */
	B_BUFF_PTR(buff) = (UBYTE *) malloc(B_SIZE(buff) * sizeof(UBYTE));

	if (B_BUFF_PTR(buff) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* Create line offset array */
	B_OFFSET_PTR(buff) = (UBYTE **) malloc(height * sizeof(UBYTE *));

	if (B_OFFSET_PTR(buff) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	ptr = B_BUFF_PTR(buff);
	index = B_OFFSET_PTR(buff);

	while (height--) {
		*index = ptr;
		index++;
		ptr += width;
	}

	buff_clear(buff);

	JLIB_LEAVE;
	return buff;
}


/*+------------------------------------------------------------------------+ */
/*|initialise an offscreen buffer                                          | */
/*+------------------------------------------------------------------------+ */
buffer_rec *buff_init(int width, int height)
{
	buffer_rec *buff;

	JLIB_ENTER("buff_init");

	/* allocate memory for the buffer record */
	buff = noscreen_buff_init(width,height);

	JLIB_TARGET_BUFFER_SETUP(buff);

	JLIB_LEAVE;
	return buff;
}


/*+------------------------------------------------------------------------+ */
/*|fill the buff with a particular colour.                                 | */
/*+------------------------------------------------------------------------+ */
void buff_fill(buffer_rec * buff, UBYTE colour)
{
	JLIB_ENTER("buff_fill");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	if ((B_CX1(buff) == 0) && (B_CY1(buff) == 0) && (B_CX2(buff) == B_MAX_X(buff)) && (B_CY2(buff) == B_MAX_Y(buff))) {
		UBYTE *buffer = B_BUFF_PTR(buff);
		int size = B_SIZE(buff);

		MEM_STORE_LONG(buffer, colour, size);
	}
	else
		buff_draw_rectNC(buff,B_CX1(buff),B_CY1(buff),B_CX2(buff),B_CY2(buff),colour);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|fill the buff with a particular colour without clipping.                | */
/*+------------------------------------------------------------------------+ */
void buff_fillNC(buffer_rec * buff, UBYTE colour)
{
	UBYTE *buffer;
	int size;

	JLIB_ENTER("buff_fillNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	buffer = B_BUFF_PTR(buff);
	size = B_SIZE(buff);
	MEM_STORE_LONG(buffer, colour, size);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|clear a buffer by filling it with zeros.                                | */
/*+------------------------------------------------------------------------+ */
void buff_clear(buffer_rec * buff)
{
	JLIB_ENTER("buff_clear");

	buff_fill(buff, 0);

	JLIB_LEAVE;
}

/*+------------------------------------------------------------------------+ */
/*|clear a buffer by filling it with zeros without clipping.               | */
/*+------------------------------------------------------------------------+ */
void buff_clearNC(buffer_rec * buff)
{
	JLIB_ENTER("buff_clearNC");

	buff_fillNC(buff, 0);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|free the storage used by a buffer to the OS.                            | */
/*+------------------------------------------------------------------------+ */
buffer_rec *buff_free(buffer_rec * buff)
{
	JLIB_ENTER("buff_free");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	JLIB_TARGET_BUFFER_FREE(buff);

	if (B_BUFF_PTR(buff) != NULL)
		free(B_BUFF_PTR(buff));

	if (B_OFFSET_PTR(buff) != NULL)
		free(B_OFFSET_PTR(buff));

	free(buff);

	JLIB_LEAVE;
	return (buffer_rec *) NULL;
}


/*+------------------------------------------------------------------------+ */
/*|Free the storage used by a screen buffer to the OS.                     | */
/*+------------------------------------------------------------------------+ */
buffer_rec *screen_buff_free(buffer_rec * buff)
{
	JLIB_ENTER("screen_buff_free");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	JLIB_TARGET_SCREEN_BUFFER_FREE(buff);

	if (B_OFFSET_PTR(buff) != NULL)
		free(B_OFFSET_PTR(buff));

	free(buff);

	JLIB_LEAVE;
	return (buffer_rec *) NULL;
}


/*+------------------------------------------------------------------------+ */
/*|Remap the colors in a buffer.                                           | */
/*+------------------------------------------------------------------------+ */
void buff_remap_colors(buffer_rec * buff, UBYTE * remap)
{
	UBYTE *buffer;
	int size;

	JLIB_ENTER("buff_remap_colors");

#ifndef JLIB_PRODUCTION
	if (remap == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));

	jlib_check_buffer(buff);
#endif

	buffer = B_BUFF_PTR(buff);

	for (size = B_SIZE(buff) - 1; size >= 0; size--)
		buffer[size] = remap[buffer[size]];

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|Set a buffers clipping region.                                          | */
/*+------------------------------------------------------------------------+ */
void buff_set_clip_region(buffer_rec *buff, int x1, int y1, int x2, int y2)
{
	JLIB_ENTER("buff_set_clip_region");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	/* Ensure clipping region is within the buffer */
	JLIB_CLAMP(x1,0,B_MAX_X(buff));
	JLIB_CLAMP(x2,0,B_MAX_X(buff));
	JLIB_CLAMP(y1,0,B_MAX_Y(buff));
	JLIB_CLAMP(y2,0,B_MAX_Y(buff));

	if (x2 < x1)
		JLIB_SWAP(x1, x2);

	if (y2 < y1)
		JLIB_SWAP(y1, y2);

	B_CX1(buff) = x1;
	B_CY1(buff) = y1;
	B_CX2(buff) = x2;
	B_CY2(buff) = y2;

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|Reset a buffers clipping region to include the whole buffer.            | */
/*+------------------------------------------------------------------------+ */
void buff_reset_clip_region(buffer_rec  *buff)
{
	JLIB_ENTER("buff_reset_clip_region");

#ifndef JLIB_PRODUCTION
	if (buff == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
#endif
	
	B_CX1(buff) = B_CY1(buff) = 0;
	B_CX2(buff) = B_MAX_X(buff);
	B_CY2(buff) = B_MAX_Y(buff);

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	JLIB_LEAVE;
}