// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

#include "gfx/bitmap.h"
#include "gfx/font.h"
#include "gfx/imageio.h"
#include "compiler/types.h"

Font::Font() 
  {
  charwidth=0;
  charheight=0;
  codebytes=0;
  }
Font::~Font() 
  {
  delete code;
  delete charcode;
  codebytes=0;
  }
Font &Font::operator =(const Font &Orig)
  {
  this->~Font();
  charwidth=Orig.charwidth;
  charheight=Orig.charheight;
  codebytes=Orig.codebytes;
  code=new UBYTE[codebytes];
  memcpy(code,Orig.code,codebytes);
  charcode=new UBYTE*[256];
  for (int CharNr=0;CharNr<256;CharNr++) { charcode[CharNr]=code+(Orig.charcode[CharNr]-Orig.code); };
  return(*this);
  }
// Expects a pcx file with 256 characters arranged in a horizontal row.
int Font::Load(char *FileName)
  {
  Bitmap Image;
  if (LoadPCX(Image,FileName)!=0) return(-1); // can't load file
  if (Image.Width()%256!=0) return(-2); // irregular picture width (256 chars expected)
  charwidth=Image.Width()/256;
  charheight=Image.Height();
  // run length encoding of the chars
  // code for one char
  // { first line;number of lines;  {line; ... ;line} }
  //   where line is { {transparent pixels;set pixels}...{ } }
  const int MaxBytesPerChar=2+(charwidth+1)*charheight;
  UBYTE *Buffer=0;
  int *CodePosition=new int[256];
  int CurrentPosition=0;
  for (int CharNr=0;CharNr<256;CharNr++)
    {
    // reserve buffer memory for the code
    UBYTE *NewBuffer=new UBYTE[CurrentPosition+MaxBytesPerChar+1];
    memcpy(NewBuffer,Buffer,CurrentPosition+1);
    delete Buffer;
    Buffer=NewBuffer;
    // encode a char
    CodePosition[CharNr]=CurrentPosition;
    Buffer[CurrentPosition+0]=0;
    Buffer[CurrentPosition+1]=(UBYTE)charwidth;
    CurrentPosition+=2;
    int EndEmptyLines=0;
    for (int sy=0;sy<charheight;sy++)
      {
      // encode a single line
      int sx=CharNr*charwidth;
      int sxEnd=sx+charwidth;
      int Runs=0,Transparent,Set;
      do 
        {
        Transparent=Set=0;
        while (sx<sxEnd && Transparent<255 && Image.GetPixel(sx,sy)==0) { sx++;Transparent++; }
        while (sx<sxEnd && Set<255         && Image.GetPixel(sx,sy)!=0) { sx++;Set++; }
        Buffer[CurrentPosition++]=(UBYTE)Transparent;
        Buffer[CurrentPosition++]=(UBYTE)Set;
        Runs++;
        }
      while (sx<sxEnd);
      if (Runs==1 && Set==0) // empty line?
        {
        if (Buffer[CodePosition[CharNr]+1]==0) // all starting lines were empty?
          { 
          CurrentPosition-=2;
          Buffer[CodePosition[CharNr]+0]++; // one more empty line in the beginning
          }
        EndEmptyLines++;
        }
      else 
        EndEmptyLines=0;
      }
    Buffer[CodePosition[CharNr]+1]-=EndEmptyLines;
    } 
  // create the final code array
  codebytes=CurrentPosition+1;
  code=new UBYTE[codebytes];
  charcode=new UBYTE*[256];
  memcpy(code,Buffer,codebytes);
  delete Buffer;
  // set up the code ptr array 
  for (CharNr=0;CharNr<256;CharNr++) charcode[CharNr]=code+CodePosition[CharNr];
  return(0); // OK
  }


void TextOut(Bitmap &Dst,int dx, int dy,Font &Src,ColorRef Color,char *Text,int Chars)
  {
  #ifdef DEBUG
  if ((Src.charwidth|Src.charheight)==0) ErrorHandler.Abort("TextOut()  Invalid font.");
  if (!Dst.IsValid()) ErrorHandler.Abort("TextOut()  Invalid dest bitmap.");
  #endif
  // check which chars are visible
  if (dy<0 || dy+Src.charheight>Dst.Height()) return; // line invisible
  if (dx<0) { int LeftChars=dx/Src.charwidth;dx=dx%Src.charwidth;Text+=LeftChars; }
  if (dx>=Dst.Width()) return;
  if (dx+Chars*Src.charwidth>Dst.Width()) Chars=(Dst.Width()-dx)/Src.charwidth;
  // draw visible chars
  if (ColorRefType(Color)==CRRGB8) Color=Dst.Palette().GetPaletteIndex(Color);
  while (Chars--)
    {
    // decode the char
    UBYTE *CodePtr=Src.charcode[*Text];
    int y=dy+CodePtr[0];
    int Lines=CodePtr[1];
    CodePtr+=2;
    while (Lines--)
      {
      int x=dx;
      int xEnd=x+Src.charwidth;
      // decode a line
      do
        {
        int Transparent=CodePtr[0];
        int Set=CodePtr[1];
        CodePtr+=2;
        if (Set==0) break;
        x+=Transparent;
        while (Set--) Dst.SetPixel(x++,y,Color);
        }
      while (x<xEnd);
      y++;
      }
    Text++;
    dx+=Src.charwidth;
    }
  }
